/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/drawingmodelmixin', function () {

    'use strict';

    // class DrawingModelMixin ================================================

    /**
     * A mix-in class that extends the public API of any drawing model object
     * that has been inserted into a sheet of a spreadsheet document.
     *
     * @constructor
     *
     * @extends DrawingModel
     *
     * @internal
     *  This is a mix-in class supposed to extend an existing instance of the
     *  class DrawingModel or any of its sub classes. Expects the symbol 'this'
     *  to be bound to an instance of DrawingModel.
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this drawing model.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model containing the drawing collection with this drawing
     *  model object.
     */
    function DrawingModelMixin(app, sheetModel) {

        // methods ------------------------------------------------------------

        /**
         * Returns the complete logical document position of this drawing
         * model.
         *
         * @returns {Number[]|Null}
         *  The logical position of the drawing object, if it is still
         *  contained in the drawing collection, including the zero-based sheet
         *  index as first array element; otherwise null.
         */
        this.getPosition = function () {
            var position = sheetModel.getDrawingCollection().getModelPosition(this, { deep: true });
            return position ? [sheetModel.getIndex()].concat(position) : null;
        };

        /**
         * Returns the logical address of the range covered by this drawing
         * model object.
         *
         * @returns {Object}
         *  The logical address of the range covered by this drawing model.
         */
        this.getRange = function () {

            var // the drawing attributes of this drawing object
                drawingAttrs = this.getMergedAttributes().drawing;

            return {
                start: [drawingAttrs.startCol, drawingAttrs.startRow],
                end: [drawingAttrs.endCol, drawingAttrs.endRow]
            };
        };

        /**
         * Returns the sheet rectangle in pixels covered by this drawing
         * object.
         *
         * @returns {Object|Null}
         *  The position of this drawing object in the sheet in pixels,
         *  according to the current sheet zoom factor, in the properties
         *  'left', 'top', 'width', and 'height'; or null if the drawing object
         *  is located entirely in hidden columns or rows and thus invisible.
         */
        this.getRectangle = function () {
            return sheetModel.getDrawingCollection().getRectangleForAttributes(this.getMergedAttributes());
        };

    } // class DrawingModelMixin

    // exports ================================================================

    return DrawingModelMixin;

});
