/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/dataseriesmodel',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/attributedmodel',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/formula/tokenarray'
    ], function (Utils, AttributedModel, SheetUtils, TokenArray) {

    'use strict';

    var SOURCE_LINK_TYPES = ['names', 'title', 'values', 'bubbles'];

    // class DataSeriesModel ==================================================

    /**
     * A class that implements a data series model to be inserted into a chart
     * model.
     *
     * Series can have formulas 'Title', 'Names', 'Values' and 'Bubbles' on
     * 'change:attributes' it refreshes all formulas to TokenArrays
     *
     * Additionally it can have 'fill' and 'line' for styling
     *
     * @constructor
     *
     * @extends DrawingModel
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this chart model.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model that contains the chart object with this data series.
     *
     * @param {Object} [initAttributes]
     *  Initial formatting attribute set for this data series model.
     */
    function DataSeriesModel(app, sheetModel, initAttributes) {

        var // self reference
            self = this,

            // token arrays for dynamic source ranges, mapped by source link type
            tokenArrays = {};

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, app, initAttributes, { additionalFamilies: ['series', 'fill', 'line'] });

        // private methods ----------------------------------------------------

        /**
         * Updates the token arrays, after the source link attributes of this
         * data series have been changed.
         */
        function changeAttributesHandler() {

            var // the merged series attributes
                attributes = self.getMergedAttributes().series;

            _(SOURCE_LINK_TYPES).each(function (type) {

                var sourceLink = attributes[type];

                // create, update, or delete token array according to type of source data
                if (_.isString(sourceLink)) {
                    // string: parse link formula
                    if (!(type in tokenArrays)) {
                        var tokenArray = new TokenArray(app, sheetModel);
                        tokenArrays[type] = tokenArray;
                        tokenArray.on('triggered', function () {
                            self.setAttributes({ series: Utils.makeSimpleObject(type, tokenArray.getFormula()) });
                        });
                    }
                    tokenArrays[type].invokeSilently('parseFormula', sourceLink);
                } else if (type in tokenArrays) {
                    // else: constant source data, delete token array
                    tokenArrays[type].destroy();
                    delete tokenArrays[type];
                }
            });
        }

        // methods ------------------------------------------------------------

        this.getFormula = function () {
            var formula = '';

            this.iterateTokens(function (token) {
                if (formula.length) {
                    formula += ',';
                }
                formula += token.getFormula();
            });

            return formula;
        };

        this.checkRangeInsideInterval = function (sheetModelIndex, interval, insert, columns) {

            var changed = false;

            this.iterateTokens(function (tokenArray) {
                if (tokenArray.transformFormula(sheetModelIndex, interval, insert, columns)) {
                    changed = true;
                }
            });

            return changed;
        };

        this.iterateTokens = function (fn) {
            _.each(tokenArrays, function (token, name) {
                if (token) {
                    fn(token, name);
                }
            });
        };

        /**
         * Returns whether any of the source links contained in this data
         * series overlaps with any of the passed cell ranges.
         *
         * @param {Array} ranges
         *  The logical addresses of the cell ranges to be checked. Each cell
         *  range address object MUST contain an additional property 'sheet'
         *  with the zero-based sheet index of the range.
         *
         * @returns {Boolean}
         *  Whether any of the passed ranges overlaps with the source links of
         *  this data series.
         */
        this.rangesOverlap = function (ranges) {
            return _(tokenArrays).any(function (tokenArray) {
                return _(tokenArray.resolveRangeList()).any(function (linkRange) {
                    return _(ranges).any(function (range) {
                        return (linkRange.sheet === range.sheet) && SheetUtils.rangeOverlapsRange(linkRange, range);
                    });
                });
            });
        };

        // initialization -----------------------------------------------------

        this.on('change:attributes', changeAttributesHandler);
        changeAttributesHandler();

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _(tokenArrays).invoke('destroy');
            tokenArrays = self = null;
        });

    } // class DataSeriesModel

    // exports ================================================================

    // derive this class from class AttributedModel
    return AttributedModel.extend({ constructor: DataSeriesModel });

});
