/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/documentstyles',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/utils/color',
     'io.ox/office/editframework/utils/border',
     'io.ox/office/editframework/model/documentstyles',
     'io.ox/office/drawinglayer/model/drawingstylecollection',
     'io.ox/office/spreadsheet/model/cellstylecollection'
    ], function (Utils, Color, Border, DocumentStyles, DrawingStyleCollection, CellStyleCollection) {

    'use strict';

    var // default values of spreadsheet document attributes
        SPREADSHEET_DOCUMENT_ATTRIBUTES = {

            /**
             * Number of columns in a sheet.
             */
            cols: 1024,

            /**
             * Number of rows in a sheet.
             */
            rows: 1048576,

            // view attributes

            /**
             * Index of the active (displayed) sheet.
             */
            activeSheet: 0
        },

        // definitions for character attributes in cells
        CHARACTER_ATTRIBUTE_DEFINITIONS = {
            fontName: { def: 'Arial' },
            fontSize: { def: 11 },
            bold: { def: false },
            italic: { def: false },
            underline: { def: false },
            strike: { def: 'none' },
            color: { def: Color.AUTO },
            url: { def: '', scope: 'element' }
        },

        // definitions for cell attributes
        CELL_ATTRIBUTE_DEFINITIONS = {
            alignHor: { def: 'auto' },
            alignVert: { def: 'bottom' },
            wrapText: { def: false },
            numberFormat: { def: { id: 0, code: 'General' }, merge: mergeNumberFormat },
            fillColor: { def: Color.AUTO },
            borderLeft: { def: Border.NONE },
            borderRight: { def: Border.NONE },
            borderTop: { def: Border.NONE },
            borderBottom: { def: Border.NONE },
            borderInsideHor: { def: Border.NONE, scope: 'element' },
            borderInsideVert: { def: Border.NONE, scope: 'element' },
            unlocked: { def: false },
            hidden: { def: false }
        },

        // definitions for column attributes
        COLUMN_ATTRIBUTE_DEFINITIONS = {
            visible: { def: true },
            width: { def: 2000 },
            autoFit: { def: true },
            customWidth: { def: false }
        },

        // definitions for row attributes
        ROW_ATTRIBUTE_DEFINITIONS = {
            visible: { def: true },
            height: { def: 500 },
            customHeight: { def: false },
            customFormat: { def: false }
        },

        // definitions for sheet attributes
        SHEET_ATTRIBUTE_DEFINITIONS = {
            visible: { def: true },
            locked: { def: false },
            baseColWidth: { def: 8 },

            // view attributes
            zoom: { def: 1 },
            selectedRanges: { def: [{ start: [0, 0], end: [0, 0] }] },
            activeIndex: { def: 0 },
            activeCell: { def: [0, 0] },
            selectedDrawings: { def: [] },
            showGrid: { def: true },
            gridColor: { def: Color.AUTO },
            splitMode: { def: 'split' },
            splitWidth: { def: 0 },
            splitHeight: { def: 0 },
            activePane: { def: 'topLeft' },
            scrollLeft: { def: 0 },
            scrollRight: { def: 0 },
            scrollTop: { def: 0 },
            scrollBottom: { def: 0 }
        },

        // definitions for data validation attributes
        VALIDATION_ATTRIBUTE_DEFINITIONS = {
            type: { def: 'all' },
            compare: { def: '' },
            value1: { def: '' },
            value2: { def: '' },
            showInfo: { def: true },
            infoTitle: { def: '' },
            infoText: { def: '' },
            showError: { def: true },
            errorTitle: { def: '' },
            errorText: { def: '' },
            errorType: { def: 'error' },
            showDropDown: { def: true },
            ignoreEmpty: { def: true }
        },

        // additional definitions for drawing attributes
        DRAWING_ATTRIBUTE_DEFINITIONS = {

            /**
             * Type of the drawing anchor (specifies which anchor attributes
             * have to be used), and runtime behavior of the drawing while
             * inserting, removing, or changing columns and rows. Supported
             * values are 'twoCell', 'oneCell', 'absolute', 'twoCellAsOneCell',
             * and 'twoCellAsAbsolute'.
             */
            anchorType: { def: 'twoCell', scope: 'element' },

            /**
             * Zero-based index of the column containing the leading border of
             * the drawing object.
             */
            startCol: { def: 0, scope: 'element' },

            /**
             * The exact position of the leading border in the column, as ratio
             * of the column width (in the range [0,1]).
             */
            startColOffset: { def: 0, scope: 'element' },

            /**
             * Zero-based index of the row containing the top border of the
             * drawing object.
             */
            startRow: { def: 0, scope: 'element' },

            /**
             * The exact position of the top border in the row, as ratio of the
             * row height (in the range [0,1]).
             */
            startRowOffset: { def: 0, scope: 'element' },

            /**
             * Zero-based index of the column containing the trailing border of
             * the drawing object.
             */
            endCol: { def: 0, scope: 'element' },

            /**
             * The exact position of the trailing border in the column, as
             * ratio of the column width (in the range [0,1]).
             */
            endColOffset: { def: 0, scope: 'element' },

            /**
             * Zero-based index of the row containing the bottom border of the
             * drawing object.
             */
            endRow: { def: 0, scope: 'element' },

            /**
             * The exact position of the bottom border in the row, as ratio of
             * the row height (in the range [0,1]).
             */
            endRowOffset: { def: 0, scope: 'element' }
        };

    // global private functions ===============================================

    /**
     * Returns a number format attribute value that results from updating an
     * existing number format with another number format. Keeps the format
     * identifier of the existing number format, if the format codes of both
     * formats are equal, and the new number format does not contain an own
     * identifier.
     */
    function mergeNumberFormat(oldFormat, newFormat) {
        return (!('id' in newFormat) && (oldFormat.code === newFormat.code)) ? oldFormat : newFormat;
    }

    // class SpreadsheetDocumentStyles ========================================

    /**
     * Provides the style sheet containers for all attribute families used in a
     * spreadsheet document, and other containers collecting special formatting
     * information for the spreadsheet document.
     *
     * @constructor
     *
     * @extends DocumentStyles
     *
     * @param {SpreadsheetApplication} app
     *  The root application instance.
     */
    function SpreadsheetDocumentStyles(app) {

        var // the font collection of the document
            fontCollection = null,

            // the cell style collection of the document
            cellStyles = null,

            // the drawing style collection of the document
            drawingStyles = null;

        // base constructor ---------------------------------------------------

        DocumentStyles.call(this, app, SPREADSHEET_DOCUMENT_ATTRIBUTES);

        // private methods ----------------------------------------------------

        function getDefaultCharAttributes() {
            return cellStyles.getStyleSheetAttributes(cellStyles.getDefaultStyleId()).character;
        }

        // methods ------------------------------------------------------------

        /**
         * Calculates the text line height in 1/100 of millimeters to be used
         * for cell text and row heights, for the passed character attributes.
         *
         * @param {Object} charAttributes
         *  Character formatting attributes influencing the line height.
         *  @param {String} charAttributes.fontName
         *      The name of the original font family (case-insensitive).
         *  @param {Number} charAttributes.fontSize
         *      The original font size, in points.
         *  @param {Boolean} [charAttributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [charAttributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         *
         * @returns {Number}
         *  The line height for the passed character attributes, in 1/100 mm.
         */
        this.getLineHeight = function (charAttributes) {

            var // the normal line height according to the font settings
                lineHeight = fontCollection.getNormalLineHeight(charAttributes);

            // use at least 120% of the font size, to compensate for browsers
            // reporting very small line heights (especially Chrome)
            lineHeight = Math.max(lineHeight, Utils.convertLength(charAttributes.fontSize, 'pt', 'px') * 1.2);

            // enlarge the resulting 'normal' line height by another 10% to add top/bottom cell padding
            return Utils.convertLengthToHmm(Math.floor(lineHeight * 1.1), 'px');
        };

        /**
         * Returns the default column width for all undefined columns in a
         * sheet, according to the font settings in the current default cell
         * style, and the passed base column width.
         *
         * @param {Number} baseWidth
         *  The base column width, as number of zero digits.
         *
         * @returns {Number}
         *  The default column width for all undefined columns, in 1/100 mm.
         */
        this.getDefaultColWidth = function (baseWidth) {

            var charAttributes = getDefaultCharAttributes(),
                digitWidth = fontCollection.getDigitWidth(charAttributes),
                colWidth = digitWidth * baseWidth + 5;

            return Utils.convertLengthToHmm(colWidth, 'px');
        };

        /**
         * Returns the default row height for all undefined rows in a sheet,
         * according to the font settings in the current default cell style.
         *
         * @returns {Number}
         *  The default row height for all undefined rows, in 1/100 mm.
         */
        this.getDefaultRowHeight = function () {
            return this.getLineHeight(getDefaultCharAttributes());
        };

        // initialization -----------------------------------------------------

        fontCollection = this.getFontCollection();
        cellStyles = new CellStyleCollection(app, this);
        drawingStyles = new DrawingStyleCollection(app, this);

        this.registerAttributeDefinitions('character', CHARACTER_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('cell', CELL_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('column', COLUMN_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('row', ROW_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('sheet', SHEET_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('validation', VALIDATION_ATTRIBUTE_DEFINITIONS)
            .registerAttributeDefinitions('drawing', DRAWING_ATTRIBUTE_DEFINITIONS);

        this.addStyleCollection(cellStyles)
            .addStyleCollection(drawingStyles);

        this.registerDestructor(function () {
           app = fontCollection = cellStyles = drawingStyles = null;
        });

    } // class SpreadsheetDocumentStyles

    // exports ================================================================

    // derive this class from class DocumentStyles
    return DocumentStyles.extend({ constructor: SpreadsheetDocumentStyles });

});
