/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/preview/view/view',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/app/extensionregistry',
     'io.ox/office/baseframework/view/baseview',
     'io.ox/office/baseframework/view/sidepane',
     'io.ox/office/baseframework/view/component',
     'io.ox/office/baseframework/view/toolbox',
     'io.ox/office/preview/view/toppane',
     'io.ox/office/preview/view/controls',
     'io.ox/office/preview/view/pagegroup',
     'io.ox/office/preview/view/pageloader',
     'io.ox/office/preview/view/nodetouch',
     'gettext!io.ox/office/preview',
     'less!io.ox/office/preview/view/style'
    ], function (Utils, ExtensionRegistry, BaseView, SidePane, Component, ToolBox, TopPane, Controls, PageGroup, PageLoader, NodeTouch, gt) {

    'use strict';

    var // predefined zoom factors
        ZOOM_FACTORS = [25, 35, 50, 75, 100, 150, 200, 300, 400, 600, 800],

        // the speed of scroll animations
        ANIMATION_DELAY = 25;

    // class PreviewView ======================================================

    /**
     * The view of the OX Preview application. Shows pages of the previewed
     * document with a specific zoom factor.
     *
     * @constructor
     *
     * @extends BaseView
     *
     * @param {PreviewApplication} app
     *  The OX Preview application that has created this view instance.
     */
    function PreviewView(app) {

        var // self reference
            self = this,

            // the preview model
            model = null,

            // the scrollable document content node
            contentRootNode = null,

            // the root DOM node of the search pane provided by the core framework
            searchPaneNode = null,

            // the top-level view pane containing global controls and the tool bar tabs
            topPane = null,

            // the main side pane containing the thumbnails
            sidePane = null,

            // the page preview control
            pageGroup = null,

            // the root node containing all page nodes
            pageContainerNode = $('<div>').addClass('page-container'),

            // all page nodes as permanent jQuery collection, for performance
            pageNodes = $(),

            // the queue for AJAX page requests
            pageLoader = new PageLoader(app),

            // all page nodes with contents, keyed by one-based page number
            loadedPageNodes = {},

            // the timer used to defer loading more pages above and below the visible area
            loadMorePagesTimer = null,

            // one-based index of the selected page
            selectedPage = 0,

            // current zoom type (percentage or keyword)
            zoomType = 100,

            // the current effective zoom factor, in percent
            zoomFactor = 100,

            // the size available in the application pane
            availableSize = { width: 0, height: 0 },

            // current scroll animation
            scrollAnimation = null;

        // base constructor ---------------------------------------------------

        BaseView.call(this, app, {
            initHandler: initHandler,
            initGuiHandler: initGuiHandler,
            contentFocusable: true,
            contentScrollable: true,
            contentMargin: 30,
            overlayMargin: { left: 8, right: Utils.SCROLLBAR_WIDTH + 8, bottom: Utils.SCROLLBAR_HEIGHT }
        });

        // private methods ----------------------------------------------------

        /**
         * Selects the specified page. Updates the status label, the thumbnail
         * preview, and optionally scrolls the view to the specified page.
         *
         * @param {Number} page
         *  The one-based index of the selected page.
         */
        function selectPage(page) {
            if ((1 <= page) && (page <= model.getPageCount()) && (page !== selectedPage)) {
                selectedPage = page;
                pageGroup.selectAndShowPage(page);
                app.getController().update();
            }
        }

        /**
         * Scrolls the application pane node by the passed distance.
         */
        function scrollContentRootNode(diffX, diffY) {
            var scrollLeft = contentRootNode.scrollLeft() + Math.round(diffX),
                scrollTop = contentRootNode.scrollTop() + Math.round(diffY);
            contentRootNode.scrollLeft(scrollLeft).scrollTop(scrollTop);
        }

        /**
         * Aborts the current scroll animation.
         */
        function abortScrollAnimation() {
            if (scrollAnimation) {
                scrollAnimation.abort();
                scrollAnimation = null;
            }
        }

        /**
         * Starts a new scroll animation.
         *
         * @param {Function} callback
         *  A callback function that has to return the scroll distance for each
         *  animation frame, as object in the properties 'x' and 'y'. The
         *  function receives the zero-based index of the current animation
         *  frame.
         *
         * @param {Numner} frames
         *  The number of animation frames.
         */
        function startScrollAnimation(callback, frames) {
            abortScrollAnimation();
            scrollAnimation = app.repeatDelayed(function (index) {
                var move = callback.call(self, index);
                scrollContentRootNode(move.x, move.y);
            }, { delay: ANIMATION_DELAY, cycles: frames })
            .always(function () { scrollAnimation = null; });
        }

        /**
         * Scrolls the view to the specified page.
         *
         * @param {Number} page
         *  The one-based index of the page to be scrolled to.
         */
        function scrollToPage(page) {

            var // the new page node (prevent selecting nodes from end of array for negative values)
                pageNode = (page > 0) ? pageNodes.eq(page - 1) : $();

            if (pageNode.length > 0) {
                contentRootNode.scrollTop(Utils.getChildNodePositionInNode(contentRootNode, pageNode).top - self.getContentMargin().top);
            }
        }

        /**
         * Loads the specified page, and stores the original page size at the
         * page node.
         */
        function loadPage(page, priority) {

            var // the page node of the specified page
                pageNode = pageNodes.eq(page - 1);

            // do not load initialized page again
            if (pageNode.children().length === 0) {
                pageLoader.loadPage(pageNode, page, { format: 'svg', priority: priority })
                .done(function () {
                    loadedPageNodes[page] = pageNode;
                    refreshLayout(page);
                })
                .fail(function () {
                    pageNode.append($('<div>').addClass('error-message').text(gt('Sorry, this page is not available at the moment.')));
                });
            }
        }

        /**
         * Cancels all running background tasks regarding updating the page
         * nodes in the visible area.
         */
        function cancelMorePagesTimer() {
            // cancel the timer that loads more pages above and below the visible
            // area, e.g. to prevent (or defer) out-of-memory situations on iPad
            if (loadMorePagesTimer) {
                loadMorePagesTimer.abort();
                loadMorePagesTimer = null;
            }
        }

        /**
         * Loads all pages that are currently visible in the application pane.
         */
        function loadVisiblePages() {

            var // find the first page that is visible at the top border of the visible area
                beginPage = _.sortedIndex(pageNodes, contentRootNode.scrollTop(), function (value) {
                    if (_.isNumber(value)) { return value; }
                    var pagePosition = Utils.getChildNodePositionInNode(contentRootNode, value);
                    return pagePosition.top + pagePosition.height - 1;
                }) + 1,
                // find the first page that is not visible anymore at the bottom border of the visible area
                endPage = _.sortedIndex(pageNodes, contentRootNode.scrollTop() + contentRootNode[0].clientHeight, function (value) {
                    if (_.isNumber(value)) { return value; }
                    return Utils.getChildNodePositionInNode(contentRootNode, value).top;
                }) + 1,
                // first page loaded with lower priority
                beginPageLowerPrio = Math.max(1, beginPage - 2),
                // one-after last page loaded with lower priority
                endPageLowerPrio = Math.min(endPage + 2, model.getPageCount() + 1);

            // loads the specified page with high priority
            function loadPageHighPriority(page) {
                loadPage(page, 'high');
            }

            // loads the specified page with low priority
            function loadPageMediumPriority(page) {
                loadPage(page, 'medium');
            }

            // fail safety: do nothing if called while view is hidden (e.g. scroll handlers)
            if (!self.isVisible()) { return; }

            // abort old requests not yet running
            pageLoader.abortQueuedRequests();

            // load visible pages with high priority
            Utils.iterateRange(beginPage, endPage, loadPageHighPriority);

            // load two pages above and below the visible area with medium priority after a short delay
            cancelMorePagesTimer();
            loadMorePagesTimer = app.executeDelayed(function () {
                Utils.iterateRange(endPage, endPageLowerPrio, loadPageMediumPriority);
                Utils.iterateRange(beginPage - 1, beginPageLowerPrio - 1, loadPageMediumPriority, { step: -1 });
            }, { delay: 500 });

            // clear all other pages
            _.each(loadedPageNodes, function (pageNode, page) {
                if ((page < beginPageLowerPrio) || (page >= endPageLowerPrio)) {
                    app.destroyImageNodes(pageNode);
                    pageNode.empty();
                    delete loadedPageNodes[page];
                }
            });

            // activate the page located at the top border of the visible area
            selectPage(beginPage);
        }

        /**
         * Recalculates the size of the specified page node, according to the
         * original page size and the current zoom type.
         */
        function calculatePageZoom(pageNode) {

            var // the original size of the current page
                pageSize = pageLoader.getPageSize(pageNode),
                // the effective zoom factor for the page
                pageZoomFactor = 100;

            // calculate the effective zoom factor
            if (_.isNumber(zoomType)) {
                pageZoomFactor = zoomType;
            } else if (zoomType === 'width') {
                pageZoomFactor = availableSize.width / pageSize.width * 100;
            } else if (zoomType === 'page') {
                pageZoomFactor = Math.min(availableSize.width / pageSize.width * 100, availableSize.height / pageSize.height * 100);
            }
            pageZoomFactor = Utils.minMax(pageZoomFactor, self.getMinZoomFactor(), self.getMaxZoomFactor());

            // set the zoom factor at the page node
            pageLoader.setPageZoom(pageNode, pageZoomFactor / 100);
        }

        /**
         * Recalculates the size of the page nodes according to the original
         * page sizes, the current zoom type, and the available space in the
         * application pane; and restores the scroll position.
         *
         * @param {Number} [page]
         *  If specified, the one-based index of a single page whose zoom will
         *  be updated.
         */
        function refreshLayout(page) {

            var // half of the content root node height
                contentCenter = contentRootNode.height() / 2,
                // find the page covering the vertical center of the visible area
                centerPage = _.sortedIndex(pageNodes, contentRootNode.scrollTop() + contentCenter, function (value) {
                    if (_.isNumber(value)) { return value; }
                    var pagePosition = Utils.getChildNodePositionInNode(contentRootNode, value);
                    return pagePosition.top + pagePosition.height - 1;
                }) + 1,
                // the position of the center page in the visible area of the application pane
                centerPagePosition = null,
                // the ratio of the center page that is located exactly at the screen center
                centerPageRatio = 0,
                // current content margin according to browser window size
                contentMargin = ((window.innerWidth <= 1024) || (window.innerHeight <= 640)) ? 0 : 30;

            if (model.getPageCount() === 0) { return; }

            // restrict center page to valid page numbers, get position of the center page
            centerPage = Utils.minMax(centerPage, 1, model.getPageCount());
            centerPagePosition = Utils.getChildNodePositionInNode(contentRootNode, pageNodes[centerPage - 1], { visibleArea: true });

            // calculate the page ratio (which part of the page is above the content center)
            centerPageRatio = (centerPagePosition.height > 0) ? ((contentCenter - centerPagePosition.top) / centerPagePosition.height) : 0;

            // set the current content margin between application pane border and page nodes
            self.setContentMargin(contentMargin);

            // the available inner size in the application pane
            availableSize.width = contentRootNode[0].clientWidth - 2 * contentMargin;
            availableSize.height = ((zoomType === 'page') ? contentRootNode.height() : contentRootNode[0].clientHeight) - 2 * contentMargin;

            // Process one or all page nodes. Detaching the page nodes while
            // updating them reduces total processing time by 95% on touch devices!
            if (_.isNumber(page)) {
                calculatePageZoom(pageNodes.eq(page - 1));
            } else {
                pageNodes.detach().each(function () { calculatePageZoom($(this)); }).appendTo(pageContainerNode);
            }

            // update 'current zoom factor' for the selected page
            zoomFactor = pageLoader.getPageZoom(pageNodes[selectedPage - 1]) * 100;

            // restore the correct scroll position with the new page sizes
            centerPagePosition = Utils.getChildNodePositionInNode(contentRootNode, pageNodes[centerPage - 1]);
            contentRootNode.scrollTop(centerPagePosition.top + Math.round(centerPagePosition.height * centerPageRatio) - contentCenter);

            // try to load more pages that became visible after updating zoom
            loadVisiblePages();
        }

        /**
         * Stores the current scroll position while the view is hidden.
         */
        function windowHideHandler() {
            contentRootNode.data({ scrollLeft: contentRootNode.scrollLeft(), scrollTop: contentRootNode.scrollTop() });
        }

        /**
         * Restores the scroll position after the view was hidden.
         */
        function windowShowHandler() {
            contentRootNode.scrollLeft(contentRootNode.data('scrollLeft') || 0).scrollTop(contentRootNode.data('scrollTop') || 0);
        }

        /**
         * Enables or disables a smaller version of the search pane.
         */
        function toggleSmallSearchPane(state) {
            searchPaneNode.find('.search-query-container').css({ width: state ? 180 : 300 });
            searchPaneNode.find('.replace-query-container').css({ width: state ? 330 : 400 });
        }

        /**
         * Toggles the small search pane mode according to the current display
         * orientation.
         */
        function updateSmallSearchPane() {
            if (Modernizr.touch) {
                if (window.matchMedia('(orientation: landscape)').matches) {
                    toggleSmallSearchPane(false);
                } else if (window.matchMedia('(orientation: portrait)').matches) {
                    toggleSmallSearchPane(true);
                } else {
                    Utils.warn('BaseView.updateSmallSearchPane(): unknown display orientation');
                }
            }
        }

        /**
         * Updates the view after the global search bar has been opened.
         */
        function searchOpenHandler() {
            updateSmallSearchPane();
            // event triggers before the DOM element becomes visible
            _.defer(function () { self.refreshPaneLayout(); });
        }

        /**
         * Updates the view after the global search bar has been closed.
         */
        function searchCloseHandler() {
            // event triggers before the DOM element becomes visible
            _.defer(function () { self.refreshPaneLayout(); });
        }

        /**
         * Handles 'orientationchange' events from mobile devices.
         */
        function orientationChangeHandler(event) {
            // use event.orientation and media queries for the orientation detection.
            // 'window.orientation' depends on where the device defines it's top
            // and therefore says nothing reliable about the orientation.
            if (event && (event.orientation === 'landscape')) {
                toggleSmallSearchPane(false);
            } else if (event && (event.orientation === 'portrait')) {
                toggleSmallSearchPane(true);
            } else {
                // set search pane size according to a media query
                updateSmallSearchPane();
            }
        }

        /**
         * Handles tracking events originating from the mouse and simulates
         * touch-like scrolling on the page nodes.
         *
         * @param {jQuery.Event} event
         *  The jQuery tracking event.
         */
        var trackingHandler = (function () {

            var // the duration to collect move events, in milliseconds
                COLLECT_DURATION = 150,
                // distances of last move events in COLLECT_DURATION
                lastMoves = null;

            // removes all outdated entries from the lastMoves array
            function updateLastMoves(moveX, moveY) {

                var now = _.now();

                while ((lastMoves.length > 0) && (now - lastMoves[0].now > COLLECT_DURATION)) {
                    lastMoves.shift();
                }
                if (_.isNumber(moveX) && _.isNumber(moveY)) {
                    lastMoves.push({ now: now, x: moveX, y: moveY });
                }
                return now;
            }

            // starts the trailing animation
            function trailingAnimation() {

                var now = updateLastMoves(),
                    move = _.reduce(lastMoves, function (memo, entry) {
                        var factor = (now - entry.now) / COLLECT_DURATION;
                        memo.x -= entry.x * factor;
                        memo.y -= entry.y * factor;
                        return memo;
                    }, { x: 0, y: 0 });

                // reduce distance for initial animation frame
                move.x /= (COLLECT_DURATION / ANIMATION_DELAY / 2);
                move.y /= (COLLECT_DURATION / ANIMATION_DELAY / 2);

                // run the scroll animation
                startScrollAnimation(function () { move.x *= 0.85; move.y *= 0.85; return move; }, 20);
            }

            function trackingHandler(event) {

                switch (event.type) {
                case 'tracking:start':
                    lastMoves = [];
                    abortScrollAnimation();
                    break;
                case 'tracking:move':
                    scrollContentRootNode(-event.moveX, -event.moveY);
                    updateLastMoves(event.moveX, event.moveY);
                    break;
                case 'tracking:end':
                    trailingAnimation();
                    break;
                case 'tracking:cancel':
                    break;
                }
            }

            return trackingHandler;
        }()); // end of trackingHandler() local scope

        /**
         * Handles pinch events.
         *
         * @param {String} phase
         * The current pinch phase ('start', 'move', 'end' or 'cancel')
         *
         * @param {jQuery.Event} event
         *  The jQuery tracking event.
         *
         * @param {Number} distance
         * The current distance in px between the two fingers
         *
         * @param {Point} midPoint
         * The current center position between the two fingers
         *
         */
        var pinchHandler = (function() {

            var minScale, maxScale, scale,
                scrollTop, scrollLeft, scrollWidth, scrollHeight,
                startDistance,
                contentWidth, contentHeight,
                windowPosition,
                contentRootScaleOrigin = {
                    x : 0,
                    y : 0
                },
                // the initial left and top position of the pageContainer in relation to the contentRootNode
                initialPageContainer = {
                    left   : 0,
                    top    : 0,
                    width  : 0,
                    height : 0
                },
                newPageContainer = {
                    left   : 0,
                    top    : 0,
                    width  : 0,
                    height : 0
                },
                translateX, translateY;

            function pinchHandler(phase, event, distance, midPoint) {

                switch (phase) {
                    case 'start':
                        minScale = self.getMinZoomFactor() / zoomFactor;
                        maxScale = self.getMaxZoomFactor() / zoomFactor;
                        scrollLeft = contentRootNode.scrollLeft();
                        scrollTop = contentRootNode.scrollTop();
                        scrollWidth = contentRootNode[0].scrollWidth;
                        scrollHeight = contentRootNode[0].scrollHeight;
                        startDistance = distance;
                        initialPageContainer.width = pageContainerNode.width();
                        initialPageContainer.height = pageContainerNode.height();
                        contentWidth = contentRootNode.width();
                        contentHeight = contentRootNode.height();
                        windowPosition = Utils.getNodePositionInPage(contentRootNode);

                        // determining the topleft position of our page container (taking care of the scrollBar position)
                        if(scrollLeft>0) {
                            initialPageContainer.left = -scrollLeft;
                        }
                        else if(initialPageContainer.width<contentWidth) {
                            initialPageContainer.left = (contentWidth-initialPageContainer.width)/2;
                        }
                        else {
                            initialPageContainer.left = 0;
                        }
                        if(scrollTop>0) {
                            initialPageContainer.top = -scrollTop;
                        }
                        else if(initialPageContainer.height<contentHeight) {
                            initialPageContainer.top = (contentHeight-initialPageContainer.height)/2;
                        }
                        else {
                            initialPageContainer.top = 0;
                        }
                        /* falls through */
                    case 'move' : {

                        contentRootScaleOrigin.x = midPoint.x - windowPosition.left;
                        contentRootScaleOrigin.y = midPoint.y - windowPosition.top;


                        // setting the proper scale origin
                        Utils.setCssAttributeWithPrefixes(pageContainerNode, 'transform-origin', Math.round(contentRootScaleOrigin.x+scrollLeft) + 'px' + ' ' + Math.round(contentRootScaleOrigin.y+scrollTop) + 'px');

                        // setting the scale factor for the transformation
                        scale = Utils.minMax(distance / startDistance, minScale, maxScale);
                        var transform = 'scale(' + scale + ')';

                        if(initialPageContainer.width<contentWidth) {
                            contentRootScaleOrigin.x += (contentWidth - initialPageContainer.width)/2;
                        }
                        if(initialPageContainer.height<contentHeight) {
                            contentRootScaleOrigin.y += (contentHeight - initialPageContainer.height)/2;
                        }
                        newPageContainer.left = ((initialPageContainer.left - contentRootScaleOrigin.x)*scale)+contentRootScaleOrigin.x;
                        newPageContainer.top  = ((initialPageContainer.top - contentRootScaleOrigin.y)*scale)+contentRootScaleOrigin.y;

                        newPageContainer.width  = initialPageContainer.width*scale;
                        newPageContainer.height = initialPageContainer.height*scale;

                        translateX = 0;
                        translateY = 0;

                        // check left & right border, translation might be necessary
                        if(newPageContainer.left>0) {
                            translateX = -newPageContainer.left;                                // translation for page container to be starting at left border
                            if(newPageContainer.width<contentWidth) {
                                translateX += (contentWidth-newPageContainer.width)/2;          // centering if needed
                            }
                        }
                        else if((newPageContainer.left+newPageContainer.width)<contentWidth) {
                            translateX = contentWidth-(newPageContainer.left+newPageContainer.width);   // translation for page container to end at the right border
                            if(newPageContainer.width<contentWidth) {
                                translateX -= (contentWidth-newPageContainer.width)/2;          // centering if needed
                            }
                        }

                        // check top & bottom border, translation might be necessary
                        if(newPageContainer.top>0) {
                            translateY = -newPageContainer.top;                                 // translation for page container to be starting at top border
                            if(newPageContainer.height<contentHeight) {
                                translateY += (contentHeight-newPageContainer.height)/2;        // centering if needed
                            }
                        }
                        else if((newPageContainer.top+newPageContainer.height)<contentHeight) {
                            translateY = contentHeight-(newPageContainer.top+newPageContainer.height);  // translation for page container to end at bottom border
                            if(newPageContainer.height<contentHeight) {
                                translateY -= (contentHeight-newPageContainer.height)/2;        // centering
                            }
                        }
                        if(translateX!==0||translateY!==0) {
                            transform += ' translate(' + Math.round(translateX/scale) + 'px,' + Math.round(translateY/scale) + 'px)';
                        }
                        Utils.setCssAttributeWithPrefixes(pageContainerNode, 'transform', transform);
                        break;
                    }

                    case 'end': {
                        Utils.setCssAttributeWithPrefixes(pageContainerNode, 'transform', '');
                        Utils.setCssAttributeWithPrefixes(pageContainerNode, 'transform-origin', '');
                        if (scale !== 1) {
                            self.setZoomType(scale * zoomFactor);
                        }
                        app.getController().update();
                        var newScrollLeft = -(newPageContainer.left+translateX),
                            newScrollTop  = -(newPageContainer.top+translateY);
                        if(newScrollLeft<0) {
                            newScrollLeft = 0;
                        }
                        if(newScrollTop<0) {
                            newScrollTop = 0;
                        }
                        contentRootNode.scrollLeft(newScrollLeft);
                        contentRootNode.scrollTop(newScrollTop);
                        break;
                    }

                    case 'cancel': {
                        break;
                    }
                }
            }
            return pinchHandler;
        }()); // end of pinchHandler() local scope


        /**
         * Handles tap events.
         *
         * @param {jQuery.Event} event
         *  The jQuery tracking event.
         *
         * @param {Number} taps
         * The number of detected taps
         *
         */
        function tapHandler(event, taps) {
            if(taps===2) {
                self.setZoomType(self.getZoomType()==='page'?'width':'page');
                app.getController().update();
                if(self.getZoomType()==='page') {
                    // the zoom type was switched to page via double-tap, we will determine the touched page and scroll to the beginning
                    var tappedVertPagePosition = (event.data.tapPosY1-Utils.getNodePositionInPage(contentRootNode).top) + contentRootNode.scrollTop(),
                        page = _.sortedIndex(pageNodes, tappedVertPagePosition, function (value) {
                        if (_.isNumber(value)) { return value; }
                        var pagePosition = Utils.getChildNodePositionInNode(contentRootNode, value);
                        return pagePosition.top + pagePosition.height - 1;
                    }) + 1;
                    selectPage(page);
                    scrollToPage(page);
                }
            }
        }

        /**
         * Handles swipe events.
         *
         * @param {String} phase
         * The current swipe phase (swipeStrictMode is true, so we only get the 'end' phase)
         *
         * @param {jQuery.Event} event
         *  The jQuery tracking event.
         *
         * @param {Number} distance
         * The swipe distance in pixel, the sign determines the swipe direction (left to right or right to left)
         *
         */
        function horSwipeHandler(phase, event, distance) {
            // swiping only if no horizontal scrolling is possible
            if(pageContainerNode.width()<=contentRootNode.width()) {
                self.showPage(distance<0?'next':'previous');
                app.getController().update();
            }
        }

        /**
         * Handles scroll events and updates the visible pages.
         *
         * @param {jQuery.Event} event
         *  The jQuery scroll event.
         */
        var scrollHandler = app.createDebouncedMethod(cancelMorePagesTimer, loadVisiblePages, {
            delay: Modernizr.touch ? 200 : 100,
            maxDelay: Modernizr.touch ? 1000 : 500
        });

        /**
         * Initialization after construction.
         */
        function initHandler() {

            model = app.getModel();
            contentRootNode = self.getContentRootNode();
            searchPaneNode = app.getWindow().nodes.search;

            self.addPane(topPane = new TopPane(app));

            // create the side pane
            self.addPane(sidePane = new SidePane(app, {
                position: 'right',
                size: Modernizr.touch ? 152 : SidePane.DEFAULT_WIDTH,
                resizable: !Modernizr.touch,
                minSize: SidePane.DEFAULT_WIDTH,
                maxSize: PageGroup.getRequiredWidth(4) + Utils.SCROLLBAR_WIDTH
            }));

            // create the page preview group
            pageGroup = new PageGroup(app, sidePane);

            // initialize the side pane
            sidePane
                .addViewComponent(new Component(app)
                    .addGroup('pages/current', pageGroup)
                );

            // initially, hide the side pane, and show the overlay tool bars
            self.toggleSidePane(false);

            // Bug 25924: sometimes, Firefox selects the entire page
            if (_.browser.Firefox) {
                contentRootNode.on('focus', function () {
                    Utils.clearBrowserSelection();
                });
            }

            // listen to orientation events when the application window is visible
            if (Modernizr.touch) {
                app.registerGlobalEventHandler(window, 'orientationchange', orientationChangeHandler);
            }

            // refresh layout when the search pane has been toggled
            this.listenTo(app.getWindow(), 'search:open', searchOpenHandler);
            this.listenTo(app.getWindow(), 'search:close', searchCloseHandler);

            // enabling the text input field and setting the proper validator when page count is known after import
            this.listenTo(app.getImportPromise(), 'done', function () {
                topPane.updatePageCount(model.getPageCount());
            });

            // search pane is a focus landmark (TODO: move to core code)
            searchPaneNode.addClass('f6-target');
        }

        function initGuiHandler() {

            var // the number of pages in the document
                pageCount = model.getPageCount(),
                // the HTML mark-up for the empty page nodes
                pageMarkup = '';

            if (pageCount >= 1) {

                // save the scroll position while view is hidden
                app.getWindow().on({ beforehide: windowHideHandler, show: windowShowHandler });

                // no layout refreshes without any pages
                self.on('refresh:layout', function () { refreshLayout(); });

                // initialize touch-like tracking with mouse, attach the scroll event handler
                contentRootNode
                    .enableTracking({selector: '.page', sourceEvents: 'mouse'})
                    .enableTouch({selector: '.page', pinchHandler: pinchHandler, tapHandler: tapHandler, horSwipeHandler: horSwipeHandler})
                    .on('tracking:start tracking:move tracking:end tracking:cancel', trackingHandler)
                    .on('scroll', scrollHandler);

                // create mark-up for all page nodes
                _.times(pageCount, function (index) {
                    pageMarkup += '<div class="page" data-page="' + (index + 1) + '"></div>';
                });

                // insert the mark-up into the container node and initialize the
                // variable 'pageNodes' BEFORE inserting everything into the living
                // DOM (variable is used when refreshing the layout)
                pageContainerNode[0].innerHTML = pageMarkup;
                pageNodes = pageContainerNode.children();

                // insert page container node into the view
                self.insertContentNode(pageContainerNode);
            }

            // set focus to application pane, and update the view
            self.grabFocus();
            app.getController().update();
        }

        // methods ------------------------------------------------------------

        /**
         * Returns whether the main side pane is currently visible.
         *
         * @returns {Boolean}
         *  Whether the main side pane is currently visible.
         */
        this.isSidePaneVisible = function () {
            return sidePane.isVisible();
        };

        /**
         * Changes the visibility of the main side pane and the overlay tool
         * box. If the side pane is visible, the overlay tool box is hidden,
         * and vice versa.
         *
         * @param {Boolean} state
         *  Whether to show or hide the main side pane.
         *
         * @returns {PreviewView}
         *  A reference to this instance.
         */
        this.toggleSidePane = function (state) {
            this.lockPaneLayout(function () {
                sidePane.toggle(state);
            });
            return this;
        };

        /**
         * Returns the one-based index of the page currently shown.
         *
         * @returns {Number}
         *  The one-based index of the current page.
         */
        this.getPage = function () {
            return selectedPage;
        };

        /**
         * Shows the specified page of the current document.
         *
         * @param {Number|String} page
         *  The one-based index of the page to be shown; or one of the keywords
         *  'first', 'previous', 'next', or 'last'.
         *
         * @returns {PreviewView}
         *  A reference to this instance.
         */
        this.showPage = function (page) {

            // convert page keyword to page number
            switch (page) {
            case 'first':
                page = 1;
                break;
            case 'previous':
                page = this.getPage() - 1;
                break;
            case 'next':
                page = this.getPage() + 1;
                break;
            case 'last':
                page = model.getPageCount();
                break;
            }

            selectPage(page);
            scrollToPage(page);
            return this;
        };

        /**
         * Returns the current zoom type.
         *
         * @returns {Number|String}
         *  The current zoom type, either as fixed percentage, or as one of the
         *  keywords 'width' (page width is aligned to width of the visible
         *  area), or 'page' (page size is aligned to visible area).
         */
        this.getZoomType = function () {
            return zoomType;
        };

        /**
         * Returns the current effective zoom factor in percent.
         *
         * @returns {Number}
         *  The current zoom factor in percent.
         */
        this.getZoomFactor = function () {
            return zoomFactor;
        };

        /**
         * Returns the minimum zoom factor in percent.
         *
         * @returns {Number}
         *  The minimum zoom factor in percent.
         */
        this.getMinZoomFactor = function () {
            return ZOOM_FACTORS[0];
        };

        /**
         * Returns the maximum zoom factor in percent.
         *
         * @returns {Number}
         *  The maximum zoom factor in percent.
         */
        this.getMaxZoomFactor = function () {
            return _.last(ZOOM_FACTORS);
        };

        /**
         * Changes the current zoom settings.
         *
         * @param {Number|String} newZoomType
         *  The new zoom type. Either a fixed percentage, or one of the
         *  keywords 'width' (page width is aligned to width of the visible
         *  area), or 'page' (page size is aligned to visible area).
         *
         * @returns {PreviewView}
         *  A reference to this instance.
         */
        this.setZoomType = function (newZoomType) {
            if (zoomType !== newZoomType) {
                zoomType = newZoomType;
                refreshLayout();
            }
            return this;
        };

        /**
         * Switches to 'fixed' zoom mode, and decreases the current zoom level
         * by one according to the current effective zoom factor, and updates
         * the view.
         *
         * @returns {PreviewView}
         *  A reference to this instance.
         */
        this.decreaseZoomLevel = function () {

            var // find last entry in ZOOM_FACTORS with a factor less than current zoom
                prevZoomFactor = Utils.findLast(ZOOM_FACTORS, function (factor) { return factor < zoomFactor; });

            return this.setZoomType(_.isNumber(prevZoomFactor) ? prevZoomFactor : this.getMinZoomFactor());
        };

        /**
         * Switches to 'fixed' zoom mode, and increases the current zoom level
         * by one according to the current effective zoom factor, and updates
         * the view.
         *
         * @returns {PreviewView}
         *  A reference to this instance.
         */
        this.increaseZoomLevel = function () {

            var // find first entry in ZOOM_FACTORS with a factor greater than current zoom
                nextZoomFactor = _.find(ZOOM_FACTORS, function (factor) { return factor > zoomFactor; });

            return this.setZoomType(_.isNumber(nextZoomFactor) ? nextZoomFactor : this.getMaxZoomFactor());
        };

        // initialization -----------------------------------------------------

        // initialize the view after import
        app.onImportSuccess(function () {

            var // file type configuration (by file extension)
                extSettings = ExtensionRegistry.getExtensionSettings(app.getFullFileName());

            // determine zoom type by application type (full width for text documents)
            zoomType = 'page';
            if (extSettings) {
                switch (extSettings.type) {
                case 'text':
                    zoomType = 'width';
                    break;
                case 'spreadsheet':
                case 'presentation':
                    zoomType = 'page';
                    break;
                }
            }

            // first set zoom, then the first page (to keep it at the top screen border)
            refreshLayout();
            self.showPage(1);
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            pageLoader.destroy();
            app = self = model = null;
            contentRootNode = searchPaneNode = sidePane = null;
            pageGroup = pageContainerNode = pageNodes = pageLoader = loadedPageNodes = null;
            loadMorePagesTimer = scrollAnimation = null;
        });

    } // class PreviewView

    // exports ================================================================

    // derive this class from class BaseView
    return BaseView.extend({ constructor: PreviewView });

});
