/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Kai Ahrens <kai.ahrens@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/preview/app/application',
    ['io.ox/core/collection',
     'io.ox/core/api/folder',
     'io.ox/office/tk/utils',
     'io.ox/office/baseframework/app/baseapplication',
     'io.ox/office/baseframework/app/extensionregistry',
     'io.ox/office/preview/model/model',
     'io.ox/office/preview/view/view',
     'io.ox/office/preview/app/controller',
     'gettext!io.ox/office/preview'
    ], function (Collection, FolderAPI, Utils, BaseApplication, ExtensionRegistry, PreviewModel, PreviewView, PreviewController, gt) {

    'use strict';

    var // the module name of this application
        MODULE_NAME = 'io.ox/office/preview';

    // class PreviewApplication ===============================================

    /**
     * The OX Preview application used to view a wide range of document types.
     *
     * @constructor
     *
     * @extends BaseApplication
     *
     * @param {Object} launchOptions
     *  All options passed to the core launcher (the ox.launch() method).
     *
     * @param {Object} [appOptions]
     *  Static application options that have been passed to the static method
     *  BaseApplication.createLauncher().
     */
    var PreviewApplication = BaseApplication.extend({ constructor: function (launchOptions, appOptions) {

        var // self reference
            self = this,

            // the unique job identifier to be used for page requests
            jobId = null,

            // user may edit the document in an OX Documents edit application
            isEditable = false;

        // base constructor ---------------------------------------------------

        BaseApplication.call(this, PreviewModel, PreviewView, PreviewController, importDocument, launchOptions, appOptions);

        // private methods ----------------------------------------------------

        /**
         * Loads the first page of the document described in the current file
         * descriptor.
         *
         * @param {Object} [point]
         *  The save point, if called from fail-restore.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  initial data of the preview document has been loaded; or rejected
         *  when an error has occurred.
         */
        function importDocument() {

            var // promise of the initial server request
                convertPromise = null,
                // promise of a Deferred that returns the collection properties
                collectionPromise = null,
                // promise of a Deferred that returns the folder properties
                folderPromise = null,
                // the file descriptor
                file = self.getFileDescriptor();

            // disable drop events
            self.getWindowNode().on('drop dragstart dragover', false);

            // show the busy blocker element
            self.getView().enterBusy({ showFileName: true, cancelHandler: function () { self.quit(); } });

            // open the file on the server
            convertPromise = self.sendConverterRequest({
                params: {
                    action: 'convertdocument',
                    convert_format: 'html',
                    convert_action: 'beginconvert'
                }
            });

            // check edit rights of user, check whether file is located in the trash folder
            if ('source' in file) {
                // TODO: file is an attachment, e.g. of a mail or task, editable?
                collectionPromise = null;
                folderPromise = null;
            } else {
                collectionPromise = new Collection(file).getProperties();
                folderPromise = FolderAPI.get({ folder: file.folder_id });
            }

            // wait for all promises and check the response objects
            return $.when(convertPromise, collectionPromise, folderPromise).then(function (convertData, collectionData, folderData) {

                // check that no error has occurred while opening the document
                if (_.isString(convertData.JobID) && (convertData.JobID.length > 0) && _.isNumber(convertData.PageCount) && (convertData.PageCount > 0)) {

                    // store whether the document can really be edited
                    isEditable = _.isObject(collectionData) && collectionData.one && collectionData.read && collectionData.modify && _.isObject(folderData) && !FolderAPI.is('trash', folderData);

                    // further initialization after import
                    jobId = convertData.JobID;
                    self.getModel().setPageCount(convertData.PageCount);

                    // resolve the promise with the response data
                    return convertData;
                }

                // special text for password protected files
                if (Utils.getStringOption(convertData, 'cause') === 'passwordProtected') {
                    convertData.message = gt('The document is protected by a password.');
                }

                // convert result to a rejected Deferred object
                return $.Deferred().reject(convertData);
            });
        }

        /**
         * Sends a close notification to the server, before the application
         * will be really closed.
         */
        function quitHandler() {
            if (!self.isImportFailed()) {
                return self.sendPreviewRequest({ params: { convert_action: 'endconvert' } });
            }
        }

        // methods ------------------------------------------------------------

        this.sendPreviewRequest = function (options) {
            return _.isString(jobId) ? this.sendConverterRequest(Utils.extendOptions({
                params: { action: 'convertdocument', job_id: jobId }
            }, options)) : $.Deferred().reject();
        };

        this.getPreviewModuleUrl = function (options) {
            return _.isString(jobId) ? this.getConverterModuleUrl(Utils.extendOptions({
                action: 'convertdocument',
                job_id: jobId
            }, options)) : undefined;
        };

        /**
         * Returns whether the current document can be edited with one of the
         * OX Document edit applications.
         */
        this.isDocumentEditable = function () {
            return isEditable && _.isString(jobId) && ExtensionRegistry.isEditable(this.getFullFileName());
        };

        /**
         * Launches the appropriate OX Document edit applications for the
         * current document.
         */
        this.editDocument = function () {

            var // the file descriptor of the current document
                file = this.getFileDescriptor(),
                // the configuration of the file extension of the current document
                extensionSettings = this.isDocumentEditable() ? ExtensionRegistry.getExtensionSettings(this.getFullFileName()) : null,
                // the edit application module identifier
                editModule = Utils.getStringOption(extensionSettings, 'module', '');

            // TODO: edit mail attachments and task attachments
            if (editModule.length > 0) {

                // launch the correct edit application
                if (ExtensionRegistry.isNative(this.getFullFileName())) {
                    ox.launch(editModule + '/main', { action: 'load', file: file });
                } else if (ExtensionRegistry.isConvertible(this.getFullFileName())) {
                    ox.launch(editModule + '/main', { action: 'convert', folderId: file.folder_id, templateFile: file, preserveFileName: true });
                } else {
                    Utils.error('PreviewApplication.editDocument(): unknown document type');
                    return;
                }

                // close this application
                _.defer(function () { self.quit(); });
            }
        };

        // initialization -----------------------------------------------------

        this.registerQuitHandler(quitHandler);

        // destroy all class members
        this.on('docs:destroy', function () {
            launchOptions = appOptions = null;
        });

    }}); // class PreviewApplication

    // static methods ---------------------------------------------------------

    /**
     * Replacement for the generic method BaseApplication.createLauncher()
     * without parameters, to launch viewer applications.
     */
    PreviewApplication.createLauncher = function () {
        return BaseApplication.createLauncher(MODULE_NAME, PreviewApplication, { chromeless: true });
    };

    // exports ================================================================

    return PreviewApplication;

});
