/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/toppane',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/forms',
     'io.ox/office/baseframework/view/toolpane',
     'io.ox/office/baseframework/view/toolbar',
     'io.ox/office/editframework/view/editcontrols',
     'io.ox/office/tk/keycodes',
     'gettext!io.ox/office/editframework'
    ], function (Utils, Forms, ToolPane, ToolBar, EditControls, KeyCodes, gt) {

    'use strict';

    var // class name shortcuts
        Button = EditControls.Button,
        RadioGroup = EditControls.RadioGroup,
        RadioList = EditControls.RadioList;

    // class ToolBarTabGroup ==================================================

    /**
     * @constructor
     *
     * @extends RadioGroup
     */
    var ToolBarTabGroup = RadioGroup.extend({ constructor: function (app, toolBarTabs) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        RadioGroup.call(this, { classes: 'tab-group', role: 'tablist' });

        // private methods ----------------------------------------------------

        function tabCreateHandler(event, tabId, options) {
            self.createOptionButton(tabId, Utils.extendOptions(options, { attributes: { role: 'tab' } }));
        }

        function tabLabelHandler(event, tabId, label) {
            Forms.setCaptionText(self.findOptionButtons(tabId), label);
        }

        function keyDownHandler(event) {

            var startNode = $(event.target),
                focusableControls = Forms.findFocusableNodes(self.getNode()),
                newSelectedNode = null;

            if (KeyCodes.matchKeyCode(event, 'LEFT_ARROW')) {
                newSelectedNode = Utils.moveNodeFocus(startNode, focusableControls, 'left');
            } else if (KeyCodes.matchKeyCode(event, 'RIGHT_ARROW')) {
                newSelectedNode = Utils.moveNodeFocus(startNode, focusableControls, 'right');
            }

            if (newSelectedNode) {
                self.triggerChange(Forms.getButtonValue(newSelectedNode), { preserveFocus: true });
                event.preventDefault();
            }
        }

        // initialization -----------------------------------------------------

        // update control according to the tab collection
        toolBarTabs.on({
            // create button element for a new tab
            'tab:create': tabCreateHandler,
            'tab:label': tabLabelHandler,
            // show/hide tab buttons
            'tab:show': function (event, tabId) { Forms.showNodes(self.findOptionButtons(tabId), true); },
            'tab:hide': function (event, tabId) { Forms.showNodes(self.findOptionButtons(tabId), false); },
            // update selected button
            'tab:activate': function (event, tabId) { self.setValue(tabId); }
        });

        // add own keyboard handling
        this.getNode().on('keydown', keyDownHandler);

    }}); // class ToolBarTabGroup

    // class ToolBarTabGroup ==================================================

    /**
     * @constructor
     *
     * @extends RadioList
     */
    var ToolBarTabList = RadioList.extend({ constructor: function (app, toolBarTabs) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        RadioList.call(this, {
            label: /*#. default title for the drop-down selector for tool bars */ gt('Toolbars'),
            tooltip: /*#. default tool tip for the drop-down selector for tool bars */ gt('Switch toolbars'),
            width: 150,
            highlight: true,
            smallerVersion: {
                css: {
                    width: 20
                },
                icon: 'fa fa-bars',
                hideLabel: true
            }
        });

        // private methods ----------------------------------------------------

        function tabCreateHandler(event, tabId, options) {
            self.createOptionButton(tabId, options);
        }

        function tabLabelHandler(event, tabId, label) {
            Forms.setCaptionText(self.getMenu().findItemNodes(tabId), label);
        }

        // initialization -----------------------------------------------------

        // update control according to the tab collection
        toolBarTabs.on({
            // create button element for a new tab
            'tab:create': tabCreateHandler,
            'tab:label': tabLabelHandler,
            // show/hide tab buttons
            'tab:show': function (event, tabId) { Forms.showNodes(self.getMenu().findItemNodes(tabId), true); },
            'tab:hide': function (event, tabId) { Forms.showNodes(self.getMenu().findItemNodes(tabId), false); },
            // update selected button
            'tab:activate': function (event, tabId) { self.setValue(tabId); }
        });

    }}); // class ToolBarTabGroup

    // class TopPane ==========================================================

    /**
     * Represents the top-level view pane in OX Document applications,
     * containing the file name, the file drop-down menu, and additional global
     * controls such as the Close button and the Undo/Redo buttons.
     *
     * @constructor
     *
     * @extends ToolPane
     *
     * @param {EditApplication} app
     *  The application containing this view pane.
     *
     * @param {ToolBarTabCollection} toolBarTabs
     *  The collection of all registered tool bar tabs.
     */
    function TopPane(app, toolBarTabs) {

        var // self reference
            self = this,

            // the tab buttons to activate the different edit tool bars
            tabGroup = null,

            // the tab buttons to activate the different edit tool bars, as drop-down list
            tabList = null,

            // the application status label
            statusLabel = null,

            // the 'View' menu group
            viewMenuGroup = null,

            // the containing nodes (lead, center, trail) of the TopPane (incl. content-nodes)
            leadWrapperNode = null,
            leadAreaNode = null,
            centerWrapperNode = null,
            centerAreaNode = null,
            trailWrapperNode = null,
            trailAreaNode = null,

            // shrink controls in multiple states
            shrinkState = 0;


        // base constructor ---------------------------------------------------

        ToolPane.call(this, app, { position: 'top', classes: 'top-pane' });

        // private methods ----------------------------------------------------

        function paneLayoutHandler() {

            var // get all sizes of the containing divs in top-pane
                areaSizes           = self.getAreaSizes(),
                // total width of pane
                paneWidth           = areaSizes.total,

                // lead, center and trailing width of the wrapped nodes
                leadWrapperWidth    = leadWrapperNode.width(),
                centerWrapperWidth  = centerWrapperNode.width(),
                trailWrapperWidth   = trailWrapperNode.width(),

                // lead, center and trailing width of the included content
                leadWidth           = leadAreaNode.outerWidth(),
                centerWidth         = centerAreaNode.outerWidth(),
                trailWidth          = trailAreaNode.outerWidth();


            // set the new widths of the nodes (wrapper + content nodes)
            function widthUpdate(){
                leadWrapperWidth    = leadWrapperNode.width();
                centerWrapperWidth  = centerWrapperNode.width();
                trailWrapperWidth   = trailWrapperNode.width();

                leadWidth           = leadAreaNode.outerWidth();
                centerWidth         = centerAreaNode.outerWidth();
                trailWidth          = trailAreaNode.outerWidth();
            }

            // get total needed width of all content-nodes
            function getTotalNeededWidth(){
                return (leadWidth + centerWidth + trailWidth);
            }

            // shrink controls in specific steps (self prio)
            function shrink(){
                switch (shrinkState) {

                    case 0:                 // on shrink state 0
                        // make the status-label smaller
                        statusLabel.activateSmallVersion();
                        shrinkState = 1;    // set new shrink state
                        break;

                    case 1:                 // on shrink state 1
                        // show the DropDown instead of the Tabs
                        tabGroup.hide();
                        tabList.show();
                        shrinkState = 2;    // set new shrink state
                        break;

                    case 2:                 // on shrink state 2
                        // make the tab-DropDown smaller
                        tabList.activateSmallVersion();
                        shrinkState = 3;    // set new shrink state
                        break;
                }
            }

            // unshrink controls in opposite direction as shrink does
            function unshrink(){
                switch (shrinkState) {

                    case 1:                 // on shrink state 1
                        // get the old styles of the status-label
                        statusLabel.deactivateSmallVersion();
                        shrinkState = 0;    // set new shrink state
                        break;

                    case 2:                 // on shrink state 2
                        // show the Tabs instead of the DropDown
                        tabGroup.show();
                        tabList.hide();
                        shrinkState = 1;    // set new shrink state
                        break;

                    case 3:                 // on shrink state 3
                        // show the normal version of the tab-DropDown
                        tabList.deactivateSmallVersion();
                        shrinkState = 2;    // set new shrink state
                        break;
                }
            }


            // NOT enough place
            if (getTotalNeededWidth() > paneWidth) {
                // reduce the place between the controls
                this.getNode().addClass('small-distance');
                // recalculate the widths
                widthUpdate();

                // if still not enough place
                if (getTotalNeededWidth() > paneWidth) {
                    // beginn to shrink the controls
                    // as long as possible, or it fits in
                    shrink();
                }

            // FREE place available
            } else {
                // unshrink the current state
                unshrink();
                // recalculate the widths
                widthUpdate();

                // if we now don't fit in the pane
                if (getTotalNeededWidth() > paneWidth) {
                    // re-shrink the old state
                    shrink();

                // if we have still enough place
                } else {
                    // place the default padding between the controls
                    this.getNode().removeClass('small-distance');
                    // recalculate the widths
                    widthUpdate();

                    // if we now don't fit in the pane
                    if (getTotalNeededWidth() > paneWidth) {
                        // re-add the smaller padding
                        this.getNode().addClass('small-distance');
                    }
                }
            }

        }

        // public methods -----------------------------------------------------

        /**
         * Returns the 'View' drop-down menu group containing controls for view
         * settings of the application.
         *
         * @returns {ComponentMenuButton}
         *  The 'View' drop-down menu group.
         */
        this.getViewMenuGroup = function () {
            return viewMenuGroup;
        };

        // initialization -----------------------------------------------------

        // create the tab button/list controls
        tabGroup = new ToolBarTabGroup(app, toolBarTabs);
        tabList = new ToolBarTabList(app, toolBarTabs);

        // create the tool bar containing the tab buttons for the edit tool pane
        this.addViewComponent(new ToolBar(app)
            .addGroup('view/toolbars/tab', tabGroup)
            .addGroup('view/toolbars/tab', tabList.hide()),
        { targetArea: 'leading' });

        // create the tool bar containing the application status label
        statusLabel = new EditControls.ApplicationStatusLabel(app, {
            smallerVersion: {
                css: {
                    width: 20
                },
                hideLabel: true
            }
        });
        this.addViewComponent(new ToolBar(app).addGroup(null, statusLabel));

        // create the 'View' drop-down menu
        viewMenuGroup = new EditControls.ComponentMenuButton(app, { label: EditControls.VIEW_LABEL, anchorPadding: -3 });

        // create the tool bar with the standard controls
        this.addViewComponent(new ToolBar(app)
            .addGroup('document/undo', new Button({ icon: 'docs-undo', tooltip: gt('Revert last operation') }), { visibleKey: 'document/editable' })
            .addGroup('document/redo', new Button({ icon: 'docs-redo', tooltip: gt('Restore last operation') }), { visibleKey: 'document/editable' })
            .addGap()
            .addGroup('view/searchpane', new Button({ icon: 'fa-search', tooltip: gt('Toggle search'), toggle: true }), { visibleKey: 'app/valid' })
            .addGap()
            .addGroup(null, viewMenuGroup)
            .addSeparator()
            .addGroup('app/quit', new Button(EditControls.QUIT_BUTTON_OPTIONS)),
        { targetArea: 'trailing' });

        // update view pane after it has been resized, or contents have been changed
        this.on('pane:layout', paneLayoutHandler);

        // show tab buttons, hide drop-down list
        tabGroup.show();
        tabList.hide();

        leadWrapperNode = this.getAreaWrapperNode('leading');
        leadAreaNode = leadWrapperNode.find('>.area-container');

        centerWrapperNode = this.getAreaWrapperNode('center');
        centerAreaNode = centerWrapperNode.find('>.area-container');

        trailWrapperNode = this.getAreaWrapperNode('trailing');
        trailAreaNode = trailWrapperNode.find('>.area-container');

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = toolBarTabs = self = tabGroup = tabList = statusLabel = viewMenuGroup = null;
        });

    } // class TopPane

    // exports ================================================================

    // derive this class from class ToolPane
    return ToolPane.extend({ constructor: TopPane });

});
