/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/editcontrols',
    ['io.ox/contacts/api',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/forms',
     'io.ox/office/tk/io',
     'io.ox/office/tk/popup/basemenu',
     'io.ox/office/tk/popup/tooltip',
     'io.ox/office/tk/control/menumixin',
     'io.ox/office/baseframework/view/basecontrols',
     'io.ox/office/editframework/utils/color',
     'io.ox/office/editframework/utils/border',
     'io.ox/office/editframework/utils/canvaswrapper',
     'io.ox/office/drawinglayer/view/drawingcontrols',
     'gettext!io.ox/office/editframework'
    ], function (ContactsAPI, Utils, KeyCodes, Forms, IO, BaseMenu, Tooltip, MenuMixin, BaseControls, Color, Border, CanvasWrapper, DrawingControls, gt) {

    'use strict';

    var // class name shortcuts
        Button = BaseControls.Button,
        Label = BaseControls.Label,
        RadioList = BaseControls.RadioList,
        TextField = BaseControls.TextField,
        ComboField = BaseControls.ComboField,
        Picture = BaseControls.Picture,

        // predefined color definitions
        BUILTIN_COLOR_DEFINITIONS = [
            { name: 'dark-red',    label: gt('Dark red'),    color: { type: 'rgb', value: 'C00000' } },
            { name: 'red',         label: gt('Red'),         color: { type: 'rgb', value: 'FF0000' } },
            { name: 'orange',      label: gt('Orange'),      color: { type: 'rgb', value: 'FFC000' } },
            { name: 'yellow',      label: gt('Yellow'),      color: { type: 'rgb', value: 'FFFF00' } },
            { name: 'light-green', label: gt('Light green'), color: { type: 'rgb', value: '92D050' } },
            { name: 'green',       label: gt('Green'),       color: { type: 'rgb', value: '00B050' } },
            { name: 'light-blue',  label: gt('Light blue'),  color: { type: 'rgb', value: '00B0F0' } },
            { name: 'blue',        label: gt('Blue'),        color: { type: 'rgb', value: '0070C0' } },
            { name: 'dark-blue',   label: gt('Dark blue'),   color: { type: 'rgb', value: '002060' } },
            { name: 'purple',      label: gt('Purple'),      color: { type: 'rgb', value: '7030A0' } }
        ],

        // definitions for the theme color table (in order as shown in the GUI color picker)
        // transformation table: 0 = pure color; negative values = 'shade'; positive values = 'tint'
        SCHEME_COLOR_DEFINITIONS = [
            { name: 'background1', label: getBackgroundColorName(1), transformations: [    0, -242, -217, -191, -166, -128 ] },
            { name: 'text1',       label: getTextColorName(1),       transformations: [  128,  166,  191,  217,  242,    0 ] },
            { name: 'background2', label: getBackgroundColorName(2), transformations: [    0, -230, -191, -128,  -64,  -26 ] },
            { name: 'text2',       label: getTextColorName(2),       transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent1',     label: getAccentColorName(1),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent2',     label: getAccentColorName(2),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent3',     label: getAccentColorName(3),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent4',     label: getAccentColorName(4),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent5',     label: getAccentColorName(5),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent6',     label: getAccentColorName(6),     transformations: [   51,  102,  153,    0, -191, -128 ] }
        ],

        LOCALE_DEFINITIONS = [
            { locale: 'en-US', label: gt('English (US)') },
            { locale: 'de-DE', label: gt('German') },
            { locale: 'fr-FR', label: gt('French') },
            { locale: 'es-ES', label: gt('Spanish') },
            { locale: 'cs-CZ', label: gt('Czech') },
            { locale: 'da-DK', label: gt('Danish') },
            { locale: 'nl-NL', label: gt('Dutch (Netherlands)') },
            { locale: 'fi-FI', label: gt('Finnish') },
            { locale: 'el-GR', label: gt('Greek') },
            { locale: 'hu-HU', label: gt('Hungarian') },
            { locale: 'it-IT', label: gt('Italian (Italy)') },
            { locale: 'pl-PL', label: gt('Polish') },
            { locale: 'pt-PT', label: gt('Portuguese (Portugal)') },
            { locale: 'ro-RO', label: gt('Romanian') },
            { locale: 'ru-RU', label: gt('Russian') },
            { locale: 'sv-SE', label: gt('Swedish (Sweden)') }
        ],

        // all available entries for the border picker control
        BORDER_ENTRIES = [

            { name: 'left',    value: 'left',    icon: 'docs-border-l', label: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'Left border') },
            { name: 'right',   value: 'right',   icon: 'docs-border-r', label: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'Right border') },
            { name: 'top',     value: 'top',     icon: 'docs-border-t', label: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'Top border') },
            { name: 'bottom',  value: 'bottom',  icon: 'docs-border-b', label: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'Bottom border') },
            { name: 'insidev', value: 'insidev', icon: 'docs-border-v', label: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'Inner vertical borders') },
            { name: 'insideh', value: 'insideh', icon: 'docs-border-h', label: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'Inner horizontal borders') },

            { name: 'none',               value: { left: false, right: false, insidev: false, top: false, bottom: false, insideh: false }, icon: 'docs-border',        label: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'No borders') },
            { name: 'outer',              value: { left: true,  right: true,  insidev: false, top: true,  bottom: true,  insideh: false }, icon: 'docs-border-tblr',   label: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'Outer borders') },
            { name: 'inner',              value: { left: false, right: false, insidev: true,  top: false, bottom: false, insideh: true },  icon: 'docs-border-hv',     label: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'Inner borders') },
            { name: 'all',                value: { left: true,  right: true,  insidev: true,  top: true,  bottom: true,  insideh: true },  icon: 'docs-border-tbhlrv', label: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'All borders') },
            { name: 'left-right',         value: { left: true,  right: true,  insidev: false, top: false, bottom: false, insideh: false }, icon: 'docs-border-lr',     label: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'Left and right borders') },
            { name: 'left-right-insidev', value: { left: true,  right: true,  insidev: true,  top: false, bottom: false, insideh: false }, icon: 'docs-border-lrv',    label: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'All vertical borders') },
            { name: 'outer-insidev',      value: { left: true,  right: true,  insidev: true,  top: true,  bottom: true,  insideh: false }, icon: 'docs-border-tblrv',  label: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'Inner vertical and outer borders') },
            { name: 'top-bottom',         value: { left: false, right: false, insidev: false, top: true,  bottom: true,  insideh: false }, icon: 'docs-border-tb',     label: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'Top and bottom borders') },
            { name: 'top-bottom-insideh', value: { left: false, right: false, insidev: false, top: true,  bottom: true,  insideh: true },  icon: 'docs-border-tbh',    label: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'All horizontal borders') },
            { name: 'outer-insideh',      value: { left: true,  right: true,  insidev: false, top: true,  bottom: true,  insideh: true },  icon: 'docs-border-tbhlr',  label: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'Inner horizontal and outer borders') }
        ],

        // all available border line styles
        BORDER_LINE_STYLES = [
            { value: 'single', label: /*#. Border line style: a solid single line */ gt.pgettext('borders', 'Single'),  icon: { style: 'solid', width: 1 } },
            { value: 'double', label: /*#. Border line style: a solid double line */ gt.pgettext('borders', 'Double'),  icon: { style: 'solid', width: 1, count: 2 } },
            { value: 'dotted', label: /*#. Border line style: a dotted single line */ gt.pgettext('borders', 'Dotted'), icon: { style: 'dotted', width: 1 } },
            { value: 'dashed', label: /*#. Border line style: a dashed single line */ gt.pgettext('borders', 'Dashed'), icon: { style: 'dashed', width: 1 } }
        ],

        ACCESS_RIGHTS_DEFINITIONS = [
            { mail: 'user1@example.com', name: 'User 1' },
            { mail: 'user2@example.com', name: 'User 2' },
            { mail: 'user3@example.com', name: 'User 3' },
            { mail: 'user4@example.com', name: 'User 4' }
        ];

    // private global functions ===============================================

    /**
     * Returns the localized name of a text color in a color scheme.
     */
    function getTextColorName(index) {
        var label =
            //#. The name of a text color in a color scheme (a color scheme consists
            //#. of two text colors, two background colors, and six accent colors).
            //#. Example result: "Text 1", "Text 2"
            //#. %1$d is the index of the text color
            //#, c-format
            gt('Text %1$d', _.noI18n(index));
        return label;
    }

    /**
     * Returns the localized name of a background color in a color scheme.
     */
    function getBackgroundColorName(index) {
        var label =
            //#. The name of a background color in a color scheme (a color scheme consists
            //#. of two text colors, two background colors, and six accent colors).
            //#. Example result: "Background 1", "Background 2"
            //#. %1$d is the index of the background color
            //#, c-format
            gt('Background %1$d', _.noI18n(index));
        return label;
    }

    /**
     * Returns the localized name of an accented color in a color scheme.
     */
    function getAccentColorName(index) {
        var label =
            //#. The name of an accent color in a color scheme (a color scheme consists
            //#. of two text colors, two background colors, and six accent colors).
            //#. Example result: "Accent 1", "Accent 2"
            //#. %1$d is the index of the accent color
            //#, c-format
            gt('Accent %1$d', _.noI18n(index));
        return label;
    }

    /**
     * Creates a bitmap for the specified border line style, and returns its
     * data URL.
     *
     * @param {Number} width
     *  The width of the generated bitmap, in pixels.
     *
     * @param {Number} height
     *  The height of the generated bitmap, in pixels.
     *
     * @param {Object} borderStyle
     *  The style descriptor of a border line, with the following properties:
     *  - {String} borderStyle.style
     *      The effective line style: one of 'solid', 'dashed', 'dotted',
     *      'dashDot', or 'dashDotDot'.
     *  - {Number} borderStyle.width
     *      Width of a (single) line in the border style, in pixels. The value
     *      0 will be interpreted as hair line.
     *  - {String} borderStyle.color
     *      The line color, as CSS color value.
     *  - {Number} [borderStyle.count=1]
     *      The number of parallel lines shown in the border.
     *
     * @returns {String}
     *  The data URL of the generated bitmap.
     */
    var getBorderStyleBitmapUrl = (function () {

        var // the canvas element used to generate the bitmaps
            canvasWrapper = new CanvasWrapper(),

            // maps unique bitmap keys values to the data URLs
            bitmapUrls = {};

        // the getBorderStyleBitmapUrl() method to be returned from the local scope
        function getBorderStyleBitmapUrl(width, height, borderStyle) {

            var // number of lines
                lineCount = borderStyle.count || 1,
                // unique key of the generated bitmap
                bitmapKey = width + ',' + height + ',' + borderStyle.style + ',' + borderStyle.width + ',' + borderStyle.color + ',' + lineCount;

            // return data URL of a bitmap already created
            if (bitmapKey in bitmapUrls) {
                return bitmapUrls[bitmapKey];
            }

            // initialize the canvas with the passed bitmap size (implicitly clears the canvas)
            canvasWrapper.initialize({ top: 0, left: 0, width: width, height: height });

            // create the bitmap
            canvasWrapper.render(function (context) {

                var // effective line width
                    lineWidth = Math.max(1, borderStyle.width),
                    // dash pattern
                    pattern = Border.getBorderPattern(borderStyle.style, lineWidth),
                    // Y offset of the first line
                    y = Math.floor(height / 2) - lineWidth * (lineCount - 1) - (lineWidth % 2) / 2;

                context.lineWidth = lineWidth;
                context.strokeStyle = borderStyle.color;
                if (borderStyle.width < 1) { context.globalAlpha = 0.4; }
                context.beginPath();
                _.times(lineCount, function () {
                    context.addLinePath(0, y, width, y, pattern);
                    y += 2 * lineWidth;
                });
                context.stroke();
            });

            // convert the bitmap to a data URL
            return (bitmapUrls[bitmapKey] = canvasWrapper.getDataURL());
        }

        return getBorderStyleBitmapUrl;
    }());

    // static class EditControls ==============================================

    /**
     * Additional classes defining specialized GUI controls for the OX editor
     * applications.
     *
     * @extends BaseControls
     * @extends DrawingControls
     */
    var EditControls = _.extend({}, BaseControls, DrawingControls);

    // constants for header labels --------------------------------------------

    /**
     * @constant
     */
    EditControls.FILE_HEADER_LABEL =
        //#. menu title: file settings and actions (download, rename, etc.)
        gt.pgettext('menu-title', 'File');

    /**
     * @constant
     */
    EditControls.FORMAT_HEADER_LABEL =
        //#. menu title: font settings, colors, text alignment, etc.
        gt.pgettext('menu-title', 'Format');

    /**
     * @constant
     */
    EditControls.INSERT_HEADER_LABEL =
        //#. menu title: insert images, charts, hyperlinks, etc.
        gt.pgettext('menu-title', 'Insert');

    /**
     * @constant
     */
    EditControls.FONT_HEADER_LABEL =
        //#. menu title: font settings (font name, size, text color, bold/italic, etc.)
        gt.pgettext('menu-title', 'Font');

    /**
     * @constant
     */
    EditControls.FILL_HEADER_LABEL =
        //#. menu title: background settings for table cells
        gt.pgettext('menu-title', 'Fill');

    /**
     * @constant
     */
    EditControls.BORDERS_HEADER_LABEL =
        //#. menu title: border settings for table cells
        gt.pgettext('menu-title', 'Borders');

    /**
     * @constant
     */
    EditControls.ALIGNMENT_HEADER_LABEL =
        //#. menu title: alignment settings for text in paragraphs or table cells
        gt.pgettext('menu-title', 'Alignment');

    /**
     * @constant
     */
    EditControls.NUMBERFORMAT_HEADER_LABEL =
        //#. menu title: number format settings for table cells
        gt.pgettext('menu-title', 'Number format');

    /**
     * @constant
     */
    EditControls.STYLES_HEADER_LABEL =
        //#. menu title: style settings (predefined formatting, e.g. "Heading", "Title", etc.) for paragraphs/table cells
        gt.pgettext('menu-title', 'Styles');

    /**
     * @constant
     */
    EditControls.TEXT_HEADER_LABEL =
        //#. menu title: generic settings for text
        gt.pgettext('menu-title', 'Text');

    /**
     * @constant
     */
    EditControls.HYPERLINKS_HEADER_LABEL =
        //#. menu title: settings for hyperlinks
        gt.pgettext('menu-title', 'Hyperlinks');

    /**
     * @constant
     */
    EditControls.TABLE_HEADER_LABEL =
        //#. menu title: settings for a single text table
        gt.pgettext('menu-title', 'Table');

    /**
     * @constant
     */
    EditControls.TABLES_HEADER_LABEL =
        //#. menu title: settings for text tables
        gt.pgettext('menu-title', 'Tables');

    /**
     * @constant
     */
    EditControls.REVIEW_HEADER_LABEL =
        //#. menu title: settings for document review (spelling, track/accept/reject changes)
        gt.pgettext('menu-title', 'Review');

    /**
     * @constant
     */
    EditControls.SPELLING_HEADER_LABEL =
        //#. menu title: settings for text spelling
        gt.pgettext('menu-title', 'Spelling');

    /**
     * @constant
     */
    EditControls.FORMULAS_HEADER_LABEL =
        //#. menu title: settings for formulas in table cells
        gt.pgettext('menu-title', 'Formulas');

    /**
     * @constant
     */
    EditControls.PARAGRAPH_STYLES_LABEL =
        //#. menu title: paragraph style sets (including sets of font-size, text color, bold/italic, etc.)
        gt.pgettext('menu-title', 'Paragraph styles');

    /**
     * @constant
     */
    EditControls.LIST_SETTINGS_LABEL =
        //#. menu title: list settings (bullet list, numbered list, list-icon, etc.)
        gt.pgettext('menu-title', 'List settings');

    /**
     * @constant
     */
    EditControls.FONT_STYLES_LABEL =
        //#. menu title: font styles (font name, size, text color, bold/italic, etc.)
        gt.pgettext('menu-title', 'Font styles');

    /**
     * @constant
     */
    EditControls.FONT_COLOR_LABEL =
        //#. menu title: font color
        gt.pgettext('menu-title', 'Font color');

    /**
     * @constant
     */
    EditControls.CELL_BORDER_LABEL =
        //#. menu title: cell border settings for table cells
        gt.pgettext('menu-title', 'Cell border');

    /**
     * @constant
     */
    EditControls.CELL_STYLES_LABEL =
        //#. menu title: cell style settings for table cells
        gt.pgettext('menu-title', 'Cell styles');

    // other control label texts ----------------------------------------------

    /**
     * @constant
     */
    EditControls.BORDER_STYLE_LABEL =
        //#. line style (solid, dashed, dotted) of borders in paragraphs and tables cells
        gt.pgettext('borders', 'Border style');

    /**
     * @constant
     */
    EditControls.BORDER_WIDTH_LABEL =
        //#. line width of borders in paragraphs and tables cells
        gt.pgettext('borders', 'Border width');

    /**
     * @constant
     */
    EditControls.BORDER_COLOR_LABEL =
        //#. line color of borders in paragraphs and tables cells
        gt.pgettext('borders', 'Border color');

    /**
     * @constant
     */
    EditControls.CELL_BORDERS_LABEL =
        //#. cell border settings for table cells
        gt.pgettext('borders', 'Cell borders');

    // constants for controls -------------------------------------------------

    /**
     * Standard options for a 'Show toolbar' check box.
     *
     * @constant
     */
    EditControls.SHOW_TOOLBARS_CHECKBOX_OPTIONS = {
        label: /*#. check box label: show/hide the upper toolbar panel */ gt('Show toolbars'),
        tooltip: /*#. check box tooltip: show/hide the upper toolbar panel */ gt('Show or hide the toolbars')
    };

    /**
     * Standard options for a 'Show collaborators' check box.
     *
     * @constant
     */
    EditControls.SHOW_COLLABORATORS_CHECKBOX_OPTIONS = {
        label: /*#. check box label: show/hide the collaborator list */ gt('Show collaborators'),
        tooltip: /*#. check box tooltip: show/hide the collaborator list */ gt('Show or hide list of collaborators')
    };

    /**
     * Standard options for a 'Bold' toggle button.
     *
     * @constant
     */
    EditControls.BOLD_BUTTON_OPTIONS = {
        icon: 'docs-font-bold',
        tooltip: gt('Bold'),
        toggle: true,
        dropDownVersion: { label: gt('Bold') }
    };

    /**
     * Standard options for an 'Italic' toggle button.
     *
     * @constant
     */
    EditControls.ITALIC_BUTTON_OPTIONS = {
        icon: 'docs-font-italic',
        tooltip: gt('Italic'),
        toggle: true,
        dropDownVersion: { label: gt('Italic') }
    };

    /**
     * Standard options for an 'Underline' toggle button.
     *
     * @constant
     */
    EditControls.UNDERLINE_BUTTON_OPTIONS = {
        icon: 'docs-font-underline',
        tooltip: gt('Underline'),
        toggle: true,
        dropDownVersion: { label: gt('Underline') }
    };

    /**
     * Standard options for a 'Strike through' toggle button.
     *
     * @constant
     */
    EditControls.STRIKEOUT_BUTTON_OPTIONS = {
        icon: 'docs-font-strikeout',
        tooltip: gt('Strike through'),
        toggle: true,
        dropDownVersion: { label: gt('Strike through') }
    };

    /**
     * Standard options for a 'Clear formatting' button.
     *
     * @constant
     */
    EditControls.CLEAR_FORMAT_BUTTON_OPTIONS = {
        icon: 'docs-reset-formatting',
        tooltip: /*#. clear all manual formatting of text or table cells (back to default style) */ gt('Clear formatting'),
        dropDownVersion: {
            label: /*#. clear all manual formatting of text or table cells (back to default style) */ gt('Clear formatting')
        }
    };

    /**
     * Standard options for a 'Insert/edit hyperlink' button.
     *
     * @constant
     */
    EditControls.HYPERLINK_BUTTON_OPTIONS = {
        icon: 'docs-hyperlink',
        label: gt('Hyperlink'),
        tooltip: gt('Insert or edit a hyperlink'),
        smallerVersion: {
            css: { width: 35 },
            hideLabel: true
        }
    };

    /**
     * Menu labels for horizontal text alignment.
     *
     * @constant
     */
    EditControls.HOR_ALIGNMENT_LABELS = {
        left:    /*#. horizontal alignment of text in paragraphs or cells */ gt.pgettext('h-alignment', 'Left'),
        center:  /*#. horizontal alignment of text in paragraphs or cells */ gt.pgettext('h-alignment', 'Center'),
        right:   /*#. horizontal alignment of text in paragraphs or cells */ gt.pgettext('h-alignment', 'Right'),
        justify: /*#. horizontal alignment of text in paragraphs or cells */ gt.pgettext('h-alignment', 'Justify'),
        auto:    /*#. horizontal alignment of text in paragraphs or cells (automatically dependent on contents: text left, numbers right) */ gt.pgettext('h-alignment', 'Automatic')
    };

    /**
     * Menu labels for vertical text alignment.
     *
     * @constant
     */
    EditControls.VERT_ALIGNMENT_LABELS = {
        top:     /*#. vertical alignment of text in table cells */ gt.pgettext('v-alignment', 'Top'),
        middle:  /*#. vertical alignment of text in table cells */ gt.pgettext('v-alignment', 'Middle'),
        bottom:  /*#. vertical alignment of text in table cells */ gt.pgettext('v-alignment', 'Bottom'),
        justify: /*#. vertical alignment of text in table cells */ gt.pgettext('v-alignment', 'Justify')
    };

    // class FileNameField ====================================================

    /**
     * A text field used to edit the file name.
     *
     * @constructor
     *
     * @extends TextField
     */
    EditControls.FileNameField = TextField.extend({ constructor: function (app) {

        var // self reference
            self = this,

            // a special busy spinner visible while committing the file name
            busyOverlayNode = $('<div class="busy-overlay">').hide().busy();

        // base constructor ---------------------------------------------------

        TextField.call(this, {
            classes: 'file-name',
            tooltip: gt('Rename document'),
            placeholder: gt('Document name'),
            width: 230,
            select: true,
            smallerVersion: {
                css: {
                    width: 100
                }
            }
        });

        // private methods ----------------------------------------------------

        /**
         * Sets a predefined tool tip in edit mode, or the document name in
         * read-only mode.
         */
        function updateToolTip() {
            Forms.setToolTip(self.getNode(), app.getModel().getEditMode() ? gt('Rename document') : _.noI18n(app.getFullFileName()));
        }

        // initialization -----------------------------------------------------

        // show a busy spinner as long as the control commits the value
        // (the time between change event and losing the focus)
        this.getInputNode().after(busyOverlayNode);
        this.on('group:change', function () { busyOverlayNode.show(); });
        this.on('group:focus group:blur', function () { busyOverlayNode.hide(); });

        // set document name as tool tip in read-only mode
        app.onInit(function () {
            app.getModel().on('change:editmode', updateToolTip);
            updateToolTip();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = self = busyOverlayNode = null;
        });

    }}); // class FileNameField

    // class ColorPicker ======================================================

    /**
     * Creates a control with a drop-down menu used to choose a color from a
     * set of color items. Shows a selection of standard colors, and a table of
     * shaded scheme colors from the current document theme.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {EditApplication} app
     *  The application containing this control group.
     *
     * @param {String|Object} autoColor
     *  Additional information needed to resolve the automatic color. Can be a
     *  string for predefined automatic colors, or an explicit color object
     *  with a value other than the automatic color. See method
     *  Color.getCssColor() for more details.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the RadioList base class.
     */
    EditControls.ColorPicker = RadioList.extend({ constructor: function (app, autoColor, initOptions) {

        var // self reference
            self = this,

            // the color box in the drop-down button
            colorBox = $('<div>').addClass('color-preview-box'),

            // the style sheet containers of the document
            documentStyles = app.getModel().getDocumentStyles();

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            itemDesign: 'grid',
            gridColumns: 10,
            updateCaptionMode: 'none',
            itemMatcher: sameColors
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Converts the passed color value to a CSS color string.
         */
        function setBackgroundColor(boxNode, color) {
            var cssColor = _.isObject(color) ? documentStyles.getCssColor(color, autoColor) : 'transparent';
            boxNode.css('background-color', cssColor);
        }

        /**
         * Initializes a new color item button.
         */
        function createItemHandler(event, buttonNode, color) {
            var boxNode = $('<span class="color-button">');
            setBackgroundColor(boxNode, color);
            buttonNode.prepend(boxNode);
        }

        /**
         * Updates the color box in the drop-down menu button.
         */
        function updateHandler(color) {
            setBackgroundColor(colorBox, color);
        }

        /**
         * Returns whether the passed color objects are considered being equal.
         */
        function sameColors(color1, color2) {

            // quick check: same reference
            if (color1 === color2) { return true; }

            // both colors must be objects
            if (!_.isObject(color1) || !_.isObject(color2)) { return false; }

            var color1Value = _.isString(color1.value) ? color1.value.toLowerCase() : null,
                color2Value = _.isString(color2.value) ? color2.value.toLowerCase() : null;

            // returns whether type and value are considered being equal
            function sameTypeAndValue() {

                // exactly the same type and value
                if ((color1.type === color2.type) && ((color1.type === 'auto') || (color1Value === color2Value))) { return true; }

                // scheme colors: match 'textN' and 'darkN', as well as 'backgroundN' and 'lightN'
                // TODO: really always, also in dark color schemes where background would be dark?
                return (color1.type === 'scheme') && (color2.type === 'scheme') && (
                    ((color1Value === 'text1') && (color2Value === 'dark1')) ||
                    ((color1Value === 'dark1') && (color2Value === 'text1')) ||
                    ((color1Value === 'text2') && (color2Value === 'dark2')) ||
                    ((color1Value === 'dark2') && (color2Value === 'text2')) ||
                    ((color1Value === 'background1') && (color2Value === 'light1')) ||
                    ((color1Value === 'light1') && (color2Value === 'background1')) ||
                    ((color1Value === 'background2') && (color2Value === 'light2')) ||
                    ((color1Value === 'light2') && (color2Value === 'background2')));
            }

            // returns whether transformations are considered being equal
            function sameTransformations() {

                // transformations may be missing in both colors
                if (_.isUndefined(color1.transformations) && _.isUndefined(color2.transformations)) { return true; }

                // arrays must be equally sized, and transformations must be similar enough (bug 30085)
                return _.isArray(color1.transformations) && _.isArray(color2.transformations) &&
                    (color1.transformations.length === color2.transformations.length) &&
                    _.all(color1.transformations, function (tf1, index) {
                        var tf2 = color2.transformations[index];
                        return _.isString(tf1.type) && (tf1.type === tf2.type) &&
                            _.isNumber(tf1.value) && _.isNumber(tf2.value) &&
                            // may not be completely equal (bug 30085)
                            (Math.abs(tf1.value - tf2.value) <= 256);
                    });
            }

            // compare colors directly
            if (sameTypeAndValue() && sameTransformations()) {
                return true;
            }

            // special treatment for fallback values in ODF mode
            return app.isODF() &&
                ((color1.type === 'scheme') && _.isString(color1.fallbackValue) && (color1.fallbackValue.toLowerCase() === color2Value)) ||
                ((color2.type === 'scheme') && _.isString(color2.fallbackValue) && (color2.fallbackValue.toLowerCase() === color1Value));
        }

        /**
         * Creates the option buttons for one table row of scheme colors.
         */
        function fillSchemeColorRow(rowIndex) {
            _.each(SCHEME_COLOR_DEFINITIONS, function (definition) {

                var // the encoded transformation
                    encoded = definition.transformations[rowIndex],
                    // decoded tint/shade value
                    tint = encoded > 0,
                    value = Math.round(Math.abs(encoded) / 255 * 100000),
                    // the theme color object
                    color = { type: 'scheme', value: definition.name },
                    // the data value added at the button
                    dataValue = definition.name,
                    // description of the color (tool tip)
                    label = definition.label,
                    // shade/tint value as integral percentage
                    percent = (100 - Math.round(value / 1000)),
                    // fallback value for format without theme support
                    fallbackValue = null;

                if (value !== 0) {
                    Color.appendTransformation(color, tint ? 'tint' : 'shade', value);
                    dataValue += (tint ? '-lighter' : '-darker') + percent;
                    label = tint ?
                        //#. The full name of a light theme color (a base color lightened by a specific percentage value)
                        //#. Example result: "Green, lighter 20%"
                        //#. %1$s is the name of the base color
                        //#. %2$d is the percentage value, followed by a literal percent sign
                        //#, c-format
                        gt('%1$s, lighter %2$d%', label, _.noI18n(percent)) :
                        //#. The full name of a dark theme color (a base color darkened by a specific percentage value)
                        //#. Example result: "Green, darker 20%"
                        //#. %1$s is the name of the base color
                        //#. %2$d is the percentage value, followed by a literal percent sign
                        //#, c-format
                        gt('%1$s, darker %2$d%', label, _.noI18n(percent));
                }

                // store the resulting RGB color in the color value object
                if (Color.isThemeColor(color)) {
                    fallbackValue = documentStyles.getCssColor(color, 'fill');
                    if (fallbackValue[0] === '#') {
                        color.fallbackValue = fallbackValue.slice(1);
                    } else {
                        Utils.warn('ColorPicker.fillSchemeColorRow(): unexpected CSS fallback color: ', fallbackValue);
                    }
                }

                self.createOptionButton(color, { section: 'theme', tooltip: label, dataValue: dataValue });
            });
        }

        /**
         * Returns the button label for the automatic color.
         */
        function getAutoColorLabel() {
            return Color.isTransparentColor(Color.AUTO, autoColor) ?
                /*#. no fill color, transparent */ gt('No color') :
                /*#. automatic text color (white on dark backgrounds, otherwise black) */ gt('Automatic color');
        }

        /**
         * Inserts all available colors into the drop-down menu.
         */
        function initializeColorTable() {

            self.clearMenu();

            // add automatic color
            self.createMenuSection('auto', { gridColumns: 1 })
                .createOptionButton(Color.AUTO, { section: 'auto', label: getAutoColorLabel(), dataValue: 'auto' });

            // add scheme colors
            if (documentStyles.hasSchemeColors()) {
                self.createMenuSection('theme', { label: gt('Theme Colors') });
                _.times(SCHEME_COLOR_DEFINITIONS[0].transformations.length, fillSchemeColorRow);
            }

            // add predefined colors
            self.createMenuSection('standard', { label: gt('Standard Colors') });
            _.each(BUILTIN_COLOR_DEFINITIONS, function (definition) {
                self.createOptionButton(definition.color, { section: 'standard', tooltip: definition.label, dataValue: definition.name });
            });
        }

        /**
         * Debounced version of the method initializeColorTable().
         */
        var initializeColorTableDebounced = app.createDebouncedMethod($.noop, initializeColorTable, { delay: 50 });

        // methods ------------------------------------------------------------

        /**
         * Changes the information needed to resolve the automatic color, and
         * repaints the menu list item for the automatic color.
         *
         * @param {String|Object} newAutoColor
         *   Can be a string for predefined automatic colors, or an explicit
         *   color object with a value other than the automatic color. See
         *   method Color.getCssColor() for more details.
         *
         * @returns {ColorPicker}
         *  A reference to this instance.
         */
        this.setAutoColor = function (newAutoColor) {
            if (!_.isEqual(autoColor, newAutoColor)) {
                autoColor = newAutoColor;
                setBackgroundColor(this.getMenu().findItemNodes(Color.AUTO).find('.color-button'), Color.AUTO);
            }
            return this;
        };

        // initialization -----------------------------------------------------

        // add marker class for additional CSS formatting
        this.getMenuNode().addClass('color-picker');

        // add the color preview box to the menu button
        this.getMenuButton().append(colorBox);

        // register an update handler that updates the color box
        this.registerUpdateHandler(updateHandler);

        // register a handler that inserts a color box into each list item
        this.getMenu().on('create:item', createItemHandler);

        // lazy initialization of the color table
        this.getMenu().one('popup:beforeshow', initializeColorTable);

        // refresh color buttons in the drop-down menu after theme has changed
        this.listenTo(app.getImportPromise(), 'done', function () {
            self.listenTo(app.getModel().getThemeCollection(), 'triggered', initializeColorTableDebounced);
        });

    }}); // class ColorPicker

    // class TextColorPicker ==================================================

    /**
     * A color picker control for selecting a text color.
     *
     * @constructor
     *
     * @extends EditControls.ColorPicker
     */
    EditControls.TextColorPicker = EditControls.ColorPicker.extend({ constructor: function (app, initOptions) {

         // base constructor --------------------------------------------------

        EditControls.ColorPicker.call(this, app, 'text', Utils.extendOptions({
            icon: 'docs-font-color',
            tooltip: gt('Text color'),
            dropDownVersion: {
                label: gt('Text color')
            }
        }, initOptions));

    }}); // class TextColorPicker

     // class FillColorPicker ==================================================

     /**
      * A color picker control for selecting a fill color.
      *
      * @constructor
      *
      * @extends EditControls.ColorPicker
      */
    EditControls.FillColorPicker = EditControls.ColorPicker.extend({ constructor: function (app, initOptions) {

        // base constructor --------------------------------------------------

        EditControls.ColorPicker.call(this, app, 'fill', Utils.extendOptions({
            icon: 'docs-cell-fill-color',
            tooltip: gt('Fill color'),
            dropDownVersion: {
                label: gt('Fill color')
            }
        }, initOptions));

    }}); // class FillColorPicker

    // class StyleSheetPicker =================================================

    /**
     * A drop-down list control used to select a style sheet from a list. The
     * drop-down list entries will visualize the formatting attributes of the
     * style sheet if possible.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {EditApplication} app
     *  The application containing this control group.
     *
     * @param {String} family
     *  The attribute family of the style sheets visualized by this control.
     *
     * @param {Object} [initOptions]
     *  Additional options passed to the RadioList constructor. Supports all
     *  options of the RadioList class, and the following additional options:
     *  @param {String|String[]} [initOptions.previewFamilies]
     *      The attribute families used to get the formatting options of the
     *      list items representing the style sheets. If omitted, only
     *      attributes of the family specified by the 'family' parameter will
     *      be used.
     *  @param {String[]} [initOptions.sections]
     *      An array specifying the order of all predefined section
     *      identifiers.
     *  @param {String} [initOptions.i18nModulePath]
     *      The module path to the localized JSON object containing the
     *      configuration of translated style sheet names. Each property of the
     *      map must be a string with the English style sheet name as key.
     *      These keys may contain regular expressions which must be placed in
     *      capturing groups (in parentheses). The values of the map are the
     *      translated style sheet names. The placeholders '$1', '$2', etc.
     *      will match the capturing groups defined in the key patterns.
     */
    EditControls.StyleSheetPicker = RadioList.extend({ constructor: function (app, family, initOptions) {

        var // self reference
            self = this,

            // the model instance
            model = null,

            // the style collection
            styleCollection = null,

            // the path to the translated style sheet names
            modulePath = Utils.getStringOption(initOptions, 'i18nModulePath', ''),

            // the configuration for translated style sheet names
            translationDatabase = {},

            // the cache with translated style sheet names
            translationCache = {};

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            itemDesign: 'grid',
            sortItems: true,
            updateCaptionMode: 'none'
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Returns the translation of the passed style sheet name for the
         * current locale. Additionally, the passed style name will be adjusted
         * further for GUI display. All words will be capitalized, automatic
         * line breaks before numbers will be prevented by converting the
         * preceding space characters to NBSP characters.
         *
         * @param {String} styleName
         *  The original (English) style sheet name.
         *
         * @returns {String}
         *  The translated and adjusted style sheet name. If no translation is
         *  available, the passed style sheet name will be adjusted and
         *  returned.
         */
        function translateStyleName(styleName) {

            var // the lower-case style name, used as database key
                styleKey = styleName.toLowerCase(),
                // the translated style name
                translatedName = null;

            // first, check the cache of the translation database
            if (styleKey in translationCache) { return translationCache[styleKey]; }

            // try to find a database entry for the base style name
            _.any(translationDatabase, function (value, key) {

                var // the matches for the pattern of the current entry
                    matches = new RegExp('^' + key + '$', 'i').exec(styleName);

                // continue with next database entry, if key does not match
                if (!_.isArray(matches)) { return; }

                // matching entry found, translate it and replace the placeholders
                translatedName = value;
                for (var index = matches.length - 1; index > 0; index -= 1) {
                    translatedName = translatedName.replace(new RegExp('\\$' + index, 'g'), matches[index]);
                }

                // exit the _.any() loop
                return true;
            });

            // adjust the resulting style name for GUI display
            translatedName = (translatedName || Utils.capitalizeWords(styleName)).replace(/ (\d)/g, '\xa0$1');

            // put the translated name into the cache
            return (translationCache[styleKey] = _.noI18n(translatedName));
        }

        /**
         * Writes the translated name of the specified style sheet into the
         * drop-down button.
         */
        function updateHandler(styleId) {
            var styleName = styleId ? styleCollection.getName(styleId) : null;
            self.setLabel(styleName ? translateStyleName(styleName) : '');
        }

        /**
         * Creates a new button in the drop-down menu for the specified style
         * sheet.
         */
        function createOptionButton(styleName, styleId) {

            var // the section identifier for the style sheet
                category = styleCollection.getUICategory(styleId) || '',
                // sorting priority
                priority = Math.floor(styleCollection.getUIPriority(styleId)),
                // the sort index stored at the button for lexicographical sorting
                sortIndex = String((priority < 0) ? (priority + 0x7FFFFFFF) : priority);

            // translate and adjust style name
            styleName = translateStyleName(styleName);

            // build a sorting index usable for lexicographical comparison:
            // 1 digit for priority sign, 10 digits positive priority,
            // followed by lower-case translated style sheet name
            sortIndex = ('000000000' + sortIndex).substr(-10);
            sortIndex = ((priority < 0) ? '0' : '1') + sortIndex + styleName.toLowerCase();

            // create the list item, pass sorting index
            self.createOptionButton(styleId, { section: category, label: styleName, tooltip: styleName, sortIndex: sortIndex });
        }

        /**
         * Fills the drop-down list with all known style names, and adds
         * preview CSS formatting to the list items.
         */
        var fillList = app.createDebouncedMethod($.noop, Utils.profileMethod('StyleSheetPicker.fillList(): family="' + family + '"', function () {
            // bug 33737, 33747: restore browser focus
            self.getMenu().guardFocusedListItem(function () {
                self.clearMenu();
                // configure the sections
                _.each(Utils.getArrayOption(initOptions, 'sections'), function (sectionId) {
                    self.createMenuSection(sectionId);
                });
                // insert the style sheets
                _.each(styleCollection.getStyleSheetNames(), createOptionButton);
            });
        }), { delay: 1000 });

        /**
         * Handles inserted or modified style sheets in the style collection.
         */
        function insertStyleSheetHandler(event, styleId) {
            // bug 33737, 33747: restore browser focus
            self.getMenu().guardFocusedListItem(function () {
                self.deleteOptionButton(styleId);
                createOptionButton(styleCollection.getName(styleId), styleId);
            });
        }

        /**
         * Handles deleted style sheets in the style collection.
         */
        function deleteStyleSheetHandler(event, styleId) {
            self.deleteOptionButton(styleId);
        }

        // initialization -----------------------------------------------------

        app.onInit(function () {
            model = app.getModel();
            styleCollection = model.getStyleCollection(family);
        });

        // add CSS class to menu button and drop-down list for special formatting
        self.getNode().addClass('style-picker family-' + family);
        self.getMenuNode().addClass('app-' + app.getDocumentType() + ' style-picker family-' + family);

        // update translated style name in drop-down button
        this.registerUpdateHandler(updateHandler);

        // add all known style sheets after import and changed style sheets
        this.listenTo(app.getImportPromise(), 'done', function () {
            app.executeDelayed(fillList, { delay: 1000 });
            self.listenTo(styleCollection, 'insert:stylesheet', insertStyleSheetHandler);
            self.listenTo(styleCollection, 'delete:stylesheet', deleteStyleSheetHandler);
            self.listenTo(styleCollection, 'modify:stylesheet', insertStyleSheetHandler);
            // also reinitialize the preview if theme settings have been changed
            self.listenTo(model.getThemeCollection(), 'triggered', fillList);
        });

        // load the translated style names if specified
        if (modulePath.length > 0) {
            this.listenTo(IO.loadResource(modulePath), 'done', function (data) {
                translationDatabase = data;
                translationCache = {};
                if (app.isImportSucceeded()) {
                    fillList();
                }
            });
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = initOptions = self = model = styleCollection = null;
        });

    }}); // class StyleSheetPicker

    // class FontFamilyPicker =================================================

    /**
     * A combobox control used to select a font family.
     *
     * @constructor
     *
     * @extends ComboField
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class ComboField.
     */
    EditControls.FontFamilyPicker = ComboField.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // the collection of fonts of the edited document
            fontCollection = null;

        // base constructor ---------------------------------------------------

        ComboField.call(this, Utils.extendOptions({
            width: 95,
            tooltip: gt('Font name'),
            typeAhead: true
        }, initOptions));

        // methods ------------------------------------------------------------

        /**
         * Fills the drop-down list with all known font names.
         */
        var fillList = app.createDebouncedMethod($.noop, function () {

            var // all font names, sorted by name
                fontNames = _.sortBy(fontCollection.getFontNames(), function (fontName) {
                    return fontName.toLowerCase();
                });

            self.getMenu().clearContents().createSectionNode('values');
            _.each(fontNames, function (fontName) {
                self.createListEntry(fontName, {
                    section: 'values',
                    label: _.noI18n(fontName),
                    labelStyle: { fontFamily: fontCollection.getCssFontFamily(fontName), fontSize: '115%' }
                });
            });
        });

        // initialization -----------------------------------------------------

        app.onInit(function () {
            fontCollection = app.getModel().getFontCollection();
        });

        // add all known fonts after import and changed fonts
        this.listenTo(app.getImportPromise(), 'done', function () {
            fillList();
            self.listenTo(fontCollection, 'triggered', fillList);
        });

    }}); // class FontFamilyPicker

    // class FontHeightPicker =================================================

    /**
     * A combobox control used to select a font height.
     *
     * @constructor
     *
     * @extends ComboField
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class ComboField.
     */
    EditControls.FontHeightPicker = ComboField.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        ComboField.call(this, Utils.extendOptions({
            width: 40,
            tooltip: gt('Font size'),
            style: 'text-align:right;',
            keyboard: 'number',
            validator: new ComboField.NumberValidator({ min: 1, max: 999.9, precision: 0.1 })
        }, initOptions));

        // initialization -----------------------------------------------------

        _.each([6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 18, 20, 22, 24, 26, 28, 32, 36, 40, 44, 48, 54, 60, 66, 72, 80, 88, 96], function (size) {
            this.createListEntry(size, { label: _.noI18n(String(size)), style: 'text-align:right;padding-right:20px;' });
        }, this);

    }}); // class FontHeightPicker

    // class LanguagePicker ===================================================

    (function () {

        var // a Promise that will resolve when the supported locales have been received
            supportedLocales = IO.sendRequest({
                method: 'POST',
                module: 'spellchecker',
                params: {
                    action: 'supportedlocales'
                }
            })
            .done(function (data) {
                var supportedLocales = Utils.getArrayOption(data, 'SupportedLocales', []);
                _.each(LOCALE_DEFINITIONS, function (definition) {
                    var localeName = definition.locale.replace('-', '_');
                    definition.supported = _.contains(supportedLocales, localeName);
                });
            });

        EditControls.LanguagePicker = RadioList.extend({ constructor: function (initOptions) {

            var // self reference
                self = this;

            // base constructor -----------------------------------------------

            RadioList.call(this, Utils.extendOptions(initOptions, {
                tooltip: gt('Text language'),
                sortItems: true
            }));

            // initialization -------------------------------------------------

            this.getMenuNode().addClass('language-picker');

            supportedLocales.always(function () {
                if (self.destroyed) { return; }
                _.each(LOCALE_DEFINITIONS, function (definition) {
                    self.createOptionButton(definition.locale, {
                        label: definition.label,
                        icon: definition.supported ? 'fa-check-square-o' : 'fa-square-o'
                    });
                });
            });

        }}); // class LanguagePicker

    }());

    // class BorderPicker =====================================================

    /**
     * A drop-down list control with different settings for single or multiple
     * components supporting outer and inner border lines. The list entries are
     * divided into two sections. The first section contains entries for single
     * border lines that can be toggled. The second section contains list
     * entries for complex border settings for multiple component borders.
     *
     * The method 'BorderPicker.setValue()' expects object values with the
     * optional properties 'top', 'bottom', 'left', 'right', insideh', and
     * 'insidev' of type Boolean or Null specifying the visibility state of the
     * respective border line. The property value null represents an ambiguous
     * state for the border line. Additionally, the method accepts an options
     * map as second parameter. The two options 'showInsideHor' and
     * 'showInsideVert' will override the respective options passed to the
     * constructor of this instance.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the base class RadioList.
     *  Additionally, the following options are supported:
     *  @param {Boolean|Function} [initOptions.showInsideHor=false]
     *      If set to true, the drop-down list will provide list entries for
     *      horizontal inner borders between multiple components that are
     *      stacked vertically. If set to a function, the visibility of the
     *      affected list items will be evaluated dynamically every time this
     *      control will be updated, according to the return value.
     *  @param {Boolean|Function} [initOptions.showInsideVert=false]
     *      If set to true, the drop-down list will provide list entries for
     *      vertical inner borders between multiple components that are located
     *      side-by-side. If set to a function, the visibility of the affected
     *      list items will be evaluated dynamically every time this control
     *      will be updated, according to the return value.
     */
    EditControls.BorderPicker = RadioList.extend({ constructor: function (initOptions) {

        var // self reference
            self = this,

            // whether list entries with inner horizontal borders will be shown
            showInsideHor = Utils.getFunctionOption(initOptions, 'showInsideHor', Utils.getBooleanOption(initOptions, 'showInsideHor', false)),

            // whether list entries with inner vertical borders will be shown
            showInsideVert = Utils.getFunctionOption(initOptions, 'showInsideVert', Utils.getBooleanOption(initOptions, 'showInsideVert', false));

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            icon: 'docs-border-tblr',
            tooltip: /*#. in paragraphs and tables cells */ gt.pgettext('borders', 'Borders')
        }, initOptions, {
            itemMatcher: matchBorder,
            updateCaptionMode: 'none'
        }));

        // private methods ----------------------------------------------------

        function isInsideHor(value) {
            return (value === 'insideh') || (value.insideh === true);
        }

        function isInsideVert(value) {
            return (value === 'insidev') || (value.insidev === true);
        }

        /**
         * Returns whether to show list items with horizontal inner borders.
         */
        function showInsideHorItems(value) {
            return _.isFunction(showInsideHor) ? showInsideHor.call(self, value) : showInsideHor;
        }

        /**
         * Returns whether to show list items with vertical inner borders.
         */
        function showInsideVertItems(value) {
            return _.isFunction(showInsideVert) ? showInsideVert.call(self, value) : showInsideVert;
        }

        /**
         * Returns whether to show list items with both inner borders.
         */
        function showInsideBothItems(value) {
            return showInsideHorItems(value) && showInsideVertItems(value);
        }

        /**
         * Returns whether the passed borders matches one of the drop-down list
         * entries, used for updating the selection in the drop-down list.
         *
         * @param {Object} value
         *  The border value to be matched against a list entry.
         *
         * @param {Object|String} entryValue
         *  The value of one of the list entries of this control.
         *
         * @returns {Boolean|Null}
         *  Whether the border value matches the list entry. All properties
         *  contained in the list entry must be equal in the border value.
         *  Other existing properties of the border value are ignored. If the
         *  passed border value contains a single property only (single border
         *  selectors from the upper part of the list), this function returns
         *  null for all list items representing other border lines than the
         *  passed border line.
         */
        function matchBorder(value, entryValue) {

            // skip single border line values
            if (_.isString(value)) { return null; }

            // return false for non-object values (deselect all entries in ambiguous state)
            if (!_.isObject(value)) { return false; }

            // update single-border list items
            if (_.isString(entryValue)) {
             // return false for null values
                return (entryValue in value) && (value[entryValue] === true);
            }

            // update compound-border list items
            return _.all(entryValue, function (visible, propName) {
                // use false for missing properties in value, but do not match existing null values
                return visible === ((propName in value) ? value[propName] : false);
            });
        }

        /**
         * Updates the icon in the drop-down menu button according to the
         * current value of this control.
         */
        function updateHandler(value) {

            // reset caption to default settings in undetermined state
            if (!_.isObject(value) && !_.isString(value)) {
                self.setCaption(self.getOptions());
                return;
            }

            // ignore single border values (after activating a single-border list item)
            if (_.isObject(value)) {
                var icon = '';
                if (value.top) { icon += 't'; }
                if (value.bottom) { icon += 'b'; }
                if (value.insideh) { icon += 'h'; }
                if (value.left) { icon += 'l'; }
                if (value.right) { icon += 'r'; }
                if (value.insidev) { icon += 'v'; }
                if (icon.length > 0) { icon = '-' + icon; }
                icon = 'docs-border' + icon;
                self.setIcon(icon);
            }
        }

        // initialization -----------------------------------------------------

        // prepare the sections for the toggle buttons and the complex border buttons
        self.createMenuSection('single').createMenuSection('compound');

        // insert all supported list items
        _.each(BORDER_ENTRIES, function (entry) {
            var single = _.isString(entry.value),
                section = single ? 'single' : 'compound',
                visible = isInsideHor(entry.value) ? (isInsideVert(entry.value) ? showInsideBothItems : showInsideHorItems) : (isInsideVert(entry.value) ? showInsideVertItems : true);
            self.createOptionButton(entry.value, { section: section, icon: entry.icon, label: entry.label, dataValue: entry.name, visible: visible });
        });

        // register custom update handler to select the correct border icon
        this.registerUpdateHandler(updateHandler);

    }}); // class BorderPicker

    // class BorderStylePicker ================================================

    /**
     * A generic drop-down list control for border styles.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Array} listEntries
     *  An array of descriptors for the entries of the drop-down list. Each
     *  descriptor is an object with the following properties:
     *  - {Any} value
     *      The value associated to the list item.
     *  - {String} label
     *      The text label for the list item.
     *  - {Object} style
     *      The style properties for the generated preview icon, with the
     *      following properties:
     *      - {String} style.style
     *          The line style: one of 'solid', 'dashed', 'dotted', 'dashDot',
     *          or 'dashDotDot'.
     *      - {Number} style.width
     *          Width of a (single) line, in pixels. The value 0 will be
     *          interpreted as hair line.
     *      - {Number} [style.count=1]
     *          Number of parallel lines.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  RadioList.
     */
    EditControls.BorderStylePicker = RadioList.extend({ constructor: function (listEntries, initOptions) {

        var // the icon-like element in the drop-down button
            styleBoxNode = null,

            // maps control values to border style descriptors
            borderStyles = {};

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            tooltip: EditControls.BORDER_STYLE_LABEL,
            updateCaptionMode: 'none'
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Returns the data URL of the preview bitmap for the passed value.
         */
        function getBitmapUrl(value) {
            var borderStyle = (value && (value in borderStyles)) ? borderStyles[value] : { style: 'solid', width: 1, color: '#333333' };
            return getBorderStyleBitmapUrl(32, 18, borderStyle);
        }

        /**
         * Creates an image element showing a preview border line of the
         * specified control value.
         */
        function createStyleBox(value) {
            return $('<img class="border-style-box" src="' + getBitmapUrl(value) + '">');
        }

        /**
         * Initializes a new list item according to the control value.
         */
        function createItemHandler(event, buttonNode, value) {
            Forms.getCaptionNode(buttonNode).prepend(createStyleBox(value));
        }

        /**
         * Updates the border style in the drop-down menu button.
         */
        function updateHandler(value) {
            styleBoxNode.attr('src', getBitmapUrl(value));
        }

        // initialization -----------------------------------------------------

        // register a handler that inserts a border box into each list item
        this.getMenu().on('create:item', createItemHandler);

        // create the bitmaps for the icons, and insert the entries into the drop-down list
        _.each(listEntries, function (entry) {
            borderStyles[entry.value] = _.extend({ color: '#333333' }, entry.icon);
            this.createOptionButton(entry.value, { label: entry.label });
        }, this);

        // insert the icon-like element showing the border line style
        styleBoxNode = createStyleBox();
        this.getCaptionNode().prepend(styleBoxNode);

        // update the line style of the border box
        this.registerUpdateHandler(updateHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            initOptions = styleBoxNode = null;
        });

    }}); // class BorderStylePicker

    // class BorderLineStylePicker ============================================

    /**
     * A drop-down list control for predefined border line styles. Creates list
     * entries for all line styles supported by browsers.
     *
     * @constructor
     *
     * @extends EditControls.BorderStylePicker
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  EditControls.BorderStylePicker.
     */
    EditControls.BorderLineStylePicker = EditControls.BorderStylePicker.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        EditControls.BorderStylePicker.call(this, BORDER_LINE_STYLES, initOptions);

    }}); // class BorderLineStylePicker

    // class BorderLineWidthPicker ============================================

    /**
     * A combo field with predefined entries for border line widths, in points.
     *
     * @constructor
     *
     * @extends ComboField
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  ComboField.
     */
    EditControls.BorderLineWidthPicker = ComboField.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        ComboField.call(this, Utils.extendOptions({
            width: 40,
            tooltip: EditControls.BORDER_WIDTH_LABEL,
            style: 'text-align:right;',
            keyboard: 'number',
            validator: new ComboField.NumberValidator({ min: 0.5, max: 10, precision: 0.1 })
        }, initOptions));

        // initialization -----------------------------------------------------

        _.each([0.5, 1, 1.5, 2, 2.5, 3, 4, 6], function (size) {
            this.createListEntry(size, {
                label: gt.format(
                    //#. A single list entry of a GUI border width picker.
                    //#. Example result: "5 points"
                    //#. %1$d is the width of the border line.
                    //#, c-format
                    gt.npgettext('borders', '%1$d point', '%1$d points', size),
                    _.noI18n(size)
                ),
                style: 'text-align:center;'
            });
        }, this);

    }}); // class BorderLineWidthPicker

    // class TableSizePicker ==================================================

    /**
     * A drop-down button and a drop-down menu containing a resizable grid
     * allowing to select a specific size.
     *
     * @constructor
     *
     * @extends Button
     * @extends MenuMixin
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options that are supported by the
     *  base class Button. Additionally, the following options are supported:
     *  @param {Integer} [initOptions.maxCols=15]
     *      The maximal number of columns to be displayed in the control.
     *  @param {Integer} [initOptions.maxRows=15]
     *      The maximal number of rows to be displayed in the control.
     */
    EditControls.TableSizePicker = Button.extend({ constructor: function (app, initOptions) {

        var // self referemce
            self = this,

            // size of a grid cell
            CELL_WIDTH = Modernizr.touch ? 27 : 18,
            CELL_HEIGHT = Modernizr.touch ? 24 : 16,

            // minimum size allowed to choose
            MIN_SIZE = { width: 1, height: 1 },
            // maximum size allowed to choose (defaulting to 15 columns and rows)
            MAX_SIZE = { width: Utils.getIntegerOption(initOptions, 'maxCols', 15, 1), height: Utils.getIntegerOption(initOptions, 'maxRows', 15, 1) },
            // minimum size visible even if selected size is smaller
            MIN_VIEW_SIZE = { width: Math.min(MAX_SIZE.width, 6), height: Math.min(MAX_SIZE.height, 4) },

            // maximum grid size, according to options, and screen size
            maxGridSize = _.clone(MAX_SIZE),

            // current grid size
            gridSize = { width: 0, height: 0 },

            // visible size of the grid (may be larger than the 'real' size)
            viewGridSize = {},

            // the drop-down menu instance (must be created after Button base constructor!)
            menu = null,

            // the button control to be inserted into the drop-down menu
            // since 'aria-label' would override 'aria-describedby' we set the title attribute only
            gridButtonNode = $(Forms.createButtonMarkup({ attributes: { title: gt('Table size'), role: 'menuitem' } })),

            // canvas element for rendering the grid lines
            gridCanvasNode = $('<canvas>').appendTo(gridButtonNode),

            // the tooltip node showing the current grid size
            sizeTooltip = null,

            // whether to grow to the left or right or to the top or bottom
            growTop = false, growLeft = false,

            // last applied page position
            lastPagePos = null;

        // base constructors --------------------------------------------------

        Button.call(this, Utils.extendOptions({ icon: 'docs-table-insert', tooltip: gt('Insert table'), attributes: { 'aria-haspopup': 'true', 'aria-expanded': 'false' } }, initOptions));
        menu = new BaseMenu({ classes: 'table-size-picker', autoLayout: false, anchor: this.getNode() });
        MenuMixin.call(this, menu, { button: this.getButtonNode() });

        // private methods ----------------------------------------------------

        /**
         * Changes the current size of the grid, and updates the badge labels.
         */
        function setGridSize(newSize) {

            var // get the renderer context
                context = gridCanvasNode[0].getContext('2d'),
                // the pixel size of the grid
                canvasWidth = 0, canvasHeight = 0,
                // the pixel size of the highlighted raea
                rectWidth = 0, rectHeight = 0;

            // validate passed size
            gridSize.width = Utils.minMax(newSize.width, MIN_SIZE.width, maxGridSize.width);
            gridSize.height = Utils.minMax(newSize.height, MIN_SIZE.height, maxGridSize.height);
            viewGridSize.width = Math.max(gridSize.width, MIN_VIEW_SIZE.width);
            viewGridSize.height = Math.max(gridSize.height, MIN_VIEW_SIZE.height);

            // prepare rendering the grid lines
            canvasWidth = viewGridSize.width * CELL_WIDTH - 1;
            canvasHeight = viewGridSize.height * CELL_HEIGHT - 1;
            gridCanvasNode.attr({ width: canvasWidth, height: canvasHeight });
            context.translate(-0.5, -0.5);
            context.clearRect(0, 0, canvasWidth, canvasHeight);

            // render the filled area
            rectWidth = gridSize.width * CELL_WIDTH;
            rectHeight = gridSize.height * CELL_HEIGHT;
            context.fillStyle = '#c8c8c8';
            context.beginPath();
            context.rect(growLeft ? (canvasWidth - rectWidth) : 0, growTop ? (canvasHeight - rectHeight) : 0, rectWidth, rectHeight);
            context.fill();

            // render the lines
            context.lineWidth = 1;
            context.strokeStyle = '#e0e0e0';
            context.beginPath();
            for (var x = CELL_WIDTH; x < canvasWidth; x += CELL_WIDTH) {
                context.moveTo(x, 0);
                context.lineTo(x, canvasHeight);
            }
            for (var y = CELL_HEIGHT; y < canvasHeight; y += CELL_HEIGHT) {
                context.moveTo(0, y);
                context.lineTo(canvasWidth, y);
            }
            context.stroke();

            // update the tooltip
            sizeTooltip.setText(_.noI18n(gt.format(
                //#. %1$d is the number of columns in the drop-down grid of a table size picker control.
                //#, c-format
                gt.ngettext('%1$d column', '%1$d columns', gridSize.width),
                _.noI18n(gridSize.width)
            ) + ' \xd7 ' + gt.format(
                //#. %1$d is the number of rows in the drop-down grid of a table size picker control.
                //#, c-format
                gt.ngettext('%1$d row', '%1$d rows', gridSize.height),
                _.noI18n(gridSize.height)
            )));
        }

        /**
         * Sets the position and growing directions of the menu node according
         * to the available space around the group node.
         */
        function initializeMenuNodePosition() {

            var // position and size of the group node
                groupPosition = Utils.getNodePositionInPage(self.getNode()),
                // sizes available for the drop-down menu per group side
                availableSizes = menu.getAvailableSizes(),
                // decide where to grow the grid to
                newGrowTop = (availableSizes.bottom.height < 200) && (availableSizes.top.height > availableSizes.bottom.height),
                newGrowLeft = availableSizes.left.width > availableSizes.right.width;

            // rerender the grid (position of selected cells) if orientation has changed
            if ((growTop !== newGrowTop) || (growLeft !== newGrowLeft)) {
                growTop = newGrowTop;
                growLeft = newGrowLeft;
                setGridSize(gridSize);
            }

            // calculate maximum size of the grid, according to available space in the window
            maxGridSize.width = Math.min(MAX_SIZE.width, Math.floor((groupPosition.width + (growLeft ? availableSizes.left.width : availableSizes.right.width)) / CELL_WIDTH));
            maxGridSize.height = Math.min(MAX_SIZE.height, Math.floor((growTop ? availableSizes.top.height : availableSizes.bottom.height) / CELL_HEIGHT));

            // initialize grid position
            menu.getNode().toggleClass('grow-top', growTop).toggleClass('grow-left', growLeft);
            menu.setNodePosition({
                top: growTop ? null : (groupPosition.top + groupPosition.height + 1),
                bottom: growTop ? (groupPosition.bottom + groupPosition.height + 1) : null,
                left: growLeft ? null : groupPosition.left,
                right: growLeft ? groupPosition.right : null
            });
        }

        /**
         * Updates the grid size according to the passed screen coordinates.
         */
        function updateGridSizeForPosition(pageX, pageY) {

            var // position/size of the grid button node
                buttonDim = Utils.extendOptions(gridButtonNode.offset(), { width: gridButtonNode.outerWidth(), height : gridButtonNode.outerHeight() }),
                // mouse position relative to grid origin
                mouseX = growLeft ? (buttonDim.left + buttonDim.width - pageX) : (pageX - buttonDim.left),
                mouseY = growTop ? (buttonDim.top + buttonDim.height - pageY) : (pageY - buttonDim.top),
                // new grid size (enlarge if the last column/row is covered more than 80% of its width/height)
                newWidth = Math.floor(mouseX / CELL_WIDTH + 1.2),
                newHeight = Math.floor(mouseY / CELL_HEIGHT + 1.2);

            // set grid size if changed
            if ((newWidth !== gridSize.width) || (newHeight !== gridSize.height)) {
                setGridSize({ width: newWidth, height: newHeight });
            }

            // store position, needed to refresh grid if its position changes from outside
            lastPagePos = { x: pageX, y: pageY };

            return false;
        }

        /**
         * Handles 'menu:beforeopen' events and initializes the drop-down grid.
         * Registers a 'mouseenter' handler at the drop-down menu that starts
         * a 'mousemove' listener when the mouse first hovers the grid element.
         */
        function menuBeforeOpenHandler() {

            // unbind all running global event handlers
            unbindGlobalEventHandlers();

            // always show the tooltip
            sizeTooltip.show();

            // initialize grid
            lastPagePos = null;
            setGridSize(MIN_SIZE);
            initializeMenuNodePosition();

            // wait for mouse to enter the grid before listening globally to mousemove events
            gridButtonNode.off('mouseenter').one('mouseenter', function () {
                $(document).on('mousemove', gridMouseMoveHandler);
            });
        }

        /**
         * Handles 'menu:open' events.
         */
        function menuOpenHandler() {
            self.getButtonNode().attr('aria-expanded', 'true');
        }

        /**
         * Handles 'menu:close' events.
         */
        function menuCloseHandler() {
            sizeTooltip.hide();
            unbindGlobalEventHandlers();
            lastPagePos = null;
            self.getButtonNode().attr('aria-expanded', 'false');
        }

        /**
         * Handles 'menu:layout' events.
         */
        function menuLayoutHandler() {
            if (lastPagePos) {
                updateGridSizeForPosition(lastPagePos.x, lastPagePos.y);
            } else {
                setGridSize(gridSize);
            }
            initializeMenuNodePosition();
        }

        /**
         * Unbind the global event listener registered at the document.
         */
        function unbindGlobalEventHandlers() {
            $(document).off('mousemove', gridMouseMoveHandler);
        }

        /**
         * Handles 'mousemove' events in the open drop-down menu element.
         */
        function gridMouseMoveHandler(event) {
            updateGridSizeForPosition(event.pageX, event.pageY);
            return false;
        }

        /**
         * Handles keyboard events in the open drop-down menu element.
         */
        function gridKeyDownHandler(event) {

            var // new size of the grid
                newGridSize = _.clone(gridSize);

            switch (event.keyCode) {
            case KeyCodes.LEFT_ARROW:
                newGridSize.width += (growLeft ? 1 : -1);
                setGridSize(newGridSize);
                return false;

            case KeyCodes.UP_ARROW:
                newGridSize.height += (growTop ? 1 : -1);
                setGridSize(newGridSize);
                return false;

            case KeyCodes.RIGHT_ARROW:
                newGridSize.width += (growLeft ? -1 : 1);
                setGridSize(newGridSize);
                return false;

            case KeyCodes.DOWN_ARROW:
                newGridSize.height += (growTop ? -1 : 1);
                setGridSize(newGridSize);
                return false;
            }
        }

        // initialization -----------------------------------------------------

        // embed the button control in the drop-down menu
        menu.appendContentNodes(gridButtonNode);

        // create the tooltip that shows the table size currently selected
        sizeTooltip = new Tooltip(_.extend({ anchor: menu.getNode() }, (Modernizr.touch ? { position: 'right bottom top' } : {})));
        // set aria attributes
        sizeTooltip.getNode().attr({ 'id': sizeTooltip.getUid(), 'aria-live': 'assertive', 'aria-relevant': 'additions', 'aria-atomic': 'true', role: 'tooltip' });
        gridButtonNode.attr('aria-describedby', sizeTooltip.getUid());

        // button click handler (convert ENTER, SPACE, and TAB keys to click events)
        Forms.setButtonKeyHandler(gridButtonNode, { tab: true });
        gridButtonNode.on('click', function (event) {
            // bug 31170: restrict to 'click' events that originate from keyboard shortcuts
            // (prevent double trigger from 'tracking:end' and 'click')
            if (_.isNumber(event.keyCode)) {
                self.triggerChange(gridSize, { sourceEvent: event });
            }
        });

        // register event handler for tracking (bug 28583, needed for touch devices)
        gridButtonNode.on('tracking:end', function (event) {
            self.triggerChange(gridSize, { sourceEvent: event });
        });

        // register event handlers
        this.on({ 'menu:beforeopen': menuBeforeOpenHandler, 'menu:open': menuOpenHandler, 'menu:close': menuCloseHandler, 'menu:layout': menuLayoutHandler });
        gridButtonNode.on('tracking:move', gridMouseMoveHandler);
        menu.getContentNode().on('keydown', gridKeyDownHandler);

        // bug 28583: enable tracking for touch devices
        gridButtonNode.enableTracking();

        // destroy all class members on destruction
        this.registerDestructor(function () {
            unbindGlobalEventHandlers();
            gridButtonNode.remove();
            sizeTooltip.destroy();

            app = menu = gridButtonNode = gridCanvasNode = sizeTooltip = null;
        });

    }}); // class TableSizePicker

    // class AccessRightsPicker ===============================================

    EditControls.AccessRightsPicker = RadioList.extend({ constructor: function (app, initOptions) {

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions(initOptions, {
            label: _.noI18n('Access Rights'),
            tooltip: _.noI18n('Access rights'),
            sortItems: true
        }));

        // initialization -----------------------------------------------------

        _.each(ACCESS_RIGHTS_DEFINITIONS, function (definition) {
            this.createOptionButton(definition.mail, { label: _.noI18n(definition.name), tooltip: _.noI18n(definition.mail) });
        }, this);

    }}); // class AccessRightsPicker

    // class ApplicationStatusLabel ===========================================

    /**
     * Shows the current status of the application when it has changed.
     *
     * @constructor
     *
     * @extends Label
     */
    EditControls.ApplicationStatusLabel = Label.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // all captions for the different application states
            captions = {
                preview: { icon: 'fa-refresh', iconClasses: 'fa-spin', label: gt('Loading document') },
                sending: { icon: 'fa-refresh', iconClasses: 'fa-spin', label: gt('Saving changes') },
                ready:   { icon: 'fa-check', label: gt('All changes saved') }
            };

        // base constructor ---------------------------------------------------

        Label.call(this, _.extend(initOptions, { classes: 'app-status-label', minWidth: 200 }));

        // initialization -----------------------------------------------------

        // make label node an ARIA live region
        this.getLabelNode().attr({
            role: 'status',
            'aria-live': 'polite',
            'aria-relevant': 'additions',
            'aria-atomic': true,
            'aria-readonly': true
        });

        // update caption according to application status
        this.listenTo(app, 'docs:state', function (state) {
            self.setCaption(captions[state]);
        });

        // destroy class members on destruction
        this.registerDestructor(function () {
            app = self = null;
        });

    }}); // class ApplicationStatusLabel

    // class AcquireEditButton ================================================

    /**
     * A button used to acquire edit rights for the document.
     *
     * @constructor
     *
     * @extends Button
     */
    EditControls.AcquireEditButton = Button.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // timer that stops the animation after a delay
            timer = null;

        // base constructor ---------------------------------------------------

        initOptions = Utils.extendOptions({
            icon: 'fa-pencil',
            label: gt('Edit'),
            tooltip: gt('Acquire edit rights'),
            smallerVersion: {
                hideLabel: true
            }
        }, initOptions);
        Button.call(this, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Starts the spinning animation while edit rights are being acquired.
         */
        function startAnimation() {
            stopAnimation();
            self.setIcon('fa-refresh', { classes: 'fa-spin' });
            timer = app.executeDelayed(stopAnimation, { delay: 30000 });
        }

        /**
         * Stops the spinning animation shown while edit rights are being
         * acquired.
         */
        function stopAnimation() {
            if (timer) { timer.abort(); timer = null; }
            self.setIcon(initOptions.icon);
        }

        // initialization -----------------------------------------------------

        // start and stop the spinning animation
        app.onInit(function () {
            self.listenTo(app, 'docs:editrights:acquire', startAnimation);
            self.listenTo(app, 'docs:editrights:decline', stopAnimation);
            self.listenTo(app.getModel(), 'change:editmode', stopAnimation);
        });

        // destroy class members on destruction
        this.registerDestructor(function () {
            stopAnimation();
            app = initOptions = self = null;
        });

    }}); // class AcquireEditButton

    // class UserBadge ========================================================

    /**
     *
     * A badge displaying properties of an OX user, consisting the user's profile pic,
     * display name, and active status.
     *
     * @constructor
     *
     * @extends Picture
     *
     * @param {Object} [initOptions]
     *  Supports all options of the Picture class, and additionally :
     *  @param [initOptions.userId=null]
     *  the OX Appsuite Id of the user to be displayed
     *  @param [initOptions.userName='Unknown']
     *  the display name of the user
     *  @param [initOptions.editor=false]
     *  whether this user is currently a document editor (possesing edit rights).
     *  @param [initOptions.pictureWidth=48]
     *  the desired picture width. Defaults to 48px.
     *  @param [initOptions.pictureHeight=48]
     *  the desired picture height. Defaults to 48px.
     */
    EditControls.UserBadge = Picture.extend({ constructor: function (initOptions) {

        var userId = Utils.getIntegerOption(initOptions, 'userId', null),
            userName = Utils.getStringOption(initOptions, 'userDisplayName', gt('Unknown')),
            isEditor = Utils.getBooleanOption(initOptions, 'editor', false),
            pictureHeight = Utils.getIntegerOption(initOptions, 'pictureHeight', 48),
            pictureWidth = Utils.getIntegerOption(initOptions, 'pictureWidth', 48),
            userNameNode = $('<a>').addClass('user-name halo-link').text(userName).data({ internal_userid : userId }),
            descriptionNode = $('<div>').addClass('user-description'),
            statusNode = $('<div>').addClass('user-status'),
            placeHolderImageUrl = ox.base + '/apps/themes/default/dummypicture.png',
            imageOptions = { tooltip: userName, alt: userName },
            self = this;

        // base constructor ---------------------------------------------------

        Picture.apply(this, [placeHolderImageUrl, Utils.extendOptions(initOptions, imageOptions)]);

        // private methods -----------------------------------------------------

        // initializes user profile pic
        function initProfilePicture(userId) {
            // use placeholder picture and quit if no user id is available
            if (!userId) {
                self.setSource(placeHolderImageUrl);
                return;
            }
            // double picture resolution for retina displays
            if (_.device('retina')) {
                pictureHeight = pictureHeight * 2;
                pictureWidth = pictureWidth * 2;
            }
            // build user photo url
            var pictureParams = { internal_userid: userId, width: pictureHeight, height: pictureWidth, scaleType: 'cover' },
                userPhotoSource = ContactsAPI.pictureHalo(null, pictureParams);
            self.setSource(userPhotoSource);
            // set also placeHolder image, when this particular user doesn't have an profile image yet.
            var testImage = new Image();
            testImage.src = userPhotoSource;
            testImage.onload = function () { if (testImage.width === 1) { self.setSource(placeHolderImageUrl); } };
            testImage.onerror = function () { Utils.error('EditControls.UserBadge.initProfilePicture(): Failed to load user profile picture'); };
        }

        // initialization -----------------------------------------------------
        // prepare profile picture, show placeholder image initially
        initProfilePicture(userId);
        descriptionNode.append(userNameNode);
        // mark user with editrights with an appropriate icon
        if (isEditor) { statusNode.append(Forms.createIconMarkup('fa-edit')); }

        this.addChildNodes(descriptionNode, statusNode);

        // ARIA
        this.getNode().attr({ role: 'dialog'});

    }}); // class UserBadge

    // class UsersLayerMenu ================================================

    /**
     * A LayerMenu displaying collaborating users.
     *
     * @constructor
     *
     * @extends LayerMenuButton
     *
     * @param {BaseApplication} app
     *  The application containing this context menu.
     *
     * @param {String} layerTitle
     *  The title on top of the menu(-layer)
     *
     * @param {Object} [initOptions]
     *  Supports all options of the LayerMenu class.
     *
     */
    EditControls.UsersLayerMenu = BaseControls.LayerMenu.extend({ constructor: function (app, layerTitle, initOptions) {

        var // the layerMenu-class
            self = this,
            users = [],
            // Whether this users list can be auto opened.
            // This users list should be auto-opened at most once in one document session.
            autoShow = true;

        // base constructor ---------------------------------------------------

        BaseControls.LayerMenu.call(this, app, layerTitle, _.extend(initOptions, {
            positioning: setDefaultPosition,
            autoClose: false,
            autoFocus: false,
            closer: true,
            rootContainerNode: app.getWindowNode()
        }));

        // private methods ----------------------------------------------------
        function fillMenu() {
            self.getNode().find('.group').not('.dragHead').remove();
            // generate badges for all collaborating users
            _.each(users, function (user) {
                self.addGroup(null, new EditControls.UserBadge({
                    userId: user.id,
                    userDisplayName: user.userDisplayName,
                    backgroundMode: true,
                    pictureHeight: 32,
                    pictureWidth: 32,
                    classes: 'user-badge',
                    editor: app.getEditUserId() === user.userId
                }));
            });
        }

        // refresh local users list
        function refreshCollaboratorsList(relevantUsers) {
            // bring edit user to top for better visual consistency
            users = _.sortBy(relevantUsers, function (user) {
                return app.getEditUserId() !== user.userId;
            });
            // redraw user badges
            fillMenu();
        }

        // Sets the default position of the collaborators list layer menu
        // (right top corner of the application content area.)
        function setDefaultPosition() {
            var appPaneNode = app.getView().getAppPaneNode(),
                marginToEdge = 40,
                defaultPosition = {
                    top: appPaneNode.position().top + marginToEdge,
                    left: appPaneNode.width() - self.getNode().width() - marginToEdge
                };
            self.setNodePosition(defaultPosition);
        }

        // initialization -----------------------------------------------------

        // lazy initialization of the drop-down menu
        self.on('popup:beforeshow', fillMenu);

        // auto show/hide collaborator list depending on user event
        app.on('docs:users', function (data) {
            var activeUsers = Utils.getArrayOption(data, 'activeUsers', []),
                relevantUsers = _.reject(activeUsers, function (user) {
                    return !user.active && user.durationOfInactivity >= 60;
                });
            // refresh local list with relevant users (active and inactive users < 60 secs)
            refreshCollaboratorsList(relevantUsers);
            // auto show once if somebody joins
            if (data && data.activeUsers && activeUsers.length > 1 && (app.getActiveUsers().length !== activeUsers.length) && autoShow) {
                self.show();
            }
            // auto hide the collaborators list if user is alone on the document
            if (relevantUsers.length < 2) { self.hide(); }
        });

        // disable auto-open functionality if user has already closed this popup by himself/herself.
        this.on('layermenu:userclose', function () {
            autoShow = false;
        });

        // update control items immediately on hide/show
        this.on('popup:show popup:hide', function () {
            app.getController().update();
        });

        // destroy all class members
        this.registerDestructor(function () {
            app = initOptions = null;
        });

    }}); // class UsersLayerMenu

    // exports ================================================================

    return EditControls;

});
