/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2011
 * Mail: info@open-xchange.com
 *
 * @author Kai Ahrens <kai.ahrens@open-xchange.com>
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/app/rtconnection',
    ['io.ox/core/event',
     'io.ox/realtime/groups',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/logger',
     'io.ox/office/tk/object/triggerobject',
     'io.ox/office/editframework/utils/editconfig'
    ], function (Events, RTGroups, Utils, Logger, TriggerObject, Config) {

    'use strict';

    var // names of internal events for the real-time connection state
        INTERNAL_EVENTS = 'online offline reset error:notMember',

        // names of runtime requests expecting an answer
        REQUEST_EVENTS = 'addresource getdocument',

        // names of runtime push messages
        PUSH_EVENTS = 'update flusherrorinfo hangup preparelosingeditrights declinedacquireeditrights acceptedacquireeditrights',

        // max number of action send calls until an update must be received
        MAX_NUM_OF_APPLYACTIONS_UNTIL_UPDATE = 50,

        // repeated time to check for pending ACKs
        CHECK_PENDING_INTERVAL = 30000;

    // class RTConnection =====================================================

    /**
     * Represents the connection to the real-time framework, used to send
     * message to and receive message from the server.
     *
     * Triggers the following events:
     * - 'update': Update notifications containing actions created by remote
     *  clients, and state information for the application (read-only, name and
     *  identifier of the current editor, etc.).
     *
     * @extends TriggerObject
     *
     * @param {EditApplication} app
     *  The application containing this operations queue.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Number} [initOptions.realTimeDelay=700]
     *      The duration in milliseconds the realtime framework will collect
     *      messages before sending them to the server.
     */
    function RTConnection(app, initOptions) {

        var // self reference
            self = this,

            // file descriptor of the edited document
            file = app.getFileDescriptor(),

            // create the core real-time group instance for the document
            rtGroup = RTGroups.getGroup('synthetic.office://operations/' + file.folder_id + '.' + file.id),

            // private event hub for internal event handling
            eventHub = new Events(),

            // duration for the realtime framework collecting messages before sending
            realTimeDelay = Utils.getIntegerOption(initOptions, 'realTimeDelay', 700, 0),

            traceActions = {},

            // pending deferreds
            pendingSendAcks = [],

            // count sending actions
            applyActionsSent = 0,

            // in prepare losing edit rights, for debug purposes only!
            inPrepareLosingEditRights = false;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Log the data
         */
        function debugLogUpdateNotification(data) {
            var // the operations array (may be missing completely)
                actions = Utils.getArrayOption(data, 'actions', []),
                // editing user id
                editUserId = Utils.getStringOption(data, 'editUserId', ''),
                // server osn
                serverOSN = Utils.getIntegerOption(data, 'serverOSN', -1),
                // has errors
                hasErrors = Utils.getBooleanOption(data, 'hasErrors', false),
                // server osn
                activeClients = Utils.getArrayOption(data, 'activeUsers', []).length,
                // write protected
                writeProtected = Utils.getBooleanOption(data, 'writeProtected', false),
                // editing user id
                lockedByUser = Utils.getStringOption(data, 'lockedByUser', ''),
                // output string
                output = '';

            // output content of update notification
            output = 'editUserId: ' + editUserId + '; ';
            output += 'server osn: ' + serverOSN + '; ';
            output += 'active users: ' + activeClients + '; ';
            output += 'hasErrors: ' + hasErrors + '; ';
            output += 'writeProtected: ' + writeProtected + '; ';
            output += 'locked by user: ' + lockedByUser;
            Utils.log(output);

            // check that all array elements are objects
            if (_.all(actions, _.isObject)) {
                try {
                    Utils.log('Last remote action received by RT-Update: ' + JSON.stringify(actions[actions.length - 1]));
                } catch (ex) {
                    // do nothing
                }
            }
        }

        /**
         * Register events for specific RT actions.
         * The event handler 'eventName' is triggered, when
         * an RT event with the action name 'eventName' is received.
         * The triggered eventName will be equal to the 'actionName'.
         * The eventHandler will be called with the received payload data
         */
        function registerRTEvents(eventNames) {
            _.each(eventNames.split(/\s+/), function (eventName) {
                rtGroup.on('receive', function (event, stanza) {
                    _.each(stanza.getAll('office', eventName), function (payload) {
                        RTConnection.log('RTConnection.eventHub.trigger: office:' + eventName);
                        eventHub.trigger(eventName, payload.data);
                    });
                });
            });
        }

        /**
         * Determines whether a certain action should be traced on its way
         * through the realtime backend.
         *
         * @returns {Boolean}
         *  Whether to trace the stanzas for the specified action.
         */
        function shouldTrace(action) {
            return traceActions[action.toLowerCase()];
        }

        /**
         * Checks the pending ACKs for their pending time. Outputs a log
         * entry if the pending time >= CHECK_PENDING_INTERVAL.
         * Debug only function.
         */
        function checkPendingAcks() {
            var // current time
                now = new Date();

            _.each(pendingSendAcks, function (entry) {
                var timeSpan;

                if (entry.deferred && entry.deferred.state() === 'pending') {
                    timeSpan = now.getTime() - entry.timeStamp.getTime();
                    if (timeSpan > CHECK_PENDING_INTERVAL) {
                        Utils.log('RTConnection.send: ACK pending for ' + Math.round(timeSpan / 1000) + ' sec,' + ' action=' + entry.action);
                    }
                }
            });
        }

       /**
         * Send a message to the connected RT object,
         * waiting for the acknowledge to this call.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  acknowledge for this request will have arrived, or
         *  rejected after a time out otherwise.
         */
        function send(action, element, data) {

            var payloads = [{ element: 'action', data: action }],
                defer = null;

            if (_.isString(element) && !_.isUndefined(data)) {
                payloads.push({
                    namespace: 'office',
                    element: element,
                    data: data,
                    verbatim: true
                });
            }

            defer = rtGroup.send({
                element: 'message',
                payloads: payloads,
                trace: shouldTrace(action),
                bufferinterval: realTimeDelay
            });

            if (RTConnection.debug) {
                // add logging code for pending ACKs bound to sends
                pendingSendAcks.push({ deferred: defer, timeStamp: new Date(), action: action });
                defer.always(function () {
                    var // current time
                        now = new Date(),
                        // time span
                        timeSpan = 0,
                        // i
                        i = 0;

                    for (i = 0; i < pendingSendAcks.length; i++) {
                        if (pendingSendAcks[i].deferred === defer) {
                            timeSpan = now.getTime() - pendingSendAcks[i].timeStamp.getTime();
                            // log message if deferred needed at least CHECK_PENDING_INTERVAL time
                            // to be resolved/rejected
                            if (timeSpan > CHECK_PENDING_INTERVAL) {
                                RTConnection.log('RTConnection.send: pending ACK resolved/rejected after ' + Math.round(timeSpan / 1000) + ' seconds.');
                            }
                            // remove entry from pending send acks array
                            pendingSendAcks.splice(i, 1);
                            break;
                        }
                    }

                    if (pendingSendAcks.length > 0) {
                        RTConnection.log('RTConnection.send: pending ACKs for send found: ' + pendingSendAcks.length);
                    }
                });
            }

            // Time out detection now uses the deferred to trigger an optional timeout handler
            defer.fail(function () {
                RTConnection.error('Send message with RT failed, payloads: ' + payloads.toString());
                // RT calls us even after connection, rtgroup are destroyed and references are cleared.
                if (!self.destroyed) { self.trigger('timeout'); }
            });

            return defer.promise();
        }

        /**
         * Send a message request to the connected RT object,
         * returning a promise object that will be resolved,
         * if the result data to this request has arrived.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  result for this request has arrived, or rejected otherwise.
         */
        function sendQuery(action, element, data) {
            var payloads = [{ element: 'action', data: action }],
                defer;

            if (_.isString(element) && !_.isUndefined(data)) {
                payloads.push({
                    namespace: 'office',
                    element: element,
                    data: data,
                    verbatim: true
                });
            }

            defer = rtGroup.query({
                element: 'message',
                payloads: payloads,
                trace: shouldTrace(action),
                bufferinterval: realTimeDelay
            });

            // Time out detection now uses the deferred to trigger an optional timeout handler
            defer.fail(function () {
                RTConnection.error('Send query message with RT failed, payloads: ' + payloads.toString());
                // RT calls us even after connection, rtgroup are destroyed and references are cleared.
                if (!self.destroyed) { self.trigger('timeout'); }
            });

            return defer.promise();
        }

        /**
         * Handles the response from the RT low-level code and extract the
         * data part from, providing it as result via the deferred/promise.
         *
         * @param {jQuery.Deferred|jQuery.Promise} orgDeferred
         *  The original promise or deferred provided by the RT low-level
         *  code.
         *
         * @param {Number} [timeout]
         *  If specified, the delay time in milliseconds, after the Promise
         *  returned by this method will be rejected, if the passed original
         *  Deferred object is still pending.
         *
         * @returns {jQuery.Deferred}
         *  The Deferred object that will be resolved if the result for the
         *  original request has arrived, or rejected otherwise.
         */
        function handleResponse(orgDeferred, timeout) {

            var // the Deferred object to be returned
                def = $.Deferred();

            orgDeferred.then(function (response) {
                var payloads = Utils.getArrayOption(response, 'payloads', []);
                if (self.destroyed) {
                    // reject deferred after destroy
                    def.reject({cause: 'closing'});
                }
                if (payloads.length > 0) {
                    // Extract the data and provide it
                    def.resolve(payloads[0].data);
                } else {
                    // Response from server is not in the expected format, therefore
                    // we reject the deferred here.
                    def.reject(response);
                }
            }, function (response) {
                var payloads = Utils.getArrayOption(response, 'payloads', []);
                if (payloads.length > 0) {
                    // Extract data even for the rejected case
                    def.reject(payloads[0].data);
                } else {
                    // If we don't find a payload just provide the original response
                    def.reject(response);
                }
            });

            // prevent endless waiting
            if (_.isNumber(timeout) && (timeout > 0)) {
                var timer = app.executeDelayed(function () {
                    if (def.state() === 'pending') {
                        RTConnection.log('Timeout reached for RT request, timeout=' + timeout + 'ms.');
                        def.reject({ cause: 'timeout' });
                    }
                }, { delay: timeout });
                def.always(function () { timer.abort(); timer = null; });
            }

            return def;
        }

        /**
         * Handles the response from the RT low-level code and adds timeout
         *
         * @param {jQuery.Deferred|jQuery.Promise} orgDeferred
         *  The original promise or deferred provided by the RT low-level
         *  code.
         *
         * @param {Number} [timeout]
         *  If specified, the delay time in milliseconds, after the Promise
         *  returned by this method will be rejected, if the passed original
         *  Deferred object is still pending.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  result for the original request has arrived, or rejected otherwise.
         */
        function handleGenericResponse(orgDeferred, timeout) {
            var // the Deferred object to be returned
                def = $.Deferred();

            orgDeferred.then(function (data) { def.resolve(data); }, function (data) { def.reject(data); });

            // prevent endless waiting
            if (_.isNumber(timeout) && (timeout > 0)) {
                app.executeDelayed(function () { def.reject({ cause: 'timeout' }); }, { delay: timeout });
            }

            return def.promise();
        }

        // methods ------------------------------------------------------------

        /**
         * Connects the real-time group and sends the initial request for the
         * document import operations.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  getActions results.
         */
        this.connect = function (connectData) {
            var connectDeferred = null,
                // optional connect data
                connectOptions = {expectWelcomeMessage: true};

            if (connectData) {
                _.extend(connectOptions, {additionalPayloads: [{element: 'connectData', namespace: 'office', data: connectData, verbatim: true }]});
            }

            if (!rtGroup.isRTWorking()) {
                // the server doesn't support real-time communication, must be
                // incorrect configured.
                return $.Deferred().reject({cause: 'bad server component' });
            }

            RTConnection.log('RTConnection.connect called');

            connectDeferred = handleResponse(rtGroup.join(connectOptions));
            // Remember *connect* deferred to be able to reject them if user
            // closes document while we are still connecting to the realtime group!
            return app.createAbortablePromise(connectDeferred);
        };

        /**
         * Send the given operations to the connected RT object
         * for further distribution to other connected clients
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  ACK of the internal send arrives.
         */
        this.sendActions = function (actions) {
            var tmp;

            RTConnection.log('RTConnection.applyactions called');
            if (RTConnection.debug) {
                if (inPrepareLosingEditRights && actions && (actions.length > 0)) {
                    tmp = actions[actions.length - 1].operations;
                    if (tmp && tmp.length > 0) {
                        RTConnection.log('preparelosingeditrights: applyactions, last operation: ' + JSON.stringify(_.last(tmp)));
                    }
                }
            }
            applyActionsSent++;
            if (applyActionsSent > MAX_NUM_OF_APPLYACTIONS_UNTIL_UPDATE) {
                self.trigger('noUpdateFromServer');
                // reset counter after we have triggered the missing update
                applyActionsSent = 0;
            }
            return send('applyactions', 'actions', actions);
        };

        /**
         * Sends user data update to the server. This user update data will
         * broadcasted to all collaborating clients.
         *
         * @param {String} action
         *  Action name string to be handled by the server.
         *
         * @param {Object} userData
         *  An object containing relevant additional user data on the document.
         *  @param {Object} userData.selection
         *      Contains a 'recent' user selection on a document. This will be
         *      used for showing user cursors of multiple collaborators while
         *      editing on documents.
         *
         * @returns {jQuery.Promise}
         */
        this.updateUserData = Config.SHOW_REMOTE_SELECTIONS ? function (action, userData) {
            RTConnection.log('RTConnection.updateUserData called', userData);
            return send(action, 'state', userData);
        } : function () { return $.when(); };

        /**
         * Sends a real-time request to the server and waits for a response.
         *
         * @param {String} action
         *  The identifier of the action sent to the server.
         *
         * @param {Number} [timeout]
         *  If specified, the delay time in milliseconds, after the Promise
         *  returned by this method will be rejected, if the query is still
         *  pending.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  answer of the server request, or rejected on error or timeout.
         */
        this.sendQuery = function (action, timeout) {
            RTConnection.log('RTConnection.sendQuery(' + action + ')');
            return handleResponse(sendQuery(action), timeout);
        };

        /**
         * Initiates flushing of the document using a
         * synchronous request.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  answer for flushDocument from the server arrives.
         */
        this.flushDocument = function () {
            RTConnection.log('RTConnection.flushDocument called');
            return handleGenericResponse(sendQuery('flushdocument'), 20000);
        };

        /**
         * Send our wish to acquire the edit rights to the connected RT object
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  ACK of the internal send arrives.
         */
        this.acquireEditRights = function () {
            RTConnection.log('RTConnection.acquireedit called');
            return send('acquireedit', 'state', { osn: app.getModel().getOperationStateNumber() });
        };

        /**
         * Acknowledge to the server that our preparation to lose the
         * edit rights are completed.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  ACK of the internal send arrives.
         */
        this.canLoseEditRights = function () {
            RTConnection.log('RTConnection.canloseeditrights called');
            if (RTConnection.debug) {
                inPrepareLosingEditRights = false;
            }
            return send('canloseeditrights', 'state', { osn: app.getModel().getOperationStateNumber() });
        };

        /**
         * Sends client log message to the server so it can be written to the
         * server side log files for later debugging.
         *
         * @return {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  ACK of the internal send arrives.
         */
        this.sendLogMessage = function (data) {
            RTConnection.log('RTConnection.sendLogMessage called');
            return send('logMessage', 'state', { message: data });
        };

        /**
         * Close the document at the connected RT object
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  closing document status of the request.
         */
        this.closeDocument = function (recentFileData) {
            var leaveOptions = {expectSignOffMessage: true};

            RTConnection.log('RTConnection.closeDocument called');
            if (recentFileData) {
                _.extend(leaveOptions, {additionalPayloads: [{element: 'recentfile', namespace: 'office', data: recentFileData, verbatim: true }]});
            }
            return handleResponse(rtGroup.leave(leaveOptions), 20000);
        };

        // initialization -----------------------------------------------------

        RTConnection.log('RTConnection initialization');
        if (RTConnection.debug) {
            app.repeatDelayed(checkPendingAcks, {delay: CHECK_PENDING_INTERVAL});
        }

        // read traceable actions from page URL
        if (_.url.hash('office:trace')) {
            _.each(_.url.hash('office:trace').split(/\s*,\s*/), function (action) {
                traceActions[action.toLowerCase()] = true;
            });
        }

        // register the internal event hub for request events and custom runtime events
        registerRTEvents(REQUEST_EVENTS + ' ' + PUSH_EVENTS);

        // forward runtime realtime events to own listeners
        rtGroup.on(INTERNAL_EVENTS, function (event) {
            self.trigger(event.type);
        });

        // forward OX Documents runtime events to own listeners
        eventHub.on(PUSH_EVENTS, function (event, data) {
            if (event.type === 'update') {
                applyActionsSent = 0;
                if (RTConnection.debug) {
                    debugLogUpdateNotification(data);
                }
            } else if (event.type === 'preparelosingeditrights' && RTConnection.debug) {
                inPrepareLosingEditRights = true;
            }
            self.trigger(event.type, data);
        });

        // disconnect from the RT object, no further calls should be made at this object from now on
        this.registerDestructor(function () {

            // destroy realtime group
            try {
                rtGroup.destroy();
                rtGroup = null;
                RTConnection.log('RTConnection.destroy(): RT group destroyed');
            } catch (ex) {
                Utils.exception(ex);
            }

            // destroy private event hub
            eventHub.destroy();
            eventHub = null;

            // remove references
            file = traceActions = null;

            if (RTConnection.debug) {
                pendingSendAcks = null;
            }
        });

    } // class RTConnection

    // global initialization ==================================================

    // whether to show console output of this RTConnection class
    RTConnection.debug = Config.DEBUG && Config.getDebugState('realtimedebugging', 'office:log-realtime');

    // extend statically with logging methods
    Logger.extend(RTConnection, { enable: RTConnection.debug, prefix: 'RT' });

    // add time stamps for RT debugging by default
    if (RTConnection.debug && Config.getUrlFlag('office:log-timestamp', true)) {
        Logger.LOG_TIMESTAMP = true;
    }

    // enable DOM console for RT debugging by default
    if (RTConnection.debug && Config.getUrlFlag('office:debug-console', true)) {
        Logger.enableDOMConsole();
    }

    // enable logging for real-time core
    require(['io.ox/realtime/rt']).done(function (CoreRT) {
        CoreRT.debug = RTConnection.debug;
    });

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: RTConnection });

});
