/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * © 2012 Open-Xchange Inc., Tarrytown, NY, USA. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/app/fileactions',
    ['io.ox/core/extensions',
     'io.ox/core/extPatterns/links',
     'io.ox/core/api/folder',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/io',
     'io.ox/office/baseframework/app/extensionregistry',
     'gettext!io.ox/office/files'
    ], function (ext, links, FolderAPI, Utils, IO, ExtensionRegistry, gt) {

    'use strict';

    // static private functions ===============================================

    /**
     * Returns whether the files in the folder described by the passed baton
     * object are considered to be editable. The trash folder for example will
     * be treated as read-only folder.
     *
     * @param {Object} baton
     *  The baton object providing access to a folder.
     *
     * @returns {Boolean|jQuery.Promise}
     *  A Boolean value specifying whether the folder is considered editable,
     *  or a Promise that will resolve with a Boolean value, if the folder data
     *  in the baton object is not available yet.
     */
    function isEditableFolder(baton) {

        // no baton.app, no trash folder (e.g. main portal)
        if (!baton.app) { return true; }

        // check that the file is not located in the trash folder
        return baton.app.folder.getData().then(function (data) {
            return !FolderAPI.is('trash', data);
        });
    }

    /**
     * Disables the passed extension point for invalid files (all files names
     * with a specific error suffix in their extension).
     *
     * @param {String} actionId
     *  The identifier of the action to be disabled.
     */
    function disableActionForInvalidFiles(actionId) {
        new links.Action(actionId, {
            id: 'disable_action',
            index: 'first',
            requires: function (data) {
                if (ExtensionRegistry.isError(data.baton.data.filename)) {
                    data.stopPropagation();
                    return false;
                }
            }
        });
    }

    // static class FileActions ===============================================

    var FileActions = {};

    // methods ----------------------------------------------------------------

    /**
     * Creates all required actions and links in the OX Drive application for a
     * specific OX Documents application.
     *
     * @param {String} appBaseName
     *  The base name of the application (name of the application source code
     *  directory).
     *
     * @param {Object} options
     *  Optional parameters:
     *  @param {String} options.newDocumentLabel
     *      The string shown for the 'Create new document' action link inserted
     *      into the tool bar of OX Drive.
     *  @param {Number} options.newDocumentIndex
     *      The index of the 'Create new document' action link that will be
     *      inserted into the tool bar of OX Drive. The 'Create new document'
     *      action links of the various OX Document applications will be
     *      ordered increasing by this index.
     */
    FileActions.create = function (appBaseName, options) {

        var // root for extension points in OX Drive application
            ACTION_POINT = 'io.ox/files/actions/office/' + appBaseName,
            // the module name of the edit application
            MODULE_NAME = 'io.ox/office/' + appBaseName;

        // private methods ----------------------------------------------------

        /**
         * Launches a new OX Documents application with the passed options.
         */
        function launchApplication(launchOptions) {
            ox.launch(MODULE_NAME + '/main', launchOptions);
        }

        // initialization -----------------------------------------------------

        // Creates a new empty document in the current folder of the OX Drive application.
        if (ExtensionRegistry.supportsEditMode(appBaseName)) {
            new links.Action(ACTION_POINT + '/new', {
                requires: function (e) {
                	if (e.collection.has('create')) { // bug 32367
	                    return isEditableFolder(e.baton);
	                }
                },
                action: function (baton) {
                    launchApplication({ action: 'new', folderId: baton.app.folder.get() });
                }
            });
        }

        // Loads an existing document from the current file in the OX Drive application.
        // If the document requires conversion to a supported file format, creates the
        // converted file and loads it.
        new links.Action(ACTION_POINT + '/edit', {
            requires: function (e) {
                var filename = e.context.filename;
                if (e.collection.has('one', 'read', 'modify') && ExtensionRegistry.isEditable(filename, MODULE_NAME) && !ExtensionRegistry.isTemplate(filename, MODULE_NAME) && ExtensionRegistry.isNative(filename, MODULE_NAME)) {
                    // prefetch application source code
                    IO.prefetchModuleSource(MODULE_NAME);
                    return isEditableFolder(e.baton);
                }
            },
            filter: function (obj) {
                return ExtensionRegistry.isEditable(obj.filename, MODULE_NAME);
            },
            action: function (baton) {
                if (ExtensionRegistry.isNative(baton.data.filename)) {
                    launchApplication({ action: 'load', file: baton.data });
                } else {
                    launchApplication({ action: 'convert', folderId: baton.data.folder_id, templateFile: baton.data, preserveFileName: true });
                }
            }
        });

        new links.Action(ACTION_POINT + '/new-from-template', {
            requires: function (e) {
                var filename = e.context.filename;
                if (e.collection.has('one', 'read') && !ExtensionRegistry.isError(filename) && ExtensionRegistry.isTemplate(filename, MODULE_NAME)) {
                    return isEditableFolder(e.baton);
                }
            },
            filter: function (obj) {
                return ExtensionRegistry.isTemplate(obj.filename, MODULE_NAME);
            },
            action: function (baton) {
                launchApplication({ action: 'convert', template: true, folderId: baton.data.folder_id, templateFile: baton.data, initial_filename: gt('unnamed') });
            }
        });

        // Creates a new document as copy of the current file in the OX Drive application.
        if (ExtensionRegistry.supportsEditMode(appBaseName)) {
            new links.Action(ACTION_POINT + '/edit-as-new', {
                requires: function (e) {
                    var filename = e.context.filename;
                    if (e.collection.has('one', 'read') && !ExtensionRegistry.isError(filename) && ExtensionRegistry.isEditable(filename, MODULE_NAME) && !ExtensionRegistry.isTemplate(filename, MODULE_NAME)) {
                        return isEditableFolder(e.baton);
                    }
                },
                filter: function (obj) {
                    return ExtensionRegistry.isEditable(obj.filename, MODULE_NAME);
                },
                action: function (baton) {
                    var convert = ExtensionRegistry.isConvertible(baton.data.filename);
                    launchApplication({ action: convert ? 'convert' : 'new', folderId: baton.data.folder_id, templateFile: baton.data, preserveFileName: convert ? true : false });
                }
            });

            new links.Action(ACTION_POINT + '/edit-template', {
                requires: function (e) {
                    var filename = e.context.filename;
                    if (e.collection.has('one', 'read', 'modify') && !ExtensionRegistry.isError(filename) && ExtensionRegistry.isTemplate(filename, MODULE_NAME) && !ExtensionRegistry.isTemplateLocked(filename, MODULE_NAME) && ExtensionRegistry.isEditable(filename, MODULE_NAME)) {
                        return isEditableFolder(e.baton);
                    }
                },
                filter: function (obj) {
                    return ExtensionRegistry.isEditable(obj.filename, MODULE_NAME);
                },
                action: function (baton) {
                    if (ExtensionRegistry.isNative(baton.data.filename)) {
                        launchApplication({ action: 'load', file: baton.data });
                    } else {
                        launchApplication({ action: 'convert', folderId: baton.data.folder_id, templateFile: baton.data, preserveFileName: true });
                    }
                }
            });
        }

        // An action link in the main tool bar of the OX Drive application used
        // to create a new empty document.
        new links.ActionLink('io.ox/files/links/toolbar/default', {
            index: 200 + Utils.getIntegerOption(options, 'newDocumentIndex', 0),
            id: appBaseName + '_new',
            label: Utils.getStringOption(options, 'newDocumentLabel', ''),
            ref: ACTION_POINT + '/new'
        });

        // An action link in the details view of a file used to load the file.
        ext.point('io.ox/files/links/inline').extend(new links.Link({
            id: appBaseName + '_edit',
            index: 150,
            prio: 'hi',
            mobile: 'hi',
            section: 'edit',
            label: gt('Edit'),
            ref: ACTION_POINT + '/edit'
        }));

        // An action link in the details view of a file used to create a new
        // document based on that file.
        ext.point('io.ox/files/links/inline').extend(new links.Link({
            id: appBaseName + '_editasnew',
            prio: 'lo',
            mobile: 'lo',
            index: 'first',
            section: 'edit',
            label: gt('Edit as new'),
            ref: ACTION_POINT + '/edit-as-new'
        }));

        ext.point('io.ox/files/links/inline').extend(new links.Link({
            id: appBaseName + '_newfromtemplate',
            prio: 'hi',
            mobile: 'hi',
            index: 150,
            label: gt('New from template'),
            ref: ACTION_POINT + '/new-from-template'
        }));

        ext.point('io.ox/files/links/inline').extend(new links.Link({
            id: appBaseName + '_edittemplate',
            prio: 'lo',
            mobile: 'lo',
            index: 'first',
            section: 'edit',
            label: gt('Edit template'),
            ref: ACTION_POINT + '/edit-template'
        }));

    };

    // static initialization ==================================================

    disableActionForInvalidFiles('io.ox/files/actions/add-to-portal');
    disableActionForInvalidFiles('io.ox/files/actions/publish');

    // exports ================================================================

    return FileActions;

});
