/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/tableResize',
    ['io.ox/office/tk/utils',
     'io.ox/office/text/dom',
     'io.ox/office/text/operations',
     'io.ox/office/text/position',
     'io.ox/office/text/table'
    ], function (Utils, DOM, Operations, Position, Table) {

    'use strict';

    var // the names of all tracking events but 'tracking:start'
        TRACKING_EVENT_NAMES = 'tracking:move tracking:scroll tracking:end tracking:cancel',
        // the names of all tracking events
        ALL_TRACKING_EVENT_NAMES = 'tracking:start tracking:move tracking:scroll tracking:end tracking:cancel';

    // static class TableResize ==================================================

    var TableResize = {};

    // static methods ---------------------------------------------------------

    /**
     * Draws a selection box for the specified resize node and registers
     * mouse handlers for moving.
     *
     * @param {Editor} editor
     *  The editor instance to use.
     *
     * @param {jQuery} officemaindiv
     *  The main window in which the resize node will be displayed temporarely
     *
     * @param {HTMLElement|jQuery} resizeNode
     *  The selected table resize node. If this value is a jQuery
     *  collection, uses the first DOM node it contains.
     */
    TableResize.drawTableCellResizeSelection = function (editor, view, officemaindiv, resizeNode) {

        var // the node containing the scroll bar
            scrollNode = view.getContentRootNode();

        /**
         * Handling all tracking events while resizing a table.
         */
        var tableResizeHandler = (function () {

            var startX = 0,
                startY = 0,
                currentX = 0,
                currentY = 0,
                shiftX = 0,
                shiftY = 0,
                // verticalResize is true, if the column width will be modified
                verticalResize = false,
                // horizontalResize is true, if the row height will be modified
                horizontalResize = false,
                // the container element used to visualize the resizing
                resizeLine = $('<div>').addClass('resizeline'),
                // the distance from body element to 'officemaindiv' in pixel
                topDistance = null,
                // the distance from body element to 'officemaindiv' in pixel
                leftDistance = null,
                // the cell node for the selected resize node
                cellNode = null,
                // the row node for the selected resize node
                rowNode =  null,
                // the table node for the selected resize node
                tableNode = null,
                // the table node attributes object for the selected resize node
                tableNodeAttrs = null,
                // the maximum table width
                maxTableWidth = 0,
                // is the selected cell the last cell in its row
                lastCell = false,
                // logical position of the selected node
                tablePosition = [],
                // table grid before shifting column
                oldTableGrid = [],
                // table width after shifting column
                newTableWidth = 0,
                // table width before shifting column
                oldTableWidth = 'auto',
                // table grid, containing relative widths
                tableGrid = [],
                // table grid, containing calculated pixel widhts
                pixelGrid = [],
                // sum of all grid values, will not be modified
                gridSum = 0,
                // the number of the grid count, that will be modified
                shiftedGrid = 0,
                // maximum shift to the left
                maxLeftShift = 0,
                // maximum shift to the right
                maxRightShift = 0,
                // maximum shift to the top
                maxTopShift = 0,
                // maximum shift to the bottom
                maxBottomShift = 0,
                // maximum right value of shift position
                maxRightValue = 0,
                // minimum left value of shift position
                minLeftValue = 0,
                // minimum width of a column in px
                // Fix for 29678, using min width like in Word
                minColumnWidth = 16,
                //zoom factor in floating point notation
                zoomFactor = view.getZoomFactor() / 100,
                // the current scroll position
                scrollX = 0, scrollY = 0,
                // the start scroll position
                startScrollX = 0, startScrollY = 0,
                //maximum height of page content in pixels (without margins)
                pageMaxHeight = editor.getPageMaxHeight() - 5; // -5 px because of difference in rounding and offsetHeight, position top in IE, FF


            /**
             * Calculates the required data, when mouse down happens on resize node.
             *
             * @param {Event} event
             *  The event object.
             */
            function startResizeTable(event) {
                // mouse down event handler
                startX = event.pageX;
                startY = event.pageY;

                if ((! startX) && (event.originalEvent.pageX) && (event.originalEvent.pageY)) {
                    startX = event.originalEvent.pageX;
                    startY = event.originalEvent.pageY;
                }

                topDistance = officemaindiv.offset().top;
                leftDistance = officemaindiv.offset().left;

                startX = startX - leftDistance;
                startY = startY - topDistance;

                if ($(resizeNode).is('div.resize.right')) {
                    verticalResize = true;
                } else if ($(resizeNode).is('div.resize.bottom')) {
                    horizontalResize = true;
                }

                // calculating maximum resize values
                cellNode = $(resizeNode).closest('td, th');
                rowNode =  $(resizeNode).closest('tr');
                tableNode = $(resizeNode).closest('table');

                if (verticalResize) {
                    $(resizeLine).css({ width: '1px', height: '100%', left: startX, top: '0px' });
                    officemaindiv.append(resizeLine);

                    // calculating maxLeftShift and maxRightShift
                    lastCell = cellNode[0].nextSibling ? false : true;
                    tablePosition = Position.getOxoPosition(editor.getNode(), tableNode.get(0), 0);
                    tableNodeAttrs = editor.getStyleSheets('table').getElementAttributes(tableNode).table;
                    oldTableGrid = tableNodeAttrs.tableGrid;
                    oldTableWidth = tableNodeAttrs.width;
                    maxTableWidth = tableNode.parent().width();

                    if (oldTableWidth === 'auto') { oldTableWidth = tableNode.outerWidth(); }
                    else { oldTableWidth = Utils.convertHmmToLength(oldTableWidth, 'px', 1); }

                    // converting from relational grid to pixel grid
                    for (var i = 0; i < oldTableGrid.length; i++) { gridSum += oldTableGrid[i]; }
                    for (var i = 0; i < oldTableGrid.length; i++) { pixelGrid.push(Math.round(oldTableGrid[i] * oldTableWidth / gridSum)); }

                    // which border was shifted?
                    shiftedGrid = Table.getGridPositionFromCellPosition(rowNode, cellNode.prevAll().length).end;

                    // Fix for 30798, cells with lists need an increased minimum width
                    if (cellNode.find(DOM.LIST_LABEL_NODE_SELECTOR).length > 0) { minColumnWidth *= 4; }

                    maxLeftShift = pixelGrid[shiftedGrid] * zoomFactor;
                    if (! lastCell) { maxRightShift = pixelGrid[shiftedGrid + 1] * zoomFactor; }
                    else { maxRightShift = (maxTableWidth - oldTableWidth) * zoomFactor; }

                } else if (horizontalResize) {
                    $(resizeLine).css({ width: '100%', height: '1px', left: '0px', top: startY});
                    officemaindiv.append(resizeLine);
                    // calculating maxTopShift
                    maxTopShift = cellNode.outerHeight() * zoomFactor;
                    // calculating maxBottomShift - max height that single row can have
                    maxBottomShift = (pageMaxHeight - cellNode.outerHeight()) * zoomFactor;
                }

                editor.getNode().css('cursor', $(resizeNode).css('cursor'));  // setting cursor for increasing drawing
                $(resizeLine).css('cursor', $(resizeNode).css('cursor'));  // setting cursor for increasing drawing

                startScrollX = scrollNode.scrollLeft();
                startScrollY = scrollNode.scrollTop();
            }

            /**
             * Calculates the required data, when mouse move happens.
             *
             * @param {Event} event
             *  The event object.
             */
            function moveResizeTable(event) {

                // mouse move event handler
                currentX = event.pageX;
                currentY = event.pageY;

                if ((! currentX) && (event.originalEvent.pageX)) {
                    currentX = event.originalEvent.pageX;
                    currentY = event.originalEvent.pageY;
                }

                topDistance = officemaindiv.offset().top;
                leftDistance = officemaindiv.offset().left;

                currentX = currentX + scrollX - leftDistance;
                currentY = currentY + scrollY - topDistance;

                if (verticalResize) {

                    maxRightValue = startX + maxRightShift;
                    minLeftValue = startX - maxLeftShift + (minColumnWidth * zoomFactor);
                    if (! lastCell) { maxRightValue -= (minColumnWidth * zoomFactor); }

                    shiftX = currentX;
                    shiftY = 0;

                    if (shiftX >= maxRightValue) {
                        shiftX = maxRightValue;
                    } else if (shiftX <= minLeftValue) {
                        shiftX = minLeftValue;
                    }

                } else if (horizontalResize) {
                    shiftX = 0;
                    shiftY = currentY;

                    if ((shiftY - startY) <= - maxTopShift) {
                        shiftY = startY - maxTopShift;
                    }
                    if ((shiftY - startY) >= maxBottomShift) {
                        shiftY = startY + maxBottomShift;
                    }
                }

                if ((_.isNumber(shiftX)) && (_.isNumber(shiftY))) {
                    $(resizeLine).css({'left': shiftX - scrollX, 'top': shiftY - scrollY});
                }
            }

            /**
             * Updates scroll position according to the passed tracking event.
             *
             * @param {Event} event
             *  The event object.
             */
            function scrollResizeTable(event) {

                // update scrollPosition with suggestion from event
                if (event.scrollX) {
                    scrollNode.scrollLeft(scrollNode.scrollLeft() + event.scrollX);
                }

                if (event.scrollY) {
                    scrollNode.scrollTop(scrollNode.scrollTop() + event.scrollY);
                }

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }


            /**
             * Calculates the required data, when mouse up happens and generates
             * operations.
             *
             * @param {Event} event
             *  The event object.
             *
             * @param {HTMLElement} resizeNode
             *  The resize node, that will be displayed temporarely
             *
             */
            function stopResizeTable(event) {

                var generator = editor.createOperationsGenerator(),
                    rowPosition = null,
                    rowHeight = 0,
                    newRowHeight = 0;

                // taking care of scroll node
                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;

                // mouse up event handler
                currentX = event.pageX;
                currentY = event.pageY;

                if ((! currentX) && (event.originalEvent.changedTouches[0].pageX)) {
                    currentX = event.originalEvent.changedTouches[0].pageX;
                    currentY = event.originalEvent.changedTouches[0].pageY;
                }

                topDistance = officemaindiv.offset().top;
                leftDistance = officemaindiv.offset().left;

                currentX = currentX + scrollX - leftDistance;
                currentY = currentY + scrollY - topDistance;

                if (verticalResize) {

                    if ((_.isNumber(currentX)) && (currentX !== startX)) {

                        maxRightValue = startX + maxRightShift;
                        minLeftValue = startX - maxLeftShift + (minColumnWidth * zoomFactor);
                        if (! lastCell) { maxRightValue -= (minColumnWidth * zoomFactor); }

                        if (currentX >= maxRightValue) {
                            currentX = maxRightValue;
                        } else if (currentX <= minLeftValue) {
                            currentX = minLeftValue;
                        }

                        shiftX = (currentX - startX) / zoomFactor;

                        newTableWidth = lastCell ? (oldTableWidth + shiftX) : oldTableWidth;

                        // -> shifting the border
                        pixelGrid[shiftedGrid] += shiftX;
                        if (! lastCell) { pixelGrid[shiftedGrid + 1] -= shiftX; }

                        // converting modified pixel grid to new relation table grid
                        for (var i = 0; i < pixelGrid.length; i++) {
                            tableGrid.push(Math.round(gridSum * pixelGrid[i] / newTableWidth));  // only ints
                        }

                        if ((! lastCell) && (editor.getStyleSheets('table').getElementAttributes(tableNode).table.width === 'auto')) {
                            newTableWidth = 'auto';
                        } else {
                            newTableWidth = Utils.convertLengthToHmm(newTableWidth, 'px');
                        }

                        generator.generateOperation(Operations.SET_ATTRIBUTES, {
                            attrs: { table: { tableGrid: tableGrid, width: newTableWidth } },
                            start: tablePosition
                        });

                        // Performance: Mark the table with information about vertical resize, so that no full update of cell
                        // attributes is required, but only a recalculation of tab stops.
                        tableNode.data('tableResize', true);

                        editor.applyOperations(generator.getOperations());

                        tableNode.removeData('tableResize');
                    }

                } else if (horizontalResize) {

                    if ((_.isNumber(currentY)) && (currentY !== startY)) {

                        rowHeight = rowNode.outerHeight() + ((currentY - startY) / zoomFactor);

                        if (rowHeight < 0) { rowHeight = 0; }
                        if (rowHeight > pageMaxHeight) { rowHeight = pageMaxHeight; }

                        newRowHeight = Utils.convertLengthToHmm(rowHeight, 'px');
                        rowPosition = Position.getOxoPosition(editor.getNode(), rowNode.get(0), 0);

                        generator.generateOperation(Operations.SET_ATTRIBUTES, {
                            attrs: { row: { height: newRowHeight } },
                            start: rowPosition
                        });

                        editor.applyOperations(generator.getOperations());
                    }
                }

            }

            /**
             * Finalizes the tracking of the table resizing process.
             */
            function finalizeResizeTable() {
                // leaveTracking();
                $(resizeNode).off(ALL_TRACKING_EVENT_NAMES);
                view.scrollToChildNode(resizeNode);

                // removing the resize line
                officemaindiv.children('div.resizeline').remove();

                // Resetting cursor, using css file again
                editor.getNode().css('cursor', '');

                // Resetting variables
                shiftX = shiftY = scrollX = scrollY = 0;

                // removing class
                $(resizeNode).removeClass('resizeActive');
            }


            // return the actual drawingMoveHandler() method
            return function (event) {

                switch (event.type) {
                case 'tracking:start':
                    startResizeTable(event);
                    break;
                case 'tracking:move':
                    moveResizeTable(event);
                    break;
                case 'tracking:scroll':
                    scrollResizeTable(event);
                    break;
                case 'tracking:end':
                    stopResizeTable(event);
                    finalizeResizeTable(event);
                    break;
                case 'tracking:cancel':
                    finalizeResizeTable(event);
                    break;
                }
            };

        }()); // end of tableResizeHandler() local scope

        /**
         * Handler for 'tracking:start' events for table resize nodes. For this
         * node all other tracking events have to be registered.
         *
         * @param {jQuery.Event} event
         *  The 'tracking:start' event, that starts the resizing of the table.
         */
        function trackingStartHandler(event) {
            $(resizeNode).off(TRACKING_EVENT_NAMES);
            $(resizeNode).on(TRACKING_EVENT_NAMES, tableResizeHandler);

            tableResizeHandler.call(this, event);
        }

        // mark this resize node, that it is already prepared for tracking
        $(resizeNode).addClass('resizeActive');

        resizeNode
        .enableTracking({
            autoScroll: true,
            borderNode: scrollNode,
            borderMargin: -30,
            borderSize: 60,
            minSpeed: 10,
            maxSpeed: 250
        })
        .on('tracking:start', trackingStartHandler);

    };

    // exports ================================================================

    return TableResize;

});
