/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/app/controller',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/app/editcontroller',
     'io.ox/office/editframework/model/format/border',
     'io.ox/office/editframework/model/format/mixedborder',
     'io.ox/office/editframework/model/format/lineheight',
     'io.ox/office/text/app/config',
     'io.ox/office/drawinglayer/view/imageutil',
     'io.ox/office/text/hyperlink',
     'io.ox/office/text/format/listcollection',
     'io.ox/office/text/format/paragraphstyles',
     'io.ox/office/text/format/tablestyles',
     'io.ox/office/text/format/drawingstyles'
    ], function (Utils, EditController, Border, MixedBorder, LineHeight, TextConfig, Image, Hyperlink, ListCollection, ParagraphStyles, TableStyles, DrawingStyles) {

    'use strict';

    // class TextController ===================================================

    /**
     * The controller of a OX Text application.
     *
     * @constructor
     *
     * @extends EditController
     */
    function TextController(app) {

        var // self reference
            self = this,

            // the text model of the passed application
            model = null,

            // the view instance
            view = null,

            // all the little controller items
            items = {

                'document/editable/text': {
                    parent: 'document/editable',
                    enable: function () { return model.isTextSelected(); }
                },

                'document/selectall': {
                    // enabled in read-only mode
                    set: function () { model.selectAll(); },
                    focus: 'never', // do not grab focus after setter (32002)
                    // restrict keyboard shortcut to application pane
                    shortcut: { keyCode: 'A', ctrlOrMeta: true, selector: '.app-pane' }
                },

                // document contents

                'document/search/highlight': {
                    // enabled in read-only mode
                    set: function (query) { model.quickSearch(query, 0); },
                    focus: 'never' // do not grab focus after setter
                },
                'document/search/nextResult': {
                    // enabled in read-only mode
                    set: function () {
                        if (app.getWindow().search.active) {
                            model.selectNextSearchResult();
                        }
                    },
                    shortcut: [{ keyCode: 'G', ctrlOrMeta: true }, { keyCode: 'F3' }],
                    focus: 'never' // do not grab focus after setter
                },
                'document/search/previousResult': {
                    // enabled in read-only mode
                    set: function () {
                        if (app.getWindow().search.active) {
                            model.selectPreviousSearchResult();
                        }
                    },
                    shortcut: [{ keyCode: 'G', ctrlOrMeta: true, shift: true }, { keyCode: 'F3', shift: true }],
                    focus: 'never' // do not grab focus after setter
                },
                'document/search/replaceAll': {
                    parent: 'document/editable',
                    set: function () { model.searchAndReplaceAll(app.getWindow().search.query, app.getWindow().search.replacement); }
                },
                'document/search/replaceSelected': {
                    parent: 'document/editable',
                    set: function () { model.replaceSelectedSearchResult(app.getWindow().search.replacement); }
                },
                'document/search/close': {
                    // enabled in read-only mode
                    set: function () { model.removeHighlighting(); }
                },

                'document/cut': {
                    parent: 'document/editable',
                    enable: function () { return model.hasSelectedRange(); },
                    set: function () { model.cut(); }
                },
                'document/copy': {
                    // enabled in read-only mode
                    enable: function () { return model.hasSelectedRange(); },
                    set: function () { model.copy(); }
                },
                'document/paste': {
                    parent: 'document/editable',
                    enable: function () { return model.hasInternalClipboard(); },
                    set: function () { model.pasteInternalClipboard(); }
                },

                // spelling

                'document/spelling/enabled': {
                    parent: 'document/editable/text',
                    enable: function () { return TextConfig.isSpellingEnabled(); }
                },
                'document/onlinespelling': {
                    parent: 'document/spelling/enabled',
                    get: function () { return model.isOnlineSpelling(); },
                    set: function (state) { model.setOnlineSpelling(state); }
                },

                // paragraphs

                'paragraph/stylesheet': {
                    parent: 'document/editable/text',
                    get: function () { return model.getAttributes('paragraph').styleId; },
                    set: function (styleId) { model.setAttributes('paragraph', { styleId: styleId }, { clear: true }); }
                },
                'paragraph/attributes': {
                    parent: 'document/editable/text',
                    get: function () { return model.getAttributes('paragraph').paragraph || {}; }
                },
                'paragraph/alignment': {
                    parent: 'paragraph/attributes',
                    get: function (attributes) { return attributes.alignment; },
                    set: function (alignment) { model.setAttribute('paragraph', 'alignment', alignment); }
//                  removed for the 7.2.0 due to clashes with browser shortcuts
//                    shortcut: [
//                        { keyCode: 'L', ctrlOrMeta: true, value: 'left' },
//                        { keyCode: 'R', ctrlOrMeta: true, value: 'right' },
//                        { keyCode: 'E', ctrlOrMeta: true, value: 'center' },
//                        { keyCode: 'J', ctrlOrMeta: true, value: 'justify' }
//                    ]
                },
                'paragraph/lineheight': {
                    parent: 'paragraph/attributes',
                    get: function (attributes) { return attributes.lineHeight; },
                    set: function (lineHeight) { model.setAttribute('paragraph', 'lineHeight', lineHeight); }
//                  removed for the 7.2.0 due to clashes with browser shortcuts
//                    shortcut: [
//                        { keyCode: '1', ctrlOrMeta: true, value: LineHeight.SINGLE },
//                        { keyCode: '5', ctrlOrMeta: true, value: LineHeight.ONE_HALF },
//                        { keyCode: '2', ctrlOrMeta: true, value: LineHeight.DOUBLE }
//                    ]
                },
                'paragraph/fillcolor': {
                    parent: 'paragraph/attributes',
                    get: function (attributes) { return attributes.fillColor; },
                    set: function (color) { model.setAttribute('paragraph', 'fillColor', color); }
                },
                'paragraph/borders': {
                    parent: 'paragraph/attributes',
                    get: function (attributes) { return ParagraphStyles.getBorderModeFromAttributes(attributes); },
                    set: function (borderMode) { model.setAttributes('paragraph', { paragraph: ParagraphStyles.getAttributesFromBorderMode(borderMode, model.getAttributes('paragraph').paragraph) }); }
                },

                // toggle default bullet list, or select bullet type
                'paragraph/list/bullet': {
                    parent: 'paragraph/attributes',
                    get: getListStyleId,
                    set: setListStyleId,
                    userData: 'bullet'
                },
                // toggle default numbered list, or select numbering type
                'paragraph/list/numbered': {
                    parent: 'paragraph/attributes',
                    get: getListStyleId,
                    set: setListStyleId,
                    userData: 'numbering'
                },

                // parent for controller items only enabled inside lists
                'paragraph/list/enabled': {
                    parent: 'paragraph/attributes',
                    enable: function () {
                        var indent = this.getValue().listLevel;
                        return _.isNumber(indent) && (0 <= indent) && (indent <= 8);
                    }
                },
                // change list level
                'paragraph/list/indent': {
                    parent: 'paragraph/list/enabled',
                    get: function (attributes) { return attributes.listLevel; },
                    set: function (indent) { model.setAttribute('paragraph', 'listLevel', indent); }
                },
                // increase list level by one
                'paragraph/list/incindent': {
                    parent: 'paragraph/list/indent',
                    enable: function () { return this.getValue() < 8; },
                    set: function () {
                        var indent = this.getValue();
                        if (indent < 8) {
                            model.setAttribute('paragraph', 'listLevel', indent + 1);
                        }
                    }
                },
                // decrease list level by one
                'paragraph/list/decindent': {
                    parent: 'paragraph/list/indent',
                    enable: function () { return this.getValue() > 0; },
                    set: function () {
                        var indent = this.getValue();
                        if (indent > 0) {
                            model.setAttribute('paragraph', 'listLevel', indent - 1);
                        }
                    }
                },

                // characters

                'character/stylesheet': {
                    parent: 'document/editable/text',
                    get: function () { return model.getAttributes('character').styleId; },
                    set: function (styleId) { model.setAttributes('character', { styleId: styleId }, { clear: true }); }
                },
                'character/attributes': {
                    parent: 'document/editable/text',
                    get: function () { return model.getAttributes('character').character || {}; }
                },
                'character/fontname': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.fontName; },
                    set: function (fontName) { model.setAttribute('character', 'fontName', fontName); }
                },
                'character/fontsize': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.fontSize; },
                    set: function (fontSize) { model.setAttribute('character', 'fontSize', fontSize); }
                },
                'character/bold': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.bold; },
                    set: function (state) { model.setAttribute('character', 'bold', state); },
                    shortcut: { keyCode: 'B', ctrlOrMeta: true, value: function (state) { return !state; } }
                },
                'character/italic': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.italic; },
                    set: function (state) { model.setAttribute('character', 'italic', state); },
                    shortcut: { keyCode: 'I', ctrlOrMeta: true, value: function (state) { return !state; } }
                },
                'character/underline': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.underline; },
                    set: function (state) { model.setAttribute('character', 'underline', state); },
                    shortcut: { keyCode: 'U', ctrlOrMeta: true, value: function (state) { return !state; } }
                },
                'character/strike': {
                    parent: 'character/attributes',
                    get: function (attributes) { return _.isString(attributes.strike) ? (attributes.strike !== 'none') : null; },
                    set: function (state) { model.setAttribute('character', 'strike', state ? 'single' : 'none'); }
                },
                'character/vertalign': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.vertAlign; },
                    set: function (align) { model.setAttribute('character', 'vertAlign', align); }
                },
                'character/color': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.color; },
                    set: function (color) { model.setAttribute('character', 'color', color); }
                },
                'character/fillcolor': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.fillColor; },
                    set: function (color) { model.setAttribute('character', 'fillColor', color); }
                },
                'character/language': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.language; },
                    set: function (language) { model.setAttribute('character', 'language', language); }
                },
                'character/hyperlink': {
                    parent: 'character/attributes',
                    enable: function () { return model.hasEnclosingParagraph(); },
                    get: function (attributes) { return attributes.url; }
                    // no direct setter (see item 'character/hyperlink/dialog')
                },
                'character/hyperlink/dialog': {
                    parent: 'character/hyperlink',
                    get: function () { return Hyperlink.hasPopup(model); },
                    set: function () { return model.insertHyperlinkDialog(); },
                    // removed for the 7.2.0 due to clashes with browser shortcuts
//                    shortcut: { keyCode: 'K', ctrlOrMeta: true },
                    focus: 'wait' // wait for the dialog before returning focus to application
                },
                'character/hyperlink/remove': {
                    parent: 'character/hyperlink',
                    set: function () { return model.removeHyperlink(); }
                },

                'character/resetAttributes': {
                    parent: 'document/editable/text',
                    set: function () { model.resetAttributes(); },
                    shortcut: { keyCode: 'SPACE', ctrl: true }
                },
                'character/insertLineBreak': {
                    parent: 'document/editable/text',
                    set: function () { return model.insertHardBreak(); },
                    focus: 'wait'
                },
                'character/insertTab': {
                    parent: 'document/editable/text',
                    set: function () { return model.insertTab(); },
                    focus: 'wait'
                },

                // tables

                'table/insert': {
                    parent: 'document/editable/text',
                    set: function (size) { model.insertTable(size); }
                },

                'document/editable/table': {
                    parent: 'document/editable/text',
                    enable: function () { return model.isPositionInTable(); }
                },
                'table/insert/row': {
                    parent: 'document/editable/table',
                    set: function () { model.insertRow(); }
                },
                'table/insert/column': {
                    parent: 'document/editable/table',
                    set: function () { model.insertColumn(); }
                },
                'table/delete/row': {
                    parent: 'document/editable/table',
                    set: function () { return model.deleteRows(); },
                    focus: 'wait'
                },
                'table/delete/column': {
                    parent: 'document/editable/table',
                    set: function () { return model.deleteColumns(); },
                    focus: 'wait'
                },

                'table/stylesheet': {
                    parent: 'document/editable/table',
                    get: function () { return model.getAttributes('table').styleId; },
                    set: function (styleId) { model.setAttributes('table', { styleId: styleId }, { clear: true }); }
                },
                'table/attributes': {
                    parent: 'document/editable/table',
                    get: function () { return model.getAttributes('table').table || {}; }
                },
                'table/cellattributes': {
                    parent: 'document/editable/table',
                    get: function () { return model.getAttributes('cell').cell || {}; }
                },
                'table/cellborder': {
                    parent: 'table/attributes',
                    get: function (attributes) { return MixedBorder.getBorderMode(attributes); },
                    set: function (borderMode) { model.setAttributes('table', { table: MixedBorder.getBorderAttributes(borderMode, model.getAttributes('table').table) }, { cellSpecificTableAttribute: true }); }
                },
                'table/borderwidth': {
                    parent: 'table/cellattributes',
                    get: function (attributes) { return TableStyles.getBorderStyleFromAttributes(attributes); },
                    set: function (borderWidth) { model.setAttributes('table', { table: TableStyles.getAttributesFromBorderStyle(borderWidth, model.getAttributes('table').table) }, { onlyVisibleBorders: true }); }
                },
                'table/fillcolor': {
                    parent: 'table/cellattributes',
                    get: function (attributes) { return attributes.fillColor; },
                    set: function (color) { model.setAttribute('cell', 'fillColor', color); }
                },

                // drawing

                'document/editable/drawing': {
                    parent: 'document/editable',
                    enable: function () { return model.isDrawingSelected(); }
                },
                'drawing/delete': {
                    parent: 'document/editable/drawing',
                    set: function () { return model.deleteSelected(); },
                    focus: 'wait' // wait for the dialog before returning focus to application
                },

                'drawing/attributes': {
                    parent: 'document/editable/drawing',
                    get: function () { return model.getAttributes('drawing').drawing || {}; }
                },
                'drawing/position': {
                    parent: 'drawing/attributes',
                    get: function (attributes) { return DrawingStyles.getPositionFromAttributes(attributes); },
                    set: function (position) { model.setAttributes('drawing', { drawing: DrawingStyles.getAttributesFromPosition(position, model.getAttributes('drawing').drawing) }); }
                },

                // images

                'image/insert/dialog': {
                    parent: 'document/editable/text',
                    set: function () { return model.showInsertImageDialog(); },
                    focus: 'wait' // wait for the dialog before returning focus to application
                },

                // debug

                'format/spellchecking' : {
                    parent: 'character/attributes',
                    enable: function () { return model.hasSelectedRange() && model.hasEnclosingParagraph(); },
                    set: function () { model.checkSpelling(); }
                },
                'debug/accessrights' : {
                    parent: 'document/editable/text',
                    enable: function () { return model.hasSelectedRange(); },
                    set: function (mail) { model.accessRightsExport(mail); }
                },
                'debug/recordoperations' : {
                    get: function () { return model.isRecordingOperations(); },
                    set: function (state) { model.setRecordingOperations(state); }
                },
                'debug/replayoperations' : {
                    parent: 'document/editable',
                    enable: function () { return app.getView().getDebugOperationsContent() !== null; },
                    set: function () { model.replayOperations(app.getView().getDebugOperationsContent()); }
                },
                'debug/operations' : {
                    parent: 'debug/enabled'
                },
                'debug/getpagehtmlcode' : {
                    set: function () { model.getFullModelDescription().done(function (docString) { app.getView().setPageHtmlCode(docString); }); }
                },
                'debug/setpagehtmlcode' : {
                    parent: 'document/editable',
                    enable: function () { return app.getView().getPageHtmlCode() !== null; },
                    set: function () { model.setFullModelNode(app.getView().getPageHtmlCode()); }
                },
                'debug/pagehtmlcode' : {
                    parent: 'debug/enabled'
                },
                'debug/togglepagebreaks' : {
                    set: function () { view.setPageBreaks(); }
                },

                //zoom

                'zoom/dec': {
                    enable: function () { return view.getZoomFactor() > view.getMinZoomFactor(); },
                    set: function () { view.decreaseZoomLevel(); }
                },

                'zoom/inc': {
                    enable: function () { return view.getZoomFactor() < view.getMaxZoomFactor(); },
                    set: function () { view.increaseZoomLevel(); }
                },

                'zoom/type': {
                    get: function () { return view.getZoomType(); },
                    set: function (zoomType) { view.setZoomType(zoomType); }
                }
            };

        // base constructor ---------------------------------------------------

        EditController.call(this, app, { updateDelay: 100 });

        // private methods ----------------------------------------------------

        /**
         * Item getter function for a list style identifier. Will be called in
         * the context of a controller item instance.
         *
         * @param {Object} attributes
         *  The merged attribute set of the selected paragraphs.
         *
         * @returns {String|Null}
         *  If unambiguous, the list style identifier contained in the passed
         *  paragraph attributes, if the type of that list matches the list
         *  type specified in the item user data ('bullets' or 'numbering'),
         *  otherwise an empty string. If ambiguous, returns the value null.
         */
        function getListStyleId(attributes) {

            var listStyleId = attributes.listStyleId,
                listLevel = attributes.listLevel,
                itemListType = this.getUserData(),
                isListType = false;

            // ambiguous attributes
            if (_.isNull(listStyleId) || _.isNull(listLevel)) { return null; }

            if (listLevel > -1) {
                switch (itemListType) {
                case 'bullet':
                    isListType = model.getListCollection().isBulletsList(listStyleId, listLevel);
                    break;
                case 'numbering':
                    isListType = model.getListCollection().isNumberingList(listStyleId, listLevel);
                    break;
                }
            }

            // return empty string, if list type in attributes does not match list type in item user data
            return isListType ? listStyleId : '';
        }

        /**
         * Item setter function for a list style identifier. Will be called in
         * the context of a controller item instance.
         *
         * @param {String} listStyleId
         *  The new list style identifier. The special value
         *  Lists.DEFAULT_VALUE can be used to toggle the default list style.
         */
        function setListStyleId(listStyleId) {

            var itemListType = this.getUserData();

            // simulate toggle behavior for default list style
            if (listStyleId === ListCollection.DEFAULT_VALUE) {
                if (this.getValue() === '') {
                    model.createDefaultList(itemListType);
                } else {
                    model.removeListAttributes();
                }
            } else {
                // list level may be null, will fall back to level 0 then...
                model.createSelectedListStyle(listStyleId, self.getItemValue('paragraph/list/indent'));
            }
        }

        // initialization -----------------------------------------------------

        // register item definitions
        this.registerDefinitions(items);

        // initialization after construction
        app.on('docs:init', function () {
            // model and view may not be available at construction time
            model = app.getModel();
            view = app.getView();
            // update GUI after changed selection
            model.on('selection', function () { self.update(); });
        });

    } // class TextController

    // exports ================================================================

    // derive this class from class EditController
    return EditController.extend({ constructor: TextController });

});
