/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/logger', ['io.ox/office/tk/config'], function (Config) {

    'use strict';

    // class Logger ===========================================================

    /**
     * Provides methods to create logging messages in the browser console.
     * Logging will always be disabled without global debug mode specified in
     * the server-side configuration option 'debugavailable'. If the
     * application URL contains the anchor option 'office:log-timestamp=true',
     * all log message will be prepended with the current system time.
     *
     * @constructor
     *
     * @param {Object} [initOptions]
     *  Additional optional parameters:
     *  @param {Boolean|String} [initOptions.enable=true]
     *      Specifies whether the logger is enabled. If set to a string, the
     *      application URL will be checked for an according anchor option set
     *      to the value 'true'.
     *  @param {String} [initOptions.prefix]
     *      If specified, all messages will be prefixed with the passed text
     *      (enclosed in bracket characters).
     */
    function Logger(initOptions) {

        var // the 'enable' option passed to the constructor (defaults to true)
            enableOpt = (_.isObject(initOptions) && ('enable' in initOptions)) ? initOptions.enable : true,

            // whether the logger is enabled
            enabled = Config.isDebug() && (_.isString(enableOpt) ? (_.url.hash(enableOpt) === 'true') : (_.isBoolean(enableOpt) && enableOpt)),

            // the prefix printed before each message
            prefix = (_.isObject(initOptions) && _.isString(initOptions.prefix)) ? ('[' + initOptions.prefix  + ']') : null;

        // private methods ----------------------------------------------------

        /**
         * Writes a message to the browser output console.
         *
         * @param {String} level
         *  The log level of the message. The string 'log' will create a
         *  generic log message, the string 'info' will create an information,
         *  the string 'warn' will create a warning message, and the string
         *  'error' will create an error message.
         */
        function log(level, args) {

            // IE9: console missing if debugger is not running
            if (!_.isObject(window.console)) { return; }

            // check that the browser console supports the operation
            if (prefix) {
                args.splice(0, 0, prefix);
            }
            if (!_.isFunction(window.console[level])) {
                args.splice(0, 0, level.toUpperCase());
                level = 'log';
            }
            if (Logger.LOG_TIMESTAMP) {
                var time = new Date();
                args.splice(0, 0, time.toLocaleTimeString() + '.' + ('00' + (time.getTime() % 1000)).slice(-3));
            }
            if (_.isFunction(window.console[level])) {
                window.console[level].apply(window.console, args);
            } else if (_.isNumber(_.browser.IE) && (_.browser.IE < 10)) {
                window.console.log(args.join(' '));
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Writes a log message to the browser output console, if debug mode is
         * enabled in the global configuration.
         *
         * @param {Any} [...]
         *  The values to be written to the console.
         */
        this.log = enabled ? function () { log('log', _.toArray(arguments)); } : $.noop;

        /**
         * Writes an info message to the browser output console, if debug mode is
         * enabled in the global configuration.
         *
         * @param {Any} [...]
         *  The values to be written to the console.
         */
        this.info = enabled ? function () { log('info', _.toArray(arguments)); } : $.noop;

        /**
         * Writes a warning message to the browser output console, if debug mode is
         * enabled in the global configuration.
         *
         * @param {Any} [...]
         *  The values to be written to the console.
         */
        this.warn = enabled ? function () { log('warn', _.toArray(arguments)); } : $.noop;

        /**
         * Writes an error message to the browser output console, if debug mode is
         * enabled in the global configuration.
         *
         * @param {Any} [...]
         *  The values to be written to the console.
         */
        this.error = enabled ? function () { log('error', _.toArray(arguments)); } : $.noop;

        /**
         * Writes an error message to the browser output console describing the
         * passed exception object, if debug mode is enabled in the global
         * configuration.
         *
         * @param {Any} exception
         *  The exception as caught in a try/catch.
         *
         * @param {Any} [...]
         *  Additional values to be written to the console.
         */
        this.exception = enabled ? function (exception) {

            var // the stack contained in the exception object
                stack = _.isObject(exception) && (exception.stack || exception.stacktrace);

            log('error', ['Exception caught:', exception].concat(_.toArray(arguments).slice(1)));
            if (stack) { log('error', ['Stack trace:', stack]); }

        } : $.noop;

    } // class Logger

    // constants --------------------------------------------------------------

    // whether to add a time stamp to all messages
    Logger.LOG_TIMESTAMP = _.url.hash('office:log-timestamp') === 'true';

    // static methods ---------------------------------------------------------

    /**
     * Extends the passed object or class with logging functionality.
     *
     * @returns {Object|Function}
     *  An object that will be extended with the logging methods provided by
     *  the Logger class. Can be a class constructor function, in this case the
     *  class will contain static logging methods.
     *
     * @returns {Object|Function}
     *  A reference to the passed and extended object or class.
     */
    Logger.extend = function (obj) {
        Logger.apply(obj, _.toArray(arguments).slice(1));
        return obj;
    };

    // exports ================================================================

    return Logger;

});
