/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/io',
    ['io.ox/core/http',
     'io.ox/office/tk/utils'
    ], function (http, Utils) {

    'use strict';

    var // static cache for JSON resources already loaded, mapped by resource module name
        resourceCache = {};

    // static class IO ========================================================

    /**
     * Provides static methods for server/client communication and host file
     * system access.
     */
    var IO = {};

    // public methods ---------------------------------------------------------

    /**
     * Creates a function that returns a Deferred object which will be resolved
     * or rejected depending on the result of the passed filter callback
     * function. Intended to be used in Deferred chains to modify the state of
     * a Deferred object depending on the result data.
     *
     * @param {Function} filter
     *  A function that receives and filters the response of a resolved
     *  Deferred object as first parameter. If this function returns undefined,
     *  the Deferred object created by the method will be rejected. Otherwise,
     *  the Deferred object will be resolved with the return value of this
     *  filter function.
     *
     * @param {Object} [context]
     *  The context object that will be bound to the passed filter function.
     *
     * @returns {Function}
     *  A new function that returns a Deferred object that has been resolved or
     *  rejected depending on the result of the passed filter callback
     *  function.
     */
    IO.createDeferredFilter = function (filter, context) {

        // create and return a function that call the filter and returns a resolved/rejected Deferred object
        return function (response) {

            var // the result Deferred object
                def = $.Deferred(),
                // call passed filter callback
                value = _.isObject(response) ? filter.call(context, response) : undefined;

            // resolve the Deferred object, if the value returned by the filter function is not undefined
            return _.isUndefined(value) ? def.reject() : def.resolve(value);
        };
    };

    /**
     * Sends a request to the server and returns the promise of a Deferred
     * object waiting for the response.
     *
     * @param {Object} options
     *  Additional options. The following options are supported:
     *  @param {String} [options.method='GET']
     *      The request method. Must be an upper-case string (for example,
     *      'GET', 'POST', etc.). Defaults to 'GET'.
     *  @param {String} options.module
     *      The name of the server module that will receive the request.
     *  @param {Object} [options.params]
     *      Parameters that will be inserted into the request URL (method GET),
     *      or into the request body (method POST).
     *  @param {Function} [options.resultFilter]
     *      A function that will be called if the request returns successfully,
     *      and filters the resulting 'data' object returned by the request.
     *      Receives the 'data' object as first parameter. If this function
     *      returns undefined, the entire request will be rejected. Otherwise,
     *      the request will be resolved with the return value of this function
     *      instead of the complete 'data' object.
     *
     * @returns {jQuery.Promise}
     *  The promise of the request. Will be resolved with the 'data' object
     *  returned by the response, if available; or the valid return value of
     *  the result filter callback function, if specified. Otherwise, the
     *  promise will be rejected. Contains the additional method 'abort()' that
     *  allows to abort the running request which rejects the promise. Calling
     *  this method has no effect, if the request is finished already.
     */
    IO.sendRequest = function (options) {

        var // extract the request method
            method = Utils.getStringOption(options, 'method', 'GET'),
            // extract the result filter callback
            resultFilter = Utils.getFunctionOption(options, 'resultFilter'),
            // the Deferred object representing the core AJAX request
            ajaxRequest = null,
            // the Promise returned by this method
            promise = null;

        // remove the internal options
        delete options.method;
        delete options.resultFilter;

        // send the AJAX request
        ajaxRequest = http[method](options);

        // reject, if the response contains 'hasErrors:true'
        promise = ajaxRequest.then(function (response) {
            return Utils.getBooleanOption(response, 'hasErrors', false) ? $.Deferred().reject(response) : response;
        });

        // filter the result of the original request according to the passed filter callback
        if (_.isFunction(resultFilter)) {
            promise = promise.then(IO.createDeferredFilter(resultFilter));
        }

        // add an abort() method, forward invocation to AJAX request
        return _(promise).extend({ abort: function () { ajaxRequest.abort(); } });
    };

    /**
     * Reads the specified file and returns the promise of a Deferred object
     * that will be resolved or rejected depending on the result of the read
     * operation. The file will be converted to a data URL containing the file
     * contents as Base-64 encoded data and passed to the resolved Promise.
     *
     * @param {File} fileDesc
     *  The descriptor of the file to be loaded.
     *
     * @returns {jQuery.Promise}
     *  The Promise of a Deferred object that will be resolved with the result
     *  object containing the data URL, or rejected if the read operation
     *  failed. If the file size is known, the Promise will be notified about
     *  the progress of the operation (a floating-point number between 0.0 and
     *  1.0). Contains the additional method 'abort()' that allows to abort the
     *  running file reader which rejects the promise. Calling this method has
     *  no effect, if importing the file is finished already.
     */
    IO.readClientFileAsDataUrl = function (fileDesc) {

        var // Deferred result object
            def = $.Deferred(),
            // create a browser file reader instance
            reader = window.FileReader ? new window.FileReader() : null;

        if (reader) {

            // register the load event handler, Deferred object will be resolved with data URL
            reader.onload = function (event) {
                if (event && event.target && _.isString(event.target.result)) {
                    def.resolve(event.target.result);
                } else {
                    def.reject();
                }
            };

            // register error event handlers, Deferred object will be rejected
            reader.onerror = function () {
                def.reject();
            };

            // register abort event handlers, Deferred object will be rejected
            reader.onabort = function () {
                def.reject('abort');
            };

            // register progress handler, Deferred object will be notified
            reader.onprogress = function (event) {
                if (event.lengthComputable) {
                    def.notify(event.loaded / event.total);
                }
            };

            // forget reference to the file reader after import
            def.always(function () { reader = null; });

            // read the file and generate a data URL
            reader.readAsDataURL(fileDesc);

        } else {
            // file reader not supported
            def.reject();
        }

        // add an abort() method, forward invocation to AJAX request
        return _(def.promise()).extend({
            abort: function () {
                if (reader) { reader.abort(); }
            }
        });
    };

    /**
     * Loads the specified JSON resource module for the current GUI locale. If
     * there is no resource module available for the locale, tries to fall back
     * to the plain language code, then to the locale 'en_US', then to the
     * language code 'en'. All JSON resources loaded once will be cached
     * internally, and will be returned immediately when calling this method
     * again.
     *
     * @param {String} modulePath
     *  The module path to the directory containing the JSON resource modules.
     *  The resource modules must be stored as RequireJS module source file
     *  '[modulePath]/[LOCALE].json'.
     *
     * @returns {jQuery.Promise}
     *  The Promise of a Deferred object that will be resolved with the JSON
     *  object loaded from the specified resource file, or that will be
     *  rejected on any error (e.g., resource file not found, or invalid JSON
     *  data).
     */
    IO.loadResource = function (modulePath) {

        // tries to load the resource file for the passed locale
        function requireResource(locale) {
            return require(['text!' + modulePath + '/' + locale + '.json']);
        }

        // the (pending, resolved, or rejected) Promise is stored in the cache
        if (!(modulePath in resourceCache)) {

            resourceCache[modulePath] =
                // first, try to load the resource module for the current locale
                requireResource(Utils.LOCALE)
                // on error, fall back to the language code without region code
                .then(null, function (response) {
                    return (Utils.LOCALE === Utils.LANGUAGE) ? response : requireResource(Utils.LANGUAGE);
                })
                // on error, fall back to en_US locale (unless current locale is already en_US)
                .then(null, function (response) {
                    return (Utils.LOCALE === 'en_US') ? response : requireResource('en_US');
                })
                // on error, fall back to English (unless current language is already English)
                .then(null, function (response) {
                    return (Utils.LANGUAGE === 'en') ? response : requireResource('en');
                })
                // on success (either current locale/language or English), try to parse the JSON string
                .then(function (response) {
                    if (!_.isString(response) || (response.length === 0)) {
                        Utils.error('IO.loadResource(): invalid response');
                        return $.Deferred().reject(response);
                    }
                    try {
                        return JSON.parse(response);
                    } catch (ex) {
                        Utils.exception(ex);
                        return $.Deferred().reject(ex);
                    }
                });
        }

        // return the Promise containing the localized resource object
        return resourceCache[modulePath];
    };

    // exports ================================================================

    return IO;

});
