/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/format/paragraphstyles',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/color',
     'io.ox/office/editframework/model/format/border',
     'io.ox/office/editframework/model/format/lineheight',
     'io.ox/office/editframework/model/format/stylesheets',
     'io.ox/office/text/dom'
    ], function (Utils, Color, Border, LineHeight, StyleSheets, DOM) {

    'use strict';

    var // definitions for paragraph attributes
        DEFINITIONS = {

            nextStyleId: {
                def: '',
                scope: 'style'
            },

            alignment: {
                def: 'left',
                format: function (element, value) {
                    element.css('text-align', value);
                },
                preview: true // 'true' calls the own 'format' method
            },

            fillColor: {
                def: Color.AUTO, // auto for paragraph fill resolves to 'transparent'
                format: function (element, color) {
                    element.css('background-color', this.getCssColor(color, 'fill'));
                }
            },

            /**
             * Line height relative to font settings. The CSS attribute
             * 'line-height' must be set separately at every descendant text
             * span because a relative line height (e.g. 200%) would not be
             * derived from the paragraph relatively to the spans, but
             * absolutely according to the paragraph's font size. Example: The
             * paragraph has a font size of 12pt and a line-height of 200%,
             * resulting in 24pt. This value will be derived absolutely to a
             * span with a font size of 6pt, resulting in a relative line
             * height of 24pt/6pt = 400% instead of the expected 200%.
             */
            lineHeight: { def: LineHeight.SINGLE },

            listLevel: { def: -1 },

            listStyleId: { def: '' },

            listLabelHidden: { def: false },

            listStartValue: { def: -1 },

            outlineLevel: { def: 9 },

            tabStops: {
                def: [],
                merge: function (tabStops1, tabStops2) {
                    // Merge tabStops2 into array tabstop1
                    // Support to clear tab stops defined in tabStops1
                    var clearedTabstops = _.filter(tabStops2, function (tabstop) {
                            return tabstop.value === 'clear';
                        }),
                        additionalTabstops = _.filter(tabStops2, function (tabstop) {
                            return tabstop.value !== 'clear';
                        }),
                        newTabstops = tabStops1 ? (additionalTabstops ? tabStops1.concat(additionalTabstops) : tabStops1) : (additionalTabstops ? additionalTabstops : []);

                    // Make entries in newTabstops unique
                    newTabstops = _.unique(newTabstops, false, function (tabstop) {
                        return tabstop.pos;
                    });

                    // Filter out cleared tabStops
                    if (clearedTabstops.length > 0) {
                        newTabstops = _.filter(newTabstops, function (tabstop) {
                            // return only tab stops which are not part of the clearedTabstops array
                            return (_.find(clearedTabstops, function (cleared) {
                                return cleared.pos === tabstop.pos;
                            }) === undefined);
                        });
                    }

                    // Sort tabStops by position
                    return _.sortBy(newTabstops, function (tabstop) {
                        return tabstop.pos;
                    });
                }
            },

            borderLeft: { def: Border.NONE },

            borderRight: { def: Border.NONE },

            borderTop: { def: Border.NONE },

            borderBottom: { def: Border.NONE },

            borderInside: { def: Border.NONE },

            indentFirstLine: { def: 0 },

            indentLeft: { def: 0 },

            indentRight: { def: 0 },

            marginTop: { def: 0 },

            marginBottom: { def: 0 },

            contextualSpacing: { def: false }

        },

        // parent families with parent element resolver functions
        PARENT_RESOLVERS = {
            cell: function (paragraph) { return DOM.isCellContentNode(paragraph.parent()) ? paragraph.closest('td') : null; }
        },

        // maps fill character attribute values to fill characters
        TAB_FILL_CHARS = { dot: '.', hyphen: '-', underscore: '_' },

        // the names of all border properties in paragraphs, mapped by border mode properties
        PARAGRAPH_BORDER_ATTRIBUTES = { left: 'borderLeft', right: 'borderRight', top: 'borderTop', bottom: 'borderBottom', insideh: 'borderInside' };

    // private static functions ===============================================

    function isMergeBorders(attributes1, attributes2) {
        return (Border.isVisibleBorder(attributes1.borderLeft) ||
            Border.isVisibleBorder(attributes1.borderRight) ||
            Border.isVisibleBorder(attributes1.borderTop) ||
            Border.isVisibleBorder(attributes1.borderBottom) ||
            Border.isVisibleBorder(attributes1.borderInside)) &&
            Utils.hasEqualProperties(attributes1, attributes2,
                ['borderLeft', 'borderRight', 'borderTop', 'borderBottom', 'borderInside',
                 'listStyleId', 'listLevel', 'indentLeft', 'indentRight', 'indentFirstLine']);
    }

    /**
     * Fills the passed text span with a sufficient number of the specified
     * fill character.
     *
     * @param {jQuery} spanNode
     *  The span node to be filled. The current CSS formatting of this node is
     *  used to calculate the number of fill characters needed.
     *
     * @param {String} fillChar
     *  The fill character.
     *
     * @param {Number} width
     *  The target width of the tabulator, in 1/100 of millimeters.
     */
    function insertTabFillChar(spanNode, fillChar, width) {

        var // 5 fill characters, used to calculate average character width
            checkString = Utils.repeatString(fillChar, 5),
            // average character width, in 1/100 mm
            charWidth = Utils.convertLengthToHmm(spanNode.contents().remove().end().text(checkString).width(), 'px') / 5,
            // number of characters needed to fill the specified width
            charCount = Math.floor(width / charWidth),
            // the fill character, repeated by the calculated number
            fillString = (charCount > 0) ? Utils.repeatString(fillChar, charCount) : '\u00a0',
            // a shortened string, if element is too wide
            shortFillString = null;

        // insert the fill string into the element
        spanNode.contents().remove().end().text(fillString);
        if (! fillString) { DOM.ensureExistingTextNode(spanNode); }

        // shorten fill string by one character, if element is too wide (e.g. due to rounding errors)
        if ((fillString.length > 1) && (Utils.convertLengthToHmm(spanNode.width(), 'px') >= width)) {
            shortFillString = fillString.slice(0, -1);
            spanNode.contents().remove().end().text(shortFillString);
            if (! shortFillString) { DOM.ensureExistingTextNode(spanNode); }
        }
    }

    // class ParagraphStyles ==================================================

    /**
     * Contains the style sheets for paragraph formatting attributes. The CSS
     * formatting will be read from and written to paragraph <p> elements.
     *
     * @constructor
     *
     * @extends StyleSheets
     *
     * @param {TextApplication} app
     *  The root application instance.
     *
     * @param {TextDocumentStyles} documentStyles
     *  Collection with the style containers of all style families.
     */
    function ParagraphStyles(app, documentStyles) {

        var // self reference
            self = this,
            // the view instance
            view = null;

        // base constructor ---------------------------------------------------

        StyleSheets.call(this, app, documentStyles, 'paragraph', {
            additionalFamilies: 'character',
            parentResolvers: PARENT_RESOLVERS,
            formatHandler: updateParagraphFormatting,
            previewHandler: setParagraphPreviewFormatting
        });

        // private methods ----------------------------------------------------

        /**
         * Updates all tabulators in the passed paragraph node. This method
         * uses both the default tab size and existing paragraph tab stop
         * definitions. Fill characters are also supported.
         *
         * @attention
         *  Currently, only left&right aligned tabs are supported!
         *
         * @param {jQuery} paragraph
         *  The paragraph node whose tabulator nodes will be updated, as jQuery
         *  object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateTabStops(paragraph, mergedAttributes) {

            var // default tab stop width from document settings
                defaultTabStop = documentStyles.getAttributes().defaultTabStop,
                // paragraph tab stop definitions
                paraTabStops = mergedAttributes.paragraph.tabStops,
                // effective left margin of the paragraph
                marginLeft = Utils.convertCssLengthToHmm(paragraph.css('margin-left'), 'px'),
                // end position of previous tabulator
                prevTabEndPos = null,
                // first node of paragraph
                firstNode = null,
                // top position of first node
                topFirst = 0,
                // height of first node
                heightFirst = 0,
                // leftmost position used a base to calculate tab raster
                zeroLeft = 0,
                // first line  indent
                indentFirstLine = mergedAttributes.paragraph.indentFirstLine,
                // intersect function
                intersect = function (xs1, xe1, xs2, xe2) {
                    return ((xs1 <= xs2) && (xe1 >= xs2) ||
                             (xs2 <= xs1) && (xe2 >= xs1));
                },
                // active paragraph tab stops
                activeParaTabStops = null,
                // zoom factor in floating point notation
                zoomFactor = view.getZoomFactor() / 100;

            if (paragraph.length) {
                // Calculate top position and height of first paragraph child. Our paragraph
                // have always at least one child element.
                firstNode = $(paragraph.get(0).firstChild);
                topFirst = firstNode.length ? Utils.convertLengthToHmm(firstNode.position().top, 'px') : 0;
                heightFirst = firstNode.length ? Utils.convertLengthToHmm(firstNode.height(), 'px') : 0;

                paragraph.children(DOM.TAB_NODE_SELECTOR).each(function () {

                    var // the current tab node of the paragraph
                        tabNode = $(this),
                        // the span within the tab node containing the fill characters
                        tabSpan = $(this.firstChild),
                        // the position of the tab node
                        pos = tabNode.position(),
                        // the left bound position of the tab node in 1/100th mm
                        leftHMM = 0,
                        // the top bound position of the tab node in 1/100th mm
                        topHMM = Utils.convertLengthToHmm(pos.top, 'px'),
                        // the top bound position of the first child node of the paragraph in 1/100th mm
                        topSpanHMM = 0,
                        // the height of the first child node of the paragraph in 1/100th mm
                        heightSpanHMM = 0,
                        // the calculated width of the tab stop
                        width = 0,
                        // the fill character of the tab stop
                        fillChar = null,
                        // the tab stop values
                        tabStop,
                        // ignore the tab position and continue with the next one
                        ignore = false,
                        // tab stop type
                        tabStopType = 'left',
                        // indent first line tab stop position
                        indentFirstLineTabStopValue = 0,
                        // first line tab stop
                        indentFirstLineTabStop = null,
                        // insert index for virtual first line indent tab stop
                        indentFirstLineTabStopIndex = 0;

                    if (mergedAttributes.paragraph.alignment === 'center') {
                        // for a centered paragraph tab stops are calculated where the first character is treated as leftmost (0) position
                        zeroLeft = firstNode.length ? -(marginLeft + Utils.convertLengthToHmm(firstNode.position().left, 'px')) : -marginLeft;
                    } else {
                        zeroLeft = marginLeft;
                    }

                     // calculate left bound position of the tab node in 1/100th mm, including zoom factor for browsers using transf.scale
                    leftHMM = zeroLeft + Utils.convertLengthToHmm(pos.left / zoomFactor, 'px');

                    if (prevTabEndPos) {
                        // Check to see if we are in a chain of tabs. Force to use the previous tab end position
                        // as start position. Some browsers provide imprecise positions therefore we use the correct
                        // previous position from the last calculation.

                        // Check previous node and that text is empty - this is a precondition for the chaining
                        // code.
                        if (tabNode.prev().length && tabNode.prev().text().length === 0) {
                            var checkPrevPrevNode = tabNode.prev().length ? tabNode.prev().prev() : null;
                            if (checkPrevPrevNode.length && DOM.isTabNode(checkPrevPrevNode) && prevTabEndPos.top === topHMM)
                                leftHMM = prevTabEndPos.right;
                        }
                    }

                    // Special case for first line indent. For the first line the negative indent
                    // must be used as a tab stop position using the absolute value. We need to set
                    // at least one character for the tabSpan to have a valid width otherwise we
                    // won't get a correct height which is needed for the check.
                    if (indentFirstLine < 0) {
                        indentFirstLineTabStopValue = Math.abs(indentFirstLine);
                        // Fix for 29265 and 30847: Removing empty text node in span with '.contents().remove().end()'.
                        // Otherwise there are two text nodes in span after '.text('abc')' in IE.
                        tabSpan.contents().remove().end().text('\u00a0');
                        heightSpanHMM = Utils.convertLengthToHmm(tabSpan.height(), 'px');
                        topSpanHMM = Utils.convertLengthToHmm(tabSpan.position().top, 'px');
                        // checkout if the first line indent is active for this special tab stop
                        if (intersect(topFirst, topFirst + heightFirst, topSpanHMM, topSpanHMM + heightSpanHMM) &&
                            (leftHMM + 10) < indentFirstLineTabStopValue) {
                            width = Math.max(0, indentFirstLineTabStopValue - leftHMM);
                        }

                        if (width > 1) {
                            // find index for the first line indent position within the paragraph tab stop array
                            indentFirstLineTabStopIndex = Utils.findLastIndex(paraTabStops, function (tab) {
                                return (indentFirstLineTabStopValue > tab.pos);
                            }) + 1;
                            // create a copy of the paragraph tab stops and add the first indent tab stop at
                            // the correct position to the active paragraph tab stop array
                            indentFirstLineTabStop = {value: 'left', pos: indentFirstLineTabStopValue, fillChar: '\u00a0', processed: false };
                            activeParaTabStops = paraTabStops.slice();
                            activeParaTabStops.splice(indentFirstLineTabStopIndex, 0, indentFirstLineTabStop);
                            // reset width - the tab stop position must be calculated using all tab stop positions
                            width = 0;
                        }
                        else {
                            // reset text within span
                            tabSpan.contents().remove();
                            tabSpan[0].appendChild(document.createTextNode(''));
                            // use only paragraph tab stops
                            activeParaTabStops = paraTabStops;
                        }
                    } else {
                        // reset text within span
                        tabSpan.contents().remove();
                        tabSpan[0].appendChild(document.createTextNode(''));
                        // use only paragraph tab stops
                        activeParaTabStops = paraTabStops;
                    }

                    // Paragraph tab stops. Only paragraph tab stop can have a fill character and
                    // define a new alignment
                    if (width <= 1) {
                        tabStop = _(activeParaTabStops).find(function (tab) { return ((leftHMM + 10) < tab.pos) && !tab.processed; });
                        if (tabStop) {

                            tabStopType = tabStop.value || 'left';

                            // calculate tab stop size based on tab stop properties
                            if (tabStopType === 'left') {
                                // left bound tab stop
                                width = Math.max(0, tabStop.pos - (leftHMM % tabStop.pos));
                            }
                            else if (tabStopType === 'right') {
                                // right bound tab stop
                                width = calcRightAlignTabstop(tabStop, tabNode, leftHMM);
                                // Ignore this tab stop if width is zero. Don't use the default
                                // tab stop which is only active to left bound tab stops!
                                ignore = (width === 0);
                            }

                            // insert fill character
                            if (width > 1) {
                                fillChar = _.isString(tabStop.fillChar) ? TAB_FILL_CHARS[tabStop.fillChar] : '\u00a0';
                                if (fillChar) {
                                    insertTabFillChar(tabSpan, fillChar, width);
                                }
                                // Set processed flag to prevent using the same tab again due to rounding errors
                                // resulting by the browser calculation.
                                tabStop.processed = true;
                            }
                        }
                    }

                    if (!ignore) {
                        // only process default tab stop if tab stop is not set to ignore
                        if (width <= 1) {
                            // tab size calculation based on default tab stop
                            width = Math.max(0, defaultTabStop - (leftHMM % defaultTabStop));
                            width = (width <= 10) ? defaultTabStop : width; // no 0 tab size allowed, check for <= 10 to prevent rounding errors
                            // reset possible fill character
                            insertTabFillChar(tabSpan, '\u00a0', width);
                        }
                        prevTabEndPos = { right: leftHMM + width, top: topHMM };
                    }

                    // always set a width to the tab div (even an ignored tab div needs to set the size to zero)
                    tabNode.css('width', (width / 100) + 'mm');
                });

                // reset processed flag for tab stops again
                paraTabStops.forEach(function (tab) {
                    tab.processed = false;
                });
            }
        }

        /**
         * Calculates the width of a right aligned tab element
         * depending on the position and the remaining size of
         * the following nodes.
         *
         * @param {Object} tabStop
         *  The right aligned tab stop.
         *
         * @param {jQuery} tabNode
         *  The right aligned tab node as jQuery.
         *
         * @param {Number} tabNodePosHMM
         *  The current position of the tab node in 1/100th mm.
         *
         * @returns {Number}
         *  The width of the right aligned tab stop. May be zero if the
         *  tab stop is not active.
         */
        function calcRightAlignTabstop(tabStop, tabNode, tabNodePosHMM) {
            var nextNodes = tabNode.nextAll(),
                node,
                rightPos = tabNodePosHMM,
                width = 0, i = 0, len = nextNodes.length;

            // Loop through all following nodes and sum up their width. Exceeding
            // the tab stop position or a new tab stops the iteration.
            for (i = 0; i < len && rightPos < tabStop.pos; i++) {
                node = $(nextNodes.get(i));
                if (DOM.isTabNode(node)) {
                    break;
                }
                rightPos += Utils.convertLengthToHmm(node.width(), 'px');
            }

            if (rightPos < tabStop.pos) {
                // Active right aligned tab stop. Subtract at least 1/10mm
                // rounding error regarding position values from the browser.
                width = tabStop.pos - rightPos - 10;
                width = Math.max(0, width);
            }

            return width;
        }

        /**
         * Will be called for every paragraph whose character attributes have
         * been changed.
         *
         * @param {jQuery} paragraph
         *  The paragraph node whose attributes have been changed, as jQuery
         *  object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateParagraphFormatting(paragraph, mergedAttributes) {

            var // the paragraph attributes of the passed attribute map
                paragraphAttributes = mergedAttributes.paragraph,
                // the character styles/formatter
                characterStyles = documentStyles.getStyleSheets('character'),

                leftMargin = 0,
                rightMargin = 0,

                prevParagraph = paragraph.prev(),
                prevAttributes = (prevParagraph.length > 0) ? self.getElementAttributes(prevParagraph) : { paragraph: {} },

                nextParagraph = paragraph.next(),
                nextAttributes = (nextParagraph.length > 0) ? self.getElementAttributes(nextParagraph) : { paragraph: {} };

            function setBorder(border, position, addSpace, para) {
                var space = addSpace + ((border && border.space) ? border.space : 0),
                    localPara = para || paragraph;
                localPara.css('padding-' + position, Utils.convertHmmToCssLength(space, 'px', 1));
                localPara.css('border-' + position, self.getCssBorder(border));
                return -space;
            }

            // Always update character formatting of all child nodes which may
            // depend on paragraph settings, e.g. automatic text color which
            // depends on the paragraph fill color. Also visit all helper nodes
            // containing text spans, e.g. numbering labels.
            Utils.iterateDescendantNodes(paragraph, function (node) {
                DOM.iterateTextSpans(node, function (span) {
                    characterStyles.updateElementFormatting(span, { baseAttributes: mergedAttributes });
                });
            }, undefined, { children: true });

            // update borders
            var leftPadding = paragraphAttributes.borderLeft && paragraphAttributes.borderLeft.space ? paragraphAttributes.borderLeft.space : 0;
            leftMargin += setBorder(paragraphAttributes.borderLeft, 'left', 0);
            rightMargin += setBorder(paragraphAttributes.borderRight, 'right', 0);

            var topMargin = paragraphAttributes.marginTop;
            var bottomMargin = paragraphAttributes.marginBottom;

            // top border is not set if previous paragraph uses the same border settings
            if (isMergeBorders(paragraphAttributes, prevAttributes.paragraph)) {
                setBorder({ style: 'none' }, 'top', topMargin);
                prevParagraph.css({
                    paddingBottom: this.getCssBorder(paragraphAttributes.borderInside.space + bottomMargin),
                    borderBottom: this.getCssBorder(paragraphAttributes.borderInside),
                    marginBottom: 0
                });
                topMargin = 0;
            } else {
                setBorder(paragraphAttributes.borderTop, 'top', 0);
            }

            // bottom border is replaced by inner border if next paragraph uses the same border settings
            if (isMergeBorders(paragraphAttributes, nextAttributes.paragraph)) {
                setBorder(paragraphAttributes.borderInside, 'bottom');
                setBorder(paragraphAttributes.borderInside, 'top', 0, nextParagraph);  // Task 30473: Updating merged borders
                prevParagraph.css('padding-bottom', this.getCssBorder(bottomMargin));
                bottomMargin = 0;
            } else {
                setBorder(paragraphAttributes.borderBottom, 'bottom', 0);
            }

            //calculate list indents
            var listStyleId = paragraphAttributes.listStyleId;
            if (listStyleId.length) {
                var listLevel = paragraphAttributes.listLevel,
                     lists = documentStyles.getContainer('lists');
                if (listLevel < 0) {
                    // is a numbering level assigned to the current paragraph style?
                    listLevel = lists.findIlvl(listStyleId, mergedAttributes.styleId);
                }
                if (listLevel !== -1 && listLevel < 10) {
                    var listItemCounter = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
                    var listObject = lists.formatNumber(paragraphAttributes.listStyleId, listLevel, listItemCounter, 1);

                    if (listObject.indent > 0) {
                        leftPadding += listObject.indent - listObject.firstLine;
                        leftMargin += listObject.firstLine;
                    }
                    var listLabel = $(paragraph).children(DOM.LIST_LABEL_NODE_SELECTOR);
                    if (listLabel.length) {
                        var listSpan = listLabel.children('span');
                        if (listObject.fontSize)
                            listSpan.css('font-size', listObject.fontSize + 'pt');
                        if (listObject.color)
                            listSpan.css('color', self.getCssTextColor(listObject.color, [paragraphAttributes.fillColor, listObject.fillColor]));
                    }
                }
            }

            // paragraph margin attributes - not applied if paragraph is in a list
            var textIndent = 0;
            if (listStyleId === '') {
                leftMargin += paragraphAttributes.indentLeft;
                rightMargin += paragraphAttributes.indentRight;
                textIndent = paragraphAttributes.indentFirstLine ? paragraphAttributes.indentFirstLine : 0;
                paragraph.css('text-indent', textIndent / 100 + 'mm');
            }

            if (textIndent < 0) {
                leftPadding -= textIndent;
                leftMargin += textIndent;
            }
            paragraph.css({
                paddingLeft: (leftPadding / 100) + 'mm',
                // now set left/right margins
                marginLeft: (leftMargin / 100) + 'mm',
                marginRight: (rightMargin / 100) + 'mm',
                'text-indent': (textIndent / 100) + 'mm'
            });

            //no distance between paragraph using the same style if contextualSpacing is set
            var noDistanceToPrev = prevAttributes.paragraph.contextualSpacing && (mergedAttributes.styleId === prevAttributes.styleId),
                noDistanceToNext = paragraphAttributes.contextualSpacing && (mergedAttributes.styleId === nextAttributes.styleId);
            if (noDistanceToPrev) {
                //remove bottom margin from previous paragraph
                prevParagraph.css('margin-bottom', 0 + 'mm');
                paragraph.css('padding-top', 0 + 'mm');
                topMargin = 0;
            }
            if (noDistanceToNext) {
                paragraph.css('padding-bottom', 0 + 'mm');
                bottomMargin = 0;
            }

            paragraph.css({
                marginTop: (topMargin / 100) + 'mm',
                marginBottom: (bottomMargin / 100) + 'mm'
            });

            // taking care of implicit paragraph nodes behind tables after loading the document
            if ((DOM.isImplicitParagraphNode(paragraph)) && (DOM.isFinalParagraphBehindTable(paragraph))) {
                paragraph.css('height', 0);
            }

            // update the size of all tab stops in this paragraph (but only if the paragraph contains tabs (Performance))
            if (paragraph.find(DOM.TAB_NODE_SELECTOR).length > 0) {
                updateTabStops(paragraph, mergedAttributes);
            }
        }

        /**
         * Will be called for paragraphs used as preview elements in the GUI.
         *
         * @param {jQuery} paragraph
         *  The preview paragraph node, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function setParagraphPreviewFormatting(paragraph, mergedAttributes) {

            var // the character styles/formatter
                characterStyles = documentStyles.getStyleSheets('character');

            // format the text spans contained in the paragraph element
            paragraph.children().each(function () {
                characterStyles.updateElementFormatting(this, { baseAttributes: mergedAttributes, preview: true });
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Updates all tabulators in the passed paragraph node. This method
         * uses both the default tab size and existing paragraph tab stop
         * definitions. Fill characters are also supported.
         *
         * @param {HTMLElement|jQuery} paragraph
         *  The paragraph node whose tabulator nodes will be updated. If this
         *  object is a jQuery collection, uses the first DOM node it contains.
         *
         * @returns {ParagraphStyles}
         *  A reference to this instance.
         */
        this.updateTabStops = function (paragraph) {
            updateTabStops($(paragraph), this.getElementAttributes(paragraph));
            return this;
        };

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        documentStyles.registerAttributeDefinitions('paragraph', DEFINITIONS);

        //initialization after construction
        app.on('docs:init', function () {
            view = app.getView();
        });

    } // class ParagraphStyles

    // static methods ---------------------------------------------------------

    /**
     * Converts the passed paragraph attributes to an object describing the
     * visibility states of all border attributes.
     *
     * @param {Object} attributes
     *  The paragraph attributes.
     *
     * @returns {Object}
     *  An object with position keys ('left', 'top', etc.), and Boolean values
     *  or Null values specifying whether the respective borders are visible.
     *  If a border attribute in the passed attribute map is ambiguous (value
     *  null or missing), the map value will be null.
     */
    ParagraphStyles.getBorderModeFromAttributes = function (attributes) {

        var // result mapping border attribute names to boolean values
            borderMode = {};

        _(PARAGRAPH_BORDER_ATTRIBUTES).each(function (attrName, propName) {
            borderMode[propName] = _.isObject(attributes[attrName]) ? Border.isVisibleBorder(attributes[attrName]) : null;
        });

        return borderMode;
    };

    /**
     * Converts the passed object describing the visibility states of paragraph
     * border attributes to a paragraph attribute map containing all border
     * attributes.
     *
     * @param {Object} borderMode
     *  An object with position keys ('left', 'top', etc.), and Boolean values
     *  specifying whether the respective borders are visible. Attributes not
     *  contained in this object will not occur in the returned attribute set.
     *  If the passed object contains a single property set to true, the
     *  current visibility state of the respective border attribute will be
     *  toggled.
     *
     * @param {Object} [origAttributes]
     *  The original attributes of the target paragraphs.
     *
     * @returns {Object}
     *  The paragraph attributes containing all border attributes that are
     *  mentioned in the passed object.
     */
    ParagraphStyles.getAttributesFromBorderMode = function (borderMode, origAttributes) {

        var // the resulting paragraph attributes
            attributes = {},
            // whether to toggle the visibility of a single border
            toggle = _.size(borderMode) === 1;

        _(borderMode).each(function (visible, propName) {
            var borderAttrName = PARAGRAPH_BORDER_ATTRIBUTES[propName];
            visible = toggle ? !Border.isVisibleBorder(origAttributes[borderAttrName]) : visible;
            attributes[borderAttrName] = visible ? Border.SINGLE : Border.NONE;
        });

        return attributes;
    };

    // exports ================================================================

    // derive this class from class StyleSheets
    return StyleSheets.extend({ constructor: ParagraphStyles });

});
