/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Malte Timmermann <malte.timmermann@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Oliver Specht <oliver.specht@open-xchange.com>
 */

define('io.ox/office/text/editor',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/dialogs',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/io',
     'io.ox/office/editframework/model/editmodel',
     'io.ox/office/editframework/model/undomanager',
     'io.ox/office/editframework/model/format/color',
     'io.ox/office/editframework/model/format/stylesheets',
     'io.ox/office/editframework/view/editdialogmixin',
     'io.ox/office/drawinglayer/view/drawingframe',
     'io.ox/office/drawinglayer/view/imageutil',
     'io.ox/office/text/app/config',
     'io.ox/office/text/dom',
     'io.ox/office/text/selection',
     'io.ox/office/text/table',
     'io.ox/office/text/hyperlink',
     'io.ox/office/text/operations',
     'io.ox/office/text/position',
     'io.ox/office/text/drawingResize',
     'io.ox/office/text/tableResize',
     'io.ox/office/text/format/characterstyles',
     'io.ox/office/text/format/textdocumentstyles',
     'io.ox/office/text/format/tablestyles',
     'io.ox/office/text/export',
     'gettext!io.ox/office/text'
    ], function (Utils, Dialogs, KeyCodes, IO, EditModel, UndoManager, Color, StyleSheets, EditDialogMixin, DrawingFrame, Image, TextConfig, DOM, Selection, Table, Hyperlink, Operations, Position, DrawingResize, TableResize, CharacterStyles, TextDocumentStyles, TableStyles, Export, gt) {

    'use strict';

    var // style attributes for heading 1 -6 based on latent styles
        HEADINGS_CHARATTRIBUTES = [
            { color: { type: 'scheme', value: 'accent1', transformations: [{ type: 'shade', value: 74902 }], fallbackValue: '376092' }, bold: true, fontSize: 14 },
            { color: { type: 'scheme', value: 'accent1', fallbackValue: '4F81BD' }, bold: true, fontSize: 13 },
            { color: { type: 'scheme', value: 'accent1', fallbackValue: '4F81BD' }, bold: true },
            { color: { type: 'scheme', value: 'accent1', fallbackValue: '4F81BD' }, bold: true, italic: true },
            { color: { type: 'scheme', value: 'accent1', transformations: [{ type: 'shade', value: 49804 }], fallbackValue: '244061' } },
            { color: { type: 'scheme', value: 'accent1', transformations: [{ type: 'shade', value: 49804 }], fallbackValue: '244061' }, italic: true }
        ],

        DEFAULT_PARAGRAPH_DEFINTIONS = { 'default': true, styleId: 'Standard', styleName: 'Normal' },

        // style attributes for lateral table style
        DEFAULT_LATERAL_TABLE_DEFINITIONS = { 'default': true, styleId: 'TableGrid', styleName: 'Table Grid', uiPriority: 59 },
        DEFAULT_LATERAL_TABLE_ATTRIBUTES = {
            wholeTable: {
                paragraph: { lineHeight: { type: 'percent', value: 100 }},
                table:
                {
                    borderTop:        { color: Color.AUTO, width: 17, style: 'single' },
                    borderBottom:     { color: Color.AUTO, width: 17, style: 'single' },
                    borderInsideHor:  { color: Color.AUTO, width: 17, style: 'single' },
                    borderInsideVert: { color: Color.AUTO, width: 17, style: 'single' },
                    borderLeft:       { color: Color.AUTO, width: 17, style: 'single' },
                    borderRight:      { color: Color.AUTO, width: 17, style: 'single' },
                    paddingBottom: 0,
                    paddingTop: 0,
                    paddingLeft: 190,
                    paddingRight: 190
                }
            }
        },

        DEFAULT_HYPERLINK_DEFINTIONS = { 'default': false, styleId: 'Hyperlink', styleName: 'Hyperlink', uiPriority: 99 },
        DEFAULT_HYPERLINK_CHARATTRIBUTES = { color: { type: 'scheme', value: 'hyperlink', fallbackValue: '0080C0' }, underline: true },

        DEFAULT_DRAWING_MARGINS = { marginTop: 317, marginLeft: 317, marginBottom: 317, marginRight: 317 },

        // internal clipboard
        clipboardOperations = [],
        clipboardId = '',

        // if set to true, events caught by the editor will be logged to console
        LOG_EVENTS = false,

        // Load performance: List of operation names for operations that need to be executed, even if the document is loaded
        // from local storage. These operations do not modify the DOM, but register additional document information, that is
        // not stored in the DOM.
        REQUIRED_LOAD_OPERATIONS = ['setDocumentAttributes', 'insertFontDescription', 'insertStyleSheet', 'insertTheme', 'insertListStyle'],

        // the display timeout id of the collaborative overlay
        overlayTimeoutId = null;

    // private global functions ===============================================

    /**
     * Returns true, if the passed keyboard event is ctrl+v, meta+v or shift+insert.
     *
     * @param event
     *  A jQuery keyboard event object.
     */
    function isPasteKeyEvent(event) {
        return (KeyCodes.matchModifierKeys(event, {ctrlOrMeta: true}) && (event.charCode === 118 || event.keyCode === KeyCodes.V) ||
                (KeyCodes.matchModifierKeys(event, {shift: true}) && event.keyCode === KeyCodes.INSERT));
    }

    /**
     * Returns true, if the passed keyboard event is ctrl+c, meta+c or ctrl+insert.
     *
     * @param event
     *  A jQuery keyboard event object.
     */
    function isCopyKeyEvent(event) {
        return (KeyCodes.matchModifierKeys(event, {ctrlOrMeta: true}) && (event.charCode === 99 || event.keyCode === KeyCodes.C ||
                event.keyCode === KeyCodes.INSERT));
    }

    /**
     * Returns true, if the passed keyboard event is ctrl+x, meta+x or shift+delete.
     *
     * @param event
     *  A jQuery keyboard event object.
     */
    function isCutKeyEvent(event) {
        return (KeyCodes.matchModifierKeys(event, {ctrlOrMeta: true}) && (event.charCode === 120 || event.keyCode === KeyCodes.X) ||
                (KeyCodes.matchModifierKeys(event, {shift: true}) && event.keyCode === KeyCodes.DELETE));
    }

    /**
     * Returns true, if the passed keyboard event is the global F6 accessibility key event
     */
    function isF6AcessibilityKeyEvent(event) {
        // ignore all modifier keys
        return event.keyCode === KeyCodes.F6;
    }

    /**
     * Returns true, if the passed keyboard event is ctrl+f, meta+f, ctrl+g, meta+g, ctrl+shift+g or meta+shift+g.
     *
     * @param event
     *  A jQuery keyboard event object.
     */
    function isSearchKeyEvent(event) {
        return (KeyCodes.matchModifierKeys(event, {ctrlOrMeta: true, shift: null}) && (event.charCode === 102 || event.keyCode === KeyCodes.F || event.charCode === 103 || event.keyCode === KeyCodes.G));
    }

    /**
     * Returns true, if the passed keyboard event is a common browser keyboard shortcut
     * that should be handled by the browser itself.
     *
     * @param {jQuery.event} event
     *  A jQuery keyboard event object.
     *
     * @returns {Boolean}
     *  Whether the passed event is a browser shortcut.
     */
    function isBrowserShortcutKeyEvent(event) {

        var // Ctrl
            ctrl = KeyCodes.matchModifierKeys(event, {ctrl: true}),
            // Ctrl and Shift
            ctrlShift = KeyCodes.matchModifierKeys(event, {ctrl: true, shift: true}),
            // Meta and Alt
            metaAlt = KeyCodes.matchModifierKeys(event, {meta: true, alt: true}),
            // Meta and Shift
            metaShift = KeyCodes.matchModifierKeys(event, {meta: true, shift: true}),
            // Ctrl and Meta
            ctrlMeta = KeyCodes.matchModifierKeys(event, {ctrl: true, meta: true}),
            // Ctrl or Meta
            ctrlOrMeta = KeyCodes.matchModifierKeys(event, {ctrlOrMeta: true}),
            // Ctrl or Meta and Shift
            ctrlOrMetaShift = KeyCodes.matchModifierKeys(event, {ctrlOrMeta: true, shift: true}),
            // Ctrl or Meta and Alt
            ctrlOrMetaAlt = KeyCodes.matchModifierKeys(event, {ctrlOrMeta: true, alt: true});


        if (event.type === 'keypress') {
            // check charCode for keypress event

            // Switch to the specified/last tab - Ctrl + 1...8, Cmd + 1...8 / Ctrl + 9, Cmd + 9
            if (ctrlOrMeta && event.charCode >= 49 && event.charCode <= 57) {
                return true;
            }

            // Open a new tab             - Ctrl + T, Cmd + T
            // Reopen the last closed tab - Ctrl + Shift + T, Cmd + Shift + T
            if ((ctrlOrMeta || ctrlOrMetaShift) && (event.charCode === 116 || event.charCode === 84)) {
                return true;
            }

            // Close the current tab     - Ctrl + W, Ctrl + F4, Cmd + W
            // Open a new browser window - Ctrl + N, Cmd + N
            if (ctrlOrMeta && (event.charCode === 119 || event.charCode === 110)) {
                return true;
            }

            // Close the current window - Ctrl + Shift + W, Cmd + Shift + W
            if (ctrlOrMetaShift && (event.charCode === 87 || event.charCode === 119)) {
                return true;
            }

            // Open a new window in incognito mode - Ctrl + Shift + N, Cmd + Shift + N, Ctrl + Shift + P, Cmd + Shift + P
            if (ctrlOrMetaShift && (event.charCode === 78 || event.charCode === 80)) {
                return true;
            }

            // Minimize the current window - Ctrl + M, Cmd + M
            if (ctrlOrMeta && event.charCode === 109) {
                return true;
            }

            // Hide browser - Ctrl + H, Cmd + H
            // Hide all other windows - Ctrl + Alt + H, Cmd + Alt + H
            if ((ctrlOrMeta || ctrlOrMetaAlt) && (event.charCode === 104 || event.charCode === 170)) {
                return true;
            }

            // Quit browser - Ctrl + Q, Cmd + Q
            if (ctrlOrMeta && event.charCode === 113) {
                return true;
            }

            // Zoom in on the page - Ctrl + '+', Cmd + '+'
            // Zoom out on the page - Ctrl + '-', Cmd + '-'
            // Reset zoom level - Ctrl + 0, Cmd + 0
            if (ctrlOrMeta && (event.charCode === 43 || event.charCode === 45 || event.charCode === 48)) {
                return true;
            }

            // Full-screen mode - Ctrl + Cmd + F
            if (ctrlMeta && (event.charCode === 102 || event.charCode === 6)) { //6 is for safari
                return true;
            }

            // Open the browsing history - Ctrl + H, Cmd + Shift + H
            if ((ctrl || metaShift) && (event.charCode === 72 || event.charCode === 104)) {
                return true;
            }

            // Open the download history - Ctrl + J, Cmd + J, Cmd + Shift + J, Cmd + Alt + 2
            if (((ctrlOrMeta || metaShift) && (event.charCode === 106 || event.charCode === 74)) || (metaAlt && event.charCode === 8220)) {
                return true;
            }

            // Bookmark the current web site - Ctrl + D, Cmd + D
            if (ctrlOrMeta && event.charCode === 100) {
                return true;
            }

            // Toggles the bookmarks bar                                               - Ctrl + Shift + B, Cmd + Shift + B
            // Save all open pages in your current window as bookmarks in a new folder - Ctrl + Shift + D, Cmd + Shift + D
            if (ctrlOrMetaShift && (event.charCode === 66 || event.charCode === 68)) {
                return true;
            }

            // Open the Bookmarks window - Ctrl + Shift + B, Cmd + Shift + B, Cmd + Alt + B
            if ((ctrlOrMetaShift || metaAlt) && (event.charCode === 66 || event.charCode === 8747)) {
                return true;
            }

            // Open Developer Tools - Ctrl + Shift + (I,J,K,S), Cmd + Alt + (I,J,K,S)
            if ((ctrlShift || metaAlt) && (event.charCode === 8260 || event.charCode === 73 ||
                                           event.charCode === 186  || event.charCode === 74 ||
                                           event.charCode === 8710 || event.charCode === 75 ||
                                           event.charCode === 8218 || event.charCode === 83)) {
                return true;
            }

            // For function keys Firefox sends a keydown event with corresponding keyCode as expected,
            // but also sends a keypress event with charCode 0 and a corresponding keyCode.
            if (_.browser.Firefox && (event.charCode === 0)) {

                // Full-screen mode - F11, Open Firebug - F12
                if (event.keyCode === KeyCodes.F11 || event.keyCode === KeyCodes.F12) {
                    return true;
                }
            }

        } else {
            // check keyCode for keyup and keydown events

            // Switch to the specified/last tab - Ctrl + 1...8, Cmd + 1...8 / Ctrl + 9, Cmd + 9
            if (ctrlOrMeta && ((event.keyCode >= KeyCodes['1'] && event.keyCode <= KeyCodes['9']))) {
                return true;
            }

            // Switch to the next/previous tab - Ctrl + Tab / Ctrl + Shift + Tab
            if ((KeyCodes.matchKeyCode(event, KeyCodes.TAB, {ctrl: true}) || KeyCodes.matchKeyCode(event, KeyCodes.TAB, {ctrl: true, shift: true}))) {
                return true;
            }

            // Open a new tab             - Ctrl + T, Cmd + T
            // Reopen the last closed tab - Ctrl + Shift + T, Cmd + Shift + T
            if ((ctrlOrMeta || ctrlOrMetaShift) && event.keyCode === KeyCodes.T) {
                return true;
            }

            // Close the current tab     - Ctrl + W, Ctrl + F4, Cmd + W
            // Open a new browser window - Ctrl + N, Cmd + N
            if (ctrlOrMeta && (event.keyCode === KeyCodes.W || event.keyCode === KeyCodes.F4 || event.keyCode === KeyCodes.N)) {
                return true;
            }

            // Close the current window - Alt + F4, Ctrl + Shift + W, Cmd + Shift + W
            if ((KeyCodes.matchModifierKeys(event, {alt: true}) && event.keyCode === KeyCodes.F4) || (ctrlOrMetaShift && event.keyCode === KeyCodes.W)) {
                return true;
            }

            // Open a new window in incognito mode - Ctrl + Shift + N, Cmd + Shift + N, Ctrl + Shift + P, Cmd + Shift + P
            if (ctrlOrMetaShift && (event.keyCode === KeyCodes.N || event.keyCode === KeyCodes.P)) {
                return true;
            }

            // Minimize the current window - Ctrl + M, Cmd + M
            if (ctrlOrMeta && event.keyCode === KeyCodes.M) {
                return true;
            }

            // Hide browser - Ctrl + H, Cmd + H
            // Hide all other windows - Ctrl + Alt + H, Cmd + Alt + H
            if ((ctrlOrMeta || ctrlOrMetaAlt) && event.keyCode === KeyCodes.H) {
                return true;
            }

            // Quit browser - Ctrl + Q, Cmd + Q
            if (ctrlOrMeta && event.keyCode === KeyCodes.Q) {
                return true;
            }

            // Zoom in on the page - Ctrl + '+', Cmd + '+'
            // Zoom out on the page - Ctrl + '-', Cmd + '-'
            // Reset zoom level - Ctrl + 0, Cmd + 0
            if (ctrlOrMeta && (event.keyCode === KeyCodes.NUM_PLUS  || event.keyCode === 187 /*Safari*/ || event.keyCode === 171 /*Firefox*/ ||
                               event.keyCode === KeyCodes.NUM_MINUS || event.keyCode === 189 /*Safari*/ || event.keyCode === 173 /*Firefox*/ ||
                               event.keyCode === KeyCodes.NUM_0     || event.keyCode === KeyCodes['0'])) {
                return true;
            }

            // Full-screen mode - F11, Ctrl + Cmd + F
            if ((event.keyCode === KeyCodes.F11) || (ctrlMeta && event.keyCode === KeyCodes.F)) {
                return true;
            }

            // Open the browsing history - Ctrl + H, Cmd + Shift + H
            if ((ctrl || metaShift) && event.keyCode === KeyCodes.H) {
                return true;
            }

            // Open the download history - Ctrl + J, Cmd + J, Cmd + Shift + J, Cmd + Alt + 2
            if (((ctrlOrMeta || metaShift) && event.keyCode === KeyCodes.J) || (metaAlt && event.keyCode === KeyCodes['2'])) {
                return true;
            }

            // Bookmark the current web site - Ctrl + D, Cmd + D
            if (ctrlOrMeta && event.keyCode === KeyCodes.D) {
                return true;
            }

            // Toggles the bookmarks bar                                               - Ctrl + Shift + B, Cmd + Shift + B
            // Save all open pages in your current window as bookmarks in a new folder - Ctrl + Shift + D, Cmd + Shift + D
            if (ctrlOrMetaShift && (event.keyCode === KeyCodes.B || event.keyCode === KeyCodes.D)) {
                return true;
            }

            // Open the Bookmarks window - Ctrl + Shift + B, Cmd + Shift + B, Cmd + Alt + B
            if ((ctrlOrMetaShift || metaAlt) && event.keyCode === KeyCodes.B) {
                return true;
            }

            // Open Firebug - F12
            if (event.keyCode === KeyCodes.F12) {
                return true;
            }

            // Open Developer Tools - Ctrl + Shift + (I,J,K,S), Cmd + Alt + (I,J,K,S)
            if ((ctrlShift || metaAlt) && (event.keyCode === KeyCodes.I ||
                                           event.keyCode === KeyCodes.J ||
                                           event.keyCode === KeyCodes.K ||
                                           event.keyCode === KeyCodes.S)) {
                return true;
            }
        }

        return false;
    }

    function getPrintableCharFromCharCode(charCode) {
        return (_.isNumber(charCode) && (charCode >= 32)) ? String.fromCharCode(charCode) : undefined;
    }

    function getPrintableChar(event) {
        return getPrintableCharFromCharCode(event.charCode) || getPrintableCharFromCharCode(event.which) || '';
    }

    // undo/redo --------------------------------------------------------------

    /**
     * Returns whether the passed undo action is an 'insertText' action (an
     * action with a 'insertText' redo operation and a 'deleteText' undo
     * operation).
     *
     * @param {UndoAction} action
     *  The action to be tested.
     *
     * @param {Boolean} single
     *  If true, the action must contain exactly one 'deleteText' undo
     *  operation and exactly one 'insertText' redo operation. Otherwise, the
     *  last operations of each of the arrays are checked, and the arrays may
     *  contain other operations.
     */
    function isInsertTextAction(action, single) {
        return (single ? (action.undoOperations.length === 1) : (action.undoOperations.length >= 1)) && (_.first(action.undoOperations).name === Operations.DELETE) &&
            (single ? (action.redoOperations.length === 1) : (action.redoOperations.length >= 1)) && (_.last(action.redoOperations).name === Operations.TEXT_INSERT);
    }

    /**
     * Tries to merge the passed undo actions. Merging works only if both
     * actions represent a single 'insertText' operation, and the passed action
     * appends a single character directly after the text of this action.
     *
     * @param {UndoAction} lastAction
     *  The existing action that will be extended if possible.
     *
     * @param {UndoAction} nextAction
     *  The new action that will be tried to merge into the existing action.
     *
     * @returns {Boolean}
     *  Whether the passed new undo action has been merged successfully into
     *  the existing undo action.
     */
    function mergeUndoActionHandler(lastAction, nextAction) {

        var // check if this and the passed action is an 'insertText' action
            validActions = isInsertTextAction(lastAction, false) && isInsertTextAction(nextAction, true),

            // the redo operation of this action and the passed action
            lastRedo = _.last(lastAction.redoOperations),
            nextRedo = nextAction.redoOperations[0];

        // check that the operations are valid for merging the actions
        if (validActions && (nextRedo.text.length >= 1) && Position.hasSameParentComponent(lastRedo.start, nextRedo.start)) {

            // check that the new action adds the character directly after the text of this action and does not change the attributes
            if ((_.last(lastRedo.start) + lastRedo.text.length === _.last(nextRedo.start)) && !('attrs' in nextRedo)) {

                // check that the last character of this action is not a space character (merge actions word by word)
                if (lastRedo.text.substr(-1) !== ' ') {

                    // merge undo operation (delete one more character)
                    lastAction.undoOperations[0].end[lastAction.undoOperations[0].end.length - 1] += nextRedo.text.length;
                    // merge redo operation (add the character)
                    lastRedo.text += nextRedo.text;
                    return true;
                }
            }
        }
        return false;
    }

    // class Editor ===========================================================

    /**
     * The text editor model. Contains and manages the entire DOM of the edited
     * text document. Implements execution of all supported operations.
     *
     * Triggers the events supported by the base class EditModel, and the
     * following additional events:
     * - 'selection': When the editor selection has been changed. Event
     *      handlers receive a reference to the selection object (instance of
     *      class Selection).
     *
     * @constructor
     *
     * @extends EditModel
     *
     * @param {EditApplication} app
     *  The application containing this editor instance.
     */
    function Editor(app) {

        var // self reference for local functions
            self = this,

            // the root element for the document contents
            editdiv = DOM.createPageNode().attr({ contenteditable: true, tabindex: 0, 'data-focus-role': 'page' }).addClass('user-select-text noI18n'),

            // the logical selection, synchronizes with browser DOM selection
            selection = new Selection(app, editdiv),

            // the undo manager of this document
            undoManager = new UndoManager(app, { mergeUndoActionHandler: mergeUndoActionHandler }),

            // container for all style sheets of all attribute families
            documentStyles = new TextDocumentStyles(app),

            // shortcuts for style sheet containers
            characterStyles = documentStyles.getStyleSheets('character'),
            paragraphStyles = documentStyles.getStyleSheets('paragraph'),
            tableStyles = documentStyles.getStyleSheets('table'),
            tableRowStyles = documentStyles.getStyleSheets('row'),
            tableCellStyles = documentStyles.getStyleSheets('cell'),
            drawingStyles = documentStyles.getStyleSheets('drawing'),
            pageStyles = documentStyles.getStyleSheets('page'),

            // values needed for pagebreaks calculus
            pageAttributes,
            pageMaxHeight,
            pagePaddingLeft,
            pagePaddingTop,
            pagePaddingBottom,
            pageWidth,
            pbState = true,
            isSmallDevice = _.device('small'), // phones, and devices smaller than tablets

            // shortcuts for other format containers
            /*fonts = documentStyles.getContainer('fonts'),*/
            /*themes = documentStyles.getContainer('themes'),*/
            listCollection = documentStyles.getContainer('lists'),

            // attributes that were set without a selection and are only set for a single character
            preselectedAttributes = null,

            // whether the preselected attributes must not be set to null (for example after splitParagraph)
            keepPreselectedAttributes = false,

            // whether mouse down occured in the editdiv (and no mouse up followed)
            activeMouseDownEvent = false,

            // the period of time, that a text input is deferred, waiting for a further text input (in ms).
            inputTextTimeout = 5,

            // the maximum number of characters that are buffered before an insertText operation is created.
            maxTextInputChars = 5,

            // whether a group of operations requires, that the undo stack must be deleted
            deleteUndoStack = false,

            // whether an operation is part of an undo group
            isInUndoGroup = false,

            undoRedoRunning = false,

            // the promise that controls an already active (deferred) text input
            inputTextPromise = null,

            // a deferred that is resolved, after the text is inserted. This is required for the undo group and must be global.
            insertTextDef = null,

            // the content, that will be printed in the current active text input process
            activeInputText = null,

            // a paragraph node, that is implicit and located behind a table, might temporarely get an increased height
            increasedParagraphNode = null,

            // all text spans that are highlighted (for quick removal)
            highlightedSpans = [],

            // all paragraphs (and their background-color) that are artificially highlighted
            highlightedParagraphs = [],

            // the color used for artifical highlighting of empty paragraphs
            selectionHighlightColor = '#3399FF',

            // the results of a quick search as oxoPositions
            searchResults = [],

            // the current index of the searchResults array
            currentSearchResultId = 0,

            // the osn of the document during loading
            docLoadOSN = null,

            // whether an update of element formatting is required
            requiresElementFormattingUpdate = true,

            // whether an operation was triggered by a GUI action (for table operations)
            guiTriggeredOperation = false,

            // whether the internal clipboard is currently pasted (used for table operations)
            pastingInternalClipboard = false,

            // whether an internal or external clipboard paste is currently running
            pasteInProgress = false,

            lastKeyDownEvent,

            lastOperationEnd,     // Need to decide: Should the last operation modify this member, or should the selection be passed up the whole call chain?!
            // whether an undo shall set the selection range after applying the operation
            useSelectionRangeInUndo = false,

            // indicates if the applied operations is due to a search&replace
            // TODO: piggyback the information onto the operation directly
            isReplaceOperation = false,

            // online spelling mode off
            onlineSpelling = TextConfig.isSpellingEnabled() && app.getUserSettingsValue('onlineSpelling'),
            // online spelling timeout handler
            onlineSpellingTimeout = null,
            // the node the spell checker is been/has been working on
            spellCheckerCurrentNode = null,
            // collects replacements of misspelled words by locale
            spellReplacmentsByLocale = {},

            // indicates an active ime session
            imeActive = false,
            // ime start position
            imeStartPos = null,
            // indicates if a nbsp char was added to an empty span
            imeAdditionalChar = false,

            // mutation observer
            editDivObserver = null,
            // iPad read-only paragraph
            roIOSParagraph = null,

            // operations recording
            recordOperations = true,

            // collects information for debounced list update
            updateListsDebouncedOptions = {},

            // collects information for debounced page break insertion
            currentProcessingNode = {},

            // whether the paragraph cache can be used (only for internal operations)
            useParagraphCache = false,

            // Performance: Caching the synchronously used paragraph element
            paragraphCache = null,

            // handling double click and triple click in IE
            // whether there is a double click event waiting for a third click
            doubleClickEventWaiting = false,
            // whether the third click occured after a double click, activating the tripleclick event
            tripleClickActive = false,
            // the period of time, that a double click event 'waits' for a further click (in ms).
            doubleClickTimeOut = 60;

        // base constructor ---------------------------------------------------

        EditModel.call(this, app, Operations, undoManager, documentStyles);

        // methods ------------------------------------------------------------

        /**
         * Returns the root DOM element representing this editor.
         *
         * @returns {jQuery}
         *  The root node containing the document model and representation, as
         *  jQuery object.
         */
        this.getNode = function () {
            return editdiv;
        };

        /**
         * Selects the entire document.
         *
         * @returns {Editor}
         *  A reference to this instance.
         */
        this.selectAll = function () {
            selection.selectAll();
            return this;
        };

        // operations API -----------------------------------------------------

        /**
         * Copies the current selection into the internal clipboard and deletes
         * the selection.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled. If no dialog is shown, the deferred is
         *  resolved immediately.
         */
        this.cut = function (event) {

            var // the clipboard event data
                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData;

            // set the internal clipboard data and
            // add the data to the clipboard event if the browser supports the clipboard api
            this.copy(event);

            // if the browser supports the clipboard api, use the copy function to add data to the clipboard
            // the iPad supports the api, but cut doesn't work
            if (clipboardData && !Utils.IPAD) {

                // prevent default cut handling for desktop browsers, but not for touch devices
                if (!Modernizr.touch) {
                    event.preventDefault();
                }

                // delete current selection
                return this.deleteSelected();

            } else {
                return app.executeDelayed(function () {
                    // focus and restore browser selection
                    selection.restoreBrowserSelection();
                    // delete restored selection
                    return self.deleteSelected();
                });
            }
        };

        /**
         * Generates operations needed to copy the current text selection to
         * the internal clipboard.
         *
         * @returns {Array}
         *  The operations array that represents the current selection.
         */
        function copyTextSelection() {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // zero-based index of the current content node
                targetPosition = 0,
                // result of the iteration process
                result = null,
                // OX Text default paragraph list style sheet
                listParaStyleId = self.getDefaultUIParagraphListStylesheet(),
                // indicates if we had to add a paragraph style
                listParaStyleInserted = false,
                // the applied list style ids
                listStyleIds = [],
                // attributes of the contentNode
                attributes;

            // visit the paragraphs and tables covered by the text selection
            result = selection.iterateContentNodes(function (contentNode, position, startOffset, endOffset) {

                var styleSheetAttributes = null;

                // paragraphs may be covered partly
                if (DOM.isParagraphNode(contentNode)) {

                    // first or last paragraph: generate operations for covered text components
                    if (_.isNumber(startOffset) || _.isNumber(endOffset)) {

                        // special handling for Firefox and IE when selecting list paragraphs.
                        // if the selection includes the end position of a non list paragraph
                        // above the list, we don't add an empty paragraph to the document
                        // fix for bug 29752
                        if ((_.browser.Firefox || _.browser.IE) &&
                                Position.getParagraphLength(editdiv, position) === startOffset &&
                                !DOM.isListLabelNode(contentNode.firstElementChild) &&
                                Utils.findNextNode(editdiv, contentNode, DOM.PARAGRAPH_NODE_SELECTOR) &&
                                DOM.isListLabelNode(Utils.findNextNode(editdiv, contentNode, DOM.PARAGRAPH_NODE_SELECTOR).firstElementChild)) {
                            return;
                        }

                        // generate a splitParagraph and setAttributes operation for
                        // contents of first paragraph (but for multiple-paragraph
                        // selections only)
                        if (!_.isNumber(endOffset)) {
                            generator.generateOperation(Operations.PARA_SPLIT, { start: [targetPosition, 0] });
                            generator.generateSetAttributesOperation(contentNode, { start: [targetPosition] }, { clearFamily: 'paragraph' });
                        }

                        // operations for the text contents covered by the selection
                        generator.generateParagraphChildOperations(contentNode, [targetPosition], { start: startOffset, end: endOffset, targetOffset: 0, clear: true });

                    } else {

                        // skip embedded implicit paragraphs
                        if (DOM.isImplicitParagraphNode(contentNode)) { return; }

                        // generate operations for entire paragraph
                        generator.generateParagraphOperations(contentNode, [targetPosition]);
                    }

                    // if we have a list add the list paragraph style and the list style
                    if (paragraphStyles.containsStyleSheet(listParaStyleId)) {

                        if (!listParaStyleInserted) {
                            listParaStyleInserted = true;

                            styleSheetAttributes = {
                                    attrs: paragraphStyles.getStyleSheetAttributeMap(listParaStyleId),
                                    type: 'paragraph',
                                    styleId: listParaStyleId,
                                    styleName: paragraphStyles.getName(listParaStyleId),
                                    parent: paragraphStyles.getParentId(listParaStyleId),
                                    uiPriority: paragraphStyles.getUIPriority(listParaStyleId)
                                };

                            // parent is an optional value, should not be send as 'null'
                            if (styleSheetAttributes.parent === null) { delete styleSheetAttributes.parent; }

                            generator.generateOperation(Operations.INSERT_STYLESHEET, styleSheetAttributes);
                        }

                        attributes = paragraphStyles.getElementAttributes(contentNode);
                        if (attributes.paragraph && attributes.paragraph.listStyleId && !_(listStyleIds).contains(attributes.paragraph.listStyleId)) {
                            generator.generateOperation(Operations.INSERT_LIST, listCollection.getListOperationFromListStyleId(attributes.paragraph.listStyleId));
                            listStyleIds.push(attributes.paragraph.listStyleId);
                        }
                    }

                // entire table: generate complete operations array for the table (if it is not an exceeded-size table)
                } else if (DOM.isTableNode(contentNode)) {

                    // skip embedded oversized tables
                    if (DOM.isExceededSizeTableNode(contentNode)) { return; }

                    // generate operations for entire table
                    generator.generateTableOperations(contentNode, [targetPosition]);

                } else {
                    Utils.error('Editor.copyTextSelection(): unknown content node "' + Utils.getNodeName(contentNode) + '" at position ' + JSON.stringify(position));
                    return Utils.BREAK;
                }

                targetPosition += 1;

            }, this, { shortestPath: true });

            // return operations, if iteration has not stopped on error
            return (result === Utils.BREAK) ? [] : generator.getOperations();
        }

        /**
         * Sets a paragraph to contentEditable = false while removing it
         * from an optional previous remembered paragraph.
         *
         * paragraph {HTMLElement|null}
         *  The paragraph where contentEditable should be set to false.
         */
        function setIOSROParagraph(paragraph) {
            if (roIOSParagraph !== null) {
                $(roIOSParagraph).removeAttr('contentEditable');
                roIOSParagraph = null;
            }
            if (paragraph && paragraph.isContentEditable) {
                paragraph.contentEditable = 'false';
                roIOSParagraph = paragraph;
            }
        }

        /**
         * Generates operations needed to copy the current cell range selection
         * to the internal clipboard.
         *
         * @returns {Array}
         *  The operations array that represents the current selection.
         */
        function copyCellRangeSelection() {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // information about the cell range
                cellRangeInfo = selection.getSelectedCellRange(),
                // merged attributes of the old table
                oldTableAttributes = null,
                // explicit attributes for the new table
                newTableAttributes = null,
                // all rows in the table
                tableRowNodes = null,
                // relative row offset of last visited cell
                lastRow = -1,
                // result of the iteration process
                result = null;

            // generates operations for missing rows and cells, according to lastRow/lastCol
            function generateMissingRowsAndCells(row/*, col*/) {

                // generate new rows (repeatedly, a row may be covered completely by merged cells)
                while (lastRow < row) {
                    lastRow += 1;
                    generator.generateOperationWithAttributes(tableRowNodes[lastRow], Operations.ROWS_INSERT, { start: [1, lastRow], count: 1, insertDefaultCells: false });
                }

                // TODO: detect missing cells, which are covered by merged cells outside of the cell range
                // (but do not generate cells covered by merged cells INSIDE the cell range)
            }

            if (!cellRangeInfo) {
                Utils.error('Editor.copyCellRangeSelection(): invalid cell range selection');
                return [];
            }

            // split the paragraph to insert the new table between the text portions
            generator.generateOperation(Operations.PARA_SPLIT, { start: [0, 0] });

            // generate the operation to create the new table
            oldTableAttributes = tableStyles.getElementAttributes(cellRangeInfo.tableNode);
            newTableAttributes = StyleSheets.getExplicitAttributes(cellRangeInfo.tableNode);
            newTableAttributes.table = newTableAttributes.table || {};
            newTableAttributes.table.tableGrid = oldTableAttributes.table.tableGrid.slice(cellRangeInfo.firstCellPosition[1], cellRangeInfo.lastCellPosition[1] + 1);
            generator.generateOperation(Operations.TABLE_INSERT, { start: [1], attrs: newTableAttributes });

            // all covered rows in the table
            tableRowNodes = DOM.getTableRows(cellRangeInfo.tableNode).slice(cellRangeInfo.firstCellPosition[0], cellRangeInfo.lastCellPosition[0] + 1);

            // visit the cell nodes covered by the selection
            result = selection.iterateTableCells(function (cellNode, position, row, col) {

                // generate operations for new rows, and for cells covered by merged cells outside the range
                generateMissingRowsAndCells(row, col);

                // generate operations for the cell
                generator.generateTableCellOperations(cellNode, [1, row, col]);
            });

            // missing rows at bottom of range, covered completely by merged cells (using relative cellRangeInfo, task 30839)
            generateMissingRowsAndCells(cellRangeInfo.lastCellPosition[0] - cellRangeInfo.firstCellPosition[0], cellRangeInfo.lastCellPosition[1] - cellRangeInfo.firstCellPosition[1] + 1);

            // return operations, if iteration has not stopped on error
            return (result === Utils.BREAK) ? [] : generator.getOperations();
        }

        /**
         * Copies the current selection into the internal clipboard and
         * attaches the clipboard data to the copy event if the browser
         * supports the clipboard api.
         */
        this.copy = function (event) {

            var // the clipboard div
                clipboard,
                // the clipboard event data
                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData,
                // html clipboard data cleaned up for export
                htmlExportData,
                // start and end of current selection
                start = selection.getStartPosition(),
                end = selection.getEndPosition(),
                // information about the cell range and containing table
                cellRangeInfo,
                // is the cell defined by the start position the first cell in the row
                isFirstCell,
                // is the cell defined by the end position the last cell in the row
                isLastCell;

            // generate a new unique id to identify the clipboard operations
            clipboardId = ox.session + '_' + Date.now();

            switch (selection.getSelectionType()) {

            case 'text':
                cellRangeInfo = selection.getSelectedCellRange();
                isFirstCell = $(Position.getLastNodeFromPositionByNodeName(editdiv, start, DOM.TABLE_CELLNODE_SELECTOR)).prev().length === 0;
                isLastCell = $(Position.getLastNodeFromPositionByNodeName(editdiv, end, DOM.TABLE_CELLNODE_SELECTOR)).next().length === 0;

                // if the selected range is inside the same table or parent table and
                // the start position is the first cell in the start row and the end position
                // is the last cell in the end row use table selection otherwise, use text selection.
                if (cellRangeInfo && isFirstCell && isLastCell && !_.isEqual(cellRangeInfo.firstCellPosition, cellRangeInfo.lastCellPosition)) {
                    clipboardOperations = copyCellRangeSelection();
                } else {
                    clipboardOperations = copyTextSelection();
                }
                break;

            case 'drawing':
                clipboardOperations = copyTextSelection();
                break;

            case 'cell':
                clipboardOperations = copyCellRangeSelection();
                break;

            default:
                clipboardId = '';
                Utils.error('Editor.copy(): unsupported selection type: ' + selection.getSelectionType());
            }

            htmlExportData = Export.getHTMLFromSelection(self, {clipboardId: clipboardId, clipboardOperations: clipboardOperations});
            // set clipboard debug pane content
            app.getView().setClipboardDebugInfo(htmlExportData);

            // if browser supports clipboard api add data to the event
            if (clipboardData) {
                // add operation data
                clipboardData.setData('text/ox-operations', JSON.stringify(clipboardOperations));
                // add plain text and html of the current browser selection
                clipboardData.setData('text/plain', selection.getTextFromBrowserSelection());
                clipboardData.setData('text/html', htmlExportData);

                // prevent default copy handling for desktop browsers, but not for touch devices
                if (!Modernizr.touch) {
                    event.preventDefault();
                }
            } else {

                // copy the currently selected nodes to the clipboard div and append it to the body
                clipboard = app.getView().createClipboardNode();

                clipboard.append(htmlExportData);

                // focus the clipboard node and select all of it's child nodes
                selection.setBrowserSelectionToContents(clipboard);

                app.executeDelayed(function () {
                    // set the focus back
                    app.getView().grabFocus();
                    // remove the clipboard node
                    clipboard.remove();
                });
            }
        };

        /**
         * Returns whether the internal clipboard contains operations.
         */
        this.hasInternalClipboard = function () {
            return clipboardOperations.length > 0;
        };

        /**
         * Deletes the current selection and pastes the internal clipboard to
         * the resulting cursor position.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled. If no dialog is shown, the deferred is
         *  resolved immediately.
         */
        this.pasteInternalClipboard = function () {

            // check if clipboard contains something
            if (!this.hasInternalClipboard()) { return $.when(); }

            // make sure that only one paste call is processed at the same time
            if (checkSetClipboardPasteInProgress()) { return $.when(); }

            // Group all executed operations into a single undo action.
            // The undo manager returns the return value of the callback function.
            return undoManager.enterUndoGroup(function () {

                var // the deferred to keep the undo group open until it is resolved or rejected
                    undoDef = $.Deferred(),
                    // target position to paste the clipboard contents to
                    anchorPosition = null,
                    // the generated paste operations with transformed positions
                    operations = null,
                    // operation generator for additional insert style sheets operations
                    generator = this.createOperationsGenerator(),
                    // the next free list style number part
                    listStyleNumber = 1,
                    // the list style map
                    listStyleMap = {};


                // builds the list style map from insertListStyle operations, mapping the list style ids
                // of the source document to the list style ids of the destination document
                function createListStyleMap(operation) {

                    var listStyleId;

                    function getFreeListStyleId() {

                        var sFreeId = 'L';

                        while (listCollection.hasListStyleId('L' + listStyleNumber)) {
                            listStyleNumber++;
                        }
                        sFreeId += listStyleNumber;
                        listStyleNumber++;
                        return sFreeId;
                    }


                    if ((operation.name === Operations.INSERT_LIST) && _.isObject(operation.listDefinition)) {

                        // deep copy the operation before changing
                        operation = _.copy(operation, true);

                        // check if we already have a list style with the list definition and if not create a new id
                        listStyleId = listCollection.getListStyleIdForListDefinition(operation.listDefinition);
                        if (!listStyleId) {
                            listStyleId = getFreeListStyleId();
                        }

                        listStyleMap[operation.listStyleId] = listStyleId;
                        operation.listStyleId = listStyleId;
                    }

                    return operation;
                }

                // transforms a position being relative to [0,0] to a position relative to anchorPosition
                function transformPosition(position) {

                    var // the resulting position
                        resultPosition = null;

                    if ((position[0] === 0) && (position.length === 2)) {
                        // adjust text offset for first paragraph
                        resultPosition = anchorPosition.slice(0, -1);
                        resultPosition.push(_.last(anchorPosition) + position[1]);
                    } else {
                        // adjust paragraph offset for following paragraphs
                        resultPosition = anchorPosition.slice(0, -2);
                        resultPosition.push(anchorPosition[anchorPosition.length - 2] + position[0]);
                        resultPosition = resultPosition.concat(position.slice(1));
                    }

                    return resultPosition;
                }

                // if the operation references a style try to add the style sheet to the document
                function addMissingStyleSheet(operation) {
                    if ((operation.name === Operations.TABLE_INSERT) && _.isObject(operation.attrs) && _.isString(operation.attrs.styleId)) {
                        // generate operation to insert a dirty table style into the document
                        generator.generateMissingStyleSheetOperation('table', operation.attrs.styleId);
                    }
                }

                // apply list style mapping to setAttributes and insertParagraph operations
                function mapListStyles(operation) {

                    if ((operation.name === 'setAttributes' || operation.name === Operations.PARA_INSERT) &&
                        operation.attrs && operation.attrs.paragraph && _.isString(operation.attrs.paragraph.listStyleId) &&
                        listStyleMap[operation.attrs.paragraph.listStyleId]) {

                        // apply the list style id from the list style map to the operation
                        operation.attrs.paragraph.listStyleId = listStyleMap[operation.attrs.paragraph.listStyleId];
                    }
                }

                // transforms the passed operation relative to anchorPosition
                function transformOperation(operation) {

                    // clone the operation to transform the positions (no deep clone,
                    // as the position arrays will be recreated, not modified inplace)
                    operation = _.clone(operation);

                    // transform position of operation
                    if (_.isArray(operation.start)) {
                        // start may exist but is relative to position then
                        operation.start = transformPosition(operation.start);
                        // attribute 'end' only with attribute 'start'
                        if (_.isArray(operation.end)) {
                            operation.end = transformPosition(operation.end);
                        }
                        addMissingStyleSheet(operation);
                    }
                    // map list style ids from source to destination document
                    mapListStyles(operation);

                    var text = '  name="' + operation.name + '", attrs=';
                    var op = _.clone(operation);
                    delete op.name;
                    Utils.log(text + JSON.stringify(op));

                    return operation;
                }

                // applying the paste operations
                function doPasteInternalClipboard() {

                    var // the apply actions deferred
                        applyDef = null,
                        // the newly created operations
                        newOperations = null;

                    // init the next free list style number part
                    listStyleNumber = parseInt(listCollection.getFreeListId().slice(1, listCollection.getFreeListId().length), 10);
                    if (!_.isNumber(listStyleNumber)) {
                        listStyleNumber = 1;
                    }

                    // paste clipboard to current cursor position
                    anchorPosition = selection.getStartPosition();
                    if (anchorPosition.length >= 2) {
                        Utils.info('Editor.pasteInternalClipboard()');
                        pastingInternalClipboard = true;

                        // show a nice message with cancel button
                        app.getView().enterBusy({
                            initHandler: function (header, footer) {
                                // add a banner for large clipboard pastes
                                footer.append(
                                        $('<div>').addClass('size-warning-node').append(
                                                $('<div>').addClass('alert alert-warning').append(
                                                        $('<div>').text(gt('Sorry, pasting from clipboard will take some time.'))
                                                )
                                        )
                                );
                            },
                            cancelHandler: function () {
                                if (applyDef) {
                                    applyDef.abort();
                                }
                            },
                            delay: 500 // fade in busy blocker after a short delay
                        });

                        // map the operations
                        operations = _(clipboardOperations).map(createListStyleMap);
                        operations = _(operations).map(transformOperation);

                        // concat with newly generated operations
                        newOperations = generator.getOperations();
                        if (newOperations.length) {
                            operations = newOperations.concat(operations);
                        }
                        handleImplicitParagraph(anchorPosition);

                        // set clipboard debug pane content
                        app.getView().setClipboardDebugInfo(operations);

                        // apply operations
                        applyDef = self.applyActions({ operations: operations }, { async: true });

                        // add progress handling
                        applyDef.progress(function (progress) {
                            app.getWindow().busy(progress);
                        });

                    } else {
                        Utils.warn('Editor.pasteInternalClipboard(): invalid cursor position');
                        applyDef = $.Deferred().reject();
                    }

                    return applyDef.promise();
                }

                // delete current selection
                this.deleteSelected()
                .then(doPasteInternalClipboard)
                .always(function () {
                    pastingInternalClipboard = false;
                    setClipboardPasteInProgress(false);
                    // leave busy state
                    app.getView().leaveBusy().grabFocus();
                    // close undo group
                    undoDef.resolve();

                }).done(function () {
                    // setting the cursor position
                    selection.setTextSelection(lastOperationEnd);
                });

                return undoDef.promise();

            }, this);
        };

        this.paste = function (event) {

            var // the clipboard div
                clipboard,
                // the clipboard event data
                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData,
                // the list items of the clipboard event data
                items = clipboardData && clipboardData.items,
                // the list of mime types of the clipboard event data
                types = clipboardData && clipboardData.types,
                // the operation data from the internal clipboard
                eventData,
                // the file reader
                reader,
                // the list item of type text/html
                htmlEventItem = null,
                // the file list item
                fileEventItem = null,
                // the event URL data
                urlEventData = null;


            // handles the result of reading file data from the file blob received from the clipboard data api
            function onLoadHandler(evt) {
                var data = evt && evt.target && evt.target.result;

                if (data && data.substring(0, 10) === 'data:image') {
                    createOperationsFromExternalClipboard([{operation: Operations.DRAWING_INSERT, data: data, depth: 0}]);
                } else {
                    app.rejectEditAttempt('image');
                }
            }

            // returns true if the html clipboard has a matching clipboard id set
            function isHtmlClipboardIdMatching(html) {
                return ($(html).find('[id=ox-clipboard-data]').attr('data-ox-clipboard-id') === clipboardId);
            }

            // returns the operations attached to the html clipboard, or null
            function getHtmlAttachedOperations(html) {
                var operations;

                try {
                    operations = JSON.parse($(html).find('[id=ox-clipboard-data]').attr('data-ox-operations') || '{}');
                } catch (e) {
                    operations = null;
                }

                return operations;
            }

            // if the browser supports the clipboard api, look for operation data
            // from the internal clipboard to handle as internal paste.
            if (clipboardData) {
                eventData = clipboardData.getData('text/ox-operations');
                if (eventData) {
                    // prevent default paste handling for desktop browsers, but not for touch devices
                    if (!Modernizr.touch) {
                        event.preventDefault();
                    }

                    // set the operations from the event to be used for the paste
                    clipboardOperations = (eventData.length > 0) ? JSON.parse(eventData) : [];
                    self.pasteInternalClipboard();
                    return;
                }

                // check if clipboardData contains a html item
                htmlEventItem = _.find(items, function (item) { return item.type.toLowerCase() === 'text/html'; });

                // Chrome doesn't paste images into a (content editable) div, check if clipboardData contains an image item
                fileEventItem = _.find(items, function (item) { return item.type.toLowerCase().indexOf('image') !== -1; });

                // check if we have a mime type to get an URL from
                urlEventData = clipboardData.getData(_.find(types, function (type) { return type.toLowerCase().indexOf('text/uri-list') !== -1; }));

                if (htmlEventItem) {
                    htmlEventItem.getAsString(function (html) {
                        var div, ops;

                        // set clipboard debug pane content
                        app.getView().setClipboardDebugInfo(html);

                        // needs strange replace of zero char created by Firefox
                        div = $('<div>').html(html.replace(/[\x00]/gi, ''));

                        if (isHtmlClipboardIdMatching(div)) {
                            // if the clipboard id matches it's an internal copy & paste
                            self.pasteInternalClipboard();

                        } else {
                            ops = getHtmlAttachedOperations(div);
                            if (_.isArray(ops)) {
                                // it's not an internal copy & paste, but we have clipboard operations piggy backed to use for clipboardOperations
                                clipboardOperations = ops;
                                self.pasteInternalClipboard();

                            } else {
                                // use html clipboard
                                ops = parseClipboard(div);
                                createOperationsFromExternalClipboard(ops);
                            }
                        }
                    });

                } else if (fileEventItem) {
                    reader = new FileReader();
                    reader.onload = onLoadHandler;
                    reader.readAsDataURL(fileEventItem.getAsFile());

                } else if (urlEventData && Image.hasUrlImageExtension(urlEventData)) {
                    createOperationsFromExternalClipboard([{operation: Operations.DRAWING_INSERT, data: urlEventData, depth: 0}]);
                }

                if (htmlEventItem || fileEventItem || urlEventData) {
                    // prevent default paste handling of the browser
                    event.preventDefault();
                    return;
                }
            }


            // append the clipboard div to the body and place the cursor into it
            clipboard = app.getView().createClipboardNode();

            // focus and select the clipboard container node
            selection.setBrowserSelectionToContents(clipboard);

            // read pasted data
            app.executeDelayed(function () {

                var clipboardData,
                    operations;

                // set the focus back
                app.getView().grabFocus();

                if (isHtmlClipboardIdMatching(clipboard)) {
                    // if the clipboard id matches it's an internal copy & paste
                    self.pasteInternalClipboard();

                } else {
                    // look for clipboard operations
                    operations = getHtmlAttachedOperations(clipboard);
                    if (_.isArray(operations)) {
                        // it's not an internal copy & paste, but we have clipboard operations piggy backed to use for clipboardOperations
                        clipboardOperations = operations;
                        self.pasteInternalClipboard();

                    } else {
                        // set clipboard debug pane content
                        app.getView().setClipboardDebugInfo(clipboard);
                        // use html clipboard
                        clipboardData = parseClipboard(clipboard);
                        createOperationsFromExternalClipboard(clipboardData);
                    }
                }

                // remove the clipboard node
                clipboard.remove();
            });
        };

        /**
         * Sets the state wether a cipboard paste is in progress
         *
         * @param {Boolean} state
         *  The clipboard pasting state
         */
        function setClipboardPasteInProgress(state) {
            pasteInProgress = state;
        }

        /**
         * Returns true if a clipboard paste is still in progress, otherwise false.
         * And then sets the state to true.
         * and the processing of this paste
         *
         * @returns {Boolean}
         *  Wether a clipboard paste is in progress
         */
        function checkSetClipboardPasteInProgress() {
            var state = pasteInProgress;
            pasteInProgress = true;
            return state;
        }


//        /**
//         * Handler for copy command on the debug operation table.
//         */
//        function operationsCopyHandler(event) {
//            return true;
//        }

//        /**
//         * Handler for the cut command on the debug operation table.
//         */
//        function operationsCutHandler(event) {
//            event.preventDefault();
//            return false;
//        }

//        /**
//         * Handler for the paste command on the debug operation table.
//         *
//         * @param {jQuery.Event} event
//         *  The event sent by jQuery using the paste key/a paste key
//         *  combination.
//         */
//        function operationsPasteHandler(event) {
//            var // clipboard data
//                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData,
//                // the operation data from the internal clipboard
//                eventData,
//                // clipboard node
//                clipboard = null,
//                // operations from clipboard
//                clipboardReplayOperations = null;
//
//            /**
//             * Creates a JSON compliant string from the operation arguments
//             * found in the operations debug table.
//             *
//             * @param {String} opArguments
//             *  The operation arguments found in the operations debug table.
//             *
//             * @param {Boolean} forceEscape
//             *  Specifies if the characters must be escaped
//             *
//             * @return {String}
//             *  A JSON compliant string of the opArguments string.
//             */
//            function createJSONStringFromOperationArguments(opArguments, forceEscape) {
//                var i = 0,
//                    // current char
//                    char = null,
//                    // inside string
//                    insideString = false,
//                    // escaped
//                    escaped = false,
//                    // attribute
//                    insideAttribute = false,
//                    // correct encoded json string
//                    jsonString = '',
//                    // json part string
//                    attributeString = '',
//                    // tmp string
//                    tmpString = '',
//                    // test
//                    charcheck = /[a-zA-Z0-9]/;
//
//                for (i = 0; i < opArguments.length; i++) {
//                    char = opArguments.charAt(i);
//                    if (char === '"') {
//                        if (!escaped) {
//                            if (!insideString) {
//                                insideString = true;
//                            } else {
//                                insideString = false;
//                            }
//                            jsonString += char;
//                        } else {
//                            if (insideString) {
//                                jsonString += char;
//                            } else {
//                                // should not happen
//                            }
//                        }
//                        escaped = false;
//                    } else if (char === '\\') {
//                        if (forceEscape) {
//                            jsonString += char;
//                        }
//                        if (escaped) {
//                            if (!forceEscape) {
//                                jsonString += char;
//                            }
//                            escaped = false;
//                        } else {
//                            escaped = true;
//                        }
//                    } else if (char === ':') {
//                        if (insideAttribute) {
//                            tmpString = '"' + attributeString + '"';
//                            jsonString += tmpString;
//                            jsonString += char;
//                            insideAttribute = false;
//                            attributeString = '';
//                        } else {
//                            jsonString += char;
//                        }
//                        escaped = false;
//                    } else if (char === '{') {
//                        jsonString += char;
//                        escaped = false;
//                    } else if (char === '}') {
//                        if (insideAttribute) {
//                            jsonString += attributeString;
//                            attributeString = '';
//                            insideAttribute = false;
//                        }
//                        jsonString += char;
//                        escaped = false;
//                    } else if (char === ',') {
//                        if (insideAttribute) {
//                            jsonString += attributeString;
//                            attributeString = '';
//                            insideAttribute = false;
//                        }
//                        jsonString += char;
//                        escaped = false;
//                    } else if (char === ']') {
//                        if (insideAttribute) {
//                            jsonString += attributeString;
//                            attributeString = '';
//                            insideAttribute = false;
//                        }
//                        jsonString += char;
//                        escaped = false;
//                    } else {
//                        if (insideString) {
//                            jsonString += char;
//                        } else {
//                            if (charcheck.test(char)) {
//                                attributeString += char;
//                                insideAttribute = true;
//                            } else {
//                                jsonString += char;
//                            }
//                        }
//                        escaped = false;
//                    }
//                }
//
//                return jsonString;
//            }
//
//            /**
//             * Parses the clipboard 'text/plain' result provided by clipboardData
//             * and creates operations to be replayed.
//             *
//             * @param {String} text
//             *
//             * @returns {Array}
//             *  The operations as objects in an array extracted from the text/plain
//             *  result of the clipboardData or null if content couldn't be parsed
//             *  successfully.
//             */
//            function parseTextPlain(text) {
//                var // parse text to retrieve operations from it
//                lines = text.match(/[^\r\n]+/g),
//                // index
//                i = 0,
//                // one line of text
//                line = null,
//                // columns
//                cols = null,
//                // complete selection
//                completeLines = true,
//                // operation
//                operation = null,
//                // opArgs
//                operationArguments = null,
//                // json string
//                json = '[',
//                // first entry
//                firstEntry = true,
//                // operations parsed from the clipboard content
//                clipboardOperations = null;
//
//                for (i = 0; i < lines.length; i++) {
//                    line = lines[i];
//                    if (line) {
//                        // parse single line
//                        cols = line.split('\t');
//                        // current format of an operation line includes 6 columns
//                        // ROW, INTERN/EXTERN, OSN, OPL, OPERATION, OP-ARGUMENTS
//                        if (cols.length === 6) {
//
//                            if ((i === 0) && (cols[0].length === 0)) {
//                                // skip possible header of the operation table
//                                continue;
//                            }
//                            operation = cols[4];
//                            operationArguments = cols[5];
//                            if (!firstEntry) {
//                                json += ',';
//                            }
//                            json += '{"name"' + ':' + '"' + operation + '"';
//                            if (operationArguments.length > 0) {
//                                json += ',' + createJSONStringFromOperationArguments(operationArguments, true) + '}';
//                            } else {
//                                json += '}';
//                            }
//                            firstEntry = false;
//                        } else {
//                            completeLines = false;
//                            break;
//                        }
//                    }
//                }
//
//                if (completeLines) {
//                    json += ']';
//                    try {
//                        clipboardOperations = JSON.parse(json);
//                    } catch (e) {
//                        // nothing to do
//                    }
//                }
//
//                return clipboardOperations;
//            }
//
//            /**
//             * Parses the clipboard 'text/plain' result provided by clipboardData
//             * and creates operations to be replayed.
//             *
//             * @param {Array} parseClipboardResult
//             *  The result of parseClipboard called on the clipoard data
//             *  to be used to create operations from it.
//             *
//             * @returns {Array}
//             *  The operations as objects in an array extracted from the text/plain
//             *  result of the clipboardData or null if content couldn't be parsed
//             *  successfully.
//             *  var splitted, text = child.nodeValue.replace(/[\r\n]/g, ' ').replace(/[\u0000-\u001F]/g, '');
//             */
//            function processClipboardResult(parseClipboardResult) {
//                var // operations generated from provided clipboard result
//                    operations = null,
//                    // index
//                    i = 0,
//                    // number check via regex
//                    numberCheckRegEx = /^\d+$/,
//                    // operation text
//                    opText = '',
//                    // operation arguments
//                    opArguments = '',
//                    // parses text
//                    parsedText = null,
//                    // special operation noOp which doesn't have arguments
//                    noOpText = 'noOp',
//                    // check for JSON property delimiter
//                    jsonPropDelimFound = false,
//                    // json string
//                    json = null,
//                    // new operation object
//                    operationObject = null;
//
//                if (parseClipboardResult && parseClipboardResult.length > 0) {
//                    // Check parsed result which contains the text content of
//                    // every column in a single entry within the array.
//                    // We are looking for to text entries following each other
//                    for (i = 0; i < parseClipboardResult.length; i++) {
//                        // extract text part from the parseClipboard result entry
//                        parsedText = parseClipboardResult[i].data;
//                        if (numberCheckRegEx.test(parsedText)) {
//                            // A text with just numbers must be one of the columns
//                            // Nr. E/I OPN OPL
//                            // Check possible found opText if it's a special operation
//                            // without arguments
//                            if (opText && opText === noOpText) {
//                                if (!operations) {
//                                    operations = [];
//                                }
//                                operations.push({name: opText});
//                            } else {
//                                opText = '';
//                                opArguments = '';
//                            }
//                        } else if (parsedText === 'E' || parsedText === 'I') {
//                            // external/interal operation flag - just ignore
//                            opText = '';
//                            opArguments = '';
//                        } else {
//                            jsonPropDelimFound = parsedText.indexOf(':');
//                            if (jsonPropDelimFound && opText.length > 0) {
//                                // delimiter means that this must be the argument part of the operation
//                                // descape parsed text
//                                json = '{' + createJSONStringFromOperationArguments(parsedText, false) + '}';
//                                try {
//                                    opArguments = JSON.parse(json);
//                                    operationObject = {name: opText};
//                                    operationObject = _.extend(operationObject, opArguments);
//                                    if (!operations) {
//                                        operations = [];
//                                    }
//                                    operations.push(operationObject);
//                                    operationObject = null;
//                                } catch (e) {
//                                    // nothing to do
//                                }
//                            } else {
//                                // no delimiter: this means it must be the operation itself
//                                opText = parsedText;
//                                opArguments = '';
//                            }
//                        }
//                    }
//                }
//
//                return operations;
//            }
//
//            // if the browser supports the clipboard api, look for operation data
//            // from the internal clipboard to handle as internal paste.
//            if (clipboardData) {
//                eventData = clipboardData.getData('text/plain');
//                if (eventData) {
//                    // prevent default paste processing
//                    event.preventDefault();
//
//                    app.executeDelayed(function () {
//                        // set the operations from the event to be used for the paste
//                        clipboardReplayOperations = parseTextPlain(eventData);
//                        self.replayOperations(clipboardReplayOperations);
//                    });
//                    return;
//                }
//            } else {
//                // IE case
//
//                // append the clipboard div to the body and place the cursor into it
//                clipboard = app.getView().createClipboardNode();
//
//                // focus and select the clipboard container node
//                selection.setBrowserSelectionToContents(clipboard);
//
//                // read pasted data
//                app.executeDelayed(function () {
//
//                    var clipboardData = parseClipboard(clipboard);
//
//                    // set the focus back
//                    app.getView().grabFocus();
//                    // remove the clipboard node
//                    clipboard.remove();
//
//                    clipboardReplayOperations = processClipboardResult(clipboardData);
//                    self.replayOperations(clipboardReplayOperations);
//                });
//            }
//
//            return true;
//        }

        /**
         * Replays the operations provided.
         *
         * @param {Array|String} operations
         *  An array of operations to be replayed (processed) or a string
         *  of operations in JSON format to be replayed.
         *
         * @param {Boolean} external
         *  Specifies if the operations are interal or external.
         */
        this.replayOperations = function (operations, options) {
            var // operations to replay
                operationsToReplay = null,
                // delay for operations replay
                delay = Utils.getIntegerOption(options, 'delay', 10);

            function replayOperationNext(operationArray, delay) {
                var // the next operation if available
                    op = (operationArray.length > 0) ? operationArray.shift() : null,
                    // the result of apply operation
                    result;
                if (op) {
                    result = self.applyOperations(op);
                    if (result) {
                        app.executeDelayed(function () {
                            replayOperationNext(operationArray, delay);
                        }, { delay: delay });
                    }
                }
            }

            if (operations && operations.length > 0) {
                if (_.isString(operations)) {
                    try {
                        operationsToReplay = JSON.parse(operations);
                    } catch (e) {
                        // nothing to do
                    }
                } else if (_.isArray(operations)) {
                    operationsToReplay = operations;
                }

                if (operationsToReplay) {
                    // now try to process the operations
                    replayOperationNext(operationsToReplay, delay);
                }
            }
        };

        /**
         * Helper function for generating a logging string, that contains
         * all strings of an array in unique, sorted form together with
         * the number of occurences in the array.
         * Input:
         * ['abc', 'def', 'abc', 'ghi', 'def']
         * Output:
         * 'abc (2), def (2), ghi'
         *
         * @param {Array} arr
         *  An array with strings.
         *
         * @returns {String}
         *  The string constructed from the array content.
         */
        function getSortedArrayString(arr) {

            var counter = {},
                arrayString = '';

            _.countBy(arr, function (elem) {
                counter[elem] = counter[elem] ? counter[elem] + 1 : 1;
            });

            _.each(_.keys(counter).sort(), function (elem, index) {
                var newString = elem + ' (' + counter[elem] + ')';
                newString += ((index + 1) < _.keys(counter).length) ? ', ' : '';
                arrayString += newString;
            });

            return arrayString;
        }

        /**
         * Generating a document string from the editdiv element. jQuery data
         * elements are saved at the sub elements with the attribute 'jquerydata'
         * assigned to these elements.
         *
         * @param {jQuery.Deferred} [maindef]
         *  The deferred used to update the progress.
         *
         * @param {Number} [startProgress]
         *  The start value of the notification progress.
         *
         * @param {Number} [endProgress]
         *  The end value of the notification progress.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected if the document string
         *  could be generated successfully.
         */
        this.getFullModelDescription = function (maindef, startProgress, endProgress) {

            var // collecting all keys from data object for debug reasons
                dataKeys = [],
                // the result deferred object
                def = $.Deferred(),
                // the jQuery collection of all nodes in the document (very fast)
                allPageNodes = null,
                // the number of nodes in the document
                nodeCounter = 0,
                // the number of chunks, that will be evaluated
                chunkNumber = 5,
                // the current notification progress
                currentNotify = startProgress,
                // the number of elements in one chunk
                chunkLength = 0,
                // the notify difference for one chunk
                notifydiff = Utils.round((endProgress - startProgress) / chunkNumber, 0.01),
                // a selected drawing
                selectedDrawing = null,
                // whether the jQuery data objects could be successfully stored as attributes
                dataError = false;

            // converting all editdiv information into a string, saving especially the data elements
            // finding all elements, that have a 'data' jQuery object set.
            // Helper function to split the array of document nodes into smaller parts.
            function prepareNodeForStorage(chunk) {

                // updating progress bar
                if (maindef) {
                    currentNotify += Utils.round(notifydiff, 0.01);
                    maindef.notify(currentNotify); // updating corresponding to the length inside each -> smaller snippets deferred.
                }

                // iterating over all elements of the current chunk
                chunk.each(function () {

                    var // the current node, as jQuery object
                        node = $(this),
                        // the jQuery data object of the node
                        dataObject = null;

                    try {
                        // simplify restauration of empty text nodes
                        if ((node.is('span')) && (DOM.isEmptySpan(this))) {
                            dataKeys.push('isempty');
                            // using nonjquerydata instead of jquerydata (32286)
                            node.attr('nonjquerydata', JSON.stringify({ isempty: true }));
                        }

                        // handling jQuery data objects
                        dataObject = node.data();
                        if (_.isObject(dataObject) && !_.isEmpty(dataObject)) {
                            dataKeys = dataKeys.concat(_.keys(dataObject));
                            node.attr('jquerydata', JSON.stringify(dataObject));  // different key -> not overwriting isempty value
                            // Utils.log('Saving data: ' + this.nodeName + ' ' + this.className + ' : ' + JSON.stringify($(this).data()));

                            // handling images that have the session used in the src attribute
                            if (DOM.isImageNode(node)) {
                                var imgNode = node.find('img'),
                                    srcAttr = imgNode.attr('src');
                                if (_.isString(srcAttr) && (srcAttr.length > 0)) {
                                    imgNode.attr('src', srcAttr.replace(/\bsession=(\w+)\b/, 'session=_REPLACE_SESSION_ID_'));
                                }
                            }
                        }

                        // removing classes, multi selection possible -> no 'else if'
                        if (node.is('span.spellerror')) {
                            // remove runtime classes
                            node.removeClass('spellerror');
                        } else if (node.is('div')) {
                            // remove runtime classes
                            node.removeClass('break-above last-on-page has-breakable-span selected p_spelled');
                        } else if (node.is('table')) {
                            // remove runtime classes
                            node.removeClass('tb-split-nb');
                        } else if (node.is('tr')) {
                            // remove runtime classes
                            node.removeClass('break-above-tr');
                        } else if (node.is('span.break-above-span')) {
                            // remove runtime classes
                            node.removeClass('break-above-span');
                            CharacterStyles.mergeSiblingTextSpans(node, true);
                            CharacterStyles.mergeSiblingTextSpans(node);
                        }
                    } catch (ex) {
                        dataError = true;
                        Utils.info('quitHandler, failed to save document in local storage (3): ' + ex.message);
                    }
                });

            }

            // removing drawing selections
            if (selection.getSelectionType() === 'drawing') {
                selectedDrawing = selection.getSelectedDrawing();
                DrawingFrame.clearSelection(selectedDrawing);
                // removing additional tracking handler options, that are specific for OX Text (Chrome only)
                selectedDrawing.removeData('trackingOptions');
                selectedDrawing.removeData('tracking-options');
            }

            // removing artificial selection of empty paragraphs
            if (!_.isEmpty(highlightedParagraphs)) { removeArtificalHighlighting(); }

            // collecting all nodes of the page content
            allPageNodes = self.getNode().children(':first').find('*');
            allPageNodes.remove('div.page-break');  // not saving page break nodes
            allPageNodes.remove('tr.pb-row');  // not saving page break rows inside tables
            //allPageNodes.find('span.break-above-span').removeClass('break-above-span'); // removing marker class for inserting pagebreak inside p when split it
            nodeCounter = allPageNodes.length;
            chunkLength = Utils.round(nodeCounter / chunkNumber + 1, 1);

            app.processArrayDelayed(prepareNodeForStorage, allPageNodes, { chunkLength: chunkLength })
            .done(function () {
                if (dataError) {
                    Utils.info('quitHandler, failed to save document in local storage (4).');
                    def.reject();
                } else {
                    if (!_.isEmpty(dataKeys)) { Utils.info('Saving data keys: ' + getSortedArrayString(dataKeys)); }
                    def.resolve(self.getNode().children(':first').html());
                }
            })
            .fail(function () {
                Utils.info('quitHandler, failed to save document in local storage (5).');
                def.reject();
            })
            .always(function () {
                allPageNodes = null;
            });

            return def.promise();
        };

        /**
         * Generating the editdiv element from an html string. jQuery data
         * elements are restored at the sub elements, if the attribute 'jquerydata'
         * is assigned to these elements.
         *
         * @param {String} htmlString
         *  The html string for the editdiv element.
         */
        this.setFullModelNode = function (htmlString) {

            var // collecting all keys assigned to the data objects
                dataKeys = [],
                // the old and new image src attribute value
                srcAttr = null;

            if (htmlString) {
                // setting new string to editdiv node
                this.getNode()
                    // .parent()
                    .children(':first')
                    .html(htmlString)
                    .find('[jquerydata], [nonjquerydata]')
                    .each(function () {

                        var // the data saved in the string for the node
                            dataObject;

                        if ($(this).attr('nonjquerydata')) {
                            dataObject = JSON.parse($(this).attr('nonjquerydata'));
                            for (var key in dataObject) {
                                if (key === 'isempty') {  // the only supported key for nonjquerydata
                                    DOM.ensureExistingTextNode(this);
                                    dataKeys.push(key);
                                }
                            }
                            $(this).removeAttr('nonjquerydata');
                        }

                        if ($(this).attr('jquerydata')) {
                            // Utils.log('Restoring style at: ' + this.nodeName + ' ' + this.className + ' : ' + $(this).attr('jquerydata'));
                            dataObject = JSON.parse($(this).attr('jquerydata'));
                            for (var key in dataObject) {
                                $(this).data(key, dataObject[key]);
                                dataKeys.push(key);
                            }

                            // handling images that have the session used in the src attribute
                            if (DOM.isImageNode(this)) {
                                srcAttr = $(this).find('img').attr('src');
                                if (srcAttr && _.isString(srcAttr)) {
                                    $(this).find('img').attr('src', srcAttr.replace(/\bsession=_REPLACE_SESSION_ID_\b/, 'session=' + ox.session));
                                }
                            }
                            $(this).removeAttr('jquerydata');
                        }
                    });

                if (!_.isEmpty(dataKeys)) { Utils.info('Setting data keys: ' + getSortedArrayString(dataKeys)); }
            }
        };

        /**
         * Debugging code for performance measurements. html code is the parameter for
         * this function. This code is sent to the server, returned again and included
         * into the DOM at the global editdiv element.
         *
         * @param {String} htmlCode
         *  The html code for the global editdiv element.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  formatting has been updated successfully, or rejected when an error
         *  has occurred.
         */
        this.applyPageHtmlCode = function (htmlCode) {

            var def = null;
            // sending html code to server
            def = app.sendFilterRequest({
                method: 'POST',
                params: {
                    action: 'gethtmlcode',
                    htmlcode: htmlCode
                },
                resultFilter: function (data) {
                    // returning undefined rejects the entire request
                    return Utils.getStringOption(data, 'htmlcode', '');
                }
            })
            .done(function (htmlString) {
                // setting answer from server to the editdiv
                if (htmlString) { editdiv.html(htmlString); }
                def.resolve();
            })
            .fail(function () { def.reject(); });

            return def.promise();
        };

        //spell checking
        // split a paragraph text into words and return an array of words and star/end indices
        this.splitParagraphInWords = function (paragraph) {
            // ' \t.,;?!":'
            var words = [];
            var normalParagraph = paragraph.replace(/\s|,|\.|;|:|_/g, ' ');
            var wordstart = 0;
            var wordend = 0;
            for (var index = 0; index < normalParagraph.length; ++index) {

                if (normalParagraph.charAt(index) === ' ') {
                    if (wordstart < index - 1) {
                        wordend = index;
                        var word = {
                                text: normalParagraph.substring(wordstart, wordend),
                                start: wordstart,
                                end: wordend
                            };
                        words[words.length] = word;
                        wordstart = wordend + 1;
                    } else {
                        wordstart++;
                    }
                }

            }
            if (normalParagraph.length > 1 && normalParagraph.charAt(normalParagraph.length - 1) !== ' ') {
                var word = {
                        text: normalParagraph.substring(wordstart,
                        normalParagraph.length),
                        start: wordstart,
                        end: normalParagraph.length
                    };
                words[words.length] = word;
            }
            return words;
        };

        function implStartOnlineSpelling() {
            if (app.isImportFinished()) {
                if (onlineSpelling === true && self.getEditMode()) {
                    // restart spell checking
                    if (onlineSpellingTimeout) { onlineSpellingTimeout.abort(); }
                    onlineSpellingTimeout = app.executeDelayed(implCheckSpelling, { delay: 1000 });
                }
            }
        }

        this.getSpellReplacements = function (word, language) {
            var replacements = [];
            if (spellReplacmentsByLocale[language])
                replacements = spellReplacmentsByLocale[language][word];
            return replacements;
        };

        function implCheckSpelling() {

            var paragraphSpelled = false;

            Utils.iterateSelectedDescendantNodes(editdiv, DOM.PARAGRAPH_NODE_SELECTOR, function (paragraph) {

                var // all text spans in the paragraph, as array
                    textSpans = [],
                    // all spans of the paragraph - each span is a word/language object
                    paraSpans = [],
                    //predecessor of the current paragraph
                    prevParagraph = Utils.findPreviousNode(editdiv, paragraph, DOM.PARAGRAPH_NODE_SELECTOR);

                // adds information of the text span located at 'spanInfo.index' in the 'textSpans' array to 'spanInfo'
//                function getTextSpanInfo() {
//                    spanInfo.span = textSpans[spanInfo.index];
//                    spanInfo.length = spanInfo.span ? spanInfo.span.firstChild.nodeValue.length : 0;
//                }

                // goes to the next text span in the 'textSpans' array and updates all information in 'spanInfo'
//                function getNextTextSpanInfo() {
//                    spanInfo.index += 1;
//                    spanInfo.start += spanInfo.length;
//                    getTextSpanInfo();
//                }
                if (!paragraphSpelled && (spellCheckerCurrentNode === null || spellCheckerCurrentNode === prevParagraph)) {
                    spellCheckerCurrentNode = paragraph;
                    if (!$(paragraph).hasClass('p_spelled')) {
                        paragraphSpelled = true;
                        $(paragraph).addClass('p_spelled');

                        // collect all non-empty text spans in the paragraph
                        Position.iterateParagraphChildNodes(paragraph, function (node) {

                            // DOM.iterateTextSpans() skips drawing nodes
                            DOM.iterateTextSpans(node, function (span) {
                                if (!DOM.isListLabelNode(span.parentNode)) {
                                    var charAttributes = characterStyles.getElementAttributes(span);
                                    characterStyles.setElementAttributes(span, { character: { spellerror: null } }, { special: true });
                                    if (charAttributes.character.url === '') {
                                        textSpans.push(span);
                                    }
                                }
                            });
                        }, undefined, { allNodes: true });

                        _(textSpans).each(function (span) {
                            var charAttributes = characterStyles.getElementAttributes(span).character;
                            paraSpans[paraSpans.length] = {word: span.firstChild.nodeValue, locale: charAttributes.language};
                        });
                        if (paraSpans.length > 0) {
                            app.sendRequest({
                                method: 'POST',
                                module: 'spellchecker',
                                params: {
                                    action: 'spellparagraph',
                                    paragraph: JSON.stringify(paraSpans)
                                },
                                resultFilter: function (data) {
                                    // returning undefined rejects the entire request
                                    return Utils.getArrayOption(data, 'spellResult', undefined, true);
                                }
                            })
                            .done(function (spellResult) {

                                var errorSpanArray = [],
                                    spanOffset = 0,
                                    spanLength;

                                _(spellResult).each(function (result) {
                                    if (result.replacements && result.locale && result.word) {
                                        if (!spellReplacmentsByLocale[result.locale])
                                            spellReplacmentsByLocale[result.locale] = {};
                                        spellReplacmentsByLocale[result.locale][result.word] = result.replacements;
                                    }
                                });

                                _(textSpans).each(function (currentSpan) {
                                    var
                                        result,
                                        localErrors = [];

                                    spanLength = currentSpan.firstChild.nodeValue.length;
                                    for (result = 0; result < spellResult.length; ++result) {
                                        var currErrorStart = spellResult[result].start,
                                            currErrorLength = spellResult[result].length;

                                        var maxError = currErrorStart + currErrorLength,
                                        minError = currErrorStart;

                                        if (maxError <= spanOffset)
                                            continue;
                                        else if (minError >= spanOffset + spanLength)
                                            break;
                                        else {
                                            if (minError < spanOffset) {
                                                currErrorLength -= spanOffset - minError;
                                                minError = 0;
                                            }
                                            if (minError <= spanOffset) {
                                                currErrorStart = 0;
                                            } else {
                                                currErrorStart = (minError - spanOffset);
                                            }

                                            if (maxError > spanOffset + spanLength)
                                                currErrorLength = spanLength - (currErrorStart - spanOffset);
                                            localErrors[localErrors.length] = {start: currErrorStart, length: currErrorLength};
                                        }
                                    }
                                    errorSpanArray[errorSpanArray.length] = {span: currentSpan, errors: localErrors};
                                    spanOffset += spanLength;
                                });

                                _(errorSpanArray).each(function (errorSpan) {
                                    var error = 0,
                                        localError,
                                        spanPositionOffset = 0,
                                        newSpan,
                                        currentSpan = errorSpan.span;
                                    for (;error < errorSpan.errors.length; ++error) {
                                        localError = errorSpan.errors[error];
                                        newSpan = null;
                                        if (localError.start - spanPositionOffset > 0 &&
                                                (localError.start - spanPositionOffset) < currentSpan.textContent.length) {
                                            DOM.splitTextSpan(currentSpan, localError.start - spanPositionOffset);
                                            spanPositionOffset += localError.start - spanPositionOffset;
                                        }
                                        // split end of text span NOT covered by the error
                                        if (localError.length > 0 && currentSpan.textContent.length > localError.length) {
                                            //if currErrorLength > currentSpanLength
                                            //then assign
                                            newSpan = DOM.splitTextSpan(currentSpan, localError.length, { append: true });
                                            spanPositionOffset += localError.length;
                                        }
                                        characterStyles.setElementAttributes(currentSpan, { character: { spellerror: true } }, { special: true });
                                        if (newSpan !== null)
                                            currentSpan = newSpan[0];
                                    }
                                });

                                // restore the selection, but not if a drawing or a table cell is selected (Task 26214)
                                if (self.isTextOnlySelected()) {
                                    selection.restoreBrowserSelection({ preserveFocus: true });
                                }
                            });

                        }
                    }
                }
            });

            //restart at the beginning if the end of the text is reached
            if (spellCheckerCurrentNode && spellCheckerCurrentNode.nextSibling === null) {
                spellCheckerCurrentNode = null;
            }

            onlineSpellingTimeout = app.executeDelayed(implCheckSpelling, { delay: 1000 });
        }

        this.isOnlineSpelling = function () {
            return onlineSpelling;
        };

        /**
         * Toggle automatic spell checking in the document.
         */
        this.setOnlineSpelling = function (state, noConfig) {
            var onlineSpellingPopup;
            if (state !== onlineSpelling || noConfig === true) {
                onlineSpelling = state;
                if (noConfig !== true)
                    app.setUserSettingsValue('onlineSpelling', onlineSpelling);
                editdiv.toggleClass('spellerror-visible', onlineSpelling);

                if (onlineSpelling === true) {
                    // start timeout handler
                    implStartOnlineSpelling();
                } else {
                    spellCheckerCurrentNode = null;
                    // stop online spelling
                    if (onlineSpellingTimeout) {
                        onlineSpellingPopup = $(self.getNode()).first().children('.inline-popup.spell-replacement');
                        onlineSpellingPopup.hide();
                        onlineSpellingTimeout.abort();
                        onlineSpellingTimeout = null;
                    }
                }
            }
        };

        /**
         * Spellcheck the current selection - works ATM only on selected word
         */
        this.checkSpelling = function () {

            var selectionText = '',
                language = 'en-US',
                //generator = this.createOperationsGenerator(),
                text = '';
                //url = '',
                //startPos = null,
                //start = selection.getStartPosition(),
                //end = selection.getEndPosition();

            if (!selection.hasRange()) {
                //TODO: select the word under/nearest to the cursor
            }

            if (this.hasEnclosingParagraph()) {
                // use range to retrieve text and possible url
                if (selection.hasRange()) {

                    // Find out the texts/locales of the selected text to provide them to the
                    // spell checking
                    selection.iterateNodes(function (node, pos, start, length) {
                        if ((start >= 0) && (length >= 0) && DOM.isTextSpan(node)) {
                            var nodeText = $(node).text();
                            if (nodeText) {
                                selectionText += text.concat(nodeText.slice(start, start + length));
                            }
                            var charAttributes = characterStyles.getElementAttributes(node).character;
                            if (charAttributes.language.length > 0)
                                language = charAttributes.language;
                        }
                    });
                }

            }

            app.sendRequest({
                module: 'spellchecker',
                params: {
                    action: 'spellwordreplacements',
                    word: encodeURIComponent(selectionText),
                    locale: language
                },
                resultFilter: function (data) {
                    // returning undefined rejects the entire request
                    return Utils.getArrayOption(data, 'SpellReplacements', undefined, true);
                }
            })
            .done(function (spellReplacements) {
                alert(selectionText + ': ' + spellReplacements.join(' '));
            });
        };

        this.setRecordingOperations = function (state) {
            recordOperations = state;
        };

        this.isRecordingOperations = function () {
            return recordOperations;
        };

        function initDocument() {

            var // container for the top-level paragraphs
                pageContentNode = DOM.getPageContentNode(editdiv),
                // the initial paragraph node in an empty document
                paragraph = DOM.createImplicitParagraphNode();

            app.on('docs:import:error', function () { editdiv.attr('contenteditable', false); });

            // create empty page with single paragraph
            pageContentNode.empty().append(paragraph);
            validateParagraphNode(paragraph);

            // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
            // newly, also in other browsers (new jQuery version?)
            repairEmptyTextNodes(paragraph);

            // initialize default page formatting
            pageStyles.updateElementFormatting(editdiv);

            // register keyboard event handlers on the clipboard node to handle key strokes when
            // a drawing is selected in which case the browser selection is inside the clipboard node
            selection.getClipboardNode().on({ keydown: processKeyDown,
                                              keypress: processKeyPressed,
                                              cut: _.bind(self.cut, self),
                                              copy: _.bind(self.copy, self) });
        }

        /**
         * Returns whether the document contains any highlighted ranges.
         */
        this.hasHighlighting = function () {
            return highlightedSpans.length > 0;
        };

        /**
         * Removes all highlighting (e.g. from quick-search) from the document
         * and clears the search results.
         */
        this.removeHighlighting = function () {

            // remove highlighting and merge sibling text spans
            _(highlightedSpans).each(function (span) {
                characterStyles.setElementAttributes(span, { character: { highlight: false } }, { special: true });
                CharacterStyles.mergeSiblingTextSpans(span);
                CharacterStyles.mergeSiblingTextSpans(span, true);
            });
            highlightedSpans = [];

            // used for search and replace
            searchResults = [];
            currentSearchResultId = 0;

            // fade in entire document
            editdiv.removeClass('highlight');
        };

        /**
         * Sets the browser selection to the search result given by the id of the searchResults array
         *
         * @param {Number} id
         *  The the id of search result to select.
         */
        this.selectSearchResult = function (id) {
            var // the search relsult as oxo positions
                result = searchResults[id];

            if (result) {
                currentSearchResultId = id;
                // Bug 28386: defer selection, otherwise Enter key event will delete selected search result
                _.defer(function () {
                    app.getView().grabFocus();
                    selection.setTextSelection(result.start, result.end);
                });
            }
        };

        /**
         * Sets the browser selection to the next search result,
         * does nothing if no results are available.
         */
        this.selectNextSearchResult = function () {

            if (searchResults.length > 0) {
                var id = (currentSearchResultId < searchResults.length - 1) ? currentSearchResultId + 1 : 0;
                this.selectSearchResult(id);
            }
        };

        /**
         * Sets the browser selection to the previous search result,
         * does nothing if no results are available.
         */
        this.selectPreviousSearchResult = function () {

            if (searchResults.length > 0) {
                var id = (currentSearchResultId > 0) ? currentSearchResultId - 1 : searchResults.length - 1;
                this.selectSearchResult(id);
            }
        };

        /**
         * Replaces the search result defined by the given id with the replacement text.
         *
         * @param {Number} id
         *  The the id of search result to replace.
         *
         * @param {String} replacement
         *  The replacement for the found text occurences.
         */
        this.replaceSearchResult = function (id, replacement) {

            var // the search result as range of oxo positions
                currentResult = searchResults[id], result,
                // the delta of the searched and replaced text
                textDelta,
                // the highlighted text node
                node, position, startPosition, endPosition,
                // whether the replacment string is empty
                isEmptyReplace = false;

            if (!currentResult) {
                return;
            }

            startPosition = _.clone(currentResult.start);
            endPosition = _.clone(currentResult.end);
            replacement = _.isString(replacement) ? replacement : '';
            textDelta = replacement.length - (_.last(endPosition) - _.last(startPosition));
            isEmptyReplace = replacement.length === 0;

            undoManager.enterUndoGroup(function () {
                // indicate that these operations should not trigger a remove of the highlighting
                // TODO: piggiback this onto the operations
                isReplaceOperation = true;

                // if we replace a text next to another higlighted text span,
                // we would inherit the highlighting if we do the replacement the common way.
                // In order to keep the span(s) which contains the text to replace,
                // we first add the replacement at the start position.
                // Taking care of empty replacement strings, task 30347
                if (! isEmptyReplace) {
                    self.insertText(replacement, startPosition);
                    endPosition[endPosition.length - 1] += replacement.length;
                }

                // then remove the highlighting
                position = _.clone(currentResult.start);
                while (_.last(position) < _.last(currentResult.end)) {
                    // loop through the range and remove the highlight character style from the spans
                    node = Position.getSelectedElement(editdiv, position, 'span');
                    position[position.length - 1] += $(node).text().length || 1;
                    characterStyles.setElementAttributes(node, { character: { highlight: false } }, { special: true });
                }

                // finally delete the previous text.
                startPosition[startPosition.length - 1] += replacement.length;
                selection.setTextSelection(startPosition, endPosition);
                self.deleteSelected();

                // reset indicator
                isReplaceOperation = false;
            });

            // update all positions that are in the same paragraph or table cell
            for (var updateId = id + 1; updateId < searchResults.length; updateId++) {
                result = searchResults[updateId];
                if (result && Position.hasSameParentComponent(currentResult.start, result.start, 1)) {
                    result.start[result.start.length - 1] += textDelta;
                    result.end[result.end.length - 1] += textDelta;
                } else {
                    break;
                }
            }

            // remove replaced result from search results
            searchResults = _.without(searchResults, currentResult);
            if (currentSearchResultId > searchResults.length - 1) {
                currentSearchResultId = 0;
            }
        };


        /**
         * Replaces the current search result with the replacement text.
         *
         * @param {String} replacement
         *  The replacement for the found text occurences.
         */
        this.replaceSelectedSearchResult = function (replacement) {
            this.replaceSearchResult(currentSearchResultId, replacement);
            this.selectSearchResult(currentSearchResultId);
        };

        /**
         * Debounced quickSearch, will postpone its execution until 2 seconds
         * have elapsed since the last time it was invoked.
         *
         * @param {String} query
         *  The text that will be searched in the document.
         *
         *  @param {Number} [selectId]
         *  The the id of search result to be selected after the search run.
         */
        this.debouncedQuickSearch = (function () {

            var lastQuery = null, lastSelectId = null;

            // direct callback: called everytime the method Editor.debouncedQuickSearch() is called
            function storeParameters(query, selectId) {
                lastQuery = query;
                lastSelectId = selectId;
            }

            // deferred callback: called once after the timeout, uses the last parameters passed to direct callback
            function executeQuickSearch() {
                self.quickSearch(lastQuery, lastSelectId);
            }

            // create and return the debounced method Editor.debouncedQuickSearch()
            return app.createDebouncedMethod(storeParameters, executeQuickSearch, { delay: 2000 });

        }()); // Editor.debouncedQuickSearch()

        /**
         * Searches and highlights the passed text in the entire document.
         * If a selectId is given the corresponding search result is selected.
         *
         * @param {String} query
         *  The text that will be searched in the document.
         *
         *  @param {Number} [selectId]
         *  The the id of search result to be selected after the search run.
         *
         * @returns {Boolean}
         *  Whether the passed text has been found in the document.
         */
        this.quickSearch = function (query, selectId) {

            // remove old highlighting and clear previous search results
            this.removeHighlighting();

            // check input parameter
            if (!_.isString(query) || (query.length === 0)) {
                return false;
            }
            query = query.toLowerCase();

            // search in all paragraph nodes (also embedded in tables etc.)
            Utils.iterateSelectedDescendantNodes(editdiv, DOM.PARAGRAPH_NODE_SELECTOR, function (paragraph) {

                var // all text spans in the paragraph, as array
                    textSpans = [],
                    // the concatenated text from all text spans
                    elementText = '',
                    // all matching ranges of the query text in the complete paragraph text
                    matchingRanges = [], start = 0,
                    // information about a text span while iterating matching ranges
                    spanInfo = { index: 0, start: 0 },
                    // start, end position of the range
                    startPosition,
                    endPosition;

                // adds information of the text span located at 'spanInfo.index' in the 'textSpans' array to 'spanInfo'
                function getTextSpanInfo() {
                    spanInfo.span = textSpans[spanInfo.index];
                    spanInfo.length = spanInfo.span ? spanInfo.span.firstChild.nodeValue.length : 0;
                }

                // goes to the next text span in the 'textSpans' array and updates all information in 'spanInfo'
                function getNextTextSpanInfo() {
                    spanInfo.index += 1;
                    spanInfo.start += spanInfo.length;
                    getTextSpanInfo();
                }

                // collect all non-empty text spans in the paragraph
                Position.iterateParagraphChildNodes(paragraph, function (node) {
                    // iterate child nodes...
                    if (DOM.isTextSpan(node)) {
                        // for spans add the text
                        textSpans.push(node);
                        elementText += $(node).text();
                    } else if (!DOM.isListLabelNode(node)) {
                        // for all but list labels (they don't have an oxo position) add a replacement (FFFC - Object replacement char)
                        textSpans.push(Utils.getDomNode(DOM.createTextSpan().text('\ufffc')));
                        elementText += '\ufffc';
                    }
                }, this, { allNodes: true });

                // replace all whitespace characters, and convert to lower case
                // for case-insensitive matching
                elementText = elementText.replace(/\s/g, ' ').toLowerCase();

                // find all occurrences of the query text in the paragraph text
//                while ((start = elementText.indexOf(query, start)) >= 0) {
//                    // try to merge with last offset range (overlapping like foofoof when query is foof)
//                    if ((matchingRanges.length > 0) && (_(matchingRanges).last().end >= start)) {
//                        _(matchingRanges).last().end = start + query.length;
//                    } else {
//                        matchingRanges.push({ start: start, end: start + query.length });
//                    }
//                    // continue at next character (occurrences of the query text may overlap)
//                    start += 1;
//                }

                // find all occurrences of the query text in the paragraph text
                while ((start = elementText.indexOf(query, start)) >= 0) {
                    matchingRanges.push({ start: start, end: start + query.length });

                    startPosition = Position.getOxoPosition(editdiv, paragraph, 0);
                    startPosition.push(start);
                    endPosition = Position.increaseLastIndex(startPosition, query.length);
                    searchResults.push({start: startPosition, end: endPosition});

                    // continue after the currently handled text
                    start += query.length;
                }

                // set highlighting to all occurrences
                getTextSpanInfo();
                _(matchingRanges).each(function (range) {

                    // find first text span that contains text from current matching range
                    while (spanInfo.start + spanInfo.length <= range.start) {
                        getNextTextSpanInfo();
                    }

                    // process all text spans covered by the current matching range
                    while (spanInfo.start < range.end) {

                        // split beginning of text span not covered by the range
                        if (spanInfo.start < range.start) {
                            DOM.splitTextSpan(spanInfo.span, range.start - spanInfo.start);
                            // update spanInfo
                            spanInfo.length -= (range.start - spanInfo.start);
                            spanInfo.start = range.start;
                        }

                        // split end of text span NOT covered by the range
                        if (range.end < spanInfo.start + spanInfo.length) {
                            var newSpan = DOM.splitTextSpan(spanInfo.span, range.end - spanInfo.start, { append: true });
                            // insert the new span into textSpans after the current span
                            textSpans.splice(spanInfo.index + 1, 0, newSpan[0]);
                            // update spanInfo
                            spanInfo.length = range.end - spanInfo.start;
                        }

                        // set highlighting to resulting text span and store it in the global list
                        characterStyles.setElementAttributes(spanInfo.span, { character: { highlight: true } }, { special: true });
                        highlightedSpans.push(spanInfo.span);

                        // go to next text span
                        getNextTextSpanInfo();
                    }

                }, this);

                // exit at a certain number of found ranges (for performance)
                if (highlightedSpans.length >= 100) {
                    return Utils.BREAK;
                }

            }, this);

            if (highlightedSpans.length > 0) {
                // fade out entire document
                editdiv.addClass('highlight');
                // make first highlighted text node visible
                //app.getView().scrollToChildNode(highlightedSpans[0]);

                // select the given search result
                if (_.isNumber(selectId)) {
                    this.selectSearchResult(selectId);
                }
            }

            // return whether any text in the document matches the passed query text
            return this.hasHighlighting();
        };

        /**
         * Searches the passed text in the entire document
         * and replaces it with the the replacement text
         *
         * @param {String} query
         *  The text that will be searched in the document.
         *
         * @param {String} replacement
         *  The replacement for the found text occurences.
         *
         *  @param {Boolean} [matchCase=false]
         *  Determins if the search is case sensitive (optional, defaults to false).
         */
        this.searchAndReplaceAll = function (query, replacement, matchCase) {

            var // whether the replacment string is empty
                isEmptyReplace = false;

            // remove old quicksearch highlighting and clear previous search results
            this.removeHighlighting();

            // check input parameter
            if (!_.isString(query) || (query.length === 0)) {
                return;
            }

            query = matchCase ? query : query.toLowerCase();
            replacement = _.isString(replacement) ? replacement : '';
            isEmptyReplace = replacement.length === 0;

            // search in all paragraph nodes (also embedded in tables etc.)
            undoManager.enterUndoGroup(function () {
                Utils.iterateSelectedDescendantNodes(editdiv, DOM.PARAGRAPH_NODE_SELECTOR, function (paragraph) {

                    var // the concatenated text from all text spans
                        elementText = '', start = 0,
                        // start, end position of the matching range
                        startPosition, endPosition,
                        // the offset to apply to the start and end position of the range
                        offset = 0,
                        // the delta of the searched and replaced text
                        textDelta = replacement.length - query.length;

                    // collect all non-empty text spans in the paragraph
                    Position.iterateParagraphChildNodes(paragraph, function (node) {
                        // iterate child nodes...
                        if (DOM.isTextSpan(node)) {
                            // for spans add the text
                            elementText += $(node).text();
                        } else if (!DOM.isListLabelNode(node)) {
                            // for all but list labels (they don't have an oxo position) add a replacement (FFFC - Object replacement char)
                            elementText += '\ufffc';
                        }
                    }, this, { allNodes: true });

                    // replace all whitespace characters, and convert to lower case
                    // for case-insensitive matching
                    elementText = elementText.replace(/\s/g, ' ');
                    if (!matchCase) {
                        elementText = elementText.toLowerCase();
                    }

                    // find all occurrences of the query text in the paragraph text
                    while ((start = elementText.indexOf(query, start)) >= 0) {

                        startPosition = Position.getOxoPosition(editdiv, paragraph, 0);
                        startPosition.push(start + offset);
                        endPosition = Position.increaseLastIndex(startPosition, query.length);

                        selection.setTextSelection(startPosition, endPosition);
                        self.deleteSelected();
                        if (! isEmptyReplace) {
                            self.insertText(replacement, startPosition);
                        }

                        // from elementText we get the occurences before any replacement takes place,
                        // all replacements but the first one need to handle the text offset due to
                        // the replacement offsets of the previous occurences.
                        offset += textDelta;

                        // continue after the currently handled text
                        start += query.length;
                    }

                }, this);
            }, this); // enterUndoGroup
        };


        // ==================================================================
        // HIGH LEVEL EDITOR API which finally results in Operations
        // and creates Undo Actions.
        // Public functions, that are called from outside the editor
        // and generate operations.
        // ==================================================================

        /**
         * Generates the operations that will delete the current selection, and
         * executes the operations.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled. If no dialog is shown, the deferred is
         *  resolved immediately.
         */
        this.deleteSelected = function () {

            var // the operations generator
                generator = this.createOperationsGenerator(),
                // the position of the first and last partially covered paragraph
                firstPosition = null, lastPosition = null,
                // a helper paragraph for setting attributes
                paragraph = null,
                // whether a paragraph is selected completely (and requires merge)
                paragraphSelected = false,
                // whether the full paragraph will be deleted
                paragraphDeleted = false,
                // whether the selection is a rectangular cell selection
                isCellSelection = selection.getSelectionType() === 'cell',
                // whether it is necessary to ask the user before deleting content
                askUser = false,
                // the resulting promise
                promise = null;

            // helper function, that is necessary because the user might be asked,
            // if he really wants to delete the content.
            function doDeleteSelected() {

                // apply the operations (undo group is created automatically)
                guiTriggeredOperation = true;  // Fix for 30597
                self.applyOperations(generator.getOperations());
                guiTriggeredOperation = false;

                if (firstPosition) {
                    // set paragraph attributes immediately, if paragraph is without content
                    paragraph = Position.getParagraphElement(editdiv, firstPosition);
                    if ((paragraph) && ($(paragraph).text().length === 0)) {
                        validateParagraphNode(paragraph);
                    }
                }

                if (! isCellSelection) {
                    paragraph = Position.getParagraphElement(editdiv, lastPosition);
                    if ((paragraph) && ($(paragraph).text().length === 0)) {
                        validateParagraphNode(paragraph);
                    }
                } else {
                    // cell selection
                    if (firstPosition) {   // not complete table selected
                        lastOperationEnd = Position.getFirstPositionInCurrentCell(editdiv, firstPosition);
                    }
                }

                // collapse selection
                selection.setTextSelection(lastOperationEnd);
            }

            if (!selection.hasRange()) { return $.when(); }

            // visit all content nodes (tables, paragraphs) in the selection
            selection.iterateContentNodes(function (node, position, startOffset, endOffset, parentCovered) {

                var // whether the node is the last child of its parent
                    isLastChild = node === node.parentNode.lastChild;

                if (DOM.isParagraphNode(node)) {

                    // remember first and last paragraph
                    if (!firstPosition) { firstPosition = position; }
                    lastPosition = position;

                    // default: the paragraph will not be removed completely
                    // -> important for merging paragraphs after the selection is deleted
                    paragraphDeleted = false;

                    // checking if a paragraph is selected completely and requires merge
                    paragraphSelected = _.isNumber(startOffset) && (startOffset === 0) && !_.isNumber(endOffset);

                    if (! DOM.isImplicitParagraphNode(node)) {

                        // do not delete the paragraph node, if it is only covered partially;
                        // or if it is the last paragraph when the parent container is cleared completely
                        if (parentCovered ? isLastChild : (_.isNumber(startOffset) || _.isNumber(endOffset))) {

                            // 'deleteText' operation needs valid start and end position
                            startOffset = _.isNumber(startOffset) ? startOffset : 0;
                            endOffset = _.isNumber(endOffset) ? endOffset : (Position.getParagraphLength(editdiv, position) - 1);

                            // delete the covered part of the paragraph
                            if (isCellSelection) {
                                paragraphDeleted = true;
                                if (containsUnrestorableElements(node)) { askUser = true; }
                                generator.generateOperation(Operations.DELETE, { start: position });
                            } else if ((startOffset === 0) && (endOffset === 0) && (Position.getParagraphLength(editdiv, position) === 0)) {  // -> do not delete from [1,0] to [1,0]
                                paragraphDeleted = true;
                                generator.generateOperation(Operations.DELETE, { start: position });
                            } else if (startOffset <= endOffset) {
                                if (containsUnrestorableElements(node, startOffset, endOffset)) { askUser = true; }
                                generator.generateOperation(Operations.DELETE, { start: position.concat([startOffset]), end: position.concat([endOffset]) });
                            }
                        } else {
                            paragraphDeleted = true;
                            if (containsUnrestorableElements(node)) { askUser = true; }
                            generator.generateOperation(Operations.DELETE, { start: position });
                        }
                    } else {
                        paragraphSelected = false;
                    }

                } else if (DOM.isTableNode(node)) {
                    // delete entire table
                    generator.generateOperation(Operations.DELETE, { start: position });
                    // checking, if this is a table with exceeded size
                    if (DOM.isExceededSizeTableNode(node) || containsUnrestorableElements(node)) { askUser = true; }
                } else {
                    Utils.error('Editor.deleteSelected(): unsupported content node');
                    return Utils.BREAK;
                }

            }, this, { shortestPath: true });

            // operations MUST be executed in reverse order to preserve the positions
            generator.reverseOperations();

            // Generate a 'mergeParagraph' operation for the first paragraph
            // which will be merged with the remaining part of the last paragraph
            // but only, if paragraphs are different but in the same parent container
            // or it is only one completely selected paragraph.
            // Merging is not allowed, if the last paragraph was removed completely
            // (paragraphDeleted) or if the firstPosition and lastPostion are equal. This
            // two cases happen in Chrome tables with selections over more than one cell.
            // In Firefox tables there is a cell selection. In this case all paragraphs
            // are deleted completely, so that no merge is required.
            if (firstPosition && lastPosition && (! isCellSelection) && (! paragraphDeleted) && (! _.isEqual(firstPosition, lastPosition)) && Position.hasSameParentComponent(firstPosition, lastPosition) && ((_.last(firstPosition) !== _.last(lastPosition)) || (paragraphSelected))) {
                generator.generateOperation(Operations.PARA_MERGE, { start: firstPosition });
            }

            // Asking the user, if he really wants to remove the content, if it contains
            // a table with exceeded size or other unrestorable elements. This cannot be undone.
            if (askUser) {
                promise = showDeleteWarningDialog(
                    gt('Delete Contents'),
                    gt('Deleting the selected elements cannot be undone. Do you want to continue?')
                );
                // return focus to editor after 'No' button
                promise.fail(function () { app.getView().grabFocus(); });
            } else {
                promise = $.when();
            }

            // delete contents on resolved promise
            return promise.done(doDeleteSelected);
        };

        /**
         * Editor API function to generate 'delete' operations.
         * This is a generic function, that can be used to delete any component (text, paragraph,
         * cell, row, table, ...). Deleting columns is not supported, because columns cannot
         * be described with a logical position.
         * The parameter 'start' and 'end' are used to specify the position of the components that
         * shall be deleted. For all components except 'text' the 'end' position will be ignored.
         * For paragraphs, cells, ... only one specific component can be deleted within this
         * operation. Only on text level a complete range can be deleted.
         *
         * @param {Number[]} start
         *  The logical start position.
         *
         * @param {Number[]} [end]
         *  The logical end position (optional). This can be different from 'start' only for text ranges
         *  inside one paragraph. A text range can include characters, fields, and drawing objects,
         *  but must be contained in a single paragraph.
         *
         * @param {Object} [options]
         *  Additional optional options, that can be used for performance reasons.
         *  @param {Boolean} [options.setTextSelection=true]
         *      If set to false, the text selection will not be set within this
         *      function. This is useful, if the caller takes care of setting the
         *      cursor after the operation. This is the case for 'Backspace' and
         *      'Delete' operations. It is a performance issue, that the text
         *      selection is not set twice.
         *  @param {Boolean} [options.handleUnrestorableContent=false]
         *      If set to true, it will be checked if the range, that shall be
         *      deleted, contains unrestorable content. In this case a dialog appears,
         *      in which the user is asked, if he really wants to delete this content.
         *      This is the case for 'Backspace' and 'Delete' operations.
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled. If no dialog is shown, the deferred is
         *  resolved immediately.
         */
        this.deleteRange = function (start, end, options) {
            var // a helper dom position element
                domPos = null,
                // whether the cursor shall be set in this function call
                setTextSelection = Utils.getBooleanOption(options, 'setTextSelection', true),
                // whether start and end position are different
                handleUnrestorableContent = Utils.getBooleanOption(options, 'handleUnrestorableContent', false),
                // whether start and end logical position are different
                isSimplePosition = ((! _.isArray(end)) || (_.isEqual(start, end))),
                // a paragraph node
                paragraph = null,
                // whether it is necessary to ask the user before deleting range
                askUser = false,
                // the resulting promise
                promise = null;

            // helper function, that is necessary because the user might be asked,
            // if he really wants to delete the range.
            function doDeleteRange() {

                // Using end as it is, not subtracting '1' like in 'deleteText'
                guiTriggeredOperation = true;  // Fix for 30597
                self.applyOperations({ name: Operations.DELETE, start: _.clone(start), end: _.clone(end) });
                guiTriggeredOperation = false;

                // Setting paragraph attributes immediately, if paragraph is without content.
                // This avoids jumping of following paragraphs.
                if ((_.last(start) === 0) && (_.last(end) === 0)) {
                    domPos = Position.getDOMPosition(editdiv, _.chain(start).clone().initial().value(), true);
                    if ((domPos) && (domPos.node) && ($(domPos.node).is(DOM.PARAGRAPH_NODE_SELECTOR)) && ($(domPos.node).text().length === 0)) {
                        validateParagraphNode(domPos.node);
                    }
                }

                // setting the cursor position
                if (setTextSelection) {
                    selection.setTextSelection(lastOperationEnd);
                }
            }

            end = end || _.clone(start);

            // checking, if an unrestorable
            if (handleUnrestorableContent) {
                paragraph = Position.getParagraphElement(editdiv, _.initial(start));
                if (isSimplePosition && paragraph && containsUnrestorableElements(paragraph, _.last(start), _.last(start))) { askUser = true; }
            }

            // Asking the user, if he really wants to remove the content, if it contains
            // a table with exceeded size or other unrestorable elements. This cannot be undone.
            if (askUser) {
                promise = showDeleteWarningDialog(
                    gt('Delete Contents'),
                    gt('Deleting the selected elements cannot be undone. Do you want to continue?')
                );
                // return focus to editor after 'No' button
                promise.fail(function () { app.getView().grabFocus(); });
            } else {
                promise = $.when();
            }

            // delete contents on resolved promise
            return promise.done(doDeleteRange);
        };

        /**
         * Editor API function to generate 'delete' operations for rows or  a complete table. This
         * function is triggered by an user event. It evaluates the current selection. If the
         * selection includes all rows of the table, the table is removed completely within one
         * operation. If the row or table contains unrestorable content, the user is asked, whether
         * he really wants to delete the rows.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled. If no dialog is shown,
         *  the deferred is resolved immediately.
         */
        this.deleteRows = function () {

            var // the operations generator
                generator = this.createOperationsGenerator(),
                // the start position of the selection
                position = selection.getStartPosition(),
                // the index of the row inside the table, in which the selection starts
                start = Position.getRowIndexInTable(editdiv, position),
                // the index of the row inside the table, in which the selection ends
                end = selection.hasRange() ? Position.getRowIndexInTable(editdiv, selection.getEndPosition()) : start,
                // the logical position of the table
                tablePos = Position.getLastPositionFromPositionByNodeName(editdiv, position, DOM.TABLE_NODE_SELECTOR),
                // the index of the last row in the table
                lastRow = Position.getLastRowIndexInTable(editdiv, position),
                // whether the complete table shall be deleted
                isCompleteTable = ((start === 0) && (end === lastRow)) ? true : false,
                // logical row position
                rowPosition = null,
                // loop counter
                i = 0,
                // whether it is necessary to ask the user before removing row or table
                askUser = false,
                // the resulting promise
                promise = null;

            // helper function, that is necessary because the user might be asked,
            // if he really wants to delete the selected rows.
            function doDeleteRows() {

                guiTriggeredOperation = true;
                self.applyOperations(generator.getOperations());
                guiTriggeredOperation = false;

                // setting the cursor position after deleting content
                selection.setTextSelection(lastOperationEnd);
            }

            // Generating only one operation, if the complete table is removed.
            // Otherwise sending one operation for removing each row.
            if (isCompleteTable) {
                generator.generateOperation(Operations.DELETE, { start: _.copy(tablePos, true) });
                if (containsUnrestorableElements(tablePos)) { askUser = true; }
            } else {
                for (i = end; i >= start; i--) {
                    rowPosition = _.clone(tablePos);
                    rowPosition.push(i);
                    generator.generateOperation(Operations.DELETE, { start: rowPosition });
                    if (containsUnrestorableElements(rowPosition)) { askUser = true; }
                }
            }

            // Asking the user, if he really wants to remove the content, if it contains
            // at least one row with exceeded size or other unrestorable elements. This cannot be undone.
            if (askUser) {
                promise = showDeleteWarningDialog(
                    gt('Delete Rows'),
                    gt('Deleting the selected rows cannot be undone. Do you want to continue?')
                );
            } else {
                promise = $.when();
            }

            // delete rows on resolved promise
            return promise.done(doDeleteRows);
        };

        this.deleteCells = function () {

            var isCellSelection = selection.getSelectionType() === 'cell',
                startPos = selection.getStartPosition(),
                endPos = selection.getEndPosition(),
                localPos = null;

            startPos.pop();  // removing character position and paragraph
            startPos.pop();
            endPos.pop();
            endPos.pop();

            var startCol = startPos.pop(),
                endCol = endPos.pop(),
                startRow = startPos.pop(),
                endRow = endPos.pop(),
                tablePos = _.clone(startPos),
                operations = [];

            for (var i = endRow; i >= startRow; i--) {

                var rowPosition = Position.appendNewIndex(tablePos, i),
                    localStartCol = startCol,
                    localEndCol = endCol;

                if ((! isCellSelection) && (i < endRow) && (i > startCol))  {
                    // removing complete rows
                    localStartCol = 0;
                    localEndCol = Position.getLastColumnIndexInRow(editdiv, rowPosition);
                }

                for (var j = localEndCol; j >= localStartCol; j--) {
                    localPos = Position.appendNewIndex(rowPosition, j);
                    operations.push({ name: Operations.DELETE, start: localPos });
                }

                // removing empty row
                var rowNode = Position.getDOMPosition(editdiv, rowPosition).node;
                if ($(rowNode).children().length === 0) {
                    localPos = Position.appendNewIndex(tablePos, i);
                    operations.push({ name: Operations.DELETE, start: localPos });
                }

                // checking if the table is empty
                var tableNode = Position.getDOMPosition(editdiv, tablePos).node;
                if (Table.getRowCount(tableNode) === 0) {
                    operations.push({ name: Operations.DELETE, start: _.clone(tablePos) });
                }
            }

            // apply the operations (undo group is created automatically)
            this.applyOperations(operations);

            // setting the cursor position
            selection.setTextSelection(lastOperationEnd);
        };

        this.mergeCells = function () {

            var isCellSelection = selection.getSelectionType() === 'cell',
                startPos = selection.getStartPosition(),
                endPos = selection.getEndPosition();

            startPos.pop();  // removing character position and paragraph
            startPos.pop();
            endPos.pop();
            endPos.pop();

            var startCol = startPos.pop(),
                endCol = endPos.pop(),
                startRow = startPos.pop(),
                endRow = endPos.pop(),
                tablePos = _.clone(startPos),
                endPosition = null,
                operations = [];

            if (endCol > startCol) {

                for (var i = endRow; i >= startRow; i--) {  // merging for each row

                    var rowPosition = Position.appendNewIndex(tablePos, i);

                    var localStartCol = startCol,
                        localEndCol = endCol;

                    if ((! isCellSelection) && (i < endRow) && (i > startCol))  {
                        // merging complete rows
                        localStartCol = 0;
                        localEndCol = Position.getLastColumnIndexInRow(editdiv, rowPosition);
                    }

                    var count = localEndCol - localStartCol,
                        cellPosition = Position.appendNewIndex(rowPosition, localStartCol);
                    operations.push({ name: Operations.CELL_MERGE, start: cellPosition, count: count });

                    endPosition = _.clone(cellPosition);
                }

                // apply the operations (undo group is created automatically)
                this.applyOperations(operations);

                endPosition.push(0);
                endPosition.push(0);

                // setting the cursor position
                selection.setTextSelection(endPosition);
            }
        };

        this.insertCell = function () {

            var isCellSelection = selection.getSelectionType() === 'cell',
                startPos = selection.getStartPosition(),
                endPos = selection.getEndPosition(),
                count = 1,  // default, adding one cell in each row
                endPosition = null;

            startPos.pop();  // removing character position and paragraph
            startPos.pop();
            endPos.pop();
            endPos.pop();

            var startCol = startPos.pop(),
                endCol = endPos.pop(),
                startRow = startPos.pop(),
                endRow = endPos.pop(),
                tablePos = _.copy(startPos, true),
                attrs = { cell: {} },
                operations = [];

            for (var i = endRow; i >= startRow; i--) {

                var rowPosition = Position.appendNewIndex(tablePos, i),
                    localEndCol = endCol;

                if ((! isCellSelection) && (i < endRow) && (i > startCol))  {
                    // removing complete rows
                    localEndCol = Position.getLastColumnIndexInRow(editdiv, rowPosition);
                }

                localEndCol++;  // adding new cell behind existing cell
                var cellPosition = Position.appendNewIndex(rowPosition, localEndCol);
                attrs.cell.gridSpan = 1;  // only 1 grid for the new cell
                operations.push({ name: Operations.CELLS_INSERT, start: cellPosition, count: count, attrs: attrs });

                // Applying new tableGrid, if the current tableGrid is not sufficient
                var tableDomPoint = Position.getDOMPosition(editdiv, tablePos),
                    rowDomPoint = Position.getDOMPosition(editdiv, rowPosition);

                if (tableDomPoint && DOM.isTableNode(tableDomPoint.node)) {

                    var tableGridCount = tableStyles.getElementAttributes(tableDomPoint.node).table.tableGrid.length,
                        rowGridCount = Table.getColSpanSum($(rowDomPoint.node).children());

                    if (rowGridCount > tableGridCount) {

                        localEndCol--;  // behind is evaluated in getTableGridWithNewColumn
                        var insertmode = 'behind',
                            tableGrid = Table.getTableGridWithNewColumn(editdiv, tablePos, localEndCol, insertmode);

                        // Setting new table grid attribute to table
                        operations.push({ name: Operations.SET_ATTRIBUTES, attrs: { table: { tableGrid: tableGrid } }, start: _.clone(tablePos) });
                        requiresElementFormattingUpdate = false;  // no call of implTableChanged -> attributes are already set in implSetAttributes
                    }

                }

                endPosition = _.clone(cellPosition);
            }

            guiTriggeredOperation = true;

            // apply the operations (undo group is created automatically)
            this.applyOperations(operations);

            guiTriggeredOperation = false;
            requiresElementFormattingUpdate = true;

            endPosition.push(0);
            endPosition.push(0);

            // setting the cursor position
            selection.setTextSelection(endPosition);
        };

        /**
         * Editor API function to generate 'delete' operations for columns or a complete table. This
         * function is triggered by an user event. It evaluates the current selection. If the
         * selection includes all columns of the table, the table is removed completely within one
         * operation. If the column or table contains unrestorable content, the user is asked, whether
         * he really wants to delete the columns.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled. If no dialog is shown,
         *  the deferred is resolved immediately.
         */
        this.deleteColumns = function () {

            var // the operations generator
                generator = this.createOperationsGenerator(),
                // the start position of the selection
                position = selection.getStartPosition(),
                // the logical position of the table
                tablePos = Position.getLastPositionFromPositionByNodeName(editdiv, position, DOM.TABLE_NODE_SELECTOR),
                // the table node
                tableNode = Position.getDOMPosition(editdiv, tablePos).node,
                // the maximum grid number of the table
                maxGrid = Table.getColumnCount(tableNode) - 1,
                // the row node at the logical start position of the selection
                rowNode = Position.getLastNodeFromPositionByNodeName(editdiv, position, 'tr'),
                // the index of the column at the logical start position of the selection
                startColIndex = Position.getColumnIndexInRow(editdiv, position),
                // the index of the column at the logical end position of the selection
                endColIndex = selection.hasRange() ? Position.getColumnIndexInRow(editdiv, selection.getEndPosition()) : startColIndex,
                // a helper object for calculating grid positions in the table
                returnObj = Table.getGridPositionFromCellPosition(rowNode, startColIndex),
                // the start grid position in the table
                startGrid = returnObj.start,
                // the end grid position in the table
                endGrid = selection.hasRange() ? Table.getGridPositionFromCellPosition(rowNode, endColIndex).end : returnObj.end,
                // whether the complete table shall be deleted
                isCompleteTable = ((startGrid === 0) && (endGrid === maxGrid)) ? true : false,
                // logical row position
                rowPos = null,
                // logical cell position
                cellPos = null,
                // the index of the last row in the table
                maxRow = null,
                // whether all rows will be deleted
                deletedAllRows = false,
                // a jQuery collection containing all rows of the specified table
                allRows = null,
                // An array, that contains for each row an array with two integer values
                // for start and end position of the cells.
                allCellRemovePositions = null,
                // a row node
                currentRowNode = null,
                // Two integer values for start and end position of the cells in the row.
                oneRowCellArray = null,
                // helper numbers for cell descriptions inside a row
                start = 0, end = 0,
                // the table grid of a table
                tableGrid = null,
                // loop counter
                i = 0, j = 0,
                // whether it is necessary to ask the user before removing row or table
                askUser = false,
                // the resulting promise
                promise = null;

            // helper function, that is necessary because the user might be asked,
            // if he really wants to delete the selected rows.
            function doDeleteColumns() {

                // apply the operations (undo group is created automatically)
                guiTriggeredOperation = true;
                self.applyOperations(generator.getOperations());
                guiTriggeredOperation = false;
                requiresElementFormattingUpdate = true;

                // setting the cursor position after deleting the content
                selection.setTextSelection(lastOperationEnd);
            }

            // Generating only one operation, if the complete table is removed.
            // Otherwise sending operations for removing each columns, maybe for rows and
            // for setting new table attributes.
            if (isCompleteTable) {
                generator.generateOperation(Operations.DELETE, { start: _.copy(tablePos, true) });
                if (containsUnrestorableElements(tableNode)) { askUser = true; }
            } else {

                // generating delete columns operation, but further operations might be necessary
                generator.generateOperation(Operations.COLUMNS_DELETE, { start: tablePos, startGrid: startGrid, endGrid: endGrid });

                // Checking, if there will be rows without cells after the columns are deleted
                maxRow = Table.getRowCount(tableNode) - 1;
                deletedAllRows = true;
                allRows = DOM.getTableRows(tableNode);
                allCellRemovePositions = Table.getAllRemovePositions(allRows, startGrid, endGrid);

                for (i = maxRow; i >= 0; i--) {
                    rowPos = _.clone(tablePos);
                    rowPos.push(i);
                    currentRowNode = Position.getDOMPosition(editdiv, rowPos).node;
                    oneRowCellArray =  allCellRemovePositions[i];
                    end = oneRowCellArray.pop();
                    start = oneRowCellArray.pop();

                    if ($(currentRowNode).children().length === (end - start + 1)) {
                        generator.generateOperation(Operations.DELETE, { start: rowPos });
                        if (containsUnrestorableElements(rowPos)) { askUser = true; }
                    } else {
                        deletedAllRows = false;
                        // checking unrestorable content in all cells, that will be removed
                        for (j = start; j <= end; j++) {
                            cellPos = _.clone(rowPos);
                            cellPos.push(j);
                            if (containsUnrestorableElements(cellPos)) { askUser = true; }
                        }
                    }
                }

                // Deleting the table explicitely, if all its content was removed
                if (deletedAllRows) {
                    generator.generateOperation(Operations.DELETE, { start: _.clone(tablePos) });
                    if (containsUnrestorableElements(tablePos)) { askUser = true; }
                } else {
                    // Setting new table grid attribute to table
                    tableGrid = _.clone(tableStyles.getElementAttributes(tableNode).table.tableGrid);
                    tableGrid.splice(startGrid, endGrid - startGrid + 1);  // removing column(s) in tableGrid (automatically updated in table node)
                    generator.generateOperation(Operations.SET_ATTRIBUTES, { attrs: { table: { tableGrid: tableGrid } }, start: _.clone(tablePos) });
                    requiresElementFormattingUpdate = false;   // no call of implTableChanged -> attributes are already set in implSetAttributes
                }
            }

            // Asking the user, if he really wants to remove the columns, if it contains
            // at least one cell with exceeded size or other unrestorable elements. This cannot be undone.
            if (askUser) {
                promise = showDeleteWarningDialog(
                    gt('Delete Columns'),
                    gt('Deleting the selected columns cannot be undone. Do you want to continue?')
                );
            } else {
                promise = $.when();
            }

            // delete columns on resolved promise
            return promise.done(doDeleteColumns);
        };

        this.insertRow = function () {

            if (! this.isRowAddable()) {
                app.getView().rejectEditTextAttempt('tablesizerow'); // checking table size (26809)
                return;
            }

            undoManager.enterUndoGroup(function () {

                var // inserting only one row
                    count = 1,
                    // whether default cells shall be inserted into the new row
                    insertDefaultCells = false,
                    // the current logical position of the selection
                    position = selection.getEndPosition(),
                    // the row number from the logical position
                    referenceRow = null,
                    // the html dom node of the row
                    rowNode = null,
                    // the logical position of the row
                    rowPos = Position.getLastPositionFromPositionByNodeName(editdiv, position, 'tr'),
                    // operations generator
                    generator = this.createOperationsGenerator();

                if (rowPos !== null) {

                    rowNode = Position.getTableRowElement(editdiv, rowPos);

                    referenceRow = _.last(rowPos);

                    rowPos[rowPos.length - 1] += 1;

                    generator.generateOperation(Operations.ROWS_INSERT, {  start: rowPos, count: count, insertDefaultCells: insertDefaultCells, referenceRow: referenceRow });

                    // cells with list in the first paragraph, should automatically receive this list information, too
                    if (rowNode) {
                        // checking the cell contents in the cloned row
                        $(rowNode).children('td').each(function () {
                            var // the container for the content nodes
                                container = DOM.getCellContentNode(this),
                                // the first paragraph in the cell determines the attributes for the new cell
                                paragraph = container[0].firstChild,
                                // the paragraph attributes
                                paraAttributes = null,
                                // the logical position of the new paragraph
                                paraPos = _.clone(rowPos);

                            if (DOM.isParagraphNode(paragraph)) {
                                paraAttributes = StyleSheets.getExplicitAttributes(paragraph);

                                if (! _.isEmpty(paraAttributes)) {
                                    paraPos.push($(this).prevAll().length);
                                    paraPos.push(0);
                                    // generating an operation for setting paragraph attributes in the new row
                                    // -> no implicit paragraph in these cells
                                    generator.generateOperation(Operations.PARA_INSERT, {  start: paraPos, attrs: paraAttributes });
                                }
                            }
                        });
                    }

                    guiTriggeredOperation = true;
                    // apply all collected operations
                    this.applyOperations(generator.getOperations());
                    guiTriggeredOperation = false;
                }

                // setting the cursor position
                selection.setTextSelection(lastOperationEnd);
            }, this);

        };

        this.insertColumn = function () {

            if (! this.isColumnAddable()) {
                app.getView().rejectEditTextAttempt('tablesizecolumn'); // checking table size (26809)
                return;
            }

            undoManager.enterUndoGroup(function () {

                var position = selection.getEndPosition(),
                    cellPosition = Position.getColumnIndexInRow(editdiv, position),
                    tablePos = Position.getLastPositionFromPositionByNodeName(editdiv, position, DOM.TABLE_NODE_SELECTOR),
                    rowNode = Position.getLastNodeFromPositionByNodeName(editdiv, position, 'tr'),
                    insertMode = 'behind',
                    gridPosition = Table.getGridPositionFromCellPosition(rowNode, cellPosition).start,
                    tableGrid = Table.getTableGridWithNewColumn(editdiv, tablePos, gridPosition, insertMode),
                    // table node element
                    table = Position.getTableElement(editdiv, tablePos),
                    // all rows in the table
                    allRows = DOM.getTableRows(table),
                    // operations generator
                    generator = this.createOperationsGenerator();

                generator.generateOperation(Operations.COLUMN_INSERT, { start: tablePos, tableGrid: tableGrid, gridPosition: gridPosition, insertMode: insertMode });
                // Setting new table grid attribute to table
                generator.generateOperation(Operations.SET_ATTRIBUTES, { attrs: { table: { tableGrid: tableGrid } }, start: _.clone(tablePos) });

                // Iterating over all rows to keep paragraph attributes of first paragraph in each new cell
                // -> these paragraphs must not be explicit
                if (allRows) {
                    allRows.each(function (i, row) {
                        var // the current logical cell position
                            cellPosition = Table.getCellPositionFromGridPosition(row, gridPosition),
                            // the current cell node
                            cellClone = $(row).children(DOM.TABLE_CELLNODE_SELECTOR).slice(cellPosition, cellPosition + 1),
                            // the container for the content nodes
                            container = null,
                            // the first paragraph in the cell determines the attributes for the new cell
                            paragraph = null,
                            // whether the current or the following cell is used to find paragraph attributes
                            currentCellUsed = true,
                            // the paragraph attributes
                            paraAttributes = null,
                            // the position of the cell inside its row
                            cellNumber = null,
                            // the logical position of the row
                            rowPos = Position.getOxoPosition(editdiv, row, 0),
                            // the logical position of the new paragraph
                            paraPos = _.clone(rowPos);

                        // using the current cell only, if it is the last cell in the row. Otherwise the attributes from the
                        // following cell need to be examined.
                        if (cellClone.next().length > 0) {
                            cellClone = cellClone.next();
                            currentCellUsed = false;
                        }

                        container = DOM.getCellContentNode(cellClone);
                        paragraph = container[0].firstChild;

                        if (DOM.isParagraphNode(paragraph)) {
                            paraAttributes = StyleSheets.getExplicitAttributes(paragraph);

                            if (! _.isEmpty(paraAttributes)) {
                                cellNumber = $(cellClone).prevAll().length;
                                if (currentCellUsed) { cellNumber++; }
                                paraPos.push(cellNumber);
                                paraPos.push(0);
                                // generating an operation for setting paragraph attributes in the new row
                                // -> no implicit paragraph in these cells
                                generator.generateOperation(Operations.PARA_INSERT, {  start: paraPos, attrs: paraAttributes });
                            }
                        }
                    });
                }

                // no call of implTableChanged in implInsertColumn
                requiresElementFormattingUpdate = false;  // no call of implTableChanged -> attributes are already set in implSetAttributes
                guiTriggeredOperation = true;

                // apply all collected operations
                this.applyOperations(generator.getOperations());

                requiresElementFormattingUpdate = true;
                guiTriggeredOperation = false;

                // setting the cursor position
                selection.setTextSelection(lastOperationEnd);

            }, this);
        };

        this.insertParagraph = function (start) {
            this.applyOperations({ name: Operations.PARA_INSERT, start: _.clone(start) });
        };

        /**
         * Inserting a table into the document.
         * The undo manager returns the return value of the callback function.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled. If no dialog is shown, the deferred is
         *  resolved immediately.
         *  The undo manager returns the return value of the callback function.
         */
        this.insertTable = function (size) {

            if (!_.isObject(size)) { return; }

            return undoManager.enterUndoGroup(function () {

                var // cursor position used to split the paragraph
                    startPosition = null,
                    // paragraph to be split for the new table, and its position
                    paragraph = null, position = null,
                    // text offset in paragraph, first and last text position in paragraph
                    offset = 0, startOffset = 0, endOffset = 0,
                    // table attributes
                    attributes = { table: { tableGrid: [], width: 'auto' } },
                    // default table style
                    tableStyleId = self.getDefaultUITableStylesheet(),
                    // a new implicit paragraph
                    newParagraph = null,
                    // operations generator
                    generator = this.createOperationsGenerator();

                function doInsertTable() {
                    startPosition = selection.getStartPosition();
                    position = startPosition.slice(0, -1);
                    paragraph = Position.getParagraphElement(editdiv, position);
                    if (!paragraph) { return; }

                    if (! DOM.isImplicitParagraphNode(paragraph)) {
                        // split paragraph, if the cursor is between two characters,
                        // or if the paragraph is the very last in the container node
                        offset = _.last(startPosition);
                        startOffset = Position.getFirstTextNodePositionInParagraph(paragraph);
                        endOffset = Position.getLastTextNodePositionInParagraph(paragraph);

                        if ((!paragraph.nextSibling) && (offset === endOffset)) {
                            // create a new empty implicit paragraph behind the table
                            newParagraph = DOM.createImplicitParagraphNode();
                            validateParagraphNode(newParagraph);
                            $(paragraph).after(newParagraph);
                            implParagraphChanged(newParagraph);
                            position = Position.increaseLastIndex(position);
                            // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
                            // newly, also in other browsers (new jQuery version?)
                            repairEmptyTextNodes(newParagraph);
                        } else if ((startOffset < offset) && (offset < endOffset)) {
                            self.splitParagraph(startPosition);
                            position = Position.increaseLastIndex(position);
                        } else if ((!paragraph.nextSibling) && (offset > 0)) {
                            self.splitParagraph(startPosition);
                            position = Position.increaseLastIndex(position);
                        } else if (offset === endOffset) {
                            // cursor at the end of the paragraph: insert before next content node
                            position = Position.increaseLastIndex(position);
                        }
                    }

                    // prepare table column widths (values are relative to each other)
                    _(size.width).times(function () { attributes.table.tableGrid.push(1000); });

                    // set default table style
                    if (_.isString(tableStyleId)) {

                        // insert a pending table style if needed
                        Table.checkForLateralTableStyle(generator, self, tableStyleId);

                        // add table style name to attributes
                        attributes.styleId = tableStyleId;

                        // default: tables do not have last row, last column and vertical bands
                        attributes.table.exclude = ['lastRow', 'lastCol', 'bandsVert'];
                    }

                    // insert the table, and add empty rows
                    generator.generateOperation(Operations.TABLE_INSERT, { start: _.clone(position), attrs: attributes });

                    generator.generateOperation(Operations.ROWS_INSERT, { start: Position.appendNewIndex(position, 0), count: size.height, insertDefaultCells: true });

                    // apply all collected operations
                    self.applyOperations(generator.getOperations());

                    // set the cursor to first paragraph in first table cell
                    selection.setTextSelection(position.concat([0, 0, 0, 0]));
                }

                if (selection.hasRange()) {
                    return this.deleteSelected()
                    .done(function () {
                        doInsertTable();
                    });
                }

                doInsertTable();
                return $.when();

            }, this); // enterUndoGroup()

        };

        this.showInsertImageDialog = function () {
            var def = Image.showInsertImageDialog(app);
            def = def.then(function (imageFragment) {
                return self.insertImageURL(imageFragment);
            });

            return def;
        };

        this.insertImageURL = function (imageHolder) {

            var params = {};
            if (imageHolder.url) {
                params.imageUrl = imageHolder.url;
                params.name = imageHolder.name;
            } else if (imageHolder.substring(0, 10) === 'data:image') {
                params.imageData = imageHolder;
            } else {
                params.imageUrl = imageHolder;
            }

            return getImageSize(params.imageUrl || params.imageData).then(function (size) {

                var def = $.Deferred(),
                    result = false;

                // exit silently if we lost the edit rights
                if (!self.getEditMode()) {
                    return def.resolve();
                }


                undoManager.enterUndoGroup(function () {

                    var start = selection.getStartPosition();

                    handleImplicitParagraph(start);
                    result = self.applyOperations({
                        name: Operations.DRAWING_INSERT,
                        start: start,
                        type: 'image',
                        attrs: { drawing: _.extend(params, size, DEFAULT_DRAWING_MARGINS) }
                    });
                });

                // setting the cursor position
                selection.setTextSelection(lastOperationEnd);

                return result ? def.resolve() : def.reject();
            })
            .fail(function () {
                app.rejectEditAttempt('image');
            });
        };

        this.insertHyperlinkDirect = function (url, text) {

            var generator = this.createOperationsGenerator(),
                start = selection.getStartPosition(),
                end = selection.getEndPosition(),
                hyperlinkStyleId = self.getDefaultUIHyperlinkStylesheet();

            if (url && url.length > 0) {

                undoManager.enterUndoGroup(function () {

                    var newText = text || url;

                    // delete selected range (this creates new operations)
                    if (selection.hasRange()) {
                        self.deleteSelected();
                    }

                   // insert new text
                    handleImplicitParagraph(start);
                    generator.generateOperation(Operations.TEXT_INSERT, { text: newText, start: _.clone(start) });

                    // Calculate end position of new text
                    // will be used for setAttributes operation
                    end = Position.increaseLastIndex(start, newText.length);

                    if (characterStyles.isDirty(hyperlinkStyleId)) {
                        // insert hyperlink style to document
                        generator.generateOperation(Operations.INSERT_STYLESHEET, {
                            attrs: characterStyles.getStyleSheetAttributeMap(hyperlinkStyleId),
                            type: 'character',
                            styleId: hyperlinkStyleId,
                            styleName: characterStyles.getName(hyperlinkStyleId),
                            parent: characterStyles.getParentId(hyperlinkStyleId),
                            uiPriority: characterStyles.getUIPriority(hyperlinkStyleId)
                        });
                        characterStyles.setDirty(hyperlinkStyleId, false);
                    }

                    generator.generateOperation(Operations.SET_ATTRIBUTES, {
                        attrs: { styleId: hyperlinkStyleId, character: { url: url } },
                        start: _.clone(start),
                        end: _.clone(end)
                    });

                    // apply all collected operations
                    self.applyOperations(generator.getOperations());
                });
            }
        };

        /**
         * Shows the hyperlink dialog for the current selection, and returns a
         * Deferred object that allows to wait for the dialog.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled.
         */
        this.insertHyperlinkDialog = function () {

            var generator = this.createOperationsGenerator(),
                text = '', url = '',
                startPos = null,
                start = selection.getStartPosition(),
                end = selection.getEndPosition();

            if (!selection.hasRange()) {
                var newSelection = Hyperlink.findSelectionRange(this, selection);
                if (newSelection.start !== null && newSelection.end !== null) {
                    startPos = selection.getStartPosition();
                    start[start.length - 1] = newSelection.start;
                    end[end.length - 1] = newSelection.end;
                    selection.setTextSelection(start, end);
                }
            }

            if (!this.hasEnclosingParagraph()) {
                return $.Deferred().reject();
            }

            // use range to retrieve text and possible url
            if (selection.hasRange()) {

                // Find out the text/url of the selected text to provide them to the
                // hyperlink dialog
                selection.iterateNodes(function (node, pos, start, length) {
                    if ((start >= 0) && (length >= 0) && DOM.isTextSpan(node)) {
                        var nodeText = $(node).text();
                        if (nodeText) {
                            text = text.concat(nodeText.slice(start, start + length));
                        }
                        if (url.length === 0) {
                            var charAttributes = characterStyles.getElementAttributes(node).character;
                            if (charAttributes.url && charAttributes.url.length > 0)
                                url = charAttributes.url;
                        }
                    }
                });
            }

            // show hyperlink dialog
            return Hyperlink.showDialog(text, url, app).done(function (data) {
                // set url to selected text
                var hyperlinkStyleId = self.getDefaultUIHyperlinkStylesheet(),
                    url = data.url;

                undoManager.enterUndoGroup(function () {

                    if (data.url === null && data.text === null) {
                        // remove hyperlink
                        // setAttribute uses a closed range therefore -1
                        end[end.length - 1] -= 1;
                        generator.generateOperation(Operations.SET_ATTRIBUTES, {
                            attrs: Hyperlink.CLEAR_ATTRIBUTES,
                            start: _.clone(start),
                            end: _.clone(end)
                        });
                    }
                    else {
                        // insert/change hyperlink
                        if (data.text !== text) {

                            // text has been changed
                            if (selection.hasRange()) {
                                self.deleteSelected();
                            }

                            // insert new text
                            handleImplicitParagraph(start);
                            generator.generateOperation(Operations.TEXT_INSERT, { text: data.text, start: _.clone(start) });

                            // Calculate end position of new text
                            // will be used for setAttributes operation
                            end = Position.increaseLastIndex(start, data.text.length);
                        }
                        else {
                            end[end.length - 1] -= 1;
                        }

                        if (characterStyles.isDirty(hyperlinkStyleId)) {
                            // insert hyperlink style to document
                            generator.generateOperation(Operations.INSERT_STYLESHEET, {
                                attrs: characterStyles.getStyleSheetAttributeMap(hyperlinkStyleId),
                                type: 'character',
                                styleId: hyperlinkStyleId,
                                styleName: characterStyles.getName(hyperlinkStyleId),
                                parent: characterStyles.getParentId(hyperlinkStyleId),
                                uiPriority: characterStyles.getUIPriority(hyperlinkStyleId)
                            });
                            characterStyles.setDirty(hyperlinkStyleId, false);
                        }

                        generator.generateOperation(Operations.SET_ATTRIBUTES, {
                            attrs: { styleId: hyperlinkStyleId, character: { url: url } },
                            start: _.clone(start),
                            end: _.clone(end)
                        });
                    }

                    // apply all collected operations
                    self.applyOperations(generator.getOperations());

                }, self);

            }).always(function () {
                if (startPos) {
                    selection.setTextSelection(startPos);
                }
            });
        };

        this.removeHyperlink = function () {

            var generator = this.createOperationsGenerator(),
                startPos = null,
                start = selection.getStartPosition(),
                end = selection.getEndPosition();

            if (!selection.hasRange()) {
                var newSelection = Hyperlink.findSelectionRange(this, selection);
                if (newSelection.start !== null && newSelection.end !== null) {
                    startPos = selection.getStartPosition();
                    start[start.length - 1] = newSelection.start;
                    end[end.length - 1] = newSelection.end;
                    selection.setTextSelection(start, end);
                }
            }

            if (selection.hasRange() && this.hasEnclosingParagraph()) {

                // remove hyperlink
                // setAttribute uses a closed range therefore -1
                end[end.length - 1] -= 1;
                generator.generateOperation(Operations.SET_ATTRIBUTES, {
                    attrs: Hyperlink.CLEAR_ATTRIBUTES,
                    start: _.clone(start),
                    end: _.clone(end)
                });

                // apply the operations (undo group is created automatically)
                this.applyOperations(generator.getOperations());

                app.executeDelayed(function () {
                    app.getView().grabFocus();
                    if (startPos) {
                        selection.setTextSelection(startPos);
                    }
                });
            }
        };

        /**
         * Inserting a tab stopp.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled. If no dialog is shown, the deferred is
         *  resolved immediately.
         *  The undo manager returns the return value of the callback function.
         */
        this.insertTab = function () {

            return undoManager.enterUndoGroup(function () {

                function doInsertTab() {
                    var start = selection.getStartPosition(),
                        operation = { name: Operations.TAB_INSERT, start: start };
                    handleImplicitParagraph(start);
                    if (_.isObject(preselectedAttributes)) { operation.attrs = preselectedAttributes; }
                    self.applyOperations(operation);
                }

                if (selection.hasRange()) {
                    return this.deleteSelected()
                    .done(function () {
                        doInsertTab();
                        selection.setTextSelection(lastOperationEnd);  // finally setting the cursor position
                    });
                }

                doInsertTab();
                selection.setTextSelection(lastOperationEnd);  // finally setting the cursor position
                return $.when();

            }, this);

        };

        /**
         * Inserting a hard line break, that leads to a new line inside a paragraph. This
         * function can be called from the side pane with a button or with 'Shift+Enter' from
         * within the document.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  dialog has been closed with the default action; or rejected, if the
         *  dialog has been canceled. If no dialog is shown, the deferred is
         *  resolved immediately.
         *  The undo manager returns the return value of the callback function.
         */
        this.insertHardBreak = function () {

            return undoManager.enterUndoGroup(function () {

                function doInsertHardBreak() {
                    var start = selection.getStartPosition();
                    handleImplicitParagraph(start);
                    self.applyOperations({ name: Operations.HARDBREAK_INSERT, start: start });
                }

                if (selection.hasRange()) {
                    return this.deleteSelected()
                    .done(function () {
                        doInsertHardBreak();
                        selection.setTextSelection(lastOperationEnd); // finally setting the cursor position
                    });
                }

                doInsertHardBreak();
                selection.setTextSelection(lastOperationEnd); // finally setting the cursor position
                return $.when();

            }, this);

        };

        this.splitParagraph = function (position) {
            this.applyOperations({ name: Operations.PARA_SPLIT, start: _.clone(position) });
        };

        this.mergeParagraph = function (position) {
            this.applyOperations({ name: Operations.PARA_MERGE, start: _.clone(position) });
        };

        this.insertText = function (text, position, attrs) {

            undoManager.enterUndoGroup(function () {

                var operation = { name: Operations.TEXT_INSERT, text: text, start: _.clone(position) };

                handleImplicitParagraph(position);
                if (_.isObject(attrs)) { operation.attrs = _.copy(attrs, true); }
                guiTriggeredOperation = true;  // Fix for 30587, remote client requires synchronous attribute setting
                this.applyOperations(operation);
                guiTriggeredOperation = false;

                // Bug 25077: Chrome cannot set curser after a trailing SPACE
                // character, need to validate the paragraph node immediately
                // (which converts trailing SPACE characters to NBSP)
                // -> only do this, if it is really necessary (performance)
                if (_.browser.WebKit && (text === ' ')) {
                    validateParagraphNode(Position.getParagraphElement(editdiv, position.slice(0, -1)));
                }

            }, this);
        };

        /**
         * creates a default list either with decimal numbers or bullets
         * @param type {String} 'numbering' or 'bullet'
         */
        this.createDefaultList = function (type) {

            var defListStyleId = listCollection.getDefaultNumId(type);

            undoManager.enterUndoGroup(function () {

                if (defListStyleId === undefined) {
                    var listOperation = listCollection.getDefaultListOperation(type);
                    this.applyOperations(listOperation);
                    defListStyleId = listOperation.listStyleId;
                }
                this.setAttributes('paragraph', { styleId: this.getDefaultUIParagraphListStylesheet(), paragraph: { listStyleId: defListStyleId, listLevel: 0 } });
            }, this);
        };

        this.createSelectedListStyle = function (listStyleId, listLevel) {

            undoManager.enterUndoGroup(function () {
                var listOperation,
                    listOperationAndStyle,
                    savedSelection = _.copy(selection.getStartPosition()),
                    _start = selection.getStartPosition(),
                    currentParaPosition,
                    para,
                    paraAttributes,
                    prevParagraph, prevAttributes,
                    nextParagraph, nextAttributes,
                    defaultListStyle = this.getDefaultUIParagraphListStylesheet();

                _start.pop();
                para = Position.getParagraphElement(editdiv, _start);
                paraAttributes = paragraphStyles.getElementAttributes(para).paragraph;


                if (! selection.hasRange() && paraAttributes.listStyleId !== '') {
                    //merge level from new listlevel with used list level and apply that new list to all consecutive paragraphs using the old list
                    if (listStyleId !== paraAttributes.listStyleId) {

                        listOperationAndStyle = listCollection.mergeListStyle(paraAttributes.listStyleId, listStyleId, paraAttributes.listLevel);
                        if (listOperationAndStyle.listOperation) {
                            this.applyOperations(listOperationAndStyle.listOperation);
                        }
                        prevParagraph = Utils.findPreviousNode(editdiv, para, DOM.PARAGRAPH_NODE_SELECTOR);
                        while (prevParagraph) {
                            prevAttributes = paragraphStyles.getElementAttributes(prevParagraph).paragraph;
                            if (prevAttributes.listStyleId === paraAttributes.listStyleId) {
                                currentParaPosition = Position.getOxoPosition(editdiv, prevParagraph, 0);
                                currentParaPosition.push(0);
                                selection.setTextSelection(currentParaPosition);
                                this.setAttributes('paragraph', { styleId: defaultListStyle, paragraph: { listStyleId: listOperationAndStyle.listStyleId } });
                                prevParagraph = Utils.findPreviousNode(editdiv, prevParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                            } else {
                                prevParagraph = null;
                            }
                        }

                        nextParagraph = Utils.findNextNode(editdiv, para, DOM.PARAGRAPH_NODE_SELECTOR);
                        while (nextParagraph) {
                            nextAttributes = paragraphStyles.getElementAttributes(nextParagraph).paragraph;
                            if (nextAttributes.listStyleId === paraAttributes.listStyleId) {
                                currentParaPosition = Position.getOxoPosition(editdiv, nextParagraph, 0);
                                currentParaPosition.push(0);
                                selection.setTextSelection(currentParaPosition);
                                this.setAttributes('paragraph', { styleId: defaultListStyle, paragraph: { listStyleId: listOperationAndStyle.listStyleId } });
                                nextParagraph = Utils.findNextNode(editdiv, nextParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                            } else {
                                nextParagraph = null;
                            }
                        }

                        selection.setTextSelection(savedSelection);
                        this.setAttributes('paragraph', { styleId: defaultListStyle, paragraph: { listStyleId: listOperationAndStyle.listStyleId } });
                    }

                } else {
                    listOperation = listCollection.getSelectedListStyleOperation(listStyleId);
                    if (listOperation) {
                        this.applyOperations(listOperation);
                    }
                    listLevel = Math.max(0, listLevel || 0);  // setting listLevel for paragraphs, that were not in lists before

                    this.setAttributes('paragraph', { styleId: defaultListStyle, paragraph: { listStyleId: listStyleId, listLevel: listLevel } });
                }

            }, this); // enterUndoGroup
        };

        this.createList = function (type, options) {

            var defListStyleId = (!options || (!options.symbol && !options.listStartValue)) ? listCollection.getDefaultNumId(type) : undefined;

            if (defListStyleId === undefined) {
                var listOperation = listCollection.getDefaultListOperation(type, options);
                this.applyOperations(listOperation);
                defListStyleId = listOperation.listStyleId;
            }
            if (options && options.startPosition) {
                var start = _.clone(options.startPosition),
                    listParaStyleId = this.getDefaultUIParagraphListStylesheet(),
                    insertStyleOperation = null;

                start.pop();
                // register pending style sheet via 'insertStyleSheet' operation
                if (_.isString(listParaStyleId) && paragraphStyles.isDirty(listParaStyleId)) {

                    var styleSheetAttributes = {
                        attrs: paragraphStyles.getStyleSheetAttributeMap(listParaStyleId),
                        type: 'paragraph',
                        styleId: listParaStyleId,
                        styleName: paragraphStyles.getName(listParaStyleId),
                        parent: paragraphStyles.getParentId(listParaStyleId),
                        uiPriority: paragraphStyles.getUIPriority(listParaStyleId)
                    };

                    // parent is an optional value, should not be send as 'null'
                    if (styleSheetAttributes.parent === null) { delete styleSheetAttributes.parent; }

                    insertStyleOperation = _.extend({ name: Operations.INSERT_STYLESHEET }, styleSheetAttributes);
                    this.applyOperations(insertStyleOperation);

                    // remove the dirty flag
                    paragraphStyles.setDirty(listParaStyleId, false);
                }

                var newOperation = {
                        name: Operations.SET_ATTRIBUTES,
                        attrs: { styleId: listParaStyleId, paragraph: { listStyleId: defListStyleId, listLevel: 0 } },
                        start: start
                    };
                this.applyOperations(newOperation);
                newOperation = _.copy(newOperation);
                newOperation.start = Position.increaseLastIndex(newOperation.start);
                this.applyOperations(newOperation);
            } else {
                this.setAttributes('paragraph', { styleId: this.getDefaultUIParagraphListStylesheet(), paragraph: { listStyleId: defListStyleId, listLevel: 0 } });
            }

        };

        /**
         * Removes bullet/numbered list formatting from the selected
         * paragraphs.
         */
        this.removeListAttributes = function () {

            var paragraph = Position.getLastNodeFromPositionByNodeName(editdiv, selection.getStartPosition(), DOM.PARAGRAPH_NODE_SELECTOR),
                prevPara = Utils.findPreviousNode(editdiv, paragraph, DOM.PARAGRAPH_NODE_SELECTOR);

            this.setAttributes('paragraph', { styleId: this.getDefaultUIParagraphStylesheet(), paragraph: { listStyleId: null, listLevel: -1 } });
            if (prevPara) {
                paragraphStyles.updateElementFormatting(prevPara);
            }
        };

        // style sheets and formatting attributes -----------------------------

        /**
         * Returns the collection of all list definitions.
         */
        this.getListCollection = function () {
            return listCollection;
        };

        /**
         * Adds the passed attributes to the set of preselected attributes that
         * will be applied to the next inserted text contents without moving
         * the text cursor or executing any other action.
         *
         * @param {Object} attributes
         *  A character attribute set that will be added to the set of
         *  preselected attributes.
         *
         * @returns {Editor}
         *  A reference to this instance.
         */
        this.addPreselectedAttributes = function (attributes) {
            preselectedAttributes = preselectedAttributes || {};
            documentStyles.extendAttributes(preselectedAttributes, attributes);
            return this;
        };

        /**
         * Returns the values of all formatting attributes of the elements in
         * the current selection associated to the specified attribute family.
         *
         * @param {String} family
         *  The name of the attribute family used to select specific elements
         *  in the current selection:
         *  - 'character': all text spans (text portions, text components),
         *  - 'paragraph': all paragraph nodes,
         *  - 'table': all table nodes,
         *  - 'drawing': all drawing object nodes.
         *
         * @returns {Object}
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute family.
         */
        this.getAttributes = function (family) {

            var // whether the selection is a simple cursor
                isCursor = selection.isTextCursor(),
                // table or drawing element contained by the selection
                element = null,
                // resulting merged attributes
                mergedAttributes = null;

            // merges the passed element attributes into the resulting attributes
            function mergeElementAttributes(elementAttributes) {

                var // whether any attribute is still unambiguous
                    hasNonNull = false;

                // merges the passed attribute value into the attributes map
                function mergeAttribute(attributes, name, value) {
                    if (!(name in attributes)) {
                        // initial iteration: store value
                        attributes[name] = value;
                    } else if (!_.isEqual(value, attributes[name])) {
                        // value differs from previous value: ambiguous state
                        attributes[name] = null;
                    }
                    hasNonNull = hasNonNull || !_.isNull(attributes[name]);
                }

                // initial iteration: store attributes and return
                if (!mergedAttributes) {
                    mergedAttributes = elementAttributes;
                    return;
                }

                // process all passed element attributes
                _(elementAttributes).each(function (attributeValues, family) {
                    if (family === 'styleId') {
                        mergeAttribute(mergedAttributes, 'styleId', attributeValues);
                    } else {
                        var mergedAttributeValues = mergedAttributes[family];
                        _(attributeValues).each(function (value, name) {
                            mergeAttribute(mergedAttributeValues, name, value);
                        });
                    }
                });

                // stop iteration, if all attributes are ambiguous
                return hasNonNull ? undefined : Utils.BREAK;
            }

            switch (family) {

            case 'character':
                selection.iterateNodes(function (node) {
                    return DOM.iterateTextSpans(node, function (span) {
                        // ignore empty text spans (they cannot be formatted via operations),
                        // but get formatting of an empty span selected by a text cursor
                        if (isCursor || (span.firstChild.nodeValue.length > 0)) {
                            return mergeElementAttributes(characterStyles.getElementAttributes(span));
                        }
                    });
                });
                if (isCursor && preselectedAttributes) {
                    // add preselected attributes (text cursor selection cannot result in ambiguous attributes)
                    documentStyles.extendAttributes(mergedAttributes, preselectedAttributes);
                }
                break;

            case 'paragraph':
                selection.iterateContentNodes(function (paragraph) {
                    return mergeElementAttributes(paragraphStyles.getElementAttributes(paragraph));
                });
                break;

            case 'cell':
                selection.iterateTableCells(function (cell) {
                    return mergeElementAttributes(tableCellStyles.getElementAttributes(cell));
                });
                break;

            case 'table':
                if ((element = selection.getEnclosingTable())) {
                    mergeElementAttributes(tableStyles.getElementAttributes(element));
                }
                break;

            case 'drawing':
                // TODO: needs change when multiple drawings can be selected
                if ((element = selection.getSelectedDrawing()[0]) && DrawingFrame.isDrawingFrame(element)) {
                    mergeElementAttributes(drawingStyles.getElementAttributes(element));
                }
                break;

            default:
                Utils.error('Editor.getAttributes(): missing implementation for family "' + family + '"');
            }

            return mergedAttributes || {};
        };

        /**
         * Changes a single attribute of the specified attribute family in the
         * current selection.
         *
         * @param {String} family
         *  The name of the attribute family containing the specified
         *  attribute.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the operation. The
         *  following options are supported:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection, before applying the new attributes.
         */
        this.setAttribute = function (family, name, value, options) {
            this.setAttributes(family, Utils.makeSimpleObject(family, Utils.makeSimpleObject(name, value)), options);
        };

        /**
         * Changes multiple attributes of the specified attribute family in the
         * current selection.
         *
         * @param {String} family
         *  The name of the attribute family containing the passed attributes.
         *
         * @param {Object} attributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute families.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the opration. The
         *  following options are supported:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection while applying the new attributes.
         */
        this.setAttributes = function (family, attributes, options) {

            // Create an undo group that collects all undo operations generated
            // in the local setAttributes() method (it calls itself recursively
            // with smaller parts of the current selection).
            undoManager.enterUndoGroup(function () {

                var // table or drawing element contained by the selection
                    element = null,
                    // operations generator
                    generator = this.createOperationsGenerator(),
                    // the style sheet container
                    styleSheets = this.getStyleSheets(family),
                    // properties for the insertStyleSheet operation
                    newStyleSheetProps = null,
                    // logical position
                    localPosition = null,
                    // another logical position
                    localDestPosition = null,
                    // the length of the paragraph
                    paragraphLength = null,
                    // whether after assigning cell attributes, it is still necessary to assign table attributes
                    createTableOperation = true;

                // generates a 'setAttributes' operation with the correct attributes
                function generateSetAttributeOperation(startPosition, endPosition) {

                    var // the options for the operation
                        operationOptions = { start: startPosition, attrs: _.clone(attributes) };

                    // add end position if specified
                    if (_.isArray(endPosition)) {
                        operationOptions.end = endPosition;
                    }

                    // generate the 'setAttributes' operation
                    generator.generateOperation(Operations.SET_ATTRIBUTES, operationOptions);
                }

                // add all attributes to be cleared
                if (Utils.getBooleanOption(options, 'clear', false)) {
                    if (family !== 'table') {  // special behaviour for tables follows below
                        attributes = documentStyles.extendAttributes(styleSheets.buildNullAttributes(), attributes);
                    }
                }

                // nothig to do if no attributes will be changed
                if (_.isEmpty(attributes)) { return; }

                // register pending style sheet via 'insertStyleSheet' operation
                if (_.isString(attributes.styleId) && styleSheets.isDirty(attributes.styleId)) {

                    newStyleSheetProps = {
                        attrs: styleSheets.getStyleSheetAttributeMap(attributes.styleId),
                        type: family,
                        styleId: attributes.styleId,
                        styleName: styleSheets.getName(attributes.styleId),
                        parent: styleSheets.getParentId(attributes.styleId),
                        uiPriority: styleSheets.getUIPriority(attributes.styleId)
                    };

                    // parent is an optional value, should not be send as 'null'
                    if (newStyleSheetProps.parent === null) { delete newStyleSheetProps.parent; }

                    generator.generateOperation(Operations.INSERT_STYLESHEET, newStyleSheetProps);

                    // remove the dirty flag
                    styleSheets.setDirty(attributes.styleId, false);
                }

                // generate 'setAttribute' operations
                switch (family) {

                case 'character':
                    if (selection.hasRange()) {
                        selection.iterateContentNodes(function (paragraph, position, startOffset, endOffset) {
                            // validate start offset (iterator passes 'undefined' for fully covered paragraphs)
                            if (!_.isNumber(startOffset)) {
                                startOffset = 0;
                            }
                            // validate end offset (iterator passes 'undefined' for fully covered paragraphs)
                            if (!_.isNumber(endOffset)) {
                                endOffset = Position.getParagraphLength(editdiv, position) - 1;
                            }
                            // set the attributes at the covered text range
                            // TODO: currently, no way to set character attributes at empty paragraphs via operation...
                            if (startOffset <= endOffset) {
                                generateSetAttributeOperation(position.concat([startOffset]), position.concat([endOffset]));
                            }
                            //reset spelling status on language changes
                            if (attributes.character && attributes.character.language) {
                                $(paragraph).removeClass('p_spelled');
                            }
                        });
                    } else {
                        if (Position.getParagraphLength(editdiv, _.clone(selection.getStartPosition())) === 0) {
                            // Task 28187: Setting character attributes at empty paragraphs
                            // In implSetAttributes it is additionally necessary to remove the character attributes
                            // from text span so that character styles at paragraph become visible
                            handleImplicitParagraph(_.clone(selection.getStartPosition()));
                            generateSetAttributeOperation(_.initial(_.clone(selection.getStartPosition())));
                        } else {
                            // using preselected attributes for non-empty paragraphs
                            self.addPreselectedAttributes(attributes);
                        }
                    }
                    break;

                case 'paragraph':
                        // #26454# styles should not touch list attributes if they don't set it itself
                    if (('styleId' in attributes) && _.isObject(attributes.paragraph) && _.isNull(attributes.paragraph.listStyleId) && _.isNull(attributes.paragraph.listLevel)) {
                        var styleAttributes = styleSheets.getStyleSheetAttributes(attributes.styleId);
                        if (styleAttributes.paragraph.listStyleId === '') {
                            delete attributes.paragraph.listStyleId;
                            delete attributes.paragraph.listLevel;
                        }
                    }

                    selection.iterateContentNodes(function (paragraph, position) {
                        handleImplicitParagraph(Position.appendNewIndex(position));
                        generateSetAttributeOperation(position);
                    });
                    break;

                case 'cell':
                    selection.iterateTableCells(function (cell, position) {
                        generateSetAttributeOperation(position);
                    });
                    break;

                case 'table':
                    if ((element = selection.getEnclosingTable())) {

                        localPosition = Position.getOxoPosition(editdiv, element, 0);  // the logical position of the table

                        if (Utils.getBooleanOption(options, 'clear', false)) {
                            // removing hard attributes at tables and cells, so that the table style will be visible
                            Table.removeTableAttributes(element, localPosition, documentStyles, editdiv, generator);
                        }

                        if (Utils.getBooleanOption(options, 'onlyVisibleBorders', false)) {
                            // setting border width directly at cells -> overwriting values of table style
                            Table.setBorderWidthToVisibleCells(element, attributes, tableCellStyles, editdiv, generator);
                            createTableOperation = false;
                        }

                        if (Utils.getBooleanOption(options, 'cellSpecificTableAttribute', false)) {
                            // setting border mode directly at cells -> overwriting values of table style
                            Table.setBorderModeToCells(element, attributes, TableStyles.getBorderStyleFromAttributes(this.getAttributes('cell').cell || {}), editdiv, generator);
                        }

                        // setting attributes to the table element
                        // -> also setting table attributes, if attributes are already assigned to table cells, to
                        // keep the getter functions simple and performant
                        if (createTableOperation) {
                            generateSetAttributeOperation(localPosition);
                        }
                    }
                    break;

                case 'drawing':
                    // TODO: needs change when multiple drawings can be selected
                    // TODO: this fails if a drawing style sheet changes the inline/floating mode instead of explicit attributes
                    if ((element = selection.getSelectedDrawing()[0]) && DrawingFrame.isDrawingFrame(element) && _.isObject(attributes.drawing)) {

                        localPosition = Position.getOxoPosition(editdiv, element, 0);

                        // when switching from inline to floated, saving current position in the drawing, so that it can
                        // be set correctly when switching back to inline.
                        // This is only necessary, if the drawing was moved in that way, that implMove needed to be called.
                        if ((attributes.drawing.inline === false) && (DOM.isInlineDrawingNode(element))) {
                            $(element).data('inlinePosition', localPosition[localPosition.length - 1]);
                            // Fixing vertical offset, if drawing is set from inline to floated (31722)
                            if ((attributes.drawing.anchorVertBase) && (attributes.drawing.anchorVertBase === 'paragraph') &&
                                (_.isNumber(attributes.drawing.anchorVertOffset)) && (attributes.drawing.anchorVertOffset === 0) &&
                                (($(element).offset().top - $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR).offset().top > 0))) {
                                attributes.drawing.anchorVertOffset = Utils.convertLengthToHmm($(element).offset().top - $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR).offset().top, 'px');
                            }
                        }

                        // when switching from floated to inline, a move of the drawing might be necessary
                        if ((attributes.drawing.inline === true) && (DOM.isFloatingDrawingNode(element)) && ($(element).data('inlinePosition'))) {

                            localDestPosition = _.clone(localPosition);
                            paragraphLength = Position.getParagraphLength(editdiv, localDestPosition);
                            if ((paragraphLength - 1) < $(element).data('inlinePosition')) {  // -> is this position still valid?
                                localDestPosition[localDestPosition.length - 1] = paragraphLength - 1;
                            } else {
                                localDestPosition[localDestPosition.length - 1] = $(element).data('inlinePosition');
                            }
                            if (! _.isEqual(localPosition, localDestPosition)) {
                                undoManager.enterUndoGroup(function () {
                                    handleImplicitParagraph(localDestPosition);
                                    this.applyOperations({ name: Operations.MOVE, start: localPosition, end: localPosition, to: localDestPosition });
                                    localPosition = _.clone(localDestPosition);
                                }, this);
                            }
                        }

                        generateSetAttributeOperation(localPosition);

                        $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated

                    }
                    break;

                default:
                    Utils.error('Editor.setAttributes(): missing implementation for family "' + family + '"');
                }

                // apply all collected operations
                this.applyOperations(generator.getOperations());

            }, this); // enterUndoGroup
        };

        /**
         * Removes all hard-set attributes depending on the current selection.
         * A selection range clears only hard-set character attributes and without
         * a range the hard-set paragraph attributes are cleared.
         *
         */
        this.resetAttributes = function () {

            function clearParagraphAttributes() {
                var paraAttributes = paragraphStyles.buildNullAttributes();
                delete paraAttributes.styleId;
                self.setAttributes('paragraph', paraAttributes);
            }

            function clearCharacterAttributes() {
                var charAttributes = characterStyles.buildNullAttributes();
                // don't reset hyperlink attribute
                delete charAttributes.styleId;
                delete charAttributes.character.url;
                self.setAttributes('character', charAttributes);
            }

            if (!selection.hasRange()) {
                undoManager.enterUndoGroup(function () {

                    var start = selection.getStartPosition(),
                        end = selection.getEndPosition(),
                        firstParagraph = Position.getLastNodeFromPositionByNodeName(editdiv, start, DOM.PARAGRAPH_NODE_SELECTOR),
                        lastParagraph = Position.getLastNodeFromPositionByNodeName(editdiv, end, DOM.PARAGRAPH_NODE_SELECTOR),
                        prevPara,
                        nextPara;

                    clearParagraphAttributes();
                    prevPara = Utils.findPreviousNode(editdiv, firstParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                    if (prevPara)
                        paragraphStyles.updateElementFormatting(prevPara);
                    nextPara = Utils.findNextNode(editdiv, lastParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                    if (nextPara)
                        paragraphStyles.updateElementFormatting(nextPara);
                    self.addPreselectedAttributes(characterStyles.buildNullAttributes());
                }, this);
            }
            else {
                clearCharacterAttributes();
            }
        };

        /**
         * Returns the current document styles.
         */
        this.getDocumentStyles = function () {
            return documentStyles;
        };

        /**
         * Returns the current selection.
         */
        this.getSelection = function () {
            return selection;
        };

        /**
         * Returns whether the current selection selects any text. This
         * includes the rectangular table cell selection mode.
         */
        this.isTextSelected = function () {
            return selection.getSelectionType() !== 'drawing';
        };

        /**
         * Returns whether the editor contains a selection range (text,
         * drawings, or table cells) instead of a simple text cursor.
         */
        this.hasSelectedRange = function () {
            return !selection.isTextCursor();
        };

        /**
         * Returns whether the editor contains a selection within a
         * single paragraph or not.
         */
        this.hasEnclosingParagraph = function () {
            return selection.getEnclosingParagraph() !== null;
        };

        // PUBLIC TABLE METHODS

        /**
         * Returns whether the editor contains a selection within a
         * table or not.
         */
        this.isPositionInTable = function () {
            return !_.isNull(selection.getEnclosingTable());
        };

        /**
         * Returns the number of columns, if the editor contains a selection within a
         * table.
         */
        this.getNumberOfColumns = function () {
            var table = selection.getEnclosingTable();
            return table && Table.getColumnCount(table);
        };

        /**
         * Returns the number of rows, if the editor contains a selection within a
         * table.
         */
        this.getNumberOfRows = function () {
            var table = selection.getEnclosingTable();
            return table && Table.getRowCount(table);
        };

        /**
         * Returns whether a further row can be added, if the editor contains a
         * selection within a table. The maximum number of rows and cells can
         * be defined in the configuration.
         */
        this.isRowAddable = function () {

            var rowCount = this.getNumberOfRows(),
                colCount = this.getNumberOfColumns(),
                maxTableRows = TextConfig.getMaxTableRows(),
                maxTableCells = TextConfig.getMaxTableCells();

            if (_.isNumber(rowCount) && _.isNumber(colCount)) {

                if ((_.isNumber(maxTableRows)) && (rowCount >= maxTableRows)) {
                    return false;
                }

                if ((_.isNumber(maxTableCells)) && ((rowCount + 1) * colCount > maxTableCells)) {
                    return false;
                }

                return true;
            }

            return false;
        };

        /**
         * Returns whether a further column can be added, if the editor contains a selection within a
         * table. The maximum number of columns and cells can be defined in the configuration.
         */
        this.isColumnAddable = function () {

            var rowCount = this.getNumberOfRows(),
                colCount = this.getNumberOfColumns(),
                maxTableColumns = TextConfig.getMaxTableColumns(),
                maxTableCells = TextConfig.getMaxTableCells();

            if (_.isNumber(rowCount) && _.isNumber(colCount)) {

                if ((_.isNumber(maxTableColumns)) && (colCount >= maxTableColumns)) {
                    return false;
                }

                if ((_.isNumber(maxTableCells)) && ((colCount + 1) * rowCount > maxTableCells)) {
                    return false;
                }

                return true;
            }

            return false;
        };

        // PUBLIC DRAWING METHODS

        /**
         * Returns whether the current selection selects one or more drawings.
         */
        this.isDrawingSelected = function () {
            return selection.getSelectionType() === 'drawing';
        };

        /**
         * Returns whether the current selection selects text, not cells and
         * not drawings.
         */
        this.isTextOnlySelected = function () {
            return selection.getSelectionType() === 'text';
        };

        /**
         * Returns the default lateral heading character styles
         */
        this.getDefaultHeadingCharacterStyles = function () {
            return HEADINGS_CHARATTRIBUTES;
        };

        /**
         * Returns the default lateral paragraph style
         */
        this.getDefaultParagraphStyleDefinition = function () {
            return DEFAULT_PARAGRAPH_DEFINTIONS;
        };

        /**
         * Returns the default lateral table definiton
         */
        this.getDefaultLateralTableDefinition = function () {
            return DEFAULT_LATERAL_TABLE_DEFINITIONS;
        };

        /**
         * Returns the default lateral table attributes
         */
        this.getDefaultLateralTableAttributes = function () {
            return DEFAULT_LATERAL_TABLE_ATTRIBUTES;
        };

        this.getDefaultLateralHyperlinkDefinition = function () {
            return DEFAULT_HYPERLINK_DEFINTIONS;
        };

        this.getDefaultLateralHyperlinkAttributes = function () {
            return DEFAULT_HYPERLINK_CHARATTRIBUTES;
        };

        /**
         * Returns the document default paragraph stylesheet id
         */
        this.getDefaultUIParagraphStylesheet = function () {
            var styleNames = [],
                styleId = null;

            if (app.isODF()) {
                styleNames = paragraphStyles.getStyleSheetNames();
                _(styleNames).each(function (name, id) {
                    var lowerId = id.toLowerCase();
                    if (lowerId === 'standard' || lowerId === 'normal') {
                        styleId = id;
                    }
                });

                if (!styleId) {
                    styleId = paragraphStyles.getDefaultStyleId();
                }
            } else {
                styleId = paragraphStyles.getDefaultStyleId();
            }

            return styleId;
        };

        /**
         * Returns the document default table stylesheet id
         * which can be used to set attributes for a new
         * table.
         */
        this.getDefaultUITableStylesheet = function () {
            var styleNames = tableStyles.getStyleSheetNames(),
                highestUIPriority = 99,
                tableStyleId = null;

            _(styleNames).each(function (name, id) {
                var uiPriority = tableStyles.getUIPriority(id);

                if (uiPriority && (uiPriority < highestUIPriority)) {
                    tableStyleId = id;
                    highestUIPriority = uiPriority;
                }
            });

            return tableStyleId;
        };

        /**
         * Returns the document default hyperlink stylesheet id
         */
        this.getDefaultUIHyperlinkStylesheet = function () {
            var styleNames = characterStyles.getStyleSheetNames(),
                hyperlinkId = null;

            _(styleNames).find(function (name, id) {
                var lowerName = name.toLowerCase();
                if (lowerName.indexOf('hyperlink') >= 0) {
                    hyperlinkId = id;
                    return true;
                }
                return false;
            });

            return hyperlinkId;
        };

        /**
         * Returns the document default paragraph list stylesheet id
         */
        this.getDefaultUIParagraphListStylesheet = function () {
            var styleNames = paragraphStyles.getStyleSheetNames(),
                 paragraphListId = null;

            _(styleNames).find(function (name, id) {
                var lowerName = name.toLowerCase();
                if (lowerName.indexOf('list paragraph') === 0) {
                    paragraphListId = id;
                    return true;
                }
            });

            return paragraphListId;
        };

        /**
         * Optimizing the actions before they are sent to the server. Especially
         * TEXT_INSERT operations can be merged. This function is a callback
         * that is started from 'sendActions' in the application.
         *
         * @param {Array} actions
         *  An array containing all actions that are investigated for optimization.
         *
         * @returns {Array}
         *  An array with optimized actions.
         */
        this.optimizeActions = function (actions) {

            var newActions = [];

            _.each(actions, function (next) {

                var insertNextOp = true,
                    last = _.last(newActions);

                if ((last) && (last.operations) && (last.operations.length === 1) && (next) && (next.operations) && (next.operations.length === 1)) {

                    if ((last.operations[0].name === Operations.TEXT_INSERT) && (next.operations[0].name === Operations.TEXT_INSERT)) {

                        // trying to merge two following text insert operations
                        if (Position.hasSameParentComponent(last.operations[0].start, next.operations[0].start)) {

                            // check that the 'next' action adds the character directly after the 'last' action
                            if ((_.last(last.operations[0].start) + last.operations[0].text.length === _.last(next.operations[0].start)) && !('attrs' in next.operations[0])) {
                                // merge insert operation (add the characters)
                                last.operations[0].text += next.operations[0].text;
                                // increasing the operation length
                                if ((last.operations[0].opl) && (next.operations[0].opl)) {
                                    last.operations[0].opl += next.operations[0].opl;
                                }
                                insertNextOp = false;
                            } else if ((_.last(last.operations[0].start) <= _.last(next.operations[0].start)) && (_.last(next.operations[0].start) <= _.last(last.operations[0].start) + last.operations[0].text.length)) {
                                // the new text is inserted into an already existing text
                                last.operations[0].text = last.operations[0].text.slice(0, _.last(next.operations[0].start) - _.last(last.operations[0].start)) + next.operations[0].text + last.operations[0].text.slice(_.last(next.operations[0].start) - _.last(last.operations[0].start));
                                // increasing the operation length
                                if ((last.operations[0].opl) && (next.operations[0].opl)) {
                                    last.operations[0].opl += next.operations[0].opl;
                                }
                                insertNextOp = false;
                            }
                        }
                    }
                }

                if (insertNextOp) {
                    newActions.push(next);
                }

            });

            return newActions;
        };

        /**
         * Load performance: Setting the document operation state number of a document that is currently
         * loaded using the local storage.
         *
         * @param {Number} osn
         *  The document operation state number of the currently loaded document.
         */
        this.setLoadDocumentOsn = function (osn) {
            docLoadOSN = osn;
        };

        /**
         * Load performance: Filtering operations corresponding to the operation name.
         *
         * @param {Object} operation
         *  An operation object.
         *
         * @returns {Boolean}
         *  Whether the specified operation is listed in the array of operations, that are required, even
         *  if the document is loaded from the local storage. Furthermore all operations must be executed,
         *  if their operation state number is larger the the operation state number of the saved document.
         *  If 'true' is returned, the operation must be executed. If this is not necessary, 'false' is returned.
         */
        this.operationFilter = function (operation) {
            var forceOperationExecution = (operation && operation.osn && docLoadOSN && (operation.osn >= docLoadOSN));
            return (forceOperationExecution || (operation && operation.name && _.contains(REQUIRED_LOAD_OPERATIONS, operation.name)));
        };

        /**
         * Load performance: Execute post process activities after loading the document from the
         * local storage.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  formatting has been updated successfully, or rejected when an error
         *  has occurred.
         */
        this.updateDocumentFormattingStorage = function () {
            // initialize default page formatting
            pageStyles.updateElementFormatting(editdiv);

            // handling page breaks correctly
            pageAttributes = pageStyles.getElementAttributes(editdiv);
            pageMaxHeight = Utils.convertHmmToLength(pageAttributes.page.height - pageAttributes.page.marginTop - pageAttributes.page.marginBottom, 'px', 1);
            pagePaddingLeft = Utils.convertHmmToLength(pageAttributes.page.marginLeft, 'px', 1);
            pagePaddingTop = Utils.convertHmmToLength(pageAttributes.page.marginTop, 'px', 1);
            pagePaddingBottom = Utils.convertHmmToLength(pageAttributes.page.marginBottom, 'px', 1);
            pageWidth = Utils.convertHmmToLength(pageAttributes.page.width, 'px', 1);

            initialPageBreaks();

            return $.when();
        };

        /**
         * Updates the formatting of all elements, after the import operations
         * of the document have been applied.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  formatting has been updated successfully, or rejected when an error
         *  has occurred.
         */
        this.updateDocumentFormatting = function () {

            var // the result Deferred object
                def = $.Deferred(),
                // the content node of the page
                pageContentNode = DOM.getPageContentNode(editdiv),
                // all top-level paragraphs, all tables (also embedded), and all drawings
                formattingNodes = pageContentNode.find('> ' + DOM.PARAGRAPH_NODE_SELECTOR + ', ' + DrawingFrame.NODE_SELECTOR + ', ' + DOM.TABLE_NODE_SELECTOR),
                // number of progress ticks for tables (1 tick per paragraph/drawing)
                TABLE_PROGRESS_TICKS = 20,
                // total number of progress ticks
                totalProgressTicks = formattingNodes.length + (TABLE_PROGRESS_TICKS - 1) * formattingNodes.filter(DOM.TABLE_NODE_SELECTOR).length,
                // current progress for formatting process
                currentProgressTicks = 0,
                // update progress bar at most 250 times
                progressBlockSize = totalProgressTicks / 250,
                // index of progress block from last progress update
                lastProgressBlock = -1;

            pageAttributes = pageStyles.getElementAttributes(self.getNode());
            pageMaxHeight = Utils.convertHmmToLength(pageAttributes.page.height - pageAttributes.page.marginTop - pageAttributes.page.marginBottom, 'px', 1);
            pagePaddingLeft = Utils.convertHmmToLength(pageAttributes.page.marginLeft, 'px', 1);
            pagePaddingTop = Utils.convertHmmToLength(pageAttributes.page.marginTop, 'px', 1);
            pagePaddingBottom = Utils.convertHmmToLength(pageAttributes.page.marginBottom, 'px', 1);
            pageWidth = Utils.convertHmmToLength(pageAttributes.page.width, 'px', 1);


            // first 90% for formatting (last 10% for updating lists)
            function updateFormatProgress(progressTicks) {
                var progressBlock = Math.floor(progressTicks / progressBlockSize);
                if (lastProgressBlock < progressBlock) {
                    def.notify(0.9 * progressTicks / totalProgressTicks);
                    lastProgressBlock = progressBlock;
                }
            }

            // updates the formatting of the passed element, returns a promise
            function updateElementFormatting(element) {

                var // the Promise for asynchronous element formatting
                    promise = null,
                    // whether the element is a direct child of the page content node
                    topLevel = !element[0].parentNode;

                // insert the detached element into the page content node
                // (performance optimization for large documents, see below)
                if (topLevel) { pageContentNode.append(element); }

                // determine the type of the passed element
                if (DOM.isParagraphNode(element)) {
                    // validate DOM contents AND update formatting of the paragraph
                    validateParagraphNode(element);
                    paragraphStyles.updateElementFormatting(element);
                    updateFormatProgress(currentProgressTicks += 1);
                }
                else if (DrawingFrame.isDrawingFrame(element)) {
                    // TODO: validate DOM of paragraphs embedded in text frames
                    drawingStyles.updateElementFormatting(element);
                    updateFormatProgress(currentProgressTicks += 1);
                }
                else if (DOM.isTableNode(element) && !DOM.isExceededSizeTableNode(element)) {
                    // Bug 28409: Validate DOM contents of embedded paragraphs without
                    // formatting (done by the following table formatting). Finds and
                    // processes all cell paragraphs contained in a top-level table
                    // (also from embedded tables).
                    if (topLevel) {
                        element.find(DOM.CELLCONTENT_NODE_SELECTOR + ' > ' + DOM.PARAGRAPH_NODE_SELECTOR).each(function () {
                            validateParagraphNode(this);
                        });
                    }
                    // update table formatting asynchronous to prevent browser alerts
                    promise = tableStyles.updateElementFormatting(element, { async: true })
                        .progress(function (localProgress) {
                            updateFormatProgress(currentProgressTicks + TABLE_PROGRESS_TICKS * localProgress);
                        })
                        .always(function () {
                            currentProgressTicks += TABLE_PROGRESS_TICKS;
                        });
                }

                return promise;
            }

            // updates the formatting of all lists, forwards the progress, returns a promise
            function updateListFormatting() {
                return updateLists({ async: true }).progress(function (progress) {
                    def.notify(0.9 + 0.1 * progress);  // last 10% for updating lists
                });
            }

            // updates page by inserting page layout
            function callInsertPageBreaks() {
                return initialPageBreaks();
            }

            // dumps profiling information to browser console
            function dumpProfilingInformation() {

                var tableCount = editdiv.find(DOM.TABLE_NODE_SELECTOR).length,
                    cellCount = editdiv.find('td').length,
                    paragraphCount = editdiv.find(DOM.PARAGRAPH_NODE_SELECTOR).length,
                    spanCount = editdiv.find('span').filter(function () { return DOM.isPortionSpan(this) || DOM.isTextComponentNode(this.parentNode); }).length,
                    drawingCount = editdiv.find(DrawingFrame.NODE_SELECTOR).length;

                function dumpUpdateFormattingCount(styleSheets, elementCount, elementName) {
                    var updateCount = styleSheets.DBG_COUNT || 0,
                        perElementCount = (elementCount === 0) ? 0 : Utils.round(updateCount / elementCount, 0.01);
                    if ((elementCount > 0) || (updateCount > 0)) {
                        Utils.info('Editor.updateDocumentFormatting(): ' + elementCount + ' ' + elementName + ' updated ' + updateCount + ' times (' + perElementCount + ' per element)');
                    }
                }

                function dumpParagraphCount(methodName, debugCount) {
                    var updateCount = debugCount || 0,
                        perElementCount = (paragraphCount === 0) ? 0 : Utils.round(updateCount / paragraphCount, 0.01);
                    Utils.info('Editor.' + methodName + '(): called ' + updateCount + ' times (' + perElementCount + ' per paragraph)');
                }

                dumpUpdateFormattingCount(characterStyles, spanCount, 'text spans');
                dumpUpdateFormattingCount(paragraphStyles, paragraphCount, 'paragraphs');
                dumpUpdateFormattingCount(tableCellStyles, cellCount, 'table cells');
                dumpUpdateFormattingCount(tableStyles, tableCount, 'tables');
                dumpUpdateFormattingCount(drawingStyles, drawingCount, 'drawing objects');
                dumpParagraphCount('validateParagraphNode', validateParagraphNode.DBG_COUNT);
            }


            // detach all child nodes of the page content node, this improves
            // performance in very large documents notably
            formattingNodes.filter(function () { return this.parentNode === pageContentNode[0]; }).detach();

            // first, update the root page node
            pageStyles.updateElementFormatting(editdiv);

            // process all elements contained in formattingNodes, then update list formatting
            app.processArrayDelayed(updateElementFormatting, formattingNodes, { chunkLength: 1 })
                .then(updateListFormatting)
                .then(callInsertPageBreaks)
                .done(function () { dumpProfilingInformation(); def.resolve(); })
                .fail(function () { def.reject(); });

            return def.promise();
        };

        /**
         * Getter method for maximum height of page (without margins)
         */
        this.getPageMaxHeight = function () {
            return Utils.convertHmmToLength(pageAttributes.page.height - pageAttributes.page.marginTop - pageAttributes.page.marginBottom, 'px', 1);
        };

        /**
         * Inital looping through current 'page content DOM' and doing addition of heights of elements to calculate page breaks positions.
         * This function is called only on document loading, and that's why it loops trough all DOM nodes
         *
         */
        function initialPageBreaks() {
            if (isSmallDevice) { return; }
            Utils.takeTime('Editor.initialPageBreaks(): ', function () {
                var
                    pageContentNode = DOM.getPageContentNode(editdiv),
                    processingNodes = $(pageContentNode).find('> div.p, > table'),
                    contentHeight = 0,
                    elementHeightWithoutMB = 0,
                    elementHeight = 0,
                    nodeHelper,
                    zoomFactor,
                    floatingDrawings,
                    fDrawingHeight,
                    fDrawingWidth = 0,
                    accHeightFromPrevDrawing = 0,
                    heightOfRectPartHelper = 0,
                    elementSpan,
                    offsetPosition = [],
                    newlySplittedSpan,
                    markup = '',
                    pageBreak,
                    elMarginBottom,
                    elMarginTop,
                    prevElMarginBottom = 0;

                _.each(processingNodes, function (element, iterator) {
                    // processing node with floated drawing(s) inside of it
                    fDrawingHeight = 0;
                    floatingDrawings = $(element).find('.drawing.float');
                    if (floatingDrawings.length > 0) {
                        _.each(floatingDrawings, function (drawing) {
                            if ($(drawing).outerWidth(true) !== pageContentNode.width()) {
                                if ($(drawing).outerHeight(true) + ($(drawing).position().top / zoomFactor) > fDrawingHeight) {
                                    fDrawingHeight = $(drawing).outerHeight(true) + ($(drawing).position().top / zoomFactor);
                                    fDrawingWidth = $(drawing).outerWidth(true);
                                }
                            }
                        });
                    }
                    if (accHeightFromPrevDrawing > fDrawingHeight) {
                        fDrawingHeight = 0;
                    }
                    elementHeightWithoutMB = element.offsetHeight;
                    elMarginBottom = parseInt($(element).css('margin-bottom'), 10);
                    elMarginTop = parseInt($(element).css('margin-top'), 10);
                    if (elMarginTop > prevElMarginBottom) { // el has margin top that is greater than margin bottom of prev el, include diff in page height
                        elementHeightWithoutMB += elMarginTop - prevElMarginBottom;
                    }
                    elementHeight = elementHeightWithoutMB + elMarginBottom;
                    prevElMarginBottom = elMarginBottom;
                    // if current content height is bellow max page height;
                    if (contentHeight + elementHeight <= pageMaxHeight  && (fDrawingHeight < 1 || contentHeight + fDrawingHeight <= pageMaxHeight) && (accHeightFromPrevDrawing < 1 || (contentHeight + accHeightFromPrevDrawing <= pageMaxHeight))) {
                        contentHeight += elementHeight;
                        if (accHeightFromPrevDrawing > elementHeight) {
                            accHeightFromPrevDrawing -= elementHeight;
                        } else {
                            heightOfRectPartHelper = accHeightFromPrevDrawing;
                            accHeightFromPrevDrawing = 0;
                        }
                    // for last element that can fit on page we need to omit margin bottom
                    } else if (contentHeight + elementHeightWithoutMB <= pageMaxHeight  && (fDrawingHeight < 1 || contentHeight + fDrawingHeight <= pageMaxHeight) && (accHeightFromPrevDrawing < 1 || (contentHeight + accHeightFromPrevDrawing <= pageMaxHeight))) {
                        contentHeight += elementHeightWithoutMB;
                        if (accHeightFromPrevDrawing > elementHeightWithoutMB) {
                            accHeightFromPrevDrawing -= elementHeightWithoutMB;
                        } else {
                            heightOfRectPartHelper = accHeightFromPrevDrawing;
                            accHeightFromPrevDrawing = 0;
                        }
                    } else {
                        prevElMarginBottom = 0;
                        if ($(element).hasClass('p') && $(element).text().length > 10 && $(element).find('.drawing').length < 1) {
                            // paragraph split preparation
                            if (accHeightFromPrevDrawing > 0 && fDrawingWidth > 0) {
                                offsetPosition = getOffsetPositionFromElement(element, contentHeight, false, accHeightFromPrevDrawing, fDrawingWidth);
                            } else {
                                offsetPosition = getOffsetPositionFromElement(element, contentHeight, false);
                            }
                            var containsSpecIndentElements = $(element.firstChild).hasClass('list-label') || $(element.firstChild).hasClass('inline tab');
                            if (offsetPosition && offsetPosition.length > 0 && !(offsetPosition[0] === 0 && (containsSpecIndentElements || accHeightFromPrevDrawing > 0))) { // if it's list and offset is 0, dont place pagebreak in between first char and list label
                                elementSpan = $(element).children('span').first();
                                while (elementSpan.text().length < 1 || !(elementSpan.is('span'))) {
                                    elementSpan = elementSpan.next();
                                }
                                if (elementSpan && elementSpan.length > 0) { //fix for #32563, if div.p contains only empty spans
                                    _.each(offsetPosition, function (offset) {
                                        if (offset === 0) {
                                            $(element).children('span').first().addClass('break-above-span');
                                            $(element).addClass('has-breakable-span');
                                        } else {
                                            var elTextLength = $(elementSpan).text().length;
                                            while (elTextLength > 0 && elTextLength <= offset) {
                                                offset -= elTextLength;
                                                elementSpan = elementSpan.next();
                                                while (!elementSpan.is('span')) { // we might fetch some inline div nodes, like div.tab or div.linebreak
                                                    elementSpan = elementSpan.next();
                                                }
                                                elTextLength = $(elementSpan).text().length;
                                            }
                                            newlySplittedSpan = DOM.splitTextSpan(elementSpan, offset, {append: true});
                                            newlySplittedSpan.addClass('break-above-span');
                                            $(element).addClass('has-breakable-span');
                                            elementSpan = newlySplittedSpan;
                                        }
                                    });
                                }
                            }
                        } //end of paragraph split preparation
                        if ($(element).hasClass('has-breakable-span')) {
                            zoomFactor = app.getView().getZoomFactor() / 100;
                            pageBreak = $();
                            var arrBreakAboveSpans = $(element).find('.break-above-span'),
                                upperPartDiv = 0,
                                totalUpperParts = 0,
                                diffPosTopAndPrev = 0,
                                elementLeftIndent = parseInt($(element).css('margin-left'), 10) + parseInt($(element).css('padding-left'), 10);// for indented elements such as lists, or special paragraphs
                            if (arrBreakAboveSpans) {
                                _.each(arrBreakAboveSpans, function (breakAboveSpan) {
                                    diffPosTopAndPrev += $(pageBreak).outerHeight(true) + upperPartDiv;
                                    upperPartDiv = $(breakAboveSpan).position().top / zoomFactor - diffPosTopAndPrev;
                                    totalUpperParts += upperPartDiv;
                                    contentHeight += upperPartDiv;
                                    nodeHelper = $(element).prev();
                                    while (nodeHelper.hasClass('page-break')) {
                                        nodeHelper = nodeHelper.prev();
                                    }
                                    nodeHelper.addClass('last-on-page');
                                    pageBreak = $('<div>').addClass('page-break').attr('contenteditable', 'false').css({
                                        'width' : pageWidth,
                                        'margin' : (pagePaddingBottom + Math.max(pageMaxHeight - contentHeight, 0)) + 'px 0px ' + pagePaddingTop + 'px -' + (pagePaddingLeft + elementLeftIndent) + 'px'
                                    });
                                    $(breakAboveSpan).before(pageBreak);
                                    contentHeight = 0;
                                });
                            }
                            contentHeight = elementHeight - totalUpperParts;
                        } else if ($(element).is('table') && element.rows.length > 1) { // table split
                            var accumulator = 0,
                                totalAcc = 0,
                                arrAccumulator = [],
                                tempContentHeight = contentHeight, //helper var for big tables
                                breakAboveRows,
                                tableHeight = elementHeight, //helper height value, for tables bigger than one page
                                summedAttrPBHeight = 0, //summed value of all rows above last page break
                                negativeMarginTablesFix = _.browser.IE ? 2 : (_.browser.Firefox ? 1 : 0); // IE needs 2px more to left for splitted tables, Firefox 1px - due to css border-collapse

                            _.each(element.rows, function (tr) { // $(element).find('> tbody > tr')
                                if (accumulator + tr.offsetHeight < pageMaxHeight - tempContentHeight) {
                                    accumulator += tr.offsetHeight;
                                } else {
                                    $(tr).addClass('break-above-tr');
                                    $(element).addClass('tb-split-nb');
                                    totalAcc += accumulator;
                                    arrAccumulator.push(totalAcc);
                                    tempContentHeight = tr.offsetHeight;
                                    totalAcc = tempContentHeight;
                                    accumulator = 0;
                                }
                            });

                            if ($(element).find('tr').first().hasClass('break-above-tr')) {
                                $(element).find('tr').first().removeClass('break-above-tr');
                                if ($(element).prev().length > 0) {
                                    markup = '<div class="page-break" contenteditable=' + (_.browser.WebKit ? '"false"' : '""') + 'style="width: ' + pageWidth +
                                            'px; margin: ' + (pagePaddingBottom + Math.max(pageMaxHeight - contentHeight, 0)) + 'px 0px ' + pagePaddingTop + 'px -' + pagePaddingLeft + 'px"></div>';
                                    $(element).before(markup);
                                }
                                $(element).addClass('break-above'); // this is marker for first node bellow pagebreak
                                contentHeight = 0;
                                arrAccumulator.shift(); // remove first el from array, which is 0 in this case
                            }

                            breakAboveRows = $(element).find('.break-above-tr');
                            if (breakAboveRows.length > 0) {
                                _.each(breakAboveRows, function (breakAboveRow, i) {
                                    contentHeight += arrAccumulator[i];
                                    markup = '<tr class=\'pb-row\'><td style=\'padding: 0px; border: none\' colspan=\'1\'><div class=\'page-break\' contenteditable=' +
                                        (_.browser.WebKit ? '\'false\'' : '\'\'') + 'style=\'width: ' + pageWidth + 'px; margin: ' + (pagePaddingBottom + Math.max(pageMaxHeight - contentHeight, 0)) +
                                        'px 0px ' + pagePaddingTop + 'px -' + (pagePaddingLeft + negativeMarginTablesFix) + 'px\'></div></td></tr>';
                                    $(breakAboveRow).before(markup);
                                    if (tableHeight > pageMaxHeight) {
                                        contentHeight = 0;
                                    } else {
                                        contentHeight = tableHeight - arrAccumulator[i];
                                    }
                                    tableHeight = tableHeight - arrAccumulator[i];
                                    summedAttrPBHeight += arrAccumulator[i];
                                });
                                contentHeight = elementHeight - summedAttrPBHeight; // height of part of table transfered to new page
                            } else {
                                contentHeight = elementHeight;
                            }
                            //end of table split
                        } else { // default insertion of page break between elements
                            nodeHelper = $(element).prev();
                            while (nodeHelper.hasClass('page-break')) {
                                nodeHelper = nodeHelper.prev();
                            }
                            if (nodeHelper.length > 0) {
                                nodeHelper.addClass('last-on-page');
                                markup = '<div class="page-break" contenteditable=' + (_.browser.WebKit ? '"false"' : '""') + 'style="width: ' + pageWidth +
                                    'px; margin: ' + (pagePaddingBottom + Math.max(pageMaxHeight - contentHeight, 0)) + 'px 0px ' + pagePaddingTop + 'px -' + pagePaddingLeft + 'px"></div>';
                                $(element).before(markup);
                            }
                            $(element).addClass('break-above'); // this is marker for first node bellow pagebreak
                            contentHeight = elementHeight; // addition if content is shifted down
                        }
                    }
                    if (fDrawingHeight > 0) { // store the diff between floated image height and height of containing p, for next iteration
                        accHeightFromPrevDrawing = fDrawingHeight - elementHeight;
                    }
                    //at the last page, we add extra padding to fill in to the size of max page height
                    if (processingNodes.length === iterator + 1) {
                        pageContentNode.css({
                            'padding-bottom': pageMaxHeight - contentHeight
                        });
                        $(element).addClass('last-on-page');
                    }
                });
            });
        }

        /**
         *  Creates invisible helper node to calculate position(s) in text where paragraph should be split.
         *  Also stores position of all rest new line beginings into node with jQuery data()
         *  Words and spaces are ocurring in pairs, sequently. Space can be one character, or more grouped, then nbsp is used
         *
         *  @param {Node|jQuery} element - element whose offset is calculated
         *  @param {Number} contentHeight - current height value of all processed nodes
         *  @param {Boolean} modifyingElement - if the current processing node is modified or untouched
         *  @param {Number} partHeight - optional - if there is floating drawing from previous paragraphs, use height to create dummy div
         *  @param {Number} partWidth - optional - if there is floating drawing from previous paragraphs, use width to create dummy div
         *  @returns {Array} offsetPosition - calculated position(s) in text where paragraph should be split
         *
         */
        function getOffsetPositionFromElement(element, contentHeight, modifyingElement, partHeight, partWidth) {

            var
                offsetPosition = [],
                elementSpan,
                multiplicator = 1,
                contentHeightCached = contentHeight,
                newElement,
                elementHeight = $(element)[0].offsetHeight,
                elementWidth = $(element)[0].offsetWidth,
                elementLineHeightStyle = $(element).css('line-height'),
                elementIndentLeft = parseInt($(element).css('padding-left'), 10) + parseInt($(element).css('margin-left'), 10),
                oxoPosInPar,
                startSpan,
                numOfPageBreaks,
                cachedDataForLineBreaks = [],
                cachedDataForLineBreaksLength,
                helperOffsetArray = [],
                elSpanPosition,
                charLength = 0,
                elSpans,
                $elDrawing, //paragraph contains image(s)
                drawingTopPos,
                drawingBottomPos;

            if (!elementIndentLeft) { elementIndentLeft = 0; }
            if (_.isEmpty($(element).data('lineBreaksData')) || modifyingElement) { // if no cached data, or element is modified, calculate all positions
                newElement = $(element).clone(true);
                if (partHeight > 0 && partWidth > 0) { // if there is drawing leftover from previous paragraphs, it will change p layout
                    var markup = '<div style="width: ' + partWidth + 'px; height: ' + partHeight + 'px; float: left"></div>';
                    newElement.prepend(markup);
                }
                newElement.prependTo('#io-ox-office-temp').css({'width': elementWidth, 'line-height': elementLineHeightStyle});

                $elDrawing = $(newElement).find('div.drawing');
                if ($elDrawing.length > 0) {
                    drawingTopPos = $elDrawing.first().position().top;
                    drawingBottomPos = drawingTopPos + $elDrawing.first().outerHeight(true);
                    if ($elDrawing.length > 1) {
                        _.each($elDrawing, function (image) {
                            var imagePosTop = $(image).position().top;
                            if (imagePosTop < drawingTopPos) { drawingTopPos = imagePosTop; }
                            if (imagePosTop + $(image).outerHeight(true) > drawingBottomPos) { drawingBottomPos = imagePosTop + $(image).outerHeight(true); }
                        });
                    }
                    //$elDrawing.next('span').css('display', 'block');
                }

                elementSpan = newElement.children('span');
                _.each(elementSpan, function (el, index) {
                    var elText = $(el).text(),
                        words = elText.match(/\S+/g),
                        markup = '',
                        spaces = elText.match(/(\s+)/gi),
                        charAtBegining;

                    charAtBegining = (elText[0]) ? elText[0].match(/\S+/g) : null; // check if text begins with characters or whitespace

                    if (words && words.length > 0) {
                        _.each(words, function (word, i) {
                            if (word.length > 0) {
                                var generatedSpace;
                                if (spaces && spaces[i]) {
                                    if (spaces[i].length > 1) {
                                        generatedSpace = '';
                                        for (var j = 0; j < spaces[i].length; j++) { //  translates for ex. '     ' into -> ' &nbsp; &nbsp; ' (5 whitespaces)
                                            if (j % 2 !== 0) {
                                                generatedSpace += ' ';
                                            } else {
                                                generatedSpace += '&nbsp;';
                                            }
                                        }
                                    } else {
                                        if (index === 0 && i === 0 && !charAtBegining) { // fix for first span with whitespace (it will render as 0px span, so we fake content with dot)
                                            generatedSpace = '.';
                                        } else {
                                            generatedSpace = ' ';
                                        }
                                    }
                                } else {
                                    generatedSpace = '';
                                }
                                if (charAtBegining) { // word, space direction
                                    if (generatedSpace.length > 0) {
                                        markup += '<span class="textData" data-charLength="' + (charLength += word.length) + '">' + Utils.escapeHTML(word) + '</span>' + '<span class="whitespaceData" data-charLength="' +  (charLength += spaces[i].length) + '">' + generatedSpace + '</span>';
                                    } else {
                                        markup += '<span class="textData" data-charLength="' + (charLength += word.length) + '">' + Utils.escapeHTML(word) + '</span>';
                                    }
                                } else { // space, word direction
                                    if (generatedSpace.length > 0) {
                                        markup += '<span class="whitespaceData" data-charLength="' +  (charLength += spaces[i].length) + '">' + generatedSpace + '</span>' + '<span class="textData" data-charLength="' + (charLength += word.length) + '">' + Utils.escapeHTML(word) + '</span>';
                                    } else {
                                        markup += '<span class="textData" data-charLength="' + (charLength += word.length) + '">' + Utils.escapeHTML(word) + '</span>';
                                    }
                                }
                            }
                        });
                        $(el).empty().append(markup);
                    }
                });

                startSpan = $(elementSpan).find('span').first();
                _.each(elementSpan, function (el) {
                    elSpans = $(el).children('.textData');
                    _.each(elSpans, function (elSpan) {
                        if ($(elSpan).text().length > 0) { //if (Utils.getDomNode(elSpan).firstChild)
                            elSpanPosition = $(elSpan).position();
                            if ((elSpanPosition.left - elementIndentLeft < 1) && elSpanPosition.top > 1 && !$(elSpan).is(startSpan)) {
                                if (drawingTopPos && drawingBottomPos) {
                                    if (elSpanPosition.top < drawingTopPos || elSpanPosition.top > drawingBottomPos) {
                                        oxoPosInPar = parseInt($(elSpan).attr('data-charLength'), 10) - $(elSpan).text().length;
                                        cachedDataForLineBreaks.push({'oxoPosInPar': oxoPosInPar, 'distanceTop': elSpanPosition.top});
                                    }
                                } else {
                                    oxoPosInPar = parseInt($(elSpan).attr('data-charLength'), 10) - $(elSpan).text().length;
                                    cachedDataForLineBreaks.push({'oxoPosInPar': oxoPosInPar, 'distanceTop': elSpanPosition.top});
                                }
                            }
                        }
                    });
                });
                $(element).data('lineBreaksData', cachedDataForLineBreaks);
                newElement.remove(); //clear dom after getting position
            } else {
                cachedDataForLineBreaks = $(element).data('lineBreaksData');
            }

            cachedDataForLineBreaksLength = cachedDataForLineBreaks.length;
            if (cachedDataForLineBreaksLength > 0) {
                helperOffsetArray = [];
                multiplicator = 0;
                if (elementHeight > pageMaxHeight) {
                    multiplicator = ~~((elementHeight + contentHeightCached) / pageMaxHeight); //get int value how many times is element bigger than page
                    numOfPageBreaks = multiplicator;
                    multiplicator -= 1;
                    if (offsetPosition.length === 0 && cachedDataForLineBreaksLength > 0) {
                        Utils.iterateArray(cachedDataForLineBreaks, function (el, iterator) {
                            if (el.distanceTop < (pageMaxHeight - contentHeightCached + pageMaxHeight * multiplicator) && multiplicator > -1) {
                                offsetPosition.push(el.oxoPosInPar);
                                multiplicator -= 1;
                            } else if (iterator === 0 && (el.distanceTop > (pageMaxHeight - contentHeightCached + pageMaxHeight * multiplicator) && multiplicator > -1)) {// first row (backwards) cannot fit into free space, it is also shifted down (pb on begining, pos 0)
                                offsetPosition.push(0);
                            }
                        }, {reverse: true});
                        if (offsetPosition.length < numOfPageBreaks) {
                            // now we know space left is not enough for last line, so we put page break before it
                            offsetPosition.push(cachedDataForLineBreaks[cachedDataForLineBreaksLength - 1].oxoPosInPar);
                        }
                    }
                    offsetPosition.reverse();
                    _.each(offsetPosition, function (offPos, iterator) {
                        if (iterator > 0) {
                            offPos -=  offsetPosition[iterator - 1];
                        }
                        helperOffsetArray.push(offPos);
                    });
                    offsetPosition = helperOffsetArray.slice();
                } else {
                    //if its not multi paragraph split
                    if (offsetPosition.length === 0 && cachedDataForLineBreaksLength > 0) {
                        Utils.iterateArray(cachedDataForLineBreaks, function (el, iterator) {
                            if (el.distanceTop < (pageMaxHeight - contentHeightCached)) {
                                offsetPosition.push(el.oxoPosInPar);
                                return Utils.BREAK;
                            } else if (iterator === 0 && (el.distanceTop > (pageMaxHeight - contentHeightCached))) {// first row (backwards) cannot fit into free space, it is also shifted down (pb on begining, pos 0)
                                offsetPosition.push(0);
                                return Utils.BREAK;
                            }
                        }, {reverse: true});
                    }
                    if (offsetPosition.length === 0) {
                        // now we know space left is not enough for last line, so we put page break before it
                        offsetPosition.push(cachedDataForLineBreaks[cachedDataForLineBreaksLength - 1].oxoPosInPar);
                    }
                }
            }

            return offsetPosition;
        }

        /**
         * Looping through current 'page content DOM' and adding heights of elements to calculate page breaks positions.
         * Starts from element which is modified and continues downwards trough DOM. Stops if no more elements are shifted to new page
         * last-on-page marker: used for checking if content is shifted to another page, if not, stop iteration
         * break-above marker: used to mark element above which page break node should be inserted
         *
         */

        function insertPageBreaks() {
            if (!pbState || isSmallDevice) { return; } // if page layout is disabled, dont run the function
            Utils.takeTime('Editor.insertPageBreaks(): ', function () {
                var
                    contentHeight = 0,
                    floatingDrawings,
                    pageContentNode = DOM.getPageContentNode(editdiv),
                    elementHeight = 0,
                    elementHeightWithoutMB = 0,
                    processingNodes,
                    elementSpan,
                    newlySplittedSpan,
                    nodeHelper,
                    zoomFactor = app.getView().getZoomFactor() / 100,
                    offsetPosition = [],
                    modifyingElement,
                    pageBreak,
                    markup = '',
                    elMarginBottom,
                    elMarginTop,
                    prevElMarginBottom = 0,
                    fDrawingHeight,
                    fDrawingWidth = 0,
                    accHeightFromPrevDrawing = 0,
                    heightOfRectPartHelper = 0,
                    //collectionOfDrawingData = {},
                    //flags for controling which pages to process, and where to stop with processing
                    controlBreak = false,
                    refreshNextPage = false,
                    refreshPrevPage = false;

                function prepareNodeAndRemovePageBreak(element) {
                    if ($(element).hasClass('last-on-page')) {
                        $(element).removeClass('last-on-page');
                        refreshNextPage = true; //content is shifted up, next page has to be recalculated also
                    }
                    nodeHelper = $(element).next();
                    //if element is last in document
                    if (nodeHelper.length === 0) {
                        $(element).addClass('last-on-page');
                    } else {
                        // remove old page break
                        while (nodeHelper.hasClass('page-break')) {
                            nodeHelper.remove();
                            nodeHelper = $(element).next();
                        }
                    }
                }

                function processNodesForPageBreaksInsert(processingNodes) {
                    Utils.iterateArray(processingNodes, function (element, iterator) {
                        // first merge back any previous paragraph splits
                        pageBreak = $(element).find('div.page-break');
                        if ($(element).hasClass('has-breakable-span') || pageBreak.length > 0) {
                            pageBreak.remove();
                            _.each($(element).children('span'), function (span) {
                                if ($(span).hasClass('break-above-span')) {
                                    $(span).removeClass('break-above-span');
                                    CharacterStyles.mergeSiblingTextSpans(span);
                                    //CharacterStyles.mergeSiblingTextSpans(span, true);
                                }
                            });
                            if ($(element).hasClass('p')) {
                                validateParagraphNode(element);
                                $(element).removeClass('has-breakable-span');
                            }
                        }
                        // if table is split, merge back
                        if ($(element).find('tr.pb-row').length > 0) {
                            $(element).find('tr.pb-row').remove();
                            $(element).find('tr.break-above-tr').removeClass('break-above-tr');
                            if ((_.browser.Firefox) && (DOM.isTableNode(element))) { Table.forceTableRendering(element); } // forcing Firefox to rerender table (32461)
                        }
                        $(element).removeClass('tb-split-nb'); // return box shadow property to unsplitted tables

                        fDrawingHeight = 0;
                        floatingDrawings = $(element).find('.drawing.float');
                        if (floatingDrawings.length > 0) {
                            _.each(floatingDrawings, function (drawing) {
                                if ($(drawing).outerWidth(true) !== pageContentNode.width()) {
                                    if ($(drawing).outerHeight(true) + ($(drawing).position().top / zoomFactor) > fDrawingHeight) {
                                        fDrawingHeight = $(drawing).outerHeight(true) + ($(drawing).position().top / zoomFactor);
                                        fDrawingWidth = $(drawing).outerWidth(true);
                                    }
                                }
                            });
                        }
                        if (accHeightFromPrevDrawing > fDrawingHeight) {
                            fDrawingHeight = 0;
                        }
                        elementHeightWithoutMB = element.offsetHeight;
                        elMarginBottom = parseInt($(element).css('margin-bottom'), 10);
                        elMarginTop = parseInt($(element).css('margin-top'), 10);

                        if (elMarginTop > prevElMarginBottom) { // el has margin top that is greater than margin bottom of prev el, include diff in page height
                            elementHeightWithoutMB += elMarginTop - prevElMarginBottom;
                        }
                        elementHeight = elementHeightWithoutMB + elMarginBottom;
                        prevElMarginBottom = elMarginBottom;
                        // if current content height is bellow max page height
                        // and if there is no image floated inside paragraph, or if so, p fits inside page with it also
                        // and if there is no floated image from one of previous paragraphs that intersects current p, or if exist, element can fit on page with it also;
                        if (contentHeight + elementHeight <= pageMaxHeight && (fDrawingHeight < 1 || contentHeight + fDrawingHeight <= pageMaxHeight) && (accHeightFromPrevDrawing < 1 || (contentHeight + accHeightFromPrevDrawing <= pageMaxHeight))) {
                            contentHeight += elementHeight;
                            if (accHeightFromPrevDrawing > elementHeight) {
                                accHeightFromPrevDrawing -= elementHeight;
                            } else {
                                heightOfRectPartHelper = accHeightFromPrevDrawing;
                                accHeightFromPrevDrawing = 0;
                            }
                            prepareNodeAndRemovePageBreak(element);
                        // for last element that can fit on page we need to omit margin bottom
                        } else if (contentHeight + elementHeightWithoutMB <= pageMaxHeight && (fDrawingHeight < 1 || contentHeight + fDrawingHeight <= pageMaxHeight) && (accHeightFromPrevDrawing < 1 || (contentHeight + accHeightFromPrevDrawing <= pageMaxHeight))) {
                            contentHeight += elementHeightWithoutMB;
                            if (accHeightFromPrevDrawing > elementHeightWithoutMB) {
                                accHeightFromPrevDrawing -= elementHeightWithoutMB;
                            } else {
                                heightOfRectPartHelper = accHeightFromPrevDrawing;
                                accHeightFromPrevDrawing = 0;
                            }
                            prepareNodeAndRemovePageBreak(element);
                        } else { //shift to new page or split
                            prevElMarginBottom = 0;
                            if ($(element).hasClass('last-on-page') || refreshPrevPage || refreshNextPage) {
                                $(element).removeClass('last-on-page');
                                refreshPrevPage = false;
                                refreshNextPage = false;
                            } else {
                                //controlBreak = true;
                            }
                            nodeHelper = $(element).next();
                            $(element).prev().addClass('last-on-page');
                            //if element is last in document
                            if (nodeHelper.length === 0) {
                                $(element).addClass('last-on-page');
                            } else { // remove old page break
                                while (nodeHelper.hasClass('page-break')) {
                                    nodeHelper.remove();
                                    nodeHelper = $(element).next();
                                }
                            }

                            if ($(element).hasClass('p') && $(element).text().length > 10 && $(element).find('.drawing').length < 1) {  // paragraph has to be split
                                offsetPosition = [];
                                modifyingElement = currentProcessingNode[0] === element; // flag if we are modifying the element, then cached line positions are invalid
                                if (accHeightFromPrevDrawing > 0 && fDrawingWidth > 0) {
                                    offsetPosition = getOffsetPositionFromElement(element, contentHeight, modifyingElement, accHeightFromPrevDrawing, fDrawingWidth);
                                } else {
                                    offsetPosition = getOffsetPositionFromElement(element, contentHeight, modifyingElement);
                                }
                                var containsSpecIndentElements = $(element.firstChild).hasClass('list-label') || $(element.firstChild).hasClass('inline tab');
                                if (offsetPosition  && offsetPosition.length > 0 && !(offsetPosition[0] === 0 && (containsSpecIndentElements || accHeightFromPrevDrawing > 0))) { // if it's list and offset is 0, dont place pagebreak in between first char and list label
                                    elementSpan = $(element).children('span').first();
                                    while (elementSpan.text().length < 1 || !(elementSpan.is('span'))) {
                                        elementSpan = elementSpan.next();
                                    }
                                    if (elementSpan && elementSpan.length > 0) { //fix for #32563, if div.p contains only empty spans
                                        _.each(offsetPosition, function (offset) {
                                            if (offset === 0) {
                                                $(element).children('span').first().addClass('break-above-span');
                                                $(element).addClass('has-breakable-span');
                                            } else {
                                                var elTextLength = $(elementSpan).text().length;
                                                while (elTextLength > 0 && elTextLength <= offset) {
                                                    offset -= elTextLength;
                                                    elementSpan = elementSpan.next();
                                                    while (!elementSpan.is('span')) { // we might fetch some inline div nodes, like div.tab or div.linebreak
                                                        elementSpan = elementSpan.next();
                                                    }
                                                    elTextLength = $(elementSpan).text().length;
                                                }
                                                newlySplittedSpan = DOM.splitTextSpan(elementSpan, offset, {append: true});
                                                newlySplittedSpan.addClass('break-above-span');
                                                $(element).addClass('has-breakable-span');
                                                elementSpan = newlySplittedSpan;
                                            }
                                        });
                                    }
                                    refreshNextPage = true;
                                }
                            } //end of paragraph split preparation

                            if ($(element).hasClass('has-breakable-span')) {
                                zoomFactor = app.getView().getZoomFactor() / 100;
                                pageBreak = $();
                                var arrBreakAboveSpans = $(element).find('.break-above-span'),
                                    upperPartDiv = 0,
                                    totalUpperParts = 0,
                                    diffPosTopAndPrev = 0,
                                    elementLeftIndent = parseInt($(element).css('margin-left'), 10) + parseInt($(element).css('padding-left'), 10); // for indented elements such as lists, or special paragraphs

                                _.each(arrBreakAboveSpans, function (breakAboveSpan) {
                                    diffPosTopAndPrev += $(pageBreak).outerHeight(true) + upperPartDiv;
                                    upperPartDiv = $(breakAboveSpan).position().top / zoomFactor - diffPosTopAndPrev;
                                    totalUpperParts += upperPartDiv;
                                    contentHeight += upperPartDiv;
                                    nodeHelper = $(element).prev();
                                    while (nodeHelper.hasClass('page-break')) {
                                        nodeHelper = nodeHelper.prev();
                                    }
                                    nodeHelper.addClass('last-on-page');
                                    markup = '<div class="page-break" contenteditable="false" style="width: ' + pageWidth +
                                            'px; margin: ' + (pagePaddingBottom + Math.max(pageMaxHeight - contentHeight, 0)) + 'px 0px ' + pagePaddingTop + 'px -' + (pagePaddingLeft + elementLeftIndent) + 'px"></div>';
                                    $(breakAboveSpan).before(markup);
                                    pageBreak = $(breakAboveSpan).prev();
                                    contentHeight = 0;
                                });
                                contentHeight = elementHeight - totalUpperParts;
                            } else if ($(element).is('table') && element.rows.length > 1) { // table split
                                var accumulator = 0,
                                    totalAcc = 0,
                                    arrAccumulator = [],
                                    tempContentHeight = contentHeight, //helper var for big tables
                                    breakAboveRows,
                                    tableHeight = elementHeight, //helper height value, for tables bigger than one page
                                    summedAttrPBHeight = 0, //summed value of all rows above last page break
                                    negativeMarginTablesFix = _.browser.IE ? 2 : (_.browser.Firefox ? 1 : 0); // IE needs 2px more to left for splitted tables, Firefox 1px - due to css border-collapse

                                $(element).find('tr.pb-row').remove(); //maybe not necessary
                                $(element).find('tr.break-above-tr').removeClass(); //maybe not necessary
                                _.each(element.rows, function (tr) { // $(element).find('> tbody > tr')
                                    if (accumulator + tr.offsetHeight < pageMaxHeight - tempContentHeight) {
                                        accumulator += tr.offsetHeight;
                                    } else {
                                        $(tr).addClass('break-above-tr');
                                        $(element).addClass('tb-split-nb');
                                        totalAcc += accumulator;
                                        arrAccumulator.push(totalAcc);
                                        tempContentHeight = tr.offsetHeight;
                                        totalAcc = tempContentHeight;
                                        accumulator = 0;
                                    }
                                });

                                if ($(element).find('tr').first().hasClass('break-above-tr')) {
                                    $(element).find('tr').first().removeClass('break-above-tr');
                                    if ($(element).prev().length > 0) {
                                        markup = '<div class="page-break" contenteditable=' + (_.browser.WebKit ? '"false"' : '""') + 'style="width: ' + pageWidth +
                                                'px; margin: ' + (pagePaddingBottom + Math.max(pageMaxHeight - contentHeight, 0)) + 'px 0px ' + pagePaddingTop + 'px -' + pagePaddingLeft + 'px"></div>';
                                        $(element).before(markup);
                                    }
                                    $(element).addClass('break-above'); // this is marker for first node bellow pagebreak
                                    contentHeight = 0;
                                    arrAccumulator.shift();
                                }

                                breakAboveRows = $(element).find('.break-above-tr');
                                if (breakAboveRows.length > 0) {
                                    _.each(breakAboveRows, function (breakAboveRow, i) {
                                        contentHeight += arrAccumulator[i];
                                        markup = '<tr class=\'pb-row\'><td style=\'padding: 0px; border: none\' colspan=\'1\'><div class=\'page-break\' contenteditable=' +
                                            (_.browser.WebKit ? '\'false\'' : '\'\'') + 'style=\'width: ' + pageWidth + 'px; margin: ' + (pagePaddingBottom + Math.max(pageMaxHeight - contentHeight, 0)) +
                                            'px 0px ' + pagePaddingTop + 'px -' + (pagePaddingLeft + negativeMarginTablesFix) + 'px\'></div></td></tr>';
                                        $(breakAboveRow).before(markup);
                                        if (tableHeight > pageMaxHeight) {
                                            contentHeight = 0;
                                        } else {
                                            contentHeight = tableHeight - arrAccumulator[i];
                                        }
                                        tableHeight = tableHeight - arrAccumulator[i];
                                        summedAttrPBHeight += arrAccumulator[i];
                                    });
                                    contentHeight = elementHeight - summedAttrPBHeight; // height of part of table transfered to new page
                                } else {
                                    contentHeight = elementHeight;
                                }
                                if (_.browser.Firefox) { Table.forceTableRendering(element); } // forcing Firefox to rerender table (32461)
                                //end of table split
                            } else { // default insertion of pb between elements
                                nodeHelper = $(element).prev();
                                while (nodeHelper.hasClass('page-break')) {
                                    nodeHelper = nodeHelper.prev();
                                }
                                if (nodeHelper.length > 0) {
                                    nodeHelper.addClass('last-on-page');
                                    markup = '<div class="page-break" contenteditable=' + (_.browser.WebKit ? '"false"' : '""') + 'style="width: ' + pageWidth +
                                        'px; margin: ' + (pagePaddingBottom + Math.max(pageMaxHeight - contentHeight, 0)) + 'px 0px ' + pagePaddingTop + 'px -' + pagePaddingLeft + 'px"></div>';
                                    $(element).before(markup);
                                }
                                $(element).addClass('break-above'); // this is marker for first node bellow pagebreak
                                contentHeight = elementHeight; // addition if content is shifted down
                            }
                        }
                        if (fDrawingHeight > 0) { // store the diff between floated image height and height of containing p, for next iteration
                            accHeightFromPrevDrawing = fDrawingHeight - elementHeight;
                        }
                        //at the last page, we add extra padding, to fill it out to the size of max page height
                        if (processingNodes.length === iterator + 1) {
                            pageContentNode.css({
                                'padding-bottom': (pageMaxHeight - contentHeight > 0) ? pageMaxHeight - contentHeight : '36px'
                            });
                        }
                        if (controlBreak) {
                            return Utils.BREAK;
                        }
                    });
                }

                if (self.isProcessingOperations()) {
                    // to prevent triggering method when not all DOM nodes are rendered (still operations in the stack, #32067)
                    _.defer(function () {insertPageBreaksDebounced(currentProcessingNode ? currentProcessingNode : {}); });
                } else {
                    if (currentProcessingNode) {
                        currentProcessingNode = $(currentProcessingNode);
                        if (currentProcessingNode.hasClass('break-above')) { // first element bellow pagebreak, we want to refresh previous page also
                            nodeHelper = currentProcessingNode.prev();
                            while (nodeHelper.hasClass('page-break')) {
                                nodeHelper = nodeHelper.prev();
                            }
                            if (nodeHelper.length > 0) {
                                currentProcessingNode = nodeHelper;
                            }
                            refreshPrevPage = true;
                        }
                        if (currentProcessingNode.hasClass('last-on-page')) {
                            refreshNextPage = true;
                        }
                        if (currentProcessingNode.parents('table').length > 0) { //if we fetch valid div.p, but inside of table
                            currentProcessingNode = currentProcessingNode.parents('table').last();
                        }
                        nodeHelper = currentProcessingNode.prev();
                        if (nodeHelper.hasClass('page-break')) {
                            while (nodeHelper.hasClass('page-break')) { // might happen when deleting multiple elements at once, some pb are left - cleanup
                                nodeHelper = nodeHelper.prev();
                            }
                            if (!nodeHelper.hasClass('page-break')) {
                                nodeHelper = nodeHelper.prevUntil('.page-break, .has-breakable-span').last();
                                if (nodeHelper.prev().hasClass('has-breakable-span')) {
                                    contentHeight = (nodeHelper.prev().height() + parseInt(nodeHelper.prev().css('margin-bottom'), 10)) - nodeHelper.prev().find('.break-above-span').last().position().top / zoomFactor;
                                }
                                processingNodes = nodeHelper.nextAll('div.p, table').andSelf();
                            } else {
                                processingNodes = currentProcessingNode.nextAll('div.p, table').andSelf();
                            }
                        } else if (nodeHelper.hasClass('has-breakable-span')) { //if node is second bellow pagebreak - fallback
                            processingNodes = currentProcessingNode.nextAll('div.p, table').andSelf();
                            contentHeight = (nodeHelper.height() + parseInt(nodeHelper.css('margin-bottom'), 10)) - nodeHelper.find('.break-above-span').last().position().top / zoomFactor;
                        } else {
                            var procNodesFirst = currentProcessingNode.prevUntil('.page-break, .has-breakable-span').last();
                            nodeHelper = procNodesFirst.prev();
                            if (nodeHelper.hasClass('has-breakable-span')) {
                                processingNodes = procNodesFirst.nextAll('div.p, table').andSelf();
                                contentHeight = (nodeHelper.height() + parseInt(nodeHelper.css('margin-bottom'), 10)) - nodeHelper.find('.break-above-span').last().position().top / zoomFactor;
                            } else {
                                processingNodes = procNodesFirst.nextAll('div.p, table').andSelf();
                            }
                        }
                        if (processingNodes.length === 0) { // if we are somewhere on first page, no pageBreak class at the top!
                            processingNodes = $(pageContentNode).find('> div.p, > table');
                        }
                    } else if (_.isEmpty(currentProcessingNode)) { // fallback option if we dont fetch any valid node
                        processingNodes = $(pageContentNode).find('> div.p, > table');
                        currentProcessingNode = processingNodes.first();
                    }

                    $(processingNodes).not(':eq(0)').removeClass('break-above'); // remove all break-above classes except the top one
                    if (!$(processingNodes).first().hasClass('break-above')) {
                        //might happend that we delete this node; first node in collection has to have this class
                        $(processingNodes).first().addClass('break-above');
                    }
                    if (processingNodes.length < 40) {
                        processNodesForPageBreaksInsert(processingNodes);
                    } else {
                        var chunkNumber = 4,
                            nodeCounter = processingNodes.length,
                            chunkLength = Utils.round(nodeCounter / chunkNumber + 1, 1),
                            def = $.Deferred();
                        app.processArrayDelayed(processNodesForPageBreaksInsert, processingNodes, { chunkLength: chunkLength })
                        .done(function () {
                            def.resolve();
                            app.getView().recalculateDocumentMargin(); // because this is also deferred, we need to call recalculate margin once again
                        })
                        .fail(function () {
                            Utils.info('quitHandler, failed process all nodes for inserting page breaks');
                            def.reject();
                        });
                    }
                    app.getView().recalculateDocumentMargin(); // because this is also deferred, we need to call recalculate margin once again
                }
            });
        }

        /**
         * Turn page breaks on or off from debug pane
         *
         */
        this.togglePageBreaks = function () {
            var
                splittedParagraphs,
                pageContentNode = DOM.getPageContentNode(editdiv),
                spans;

            if (pbState) {
                pageContentNode.find('div.page-break').remove();
                pageContentNode.find('tr.pb-row').remove();
                pageContentNode.find('table.tb-split-nb').removeClass('tb-split-nb');
                pageContentNode.find('tr.break-above-tr').removeClass('break-above-tr');
                pageContentNode.find('div.break-above').removeClass('break-above');
                pageContentNode.find('div.last-on-page').removeClass('last-on-page');

                splittedParagraphs = pageContentNode.find('.has-breakable-span');
                if (splittedParagraphs.length > 0) {
                    _.each(splittedParagraphs, function (splittedItem) {
                        spans = $(splittedItem).find('.break-above-span').removeClass('break-above-span');
                        validateParagraphNode(splittedItem);
                        if (spans.length > 0) {
                            _.each(spans, function (span) {
                                CharacterStyles.mergeSiblingTextSpans(span);
                                CharacterStyles.mergeSiblingTextSpans(span, true);
                            });
                        }
                        $(splittedItem).removeClass('has-breakable-span');
                    });
                }
                app.getView().recalculateDocumentMargin();
                pbState = false;
            } else {
                initialPageBreaks();
                app.getView().recalculateDocumentMargin();
                pbState = true;
            }
        };

        /**
         * Called when all initial document operations have been processed.
         * Can be used to start post-processing tasks which need a fully
         * processed document.
         */
        function documentLoaded() {

            insertHyperlinkPopup();
            insertCollaborativeOverlay();
            insertSpellReplacementPopup();
            insertMissingCharacterStyles();
            insertMissingParagraphStyles();
            insertMissingTableStyles();

            if (onlineSpelling) {
                self.setOnlineSpelling(true, true);
            }

            // Special observer for iPad/Safari mobile. Due to missing events to detect
            // and process DOM changes we have to add special code to handle these cases.
            if (Utils.IPAD) {
                var MutationObserver = window.MutationObserver || window.WebKitMutationObserver || window.MozMutationObserver;

                editDivObserver = new MutationObserver(function (mutations) {

                    if (!self.isProcessingOperations()) {
                        // Optimization: we only check DOM mutations when we don't process our own
                        // operations.

                        mutations.forEach(function (mutation) {
                            var i = 0;

                            if (mutation.addedNodes.length > 0) {

                                for (i = 0; i < mutation.addedNodes.length; i++) {
                                    var node = mutation.addedNodes[i];

                                    if (node.nodeType === 1 && node.tagName === 'IMG' && node.className === '-webkit-dictation-result-placeholder') {
                                        // We have detected a Siri input. Switch model to read-only/disconnect and disconnect observer
                                        app.rejectEditAttempt('siri');
                                        editDivObserver.disconnect();
                                    }
                                }
                            }
                        });
                    }
                });

                editDivObserver.observe(editdiv[0], {childList: true, subtree: true});

                // Prevent text suggestions which are available via context menu on iPad/iPhone.
                selection.on('change', function (event, options) {
                    var paragraph, text;

                    // Performance: Ignoring handler for 'simpleTextSelection' operations
                    if (! Utils.getBooleanOption(options, 'simpleTextSelection', false)) {
                        if (selection.hasRange()) {
                            if (selection.getSelectionType() === 'text' && self.getEditMode() === true) {
                                paragraph = selection.getEnclosingParagraph();
                                text = null;

                                if (paragraph) {
                                    text = selection.getSelectedText();
                                    if (text && text.length > 0 && text.indexOf(' ') === -1) {
                                        setIOSROParagraph(paragraph);
                                    } else {
                                        setIOSROParagraph(null);
                                    }
                                }
                            }
                        } else {
                            // in case the text cursor is set to a different position
                            setIOSROParagraph(null);
                        }
                    }
                });
            }
        }

        this.accessRightsExport = function (mail) {
            var newWindow = null,
                startPosition = selection.getStartPosition(),
                endPosition = selection.getEndPosition(),
                startSpan = null,
                endSpan = null,
                nodeInfo = null;

            if (startPosition.length > 2 || endPosition.length > 2)
                return; //no export with selection from, to or inside of tables
            newWindow = window.open();
            nodeInfo = Position.getDOMPosition(editdiv, endPosition);
            nodeInfo.node = nodeInfo.node.parentNode;
            DOM.splitTextSpan(nodeInfo.node, nodeInfo.offset);
            endSpan = DOM.splitTextSpan(nodeInfo.node, 0)[0];
            $(endSpan).addClass('intangible-access-end');
            $(endSpan).text('>');

            nodeInfo = Position.getDOMPosition(editdiv, startPosition);
            nodeInfo.node = nodeInfo.node.parentNode;
            startSpan = DOM.splitTextSpan(nodeInfo.node, nodeInfo.offset)[0];
            DOM.splitTextSpan(startSpan, startSpan.firstChild.nodeValue.length);
            $(startSpan).addClass('intangible-access-start');
            $(startSpan).text('<');


            newWindow.document.write($(editdiv).html());

            $(startSpan).remove();
            $(endSpan).remove();

            function markNewDoc() {
                var accessStart = $(newWindow.document.body).find('span.intangible-access-start'),
                accessEnd = $(newWindow.document.body).find('span.intangible-access-end'),
                startPara = $(accessStart).parent(),
                paraSuccessors = startPara.nextAll().andSelf(),
                article = $('<article>').attr('id', '1'),
                //firstParagraph = $('<div>'),
                copyCompleted = false;


                $(newWindow.document.body).find('div.inline-popup').remove();

                $(newWindow.document.body).prepend(
                        $('<span>').attr('id', 'access').attr('itemtype', 'http://schema.org/Intangible/Access').attr('itemref', '1')
                        .attr('itemscope', 'itemscope').text('Access to this webpage is granted to:')

                            .append($('<span>').attr('itemprop', 'person').attr('itemtype', 'http://schema.org/Person')
                            .append(mail ? $('<span>').attr('itemprop', 'email').text(mail) : DOM.createTextSpan().attr('itemprop', 'email'))));

                //create article element
                $(accessStart).parent().before(article);

                //copy next paragraphs/tables etc.
                $(paraSuccessors).each(function (index, para) {
                    var accessEndParent = $(accessEnd).parent()[0];
                    if ($(para).is(accessEndParent)) {
                        copyCompleted = true;
                    }
                    //move complete Element
                    $(article).append(para);
                    if (copyCompleted) {
                        return false;
                    }
                });

                $(accessStart).remove();
                $(accessEnd).remove();

                //make list labels and tabs inline
                $(newWindow.document.body).find(DOM.LIST_LABEL_NODE_SELECTOR).css('display', 'inline-block').css('cursor', 'crosshair');
                $(newWindow.document.body).find(DOM.TAB_NODE_SELECTOR).css('display', 'inline-block');
            }
            markNewDoc();

        };
        // ==================================================================
        // END of Editor API
        // ==================================================================

        // ==================================================================
        // Private functions for document post-processing
        // ==================================================================

        /**
         * Inserts a hyperlink popup div into the DOM which is used to show
         * hyperlink information and change/remove functions.
         */
        function insertHyperlinkPopup() {
            Hyperlink.createPopup(self, app, selection);
        }

        /**
         * Inserts a spell replacement popup div into the DOM which is used to show and select
         * the spell checker's suggestions for the a misspelled word.
         */
        function insertSpellReplacementPopup() {

            var spellReplacementPopup = $('<div>', { contenteditable: false }).addClass('inline-popup spell-replacement f6-target').css('display', 'none'),
                page = $(self.getNode()).first(),
                // Performance: Delay spelling for specified operations
                spellReplacementPromise = null;

            function handleSpellReplacementPopup() {

                function getSpellErrorWord(editor, selection) {
                    var result = { url: null, spellError: false, word: '', replacements: [] };

                    if (!selection.hasRange()) {
                        // find out a possible URL set for the current position
                        var startPosition = selection.getStartPosition(),
                            obj = Position.getDOMPosition(editor.getNode(), startPosition),
                            characterStyles = editor.getStyleSheets('character'),
                            isParaStart = startPosition[startPosition.length - 1] === 0;

                        if (!isParaStart && obj && obj.node && DOM.isTextSpan(obj.node.parentNode)) {
                            var attributes = characterStyles.getElementAttributes(obj.node.parentNode).character;
                            if (attributes.url.length > 0) {
                                // Now we have to check some edge cases to prevent to show
                                // the popup for a paragraph which contains only an empty span
                                // having set the url attribute.
                                var span = $(obj.node.parentNode);
                                if ((span.text().length > 0) ||
                                    (span.next().length) ||
                                    (DOM.isTextSpan(span.next())))
                                    result.url = attributes.url;
                            }
                            else if ($(obj.node.parentNode).hasClass('spellerror') === true) {
                                // Now we have to check some edge cases to prevent to show
                                // the popup for a paragraph which contains only an empty span
                                // having set the url attribute.
                                var span = $(obj.node.parentNode);
                                if ((span.text().length > 0) ||
                                    (span.prev().length) ||
                                    (DOM.isTextSpan(span.next())))
                                    result.spellError = true;
                                var newSelection = Hyperlink.findSelectionRange(editor, selection),
                                    locale = attributes.language.replace(/-/, '_');
                                result.word = newSelection.text;
                                result.start = newSelection.start;
                                result.end = newSelection.end;
                                result.replacements = editor.getSpellReplacements(result.word, locale);
                            }
                        }
                    }
                    return result;
                }

                function replaceWord(paraPosition, startEndIndex, word, replacement) {
                    var startPosition = _.clone(paraPosition),
                        endPosition = _.clone(paraPosition);
                    startPosition[endPosition.length - 1] = startEndIndex.start;
                    endPosition[endPosition.length - 1] = startEndIndex.end;
                    undoManager.enterUndoGroup(function () {
                        selection.setTextSelection(startPosition, endPosition);
                        self.deleteSelected();  // this selection can never contain unrestorable content -> no deferred required
                        self.insertText(replacement, startPosition, preselectedAttributes);
                        selection.setTextSelection(endPosition);
                    });
                }

                var result = getSpellErrorWord(self, selection);
                //check if it is a wrong word!
                if (result !== null && !result.url && result.spellError && result.replacements && result.replacements.length) {

                    var selectionStartPosition = selection.getStartPosition(),
                        obj = null;

                    obj = Position.getDOMPosition(self.getNode(), selectionStartPosition);
                    if (obj && obj.node && DOM.isTextSpan(obj.node.parentNode)) {
                        var pos = _.last(selectionStartPosition),
                            //startEndPos = Hyperlink.findURLSelection(self, urlSelection, result.url),
                            startEndPos = {start: result.start, end: result.end},
                            left, top, height, width;

                        if (pos !== startEndPos.end) {
                            // find out position of the first span of our selection
                            selectionStartPosition[selectionStartPosition.length - 1] = startEndPos.start;
                            obj = Position.getDOMPosition(self.getNode(), selectionStartPosition, true);
                            left = $(obj.node).offset().left;

                            // find out position of the last span of our selection
                            selectionStartPosition[selectionStartPosition.length - 1] = startEndPos.end - 1;
                            obj = Position.getDOMPosition(self.getNode(), selectionStartPosition, true);
                            top = $(obj.node).offset().top;
                            height = $(obj.node).height();

                            // calculate position relative to the application pane
                            var parent = page,
                                parentLeft = parent.offset().left,
                                parentTop = parent.offset().top,
                                parentWidth = parent.width(),
                                linkClasses = 'link',
                                zoomFactor = app.getView().getZoomFactor();

                            left = (left - parentLeft) / zoomFactor * 100;
                            top = (top - parentTop) / zoomFactor * 100 + height;

                            spellReplacementPopup.children().remove();
                            if (Modernizr.touch) {
                                // some CSS changes for touch devices
                                linkClasses += ' touch';
                            }
                            _(result.replacements).each(function (replacement, id) {
                                if (id > 0) {
                                    spellReplacementPopup.append($('<br>'));
                                }
                                spellReplacementPopup.append(
                                        $('<a>').attr({ href: '#', tabindex: 0 })
                                            .click(function () {
                                                replaceWord(selectionStartPosition, startEndPos, result.word, replacement);
                                                return false;
                                            })
                                            .text(replacement)
                                            .addClass(linkClasses)
                                            );

                            });

                            spellReplacementPopup.css({left: left, top: top, width: ''});
                            width = spellReplacementPopup.width();
                            if ((left + width) > (parentLeft + parentWidth)) {
                                left -= (((left + width) - parentWidth) + parentLeft);
                                left = Math.max(parseInt(page.css('padding-left'), 10), left);
                                if (!_.browser.IE)
                                    width = Math.min(width, parentWidth);
                                spellReplacementPopup.css({left: left, width: width});
                            }
                            if (pos === startEndPos.start || result.beforeHyperlink) {
                                // special case: at the start of a hyperlink we want to
                                // write with normal style
                                self.addPreselectedAttributes(Hyperlink.CLEAR_ATTRIBUTES);
                            }
                            spellReplacementPopup.show();
                            app.getView().scrollToChildNode(spellReplacementPopup);
                        }
                        else {
                            spellReplacementPopup.hide();
                        }
                    }
                }
                else {
                    spellReplacementPopup.hide();
                }

            }  // end of handleSpellReplacementPopup

            if (spellReplacementPopup[0]) {
                spellReplacementPopup.on('keydown', {nextFocus: self.getNode()}, Hyperlink.popupProcessKeydown);

                var found = page.children('.inline-popup.spell-replacement');
                if (!found[0]) {

                    page.append(spellReplacementPopup);

                    self.on('change:editmode', function (event, editMode) {
                        if (!editMode) {
                            spellReplacementPopup.hide();
                        }
                    });

                    selection.on('change', function (event, options) {

                        var spellReplacementDelay = 0;

                        // calling function
                        if ((self.getEditMode()) && (onlineSpelling === true)) {

                            // Performance: Aborting previously started spell replacement processes
                            if (spellReplacementPromise) {
                                spellReplacementPromise.abort();
                            }

                            // Performance: Delay spelling for insert operations, for example insertText
                            spellReplacementDelay = Utils.getBooleanOption(options, 'simpleTextSelection', false) ? 1000 : 0;

                            spellReplacementPromise = app.executeDelayed(function () {
                                handleSpellReplacementPopup();
                            }, { delay: spellReplacementDelay });

                        }
                    });
                }
            }
        }

        /**
         * Checks stored character styles of a document and adds 'missing'
         * character styles (e.g. hyperlink).
         */
        function insertMissingCharacterStyles() {
            var styleNames = characterStyles.getStyleSheetNames(),
                parentId = characterStyles.getDefaultStyleId(),
                hyperlinkMissing = true;

            _(styleNames).find(function (name) {
                var lowerName = name.toLowerCase();
                if (lowerName.indexOf('hyperlink') >= 0) {
                    hyperlinkMissing = false;
                    return true;
                }
                return false;
            });

            if (hyperlinkMissing) {
                var hyperlinkAttr = { character: self.getDefaultLateralHyperlinkAttributes() },
                    hyperlinkDef = self.getDefaultLateralHyperlinkDefinition();
                characterStyles.insertStyleSheet(
                        hyperlinkDef.styleId, hyperlinkDef.styleName,
                        parentId, hyperlinkAttr,
                        { hidden: false, priority: hyperlinkDef.uiPriority, defStyle: false, dirty: true });
            }
        }

        /**
         * Check the stored paragraph styles of a document and adds 'missing'
         * heading / default and other paragraph styles.
         */
        function insertMissingParagraphStyles() {
            var headings = [0, 1, 2, 3, 4, 5],
                styleNames = paragraphStyles.getStyleSheetNames(),
                parentId = paragraphStyles.getDefaultStyleId(),
                hasDefaultStyle = _.isString(parentId) && (parentId.length > 0),
                paragraphListStyleId = null,
                visibleParagraphDefaultStyleId = null,
                hiddenDefaultStyle = false,
                defParaDef;

            if (!hasDefaultStyle) {
                // add a missing default paragraph style
                defParaDef = self.getDefaultParagraphStyleDefinition();
                paragraphStyles.insertStyleSheet(defParaDef.styleId, defParaDef.styleName, null, null,
                        { hidden: false, priority: 1, defStyle: defParaDef['default'], dirty: true });
                parentId = defParaDef.styleId;
                visibleParagraphDefaultStyleId = parentId;
            } else {
                hiddenDefaultStyle = paragraphStyles.isHidden(parentId);
                if (!hiddenDefaultStyle) {
                    visibleParagraphDefaultStyleId = parentId;
                }
            }

            // Search through all paragraph styles to find out what styles
            // are missing.
            _(styleNames).each(function (name, id) {
                var styleAttributes = paragraphStyles.getStyleSheetAttributes(id).paragraph,
                    outlineLvl = styleAttributes.outlineLevel,
                    lowerName = name.toLowerCase(),
                    lowerId = id.toLowerCase();

                if (_.isNumber(outlineLvl) && (outlineLvl >= 0 && outlineLvl < 6)) {
                    headings = _(headings).without(outlineLvl);
                }

                if (!paragraphListStyleId) {
                    if (lowerName.indexOf('list paragraph') === 0) {
                        paragraphListStyleId = id;
                    }
                }

                // Check and store the headings style which is used
                // by odf documents as seed for other heading styles.
                if (hiddenDefaultStyle && !visibleParagraphDefaultStyleId) {
                    if (lowerId === 'standard' || lowerId === 'normal') {
                        visibleParagraphDefaultStyleId = id || parentId;
                    }
                }
            });

            // add the missing paragraph heading styles using predefined values
            if (headings.length > 0) {
                var defaultCharStyles = self.getDefaultHeadingCharacterStyles(),
                    headingsParentId = hiddenDefaultStyle ? visibleParagraphDefaultStyleId : parentId,
                    headingsNextId = hiddenDefaultStyle ? visibleParagraphDefaultStyleId : parentId;

                _(headings).each(function (level) {
                    var attr = {},
                        charAttr = defaultCharStyles[level];
                    attr.character = charAttr;
                    attr.paragraph = { outlineLevel: level, nextStyleId: headingsNextId };
                    paragraphStyles.insertStyleSheet('heading' + (level + 1), 'heading ' + (level + 1),
                            headingsParentId, attr, { hidden: false, priority: 9, defStyle: false, dirty: true });
                });
            }

            // add missing paragraph list style
            if (!paragraphListStyleId) {
                paragraphStyles.insertStyleSheet(
                    'ListParagraph',
                    'List Paragraph', hiddenDefaultStyle ? (visibleParagraphDefaultStyleId || parentId) : parentId,
                    { paragraph: { indentLeft: 1270, contextualSpacing: true, nextStyleId: 'ListParagraph' }},
                    { hidden: false, priority: 34, defStyle: false, dirty: true });
            }
        }

        /**
         * Check the stored table styles of a document and adds a 'missing'
         * default table style. This ensures that we can insert tables that
         * are based on a reasonable default style.
         */
        function insertMissingTableStyles() {

            var styleNames = tableStyles.getStyleSheetNames(),
                parentId = tableStyles.getDefaultStyleId(),
                hasDefaultStyle = _.isString(parentId) && (parentId.length > 0),
                defTableDef = self.getDefaultLateralTableDefinition(),
                defTableAttr = self.getDefaultLateralTableAttributes();

            // inserts the table style sheet described by the passed parameters, if it does not exist yet, and marks it as dirty
            function insertMissingTableStyle(styleId, styleName, uiPriority, attrs) {
                // check by style sheet name (they are always loaded in English from Word, while ste style IDs are localized)
                if (!tableStyles.containsStyleSheetByName(styleName)) {
                    tableStyles.insertStyleSheet(styleId, styleName, null, attrs, { priority: uiPriority, dirty: true });
                }
            }

            // inserts a group of table style definitions for the primary text color and all six accent colors
            function insertMissingTableStyleGroup(baseStyleId, baseStyleName, uiPriority, generateAttributesFunc) {
                insertMissingTableStyle(baseStyleId, baseStyleName, uiPriority, generateAttributesFunc('text1'));
                for (var index = 1; index <= 6; index += 1) {
                    insertMissingTableStyle(baseStyleId + '-Accent' + index, baseStyleName + ' Accent ' + index, uiPriority, generateAttributesFunc('accent' + index));
                }
            }

            if (!hasDefaultStyle) {
                // Add a missing default table style
                tableStyles.insertStyleSheet(defTableDef.styleId, defTableDef.styleName, null, defTableAttr, { priority: 59, defStyle: defTableDef['default'], dirty: true });
            } else {
                // Search for a style defined in the document that can be used for tables
                // If we cannot find it we have to add it.
                var lowestUIPriority = 99,
                    tableStyleId = null;
                _(styleNames).each(function (name, id) {
                    var uiPriority = tableStyles.getUIPriority(id);

                    if (uiPriority && (uiPriority < lowestUIPriority)) {
                        tableStyleId = id;
                        lowestUIPriority = uiPriority;
                    }
                });

                if ((!tableStyleId) || ((tableStyleId === tableStyles.getDefaultStyleId()) && (lowestUIPriority === 99))) {
                    // OOXML uses a default table style which contains no border
                    // definitions. Therfore we add our own default table style
                    // if we only find the default style with uiPriority 99
                    tableStyles.insertStyleSheet(defTableDef.styleId, defTableDef.styleName, parentId, defTableAttr, { priority: 59, dirty: true });
                }
            }

            // adding further table styles, if not already available
            insertMissingTableStyleGroup('LightShading',   'Light Shading',    60, TableStyles.getLightShadingTableStyleAttributes);
            insertMissingTableStyleGroup('MediumShading1', 'Medium Shading 1', 63, TableStyles.getMediumShading1TableStyleAttributes);
            insertMissingTableStyleGroup('MediumShading2', 'Medium Shading 2', 64, TableStyles.getMediumShading2TableStyleAttributes);
            insertMissingTableStyleGroup('MediumGrid1',    'Medium Grid 1',    67, TableStyles.getMediumGrid1TableStyleAttributes);
        }

        // ====================================================================
        // Private functions for the hybrid edit mode
        // ====================================================================

        function processMouseDown(event) {

            var // whether the user has privileges to edit the document
                readOnly = self.getEditMode() !== true,
                // mouse click on a table
                tableCell = $(event.target).closest(DOM.TABLE_CELLNODE_SELECTOR),
                tableCellPoint = null,
                // drawing start and end position for selection
                startPosition = null, endPosition = null,
                // mouse click on a drawing node
                drawingNode = null,
                // mouse click on a tabulator node
                tabNode = null,
                // move handler node for drawings
                moveNode = null,
                // mouse click on a table resize node
                resizerNode = null;

            function isClickableSpellReplacement(element) {
                return element.parent().is('.inline-popup.spell-replacement') && (element.is('a') || (!readOnly && element.is('span.link')));
            }

            // expanding a waiting double click to a triple click
            if (doubleClickEventWaiting) {
                tripleClickActive = true;
                event.preventDefault();
                return;
            }

            activeMouseDownEvent = true;

            // make sure that the clickable parts in the hyperlink popup are
            // processed by the browser
            if (Hyperlink.isClickableNode($(event.target), readOnly)) {
                return;
            } else if (Hyperlink.isPopupOrChildNode($(event.target))) {
                // don't call selection.processBrowserEvent as we don't have
                // a valid position in the document if we process mouse event
                // on the popup
                event.preventDefault();
                return;
            }

            // Fix for 29257 and 29380 and 31623
            if ((_.browser.IE === 11) && ($(event.target).is('table') || DOM.isListLabelNode(event.target) ||
                    (event.target.parentNode && DOM.isListLabelNode(event.target.parentNode)) ||
                    DOM.isPlaceholderNode(event.target) ||
                    (event.target.parentNode && DOM.isPlaceholderNode(event.target.parentNode)) ||
                    (event.target.parentNode && event.target.parentNode.parentNode && DOM.isPlaceholderNode(event.target.parentNode.parentNode)))) {
                event.preventDefault();
                return;
            }

            // Avoiding table grabbers in IE (29409)
            if ((_.browser.IE === 11) && ($(event.target).is('div.cell'))) {
                event.preventDefault();  // preventing default to avoid grabbers, but no 'return'
            }

            if (isClickableSpellReplacement($(event.target))) {
                return;
            }

            if (tableCell.length > 0) {
                // saving initial cell as anchor cell for table cell selection
                // -> this has to work also in read-only mode
                tableCellPoint = DOM.Point.createPointForNode(tableCell);
                selection.setAnchorCellRange(new DOM.Range(tableCellPoint, new DOM.Point(tableCellPoint.node, tableCellPoint.node + 1)));
            }

            // tabulator handler for IE
            tabNode = $(event.target).closest(DOM.TAB_NODE_SELECTOR);
            if ((_.browser.IE) && (tabNode.length > 0)) {
                selection.setTextSelection(Position.getTextLevelOxoPosition(DOM.Point.createPointForNode(tabNode), editdiv, true));
                return;
            }

            // checking for a selection on a drawing node
            drawingNode = $(event.target).closest(DrawingFrame.NODE_SELECTOR);

            // in read only mode allow text selection only (and drawing selection, task 30041)
            if (readOnly && drawingNode.length === 0) {
                selection.processBrowserEvent(event);
                return;
            }

            // checking for a selection on a resize node
            resizerNode = $(event.target).closest(DOM.RESIZE_NODE_SELECTOR);

            if (drawingNode.length > 0) {

                // prevent default click handling of the browser
                event.preventDefault();

                // set focus to the document container (may be located in GUI edit fields)
                app.getView().grabFocus();

                // do nothing if the drawing is already selected (prevent endless
                // recursion when re-triggering the event, to be able to start
                // moving the drawing immediately, see below)
                if (!DrawingFrame.isSelected(drawingNode)) {

                    // select the drawing
                    startPosition = Position.getOxoPosition(editdiv, drawingNode[0], 0);
                    endPosition = Position.increaseLastIndex(startPosition);
                    selection.setTextSelection(startPosition, endPosition);

                    // The following triggering of the event is necessary to move a drawing without a further mousedown
                    // or touchstart event. Without this additional trigger the first mousedown/touchstart would only select
                    // the drawing. For moving it, a further mousedown/touchstart is requrired. By triggering this event
                    // again, it is possible to move a drawing already with the first mousedown/touchstart.
                    // For moving a drawing, a tracker node was created and registered for 'tracking:start' in the
                    // preceeding DrawingResize.drawDrawingSelection() call. This additional mousedown/touchstart
                    // generates a 'tracking:start' event, that will be handled by the tracker node.
                    // The event target for this event in 'processMouseDown' could be the 'img' or the 'div.drawing'. But
                    // it has to be sent to the tracker. Therefore it is necessary to adapt the event target, before
                    // triggering it again.
                    moveNode = drawingNode.find('.tracker');
                    if (moveNode.length > 0) {
                        event.target = moveNode[0];
                        moveNode.trigger(event);
                    }
                }
                return false;

            } else if (resizerNode.length > 0) {

                if (! DOM.isActiveResizeNode(resizerNode)) {
                    // Because the event is triggered again to avoid a second click, it is necessary to mark the resize node
                    // as 'active'. Otherwise the event would be triggered endlessly.

                    // prevent default click handling of the browser
                    event.preventDefault();
                    // set focus to the document container (may be located in GUI edit fields)
                    app.getView().grabFocus();
                    // draw the resizer
                    TableResize.drawTableCellResizeSelection(self, app.getView(), app.getWindowNode(), resizerNode);

                    // send initial mouse down event to the registered handlers
                    // The following triggering of the event is necessary to resize the table without a further mousedown
                    // or touchstart event. Without this additional trigger the first mousedown/touchstart would only select
                    // the resize node. For moving it, a further mousedown/touchstart is requrired. By triggering this event
                    // again, it is possible to move a resize node already with the first mousedown/touchstart.
                    resizerNode.trigger(event);
                }

            } else {

                // clicked somewhere else: calculate logical selection from browser selection,
                // after browser has processed the mouse event
                selection.processBrowserEvent(event);
            }
        }

        function processMouseUp(event) {

            var readOnly = self.getEditMode() !== true,
                drawing = null;

            activeMouseDownEvent = false;

            // make sure that the clickable parts in the hyperlink popup are
            // processed by the browser but we don't call selection.processBrowserEvent
            // as otherwise we have a bad browser selection which produce assertions
            // and have other strange effects (especially for IE)
            if (Hyperlink.isClickableNode($(event.target), readOnly)) {
                return;
            }

            // handle mouse events in edit mode only
            if (readOnly) {
                selection.processBrowserEvent(event);
                return;
            }

            if (selection.getSelectionType() === 'drawing') {
                // mouse up while drawing selected: selection does not change,
                // but scroll drawing completely into the visible area
                drawing = selection.getSelectedDrawing();
                app.getView().scrollToChildNode(drawing);

                if (_.browser.IE < 11 || Utils.IPAD) {
                    // in IE (< 11) processBrowserEvent was not called in mouseDown. Therefore
                    // it is now necessary to call it in mouseup, even if drawings are
                    // selected. (Fix for 29382: Excluding IE 11)

                    // on the iPad when deselecting an image,
                    // processBrowserEvent is called in mouseDown, but applyBrowserSelection is not.
                    // so we need to call processBrowserEvent also in mouseUp to finally call applyBrowserSelection.
                    selection.processBrowserEvent(event);
                } else if (_.browser.IE === 11) {
                    // Avoiding windows frame around the drawing by repainting OX Text drawing frame
                    DrawingFrame.clearSelection(drawing);
                    DrawingResize.drawDrawingSelection(app, drawing);
                }

            } else {
                // calculate logical selection from browser selection, after
                // browser has processed the mouse event
                selection.processBrowserEvent(event);
            }
        }

        function processMouseUpOnDocument(event) {

            if (activeMouseDownEvent) {
                activeMouseDownEvent = false;
                processMouseUp(event);
            }
        }

        // Registering keypress handler at the document for Chrome
        // to avoid problems with direct keypress to document
        // -> This handler is triggered once without event, therefore
        // the check for the event is required.
        function processKeypressOnDocument(event) {

            // Fix for the following scenario:
            // - local client uses Chrome browser
            // - local client selects drawing, without have edit privileges
            // - local client removes the focus from the Chrome browser
            // - remote client removes the selected drawing
            // - local client set browser focus by clicking on top of browser (or side pane)
            // - local client can type a letter direct with keypress into the document
            //   -> the letter is inserted into the document, no operation, no read-only warning
            // After removing the drawing in the remote client, the focus cannot be set correctly
            // in the local client, because the browser does not have the focus. Therefore the focus
            // remains in the clipboard and is then switched to the body element.

            var readOnly = self.getEditMode() !== true;

            // special handling for the following scenario:
            // - there is an event (once this handler is called without event)
            // - the target of the event was the body. So we cannot conflict with events triggered in the side pane, that bubble up to the body.
            // - the document is in read-only mode -> this might be removed in the future, if more than one user is able to edit the document.
            // - this edit application is visible
            // - the event is not triggered a second time -> this event is branded with the property '_triggeredAgain' to avoid endless loop.
            // - the currently focused element is the body element (which is a second test similar to the check of the event target).
            if ((event) && ($(event.target).is('body')) && (readOnly) && (app.isActive()) && (! event._triggeredAgain) && ($(window.document.activeElement).is('body'))) {
                event.preventDefault();
                // Trigger the same event again at the div.page once more (-> read-only warning will be displayed)
                event.target = editdiv[0];
                event._triggeredAgain = true;
                editdiv.trigger(event);
            }
        }

        // Fix for 29751: IE supports double click for word selection
        function processDoubleClick(event) {
            if (DOM.isTextSpan(event.target)) {
                doubleClickEventWaiting = true;
                // Executing the code deferred, so that a triple click becomes possible.
                app.executeDelayed(function () {
                    // modifying the double click to a triple click, if a further mousedown happened
                    if (tripleClickActive) { event.type = 'tripleclick'; }
                    doubleClickEventWaiting = false;
                    tripleClickActive = false;
                    selection.processBrowserEvent(event);
                }, { delay: doubleClickTimeOut });

            }
        }

        function processKeyDown(event) {

            var readOnly = self.getEditMode() !== true,
                isCellSelection = false,
                currentBrowserSelection = null,
                checkPosition = null,
                returnObj = null,
                startPosition,
                endPosition,
                paraLen;

            lastKeyDownEvent = event;   // for some keys we only get keyDown, not keyPressed!

            if (roIOSParagraph) {
                if (self.getEditMode() === true) {
                    $(roIOSParagraph).removeAttr('contenteditable');
                }
                roIOSParagraph = null;
            }

            // Fix for 32910: Client reports internal error if user types as soon as possible after pasting lengthly content with lists
            if (pastingInternalClipboard) {
                event.preventDefault();
                return false;
            }

            // special handling for IME input on IE
            // having a browser selection that spans up a range,
            // the IME input session start event (compositionstart) is omitted.
            // Normally we have to call deleteSelected() to remove the selection within
            // compositionstart. Unfortunately IE is NOT able to work correctly if the
            // browser selection is changed during a IME session. Therefore this code
            // cancels the IME session whenever we detect a selected range to preserve
            // a consistent document state.
            // TODO: We have to find a better solution to support IME and selections.
            if (_.browser.IE && (event.keyCode === KeyCodes.IME_INPUT) && selection.hasRange()) {
                event.preventDefault();
                return false;
            }

            if (DOM.isIgnorableKey(event.keyCode) || isBrowserShortcutKeyEvent(event)) {
                return;
            }

            if (Hyperlink.isClickableNode($(event.target), readOnly)) {
                // We need the keyboard events for the popup which normally contains
                // an external URL which must be opened by the browser.
                return;
            }

            dumpEventObject(event);

            // 'escape' converts a cell selection into a text selection
            if ((event.keyCode === KeyCodes.ESCAPE) && (selection.getSelectionType() === 'cell')) {
                Table.resolveCellSelection(selection, editdiv);
                event.preventDefault();
                return false;
            }

            // resetting values for table cell selection, if no 'shift' is pressed
            if ((selection.getAnchorCellRange() !== null) && (! event.shiftKey)) {
                selection.setAnchorCellRange(null);
            }

            if (DOM.isCursorKey(event.keyCode)) {

                isCellSelection = (selection.getSelectionType() === 'cell');

                if (isCellSelection) {
                    currentBrowserSelection = selection.getBrowserSelection();
                }

                // Setting cursor up and down in tables on MS IE or Webkit
                if ((event.keyCode === KeyCodes.UP_ARROW || event.keyCode === KeyCodes.DOWN_ARROW) && (!event.shiftKey) && (!event.ctrlKey)) {
                    // Setting cursor up and down in tables on MS IE or Webkit
                    if ((_.browser.IE || _.browser.WebKit) && self.isPositionInTable()) {
                        if (Table.updownCursorTravel(event, selection, editdiv, { up: event.keyCode === KeyCodes.UP_ARROW, down: event.keyCode === KeyCodes.DOWN_ARROW, readonly: readOnly })) {
                            return;
                        }
                    }
                    // Setting cursor up and down before and behind exceeded-size tables in Firefox (27395)
                    if (_.browser.Firefox) {
                        if (Table.updownExceededSizeTable(event, selection, editdiv, { up: event.keyCode === KeyCodes.UP_ARROW, down: event.keyCode === KeyCodes.DOWN_ARROW })) {
                            return;
                        }
                    }
                }

                // Changing table cell selections in FireFox when shift is used with arrow keys.
                // Switching from text selection to cell selection already happened in 'selection.moveTextCursor()'.
                if (_.browser.Firefox && DOM.isArrowCursorKey(event.keyCode) && event.shiftKey && isCellSelection && self.isPositionInTable()) {

                    event.preventDefault();

                    if ((event.keyCode === KeyCodes.UP_ARROW) || (event.keyCode === KeyCodes.DOWN_ARROW)) {
                        Table.changeCellSelectionVert(app, selection, currentBrowserSelection, { backwards: event.keyCode === KeyCodes.UP_ARROW });
                    }

                    if ((event.keyCode === KeyCodes.LEFT_ARROW) || (event.keyCode === KeyCodes.RIGHT_ARROW)) {
                        Table.changeCellSelectionHorz(app, selection, currentBrowserSelection, { backwards: event.keyCode === KeyCodes.LEFT_ARROW });
                    }

                    return;
                }

                if (_.browser.IE && DOM.isArrowCursorKey(event.keyCode) && event.shiftKey &&
                    (self.isPositionInTable() || Position.isPositionInTable(editdiv, selection.getEndPosition()) || Position.isPositionInTable(editdiv, selection.getStartPosition()))) {

                    if ((event.keyCode === KeyCodes.LEFT_ARROW) || (event.keyCode === KeyCodes.RIGHT_ARROW)) {
                        if (Table.changeCellSelectionHorzIE(app, editdiv, selection, currentBrowserSelection, { backwards: event.keyCode === KeyCodes.LEFT_ARROW })) {
                            event.preventDefault();
                            return;
                        }
                    }

                }

                // any navigation key: change drawing selection to text selection before
                selection.selectDrawingAsText();

                // let browser process the key event (this may create a drawing selection)
                selection.processBrowserEvent(event, { readonly: readOnly }); // processing is dependent from read-only mode

                return;
            }

            // handle just cursor, copy, search and the global F6 accessibility events if in read only mode
            if (readOnly && !isCopyKeyEvent(event) && !isF6AcessibilityKeyEvent(event) && ! KeyCodes.matchKeyCode(event, 'TAB', { shift: null }) && !KeyCodes.matchKeyCode(event, 'ESCAPE', { shift: null }) && !isSearchKeyEvent(event)) {
                app.rejectEditAttempt();
                event.preventDefault();
                return;
            }

            if (event.keyCode === KeyCodes.DELETE && !event.shiftKey) {

                event.preventDefault();

                // if there is an active 'inputTextPromise', it has to be set to null now, so that
                // following text input is included into a new deferred
                inputTextPromise = null;

                // Executing the code for 'DELETE' deferred. This is necessary because of deferred input of text. Defer time is 'inputTextTimeout'
                app.executeDelayed(function () {

                    startPosition = selection.getStartPosition();

                    if (selection.hasRange()) {
                        self.deleteSelected()
                        .done(function () {
                            selection.setTextSelection(startPosition, null, { simpleTextSelection: true });
                        });
                    } else {
                        endPosition = selection.getEndPosition();

                        // skipping over floated drawings and exceeded size tables
                        returnObj = Position.skipDrawingsAndTables(_.clone(startPosition), _.clone(endPosition), editdiv, { backwards: false });
                        paraLen = returnObj.paraLen;
                        startPosition = returnObj.start;
                        endPosition = returnObj.end;

                        if (startPosition[startPosition.length - 1] < paraLen) {
                            self.deleteRange(startPosition, endPosition, { setTextSelection: false, handleUnrestorableContent: true })
                            .always(function () {
                                selection.setTextSelection(startPosition, null, { simpleTextSelection: true });
                            });

                        } else {
                            var mergeselection = _.clone(startPosition),
                                characterPos = mergeselection.pop(),
                                nextParagraphPosition = _.copy(mergeselection);

                            nextParagraphPosition[nextParagraphPosition.length - 1] += 1;

                            var domPos = Position.getDOMPosition(editdiv, nextParagraphPosition),
                                nextIsTable = false,
                                isLastParagraph = false;

                            if (domPos) {
                                if (DOM.isTableNode(domPos.node)) {
                                    nextIsTable = true;
                                }
                            } else {
                                nextParagraphPosition[nextParagraphPosition.length - 1] -= 1;
                                isLastParagraph = true;
                            }

                            if (DOM.isMergeableParagraph(Position.getParagraphElement(editdiv, mergeselection))) {
                                if (paraLen === 0) {  // Simply remove an empty paragraph
                                    self.deleteRange(_.initial(_.clone(startPosition)));
                                } else {              // Merging two paragraphs
                                    self.mergeParagraph(mergeselection);
                                }
                            }

                            if (nextIsTable) {
                                if (characterPos === 0) {
                                    // removing empty paragraph
                                    var localPos = _.clone(startPosition);
                                    localPos.pop();
                                    self.deleteRange(localPos);
                                    nextParagraphPosition[nextParagraphPosition.length - 1] -= 1;
                                }
                                startPosition = Position.getFirstPositionInParagraph(editdiv, nextParagraphPosition);
                            } else if (isLastParagraph) {
                                // it is necessary, to exchange the paragraph in an empty cell or document with an implicit paragraph
                                checkPosition = _.clone(nextParagraphPosition);
                                checkPosition.push(0);
                                checkImplicitParagraph(checkPosition);

                                if (Position.isPositionInTable(editdiv, nextParagraphPosition)) {

                                    returnObj = Position.getFirstPositionInNextCell(editdiv, nextParagraphPosition);
                                    startPosition = returnObj.position;
                                    var endOfTable = returnObj.endOfTable;
                                    if (endOfTable) {
                                        startPosition[startPosition.length - 1] += 1;
                                        startPosition = Position.getFirstPositionInParagraph(editdiv, startPosition);
                                    }
                                }
                            }

                            selection.setTextSelection(startPosition, null, { simpleTextSelection: true });
                        }
                    }

                }, { delay: inputTextTimeout });

            }
            else if (event.keyCode === KeyCodes.BACKSPACE && !event.altKey) {

                event.preventDefault();

                // if there is an active 'inputTextPromise', it has to be set to null now, so that
                // following text input is included into a new deferred
                inputTextPromise = null;

                // Executing the code for 'BACKSPACE' deferred. This is necessary because of deferred input of text. Defer time is 'inputTextTimeout'
                app.executeDelayed(function () {

                    startPosition = selection.getStartPosition();

                    if (selection.hasRange()) {
                        self.deleteSelected()
                        .done(function () {
                            selection.setTextSelection(startPosition, null, { simpleTextSelection: true });
                        });
                    } else {
                        // skipping over floated drawings and exceeded size tables
                        returnObj = Position.skipDrawingsAndTables(_.clone(startPosition), _.clone(selection.getEndPosition()), editdiv, { backwards: true });
                        if (returnObj.start === null || returnObj.end === null) { // #33154 backspace on table with position [0, 0] and exceeded size limit
                            return;
                        }

                        startPosition = returnObj.start;
                        endPosition = returnObj.end;

                        if (startPosition[startPosition.length - 1] > 0) {

                            startPosition[startPosition.length - 1] -= 1;
                            endPosition[endPosition.length - 1] -= 1;
                            self.deleteRange(startPosition, endPosition, { setTextSelection: false, handleUnrestorableContent: true })
                            .fail(function () {
                                // keeping the current selection
                                startPosition[startPosition.length - 1] += 1;
                            })
                            .always(function () {
                                selection.setTextSelection(startPosition, null, { simpleTextSelection: true });
                            });

                        } else if (startPosition[startPosition.length - 2] >= 0) {

                            var paragraph = Position.getLastNodeFromPositionByNodeName(editdiv, startPosition, DOM.PARAGRAPH_NODE_SELECTOR),
                                listLevel = -1,
                                atParaStart = startPosition[startPosition.length - 1] === 0,
                                styleId = null,
                                elementAttributes = null,
                                styleAttributes = null,
                                paraAttributes = null;

                            if (! _(startPosition).all(function (value) { return (value === 0); })) {

                                startPosition[startPosition.length - 2] -= 1;
                                startPosition.pop();  // -> removing last value from startPosition !

                                var length = Position.getParagraphLength(editdiv, startPosition),
                                    domPos = Position.getDOMPosition(editdiv, startPosition),
                                    prevIsTable = false,
                                    paraBehindTable = null,
                                    fromStyle = false,
                                    prevPara;

                                if ((domPos) && (DOM.isTableNode(domPos.node))) {
                                    prevIsTable = true;
                                }

                                if (atParaStart) {

                                    elementAttributes = paragraphStyles.getElementAttributes(paragraph);
                                    styleId = elementAttributes.styleId;
                                    paraAttributes = elementAttributes.paragraph;
                                    listLevel = paraAttributes.listLevel;
                                    styleAttributes = paragraphStyles.getStyleSheetAttributes(styleId).paragraph;

                                    if (paraAttributes.listStyleId !== '') {
                                        fromStyle = paraAttributes.listStyleId === styleAttributes.listStyleId;
                                    }

                                    if (fromStyle) {
                                        // works in OOX as listStyle 0 is a non-numbering style
                                        // will not work in OpenDocument
                                        self.setAttribute('paragraph', 'listStyleId', 'L0');
                                        return;
                                    } else if (listLevel >= 0) {
                                        implModifyListLevel(listLevel, false, -1);
                                        return;
                                    } else if (styleId === self.getDefaultUIParagraphListStylesheet()) {
                                        self.setAttributes('paragraph', { styleId: self.getDefaultUIParagraphStylesheet() });
                                        // invalidate previous paragraph to recalculate bottom distance
                                        prevPara = Utils.findPreviousNode(editdiv, paragraph, DOM.PARAGRAPH_NODE_SELECTOR);
                                        if (prevPara)
                                            paragraphStyles.updateElementFormatting(prevPara);
                                        return;
                                    }
                                }

                                if (startPosition[startPosition.length - 1] >= 0) {
                                    if (! prevIsTable) {
                                        if (Position.getParagraphLength(editdiv, startPosition) === 0) {
                                            // simply remove first paragraph, if it is empty
                                            self.deleteRange(startPosition);
                                        } else {
                                            // merge two following paragraphs, if both are not empty
                                            self.mergeParagraph(startPosition);
                                        }
                                    } else {
                                        // it is necessary, to exchange an empty final paragraph behind a table with an implicit paragraph
                                        checkPosition = _.clone(startPosition);
                                        checkPosition[checkPosition.length - 1] += 1; // increasing paragraph again
                                        checkPosition.push(0);
                                        checkImplicitParagraph(checkPosition);

                                        // if it is not an implicit paragraph now, it can be removed via operation
                                        paraBehindTable = Position.getParagraphElement(editdiv, checkPosition);
                                        if ((Position.getParagraphNodeLength(paraBehindTable) === 0) && (! DOM.isImplicitParagraphNode(paraBehindTable))) {
                                            // remove paragraph explicitely
                                            self.applyOperations({ name: Operations.DELETE, start: checkPosition });
                                        }
                                    }
                                }

                                if (prevIsTable) {
                                    startPosition = Position.getLastPositionInParagraph(editdiv, startPosition, { ignoreImplicitParagraphs: true });
                                } else {
                                    var isFirstPosition = (startPosition[startPosition.length - 1] < 0) ? true : false;
                                    if (isFirstPosition) {
                                        if (Position.isPositionInTable(editdiv, startPosition)) {
                                            // it is necessary, to exchange the paragraph in an empty cell with an implicit paragraph
                                            checkPosition = _.clone(startPosition);
                                            checkPosition.pop();
                                            checkPosition.push(0, 0);
                                            checkImplicitParagraph(checkPosition);

                                            returnObj = Position.getLastPositionInPrevCell(editdiv, startPosition, { ignoreImplicitParagraphs: true });
                                            startPosition = returnObj.position;
                                            var beginOfTable = returnObj.beginOfTable;
                                            if (beginOfTable) {
                                                startPosition[startPosition.length - 1] -= 1;
                                                startPosition = Position.getLastPositionInParagraph(editdiv, startPosition, { ignoreImplicitParagraphs: true });
                                            }
                                        } else {
                                            startPosition.push(length);
                                        }
                                    } else {
                                        startPosition.push(length);
                                    }
                                }
                            } else {
                                // The position contains only '0's -> Backspace at the beginning of the document -> handle paragraphs with list style.
                                // If no list style is set, checkImplicitParagraph handles the removal of the paragraph.
                                elementAttributes = paragraphStyles.getElementAttributes(paragraph);
                                styleId = elementAttributes.styleId;
                                paraAttributes = elementAttributes.paragraph;
                                listLevel = paraAttributes.listLevel;
                                styleAttributes = paragraphStyles.getStyleSheetAttributes(styleId).paragraph;

                                if ((paraAttributes.listStyleId !== '') && (paraAttributes.listStyleId === styleAttributes.listStyleId)) {
                                    // works in OOX as listStyle 0 is a non-numbering style will not work in OpenDocument
                                    self.setAttribute('paragraph', 'listStyleId', 'L0');
                                } else if (listLevel >= 0) {
                                    implModifyListLevel(listLevel, false, -1);
                                } else if (styleId === self.getDefaultUIParagraphListStylesheet()) {
                                    self.setAttributes('paragraph', { styleId: self.getDefaultUIParagraphStylesheet() });
                                } else {
                                    // Default, if the paragraph is NOT a list paragraph
                                    checkImplicitParagraph(_.clone(startPosition));
                                }
                            }

                            selection.setTextSelection(startPosition, null, { simpleTextSelection: true });
                        }
                    }

                }, { delay: inputTextTimeout });

            }
            else if (Utils.boolXor(event.metaKey, event.ctrlKey) && !event.altKey) {

                // prevent browser from evaluating the key event, but allow cut, copy and paste events
                if (!isPasteKeyEvent(event) && !isCopyKeyEvent(event) && !isCutKeyEvent(event)) {
                    event.preventDefault();
                }

                //self.clearPreselectedAttributes();
            } else if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                event.preventDefault();

                if (self.isPositionInTable()) {

                    // if there is an active 'inputTextPromise', it has to be set to null now, so that
                    // following text input is included into a new deferred
                    inputTextPromise = null;

                    // Executing the code for 'ENTER' deferred. This is necessary because of deferred input of text.
                    // Defer time is 'inputTextTimeout'
                    app.executeDelayed(function () {

                        if (event.shiftKey) {
                            // Jump into first position of previous cell. Do not jump, if there is no previous cell
                            returnObj = Position.getFirstPositionInPreviousCell(editdiv, selection.getStartPosition());
                            if (! returnObj.beginOfTable) {
                                selection.setTextSelection(returnObj.position);
                            }
                        } else {
                            // Jumping into the next table cell or, if this is already the last cell of a table, insert a new row.
                            returnObj = Position.getFirstPositionInNextCell(editdiv, selection.getStartPosition());

                            if (returnObj.endOfTable) {

                                if (readOnly) {
                                    app.rejectEditAttempt();
                                    event.preventDefault();
                                    return;
                                }

                                if (self.isRowAddable()) {
                                    self.insertRow();
                                    selection.setTextSelection(lastOperationEnd);
                                } else {
                                    app.getView().rejectEditTextAttempt('tablesizerow'); // checking table size (26809)
                                    return;
                                }
                            } else {
                                selection.setTextSelection(returnObj.position);
                            }
                        }

                    }, { delay: inputTextTimeout });
                } else {

                    if (readOnly) {
                        app.rejectEditAttempt();
                        event.preventDefault();
                        return;
                    }

                    // (shift)Tab: Change list indent (if in list) when selection is at first position in paragraph
                    var paragraph = Position.getLastNodeFromPositionByNodeName(editdiv, selection.getStartPosition(), DOM.PARAGRAPH_NODE_SELECTOR),
                        mustInsertTab = true;  // always true, independent from shift-key
                    if (!selection.hasRange() &&
                            _.last(selection.getStartPosition()) === Position.getFirstTextNodePositionInParagraph(paragraph)) {
                        var elementAttributes = paragraphStyles.getElementAttributes(paragraph),
                            styleId = elementAttributes.styleId,
                            paraAttributes = elementAttributes.paragraph,
                            listLevel = paraAttributes.listLevel,
                            styleAttributes = paragraphStyles.getStyleSheetAttributes(styleId).paragraph;

                        if (paraAttributes.listStyleId !== '') {
                            mustInsertTab = false;
                            var fromStyle = listLevel === -1 || paraAttributes.listStyleId === styleAttributes.listStyleId;
                            if (listLevel === -1) {
                                listLevel = paraAttributes.listLevel;
                            }
                            if (listLevel !== -1) {

                                if (!fromStyle) {
                                    implModifyListLevel(listLevel, !event.shiftKey, 0);
                                } else {
                                    // numbering via paragraph style (e.g. outline numbering)
                                    var newStyleId = listCollection.findPrevNextStyle(paraAttributes.listStyleId, styleId, event.shiftKey);
                                    if (newStyleId) {
                                        self.setAttributes('paragraph', { styleId: newStyleId });
                                    }
                                }
                            }
                        }
                    }
                    if (mustInsertTab) {
                        // if there is an active 'inputTextPromise', it has to be set to null now, so that
                        // following text input is included into a new deferred
                        inputTextPromise = null;

                        // Executing the code for 'TAB' deferred. This is necessary because of deferred input of text.
                        // Defer time is 'inputTextTimeout'
                        app.executeDelayed(function () {
                            self.insertTab();
                        }, { delay: inputTextTimeout });
                    }
                }
            } else if (event.keyCode === KeyCodes.ESCAPE) {

                // tracking of drawing node or table resizers canceled: do nothing else
                if ($.cancelTracking()) { return; }

                // deselect drawing node before the search bar
                if (self.isDrawingSelected()) {
                    selection.setTextSelection(selection.getEndPosition());
                    return;
                }

                // close hyperlink pop-up menu
                if (Hyperlink.closePopup(self)) { return; }

                // close spelling pop-up menu
                var spellingMenu = editdiv.children('.inline-popup.spell-replacement');
                if (spellingMenu.css('display') !== 'none') {
                    spellingMenu.hide();
                    return;
                }

                // close search& eplace tool bar
                app.getWindow().search.close();
                selection.restoreBrowserSelection();
            }
        }

        function processKeyPressed(event) {

            var // Editor mode
                readOnly = self.getEditMode() !== true,
                // the char element
                c = null,
                hyperlinkSelection = null;

            dumpEventObject(event);

            if (lastKeyDownEvent && DOM.isIgnorableKey(lastKeyDownEvent.keyCode)) {
                return;
            }

            // Fix for 32910: Client reports internal error if user types as soon as possible after pasting lengthly content with lists
            if (pastingInternalClipboard) {
                event.preventDefault();
                return false;
            }

            if (lastKeyDownEvent && DOM.isCursorKey(lastKeyDownEvent.keyCode)) {
                // Fix for 32008: CTRL+POS1/END create special characters on Mac/Chrome
                if (_.browser.WebKit && _.browser.MacOS && _.browser.Chrome && event.ctrlKey) {
                    if ((lastKeyDownEvent.keyCode === 35) || (lastKeyDownEvent.keyCode === 36)) {
                        event.preventDefault();
                    }
                }
                return;
            }

            if (isBrowserShortcutKeyEvent(event)) {
                return;
            }

            if (Hyperlink.isClickableNode($(event.target), readOnly)) {
                // We need the keyboard events for the popup which normally contains
                // an external URL which must be opened by the browser.
                return;
            }

            // handle just cursor, copy, escape, search and the global F6 accessibility events if in read only mode
            if (readOnly && !isCopyKeyEvent(event) && !isF6AcessibilityKeyEvent(event) && ! KeyCodes.matchKeyCode(event, 'TAB', { shift: null }) && !KeyCodes.matchKeyCode(event, 'ESCAPE', { shift: null }) && !isSearchKeyEvent(event)) {
                app.rejectEditAttempt();
                event.preventDefault();
                return;
            }

            // Special behavior for the iPad! Due to strange event order in mobile Safari we have to
            // pass the keyPressed event to the browser. We use the textInput event to cancel the DOM
            // modification instead. The composition system needs the keyPressed event to start a
            // a new composition.
            if (!Utils.IPAD) {
                // prevent browser from evaluating the key event, but allow cut, copy and paste events
                if (!isPasteKeyEvent(event) && !isCopyKeyEvent(event) && !isCutKeyEvent(event)) {
                    event.preventDefault();
                }
            } else {
                // We still get keydown/keypressed events during a composition session in mobile Safari.
                // According to the W3C specification this shouldn't happen. Therefore we have to prevent
                // to call our normal keyPressed code.
                if (imeActive) {
                    return;
                }
            }

            c = getPrintableChar(event);

            // TODO
            // For now (the prototype), only accept single chars, but let the browser process, so we don't need to care about DOM stuff
            // TODO: But we at least need to check if there is a selection!!!

            if ((!event.ctrlKey || (event.ctrlKey && event.altKey && !event.shiftKey)) && !event.metaKey && (c.length === 1)) {

                undoManager.enterUndoGroup(function () {

                    var // the logical start position of the selection
                        startPosition = null,
                        // the string to be inserted with insert text operation
                        insertText = null,
                        // whether the insertTextDef was created
                        createdInsertTextDef = false;

                    // Helper function for collecting and inserting characters. This function is typically executed synchronously,
                    // but in special cases with a selection containing unrestorable content, it is executed deferred.
                    function doInsertCharacter() {

                        // merging input text, if there is already a promise and there are not already 5 characters deferred
                        if ((inputTextPromise) && (activeInputText) && (activeInputText.length < maxTextInputChars)) {
                            // deleting existing deferred for insert text and creating a new deferred
                            // containing the new and the old text
                            inputTextPromise.abort();
                            activeInputText += c;
                        } else {
                            // creating a completely new deferred for insert text
                            inputTextPromise = null;
                            activeInputText = c;
                        }

                        insertText = activeInputText;
                        // Calling insertText deferred. Restarting, if there is further text input in the next 'inputTextTimeout' ms.
                        inputTextPromise = app.executeDelayed(function () {
                            startPosition = selection.getStartPosition();
                            self.insertText(insertText, startPosition, preselectedAttributes);
                            // set cursor behind character
                            selection.setTextSelection(Position.increaseLastIndex(startPosition, insertText.length), null, { simpleTextSelection: true, insertOperation: true });
                            // Setting to null, so that new timeouts can be started.
                            // If there is already a running new inputTextPromise (fast typing), then this is also set to null, so that it cannot
                            // add further characters. But this is not problematic. Example: '123 456' -> After typing space, the inputTextPromise
                            // is set to null. Then it can happen that '4' and '5' are typed, and then this deferred function is executed. In this
                            // case the inputTextPromise for the '45' is terminated, even though there would be sufficient time to add the '6' too.
                            // But this is not important, because this concatenation is only required for faster type feeling.
                            inputTextPromise = null;
                            // Resolving the insertTextDef for the undoGroup. This is very important, because otherwise there could be
                            // pending actions that will never be sent to the server (33226). Every new character in 'processKeyPressed'
                            // needs the same 'insertTextDef', because this is registered for the undoGroup with the return value of
                            // enterUndoGroup. A first character registers the promise of the 'insertTextDef' and a second very fast
                            // following character must use the same (already created) insertTextDef. It is finally resolved
                            // during the execution of this deferred function. What happens, if there is already a new inputTextPromise?
                            // Again the example '123 456'. After typing '123 ' the inputTextPromise is set to null. Then '456' is typed
                            // and collected. Then this deferred function is executed for '123 ', deletes the inputTextPromise (so that
                            // '456' is also completed) and resolves the insertTextDef, so that it will be set to null (via the 'always'
                            // function). This makes also the '456' a complete block with resolved undoGroup. If later this function is
                            // executed for '456', the insertTextDef might already be set to null. But in this case it simply does not
                            // need to be resolved, because this already happened at the previous run of the function with '123 '.
                            // The correct undoGroups are anyhow created by merging following insertText operations (mergeUndoActionHandler).
                            if (insertTextDef) {  // -> the insertTextDef might be already resolved in previous call of this function
                                insertTextDef.resolve();  // resolving the deferred for the undo group
                            }
                        }, { delay: inputTextTimeout });

                        // After a space, always a new text input promise is required -> text will be written.
                        // Setting inputTextPromise to null, makes it unreachable for abort() .
                        if (c === ' ') { inputTextPromise = null; }
                    }

                    // Setting insertTextDef, if it is still null.
                    // Because of the undoGroup this insertTextDef must be global (33226). It cannot be local and be created
                    // in every run inside this 'undoManager.enterUndoGroup'.

                    if (!insertTextDef) {
                        createdInsertTextDef = true;
                        insertTextDef = $.Deferred()
                        .always(function() {
                            insertTextDef = null;
                        });
                    }

                    if (selection.hasRange()) {
                        self.deleteSelected()
                        .done(function () {
                            // inserting the character with taking care of existing selection range
                            // -> this might lead to a deferred character insertion
                            doInsertCharacter();
                        })
                        .fail(function () {
                            if (insertTextDef) {
                                insertTextDef.resolve();
                            }
                        });
                    } else {

                        // check left text to support hyperlink auto correction
                        if ((event.keyCode === KeyCodes.SPACE) || (event.charCode === KeyCodes.SPACE)) {
                            app.executeDelayed(function () {
                                startPosition = selection.getStartPosition();
                                hyperlinkSelection = Hyperlink.checkForHyperlinkText(self, selection.getEnclosingParagraph(), startPosition);

                                if (hyperlinkSelection !== null) {
                                    Hyperlink.insertHyperlink(self,
                                                              hyperlinkSelection.start,
                                                              hyperlinkSelection.end,
                                                              (hyperlinkSelection.url === null) ? hyperlinkSelection.text : hyperlinkSelection.url);
                                    self.addPreselectedAttributes(Hyperlink.CLEAR_ATTRIBUTES);
                                }
                            }, { delay: inputTextTimeout });
                        }

                        // inserting the character without taking care of existing selection range
                        doInsertCharacter();
                    }

                    // returning a promise for the undo group
                    return createdInsertTextDef ? insertTextDef.promise() : undefined;

                }); // enterUndoGroup
            }
            else if (c.length > 1) {
                // TODO?
            }
            else {

                if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null })) {

                    // if there is an active 'inputTextPromise', it has to be set to null now, so that
                    // following text input is included into a new deferred
                    inputTextPromise = null;

                    // 'Enter' key is possible with and without shift-key. With shift-key a hard break is inserted.
                    if (!event.shiftKey) {

                        // Executing the code for 'ENTER' deferred. This is necessary because of deferred input of text.
                        // Defer time is 'inputTextTimeout'
                        app.executeDelayed(function () {

                            var // whether the selection has a range (saving state, because the selection will be removed by deleteSelected())
                                hasSelection = selection.hasRange(),
                                // the logical start position of the selection
                                startPosition = selection.getStartPosition(),
                                // the index of the last value in the logical start position
                                lastValue = startPosition.length - 1,
                                // a new logical position
                                newPosition = _.clone(startPosition),
                                // an object containing the start and end position of a hyperlink
                                hyperlinkSelection = null,
                                // a logical table position
                                localTablePos = null,
                                // whether the current position is the start position in a paragraph
                                isFirstPositionInParagraph = (startPosition[lastValue] === 0),
                                // whether the current position is the start position of the first paragraph in the document or table cell
                                isFirstPositionOfFirstParagraph = (isFirstPositionInParagraph && (startPosition[lastValue - 1] === 0)),
                                // the paragraph element addressed by the passed logical position
                                paragraph = Position.getLastNodeFromPositionByNodeName(editdiv, startPosition, DOM.PARAGRAPH_NODE_SELECTOR),
                                // Performance: Whether a simplified selection can be used
                                isSimpleTextSelection = true, isSplitOperation = true,
                                // a deferred object required for the undo group
                                enterDef = $.Deferred();

                            // Helper function for executing the 'Enter' key event. This function is typically executed synchronously,
                            // but in special cases with a selection containing unrestorable content, it is executed deferred.
                            function doHandleEnterKey() {

                                // check for a possible hyperlink text
                                hyperlinkSelection = Hyperlink.checkForHyperlinkText(self, paragraph, startPosition);
                                // set hyperlink style and url attribute
                                if (hyperlinkSelection !== null) {
                                    Hyperlink.insertHyperlink(self, hyperlinkSelection.start, hyperlinkSelection.end, hyperlinkSelection.url);
                                }

                                if ((isFirstPositionInParagraph) && (isFirstPositionOfFirstParagraph) && (lastValue >= 4) &&
                                    (Position.isPositionInTable(editdiv, [0])) &&
                                    _(startPosition).all(function (value) { return (value === 0); })) {
                                    //at first check if a paragraph has to be inserted before the current table
                                    self.insertParagraph([0]);
                                    // Setting attributes to new paragraph immediately (task 25670)
                                    paragraphStyles.updateElementFormatting(Position.getParagraphElement(editdiv, [0]));
                                    newPosition = [0, 0];
                                } else if ((isFirstPositionInParagraph) && (isFirstPositionOfFirstParagraph) && (lastValue >= 4) &&
                                           (localTablePos = Position.getTableBehindTablePosition(editdiv, startPosition)) || // Inserting an empty paragraph between to tables
                                           (localTablePos = Position.getTableAtCellBeginning(editdiv, startPosition))) { // Inserting an empty paragraph at beginning of table cell
                                    // Inserting an empty paragraph between to tables
                                    self.insertParagraph(localTablePos);
                                    // Setting attributes to new paragraph immediately (task 25670)
                                    paragraphStyles.updateElementFormatting(Position.getParagraphElement(editdiv, localTablePos));
                                    newPosition = Position.appendNewIndex(localTablePos);
                                } else {
                                    // demote or end numbering instead of creating a new paragraph
                                    var listLevel = paragraphStyles.getElementAttributes(paragraph).paragraph.listLevel,
                                        paragraphLength = Position.getParagraphNodeLength(paragraph),
                                        endOfParagraph = paragraphLength === _.last(startPosition),
                                        split = true,
                                        numAutoCorrect = {};

                                    if (!hasSelection && listLevel >= 0 && paragraphLength === 0) {
                                        listLevel--;
                                        undoManager.enterUndoGroup(function () {
                                            if (listLevel < 0) {
                                                //remove list label and update paragraph
                                                $(paragraph).children(DOM.LIST_LABEL_NODE_SELECTOR).remove();
                                                self.setAttributes('paragraph', { styleId: self.getDefaultUIParagraphStylesheet(), paragraph: { listStyleId: null, listLevel: -1 } });
                                                implParagraphChanged(paragraph);
                                            } else {
                                                self.setAttribute('paragraph', 'listLevel', listLevel);
                                            }
                                        });
                                        split = false;
                                    }

                                    if (!hasSelection && listLevel < 0 && paragraphLength > 2) {
                                        // detect Numbering/Bullet labels at paragraph start

                                        if (paragraph !== undefined) {
                                            var paraText = paragraph.textContent;
                                            // Task 30826 -> No automatic list generation, if split happens inside the first '. ' substring
                                            if (_.last(startPosition) > ((paraText.indexOf('. ') + 1))) {
                                                if (((paraText.indexOf('. ') >= 0) && (paragraphLength > 3) && (paraText.indexOf('. ') < (paraText.length - 2))) ||
                                                    ((paraText.indexOf(' ') >= 0) && (paragraphLength > 2) && (paraText.indexOf('* ') === 0 || paraText.indexOf('- ') === 0))) {
                                                    // Fix for 29508: Not detecting '123. ' or ' xmv. ' as list
                                                    // Fix for 29732: Detecting '- ' and '* ' automatically as list
                                                    var labelText = paraText.split(' ')[0];
                                                    numAutoCorrect.listDetection = listCollection.detectListSymbol(labelText);
                                                    if (numAutoCorrect.listDetection.numberFormat !== undefined) {
                                                        numAutoCorrect.startPosition = _.clone(startPosition);
                                                        numAutoCorrect.startPosition[numAutoCorrect.startPosition.length - 1] = 0;
                                                        numAutoCorrect.endPosition = selection.getEndPosition();
                                                        numAutoCorrect.endPosition[numAutoCorrect.endPosition.length - 1] = labelText.length;
                                                    }
                                                }
                                            }
                                        }
                                    }

                                    if (split === true) {
                                        newPosition[lastValue - 1] += 1;
                                        newPosition[lastValue] = 0;

                                        undoManager.enterUndoGroup(function () {

                                            var localParaPos = _.clone(startPosition),
                                                charStyles = characterStyles.getElementAttributes(paragraph.lastChild).character,
                                                listStyleId = null,
                                                attrs = {},
                                                newParagraph = null,
                                                paraAttrs = null,
                                                charAttrs = null,
                                                paraStyleId = null,
                                                isListParagraph = false;

                                            if (paragraphLength > 0 && endOfParagraph && charStyles && charStyles.url) {

                                                // Special handling: Inserting new paragraph after return at end of paragraph after hyperlink (task 30742)
                                                paraAttrs = StyleSheets.getExplicitAttributes(paragraph);
                                                charAttrs = StyleSheets.getExplicitAttributes(paragraph.lastChild);
                                                paraStyleId = StyleSheets.getElementStyleId(paragraph);
                                                isListParagraph = isListStyleParagraph(null, paraAttrs);

                                                if (charAttrs && charAttrs.character && charAttrs.character.url) { delete charAttrs.character.url; }
                                                if (paraStyleId) { attrs.styleId = paraStyleId; }
                                                if (paraAttrs.paragraph && !_.isEmpty(paraAttrs.paragraph)) { attrs.paragraph = paraAttrs.paragraph; }
                                                if (charAttrs.character && !_.isEmpty(charAttrs.character)) { attrs.character = charAttrs.character; }
                                                // use only paragraph attributes in list paragraphs as character attributes, otherwise merge character attributes, task 30794
                                                if (paraAttrs.character && !_.isEmpty(paraAttrs.character)) {
                                                    if (isListParagraph) {
                                                        attrs.character = paraAttrs.character;
                                                    } else {
                                                        attrs.character = _.extend(paraAttrs.character, attrs.character || {});
                                                    }
                                                }

                                                // checking for 'nextStyleId'
                                                styleAttributes = paragraphStyles.getStyleSheetAttributeMap(paraStyleId);
                                                if (styleAttributes.paragraph && styleAttributes.paragraph.nextStyleId) { attrs.styleId = styleAttributes.paragraph.nextStyleId; }

                                                self.applyOperations({ name: Operations.PARA_INSERT, start: newPosition.slice(0, -1), attrs: attrs });

                                                newParagraph = Position.getParagraphElement(editdiv, newPosition.slice(0, -1));
                                                implParagraphChangedSync($(newParagraph));

                                                selection.setTextSelection(newPosition, null, { simpleTextSelection: false, splitOperation: false });
                                                isSplitOperation = false;

                                                // updating lists, if required
                                                if (isListParagraph) {
                                                    if (paraAttrs && paraAttrs.paragraph) {
                                                        listStyleId = paraAttrs.paragraph.listStyleId;
                                                        listLevel = paraAttrs.paragraph.listLevel;
                                                    }
                                                    updateListsDebounced({useSelectedListStyleIDs: true, listStyleId: listStyleId, listLevel: listLevel});
                                                }

                                                // in Internet Explorer it is necessary to add new empty text nodes in paragraphs again
                                                // newly, also in other browsers (new jQuery version?)
                                                repairEmptyTextNodes(newParagraph);

                                            } else {

                                                keepPreselectedAttributes = true;  // preselected attributes must also be valid in following paragraph (26459)
                                                useParagraphCache = true;
                                                paragraphCache = paragraph;  // Performance: Caching the currently used paragraph for temporary usage
                                                handleImplicitParagraph(startPosition);
                                                guiTriggeredOperation = true; // Fix for 30597
                                                self.splitParagraph(startPosition);
                                                guiTriggeredOperation = false;
                                                paragraphCache = null;  // Performance: Deleting the currently used paragraph cache
                                                useParagraphCache = false;

                                                // Special behaviour for splitting an empty paragraph
                                                if (_.last(localParaPos) === 0) { //  || (iPad)) {
                                                    // Fix for 28568: Splitted empty paragraph needs immediately content -> calling validateParagraphNode
                                                    validateParagraphNode(paragraph);
                                                }

                                                // Avoiding cursor jumping after 'Enter' at the end of a paragraph, because new paragraph is empty.
                                                if (paragraphLength > 0 && endOfParagraph) {
                                                    implParagraphChangedSync($(paragraph.nextSibling));
                                                }

                                                // Performance: Saving paragraph and final position, to get faster access to DOM position during setting the cursor after inserting text
                                                selection.setInsertTextInfo(paragraph.nextSibling, 0);

                                                // checking 'nextStyleId' at paragraphs
                                                if (endOfParagraph) {
                                                    var styleId = StyleSheets.getElementStyleId(paragraph),
                                                        styleAttributes = paragraphStyles.getStyleSheetAttributeMap(styleId);

                                                    if (styleAttributes.paragraph && styleAttributes.paragraph.nextStyleId && styleAttributes.paragraph.nextStyleId !== styleId) {
                                                        self.applyOperations({
                                                            name: Operations.SET_ATTRIBUTES,
                                                            start: newPosition.slice(0, -1),
                                                            attrs: { styleId: styleAttributes.paragraph.nextStyleId }
                                                        });
                                                    }
                                                }
                                            }
                                        });

                                        // now apply 'AutoCorrection'
                                        if (numAutoCorrect.listDetection && numAutoCorrect.listDetection.numberFormat !== undefined) {
                                            undoManager.enterUndoGroup(function () {
                                                self.deleteRange(numAutoCorrect.startPosition, numAutoCorrect.endPosition);
                                                self.createList(numAutoCorrect.listDetection.numberFormat === 'bullet' ? 'bullet' : 'numbering', {
                                                    listStartValue: numAutoCorrect.listDetection.listStartValue,
                                                    symbol: numAutoCorrect.listDetection.symbol,
                                                    startPosition: numAutoCorrect.startPosition,
                                                    numberFormat: numAutoCorrect.listDetection.numberFormat
                                                });
                                            });
                                        }

                                    }
                                }

                                selection.setTextSelection(newPosition, null, { simpleTextSelection: isSimpleTextSelection, splitOperation: isSplitOperation });

                                keepPreselectedAttributes = false;

                            }  // function doHandleEnterKey

                            undoManager.enterUndoGroup(function () {

                                if (selection.hasRange()) {
                                    self.deleteSelected()
                                    .done(function () {
                                        doHandleEnterKey();
                                    })
                                    .always(function () {
                                        enterDef.resolve();   // closing the undo group
                                    });
                                    return enterDef.promise();  // for the undo group
                                } else {
                                    doHandleEnterKey();
                                }

                                return $.when();
                            });  // undo group

                        }, { delay: inputTextTimeout });

                    } else if (event.shiftKey) {
                        // insert a hard break
                        self.insertHardBreak();
                    }
                }
            }
        }

        /**
         * Handler for the 'compositionstart' event.
         * The event is dispatched before a text composition system begins a new composition session,
         * and before the DOM is modified due to the composition process.
         *
         * @param event {jQuery.event}
         *  The 'compositionstart' event
         */
        function processCompositionStart() {
            //Utils.log('processCompositionStart');

            var insertSpan;

            // delete current selection first
            self.deleteSelected();

            imeActive = true;
            // determin the span where the IME text will be inserted into
            imeStartPos = selection.getStartPosition();
            insertSpan = $(Position.getLastNodeFromPositionByNodeName(editdiv, imeStartPos, 'span'));

            // if the span is empty webkit browsers replace the content of the paragraph node
            // to avoid this we add a non-breaking space charater here which will be removed in processCompositionEnd()
            // this workaround is not needed for Firefox and IE and would lead to runtime errors
            if (_.browser.WebKit && (insertSpan.text().length === 0)) {
                imeAdditionalChar = true;
                insertSpan.text('\u00a0');
                selection.setBrowserSelection(DOM.Range.createRange(insertSpan, 1, insertSpan, 1));
            } else {
                imeAdditionalChar = false;
            }

            // Special code for iPad. When we detect a compositionstart we have to
            // remove the last char from the activeInputText. Unfortunately we only
            // know now that the char should be processed by the composition system (IME).
            // Therefore we have to remove this char from the active input text.
            if (_.browser.iOS && _.browser.WebKit) {
                // check and abort running input text promise
                if (inputTextPromise) {
                    inputTextPromise.abort();
                }
                // remove latest character from active input text
                if (activeInputText) {
                    activeInputText = activeInputText.slice(0, -1);
                }
            }
        }

        /**
         * Handler for the 'compositionupdate' event.
         * The event is dispatched during a composition session when a text composition system updates its active text passage with a new character.
         *
         * @param event {jQuery.event}
         *  The 'compositionupdate' event
         */
        function processCompositionUpdate() {
            //Utils.log('processCompositionUpdate');
        }

        /**
         * Handler for the 'compositionend' event.
         * The event is dispatched after the text composition system completes or cancels the current composition session
         * (e.g., the IME is closed, minimized, switched out of focus, or otherwise dismissed, and the focus switched back to the user agent).
         *
         * @param event {jQuery.event}
         *  The 'compositionend' event
         */
        function processCompositionEnd(event) {
            //Utils.log('processCompositionEnd');

            var imeText = event.originalEvent.data,
                startPosition = imeStartPos ? _.clone(imeStartPos) : selection.getStartPosition(),
                endPosition,

            compositionEndDeferred = function () {
                //Utils.log('processCompositionEnd - deferred');

                if (imeText.length > 0) {
                    // remove the inserted IME text from the DOM
                    // and also remove the non-breaking space if it has been added in 'processCompositionStart'
                    endPosition = Position.increaseLastIndex(startPosition, imeText.length - (imeAdditionalChar ? 0 : 1));
                    implDeleteText(startPosition, endPosition);

                    // insert the IME text provided by the event data
                    self.insertText(imeText, startPosition, preselectedAttributes);

                    // set the text seleection after the inserted IME text
                    selection.setTextSelection(imeAdditionalChar ? endPosition : Position.increaseLastIndex(endPosition));

                    // during the IME input we prevented calling 'restoreBrowserSelection', so we need to call it now
                    selection.restoreBrowserSelection();
                }
            };

            if (_.browser.WebKit && !Modernizr.touch) {
                // according to the w3c 'http://www.w3.org/TR/DOM-Level-3-Events/#event-type-compositionend'.
                // the 'compositionend' event is dispatched immediately after the text composition system completes the composition session.
                // webkit desktop browsers call 'compositionend' before writing back the IME data to the DOM.
                // therefore we need to defer the deletion of the IME inserted chars and call of insertText operation.
                app.executeDelayed(compositionEndDeferred);
            } else {
                // all other desktop browser and the iPad can process 'compositionend' synchronously.
                compositionEndDeferred();
            }

            imeActive = false;
        }

        function processTextInput(event) {

            var currBrowserSel = null,
                focusPos = null;

            dumpEventObject(event);
            if (Utils.IPAD && !imeActive) {
                var browserSelection = window.getSelection();
                if (!browserSelection.isCollapsed && browserSelection.rangeCount === 1) {
                    // Special code: The prevent default cancels the insertion of a suggested text.
                    // Unfortunately the text is inserted for a short fraction and afterwards replaced
                    // by the old text. Due to this process the cursor position is sometimes incorrectly
                    // set. This code tries to restore the latest old cursor position.
                    currBrowserSel = selection.getBrowserSelection();
                    if (currBrowserSel && currBrowserSel.active && currBrowserSel.active.end) {
                        focusPos = Position.getTextLevelOxoPosition(currBrowserSel.active.end, editdiv, true);
                        app.executeDelayed(function () {
                            selection.setTextSelection(focusPos);
                        });
                    }
                }
                event.preventDefault();
                return false;
            }
        }

        function processInput(event) {
            dumpEventObject(event);
        }

        /**
         * Handler to process dragStart event from the browser.
         *
         * @param {Object} event
         *  The browser event sent via dragStart
         */
        function processDragStart(event) {
            var dataTransfer = event && event.originalEvent && event.originalEvent.dataTransfer,
                 dataTransferOperations = null;

            switch (selection.getSelectionType()) {

            case 'text':
            case 'drawing':
                dataTransferOperations = copyTextSelection();
                break;

            case 'cell':
                dataTransferOperations = copyCellRangeSelection();
                break;

            default:
                Utils.error('Editor.processDragStart(): unsupported selection type: ' + selection.getSelectionType());
            }

            // if browser supports DnD api add data to the event
            if (dataTransfer) {
                // add operation data
                if (!_.browser.IE) {
                    if (dataTransferOperations) {
                        dataTransfer.setData('text/ox-operations', JSON.stringify(dataTransferOperations));
                    }
                    // add plain text and html of the current browser selection
                    dataTransfer.setData('text/plain', selection.getTextFromBrowserSelection());
                    dataTransfer.setData('text/html', Export.getHTMLFromSelection(self));
                } else {
                    // IE just supports 'Text' & Url. Text is more generic
                    dataTransfer.setData('Text', selection.getTextFromBrowserSelection());
                }
            }
        }

        /**
         * Handler to process drop events from the browser.
         *
         * @param {Object} event
         *  The drop event sent by the browser.
         */
        function processDrop(event) {

            event.preventDefault();

            if (self.getEditMode() !== true) {
                return false;
            }

            var files = event.originalEvent.dataTransfer.files;

            if (!files || files.length === 0) {

                // try to find out what type of data has been dropped
                var types = event.originalEvent.dataTransfer.types,
                    detectedDropDataType = null, div = null, url = null, text = null,
                    assuranceLevel = 99, lowerCaseType = null, operations = null;

                if (types && types.length > 0) {

                    _(types).each(function (type) {

                        lowerCaseType = type.toLowerCase();

                        if (lowerCaseType === 'text/ox-operations') {
                            operations = event.originalEvent.dataTransfer.getData(type);
                            // set the operations from the event to be used for the paste
                            clipboardOperations = (operations.length > 0) ? JSON.parse(operations) : [];
                            assuranceLevel = 0;
                            detectedDropDataType = 'operations';
                        }
                        else if (lowerCaseType === 'text/html') {

                            var html = event.originalEvent.dataTransfer.getData(type);
                            if (html && assuranceLevel > 1) {
                                // needs strange replace of zero char created by Firefox
                                div = $('<div>').html(html.replace(/[\x00]/gi, ''));
                                if (div.children().length > 0) {
                                    // Unfortunately we sometimes get broken html from Firefox (D&D from Chrome).
                                    // So it's better to use the plain text part.
                                    assuranceLevel = 1;
                                    detectedDropDataType = 'html';
                                }
                            }
                        } else if (lowerCaseType === 'text/uri-list' ||
                                    lowerCaseType === 'url') {

                            var list = event.originalEvent.dataTransfer.getData(type);
                            if (list && (list.length > 0) && (assuranceLevel > 3)) {
                                assuranceLevel = 3;
                                detectedDropDataType = 'link';
                                url = list;
                            }
                        } else if (lowerCaseType === 'text/x-moz-url') {
                            // FF sometimes (image D&D Chrome->FF) provides only this type
                            // instead of text/uri-list so we are forced to support it, too.
                            var temp = event.originalEvent.dataTransfer.getData(type);
                            // x-moz-url is defined as link + '\n' + caption
                            if (temp && temp.length > 0 && (assuranceLevel > 2)) {
                                var array = temp.split('\n');
                                url = array[0];
                                if (array.length > 1) {
                                    text = array[1];
                                }
                                assuranceLevel = 2;
                                detectedDropDataType = 'link';
                            }
                        }
                        else if (lowerCaseType === 'text/plain' || lowerCaseType === 'text') {

                            var plainText = event.originalEvent.dataTransfer.getData(type);
                            if (plainText && (assuranceLevel > 4)) {
                                assuranceLevel = 4;
                                detectedDropDataType = 'text';
                                text = plainText;
                            }
                        }
                    });
                } else {
                    // IE sometimes don't provide any types but they are accessible getData()
                    // So try to check if we have a Url to check for.
                    url = event.originalEvent.dataTransfer.getData('Url');
                    if (url && url.length > 0) {
                        detectedDropDataType = 'link';
                    }
                }

                if (detectedDropDataType === 'operations') {
                    // use clipboard code to insert operations to document
                    self.pasteInternalClipboard();
                } else if (detectedDropDataType === 'html') {
                    if (div && div.children().length > 0) {
                        // drag&drop detected html
                        var ops = parseClipboard(div);
                        createOperationsFromExternalClipboard(ops);
                    }
                } else if (detectedDropDataType === 'link') {
                    // insert detected hyperlink
                    var setText = text || url;
                    if (setText && setText.length) {
                        if (Image.hasUrlImageExtension(url)) {
                            self.insertImageURL(url);
                        } else {
                            self.insertHyperlinkDirect(url, setText);
                        }
                    }
                } else if (detectedDropDataType === 'text') {
                    if (text && text.length > 0) {
                        insertPlainTextFormatted(text);
                    }
                } else {
                    // fallback try to use 'text' to at least get text
                    text = event.originalEvent.dataTransfer.getData('text');
                    if (text && text.length > 0) {
                        insertPlainTextFormatted(text);
                    }
                }
            } else {
                processDroppedImages(event);
            }

            return false;
        }

        function processDroppedImages(event) {

            var images = event.originalEvent.dataTransfer.files;

            //checks if files were dropped from the browser or the file system
            if (!images || images.length === 0) {
                self.insertImageURL(event.originalEvent.dataTransfer.getData('text'));
                return;
            } else {
                for (var i = 0; i < images.length; i++) {
                    var img = images[i];
                    var imgType = /image.*/;

                    //cancels insertion if the file is not an image
                    if (!img.type.match(imgType)) {
                        continue;
                    }

                    IO.readClientFileAsDataUrl(img).done(_.bind(self.insertImageURL, self));
                }
            }
        }

        /**
         * Uses a plain text string and interprets control characters, e.g.
         * \n, \t to have a formatted input for the editor.
         *
         * @param {String} plainText
         *  The plain text with optional control characters.
         */
        function insertPlainTextFormatted(plainText) {
            var lines, insertParagraph = false, result = [];

            if (_.isString(plainText) && plainText.length > 0) {

                lines = plainText.match(/[^\r\n]+/g);

                _(lines).each(function (line) {
                    if (insertParagraph) {
                        result.push({operation: Operations.PARA_INSERT, depth: 0, listLevel: -1});
                    }
                    result.push({operation: Operations.TEXT_INSERT, data: line, depth: 0});
                    insertParagraph = true;
                });

                createOperationsFromExternalClipboard(result);
            }
        }

        /**
         * Parses the clipboard div for pasted text content
         * @param {jQuery} clipboard
         * @returns {Array} the clipboard data to create operations from
         */
        function parseClipboard(clipboard) {

            var result = [],
                fileId = app.getFileDescriptor().id,
                acceptTextNodes = true;

            (function findTextNodes(current, depth, listLevel, type) {

                var // the length (of chars, tabs, drawings...) to be added to the oxo position
                    insertLength = 0,
                    // the current child node
                    child,
                    // text node content
                    text,
                    // text node content splitted by tab and text
                    splitted,
                    // additional data of the current child node
                    childData,
                    // determins if the node gets parsed recursively or is skipped
                    nextLevel;

                for (var i = 0; i < current.childNodes.length; i++) {
                    child = current.childNodes[i];
                    nextLevel = true;

                    if (child.nodeType === 3 && acceptTextNodes) {
                        // handle non-whitespace characters and non-breaking spaces only
                        // except for our own text portion spans
                        if (DOM.isTextSpan(child.parentNode) || (/\S|\u00a0/.test(child.nodeValue))) {
                            splitted = child.nodeValue.match(/[^\t]+|\t/g) || '';
                            for (var j = 0; j < splitted.length; j++) {
                                if (splitted[j] === '\t') {
                                    // tab
                                    result.push({operation: Operations.TAB_INSERT, depth: depth});
                                    insertLength ++;
                                } else {
                                    // text
                                    // replace '\r' and '\n' with space to fix pastes from aoo
                                    // and replace invisible control characters
                                    text = splitted[j].replace(/[\r\n]/g, ' ').replace(/[\u0000-\u001F]/g, '');
                                    insertLength += text.length;
                                    result.push({operation: Operations.TEXT_INSERT, data: text, depth: depth});
                                }
                            }
                        }
                    } else if ((child.nodeType === 8) && child.nodeValue && (child.nodeValue.toLowerCase() === 'endfragment')) {
                        // special handling for Win 8, do no longer accept text nodes after an <!--EndFragment-->
                        acceptTextNodes = false;

                    } else {
                        // insert paragraph for <div>, heading tags and for
                        //      <br> tags not nested inside a <div>, <p> or <li>
                        //      <p> tags not nested inside a <li>
                        if ($(child).is('div, h1, h2, h3, h4, h5, h6') ||
                            $(child).is('br') && (!$(child.parentNode).is('p, div, li') || $(child.parentNode).hasClass('clipboard')) ||
                            $(child).is('p') && !$(child.parentNode).is('li')) {
                            result.push({operation: Operations.PARA_INSERT, depth: depth, listLevel: listLevel});

                        } else if ($(child).is('p') && $(child.parentNode).is('li') && ($(child.parentNode).children('p').first().get(0) !== child)) {
                            // special handling for pasting lists from Word, the second <p> doesn't have a bullet or numbering
                            // <ol>
                            //   <li>
                            //     <p><span>foo</span></p>      =>   1. foo
                            //     <p><span>bar</span></p>              bar
                            //   </li>
                            // </ol>
                            result.push({operation: 'insertListParagraph', depth: depth, listLevel: listLevel});

                        } else if ($(child).is('img')) {

                            // the img alt tag optionally stores the document media URL, the session id and the file id
                            try {
                                childData = JSON.parse($(child).attr('alt'));
                            } catch (e) {
                                childData = {};
                            }
                            if (ox.session === childData.sessionId && fileId === childData.fileId && childData.altsrc) {
                                // The image URL points to a document and the copy&paste action
                                // takes place inside the same editor instance, so we use relative document media URL.
                                result.push({operation: Operations.DRAWING_INSERT, data: childData.altsrc, depth: depth});

                            } else if (DOM.isDocumentImageNode(child)) {
                                // The image URL points to a document, but copy&paste is done from one editor instance to another
                                // and the image src has not been replaced with a base64 data URL (e.g. IE 9 does not support this).
                                // So the base64 data URL can be generated from the image node as long as the session is valid.
                                result.push({operation: Operations.DRAWING_INSERT,
                                             data: DOM.getBase64FromImageNode(child, Image.getMimeTypeFromImageUri(DOM.getUrlParamFromImageNode(child, 'get_filename'))),
                                             depth: depth});

                            } else {
                                // The image src is an external web URL or already a base64 data URL
                                result.push({operation: Operations.DRAWING_INSERT, data: child.src, depth: depth});

                            }
                            insertLength ++;
                        } else if ($(child).is('style')) {
                            // don't parse <style> elements
                            nextLevel = false;
                        }  else if ($(child).is('a')) {
                            // before we can add the operation for inserting the hyperlink we need to add the insertText operation
                            // so we first parse the next recursion level
                            var len = findTextNodes(child, depth + 1, listLevel, type);
                            // and then add the hyperlink
                            result.push({operation: 'insertHyperlink', data: child.href, length: len, depth: depth});
                            // we already traversed the tree, so don't do it again
                            nextLevel = false;

                        } else if ($(child).is('ol, ul')) {
                            // for list root level create a new array, otherwise add to current
                            if (listLevel === -1) { type = []; }
                            // add current list level type
                            type[listLevel + 1] = $(child).css('list-style-type') || $(child).attr('type') || ($(child).is('ol') ? 'decimal' : 'disc');
                            // insert list entry for current list level
                            result.push({operation: 'insertList', depth: depth, listLevel: listLevel, type: type});
                            // look for deeper list levels
                            insertLength += findTextNodes(child, depth + 1, listLevel + 1, type);
                            // we already traversed the tree, so don't do it again
                            nextLevel = false;

                        } else if ($(child).is('li')) {
                            // insert paragraph and list element for current list level
                            result.push({operation: Operations.PARA_INSERT, depth: depth, listLevel: listLevel});
                            result.push({operation: 'insertListElement', depth: depth, listLevel: listLevel, type: type});
                        }

                        if (nextLevel) {
                            insertLength += findTextNodes(child, depth + 1, listLevel, type);
                        }
                    }
                }

                return insertLength;

            } (clipboard.get(0) /*node*/, 0 /*depth*/, -1 /*listLevel*/, undefined /*type*/));

            return result;
        }

        /**
         * Creates operations from the clipboard data returned by parseClipboard(...)
         * @param {Array} clipboardData
         */
        function createOperationsFromExternalClipboard(clipboardData) {

            var // the operations generator
                generator = self.createOperationsGenerator(),
                // the operation start position
                start = selection.getStartPosition(),
                // the operation end position
                end = _.clone(start),
                // indicates if the previous operation was insertHyperlink
                hyperLinkInserted = false,
                // used to cancel operation preparing in processArrayDelayed
                cancelled = false,
                // the current paragraph style sheet id
                styleId,
                // the next free list style number part
                listStyleNumber = 1,
                // the default paragraph list style sheet id
                listParaStyleId = null,
                // the list stylesheet id
                listStyleId = null,
                // indicates if previous operations inserted a list
                listInserted = false;


            // the operation after insertHyperlink needs to remove the hyperlink style again
            function removeHyperLinkStyle(start, end, nextEntry) {
                if (hyperLinkInserted && (!nextEntry || nextEntry.operation !== 'insertHyperlink')) {
                    // // generate the 'setAttributes' operation to remove the hyperlink style
                    generator.generateOperation(Operations.SET_ATTRIBUTES, {
                        attrs: Hyperlink.CLEAR_ATTRIBUTES,
                        start: _.clone(start),
                        end: _.clone(end)
                    });
                    hyperLinkInserted = false;
                }
            }


            // handle implicit paragraph
            function doHandleImplicitParagraph(start) {
                var position = _.clone(start),
                    paragraph;

                if (position.pop() === 0) {  // is this an empty paragraph?
                    paragraph = Position.getParagraphElement(editdiv, position);
                    if ((paragraph) && (DOM.isImplicitParagraphNode(paragraph)) && (Position.getParagraphNodeLength(paragraph) === 0)) {
                        // creating new paragraph explicitely
                        generator.generateOperation(Operations.PARA_INSERT, {
                            start: position
                        });
                    }
                }
            }


            // the list collection is only aware of the list styles that have already been applied by an 'insertList' operation,
            // so we need to handle the next free list style id generation ourselfs.
            function getFreeListStyleId() {
                var sFreeId = 'L';

                while (listCollection.hasListStyleId('L' + listStyleNumber)) {
                    listStyleNumber++;
                }
                sFreeId += listStyleNumber;
                listStyleNumber++;
                return sFreeId;
            }


            // parse entry and generate operations
            function generateOperationCallback(chunk, index, dataArray) {

                var def = null,
                    entry = chunk[0],
                    lastPos,
                    position,
                    attributes,
                    listOperation = null,
                    hyperlinkStyleId,
                    styleAttributes;


                // next operation's start is previous operation's end
                start = _.clone(end);
                end = _.clone(end);
                lastPos = start.length - 1;

                // cancel the clipboard paste if we loose the edit rigths or if the 'cancel' button is clicked.
                if (cancelled || !self.getEditMode()) { return Utils.BREAK; }

                switch (entry.operation) {

                case Operations.PARA_INSERT:
                    // generate the 'insertParagraph' operation
                    generator.generateOperation(Operations.PARA_SPLIT, {
                        start: _.clone(start)
                    });

                    end[lastPos - 1] ++;
                    end[lastPos] = 0;


                    // set attributes only if it's not a list paragraph, these are handled in 'insertListElement'
                    if (entry.listLevel === -1) {
                        // use the nextStyleId if the current paragraph style defines one
                        styleAttributes = paragraphStyles.getStyleSheetAttributeMap(styleId, 'paragraph');
                        styleId = styleAttributes.paragraph && styleAttributes.paragraph.nextStyleId;

                        if (listInserted) {
                            // remove list style
                            if ((!_.isString(styleId)) || (styleId === self.getDefaultUIParagraphListStylesheet())) {
                                styleId = self.getDefaultUIParagraphStylesheet();
                            }
                            attributes = { styleId: styleId, paragraph: { listStyleId: null, listLevel: -1 } };
                            listInserted = false;

                        } else if (_.isString(styleId)) {
                            attributes = { styleId: styleId };
                        }

                        if (_.isString(styleId)) {
                            // generate operation to insert a dirty paragraph style into the document
                            generator.generateMissingStyleSheetOperation('paragraph', styleId);

                            // generate the 'setAttributes' operation
                            generator.generateOperation(Operations.SET_ATTRIBUTES, {
                                start: end.slice(0, -1),
                                attrs: attributes
                            });
                        }
                    }
                    break;


                case Operations.TEXT_INSERT:
                    // generate the 'insertText' operation
                    generator.generateOperation(Operations.TEXT_INSERT, {
                        text: entry.data,
                        start: _.clone(start)
                    });

                    end[lastPos] += entry.data.length;
                    removeHyperLinkStyle(start, end, dataArray[index + 1]);
                    break;


                case Operations.TAB_INSERT:
                    // generate the 'insertTab' operation
                    generator.generateOperation(Operations.TAB_INSERT,
                        _.isObject(preselectedAttributes) ? { start: _.clone(start), attrs: preselectedAttributes } : { start: _.clone(start) }
                    );

                    end[lastPos] ++;
                    removeHyperLinkStyle(start, end, dataArray[index + 1]);
                    break;


                case Operations.DRAWING_INSERT:
                    if ((!_.isString(entry.data)) || (entry.data.length < 1)) { break; }
                    // check if we got a webkit fake URL instead of a data URI,
                    // there's currently no way to access the image data of a webkit fake URL
                    // and check for local file url
                    if ((entry.data.substring(0, 15) === 'webkit-fake-url') || (entry.data.substring(0, 7) === 'file://')) {
                        app.rejectEditAttempt('image');
                        break;
                    }

                    def = $.Deferred();

                    getImageSize(entry.data).then(function (size) {
                        // exit silently if we lost the edit rights
                        if (!self.getEditMode()) {  return $.when(); }

                        // check for base64 image data or image url
                        attributes = (entry.data.substring(0, 10) === 'data:image') ? { imageData: entry.data } : { imageUrl: entry.data };

                        // generate the 'insertDrawing' operation
                        generator.generateOperation(Operations.DRAWING_INSERT, {
                            start: _.clone(start),
                            type: 'image',
                            attrs: { drawing: _.extend(attributes, size, DEFAULT_DRAWING_MARGINS) }
                        });

                        return $.when();
                    })
                    .then(function () {
                        end[lastPos] ++;
                        removeHyperLinkStyle(start, end, dataArray[index + 1]);
                    }, function () {
                        app.rejectEditAttempt('image');
                    })
                    .always(function () {
                        // always resolve to continue processing paste operations
                        def.resolve();
                    });
                    break;


                case 'insertHyperlink':
                    if (entry.data && _.isNumber(entry.length)) {
                        hyperlinkStyleId = self.getDefaultUIHyperlinkStylesheet();

                        // generate operation to insert a dirty character style into the document
                        if (_.isString(hyperlinkStyleId)) {
                            generator.generateMissingStyleSheetOperation('character', hyperlinkStyleId);
                        }

                        // the text for the hyperlink has already been inserted and the start position is right after this text,
                        // so the start for the hyperlink attribute is the start position minus the text length
                        position = _.clone(start);
                        position[lastPos] -= entry.length;
                        if (position[lastPos] < 0) {
                            position[lastPos] = 0;
                        }

                        // generate the 'insertHyperlink' operation
                        generator.generateOperation(Operations.SET_ATTRIBUTES, {
                            attrs: { styleId: hyperlinkStyleId, character: { url: entry.data } },
                            start: position,
                            end: _.clone(end)
                        });

                        hyperLinkInserted = true;
                    }
                    break;


                case 'insertList':
                    // for the list root level insert list style with all style levels
                    if (entry.listLevel === -1) {
                        listOperation = listCollection.getListOperationFromHtmlListTypes(entry.type);
                        listStyleId = getFreeListStyleId();

                        // generate the 'insertList' operation
                        generator.generateOperation(Operations.INSERT_LIST, {
                            listStyleId: listStyleId,
                            listDefinition: listOperation.listDefinition
                        });
                    }
                    break;


                case 'insertListElement':
                    if (listStyleId && _.isNumber(entry.listLevel)) {
                        listParaStyleId = this.getDefaultUIParagraphListStylesheet();

                        // generate operation to insert a dirty paragraph style into the document
                        if (_.isString(listParaStyleId)) {
                            generator.generateMissingStyleSheetOperation('paragraph', listParaStyleId);
                        }

                        // generate the 'setAttributes' operation
                        generator.generateOperation(Operations.SET_ATTRIBUTES, {
                            start: start.slice(0, -1),
                            attrs: { styleId: listParaStyleId, paragraph: { listStyleId: listStyleId, listLevel: entry.listLevel } }
                        });

                        listInserted = true;
                    }
                    break;

                // needed for additional paragraph inside a list element, from the second paragraph on they have no bullet or numberring
                case 'insertListParagraph':
                    // generate the 'insertParagraph' operation
                    generator.generateOperation(Operations.PARA_SPLIT, {
                        start: _.clone(start)
                    });

                    end[lastPos - 1] ++;
                    end[lastPos] = 0;

                    // set attributes only if it's a list paragraph
                    if (listInserted && _.isString(listParaStyleId) && entry.listLevel > -1) {

                        // generate the 'setAttributes' operation
                        generator.generateOperation(Operations.SET_ATTRIBUTES, {
                            start: end.slice(0, -1),
                            attrs: { styleId: listParaStyleId, paragraph: { listStyleId: null, listLevel: -1 } }
                        });
                    }
                    break;


                default:
                    Utils.log('createOperationsFromExternalClipboard(...) - unhandled operation: ' + entry.operation);
                    break;
                }

                return def;
            }

            // make sure that only one paste call is processed at the same time
            if (checkSetClipboardPasteInProgress()) { return $.when(); }

            // to paste at the cursor position don't create a paragraph as first operation
            if (clipboardData.length > 1 && clipboardData[0].operation === Operations.PARA_INSERT) {
                clipboardData.shift();
            } else if (clipboardData.length > 2 && clipboardData[0].operation === 'insertList' && clipboardData[1].operation === Operations.PARA_INSERT) {
                clipboardData.splice(1, 1);
            }

            // init the paragraph style sheet id
            styleId = StyleSheets.getElementStyleId(Position.getLastNodeFromPositionByNodeName(editdiv, start, DOM.PARAGRAPH_NODE_SELECTOR));

            // init the next free list style number
            listStyleNumber = parseInt(listCollection.getFreeListId().slice(1, listCollection.getFreeListId().length), 10);
            if (!_.isNumber(listStyleNumber)) {
                listStyleNumber = 1;
            }

            // create operation to replace an implicit pragraph with a real one
            doHandleImplicitParagraph(start);


            return undoManager.enterUndoGroup(function () {

                var // the deferred to keep the undo group open until it is resolved or rejected
                    undoDef = $.Deferred(),
                    // the apply actions deferred
                    applyDef,
                    // the generate operations deferred
                    generateDef,
                    // the generated operations
                    operations = null,
                    // to save performance handle busy with every 50th progress call only
                    progressInterval = 50,
                    // the current progress count
                    progressCounter = 0;


                // show a nice message with cancel button
                app.getView().enterBusy({
                    initHandler: function (header, footer) {
                        // add a banner for large clipboard pastes
                        footer.append(
                            $('<div>').addClass('size-warning-node').append(
                                $('<div>').addClass('alert alert-warning').append(
                                    $('<div>').text(gt('Sorry, pasting from clipboard will take some time.'))
                                )
                            )
                        );
                    },
                    cancelHandler: function () {
                        cancelled = true;

                        if (applyDef && applyDef.abort) {
                            applyDef.abort();
                        }
                    },
                    delay: 500 // fade in busy blocker after a short delay
                });

                // generate operations
                generateDef = app.processArrayDelayed(generateOperationCallback, clipboardData, { context: self, chunkLength: 1, delay: 0 })
                .then(function () {
                    // cancel the clipboard paste if we loose the edit rigths or if the 'cancel' button is clicked.
                    if (cancelled || !self.getEditMode()) { return $.Deferred().reject(); }

                    // delete current selection
                    return self.deleteSelected();
                })
                .then(function () {
                    // cancel the clipboard paste if we loose the edit rigths or if the 'cancel' button is clicked.
                    if (cancelled || !self.getEditMode()) { return $.Deferred().reject(); }

                    // reduce interval and raise chunk size to reflect applyActions using a chunk size of 50
                    progressInterval = 1;
                    operations = generator.getOperations();

                    // apply generated operations
                    applyDef = self.applyActions({ operations: operations }, { async: true });
                    return applyDef;
                })
                .then(function () {
                    selection.setTextSelection(end);
                    return $.when();
                })
                .always(function () {
                    // leave busy state
                    app.getView().leaveBusy().grabFocus();
                    // close undo group
                    undoDef.resolve();
                });

                // add progress handling
                // at first we get progress data from processArrayDelayed(), using a chunk size of 1
                // after that we get progress data from applyActions(), using a chunk size of 50
                generateDef.progress(function (partialProgress) {
                    // combining progress informaions
                    var progress = (operations) ? (0.5 + (partialProgress * 0.5)) : (partialProgress * 0.5);

                    // update progress bar every time for applyActions() and at every progressInterval for processArrayDelayed()
                    if (operations || (progressCounter % progressInterval === 0)) {
                        app.getWindow().busy(progress);
                    }
                    progressCounter++;
                });

                return undoDef.promise();

            }).always(function () {
                setClipboardPasteInProgress(false);
            }); // enterUndoGroup()
        }


        // ==================================================================
        // Private functions
        // ==================================================================

        this.registerOperationHandler(Operations.DELETE, function (operation) {

            var // node info about the paragraph to be deleted
                nodeInfo = null,
                // attribute type of the start node
                type = null,
                // generator for the undo/redo operations
                generator = undoManager.isUndoEnabled() ? this.createOperationsGenerator() : null;

            // undo/redo generation
            if (generator) {

                nodeInfo = Position.getDOMPosition(editdiv, operation.start, true);
                type = resolveElementType(nodeInfo.node);

                switch (type) {

                case 'text':
                    var position = operation.start.slice(0, -1),
                        paragraph = Position.getCurrentParagraph(editdiv, position),
                        start = operation.start[operation.start.length - 1],
                        end = _.isArray(operation.end) ? operation.end[operation.end.length - 1] : start;

                    generator.generateParagraphChildOperations(paragraph, position, { start: start, end: end, clear: true });
                    undoManager.addUndo(generator.getOperations(), operation);
                    break;

                case 'paragraph':
                    if (!DOM.isImplicitParagraphNode(nodeInfo.node)) {
                        generator.generateParagraphOperations(nodeInfo.node, operation.start);
                        undoManager.addUndo(generator.getOperations(), operation);
                    }
                    break;

                case 'cell':
                    generator.generateTableCellOperations(nodeInfo.node, operation.start);
                    undoManager.addUndo(generator.getOperations(), operation);
                    break;

                case 'row':
                    generator.generateTableRowOperations(nodeInfo.node, operation.start);
                    undoManager.addUndo(generator.getOperations(), operation);
                    break;

                case 'table':
                    if (!DOM.isExceededSizeTableNode(nodeInfo.node)) {
                        generator.generateTableOperations(nodeInfo.node, operation.start); // generate undo operations for the entire table
                        undoManager.addUndo(generator.getOperations(), operation);
                    }
                    if (DOM.isTableInTableNode(nodeInfo.node)) {
                        if (($(nodeInfo.node).next().length > 0) && (DOM.isImplicitParagraphNode($(nodeInfo.node).next()))) {
                            $(nodeInfo.node).next().css('height', '');  // making paragraph behind the table visible after removing the table
                        }
                    }
                    break;
                }
            }

            // finally calling the implementation function to delete the content
            return implDelete(operation.start, operation.end);
        });

        this.registerOperationHandler(Operations.MOVE, function (operation) {
            if (undoManager.isUndoEnabled()) {
                // Todo: Ignoring 'end', only 'start' === 'end' is supported
                var undoOperation = { name: Operations.MOVE, start: operation.to, end: operation.to, to: operation.start };
                undoManager.addUndo(undoOperation, operation);
            }
            return implMove(operation.start, operation.end, operation.to);
        });

        this.registerOperationHandler(Operations.TEXT_INSERT, function (operation) {
            if (implInsertText(operation.start, operation.text, operation.attrs)) {
                if (undoManager.isUndoEnabled()) {
                    var end = Position.increaseLastIndex(operation.start, operation.text.length - 1),
                        undoOperation = { name: Operations.DELETE, start: operation.start, end: end };
                    undoManager.addUndo(undoOperation, operation);
                }
                return true;
            } else {
                return false;
            }
        });

        this.registerOperationHandler(Operations.FIELD_INSERT, function (operation) {
            if (implInsertField(operation.start, operation.type, operation.representation, operation.attrs)) {
                if (undoManager.isUndoEnabled()) {
                    var undoOperation = { name: Operations.DELETE, start: operation.start };
                    undoManager.addUndo(undoOperation, operation);
                }
                return true;
            } else {
                return false;
            }
        });

        this.registerOperationHandler(Operations.TAB_INSERT, function (operation) {
            if (implInsertTab(operation.start, operation.attrs)) {
                if (undoManager.isUndoEnabled()) {
                    var undoOperation = { name: Operations.DELETE, start: operation.start };
                    undoManager.addUndo(undoOperation, operation);
                }
                return true;
            } else {
                return false;
            }
        });

        this.registerOperationHandler(Operations.HARDBREAK_INSERT, function (operation) {
            if (implInsertHardBreak(operation.start, operation.attrs)) {
                if (undoManager.isUndoEnabled()) {
                    var undoOperation = { name: Operations.DELETE, start: operation.start };
                    undoManager.addUndo(undoOperation, operation);
                }
                return true;
            } else {
                return false;
            }
        });

        this.registerOperationHandler(Operations.DRAWING_INSERT, function (operation) {
            if (implInsertDrawing(operation.type, operation.start, operation.attrs)) {
                if (undoManager.isUndoEnabled()) {
                    var undoOperation = { name: Operations.DELETE, start: operation.start };
                    undoManager.addUndo(undoOperation, operation);
                }
                return true;
            } else {
                return false;
            }
        });

        this.registerOperationHandler(Operations.INSERT_LIST, function (operation) {
            undoManager.addUndo({ name: Operations.DELETE_LIST, listStyleId: operation.listStyleId }, operation);
            listCollection.insertList(operation);
        });

        this.registerOperationHandler(Operations.DELETE_LIST, function (operation) {
            // no Undo, cannot be removed by UI
            listCollection.deleteList(operation.listStyleId);
        });

        this.registerOperationHandler(Operations.SET_ATTRIBUTES, function (operation) {
            // undo/redo generation is done inside implSetAttributes()
            return implSetAttributes(operation.start, operation.end, operation.attrs);
        });

        this.registerOperationHandler(Operations.PARA_INSERT, function (operation) {

            var // the new paragraph
                paragraph = DOM.createParagraphNode(),
                // insert the paragraph into the DOM tree
                inserted = insertContentNode(_.clone(operation.start), paragraph),
                // text position at the beginning of the paragraph
                startPosition = null,
                // Performance: Saving data for list updates
                paraAttrs = null, listStyleId = null, listLevel = null,
                //page breaks
                currentElement = Position.getContentNodeElement(editdiv, operation.start);

            // insertContentNode() writes warning to console
            if (!inserted) { return false; }

            // insert required helper nodes
            validateParagraphNode(paragraph);

            // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
            // newly, also in other browsers (new jQuery version?)
            repairEmptyTextNodes(paragraph);

            startPosition = Position.appendNewIndex(operation.start);

            // removing a following implicit paragraph (for example in a table cell)
            // -> exchanging an implicit paragraph with a non-implicit paragraph
            if (DOM.isImplicitParagraphNode(paragraph.next())) {
                paragraph.next().remove();
            }

            // generate undo/redo operations
            if (undoManager.isUndoEnabled()) {
                undoManager.addUndo({ name: Operations.DELETE, start: operation.start }, operation);
            }

            // apply the passed paragraph attributes
            if (_.isObject(operation.attrs)) {
                paragraphStyles.setElementAttributes(paragraph, operation.attrs);
            }

            // set cursor to beginning of the new paragraph
            // But not, if this was triggered by an insertColumn or insertRow operation (Task 30859)
            if (!guiTriggeredOperation) {
                lastOperationEnd = startPosition;
            }

            // register paragraph for deferred formatting, especially empty paragraphs
            // or update them immediately after the document import has finished (Task 28370)
            if (app.isImportFinished()) {
                implParagraphChangedSync($(paragraph));
            } else {
                implParagraphChanged(paragraph);
            }

            // checking paragraph attributes for list styles
            paraAttrs = StyleSheets.getExplicitAttributes(paragraph);

            // updating lists, if required
            if (isListStyleParagraph(null, paraAttrs)) {
                if (paraAttrs && paraAttrs.paragraph) {
                    listStyleId = paraAttrs.paragraph.listStyleId;
                    listLevel = paraAttrs.paragraph.listLevel;
                    // mark this paragraph for later list update. This is necessary, because it does not yet contain
                    // the list label node (DOM.LIST_LABEL_NODE_SELECTOR) and is therefore ignored in updateLists.
                    $(paragraph).data('updateList', 'true');
                }

                updateListsDebounced({useSelectedListStyleIDs: true, paraInsert: true, listStyleId: listStyleId, listLevel: listLevel});
            }

            //render pagebreaks after insert
            insertPageBreaksDebounced(currentElement);

            return true;
        });

        this.registerOperationHandler(Operations.PARA_SPLIT, function (operation) {
            if (undoManager.isUndoEnabled()) {
                var paragraphPos = operation.start.slice(0, -1);
                var undoOperation = { name: Operations.PARA_MERGE, start: paragraphPos };
                undoManager.addUndo(undoOperation, operation);
            }
            return implSplitParagraph(operation.start);
        });

        this.registerOperationHandler(Operations.PARA_MERGE, function (operation) {

            var // the paragraph that will be merged with its next sibling
                paragraphInfo = Position.getDOMPosition(editdiv, operation.start, true),
                // current and next paragraph, as jQuery objects
                thisParagraph = null, nextParagraph = null,
                // text position at end of current paragraph, logical position of next paragraph
                paraEndPosition = null, nextParaPosition = null,
                // first child node of next paragraph
                firstChildNode = null,
                // the undo/redo operations
                generator = undoManager.isUndoEnabled() ? this.createOperationsGenerator() : null,
                // Performance: Saving data for list updates
                paraAttrs = null, listStyleId = null, listLevel = null,
                currentElement = Position.getContentNodeElement(editdiv, operation.start);

            // get current paragraph
            if (!paragraphInfo || !DOM.isParagraphNode(paragraphInfo.node)) {
                Utils.warn('Editor.mergeParagraph(): no paragraph found at position ' + JSON.stringify(operation.start));
                return false;
            }
            thisParagraph = $(paragraphInfo.node);
            paraEndPosition = Position.appendNewIndex(operation.start, Position.getParagraphLength(editdiv, operation.start));

            // get next paragraph
            nextParagraph = thisParagraph.next();
            // but make sure next paragraph is only valid node, and not pagebreak for example
            while (nextParagraph.length > 0 && !nextParagraph.is(DOM.CONTENT_NODE_SELECTOR)) {
                nextParagraph = nextParagraph.next();
            }

            if (!DOM.isParagraphNode(nextParagraph)) {
                Utils.warn('Editor.mergeParagraph(): no paragraph found after position ' + JSON.stringify(operation.start));
                return false;  // forcing an error, if there is no following paragraph
            }
            nextParaPosition = Position.increaseLastIndex(operation.start);

            // generate undo/redo operations
            if (generator) {
                generator.generateOperation(Operations.PARA_SPLIT, { start: paraEndPosition });
                generator.generateSetAttributesOperation(nextParagraph, { start: nextParaPosition }, { clearFamily: 'paragraph' });
                undoManager.addUndo(generator.getOperations(), operation);
            }

            // remove dummy text node from current paragraph
            if (DOM.isDummyTextNode(thisParagraph[0].lastChild)) {
                $(thisParagraph[0].lastChild).remove();
            }

            // remove list label node from next paragraph
            nextParagraph.children(DOM.LIST_LABEL_NODE_SELECTOR).remove();

            // append all children of the next paragraph to the current paragraph, delete the next paragraph
            firstChildNode = nextParagraph[0].firstChild;
            thisParagraph.append(nextParagraph.children());
            nextParagraph.remove();

            // remove one of the sibling text spans at the concatenation point,
            // if one is empty; otherwise try to merge equally formatted text spans
            if (DOM.isTextSpan(firstChildNode) && DOM.isTextSpan(firstChildNode.previousSibling)) {
                if (DOM.isEmptySpan(firstChildNode)) {
                    $(firstChildNode).remove();
                } else if (DOM.isEmptySpan(firstChildNode.previousSibling)) {
                    $(firstChildNode.previousSibling).remove();
                } else {
                    CharacterStyles.mergeSiblingTextSpans(firstChildNode);
                }
            }

            // refresh DOM
            implParagraphChanged(thisParagraph);

            // checking paragraph attributes for list styles
            paraAttrs = StyleSheets.getExplicitAttributes(thisParagraph);

            // updating lists, if required
            if (isListStyleParagraph(null, paraAttrs)) {
                if (paraAttrs && paraAttrs.paragraph) {
                    listStyleId = paraAttrs.paragraph.listStyleId;
                    listLevel = paraAttrs.paragraph.listLevel;
                }
                updateListsDebounced({useSelectedListStyleIDs: true, listStyleId: listStyleId, listLevel: listLevel});
            }

            // new cursor position at merge position
            lastOperationEnd = _.clone(paraEndPosition);

            if ($(currentElement).data('lineBreaksData')) {
                $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
            }
            //render pagebreaks after merge
            insertPageBreaksDebounced(currentElement);

            return true;
        });

        this.registerOperationHandler(Operations.TABLE_INSERT, function (operation) {

            var // the new table
                table = $('<table>').attr('role', 'grid').append($('<colgroup>')),
                // insert the table into the DOM tree
                inserted = insertContentNode(_.clone(operation.start), table),
                // new implicit paragraph behind the table
                newParagraph = null,
                // default table cell for too large tables (only for operation.sizeExceeded)
                cell = null, placeHolder = null,
                // number of rows in the table (only for operation.sizeExceeded)
                tableRows,
                // number of columns in the table (only for operation.sizeExceeded)
                tableColumns,
                // number of cells in the table (only for operation.sizeExceeded)
                tableCells,
                // the maximum allowed number of rows in the table (only for operation.sizeExceeded)
                maxRows,
                // the maximum allowed number of columns in the table (only for operation.sizeExceeded)
                maxCols,
                // the maximum allowed number of cells in the table (only for operation.sizeExceeded)
                maxCells,
                // the sentence stating the current number of rows/columns/cells
                countLabel = '',
                // the sentence stating the maximum number of rows/columns/cells
                limitLabel = '',
                // element from which pagebreaks calculus is started downwards
                currentElement;

            if (operation.start.length > 2) {
                // if its paragraph creation inside of table
                currentElement = Position.getContentNodeElement(editdiv, operation.start.slice(0, 1));
            } else {
                currentElement = Position.getContentNodeElement(editdiv, operation.start);
            }

            // insertContentNode() writes warning to console
            if (!inserted) { return false; }

            // generate undo/redo operations
            if (undoManager.isUndoEnabled()) {
                undoManager.addUndo({ name: Operations.DELETE, start: operation.start }, operation);
            }

            // Special replacement setting for a table, that cannot be displayed because of its size.
            if (_.isObject(operation.sizeExceeded)) {

                maxRows = TextConfig.getMaxTableRows();
                maxCols = TextConfig.getMaxTableColumns();
                maxCells = TextConfig.getMaxTableCells();

                tableRows = Utils.getIntegerOption(operation.sizeExceeded, 'rows', 0);
                tableColumns = Utils.getIntegerOption(operation.sizeExceeded, 'columns', 0);
                tableCells = tableRows * tableColumns;

                cell = $('<td>').append(
                    placeHolder = $('<div>').addClass('placeholder').append(
                        $('<div>').addClass('abs background-icon').append(Utils.createIcon('fa-table')),
                        $('<p>').text(gt('Table')),
                        $('<p>').text(gt('This table is too large to be displayed here.'))
                    )
                );

                if ((tableRows > 0) && (tableColumns > 0)) {

                    if (_.isNumber(maxRows) && (tableRows > maxRows)) {

                        countLabel = gt.format(
                            //#. %1$d is the number of rows in an oversized text table
                            //#, c-format
                            gt.ngettext('The table contains %1$d row.', 'This table contains %1$d rows.', tableRows),
                            _.noI18n(tableRows)
                        );
                        limitLabel = gt.format(
                            //#. %1$d is the maximum allowed number of rows in a text table
                            //#, c-format
                            gt.ngettext('Tables are limited to %1$d row.', 'Tables are limited to %1$d rows.', maxRows),
                            _.noI18n(maxRows)
                        );

                    } else if (_.isNumber(maxCols) && (tableColumns > maxCols)) {

                        countLabel = gt.format(
                            //#. %1$d is the number of columns in an oversized text table
                            //#, c-format
                            gt.ngettext('The table contains %1$d column.', 'This table contains %1$d columns.', tableColumns),
                            _.noI18n(tableColumns)
                        );
                        limitLabel = gt.format(
                            //#. %1$d is the maximum allowed number of columns in a text table
                            //#, c-format
                            gt.ngettext('Tables are limited to %1$d column.', 'Tables are limited to %1$d columns.', maxCols),
                            _.noI18n(maxCols)
                        );

                    } else if (_.isNumber(maxCells) && (tableCells > maxCells)) {

                        countLabel = gt.format(
                            //#. %1$d is the number of cells in an oversized text table
                            //#, c-format
                            gt.ngettext('The table contains %1$d cell.', 'This table contains %1$d cells.', tableCells),
                            _.noI18n(tableCells)
                        );
                        limitLabel = gt.format(
                            //#. %1$d is the maximum allowed number of cells in a text table
                            //#, c-format
                            gt.ngettext('Tables are limited to %1$d cell.', 'Tables are limited to %1$d cells.', maxCells),
                            _.noI18n(maxCells)
                        );

                    }

                    if (countLabel && limitLabel) {
                        placeHolder.append($('<p>').text(countLabel + ' ' + limitLabel));
                    }
                }

                table.addClass('size-exceeded')
                    .attr('contenteditable', false)
                    .append($('<tr>').append(cell))
                    .data('attributes', operation.attrs);  // setting attributes from operation directly to data without using 'setElementAttributes'
            }

            // apply the passed table attributes
            else if (_.isObject(operation.attrs) && _.isObject(operation.attrs.table)) {

//                We don't want operations to create operations themself, so we can't
//                call 'checkForLateralTableStyle' here.
//
//                styleId = operation.attrs.styleId;
//
//                // check if the table style is available and if not use the default style
//                if (styleId) {
//                    if (!tableStyles.containsStyleSheet(styleId)) {
//                        styleId = self.getDefaultUITableStylesheet();
//                    }
//                    Table.checkForLateralTableStyle(self, styleId);
//                    operation.attrs.styleId = styleId;
//                }

                if (pastingInternalClipboard  && _.browser.Firefox) { table.data('internalClipboard', true); }  // Fix for task 29401
                tableStyles.setElementAttributes(table, operation.attrs);
            }

            // call pagebreaks re-render after inserting table
            insertPageBreaksDebounced(currentElement);

            // adding an implicit paragraph behind the table (table must not be the last node in the document)
            if (table.next().length === 0) {
                newParagraph = DOM.createImplicitParagraphNode();
                validateParagraphNode(newParagraph);
                // insert the new paragraph behind the existing table node
                table.after(newParagraph);
                implParagraphChanged(newParagraph);
                // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
                // newly, also in other browsers (new jQuery version?)
                repairEmptyTextNodes(newParagraph);
            } else if (DOM.isImplicitParagraphNode(table.next())) {
                if (DOM.isTableInTableNode(table)) {
                    table.next().css('height', 0);  // hiding the paragraph behind the table inside another table
                }
            }

            if (undoRedoRunning  && _.browser.Firefox) { table.data('undoRedoRunning', true); }  // Fix for task 30477

            return true;
        });

        this.registerOperationHandler(Operations.COLUMNS_DELETE, function (operation) {

            var table = Position.getTableElement(editdiv, operation.start);

            if (table) {
                if (undoManager.isUndoEnabled()) {

                    var allRows = DOM.getTableRows(table),
                        allCellRemovePositions = Table.getAllRemovePositions(allRows, operation.startGrid, operation.endGrid),
                        generator = this.createOperationsGenerator();

                    allRows.each(function (index) {

                        var rowPos = operation.start.concat([index]),
                            cells = $(this).children(),
                            oneRowCellArray =  allCellRemovePositions[index],
                            end = oneRowCellArray.pop(),
                            start = oneRowCellArray.pop();  // more than one cell might be deleted in a row

                        // start<0: no cell will be removed in this row
                        if (start >= 0) {

                            if (end < 0) {
                                // remove all cells until end of row
                                end = cells.length;
                            } else {
                                // closed range to half-open range
                                end = Math.min(end + 1, cells.length);
                            }

                            // generate operations for all covered cells
                            cells.slice(start, end).each(function (index) {
                                generator.generateTableCellOperations(this, rowPos.concat([start + index]));
                            });
                        }
                    });
                    undoManager.addUndo(generator.getOperations(), operation);
                }
                implDeleteColumns(operation.start, operation.startGrid, operation.endGrid);
            }

            return true;
        });

        this.registerOperationHandler(Operations.CELL_MERGE, function (operation) {
            if (undoManager.isUndoEnabled()) {
                var content = null,
                    gridSpan = null,
                    undoOperation = { name: Operations.CELL_SPLIT, start: operation.start, content: content, gridSpan: gridSpan };
                undoManager.addUndo(undoOperation, operation);
            }
            return implMergeCell(_.copy(operation.start, true), operation.count);
        });

        this.registerOperationHandler(Operations.CELLS_INSERT, function (operation) {

            if (undoManager.isUndoEnabled()) {
                var count = Utils.getIntegerOption(operation, 'count', 1, 1),
                    undoOperations = [];

                // TODO: create a single DELETE operation for the cell range, once this is supported
                _(count).times(function () {
                    undoOperations.push({ name: Operations.DELETE, start: operation.start });
                });
                undoManager.addUndo(undoOperations, operation);
            }

            return implInsertCells(operation.start, operation.count, operation.attrs);
        });

        this.registerOperationHandler(Operations.ROWS_INSERT, function (operation) {

            if (undoManager.isUndoEnabled()) {
                var count = Utils.getIntegerOption(operation, 'count', 1, 1),
                    undoOperations = [];

                // TODO: create a single DELETE operation for the row range, once this is supported
                _(count).times(function () {
                    undoOperations.push({ name: Operations.DELETE, start: operation.start });
                });

                undoManager.addUndo(undoOperations, operation);
            }

            return implInsertRows(operation.start, operation.count, operation.insertDefaultCells, operation.referenceRow, operation.attrs);

        });

        this.registerOperationHandler(Operations.COLUMN_INSERT, function (operation) {
            if (undoManager.isUndoEnabled()) {
                undoManager.enterUndoGroup(function () {

                    // COLUMNS_DELETE cannot be the answer to COLUMN_INSERT, because the cells of the new column may be inserted
                    // at very different grid positions. It is only possible to remove the new cells with deleteCells operation.
                    var localPos = _.clone(operation.start),
                        table = Position.getDOMPosition(editdiv, localPos).node,  // -> this is already the new grid with the new column!
                        allRows = DOM.getTableRows(table),
                        allCellInsertPositions = Table.getAllInsertPositions(allRows, operation.gridPosition, operation.insertMode),
                        cellPosition = null;

                    for (var i = (allCellInsertPositions.length - 1); i >= 0; i--) {
                        cellPosition = Position.appendNewIndex(localPos, i);
                        cellPosition.push(allCellInsertPositions[i]);
                        undoManager.addUndo({ name: Operations.DELETE, start: cellPosition });
                    }

                    undoManager.addUndo(null, operation);  // only one redo operation

                }, this); // enterUndoGroup()
            }
            return implInsertColumn(operation.start, operation.gridPosition, operation.insertMode);
        });


        /**
         * Removes empty text nodes from the passed paragraph, checks whether
         * it needs a dummy text node, and converts consecutive white-space
         * characters.
         *
         * @param {HTMLElement|jQuery} paragraph
         *  The paragraph element to be validated. If this object is a jQuery
         *  collection, uses the first DOM node it contains.
         */
        function validateParagraphNode(paragraph) {

            var // current sequence of sibling text nodes
                siblingTextNodes = [],
                // array of arrays collecting all sequences of sibling text nodes
                allSiblingTextNodes = [siblingTextNodes],
                // whether the paragraph contains any text
                hasText = false,
                // whether the last child node is the dummy element
                hasLastDummy = false,
                // whether we have to add or preserve a dummy element for hard-breaks
                mustAddOrPreserveDummyForHardBreak = false;

            validateParagraphNode.DBG_COUNT = (validateParagraphNode.DBG_COUNT || 0) + 1;

            // convert parameter to a DOM node
            paragraph = Utils.getDomNode(paragraph);

            // whether last node is the dummy node
            hasLastDummy = DOM.isDummyTextNode(paragraph.lastChild);

            // remove all empty text spans which have sibling text spans, and collect
            // sequences of sibling text spans (needed for white-space handling)
            Position.iterateParagraphChildNodes(paragraph, function (node) {

                // visit all text spans embedded in text container nodes (fields, tabs, ... (NOT numbering labels))
                if (DOM.isTextSpan(node) || DOM.isTextComponentNode(node)) {
                    DOM.iterateTextSpans(node, function (span) {

                        //remove spell attributes - if there are any
                        characterStyles.setElementAttributes(span, { character: { spellerror: null } }, { special: true });
                        if (DOM.isEmptySpan(span)) {
                            // remove this span, if it is an empty portion and has a sibling text portion (should not happen anymore)
                            if (DOM.isTextSpan(span.previousSibling) || DOM.isTextSpan(span.nextSibling)) {
                                Utils.warn('Editor.validateParagraphNode(): empty text span with sibling text span found');
                                $(span).remove();
                            }
                            // otherwise simply ignore the empty span
                        } else {
                            // append text node to current sequence
                            siblingTextNodes.push(span.firstChild);
                        }
                    });

                // anything else (no text span or text container node): start a new sequence of text nodes
                } else {
                    allSiblingTextNodes.push(siblingTextNodes = []);
                }

            }, undefined, { allNodes: true });

            // Special case for hyperlink formatting and empty paragraph (just one empty textspan)
            // Here we always don't want to have the hyperlink formatting, we hardly reset these attributes
            if (DOM.isTextSpan(paragraph.firstElementChild) && (paragraph.children.length === 1) &&
                (paragraph.firstElementChild.textContent.length === 0)) {
                var url = characterStyles.getElementAttributes(paragraph.firstElementChild).character.url;
                if ((url !== null) && (url.length > 0)) {
                    characterStyles.setElementAttributes(paragraph.firstElementChild, Hyperlink.CLEAR_ATTRIBUTES);
                }
            } else if (DOM.isListLabelNode(paragraph.firstElementChild) && (paragraph.children.length === 2) &&
                       (paragraph.firstElementChild.nextSibling.textContent.length === 0)) {
                var url = characterStyles.getElementAttributes(paragraph.firstElementChild.nextSibling).character.url;
                if ((url !== null) && (url.length > 0)) {
                    characterStyles.setElementAttributes(paragraph.firstElementChild.nextSibling, Hyperlink.CLEAR_ATTRIBUTES);
                }
            }

            // Special case for hard-breaks. Check for a hard-break as a node before the last child which
            // is not a dummy node. We need to have a dummy node for a hard-break at the end of the
            // paragraph to support correct formatting & cursor travelling.
            if (paragraph.children.length >= 2) {
                var prevNode = paragraph.lastChild.previousSibling;

                if (paragraph.children.length >= 3 && hasLastDummy) {
                    // Second case: We have a dummy node and must preserve it.
                    // prevNode is in that case the previous node of prev node.
                    // (hard-break) - (span) - (br)
                    prevNode = prevNode.previousSibling;
                }

                if (prevNode && DOM.isHardBreakNode(prevNode)) {
                    mustAddOrPreserveDummyForHardBreak = true;
                }
            }

            // Convert consecutive white-space characters to sequences of SPACE/NBSP
            // pairs. We cannot use the CSS attribute white-space:pre-wrap, because
            // it breaks the paragraph's CSS attribute text-align:justify. Process
            // each sequence of sibling text nodes for its own (the text node
            // sequences may be interrupted by other elements such as hard line
            // breaks, drawings, or other objects).
            // TODO: handle explicit NBSP inserted by the user (when supported)
            _(allSiblingTextNodes).each(function (siblingTextNodes) {

                var // the complete text of all sibling text nodes
                    text = '',
                    // offset for final text distribution
                    offset = 0;

                // collect the complete text in all text nodes
                _(siblingTextNodes).each(function (textNode) { text += textNode.nodeValue; });

                // ignore empty sequences
                if (text.length > 0) {
                    hasText = true;

                    // process all white-space contained in the text nodes
                    text = text
                        // normalize white-space (convert to SPACE characters)
                        .replace(/\s/g, ' ')
                        // text in the node sequence cannot start with a SPACE character
                        .replace(/^ /, '\xa0')
                        // convert SPACE/SPACE pairs to SPACE/NBSP pairs
                        .replace(/ {2}/g, ' \xa0')
                        // text in the node sequence cannot end with a SPACE character
                        .replace(/ $/, '\xa0');

                    // distribute converted text to the text nodes
                    _(siblingTextNodes).each(function (textNode) {
                        var length = textNode.nodeValue.length;
                        textNode.nodeValue = text.substr(offset, length);
                        offset += length;
                    });
                }
            });

            // insert an empty text span if there is no other content (except the dummy node)
            if (!paragraph.hasChildNodes() || (hasLastDummy && (paragraph.childNodes.length === 1))) {
                $(paragraph).prepend(DOM.createTextSpan());
            }

            // append dummy text node if the paragraph contains no text,
            // or remove it if there is any text
            if (!hasText && !hasLastDummy) {
                $(paragraph).append(DOM.createDummyTextNode());
            } else if (!hasLastDummy && mustAddOrPreserveDummyForHardBreak) {
                // we need to add a dummy text node after a last hard-break
                $(paragraph).append(DOM.createDummyTextNode());
            } else if (hasText && hasLastDummy && !mustAddOrPreserveDummyForHardBreak) {
                $(paragraph.lastChild).remove();
            }
        }

        // ==================================================================
        // Private image methods
        // ==================================================================

//        function sendImageSize(position) {
//
//            // sending size of image to the server in an operation -> necessary after loading the image
//            var imagePos = Position.getDOMPosition(editdiv, _.copy(position), true);
//
//            if (imagePos && DOM.isImageNode(imagePos.node)) {
//
//                $('img', imagePos.node).one('load', function () {
//
//                    var width, height, para, maxWidth, factor, updatePosition, newOperation;
//
//                    // Workaround for a strange Chrome behavior, even if we use .one() Chrome fires the 'load' event twice.
//                    // One time for the image node rendered and the other time for a not rendered image node.
//                    // We check for the rendered image node
//                    if (Utils.containsNode(editdiv, this)) {
//                        width = Utils.convertLengthToHmm($(this).width(), 'px');
//                        height = Utils.convertLengthToHmm($(this).height(), 'px');
//
//                        // maybe the paragraph is not so big
//                        para = imagePos.node.parentNode;
//                        maxWidth = Utils.convertLengthToHmm($(para).outerWidth(), 'px');
//
//                        if (width > maxWidth) {
//                            factor = Utils.round(maxWidth / width, 0.01);
//                            width = maxWidth;
//                            height = Math.round(height * factor);
//                        }
//
//                        // updating the logical position of the image div, maybe it changed in the meantime while loading the image
//                        updatePosition = Position.getOxoPosition(editdiv, this, 0);
//                        newOperation = { name: Operations.SET_ATTRIBUTES, attrs: { drawing: { width: width, height: height } }, start: updatePosition };
//
//                        self.applyOperations(newOperation);
//                    }
//                });
//            }
//        }


        /**
         * Calculates the size of the image defined by the given url,
         * limiting the size to the paragraph size of the current selection.
         *
         * @param {String} url
         *  The image url or base64 data url
         *
         * @returns {jQuery.Promise}
         *  A promise where the resolve result contains the size object
         *  including width and height in Hmm {width: 123, height: 456}
         */
        function getImageSize(url) {

            var // the result deferred
                def = $.Deferred(),
                // the image for size rendering
                image = $('<img>'),
                // the image url
                absUrl,
                // the clipboard holding the image
                clipboard = null;

            if (!url) {
                return def.reject();
            }

            absUrl = Image.getFileUrl(app, url);

            // append the clipboard div to the body and place the image into it
            clipboard = app.getView().createClipboardNode();
            clipboard.append(image);

            // if you set the load handler BEFORE you set the .src property on a new image, you will reliably get the load event.
            image.one('load', function () {

                var width, height, para, maxWidth, maxHeight, factor, start,
                    pageAttributes = null;

                // Workaround for a strange Chrome behavior, even if we use .one() Chrome fires the 'load' event twice.
                // One time for the image node rendered and the other time for a not rendered image node.
                // We check for the rendered image node
                if (Utils.containsNode(clipboard, image)) {
                    width = Utils.convertLengthToHmm(image.width(), 'px');
                    height = Utils.convertLengthToHmm(image.height(), 'px');

                    // maybe the paragraph is not so big
                    start = selection.getStartPosition();
                    start.pop();
                    para = Position.getParagraphElement(editdiv, start);
                    if (para) {
                        maxWidth = Utils.convertLengthToHmm($(para).outerWidth(), 'px');
                        pageAttributes = pageStyles.getElementAttributes(self.getNode());
                        // reading page attributes, they are always available -> no need to check existence
                        maxHeight = pageAttributes.page.height - pageAttributes.page.marginTop - pageAttributes.page.marginBottom;

                        if (width > maxWidth) {
                            factor = Utils.round(maxWidth / width, 0.01);
                            width = maxWidth;
                            height = Math.round(height * factor);
                        }

                        if ((maxHeight) && (height > maxHeight)) {
                            factor = Utils.round(maxHeight / height, 0.01);
                            height = maxHeight;
                            width = Math.round(width * factor);
                        }
                    }

                    def.resolve({ width: width, height: height });
                }
            })
            .error(function () {
                Utils.warn('Editor.getImageSize(): image load error');
                def.reject();
            })
            .attr('src', absUrl);

            // always remove the clipboard again
            def.always(function () {
                clipboard.remove();
            });

            return def.promise();
        }

        // ====================================================================
        //  IMPLEMENTATION FUNCTIONS
        // Private methods, that are called from method applyOperations().
        // The operations itself are never generated inside an impl*-function.
        // ====================================================================

        /**
         * Has to be called after changing the structure of a paragraph node, if
         * the update has to be done immediately (synchronously).
         *
         * @param {jQuery} paragraphs
         *  All paragraph nodes that need to be updated included into a
         *  jQuery object
         */
        function implParagraphChangedSync(paragraphs) {

            paragraphs.each(function () {
                // the paragraph may have been removed from the DOM in the meantime
                if (Utils.containsNode(editdiv, this)) {
                    validateParagraphNode(this);
                    paragraphStyles.updateElementFormatting(this);
                }
            });
            // paragraph validation changes the DOM, restore selection
            // -> Not restoring browser selection, if the edit rights are not available (Task 29049)
            if ((!imeActive) && (self.getEditMode())) {
                selection.restoreBrowserSelection();
            }
        }

        /**
         * Has to be called every time after changing the structure of a
         * paragraph node.
         *
         * @param {HTMLElement|jQuery|Number[]} paragraph
         *  The paragraph element as DOM node or jQuery object, or the logical
         *  position of the paragraph or any of its child components.
         */
        var implParagraphChanged = $.noop;
        app.on('docs:import:success', function () {

            var // all paragraph nodes that need to be updated
                paragraphs = $(),
                // standard delay time (in ms)
                delay = Utils.IE9 ? 200 : 100;

            // direct callback: called every time when implParagraphChanged() has been called
            function registerParagraph(paragraph) {
                if (_.isArray(paragraph)) {
                    paragraph = Position.getCurrentParagraph(editdiv, paragraph);
                }
                // store the new paragraph in the collection (jQuery keeps the collection unique)
                if (paragraph) {
                    // reset to 'not spelled'
                    $(paragraph).removeClass('p_spelled');
                    paragraphs = paragraphs.add(paragraph);
                }
            }

            // deferred callback: called once, after current script ends
            function updateParagraphs() {
                implParagraphChangedSync(paragraphs);
                paragraphs = $();
            }

            // create and return the deferred implParagraphChanged() method
            implParagraphChanged = app.createDebouncedMethod(registerParagraph, updateParagraphs, { delay: delay, maxDelay: delay * 10 });
        });

        /**
         * Has to be called every time after changing the cell structure of a
         * table. It recalculates the position of each cell in the table and
         * sets the corresponding attributes. This can be set for the first or
         * last column or row, or even only for the south east cell.
         *
         * @param {HTMLTableElement|jQuery} table
         *  The table element whose structure has been changed. If this object
         *  is a jQuery collection, uses the first node it contains.
         */
        var implTableChanged = $.noop;
        app.on('docs:import:success', function () {

            var // all table nodes that need to be updated
                tables = $();

            // direct callback: called every time when implTableChanged() has been called
            function registerTable(table) {
                // store the new table in the collection (jQuery keeps the collection unique)
                tables = tables.add(table);
            }

            // deferred callback: called once, after current script ends
            function updateTables() {
                tables.each(function () {
                    // the table may have been removed from the DOM in the meantime
                    if (Utils.containsNode(editdiv, this)) {
                        tableStyles.updateElementFormatting(this);
                    }
                });
                tables = $();
            }

            // create and return the deferred implTableChanged() method
            implTableChanged = app.createDebouncedMethod(registerTable, updateTables);
        });

        /**
         * Prepares a 'real' paragraph after insertion of text, tab, drawing, ...
         * by exchanging an 'implicit' paragraph (in empty documents, empty cells,
         * behind tables, ...) with the help of an operation. Therefore the server is
         * always informed about creation and removal of paragraphs and implicit
         * paragraphs are only required for user input in the browser.
         *
         * @param {Number[]} position
         *  The logical text position.
         */
        function handleImplicitParagraph(position) {

            var // the searched implicit paragraph
                paragraph = null,
                // the new created paragraph
                newParagraph = null;

            position = _.clone(position);
            if (position.pop() === 0) {  // is this an empty paragraph?
                paragraph = (useParagraphCache && paragraphCache) || Position.getParagraphElement(editdiv, position);
                if ((paragraph) && (DOM.isImplicitParagraphNode(paragraph)) && (Position.getParagraphNodeLength(paragraph) === 0)) {
                    // removing implicit paragraph node
                    $(paragraph).remove();
                    // creating new paragraph explicitely
                    self.applyOperations({ name: Operations.PARA_INSERT, start: position });
                    // Setting attributes to new paragraph immediately (task 25670)
                    newParagraph = Position.getParagraphElement(editdiv, position);
                    paragraphStyles.updateElementFormatting(newParagraph);
                    // using the new paragraph as global cache
                    if (useParagraphCache && paragraphCache) { paragraphCache = newParagraph; }
                }
            }
        }

        /**
         * Removes an 'real' paragraph and prepares an 'implicit' paragraph after
         * pressing 'backspace' (or 'delete'?) in an empty paragraph. The 'real'
         * paragraph is removed with the help of an operation. Therefore the server is
         * always informed about creation and removal of paragraphs. The implicit
         * paragraphs are only required for user input in the browser. This is
         * important for new documents, document cells and paragraphs behind
         * tables at the end of the document.
         *
         * @param {Number[]} position
         *  The logical text position.
         */
        function checkImplicitParagraph(position) {
            var paragraph = null,
                newParagraph = null;

            if (position.pop() === 0) {  // is this an empty paragraph?
                paragraph = Position.getParagraphElement(editdiv, position);

                // the paragraph must not be marked as 'implicit' already and the paragraph must be empty and
                // the paragraph must have no neighbours or must be the final paragraph behind a table.
                if ((! DOM.isImplicitParagraphNode(paragraph)) && (Position.getParagraphNodeLength(paragraph) === 0) &&
                    ((DOM.isParagraphWithoutNeighbour(paragraph)) || (DOM.isFinalParagraphBehindTable(paragraph)))) {
                    // adding implicit paragraph node
                    newParagraph = DOM.createImplicitParagraphNode();
                    $(paragraph).after(newParagraph);
                    validateParagraphNode(newParagraph);
                    implParagraphChanged(newParagraph);
                    // Setting attributes to new paragraph immediately (task 25670)
                    paragraphStyles.updateElementFormatting(newParagraph);
                    // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
                    // newly, also in other browsers (new jQuery version?)
                    repairEmptyTextNodes(newParagraph);
                    // remove paragraph explicitely
                    self.applyOperations({ name: Operations.DELETE, start: position });
                }
            }
        }

        /**
         * Prepares the text span at the specified logical position for
         * insertion of a new text component or character. Splits the text span
         * at the position, if splitting is required. Always splits the span,
         * if the position points between two characters of the span.
         * Additionally splits the span, if there is no previous sibling text
         * span while the position points to the beginning of the span, or if
         * there is no next text span while the position points to the end of
         * the span.
         *
         * @param {Number[]} position
         *  The logical text position.
         *
         * @returns {HTMLSpanElement|Null}
         *  The text span that precedes the passed offset. Will be the leading
         *  part of the original text span addressed by the passed position, if
         *  it has been split, or the previous sibling text span, if the passed
         *  position points to the beginning of the span, or the entire text
         *  span, if the passed position points to the end of a text span and
         *  there is a following text span available. Returns null, if the
         *  passed logical position is invalid.
         */
        function prepareTextSpanForInsertion(position) {

            var // node info at passed position (DOM text node level)
                nodeInfo = Position.getDOMPosition(editdiv, position);

            // check that the parent is a text span
            if (!nodeInfo || !nodeInfo.node || !DOM.isPortionSpan(nodeInfo.node.parentNode)) {
                Utils.warn('Editor.prepareTextSpanForInsertion(): expecting text span at position ' + JSON.stringify(position));
                return null;
            }
            nodeInfo.node = nodeInfo.node.parentNode;

            // do not split at beginning with existing preceding text span
            if ((nodeInfo.offset === 0) && DOM.isTextSpan(nodeInfo.node.previousSibling)) {
                return nodeInfo.node.previousSibling;
            }

            // return current span, if offset points to its end with following text span
            if ((nodeInfo.offset === nodeInfo.node.firstChild.nodeValue.length) && DOM.isTextSpan(nodeInfo.node.nextSibling)) {
                return nodeInfo.node;
            }

            // otherwise, split the span
            return DOM.splitTextSpan(nodeInfo.node, nodeInfo.offset)[0];
        }

        /**
         * Inserts a simple text portion into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new text portion.
         *
         * @param {String} text
         *  The text to be inserted.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new text portion, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @returns {Boolean}
         *  Whether the text portion has been inserted successfully.
         */
        function implInsertText(start, text, attrs) {

            var // text span that will precede the field
                span = prepareTextSpanForInsertion(start),
                // new new text span
                newSpan = null,
                //variables for page breaks rendering
                currentElement = null,
                prevHeight = null;

            if (!span) { return false; }

            currentElement = span.parentNode;
            if (!$(currentElement.parentNode).hasClass('pagecontent')) { //its a div.p inside table(s)
                currentElement = $(currentElement).parents('table').last()[0];
            }
            prevHeight = currentElement.offsetHeight;

            // split the text span again to get actual character formatting for
            // the span, and insert the text
            newSpan = DOM.splitTextSpan(span, span.firstChild.nodeValue.length, { append: true }).contents().remove().end().text(text);
            if (!text) (DOM.ensureExistingTextNode(newSpan));  // empty text should never happen

            if (DOM.isImplicitParagraphNode(newSpan.parent())) { return false; }  // do not write text into an implicit paragraph, improvement for task 30906

            // apply the passed text attributes
            if (_.isObject(attrs)) {
                characterStyles.setElementAttributes(newSpan, attrs);
            }

            // try to merge with preceding and following span
            CharacterStyles.mergeSiblingTextSpans(newSpan, true);
            CharacterStyles.mergeSiblingTextSpans(newSpan);

            // validate paragraph, store new cursor position
            if ((guiTriggeredOperation) || (!app.isImportFinished())) {
                implParagraphChanged(newSpan.parent());  // Performance and task 30587: Local client can defer attribute setting, Task 30603: deferring also during loading document
            } else {
                implParagraphChangedSync(newSpan.parent());  // Performance and task 30587: Remote client must set attributes synchronously
            }

            if ($(currentElement).data('lineBreaksData')) {
                $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
            }
            if (prevHeight !== currentElement.offsetHeight) {
                insertPageBreaksDebounced(currentElement);
            }

            lastOperationEnd = Position.increaseLastIndex(start, text.length);

            // Performance: Saving paragraph and final position, to get faster access to DOM position during setting the cursor after inserting text
            selection.setInsertTextInfo(newSpan.parent(), _.last(lastOperationEnd));

            return true;
        }

        /**
         * Inserts a text field component into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new text field.
         *
         * @param {String} type
         *  A property describing the field type.
         *
         * @param {String} representation
         *  A fallback value, if the placeholder cannot be substituted with a
         *  reasonable value.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new text field, as map of attribute
         *  maps (name/value pairs), keyed by attribute family.
         *
         * @returns {Boolean}
         *  Whether the text field has been inserted successfully.
         */
        function implInsertField(start, type, representation, attrs) {

            var // text span that will precede the field
                span = prepareTextSpanForInsertion(start),
                // new text span for the field node
                fieldSpan = null;

            if (!span) { return false; }

            // split the text span again to get initial character formatting
            // for the field, and insert the field representation text

            // Fix for 29265: Removing empty text node in span with '.contents().remove().end()'.
            // Otherwise there are two text nodes in span after '.text(representation)' in IE.
            fieldSpan = DOM.splitTextSpan(span, 0).contents().remove().end().text(representation);
            if (!representation) { DOM.ensureExistingTextNode(fieldSpan); }

            // insert a new text field before the addressed text node, move
            // the field span element into the field node
            DOM.createFieldNode().append(fieldSpan).insertAfter(span);

            // apply the passed field attributes
            if (_.isObject(attrs)) {
                characterStyles.setElementAttributes(fieldSpan, attrs);
            }

            // validate paragraph, store new cursor position
            implParagraphChanged(span.parentNode);
            lastOperationEnd = Position.increaseLastIndex(start);
            return true;
        }

        /**
         * Inserts a horizontal tabulator component into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new tabulator.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new tabulator component, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @returns {Boolean}
         *  Whether the tabulator has been inserted successfully.
         */
        function implInsertTab(start, attrs) {

            var // text span that will precede the field
                span = prepareTextSpanForInsertion(start),
                // new text span for the tabulator node
                tabSpan = null,
                // new tabulator node
                newTabNode,
                // element from which we calculate pagebreaks
                currentElement = Position.getContentNodeElement(editdiv, start.slice(0, -1));


            if (!span) { return false; }

            // split the text span to get initial character formatting for the tab
            tabSpan = DOM.splitTextSpan(span, 0);
            if (_.browser.IE) {
                // Prevent resizing rectangle provided by IE for tab div even
                // set to contenteditable=false. Therefore we need to set IE
                // specific unselectable=on.
                tabSpan.attr('unselectable', 'on');
            } else if (_.browser.iOS && _.browser.Safari) {
                // Safari mobile allows to visit nodes which has contenteditable=false, therefore
                // we have to use a different solution. We use -webkit-user-select: none to force
                // Safari mobile to jump over the text span with the fill characters.
                tabSpan.css('-webkit-user-select', 'none');
            }

            newTabNode = DOM.createTabNode();

            // insert a tab container node before the addressed text node, move
            // the tab span element into the tab container node
            newTabNode.append(tabSpan).insertAfter(span);

            // apply the passed tab attributes
            if (_.isObject(attrs)) {
                characterStyles.setElementAttributes(tabSpan, attrs);
            }

            // validate paragraph, store new cursor position
            implParagraphChanged(span.parentNode);
            lastOperationEnd = Position.increaseLastIndex(start);

            if ($(currentElement).data('lineBreaksData')) {
                $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
            }
            // call for debounced render of pagebreaks
            insertPageBreaksDebounced(currentElement);

            return true;
        }

        /**
         * Inserts a hard break component into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new tabulator.
         *
         * @param {Object} [attrs]
         *  Attributes to be applied at the new hard-break component, as map of
         *  attribute maps (name/value pairs), keyed by attribute family.
         *
         * @returns {Boolean}
         *  Whether the tabulator has been inserted successfully.
         */
        function implInsertHardBreak(start, attrs) {

            var // text span that will precede the hard-break node
                span = prepareTextSpanForInsertion(start),
                // new text span for the hard-break node
                hardbreakSpan = null,
                // element from which we calculate pagebreaks
                currentElement = Position.getContentNodeElement(editdiv, start.slice(0, -1));

            if (!span) { return false; }

            // split the text span to get initial character formatting for the hard-break
            hardbreakSpan = DOM.splitTextSpan(span, 0);

            // insert a hard-break container node before the addressed text node, move
            // the hard-break span element into the hard-break container node
            // Fix for 29265: Replacing '.empty()' of hardbreakSpan with '.contents().remove().end()'
            // to get rid of empty text node in span in IE.
            hardbreakSpan.contents().remove().end().append($('<br>'));
            DOM.createHardBreakNode().append(hardbreakSpan).insertAfter(span);

            // apply the passed tab attributes
            if (_.isObject(attrs)) {
                characterStyles.setElementAttributes(hardbreakSpan, attrs);
            }

            // validate paragraph, store new cursor position
            implParagraphChanged(span.parentNode);
            lastOperationEnd = Position.increaseLastIndex(start);

            if ($(currentElement).data('lineBreaksData')) {
                $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
            }
            // call for debounced render of pagebreaks
            insertPageBreaksDebounced(currentElement);

            return true;
        }

        function implInsertDrawing(type, start, attrs) {

            var // text span that will precede the field
                span = prepareTextSpanForInsertion(start),
                // deep copy of attributes, because they are modified in webkit browsers
                attributes = _.copy(attrs, true),
                // drawing attributes from attribute object
                drawingAttrs = (_.isObject(attributes) && _.isObject(attributes.drawing)) ? attributes.drawing : {},
                // new drawing node
                drawingNode = null,
                // image aspect ratio
                factor,
                // 8 px offset to substract from image width in case of webkit
                webKitOffset = Utils.convertLengthToHmm(8, 'px'),
                // element from which we start DOM nodes processing for pagebreaks
                currentElement = Position.getContentNodeElement(editdiv, start.slice(0, -1));

            if (!span) { return false; }

            // insert the drawing with default settings between the two text nodes (store original URL for later use)
            drawingNode = DrawingFrame.createDrawingFrame(type).insertAfter(span);

            if ((type === 'image') || (type === 'ole')) {
                // fix for webkit browsers to avoid showing a line above the image
                // due to rendering the span above the drawing node
                if (_.browser.WebKit && (drawingAttrs.width > webKitOffset) && (drawingAttrs.height > 0)) {
                    factor = (drawingAttrs.width > 0) ? (drawingAttrs.height / drawingAttrs.width) : 1;
                    attributes.drawing.width = Math.max(attributes.drawing.width - webKitOffset, 100);
                    attributes.drawing.height = Math.round(attributes.drawing.width * factor);
                }

            } else if (type === 'horizontal_line') {
                drawingNode.addClass('horizontal-line');
            }

            // apply the passed drawing attributes
            if (_.isObject(attributes)) {
                drawingStyles.setElementAttributes(drawingNode, attributes);
            }

            // validate paragraph, store new cursor position
            implParagraphChanged(span.parentNode);
            lastOperationEnd = Position.increaseLastIndex(start);

            if ($(currentElement).data('lineBreaksData')) {
                $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
            }
            insertPageBreaksDebounced(currentElement);

            return true;
        }

        /**
         * Internet Explorer helper function. Repairs empty text nodes, that are
         * removed by IE after inserting a node into the DOM. This function
         * adds this empty text nodes again.
         *
         * After updating to a new version of jQuery, this seems to be
         * nessecary in other browsers too.
         *
         * @param {HTMLElement|jQuery} element
         *  The DOM element whose empty text nodes shall be added again.
         */
        function repairEmptyTextNodes(element) {

            if (!_.browser.IE) { return; }  // only necessary for IE

            if ($(element).is('span')) {
                DOM.ensureExistingTextNode(element);
            } else if (DOM.isParagraphNode(element)) {
                $(element).find('> span').each(function () {
                    DOM.ensureExistingTextNode(this);
                });
            } else if (DOM.isTableCellNode(element)) {
                DOM.getCellContentNode(element).find('> ' + DOM.PARAGRAPH_NODE_SELECTOR + ' > span').each(function () {
                    DOM.ensureExistingTextNode(this);
                });
            } else if (DOM.isTableRowNode(element)) {
                $(element).find('> td').each(function () {
                    DOM.getCellContentNode(this).find('> ' + DOM.PARAGRAPH_NODE_SELECTOR + ' > span').each(function () {
                        DOM.ensureExistingTextNode(this);
                    });
                });
            }
        }

        /**
         * Returns the family of the attributes supported by the passed DOM
         * element.
         *
         * @param {HTMLElement|jQuery} element
         *  The DOM element whose associated attribute family will be returned.
         *  If this object is a jQuery collection, returns its first node.
         */
        function resolveElementFamily(element) {

            var // the element, as jQuery object
                $element = $(element),
                // the resulting style family
                family = null;

            if (DOM.isTextSpan($element) || DOM.isTextSpanContainerNode($element) || DOM.isHardBreakNode($element)) {
                family = 'character';
            } else if (DOM.isParagraphNode($element)) {
                family = 'paragraph';
            } else if (DOM.isTableNode($element)) {
                family = 'table';
            } else if ($element.is('tr')) {
                family = 'row';
            } else if ($element.is('td')) {
                family = 'cell';
            } else if (DrawingFrame.isDrawingFrame($element)) {
                family = 'drawing';
            } else {
                Utils.warn('Editor.resolveElementFamily(): unsupported element');
            }

            return family;
        }

        /**
         * Returns the type of the attributes supported by the passed DOM
         * element.
         *
         * @param {HTMLElement|jQuery} element
         *  The DOM element whose associated attribute type will be returned.
         *  If this object is a jQuery collection, returns its first node.
         */
        function resolveElementType(element) {

            var // the element, as jQuery object
                $element = $(element),
                // the resulting style type
                type = null;

            if (DOM.isTextSpan($element) || DOM.isTextSpanContainerNode($element) || DrawingFrame.isDrawingFrame($element) || DOM.isHardBreakNode($element)) {
                type = 'text';
            } else if (DOM.isParagraphNode($element)) {
                type = 'paragraph';
            } else if (DOM.isTableNode($element)) {
                type = 'table';
            } else if ($element.is('tr')) {
                type = 'row';
            } else if ($element.is('td')) {
                type = 'cell';
            } else {
                Utils.warn('Editor.resolveElementType(): unsupported element');
            }

            return type;
        }

        /**
         * Changes a specific formatting attribute of the specified element or
         * text range. The type of the attributes will be determined from the
         * specified range.
         *
         * @param {Number[]} start
         *  The logical start position of the element or text range to be
         *  formatted.
         *
         * @param {Number[]} [end]
         *  The logical end position of the element or text range to be
         *  formatted.
         *
         * @param {Object} attributes
         *  A map with formatting attribute values, mapped by the attribute
         *  names, and by attribute family names.
         */
        function implSetAttributes(start, end, attributes) {

            var // node info for start/end position
                startInfo = null, endInfo = null,
                // the main attribute family of the target components
                styleFamily = null,
                // the style sheet container for the target components
                styleSheets = null,
                // options for StyleSheets method calls
                options = null,
                // the last text span visited by the character formatter
                lastTextSpan = null,
                // undo operations going into a single action
                undoOperations = [],
                // redo operation
                redoOperation = null,
                // Performance: Saving data for list updates
                paraAttrs = null, listStyleId = null, listLevel = null, oldListStyleId = null,
                // a helper attribute object with values set to null
                nullAttributes = null,
                // a helper iterator key
                localkey = null,
                //element which is passed to page breaks calculation
                currentElement,
                //flags for triggering page breaks
                isCharOrDrawingOrRow,
                isClearFormatting,
                isAttributesInParagraph;


            // sets or clears the attributes using the current style sheet container
            function setElementAttributes(element) {
                styleSheets.setElementAttributes(element, attributes, options);
            }

            // change listener used to build the undo operations
            function changeListener(element, oldAttributes, newAttributes) {

                var // selection object representing the passed element
                    range = Position.getPositionRangeForNode(editdiv, element),
                    // the attributes of the current family for the undo operation
                    undoAttributes = {},
                    // the operation used to undo the attribute changes
                    undoOperation = { name: Operations.SET_ATTRIBUTES, start: range.start, end: range.end, attrs: undoAttributes },
                    // last undo operation (used to merge character attributes of sibling text spans)
                    lastUndoOperation = ((undoOperations.length > 0) && (styleFamily === 'character')) ? _.last(undoOperations) : null;

                function insertUndoAttribute(family, name, value) {
                    undoAttributes[family] = undoAttributes[family] || {};
                    undoAttributes[family][name] = value;
                }

                // find all old attributes that have been changed or cleared
                _(oldAttributes).each(function (attributeValues, family) {
                    if (family === 'styleId') {
                        // style sheet identifier is a string property
                        if (attributeValues !== newAttributes.styleId) {
                            undoAttributes.styleId = attributeValues;
                        }
                    } else {
                        // process attribute map of the current family
                        _(attributeValues).each(function (value, name) {
                            if (!(family in newAttributes) || !_.isEqual(value, newAttributes[family][name])) {
                                insertUndoAttribute(family, name, value);
                            }
                        });
                    }
                });

                // find all newly added attributes
                _(newAttributes).each(function (attributeValues, family) {
                    if (family === 'styleId') {
                        // style sheet identifier is a string property
                        if (!_.isString(oldAttributes.styleId)) {
                            undoAttributes.styleId = null;
                        }
                    } else {
                        // process attribute map of the current family
                        _(attributeValues).each(function (value, name) {
                            if (!(family in oldAttributes) || !(name in oldAttributes[family])) {
                                insertUndoAttribute(family, name, null);
                            }
                        });
                    }
                });

                // try to merge 'character' undo operation with last array entry, otherwise add operation to array
                if (lastUndoOperation && (_.last(lastUndoOperation.end) + 1 === _.last(undoOperation.start)) && _.isEqual(lastUndoOperation.attrs, undoOperation.attrs)) {
                    lastUndoOperation.end = undoOperation.end;
                } else {
                    undoOperations.push(undoOperation);
                }

                // invalidate spell result if language attribute changes
                if (newAttributes.character && newAttributes.character.language) {
                    $(element).closest(DOM.PARAGRAPH_NODE_SELECTOR).removeClass('p_spelled');
                }
            }

            // fail if attributes is not of type object
            if (!_.isObject(attributes)) {
                return false;
            }

            // do nothing if an empty attributes object has been passed
            if (_.isEmpty(attributes)) {
                return;
            }

            // resolve start and end position
            if (!_.isArray(start)) {
                Utils.warn('Editor.implSetAttributes(): missing start position');
                return false;
            }
            startInfo = Position.getDOMPosition(editdiv, start, true);
            if (!startInfo || !startInfo.node) {
                Utils.warn('Editor.implSetAttributes(): invalid start position: ' + JSON.stringify(start));
                return false;
            }
            endInfo = _.isArray(end) ? Position.getDOMPosition(editdiv, end, true) : startInfo;
            if (!endInfo || !endInfo.node) {
                Utils.warn('Editor.implSetAttributes(): invalid end position: ' + JSON.stringify(end));
                return false;
            }
            end = end || start;

            // get attribute family of start and end node
            startInfo.family = resolveElementFamily(startInfo.node);
            endInfo.family = resolveElementFamily(endInfo.node);
            if (!startInfo.family || !endInfo.family) { return; }

            // options for the StyleSheets method calls (build undo operations while formatting)
            options = undoManager.isUndoEnabled() ? { changeListener: changeListener } : null;

            // characters (start or end may point to a drawing node, ignore that but format as
            // characters if the start object is different from the end object)
            if ((startInfo.family === 'character') || (endInfo.family === 'character') ||
                    ((startInfo.node !== endInfo.node) && DrawingFrame.isDrawingFrame(startInfo.node) && DrawingFrame.isDrawingFrame(endInfo.node))) {

                // check that start and end are located in the same paragraph
                if (startInfo.node.parentNode !== endInfo.node.parentNode) {
                    Utils.warn('Editor.implSetAttributes(): end position in different paragraph');
                    return;
                }

                // visit all text span elements covered by the passed range
                // (not only the direct children of the paragraph, but also
                // text spans embedded in component nodes such as fields and tabs)
                styleFamily = 'character';
                styleSheets = characterStyles;
                Position.iterateParagraphChildNodes(startInfo.node.parentNode, function (node) {

                    // DOM.iterateTextSpans() visits the node itself if it is a
                    // text span, otherwise it visits all descendant text spans
                    // contained in the node except for drawings which will be
                    // skipped (they may contain their own paragraphs).
                    DOM.iterateTextSpans(node, function (span) {
                        setElementAttributes(span);
                        // try to merge with the preceding text span
                        CharacterStyles.mergeSiblingTextSpans(span, false);
                        // remember span (last visited span will be merged with its next sibling)
                        lastTextSpan = span;
                    });

                }, undefined, {
                    // options for Position.iterateParagraphChildNodes()
                    allNodes: true,
                    start: _(start).last(),
                    end: _(end).last(),
                    split: true
                });

                // try to merge last text span in the range with its next sibling
                if (lastTextSpan) {
                    CharacterStyles.mergeSiblingTextSpans(lastTextSpan, true);
                }

            // otherwise: only single components allowed at this time
            } else {

                // check that start and end point to the same element
                if (startInfo.node !== endInfo.node) {
                    Utils.warn('Editor.implSetAttributes(): no ranges supported for attribute family "' + startInfo.family + '"');
                    return;
                }

                // format the (single) element
                styleFamily = startInfo.family;
                styleSheets = self.getStyleSheets(styleFamily);

                // Performance: Saving old list style id, before it is removed
                if ((app.isImportFinished()) && (styleFamily === 'paragraph')) {
                    paraAttrs = StyleSheets.getExplicitAttributes(startInfo.node);
                    if (paraAttrs && paraAttrs.paragraph && paraAttrs.paragraph.listStyleId) { oldListStyleId = paraAttrs.paragraph.listStyleId; }
                }

                setElementAttributes(startInfo.node);
            }

            // create the undo action
            if (undoManager.isUndoEnabled()) {
                redoOperation = { name: Operations.SET_ATTRIBUTES, start: start, end: end, attrs: attributes };
                undoManager.addUndo(undoOperations, redoOperation);
            }

            // update numberings and bullets (but updateListsDebounced can only be called after successful document import)
            if ((app.isImportFinished()) && (styleFamily === 'paragraph')) {
                if ((('styleId' in attributes) || _.isObject(attributes.paragraph) && (('listLevel' in attributes.paragraph) || ('listStyleId' in attributes.paragraph)))) {

                    // determining listStyleId und listLevel
                    if ((_.isObject(attributes.paragraph)) && ('listStyleId' in attributes.paragraph) && (attributes.paragraph.listStyleId !== null)) {
                        listStyleId = attributes.paragraph.listStyleId;  // list style assigned to paragraph
                        listLevel = ('listLevel' in attributes.paragraph) ? attributes.paragraph.listLevel : 0;
                    } else {
                        // list level modified -> checking paragraph attributes for list styles to receive list style id
                        paraAttrs = StyleSheets.getExplicitAttributes(startInfo.node);
                        // updating lists, if required
                        if (isListStyleParagraph(null, paraAttrs)) {
                            if (paraAttrs && paraAttrs.paragraph) {
                                listStyleId = paraAttrs.paragraph.listStyleId;
                                listLevel = paraAttrs.paragraph.listLevel;
                            }
                        }
                    }

                    // handling the case, that the list style id is removed
                    if (!listStyleId && oldListStyleId) { listStyleId = oldListStyleId; }

                    // defining list style Id or listLevel, that need to be updated
                    if (listStyleId || listLevel) {
                        // mark this paragraph for later list update. This is necessary, because it does not yet contain
                        // the list label node (DOM.LIST_LABEL_NODE_SELECTOR) and is therefore ignored in updateLists.
                        $(startInfo.node).data('updateList', 'true');
                        // registering this list style for update
                        // special handling for removal of listLevel, for example after 'Backspace'
                        if (listStyleId && (listLevel === null || listLevel === -1)) {
                            $(startInfo.node).data('removeLabel', 'true');
                        }

                        // not registering listStyle for updating, if this is a bullet list -> updating only specific paragraphs
                        if (listCollection.isAllLevelsBulletsList(listStyleId)) { listStyleId = null; }

                        updateListsDebounced({useSelectedListStyleIDs: true, paraInsert: true, listStyleId: listStyleId, listLevel: listLevel});
                    }

                } else if ('character' in attributes) {  // modified character attributes may lead to modified list labels

                    // checking paragraph attributes for list styles to receive list style id
                    paraAttrs = StyleSheets.getExplicitAttributes(startInfo.node);
                    // updating lists, after character attributes were modified
                    if (isListStyleParagraph(null, paraAttrs)) {
                        if (paraAttrs && paraAttrs.paragraph) {
                            listStyleId = paraAttrs.paragraph.listStyleId;
                            listLevel = paraAttrs.paragraph.listLevel;
                            updateListsDebounced({useSelectedListStyleIDs: true, paraInsert: true, listStyleId: listStyleId, listLevel: listLevel});
                        }
                    }
                }
            }

            if ((styleFamily === 'paragraph') && (attributes.character) && (Position.getParagraphNodeLength(startInfo.node) === 0)) {
                // Task 28187: Setting character attributes at empty paragraphs
                // -> removing character attributes from text span so that character styles at paragraph become visible (also 30927)
                nullAttributes = _.copy(attributes, true);
                for (localkey in nullAttributes.character) { nullAttributes.character[localkey] = null; }
                characterStyles.setElementAttributes(DOM.findFirstPortionSpan(startInfo.node), nullAttributes);
            }

            // adjust tabulators, if character or drawing attributes have been changed
            // (changing paragraph or table attributes updates tabulators automatically)
            if (app.isImportFinished() && ((styleFamily === 'character') || (styleFamily === 'drawing'))) {
                paragraphStyles.updateTabStops(startInfo.node.parentNode);
            }
            // after document is loaded, trigger pagebreaks reposition on certain events like font size change, height of drawing or row changed
            isCharOrDrawingOrRow = 'character' in attributes || 'drawing' in attributes || 'row' in attributes;
            isClearFormatting = 'styleId' in attributes && attributes.styleId === null;
            isAttributesInParagraph = _.isObject(attributes.paragraph) && (('lineHeight' in attributes.paragraph) || ('listLevel' in attributes.paragraph) || ('listStyleId' in attributes.paragraph));

            if (app.isImportFinished() && (isCharOrDrawingOrRow || isClearFormatting || isAttributesInParagraph)) {
                if (start.length > 1) {
                    // if its paragraph creation inside of table
                    currentElement = Position.getContentNodeElement(editdiv, start.slice(0, 1));
                } else {
                    currentElement = Position.getContentNodeElement(editdiv, start);
                }
                if ($(currentElement).data('lineBreaksData')) {
                    $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
                }
                insertPageBreaksDebounced(currentElement);
            }

            return true;
        }

        /**
         * Inserts the passed content node at the specified logical position.
         *
         * @param {Number[]} position
         *  The logical position of the new content node.
         *
         * @param {HTMLElement|jQuery} node
         *  The new content node. If this object is a jQuery collection, uses
         *  the first node it contains.
         *
         * @returns {Boolean}
         *  Whether the content node has been inserted successfully.
         */
        function insertContentNode(position, node) {

            var origPosition = _.clone(position),
                index = _.last(position),
                contentNode = null,
                insertBefore = true,
                insertAtEnd = false,
                parentNode = null;

            function getContainerNode(position) {

                var // logical position of the paragraph container node
                    parentPosition = position.slice(0, -1),
                    // the container node for the paragraph
                    containerInfo = Position.getDOMPosition(editdiv, parentPosition, true),
                    // the parent container node
                    containerNode = null;

                // resolve component node to the correct node that contains the
                // child content nodes (e.g. resolve table cell elements to the
                // embedded div.cellcontent elements, or drawing elements to the
                // embedded div.content elements)
                if (containerInfo && containerInfo.node) {
                    containerNode = DOM.getChildContainerNode(containerInfo.node)[0];
                }

                return containerNode;
            }

            // trying to find existing paragraphs, tables or the parent element
            if (index > -1) {
                contentNode = Position.getContentNodeElement(editdiv, position);

                // content node does not exist -> maybe the new paragraph/table shall be added to the end of the document
                if (! contentNode) {

                    if (index === 0) {
                        // trying to find parent node, because this is the first paragraph/table in document or table cell
                        parentNode = getContainerNode(position);
                        if (! parentNode) {
                            Utils.warn('Editor.insertContentNode(): cannot find parent node at position ' + JSON.stringify(position));
                            return false;
                        }
                        insertBefore = false;
                        insertAtEnd = true;

                    } else {
                        // Further try to find an existing element, behind that the new paragraph/table shall be appended
                        position[position.length - 1]--;
                        contentNode = Position.getContentNodeElement(editdiv, position);

                        if (contentNode) {
                            insertBefore = false;
                        } else {
                            Utils.warn('Editor.insertContentNode(): cannot find content node at position ' + JSON.stringify(origPosition));
                            return false;
                        }
                    }
                }
            } else if (index === -1) { // Definition: Adding paragraph/table to the end if index is -1
                parentNode = getContainerNode(position);
                if (! parentNode) {
                    Utils.warn('Editor.insertContentNode(): cannot find parent node at position ' + JSON.stringify(position));
                    return false;
                }
                insertBefore = false;
                insertAtEnd = true;
            } else {
                Utils.warn('Editor.insertContentNode(): invalid logical position ' + JSON.stringify(position));
                return false;
            }

            // modify the DOM
            if (insertAtEnd) {
                $(node).first().appendTo(parentNode);
            } else if (insertBefore) {
                $(node).first().insertBefore(contentNode);
            } else {
                $(node).first().insertAfter(contentNode);
            }

            return true;
        }

        function implSplitParagraph(position) {

            var posLength = position.length - 1,
                offset = _.last(position),
                paraPosition = position.slice(0, -1),
                paragraph = (useParagraphCache && paragraphCache) || Position.getParagraphElement(editdiv, paraPosition),
                newParaPosition = Position.increaseLastIndex(paraPosition),
                newParagraph = $(paragraph).clone(true).find('div.page-break').remove().end().insertAfter(paragraph), //remove eventual page breaks before inserting to DOM
                startPosition = null,
                hasFloatedChildren = DOM.containsFloatingDrawingNode(paragraph),
                paragraphCacheSafe = (useParagraphCache && paragraphCache) || null, // Performance: Saving global paragraph cache
                // Performance: Saving data for list updates
                paraAttrs = null, listStyleId = null, listLevel = null,
                // whether the split paragraph contains selected drawing(s)
                updateSelectedDrawing = false,
                // whether the document is in read-only mode
                readOnly = self.getEditMode() !== true,
                currentElement,
                selectedDrawings;

            // checking if a selected drawing is affected by this split
            if (readOnly && selection.getSelectionType() === 'drawing') {
                selectedDrawings = $(paragraph).find('div.drawing.selected');
                if (selectedDrawings.length > 0) { updateSelectedDrawing = true; }
            }

            if (position.length > 2) {
                // if its paragraph split inside of table
                currentElement = $(paragraph).parents('table').last();
            } else {
                currentElement = paragraph;
            }

            // move leading floating drawings to the new paragraph, if passed position
            // points inside floating drawings or directly after the last floating drawing
            if ((hasFloatedChildren) && (0 < offset) && (offset <= Position.getLeadingFloatingDrawingCount(paragraph))) {
                offset = position[posLength] = 0;
            }

            // delete trailing part of original paragraph
            if (offset !== -1) {
                if (useParagraphCache) { paragraphCache = paragraph; } // Performance: implDeleteText can use cached paragraph
                implDeleteText(position, Position.appendNewIndex(paraPosition, -1), { allowEmptyResult: true });
                if (hasFloatedChildren) {
                    // delete all image divs that are no longer associated with following floating drawings
                    Position.removeUnusedDrawingOffsetNodes(paragraph);
                }
                if (offset === 0) { // if split is at the beginning of paragraph, remove page breaks immediately for better user experience
                    var pageBreaks = $(paragraph).find('div.page-break');
                    if (pageBreaks.length > 0) {
                        pageBreaks.remove();
                    }
                }
            }

            // delete leading part of new paragraph
            startPosition = Position.appendNewIndex(newParaPosition);
            if (offset > 0) {
                if (useParagraphCache) { paragraphCache = newParagraph; } // Performance: implDeleteText cannot use cached paragraph
                implDeleteText(startPosition, Position.increaseLastIndex(startPosition, offset - 1));
                if (hasFloatedChildren) {
                    // delete all empty text spans in cloned paragraph before floating drawings
                    // TODO: implDeleteText() should have done this already
                    Position.removeUnusedDrawingOffsetNodes(newParagraph);
                }
            }

            if (useParagraphCache) { paragraphCache = paragraphCacheSafe; } // Performance: Restoring global paragraph cache

            // update formatting of the paragraphs
            implParagraphChanged(paragraph);
            implParagraphChanged(newParagraph);

            // checking paragraph attributes for list styles
            paraAttrs = StyleSheets.getExplicitAttributes(paragraph);

            // updating lists, if required
            if (isListStyleParagraph(null, paraAttrs)) {
                if (paraAttrs && paraAttrs.paragraph) {
                    listStyleId = paraAttrs.paragraph.listStyleId;
                    listLevel = paraAttrs.paragraph.listLevel;
                }
                updateListsDebounced({useSelectedListStyleIDs: true, listStyleId: listStyleId, listLevel: listLevel});
            }

            // updating the drawing node in selection (only supporting single selection yet -> TODO: support of multi selection)
            // -> it can happen, that the old drawing node is no longer in the dom
            if (updateSelectedDrawing && selection.getSelectedDrawing().closest(DOM.PAGECONTENT_NODE_SELECTOR).length === 0) {
                selectedDrawings = $(newParagraph).find('div.drawing.selected');
                if (selectedDrawings.length > 0) {
                    selection.setSelectedDrawing(selectedDrawings);
                }
            }

            if ($(currentElement).data('lineBreaksData')) {
                $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
            }
            if ($(newParagraph).data('lineBreaksData')) {
                $(newParagraph).removeData('lineBreaksData'); // for the cloned paragraph also
            }
            //updating page breaks
            insertPageBreaksDebounced(currentElement);

            lastOperationEnd = startPosition;

            // in Internet Explorer it is necessary to add new empty text nodes in paragraphs again
            // newly, also in other browsers (new jQuery version?)
            repairEmptyTextNodes(newParagraph);

            return true;
        }

        /**
         * Returns a valid text position at the passed component position
         * (paragraph or table). Used to calculate the new text cursor position
         * after deleting a component.
         */
        function getValidTextPosition(position) {
            var textPosition = Position.getFirstPositionInParagraph(editdiv, position);
            // fall-back to last position in document (e.g.: last table deleted)
            if (!textPosition) {
                textPosition = Position.getLastPositionInParagraph(editdiv, [editdiv[0].childNodes.length - 1]);
            }
            return textPosition;
        }

        /**
         * Deletes a table at the specified position.
         *
         * @param {Number[]} position
         *  The logical position of the table to be deleted.
         *
         * @returns {Boolean}
         *  TRUE if the function has been processed successfully
         *  otherwise FALSE.
         */
        function implDeleteTable(position) {

            var tablePosition = Position.getLastPositionFromPositionByNodeName(editdiv, position, DOM.TABLE_NODE_SELECTOR),
                tableNode = Position.getTableElement(editdiv, tablePosition);

            if (tableNode) {
                $(tableNode).remove();
                lastOperationEnd = getValidTextPosition(tablePosition);
            } else {
                Utils.warning('Editor.implDeleteTable(): not tableNode found ' + JSON.stringify(position));
                return false;
            }

            // the deleted paragraphs can be part of a list, update all lists
            if (elementsContainListStyleParagraph(tableNode)) {
                updateListsDebounced({useSelectedListStyleIDs: true, listStyleId: getListStyleIDsOfParagraphs(tableNode), listLevel: null});
            }

            return true;
        }

        function implDeleteRows(pos, startRow, endRow) {

            var localPosition = _.copy(pos, true);

            if (! Position.isPositionInTable(editdiv, localPosition)) {
                Utils.warning('Editor.implDeleteRows(): position not in table ' + JSON.stringify(pos));
                return false;
            }

            var table = Position.getDOMPosition(editdiv, localPosition).node,
                rowNodes = DOM.getTableRows(table).slice(startRow, endRow + 1);

            rowNodes.remove();

            if (guiTriggeredOperation) {  // called from 'deleteRows' -> temporary branding for table
                $(table).data('gui', 'remove');
                // Performance: Simple tables do not need new formatting.
                // Tables with styles need to be updated below the removed row. Also the
                // row above need to be updated, because it may have become the final row.
                $(table).data('reducedTableFormatting', startRow > 0 ? startRow - 1 : 0);
            }

            // undo also needs table refresh for right border in Firefox (32374)
            if (undoRedoRunning  && _.browser.Firefox) { $(table).data('undoRedoRunning', true); }

            // Setting cursor
            var lastRow = Table.getRowCount(table) - 1;

            if (lastRow >= 0) {
                if (endRow > lastRow) {
                    endRow = lastRow;
                }
                localPosition.push(endRow);
                localPosition.push(0);  // using first cell in row
                localPosition.push(0);  // using first paragraph or table
                localPosition = Position.getFirstPositionInParagraph(editdiv, localPosition);

                // recalculating the attributes of the table cells
                if (requiresElementFormattingUpdate) {
                    implTableChanged(table);
                }

                lastOperationEnd = localPosition;
            }

            // the deleted paragraphs can be part of a list, update all lists
            if (elementsContainListStyleParagraph(rowNodes)) {
                updateListsDebounced({useSelectedListStyleIDs: true, listStyleId: getListStyleIDsOfParagraphs(rowNodes), listLevel: null});
            }

            return true;
        }

        /**
         * Removing all cell contents
         */
        function implClearCell(cellNode) {

            var // the container for the content nodes
                container = DOM.getCellContentNode(cellNode),
                // the last paragraph in the cell
                paragraph = null;

            paragraph = DOM.createImplicitParagraphNode();
            container.empty().append(paragraph);
            // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
            // newly, also in other browsers (new jQuery version?)
            repairEmptyTextNodes(paragraph);

            // validate the paragraph (add the dummy node)
            validateParagraphNode(paragraph);
            // and NOT formatting the implicit paragraph -> leads to 'jumping' of table after inserting row or column
            // paragraphStyles.updateElementFormatting(paragraph);
        }

        function implInsertRows(start, count, insertDefaultCells, referenceRow, attrs) {

            var localPosition = _.copy(start, true),
                useReferenceRow = _.isNumber(referenceRow) ? true : false,
                newRow = null,
                currentElement;

            if (start.length > 2) {
                // if its paragraph creation inside of table
                currentElement = Position.getContentNodeElement(editdiv, start.slice(0, 1));
            } else {
                currentElement = Position.getContentNodeElement(editdiv, start.slice(0, 1));
            }

            if (! Position.isPositionInTable(editdiv, localPosition)) {
                return false;
            }

            insertDefaultCells = insertDefaultCells ? true : false;

            if (!_.isNumber(count)) {
                count = 1; // setting default for number of rows
            }

            var tablePos = _.copy(localPosition, true);
            tablePos.pop();

            var table = Position.getDOMPosition(editdiv, tablePos).node,
                tableRowDomPos = Position.getDOMPosition(editdiv, localPosition),
                tableRowNode = null,
                row = null,
                cellsInserted = false;

            if (guiTriggeredOperation) {  // called from 'insertRow' -> temporary branding for table
                $(table).data('gui', 'insert');
            }

            if (tableRowDomPos) {
                tableRowNode = tableRowDomPos.node;
            }

            if (useReferenceRow) {

                if (guiTriggeredOperation) {
                    // Performance: Called from 'insertRow' -> simple table do not need new formatting.
                    // Tables with conditional styles only need to be updated below the referenceRow
                    // (and the row itself, too, for example southEast cell may have changed)
                    $(table).data('reducedTableFormatting', referenceRow);
                }

                row = DOM.getTableRows(table).eq(referenceRow).clone(true);

                // clear the cell contents in the cloned row
                row.children('td').each(function () {
                    implClearCell(this);
                });

                cellsInserted = true;

            } else if (insertDefaultCells) {

                var columnCount = Table.getColumnCount(table),
                    // prototype elements for row, cell, and paragraph
                    paragraph = DOM.createImplicitParagraphNode(),
                    cell = DOM.createTableCellNode(paragraph);

                // insert empty text node into the paragraph
                validateParagraphNode(paragraph);

                row = $('<tr>').attr('role', 'row');

                // clone the cells in the row element
                _.times(columnCount, function () { row.append(cell.clone(true).attr('role', 'gridcell')); });

                cellsInserted = true;

            } else {
                row = $('<tr>').attr('role', 'row');
            }

            _.times(count, function () {
                newRow = row.clone(true);

                // apply the passed attributes
                if (_.isObject(attrs)) {
                    tableRowStyles.setElementAttributes(newRow, attrs);
                }

                if (tableRowNode) {
                    // insert the new row before the existing row at the specified position
                    $(tableRowNode).before(newRow);
                } else {
                    // append the new row to the table
                    $(table).append(newRow);
                }

                // in Internet Explorer it is necessary to add new empty text nodes in rows again
                // newly, also in other browsers (new jQuery version?)
                repairEmptyTextNodes(newRow);

            });

            // recalculating the attributes of the table cells
            if (cellsInserted && requiresElementFormattingUpdate) {
                implTableChanged(table);
            }

            // Setting cursor
            if ((insertDefaultCells) || (useReferenceRow)) {
                localPosition.push(0);
                localPosition.push(0);
                localPosition.push(0);

                lastOperationEnd = localPosition;
            }
            insertPageBreaksDebounced(currentElement);

            return true;
        }

        function implInsertCells(start, count, attrs) {

            var localPosition = _.clone(start),
                tableNode = Position.getLastNodeFromPositionByNodeName(editdiv, start, DOM.TABLE_NODE_SELECTOR),
                tableCellDomPos = null,
                tableCellNode = null,
                paragraph = null,
                cell = null,
                row = null;

            if (!tableNode) {
                return;
            }

            if (!_.isNumber(count)) {
                count = 1; // setting default for number of rows
            }

            tableCellDomPos = Position.getDOMPosition(editdiv, localPosition);

            if (tableCellDomPos) {
                tableCellNode = tableCellDomPos.node;
            }

            // prototype elements for row, cell, and paragraph
            paragraph = DOM.createImplicitParagraphNode();

            // insert empty text node into the paragraph
            validateParagraphNode(paragraph);

            cell = DOM.createTableCellNode(paragraph);

            // apply the passed table cell attributes
            if (_.isObject(attrs)) {
                tableCellStyles.setElementAttributes(cell, attrs);
            }

            if (tableCellNode) {
                _.times(count, function () { $(tableCellNode).before(cell.clone(true)); });
                row = $(tableCellNode).parent();
            } else {
                var rowPos = localPosition.slice(0, -1);
                row = Position.getDOMPosition(editdiv, rowPos).node;
                _.times(count, function () { $(row).append(cell.clone(true)); });
            }

            // in Internet Explorer it is necessary to add new empty text nodes in rows again
            // newly, also in other browsers (new jQuery version?)
            repairEmptyTextNodes(row);

            // recalculating the attributes of the table cells
            if (requiresElementFormattingUpdate) {
                implTableChanged(tableNode);
            }

            return true;
        }

        /**
         * Delete table cells at the specified position.
         *
         * @param {Number[]} pos
         *  The logical position of the table to be deleted.
         *
         * @param start
         *  The logical start position of the cell range.
         *
         * @param end
         *  The logical end position of the cell range.
         *
         * @returns {Boolean}
         *  TRUE if the function has been processed successfully,
         *  otherwise FALSE.
         */
        function implDeleteCells(pos, start, end) {

            var localPosition = _.copy(pos, true),
                tableRowDomPos = null,
                row = null,
                table = null,
                maxCell = 0,
                cellNodes = null,
                allCellNodes = [];

            if (! Position.isPositionInTable(editdiv, localPosition)) {
                Utils.warn('Editor.implDeleteCells(): position not in table ' + JSON.stringify(pos));
                return false;
            }

            tableRowDomPos = Position.getDOMPosition(editdiv, localPosition);

            if (tableRowDomPos) {
                row = tableRowDomPos.node;
            }

            if (row) {

                maxCell = $(row).children().length - 1;

                if (start <= maxCell) {

                    if (end > maxCell) {
                        cellNodes = $(row).children().slice(start);

                    } else {
                        cellNodes = $(row).children().slice(start, end + 1);
                    }

                    cellNodes.remove(); // removing all following cells
                    allCellNodes.push(cellNodes);
                }
            }

            // setting cursor position
            localPosition.push(0);
            localPosition.push(0);
            localPosition.push(0);

            if (row && requiresElementFormattingUpdate) {
                table = $(row).closest('table');
                // undo of insertColumn also needs table refresh for right border in Firefox
                if (undoRedoRunning  && _.browser.Firefox) { table.data('undoRedoRunning', true); }
                implTableChanged(table);
            }

            lastOperationEnd = localPosition;

            // the deleted paragraphs can be part of a list, update all lists
            if (elementsContainListStyleParagraph(allCellNodes)) {
                updateListsDebounced({useSelectedListStyleIDs: true, listStyleId: getListStyleIDsOfParagraphs(allCellNodes), listLevel: null});
            }

            return true;
        }

        function implDeleteColumns(start, startGrid, endGrid) {

            var localPosition = _.copy(start, true),
                cellNodes = null,
                allCellNodes = [];

            if (! Position.isPositionInTable(editdiv, localPosition)) {
                return;
            }

            var table = Position.getDOMPosition(editdiv, localPosition).node,
                allRows = DOM.getTableRows(table),
                endColInFirstRow = -1;

            if (guiTriggeredOperation) {  // called from 'deleteColumns' -> temporary branding for table
                $(table).data('gui', 'remove');
            }

            allRows.each(
                function (i, row) {
                    var startCol = Table.getCellPositionFromGridPosition(row, startGrid, false),
                        endCol = Table.getCellPositionFromGridPosition(row, endGrid, false);

                    if ((i === 0) && (endCol !== -1)) {
                        endColInFirstRow = endCol;
                    }

                    if (startCol !== -1) {  // do nothing if startCol is out of range for this row

                        if (endCol === -1) {
                            // checking whether undo of operations is possible and remove cells
                            cellNodes = $(row).children().slice(startCol);
                        } else {
                            // checking whether undo of operations is possible and remove all cells in the range
                            cellNodes = $(row).children().slice(startCol, endCol + 1);
                        }

                        cellNodes.each(function (i, cell) {
                            checkDisableUndoStack('cell', cell);
                        }).remove();  // removing cell nodes
                        allCellNodes.push(cellNodes);
                    }
                }
            );

            // Setting cursor
            var lastColInFirstRow = DOM.getTableRows(table).first().children().length - 1;
            if ((endColInFirstRow > lastColInFirstRow) || (endColInFirstRow === -1)) {
                endColInFirstRow = lastColInFirstRow;
            }
            localPosition.push(0);
            localPosition.push(endColInFirstRow);
            localPosition.push(0);
            localPosition.push(0);

            // delete undo stack immediately if this is necessary and not a part of an undo group
            if ((! isInUndoGroup) && (deleteUndoStack)) {
                undoManager.clearUndoActions();
                deleteUndoStack = false;
            }

            // recalculating the attributes of the table cells
            if (requiresElementFormattingUpdate) {
                implTableChanged(table);
            }

            // the deleted paragraphs can be part of a list, update all lists
            if (elementsContainListStyleParagraph(allCellNodes)) {
                updateListsDebounced({useSelectedListStyleIDs: true, listStyleId: getListStyleIDsOfParagraphs(allCellNodes), listLevel: null});
            }

            lastOperationEnd = localPosition;
        }

        function implInsertColumn(start, gridPosition, insertMode) {

            var localPosition = _.copy(start, true);

            if (! Position.isPositionInTable(editdiv, localPosition)) {
                return false;
            }

            var table = Position.getDOMPosition(editdiv, localPosition).node,
                allRows = DOM.getTableRows(table);

            if (guiTriggeredOperation) {  // called from 'insertColumn' -> temporary branding for table
                $(table).data('gui', 'insert');
            }

            allRows.each(
                function (i, row) {
                    var cellPosition = Table.getCellPositionFromGridPosition(row, gridPosition),
                        cellClone = $(row).children(DOM.TABLE_CELLNODE_SELECTOR).slice(cellPosition, cellPosition + 1).clone(true);

                    implClearCell(cellClone);

                    if (insertMode === 'behind') {
                        cellClone.insertAfter($(row).children().get(cellPosition));
                    } else {
                        cellClone.insertBefore($(row).children().get(cellPosition));
                    }

                    // in Internet Explorer it is necessary to add new empty text nodes in columns again
                    // newly, also in other browsers (new jQuery version?)
                    repairEmptyTextNodes(cellClone);
                }
            );

            // recalculating the attributes of the table cells
            if (requiresElementFormattingUpdate) {
                implTableChanged(table);
            }

            // Setting cursor to first position in table
            localPosition.push(0);
            localPosition.push(gridPosition);
            localPosition.push(0);
            localPosition.push(0);

            lastOperationEnd = localPosition;

            return true;
        }

        function implDeleteText(startPosition, endPosition, options) {

            var // info about the parent paragraph node
                position = null, paragraph = null,
                // last index in start and end position
                startOffset = 0, endOffset = 0,
                // whether a floated drawing was removed
                floatedDrawingRemoved = false,
                // next sibling text span of last visited child node
                nextTextSpan = null,
                // a counter for the removed elements
                removeCounter = 0,
                // the number of text positions to be removed
                removePositions = 0,
                // in internal calls (from other operations except delete), it can be allowed,
                // that there is nothing to delete
                allowEmptyResult = Utils.getBooleanOption(options, 'allowEmptyResult', false);

            // get paragraph node from start position
            if (!_.isArray(startPosition) || (startPosition.length < 2)) {
                Utils.warn('Editor.implDeleteText(): missing start position');
                return false;
            }
            position = startPosition.slice(0, -1);
            paragraph = (useParagraphCache && paragraphCache) || Position.getParagraphElement(editdiv, position);
            if (!paragraph) {
                Utils.warn('Editor.implDeleteText(): no paragraph found at position ' + JSON.stringify(position));
                return false;
            }

            // validate end position
            if (_.isArray(endPosition) && !Position.hasSameParentComponent(startPosition, endPosition)) {
                Utils.warn('Editor.implDeleteText(): range not in same paragraph');
                return false;
            }

            // start and end offset in paragraph
            startOffset = _.last(startPosition);
            endOffset = _.isArray(endPosition) ? _.last(endPosition) : startOffset;
            if (endOffset === -1) { endOffset = undefined; }
            if (endOffset !== undefined) {
                removePositions = endOffset - startOffset + 1;
            }

            // visit all covered child nodes (iterator allows to remove the visited node)
            Position.iterateParagraphChildNodes(paragraph, function (node, nodeStart, nodeLength) {

                var // previous text span of current node
                    prevTextSpan = null;

                // remove preceding position offset node of floating drawing objects
                if (DOM.isFloatingDrawingNode(node)) {
                    $(node).prev(DOM.OFFSET_NODE_SELECTOR).remove();
                    floatedDrawingRemoved = true;
                }

                // get sibling text spans
                prevTextSpan = DOM.isTextSpan(node.previousSibling) ? node.previousSibling : null;
                nextTextSpan = DOM.isTextSpan(node.nextSibling) ? node.nextSibling : null;

                // fix for msie, where empty text spans migth be removed
                // newly, also in other browsers (new jQuery version?)
                if (DOM.isTextSpanWithoutTextNode(node)) {
                    repairEmptyTextNodes(node);
                }
                if ((!prevTextSpan) && (DOM.isTextSpanWithoutTextNode(node.previousSibling))) {
                    prevTextSpan = node.previousSibling;
                    repairEmptyTextNodes(prevTextSpan);
                }
                if ((!nextTextSpan) && (DOM.isTextSpanWithoutTextNode(node.nextSibling))) {
                    nextTextSpan = node.previousSibling;
                    repairEmptyTextNodes(nextTextSpan);
                }

                // clear text in text spans
                if (DOM.isTextSpan(node)) {

                    // only remove the text span, if it has a sibling text span
                    // (otherwise, it separates other component nodes)
                    if (prevTextSpan || nextTextSpan) {
                        $(node).remove();
                    } else {
                        // remove text, but keep text span element
                        node.firstChild.nodeValue = '';
                    }
                    removeCounter += nodeLength;
                    return;
                }

                // other component nodes (drawings or text components)
                if (DOM.isTextComponentNode(node) || DrawingFrame.isDrawingFrame(node)) {

                    // safely release image data (see comments in BaseApplication.destroyImageNodes())
                    app.destroyImageNodes(node);

                    // remove the visited node
                    $(node).remove();

                    // remove previous empty sibling text span (not next sibling which would break iteration)
                    if (DOM.isEmptySpan(prevTextSpan) && nextTextSpan) {
                        $(prevTextSpan).remove();
                    }
                    removeCounter += nodeLength;
                    return;
                }

                Utils.error('Editor.implDeleteText(): unknown paragraph child node');
                return Utils.BREAK;

            }, undefined, { start: startOffset, end: endOffset, split: true });

            // removed must be > 0 and it must be the difference from endOffset - startOffset + 1
            // check, if at least one element was removed
            if ((! allowEmptyResult) && (removeCounter === 0)) {
                Utils.warn('Editor.implDeleteText(): no component found from position: ' + JSON.stringify(startPosition) + ' to ' + JSON.stringify(endPosition));
                return false;
            }

            // check, if the removed part has the correct length
            if ((! allowEmptyResult) && (removePositions > 0) && (removePositions !== removeCounter)) {
                Utils.warn('Editor.implDeleteText(): incorrect number of removals: Required: ' + removeCounter + ' Done: ' + removeCounter);
                return false;
            }

            // remove next sibling text span after deleted range, if empty,
            // otherwise try to merge with equally formatted preceding text span
            if (nextTextSpan && DOM.isTextSpan(nextTextSpan.previousSibling)) {
                if (DOM.isEmptySpan(nextTextSpan)) {
                    $(nextTextSpan).remove();
                } else {
                    CharacterStyles.mergeSiblingTextSpans(nextTextSpan);
                }
            }

            // validate paragraph node, store operation position for cursor position
            if ((guiTriggeredOperation) || (undoRedoRunning) || (!app.isImportFinished())) {
                implParagraphChanged(paragraph);  // Performance and task 30597: Local client can defer attribute setting, Task 30603: deferring also during loading document
            } else {
                implParagraphChangedSync($(paragraph));  // Performance and task 30597: Remote client must set attributes synchronously
            }

            lastOperationEnd = _.clone(startPosition);

            return true;
        }

        /**
         * Checking whether a node contains an element, that cannot be restored
         * in undo action. This are tables with exceeded size or drawings that
         * are not of type 'image'.
         *
         * @param {Node|jQuery|Number[]} node
         *  The node, whose descendants are investigated. For convenience reasons
         *  this can also be a logical position that is transformed into a node
         *  automatically.
         *
         * @returns {Boolean}
         *  Returns true, if the node contains an unrestorable descendant,
         *  otherwise false.
         */
        function containsUnrestorableElements(node, startOffset, endOffset) {

            var // whether the node contains an unrestorable descendant
                contains = false,
                // a dom point calculated from logical position
                domPos = null,
                // a locally defined helper node
                searchNode = null,
                // the logical position of a non-image drawing
                drawingPos = null;

            // supporting for convenience also logical positions, that are automatically
            // converted into dom nodes
            if (_.isArray(node)) {
                domPos = Position.getDOMPosition(editdiv, node, true);
                if (domPos && domPos.node) { searchNode = domPos.node; }
            } else {
                searchNode = node;
            }

            if (searchNode) {
                if ($(searchNode).find(DOM.TABLE_SIZE_EXCEEDED_NODE_SELECTOR).length > 0) {
                    contains = true;
                } else {
                    // checking if the table or paragraph contains drawings, that are not of type 'image'
                    // or if the table contains another table with exceeded size
                    $(DrawingFrame.NODE_SELECTOR, searchNode).each(function (i, drawing) {
                        if (DOM.isNonImageDrawingNode(drawing)) {
                            // if only a part of a paragraph is deleted, it has to be checked,
                            // if the non-image drawing is inside this selected part.
                            if (_.isNumber(startOffset) && (_.isNumber(endOffset))) {
                                // the logical position of the non-image drawing
                                drawingPos = Position.getOxoPosition(editdiv, drawing, 0);
                                if (_.isArray(drawingPos) && (startOffset <= _.last(drawingPos)) && (_.last(drawingPos) <= endOffset)) {
                                    contains = true;
                                }
                            } else {
                                contains = true;
                            }
                        }
                    });
                }
            }

            return contains;
        }

        /**
         * Checking whether there can be an undo for the delete operation.
         * Undo is not possible, if a table with exceeded size is removed,
         * or if a drawing, that is not of type image is removed.
         *
         * @param {String} type
         *  The type of the node to be removed. This can be 'text', 'paragraph',
         *  'cell', 'row' and table.
         *
         * @param {Node|jQuery} node
         *  The node to be removed.
         *
         * @param {Number[]} start
         *  The logical start position of the element or text range to be
         *  deleted.
         *
         * @param {Number[]} [end]
         *  The logical end position of the element or text range to be
         *  deleted. This is only relevant for type 'text'.
         */
        function checkDisableUndoStack(type, node, start, end) {

            var // info about the parent paragraph or table node
                position = null, paragraph = null,
                // last index in start and end position
                startOffset = 0, endOffset = 0;

            // deleting the different types:
            switch (type) {

            case 'text':
                // get paragraph node from start position
                if (!_.isArray(start) || (start.length === 0)) {
                    Utils.warn('Editor.disableUndoStack(): missing start position');
                    return false;
                }
                position = start.slice(0, -1);
                paragraph = Position.getParagraphElement(editdiv, position);
                if (!paragraph) {
                    Utils.warn('Editor.disableUndoStack(): no paragraph found at position ' + JSON.stringify(position));
                    return false;
                }

                // validate end position
                if (_.isArray(end) && !Position.hasSameParentComponent(start, end)) {
                    Utils.warn('Editor.disableUndoStack(): range not in same paragraph');
                    return false;
                }

                // start and end offset in paragraph
                startOffset = _.last(start);
                endOffset = _.isArray(end) ? _.last(end) : startOffset;
                if (endOffset === -1) { endOffset = undefined; }

                // visit all covered child nodes (iterator allows to remove the visited node)
                Position.iterateParagraphChildNodes(paragraph, function (localnode) {
                    if (DOM.isNonImageDrawingNode(localnode)) {
                        deleteUndoStack = true;
                    }
                }, undefined, { start: startOffset, end: endOffset });
                break;

            case 'paragraph':
                if (containsUnrestorableElements(node)) {
                    deleteUndoStack = true;
                }
                break;

            case 'cell':
                if (containsUnrestorableElements(node)) {
                    deleteUndoStack = true;
                }
                break;

            case 'row':
                if (containsUnrestorableElements(node)) {
                    deleteUndoStack = true;
                }
                break;

            case 'table':
                if (DOM.isExceededSizeTableNode(node) || containsUnrestorableElements(node)) {
                    deleteUndoStack = true;
                }
                break;
            }
        }

        /**
         * Removes a specified element or a text range. The type of the element
         * will be determined from the parameters start and end.
         * specified range. Currently, only single components can be deleted,
         * except for text ranges in a paragraph. A text range can include
         * characters, fields, and drawing objects, but must be contained in a
         * single paragraph.
         *
         * @param {Number[]} start
         *  The logical start position of the element or text range to be
         *  deleted.
         *
         * @param {Number[]} [end]
         *  The logical end position of the element or text range to be
         *  deleted. Can be omitted, if the end position is equal to the start
         *  position (single component)
         *
         *  @return {Boolean}
         *   TRUE if the function has been processed successfully, otherwise
         *   FALSE.
         */
        function implDelete(start, end) {

            var // node info for start/end position
                startInfo = null, endInfo = null,
                // position description for cells
                rowPosition, startCell, endCell,
                // position description for rows
                tablePosition, startRow, endRow,
                // a new implicit paragraph node
                newParagraph = null,
                // Performance: Saving data for list updates
                paraAttrs = null, listStyleId = null, listLevel = null,
                // a temporary helper position
                localPosition = null,
                // result of function (default true)
                result = true,
                // starting point for inserting page breaks downwards
                position = start.length > 1 ? start.slice(0, 1) : start.slice(0),
                // used for pagebreaks
                currentElement = null,
                // the height of the paragraph before deleting content
                prevHeight = null;

            // resolve start and end position
            if (!_.isArray(start)) {
                Utils.warn('Editor.implDelete(): missing start position');
                return false;
            }
            startInfo = Position.getDOMPosition(editdiv, start, true);
            if (!startInfo || !startInfo.node) {
                Utils.warn('Editor.implDelete(): invalid start position: ' + JSON.stringify(start));
                return false;
            }
            endInfo = _.isArray(end) ? Position.getDOMPosition(editdiv, end, true) : startInfo;
            if (!endInfo || !endInfo.node) {
                Utils.warn('Editor.implDelete(): invalid end position: ' + JSON.stringify(end));
                return false;
            }

            end = end || start;

            // get attribute type of start and end node
            startInfo.type = resolveElementType(startInfo.node);
            endInfo.type = resolveElementType(endInfo.node);

            // check that start and end point to the same element type
            if ((!startInfo.type || !endInfo.type) || (startInfo.type !== endInfo.type)) {
                Utils.warn('Editor.implDelete(): problem with node types: ' + startInfo.type + ' and ' + endInfo.type);
                return false;
            }

            // check that start and end point to the same element for non text types (only one cell, row, paragraph, ...)
            if ((startInfo.type !== 'text') && (startInfo.node !== endInfo.node)) {
                Utils.warn('Editor.implDelete(): no ranges supported for node type "' + startInfo.type + '"');
                return false;
            }

            // check that for text nodes start and end point have the same parent
            if ((startInfo.type === 'text') && (startInfo.node.parentNode !== endInfo.node.parentNode)) {
                Utils.warn('Editor.implDelete(): deleting range only supported inside one paragraph.');
                return false;
            }

            // checking whether undo of operations is possible
            checkDisableUndoStack(startInfo.type, startInfo.node, _.clone(start), _.clone(end));

            // deleting the different types:
            switch (startInfo.type) {

            case 'text':
                var isLastCharInPar = !startInfo.node.nextSibling && (startInfo.offset === startInfo.node.textContent.length - 1);
                // setting the paragraph node (required for page breaks)
                currentElement = startInfo.node.parentNode;
                if (!$(currentElement.parentNode).hasClass('pagecontent')) { //its a div.p inside table(s)
                    currentElement = $(currentElement).parents('table').last()[0];
                }
                // caching paragraph height before removing text
                prevHeight = currentElement.offsetHeight;
                // removing text
                result = implDeleteText(start, end);
                // using the same paragraph node again
                if ($(currentElement).data('lineBreaksData')) {
                    $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
                }
                // trigger repainting only if height changed or element is on two pages,
                // or paragraph is on more pages, and last char in paragraph is not deleted (performance improvement)
                if (prevHeight !== currentElement.offsetHeight || ($(currentElement).hasClass('has-breakable-span')) && !isLastCharInPar) {
                    insertPageBreaksDebounced(currentElement);
                }

                break;

            case 'paragraph':
                if (DOM.isImplicitParagraphNode(startInfo.node)) {
                    Utils.warn('Editor.implDelete(): Error: Operation tries to delete an implicit paragraph!');
                    return false;
                }

                // Setting lastOperationEnd for the special case, that insertParagraph was used instead of splitParagraph
                // -> if there is a previous paragraph, use the last position of this previous paragraph (task 30742)
                if (($(startInfo.node).prev().length > 0) && (DOM.isParagraphNode($(startInfo.node).prev()))) {
                    localPosition = _.clone(start);
                    localPosition[localPosition.length - 1]--;
                    localPosition = Position.getLastPositionInParagraph(editdiv, localPosition);
                    lastOperationEnd = localPosition;
                }

                if ((DOM.isParagraphWithoutNeighbour(startInfo.node)) || (DOM.isFinalParagraphBehindTable(startInfo.node))) {
                    newParagraph = DOM.createImplicitParagraphNode();
                    $(startInfo.node).parent().append(newParagraph);
                    validateParagraphNode(newParagraph);
                    implParagraphChanged(newParagraph);
                    // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
                    // newly, also in other browsers (new jQuery version?)
                    repairEmptyTextNodes(newParagraph);

                }
                $(startInfo.node).remove(); // remove the paragraph from the DOM
                // the deleted paragraphs can be part of a list, update all lists
                paraAttrs = StyleSheets.getExplicitAttributes(startInfo.node);
                // updating lists, if required
                if (isListStyleParagraph(null, paraAttrs)) {
                    if (paraAttrs && paraAttrs.paragraph) {
                        listStyleId = paraAttrs.paragraph.listStyleId;
                        listLevel = paraAttrs.paragraph.listLevel;
                    }
                    updateListsDebounced({useSelectedListStyleIDs: true, listStyleId: listStyleId, listLevel: listLevel});
                }
                // get currentElement again (maybe whole paragraph has deleted), and pass it to repaint pagebreaks from that page only
                currentElement = Position.getContentNodeElement(editdiv, position);
                insertPageBreaksDebounced(currentElement);

                break;

            case 'cell':
                rowPosition = _.clone(start);
                startCell = rowPosition.pop();
                endCell = startCell;
                result = implDeleteCells(rowPosition, startCell, endCell);
                break;

            case 'row':
                tablePosition = _.clone(start);
                startRow = tablePosition.pop();
                endRow = startRow;
                result = implDeleteRows(tablePosition, startRow, endRow);
                // get currentElement again (maybe whole paragraph has deleted), and pass it to repaint pagebreaks from that page only
                currentElement = Position.getContentNodeElement(editdiv, position);
                insertPageBreaksDebounced(currentElement);

                break;

            case 'table':
                result = implDeleteTable(start);
                // get currentElement again (maybe whole paragraph has deleted), and pass it to repaint pagebreaks from that page only
                currentElement = Position.getContentNodeElement(editdiv, position);
                insertPageBreaksDebounced(currentElement);

                break;

            default:
                Utils.error('Editor.implDelete(): unsupported node type: ' + startInfo.type);
            }

            // delete undo stack immediately if this is necessary and not a part of an undo group
            if ((! isInUndoGroup) && (deleteUndoStack)) {
                undoManager.clearUndoActions();
                deleteUndoStack = false;
            }

            return result;
        }

        function implMove(_start, _end, _to) {

            var start = _.copy(_start, true),
                to = _.copy(_to, true),
                sourcePos = Position.getDOMPosition(editdiv, start, true),
                destPos = Position.getDOMPosition(editdiv, to, true),
                insertBefore = true,
                splitNode = false;

            // Fix for 28634 -> Moving a drawing to positio with tab or hard break
            if (DOM.isHardBreakNode(destPos.node) || DOM.isTabNode(destPos.node) || DOM.isFieldNode(destPos.node)) {
                destPos.node = destPos.node.previousSibling;
            }

            if (destPos.offset === 0) {
                insertBefore = true;
            } else if ((destPos.node.length) && (destPos.offset === (destPos.node.length - 1))) {
                insertBefore = false;
            } else if ((DrawingFrame.isDrawingFrame(destPos.node)) && (destPos.offset === 1)) {
                insertBefore = false;
            } else {
                splitNode = true;  // splitting node is required
                insertBefore = false;  // inserting after new created text node
            }

            if ((sourcePos) && (destPos)) {

                var sourceNode = sourcePos.node,
                    destNode = destPos.node,
                    useOffsetDiv = true,
                    offsetDiv = sourceNode.previousSibling,
                    doMove = true;

                if ((sourceNode) && (destNode)) {

                    if (!DrawingFrame.isDrawingFrame(sourceNode)) {
                        doMove = false; // supporting only drawings at the moment
                        Utils.warn('Editor.implMove(): moved  node is not a drawing: ' + Utils.getNodeName(sourceNode));
                    } else {
                        // also move the offset divs
                        if ((!offsetDiv) || (!DOM.isOffsetNode(offsetDiv))) {
                            useOffsetDiv = false;
                        }
                    }

                    if (doMove) {

                        if (splitNode) {
                            destNode = DOM.splitTextSpan(destNode, destPos.offset + 1)[0];
                        } else {
                            if (destNode.nodeType === 3) {
                                destNode = destNode.parentNode;
                            }
                        }

                        // using empty span as reference for inserting new components
                        if ((DrawingFrame.isDrawingFrame(destNode)) && (DOM.isOffsetNode(destNode.previousSibling))) {
                            destNode = destNode.previousSibling;  // switching temporary to offset
                        }

                        // there can be empty text spans before the destination node
                        if ((DOM.isTextSpan(destNode)) || (DrawingFrame.isDrawingFrame(destNode)) || (DOM.isOffsetNode(destNode))) {
                            while (DOM.isEmptySpan(destNode.previousSibling)) {
                                destNode = destNode.previousSibling;
                            }
                        }

                        if ((insertBefore) && (DOM.isTextSpan(destNode))) {
                            destNode = DOM.splitTextSpan(destNode, 0)[0]; // taking care of empty text span before drawing
                            insertBefore = false;  // insert drawing behind new empty text span
                        }

                        // removing empty text spans behind or after the source node
                        if ((sourceNode.previousSibling) && (sourceNode.nextSibling)) {
                            if ((DOM.isTextSpan(sourceNode.previousSibling)) && (DOM.isEmptySpan(sourceNode.nextSibling))) {
                                $(sourceNode.nextSibling).remove();
                            } else if ((DOM.isEmptySpan(sourceNode.previousSibling)) && (DOM.isTextSpan(sourceNode.nextSibling))) {
                                $(sourceNode.previousSibling).remove();
                            }
                        }

                        if ((sourceNode.previousSibling) && (sourceNode.previousSibling.previousSibling) && (sourceNode.nextSibling) && (DOM.isOffsetNode(sourceNode.previousSibling))) {
                            if ((DOM.isTextSpan(sourceNode.previousSibling.previousSibling)) && (DOM.isEmptySpan(sourceNode.nextSibling))) {
                                $(sourceNode.nextSibling).remove();
                            } else if ((DOM.isEmptySpan(sourceNode.previousSibling.previousSibling)) && (DOM.isTextSpan(sourceNode.nextSibling))) {
                                $(sourceNode.previousSibling.previousSibling).remove();
                            }
                        }

                        // moving the drawing
                        if (insertBefore) {
                            $(sourceNode).insertBefore(destNode);
                        } else {
                            $(sourceNode).insertAfter(destNode);
                        }

                        // moving also the corresponding div before the moved drawing
                        if (useOffsetDiv) {
                            $(offsetDiv).insertBefore(sourceNode);
                        }

                        implParagraphChanged(to);
                    }
                }
            }

            return true;
        }

        function implMergeCell(start, count) {

            var rowPosition = _.copy(start, true),
                localStartCol = rowPosition.pop(),
                localEndCol = localStartCol + count,
                // Counting the colSpan off all cells in the range
                row = Position.getDOMPosition(editdiv, rowPosition).node,
                allSelectedCells = $(row).children().slice(localStartCol, localEndCol + 1),
                colSpanSum = Table.getColSpanSum(allSelectedCells),
                // Shifting the content of all following cells to the first cell
                targetCell = $(row).children().slice(localStartCol, localStartCol + 1),
                sourceCells = $(row).children().slice(localStartCol + 1, localEndCol + 1);

            Table.shiftCellContent(targetCell, sourceCells);

            sourceCells.remove();

            // apply the passed table cell attributes
            tableCellStyles.setElementAttributes(targetCell, { cell: { gridSpan: colSpanSum } });

            return true;
        }

        /**
         * modify current listlevel - if any - depending on the increase flag
         */
        function implModifyListLevel(currentlistLevel, increase, minLevel) {
            undoManager.enterUndoGroup(function () {
                if (increase && currentlistLevel < 8) {
                    currentlistLevel += 1;
                    self.setAttribute('paragraph', 'listLevel', currentlistLevel);
                } else if (!increase && currentlistLevel > minLevel) {
                    currentlistLevel -= 1;
                    self.setAttribute('paragraph', 'listLevel', currentlistLevel);
                    if (currentlistLevel < 0) {
                        self.setAttribute('paragraph', 'listStyleId', null);
                    }
                }
            });
        }

        /**
         * Updates all paragraphs that are part of any bullet or numbering
         * list.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the list update. The
         *  following options are supported:
         *  @param {Boolean} [options.async=false]
         *      If set to true, all lists are updated asynchronously. This
         *      should happen only once when the document is loaded.
         *
         * @returns {jQuery.Promise}
         *  A promise that is resolved, when all paragraphs are updated.
         */
        function updateLists(options) {

            var listItemCounter = [],
                listParagraphIndex = [], // paragraph index in a list
                // list of all paragraphs in the document
                paragraphNodes = null,
                // the deferred, whose promise will be returned
                def = null,
                // Performance: Whether this list update was triggered by an action with specified style id
                useSelectedListStyleIDs = updateListsDebouncedOptions.useSelectedListStyleIDs,
                // Performance: Whether there are list styles registered for updating
                doNothing = false,
                // Performance: An array with listStyleIds of splitted paragraphs
                allListStyleIds = null;  // Performance: Only for splitted paragraphs
                // Performance: An array with listStyleLevels of splitted paragraphs (same order as in listStyleIds)
                // allListStyleLevels = null;

            function updateListInParagraph(para) {

                // always remove an existing label
                var elementAttributes = paragraphStyles.getElementAttributes(para),
                    paraAttributes = elementAttributes.paragraph,
                    listStyleId = paraAttributes.listStyleId,
                    oldLabel = null,
                    updateParaTabstops = false,
                    removeLabel = false,
                    updateList = false;

                // Updating paragraphs, that are no longer part of lists (for example after 'Backspace')
                if ($(para).data('removeLabel')) {
                    removeLabel = true;
                    $(para).removeData('removeLabel');
                }

                if ($(para).data('updateList')) {
                    updateList = true;
                    $(para).removeData('updateList');
                }

                // Performance II: Do nothing, if the list style ID does not fit -> only modify paragraphs with correct list style ID
                // -> could be improved with check of list level (taking care of styles including upper levels)
                if ((useSelectedListStyleIDs) && (! _.contains(allListStyleIds, listStyleId)) && (! removeLabel) && (! updateList)) { return; }

                oldLabel = $(para).children(DOM.LIST_LABEL_NODE_SELECTOR);
                updateParaTabstops = oldLabel.length > 0;
                oldLabel.remove();

                if (listStyleId  !== '') {

                    var listLevel = paraAttributes.listLevel,
                        noListLabel = paraAttributes.listLabelHidden === true;
                    if (listLevel < 0) {
                        // is a numbering level assigned to the current paragraph style?
                        listLevel = listCollection.findIlvl(listStyleId, elementAttributes.styleId);
                    }
                    if (listLevel !== -1 && listLevel < 10) {
                        updateParaTabstops = true;
                        if (!listItemCounter[paraAttributes.listStyleId]) {
                            listItemCounter[paraAttributes.listStyleId] = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
                            listParagraphIndex[paraAttributes.listStyleId] = 0;
                        }
                        if (!noListLabel) {
                            listItemCounter[paraAttributes.listStyleId][listLevel]++;
                            listParagraphIndex[paraAttributes.listStyleId]++;
                        }
                        if (paraAttributes.listStartValue >= 0)
                            listItemCounter[paraAttributes.listStyleId][listLevel] = paraAttributes.listStartValue;
                        // TODO: reset sub-levels depending on their 'levelRestartValue' attribute
                        var subLevelIdx = listLevel + 1;
                        for (; subLevelIdx < 10; subLevelIdx++)
                            listItemCounter[paraAttributes.listStyleId][subLevelIdx] = 0;
                        // fix level counts of non-existing upper levels
                        subLevelIdx = listLevel - 1;
                        for (; subLevelIdx >= 0; subLevelIdx--)
                            if (listItemCounter[paraAttributes.listStyleId][subLevelIdx] === 0)
                                listItemCounter[paraAttributes.listStyleId][subLevelIdx] = 1;

                        var listObject = listCollection.formatNumber(paraAttributes.listStyleId, listLevel,
                            listItemCounter[paraAttributes.listStyleId], listParagraphIndex[paraAttributes.listStyleId]),
                            tab = !listObject.labelFollowedBy || listObject.labelFollowedBy === 'listtab';

                        if (!tab && (listObject.labelFollowedBy === 'space')) {
                            listObject.text += String.fromCharCode(0x00a0);//add non breaking space
                        }

                        var numberingElement = DOM.createListLabelNode(noListLabel ? '' : listObject.text);

                        var span = DOM.findFirstPortionSpan(para);
                        var elementCharAttributes = characterStyles.getElementAttributes(span).character;
                        var listSpans = numberingElement.children('span');
                        if (listObject.imgsrc) {
                            var absUrl = app.getFilterModuleUrl({ action: 'getfile', get_filename: listObject.imgsrc }),
                            imageWidth =  Utils.convertHmmToLength(listObject.imgwidth, 'pt');
                            if (!imageWidth)
                                imageWidth = elementCharAttributes.fontSize;

                            var image = $('<div>', { contenteditable: false })
                            .addClass('drawing')
                            .data('url', listObject.imgsrc)
                            .append($('<div>').addClass('content')
                                    .append($('<img>', { src: absUrl }).css('width', imageWidth + 'pt'))
                                    );

                            characterStyles.updateElementLineHeight(image, paraAttributes.lineHeight, elementCharAttributes);
                            $(image).css('height', elementCharAttributes.fontSize + 'pt');
                            $(image).css('width', elementCharAttributes.fontSize + 'pt');
                            numberingElement.prepend(image);

                        } else if (elementAttributes.character || elementAttributes.styleId) {
                            var paraCharStyles = paragraphStyles.getStyleSheetAttributes(elementAttributes.styleId);
                            // Map certain character styles from the paragraph style to the label ignoring others
                            // which doesn't fit this special use case. We prefer paragraph character attributes over
                            // style character attributes.
                            // Don't map background color and underline as MS Word also don't support them for
                            // list-labels.
                            if (elementAttributes.character.fontSize || paraCharStyles.character.fontSize) {
                                listSpans.css('font-size', (elementAttributes.character.fontSize || paraCharStyles.character.fontSize) + 'pt');
                            }
                            if (elementAttributes.character.fontName || paraCharStyles.character.fontName) {
                                listSpans.css('font-family', characterStyles.getCssFontFamily(elementAttributes.character.fontName || paraCharStyles.character.fontName));
                            }
                            if (elementAttributes.character.bold || paraCharStyles.character.bold) {
                                listSpans.css('font-weight', (elementAttributes.character.bold || paraCharStyles.character.bold) ? 'bold' : 'normal');
                            }
                            if (elementAttributes.character.italic || paraCharStyles.character.italic) {
                                listSpans.css('font-style', (elementAttributes.character.italic || paraCharStyles.character.italic) ? 'italic' : 'normal');
                            }
                            if (paraCharStyles.character.underline) {
                                var value = Utils.getTextDecorationType(listSpans);
                                listSpans.css('text-decoration', Utils.toggleToken(value, 'underline', paraCharStyles.character.underline, 'none'));
                            }
                            if (elementAttributes.character.strike || paraCharStyles.character.strike) {
                                var value = Utils.getTextDecorationType(listSpans);
                                listSpans.css('text-decoration', Utils.toggleToken(value, 'line-through', (elementAttributes.character.strike || paraCharStyles.character.strike) !== 'none', 'none'));
                            }
                            if (elementAttributes.character.color || paraCharStyles.character.color) {
                                listSpans.css('color', characterStyles.getCssColor((elementAttributes.character.color || paraCharStyles.character.color), 'text'));
                            }
                            if (paraCharStyles.character.fillColor) {
                                listSpans.css('background-color', characterStyles.getCssColor(paraCharStyles.character.fillColor, 'fill'));
                            }
                        }

                        if (listObject.color) {
                            listSpans.css('color', documentStyles.getCssTextColor(listObject.color, [paraAttributes.fillColor, listObject.fillColor]));
                        }
                        characterStyles.updateElementLineHeight(numberingElement, paraAttributes.lineHeight, elementAttributes.character);
                        var minWidth = 0,
                            isNegativeIndent = listObject.firstLine < listObject.indent;

                        if (isNegativeIndent) {
                            var labelWidth = listObject.indent - listObject.firstLine;
                            if (tab)
                                minWidth = labelWidth;
                            numberingElement.css('margin-left', (-listObject.indent + listObject.firstLine) / 100 + 'mm');
                        } else {
                            numberingElement.css('margin-left', (listObject.firstLine - listObject.indent) / 100 + 'mm');
                        }
                        numberingElement.css('min-width', minWidth / 100 + 'mm');

//                        // checking alignment of the numbering element
//                        // -> preparations for right-alignment of list labels
//                        var levelAttrs = lists.getListLevel(listStyleId, listLevel);
//                        if (levelAttrs && levelAttrs.textAlign && levelAttrs.textAlign === 'right') {
//                            numberingElement.css('text-align', 'right');
//                            numberingElement.css('margin-right', '3mm');
//                        } else {
//                            numberingElement.css('text-align', 'left');
//                            numberingElement.css('margin-right', '0');
//                        }

                        $(para).prepend(numberingElement);
                        if (tab || noListLabel) {
                            var realWidth = Utils.convertLengthToHmm(numberingElement[0].offsetWidth, 'px'),
                            realEndPos = listObject.firstLine + realWidth,
                            defaultTabstop = documentStyles.getAttributes().defaultTabStop,
                            paraAttributes = paragraphStyles.getElementAttributes(para).paragraph,
                            targetPosition = 0;
                            //listLabel = $(para).children(DOM.LIST_LABEL_NODE_SELECTOR);

                            if (isNegativeIndent && listObject.indent >= realEndPos) {
                                targetPosition = listObject.indent;
                            } else {
                                var tabstop = null;
                                if (paraAttributes && paraAttributes.tabStops) {
                                    tabstop = _.find(paraAttributes.tabStops, function (tab) {
                                        return realEndPos <= tab.pos;
                                    });
                                }
                                if (tabstop) {
                                    targetPosition = tabstop.pos;
                                }
                                else if (defaultTabstop > 0) {
                                    targetPosition = (1 + (Math.floor(realEndPos / defaultTabstop))) * defaultTabstop;
                                }
                                else {
                                    targetPosition = realEndPos;
                                }
                            }
                            numberingElement.css('min-width', (targetPosition - listObject.firstLine) / 100 + 'mm');
                            //numberingElement.css('cursor', 'row-resize');

                            //$(listLabel).css('min-width', (targetPosition - listObject.firstLine) / 100 + 'mm');
                        }
                    }
                }

                if (updateParaTabstops) {
                    paragraphStyles.updateTabStops(para);
                }
            }

            function updateListsCallback(paragraphs) {
                paragraphs.each(function () {
                    updateListInParagraph(this);
                });
            }

            // receiving list of all document paragraphs
            paragraphNodes = editdiv.find(DOM.PARAGRAPH_NODE_SELECTOR);

            // Performance I: Simplified process, if list update was triggered by splitting or deleting of paragraphs
            // Only check those paragraphs, that have a list label node.
            if (useSelectedListStyleIDs) {
                // Reducing the number of paragraphs, that need to be updated. This can happen now before it is checked,
                // if the paragraph has a list style ID set. Affected are only paragraphs, that contain already the
                // list label node (DOM.LIST_LABEL_NODE_SELECTOR), or, if they are newly inserted, they are branded
                // with the 'updateList' or 'removeLabel' (after backspace) data.
                if (updateListsDebouncedOptions.paraInsert) {
                    paragraphNodes = paragraphNodes.filter(function () {
                        return (DOM.isListLabelNode(this.firstChild) || $(this).data('updateList'));
                    });
                } else {
                    paragraphNodes = paragraphNodes.filter(function () {
                        return DOM.isListLabelNode(this.firstChild);
                    });
                }

                allListStyleIds = updateListsDebouncedOptions.listStyleIds;

                // Update of bullet lists can be completely ignored, except new paragraphs were inserted
                if (!updateListsDebouncedOptions.paraInsert) {
                    allListStyleIds = _.reject(allListStyleIds, function (styleID) {
                        return listCollection.isAllLevelsBulletsList(styleID);
                    });
                }

                // If the list is empty, there is nothing to do
                doNothing = _.isEmpty(allListStyleIds);
            }

            if (Utils.getBooleanOption(options, 'async', false)) {
                def = app.processArrayDelayed(updateListsCallback, paragraphNodes, { chunkLength: 20 });
            } else {
                if (! doNothing) { updateListsCallback(paragraphNodes); }
                def = $.when();
            }

            // enabling new registration for debounced list update
            updateListsDebouncedOptions = {};

            return def.promise();
        }

        /**
         * Check, wheter a given paragraph node describes a paragraph that
         * is part of a list (bullet or numbering list).
         *
         * @param {HTMLElement|jQuery} node
         *  The paragraph whose attributes will be checked. If this object is a
         *  jQuery collection, uses the first DOM node it contains.
         *
         * @param {Object} [attributes]
         *  An optional map of attribute maps (name/value pairs), keyed by attribute.
         *  It this parameter is defined, the parameter 'paragraph' can be omitted.
         *
         * @returns {Boolean}
         *  Whether the content node is a list paragraph node.
         */
        function isListStyleParagraph(paragraph, attributes) {
            var paraAttrs = attributes || StyleSheets.getExplicitAttributes(paragraph);
            return (paraAttrs && paraAttrs.paragraph && paraAttrs.paragraph.listStyleId);
        }

        /**
         * Check, wheter a given collection of nodes contain paragraphs with list
         * styles.
         *
         * @param {Array|HTMLElement|jQuery} nodes
         *  The collection of nodes, that will be checked for paragraphs with list styles.
         *  This can be an array with html nodes or jQuery nodes. It can also be a html node
         *  or a jQuery object not included into an array. Above all nodes is iterated.
         *
         * @returns {Boolean}
         *  Whether at least one of the specified nodes contains at least one list paragraph node.
         */
        function elementsContainListStyleParagraph(nodes) {

            var isListParagraph = false;

            if ((!nodes) || (_.isArray(nodes) && _.isEmpty(nodes))) { return isListParagraph; }

            _.chain(nodes).getArray().any(function (oneNode) {
                $(oneNode).each(function (index, node) {
                    $(node).find(DOM.PARAGRAPH_NODE_SELECTOR).each(function (i, paragraph) {
                        if (isListStyleParagraph(paragraph)) {
                            isListParagraph = true;
                            return !isListParagraph; // stopping 'each' iteration, if list paragraph was found
                        }
                    });
                    return !isListParagraph; // stopping 'each' iteration, if list paragraph was found
                });
                return isListParagraph; // stopping 'any' iteration, if list paragraph was found
            });

            return isListParagraph;
        }

        /**
         * Collecting all list style IDs of the paragraphs inside the specified elements.
         *
         * @param {Array|HTMLElement|jQuery} nodes
         *  The collection of nodes, that will be checked for paragraphs with list styles.
         *  This can be an array with html nodes or jQuery nodes. It can also be a html node
         *  or a jQuery object not included into an array. Above all nodes is iterated.
         *
         * @returns {[Array]}
         *  An array containing the list style IDs of the paragraphs inside the specified elements.
         */
        function getListStyleIDsOfParagraphs(nodes) {

            var listStyleIDs = null;

            if ((!nodes) || (_.isArray(nodes) && _.isEmpty(nodes))) { return listStyleIDs; }

            _.chain(nodes).getArray().any(function (oneNode) {
                $(oneNode).each(function (index, node) {
                    $(node).find(DOM.PARAGRAPH_NODE_SELECTOR).each(function (i, paragraph) {
                        // checking paragraph attributes for list styles
                        var paraAttrs = StyleSheets.getExplicitAttributes(paragraph);
                        // updating lists, if required
                        if (isListStyleParagraph(null, paraAttrs) && paraAttrs && paraAttrs.paragraph) {
                            listStyleIDs = listStyleIDs || [];
                            if (! _.contains(listStyleIDs, paraAttrs.paragraph.listStyleId)) { listStyleIDs.push(paraAttrs.paragraph.listStyleId); }
                        }
                    });
                });
            });

            return listStyleIDs;
        }

        /**
         * Collecting information about the event, that triggered the list update. This update
         * can be significantly accelerated by modifying only paragraphs with specified list IDs.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the operation. The
         *  following options are supported:
         *  @param {Boolean} [options.useSelectedListStyleIDs=false]
         *      If set to true, the update list process can be accelerated by only modifying paragraphs
         *      with correct list style ID set.
         *  @param {Boolean} [options.listStyleId='']
         *      The listStyleId of the paragraph.
         *  @param {Boolean} [options.listStyleLevel = -1]
         *      The listStyleLevel of the paragraph.
         */
        function registerListUpdate(options) {

            // Performance: Registering whether only specified style IDs need to be updated in this list update (never overwriting, after 'false' was set).
            // This is the case after splitting of a paragraph or after deleting a paragraph
            if ((updateListsDebouncedOptions.useSelectedListStyleIDs === undefined) || (updateListsDebouncedOptions.useSelectedListStyleIDs === true)) {
                updateListsDebouncedOptions.useSelectedListStyleIDs = Utils.getBooleanOption(options, 'useSelectedListStyleIDs', false);

                // checking, whether new paragraphs with list style IDs were inserted.
                // It is sufficient, that only one call of this function sets the value to 'true'.
                if ((updateListsDebouncedOptions.paraInsert === undefined) || (updateListsDebouncedOptions.paraInsert === false)) {
                    updateListsDebouncedOptions.paraInsert = Utils.getBooleanOption(options, 'paraInsert', false);
                }

                // also collecting all the affected listStyleIDs
                if (Utils.getBooleanOption(options, 'useSelectedListStyleIDs', false)) {
                    updateListsDebouncedOptions.listStyleIds = updateListsDebouncedOptions.listStyleIds || [];
                    if (_.isArray(options.listStyleId)) {
                        updateListsDebouncedOptions.listStyleIds = updateListsDebouncedOptions.listStyleIds.concat(options.listStyleId);
                    } else {
                        updateListsDebouncedOptions.listStyleIds.push(Utils.getStringOption(options, 'listStyleId', ''));
                    }
                }
            }
        }

        /**
         * Removing the artificial selection of empty paragraphs
         */
        function removeArtificalHighlighting() {
            _.each(highlightedParagraphs, function (element) {
                var backgroundColor = element.backgroundColor || '';
                $(element.node).css('background-color', backgroundColor);
            });
            highlightedParagraphs = [];
        }

        /**
         * Colecting data about event that triggered inserting page breaks
         *
         * @param {HTMLElement|jQuery} currentNode
         *  DOM node where the cursor position is - we process DOM from that element downward
         *
         */
        function registerPageBreaksInsert(currentNode) {
            currentProcessingNode = currentNode;
        }

        /**
         * Inserts a collaborative overlay element.
         */
        function insertCollaborativeOverlay() {
            editdiv.append($('<div>').addClass('collaborative-overlay').attr('contenteditable', false));
        }

        /**
         * Handle user data update which is triggered by a 'docs:users' event from EditApplication.
         * - handle selection updates and displays them
         *
         * @param {Object} data
         * Server notification message object, containing user information e.g. user name, user selection etc
         */
        function userDataUpdateHandler(data) {
            // quit if we dont have collaborating users
            if (!data.activeUsers) { return; }
            // clean cursors on overlay
            editdiv.find('.collaborative-overlay').children().remove();
            var overlay = editdiv.find('.collaborative-overlay'),
                caretSpan = $('<span>').addClass('collaborative-caret').text('|'),
                caretSpanHeight = 0,
                selectionStart = selection.getStartPosition(),
                selectionEnd = selection.getEndPosition(),
                zoomFactor = app.getView().getZoomFactor();
            _.each(data.activeUsers, function (user, index) {
                if (!user.userData || !user.userData.selection || user.userData.selection.type === 'drawing') { return; }
                var selectionPoints = [Position.getDOMPosition(editdiv, user.userData.selection.end)],
                    caretAbsolutePositions = [];
                // if we have a selection area
                if (!_.isEqual(user.userData.selection.start, user.userData.selection.end)) {
                    selectionPoints.unshift(Position.getDOMPosition(editdiv, user.userData.selection.start));
                }
                // draw collaborative carets and usernames
                _.each(selectionPoints, function (selectionPoint, selectionPointIndex, selectionPointsArray) {
                    if (!selectionPoint || user.userId === app.getClientId()) { return; }
                    var cursorElement = $(selectionPoint.node.parentNode),
                        cursorElementLineHeight = parseFloat(cursorElement.css('line-height')),
                        usernameOverlay = $('<div>').addClass('collaborative-username'),
                        caretOverlay = $('<div>').addClass('collaborative-cursor'),
                        caretHandle = $('<div>').addClass('collaborative-cursor-handle');
                    if (_.isNumber(selectionPoint.offset)) {
                        // break cursor element on the text offset
                        if (selectionPoint.offset === 0) {
                            cursorElement.before(caretSpan);
                        } else {
                            DOM.splitTextSpan(cursorElement, selectionPoint.offset, {append: true});
                            cursorElement.after(caretSpan);
                        }
                        caretSpanHeight = cursorElementLineHeight;
                        usernameOverlay.css('top', cursorElement.css('line-height'));
                        // create caret overlay and calculate its position
                        var caretTop = (caretSpan.offset().top - editdiv.offset().top - cursorElementLineHeight + caretSpan.outerHeight()) / zoomFactor * 100,
                            caretLeft = (caretSpan.offset().left - editdiv.offset().left) / zoomFactor * 100,
                            className = 'user-' + (index % Utils.SCHEME_COLOR_COUNT);
                        //save absolute position of caret decoy for selection rendering
                        caretAbsolutePositions.push([caretTop, caretLeft]);
                        // apply decoy position to caret overlay
                        caretOverlay.css({ top: caretTop, left: caretLeft, height: caretSpanHeight}).addClass(className);
                        // insert username only once
                        if (selectionPointIndex === selectionPointsArray.length - 1) {
                            usernameOverlay.html(user.userDisplayName).addClass(className);
                            caretOverlay.append(usernameOverlay, caretHandle);
                            caretHandle.hover(function () { usernameOverlay.show(); }, function () { usernameOverlay.hide(); });
                        }
                        overlay.append(caretOverlay);
                        // restore original state of document
                        caretSpan.remove();
                        if (selectionPoint.offset > 0) { CharacterStyles.mergeSiblingTextSpans(cursorElement, true); }
                    }
                });
                // draw collaborative selection area highlighting
                if (caretAbsolutePositions.length  ===  2) {
                    var startNode = $(selectionPoints[0].node.parentNode),
                        endNode = $(selectionPoints[1].node.parentNode),
                        startParentNode = startNode.parent(),
                        endParentNode = endNode.parent(),
                        selectionOverlayGroup = $('<div>').addClass('collaborative-selection-group'),
                        highlightWidth = Math.max(startParentNode.width(), endParentNode.width()),
                        editdivPos = editdiv.offset(),
                        pageContentNodePos = DOM.getPageContentNode(editdiv).offset(),
                        startTop = caretAbsolutePositions[0][0],
                        startLeft = caretAbsolutePositions[0][1],
                        endTop = caretAbsolutePositions[1][0],
                        endLeft = caretAbsolutePositions[1][1],
                        startNodeCell = startNode.closest('td'),
                        endNodeCell = endNode.closest('td'),
                        isTableSelection = startNodeCell.length && endNodeCell.length && (!startNodeCell.is(endNodeCell)),
                        isListSelection = DOM.isListLabelNode(Utils.findPreviousSiblingNode(startNode)) ||
                            DOM.isListLabelNode(Utils.findPreviousSiblingNode(endNode));

                    // special handling of pure table selection
                    if (isTableSelection) {
                        // handle special mega cool firefox cell selection
                        if (user.userData.selection.type === 'cell') {
                            var ovStartPos = startNodeCell.offset(),
                                ovEndPos = endNodeCell.offset(),
                                ovWidth = (ovEndPos.left  - ovStartPos.left) / zoomFactor * 100 + endNodeCell.outerWidth(),
                                ovHeight = (ovEndPos.top  - ovStartPos.top) / zoomFactor * 100  + endNodeCell.outerHeight(),
                                ov = $('<div>').css({
                                    top: (ovStartPos.top - editdivPos.top) / zoomFactor * 100,
                                    left: (ovStartPos.left - editdivPos.left) / zoomFactor * 100,
                                    width: ovWidth,
                                    height: ovHeight
                                });
                            selectionOverlayGroup.append(ov);
                        } else { // normal table selection (Chrome, IE): iterate cells between start and end
                            var cells = $(Utils.findFarthest(editdiv, endNodeCell, 'table')).find('td'),
                                cellsToHilite = cells.slice(cells.index(startNodeCell), cells.index(endNodeCell) + 1);
                            cellsToHilite.each(function () {
                                var cellOv = $('<div>').css({
                                    top: ($(this).offset().top - editdivPos.top) / zoomFactor * 100,
                                    left: ($(this).offset().left - editdivPos.left) / zoomFactor * 100,
                                    width: $(this).outerWidth(),
                                    height: $(this).outerHeight()
                                });
                                selectionOverlayGroup.append(cellOv);
                            });
                        }
                    } else { // paragraph / mixed selection
                        var startHeight = parseFloat(startNode.css('line-height')),
                            endHeight = parseFloat(endNode.css('line-height')),
                            startBottom = startTop + startHeight,
                            endBottom = endTop + endHeight;
                        // selection area is in a line
                        if (startBottom === endBottom) {
                            var selectionOverlay = $('<div>').css({
                                top: Math.min(startTop, endTop),
                                left: startLeft,
                                width: endLeft - startLeft,
                                height: Math.max(startHeight, endHeight)
                            });
                            selectionOverlayGroup.append(selectionOverlay);
                        } else { // multi line selection area
                            // start and end node are empty lines (paragraphs)
                            if (!highlightWidth) { highlightWidth = editdiv.width(); }
                            var headOverlay = $('<div>').css({
                                    top: startTop,
                                    left: startLeft,
                                    width: startParentNode.offset().left + startParentNode.width() - startLeft - editdivPos.left,
                                    height: startHeight
                                }),
                                bodyOverlay = $('<div>').css({
                                    top: startTop + startHeight,
                                    left: (isListSelection ? pageContentNodePos.left - editdivPos.left : Math.min(startParentNode.offset().left, endParentNode.offset().left) - editdivPos.left) / zoomFactor * 100,
                                    width: isListSelection ? editdiv.width() : highlightWidth,
                                    height: endTop - startTop - startHeight
                                }),
                                tailOverlay = $('<div>').css({
                                    top: endTop,
                                    left: isListSelection ? (endNode.offset().left - editdivPos.left) / zoomFactor * 100  : (endParentNode.offset().left - editdivPos.left) / zoomFactor * 100,
                                    width: endLeft - ((endParentNode.offset().left - editdivPos.left) / zoomFactor * 100),
                                    height: endHeight
                                });
                            selectionOverlayGroup.append(headOverlay, bodyOverlay, tailOverlay);
                        }
                    }
                    selectionOverlayGroup.children().addClass('selection-overlay user-' + (index % Utils.SCHEME_COLOR_COUNT));
                    overlay.append(selectionOverlayGroup);
                }
            });
            // return original position of browser cursor after merge
            selection.setTextSelection(selectionStart, selectionEnd, { userDataUpdateHandler: true });
            // show user name for 3 seconds, after resetting timeout from previous user data update call.
            clearTimeout(overlayTimeoutId);
            overlayTimeoutId = setTimeout(function () { overlay.find('.collaborative-username').fadeOut(); }, 3000);

        }

        /**
         * Updating the selection of a remote client without write privileges.
         */
        function updateRemoteSelection() {

            var // whether the selection before applying the external operation was a range selection
                wasRange = false,
                // whether a drawing is selected
                drawingSelected = false,
                // whether the browser selection can be evaluated
                useBrowserSelection = true,
                // whether a cell range is selected
                cellSelected = false,
                // old logical start and end positions
                oldStartPosition = null, oldEndPosition = null,
                // the drawing node of a selected drawing
                drawingNode,
                // logical position of the selected drawing
                drawingStartPosition, drawingEndPosition,
                // a new calculated valid logical position
                validPos,
                // the node containing the focus
                focusNode = null;

            // checking if the selection before applying the external operation was a range selection
            if (selection.hasRange()) {
                wasRange = true;
                oldStartPosition = _.clone(selection.getStartPosition());
                oldEndPosition = _.clone(selection.getEndPosition());
                drawingSelected = (selection.getSelectionType() === 'drawing');
                cellSelected = (selection.getSelectionType() === 'cell');
            }

            if (drawingSelected || cellSelected) { useBrowserSelection = false; }

            // using the browser selection as new selection
            if (useBrowserSelection) {
                selection.updateSelectionAfterBrowserSelection();
                // check if it is necessary to restore a selection range (maybe the browser removed a selection range)
                if (wasRange && !selection.hasRange()) {
                    validPos = Position.findValidSelection(editdiv, oldStartPosition, oldEndPosition, selection.getLastDocumentPosition());
                    selection.setTextSelection(validPos.start, validPos.end, { remoteSelectionUpdate: true });
                }
            } else if (drawingSelected) {

                drawingNode = selection.getSelectedDrawing();

                if ($(drawingNode).closest(DOM.PAGECONTENT_NODE_SELECTOR).length > 0) {
                    // is the drawing still in the dom?
                    drawingStartPosition = Position.getOxoPosition(editdiv, drawingNode, 0);
                    drawingEndPosition = Position.increaseLastIndex(drawingStartPosition);

                    if (!_.isEqual(drawingStartPosition), selection.getStartPosition()) {
                        selection.setTextSelection(drawingStartPosition, drawingEndPosition, { remoteSelectionUpdate: true });
                    }
                } else {
                    validPos = Position.findValidSelection(editdiv, selection.getStartPosition(), selection.getEndPosition(), selection.getLastDocumentPosition());
                    selection.setTextSelection(validPos.start, validPos.end, { preserveFocus: false, remoteSelectionUpdate: true });

                    // The div.page must get the focus. Unfortunately this fails in Chrome,
                    // if Chrome does not have the focus.
                    if (! $(window.document.activeElement).is(DOM.PAGE_NODE_SELECTOR)) {
                        focusNode = window.document.activeElement;
                        Utils.info('Missing browser focus -> changing focus failed. Focus at ' + focusNode.nodeName + '.' + focusNode.className);
                    }
                }

            } else if (cellSelected) {

                // Updating position for cell selection
                selection.updateSelectionAfterBrowserSelection();

                // maybe the cell selection was destroyed by the operation -> selection.getSelectedCellRange() returns null
                if (! selection.getSelectedCellRange()) {
                    validPos = Position.findValidSelection(editdiv, oldStartPosition, oldEndPosition, selection.getLastDocumentPosition());
                    selection.setTextSelection(validPos.start, validPos.end, { remoteSelectionUpdate: true });
                }

            }
        }

        /**
         * Shows a warning dialog with Yes/No buttons before deleting document
         * contents that cannot be restored.
         *
         * @param {String} title
         *  The title of the dialog box.
         *
         * @param {String} message
         *  The message text shown in the dialog box.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved if the Yes
         *  button has been pressed, or rejected if the No button has been pressed.
         */
        function showDeleteWarningDialog(title, message) {

            var // the displayed dialog
                dialog = new Dialogs.ModalQueryDialog(message, { title: title });

            // close dialog automatically after losing edit rights
            EditDialogMixin.call(dialog, app);

            return dialog.show();
        }

        /**
         * iterate over _all_ paragraphs and update numbering symbols and index
         */
        var
            updateListsDebounced = $.noop,
            insertPageBreaksDebounced = $.noop;
        app.on('docs:import:success', function () {
            // Adding delay for debounced list update, so that the browser gets the chance to render the new page (iPad mini)
            updateListsDebounced = app.createDebouncedMethod(registerListUpdate, updateLists, { delay: 10 });
            insertPageBreaksDebounced = app.createDebouncedMethod(registerPageBreaksInsert, insertPageBreaks, { delay: 50 });
        });

        /**
         * Dumps the passed event object to the browser console.
         */
        var dumpEventObject = LOG_EVENTS ? function (event) {
            Utils.log('type=' + event.type + ' keyCode=' + event.keyCode + ' charCode=' + event.charCode + ' shift=' + event.shiftKey + ' ctrl=' + event.ctrlKey + ' alt=' + event.altKey);
        } : $.noop;

        // initialization -----------------------------------------------------

        // initialize document contents
        app.on('docs:init', initDocument)
            .on('docs:import:after', function () {
                selection.selectTopPosition();
                app.getView().grabFocus();
            })
            .on('docs:import:success', documentLoaded);

        // restore selection after undo/redo operations
        undoManager.on('undo:before redo:before', function () {
            undoRedoRunning = true;
        });

        // restore selection after undo/redo operations
        undoManager.on('undo:after redo:after', function () {
            var para = null,
                newParagraph = null,
                pageContentNode = null;
            // setting text selection to 'lastOperationEnd' after undo/redo.
            // Special behaviour for setAttributes operations, where an existing
            // browser selection needs to be restored (useSelectionRangeInUndo).
            // -> Fix for task 28520: An setAttributes after a splitParagraph can have an invalid text selection
            // and therefore must use lastOperationEnd
            if ((useSelectionRangeInUndo) && (!selection.isTextLevelSelection() || selection.isValidTextSelection())) {
                selection.restoreBrowserSelection({ preserveFocus: true });
            } else {
                // #31971, #32322
                pageContentNode = DOM.getPageContentNode(editdiv);
                if (pageContentNode.find(DOM.CONTENT_NODE_SELECTOR).length === 0 && pageContentNode.find('.page-break').length > 0) { //cleanup of dom
                    pageContentNode.find('.page-break').remove();

                    newParagraph = DOM.createImplicitParagraphNode();
                    pageContentNode.append(newParagraph);
                    validateParagraphNode(newParagraph);
                    implParagraphChanged(newParagraph);
                    // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
                    // newly, also in other browsers (new jQuery version?)
                    repairEmptyTextNodes(newParagraph);
                }

                if (_.browser.IE) {
                    // 29397: Not using implicit paragraph in IE
                    para = Position.getLastNodeFromPositionByNodeName(editdiv, lastOperationEnd, DOM.PARAGRAPH_NODE_SELECTOR);
                    if (DOM.isImplicitParagraphNode(para) && para.previousSibling) {
                        lastOperationEnd = Position.getLastPositionInParagraph(editdiv, Position.getOxoPosition(editdiv, para.previousSibling, 0));
                    }
                }
                selection.setTextSelection(lastOperationEnd);
            }
            undoRedoRunning = false;
        });

        // undogroup:open : checking if the operations in a group can be undone
        undoManager.on('undogroup:open', function () {
            deleteUndoStack = false;  // default: undo stack must not be deleted
            isInUndoGroup = true;
        });

        // undogroup:closed : delete undo stack, if required
        undoManager.on('undogroup:close', function () {
            if (deleteUndoStack) {
                undoManager.clearUndoActions();
            }
            deleteUndoStack = false;
            isInUndoGroup = false;
        });

        // remove highlighting before changing the DOM which invalidates the positions in highlightRanges
        this.on('operations:before', function (event, operations, external) {
            // reset read-only paragraph
            if (roIOSParagraph) {
                if (self.getEditMode() === true) {
                    $(roIOSParagraph).removeAttr('contenteditable');
                }
                roIOSParagraph = null;
            }

            if (!isReplaceOperation) {
                // clear search results and trigger the debounced quick search
                self.removeHighlighting();
                if (app.getWindow().search.active) {
                    self.debouncedQuickSearch(app.getWindow().search.query);
                }
            }

            // clear preselected attributes before applying any operations
            if (! keepPreselectedAttributes) {
                preselectedAttributes = null;
            }

            // Insert drawing operations with a document URL as imageUrl attribute also have the imageData attribute set.
            // Before the operation is applied we need to determine which attribute to use and which to remove.
            if (!external) {
                _(operations).each(function (operation) {
                    if (operation.name === Operations.DRAWING_INSERT && (operation.type === 'image' || operation.type === 'ole') && operation.attrs && operation.attrs.drawing) {
                        Image.postProcessOperationAttributes(operation.attrs.drawing, app.getFileDescriptor().id);
                    }
                });
            }
        });

        this.on('operations:success', function (event, operations) {
            // saving information about the type of operation
            useSelectionRangeInUndo = (_.last(operations).name === Operations.SET_ATTRIBUTES);

            // useSelectionRangeInUndo also needs to be true, if after setAttributes an operation like deleteListStyle comes (32005)
            // -> maybe there needs to be a list of operations that can be ignored like 'Operations.DELETE_LIST' following the setAttributes operation
            // -> leaving this simple now because of performance reasons
            if ((!useSelectionRangeInUndo) && (operations.length > 1) && (_.last(operations).name === Operations.DELETE_LIST) && (operations[operations.length - 2].name === Operations.SET_ATTRIBUTES)) {
                useSelectionRangeInUndo = true;
            }

            // restart spell checking after applying operations
            spellCheckerCurrentNode = null;
            implStartOnlineSpelling();

            // In read-only mode update the selection after an external operation (and after the document is loaded successfully)
            // -> maybe this need to be deferred
            if (self.getEditMode() !== true && app.isImportFinished()) { updateRemoteSelection(); }
        });

        // edit mode changed
        this.on('change:editmode', function (event, editMode) {

            // set CSS class for formatting dependent on edit mode (mouse pointers)
            editdiv.toggleClass('edit-mode', editMode);

            // Fix for 30992, setting valid text position after receiving edit rights
            if (editMode) {
                if (Position.isValidTextPosition(editdiv, selection.getStartPosition())) {
                    selection.setTextSelection(selection.getStartPosition());
                } else if ((!_.isEmpty(selection.getStartPosition())) && (Position.isValidTextPosition(editdiv, Position.getFirstPositionInParagraph(editdiv, [selection.getStartPosition()[0]])))) {
                    selection.setTextSelection(Position.getFirstPositionInParagraph(editdiv, [selection.getStartPosition()[0]]));
                } else if (lastOperationEnd && Position.isValidTextPosition(editdiv, lastOperationEnd)) {
                    selection.setTextSelection(lastOperationEnd);
                } else {
                    selection.selectTopPosition();
                }
            }

            // in read-only mode change drawing selection to text selection
            if (!editMode) {
                selection.selectDrawingAsText();
            }

            // We are not able to disable the context menu in read-only mode on iPad
            // or other touch devices. Therefore we just set contenteditable dependent
            // on the current editMode.
            if (Modernizr.touch) {

                // prevent virtual keyboard on touch devices in read-only mode
                editdiv.attr('contenteditable', editMode);
            }

            // disable IE table manipulation handlers in edit mode
            // Utils.getDomNode(editdiv).onresizestart = function () { return false; };
            // The resizestart event does not appear to bubble in IE9+, so we use the selectionchange event to bind
            // the resizestart handler directly once the user selects an object (as this is when the handles appear).
            // The MS docs (http://msdn.microsoft.com/en-us/library/ie/ms536961%28v=vs.85%29.aspx) say it's not
            // cancelable, but it seems to work in practice.

            // The oncontrolselect event fires when the user is about to make a control selection of the object.
            // Non-standard event defined by Microsoft for use in Internet Explorer.
            // This event fires before the element is selected, so inspecting the selection object gives no information about the element to be selected.
            if (_.browser.IE) {
                Utils.getDomNode(editdiv).oncontrolselect = function (e) {
                    if (DOM.isImageNode(e.srcElement)) {
                        // if an image was selected, return browser selection to the image clipboard
                        selection.restoreBrowserSelection();
                    }
                    // return false to suppress IE default size grippers
                    return false;
                };
            }

            // focus back to editor
            app.getView().grabFocus();
        });

        // scroll to cursor, forward selection change events to own listeners
        selection.on('change', function (event, options) {

            var // whether an implicit paragraph behind a table has to be increased
                increaseParagraph = false,
                domNode = null,
                paraNode = null,
                rowNode = null,
                tableNode = null,
                explicitAttributes = null,
                startPosition = null,
                nodeInfo = null,
                insertOperation = Utils.getBooleanOption(options, 'insertOperation', false),
                splitOperation = Utils.getBooleanOption(options, 'splitOperation', false),
                readOnly = self.getEditMode() !== true,
                // selection hightlighting: dom point of start and end position
                startNodeInfo = null, endNodeInfo = null,
                // selection hightlighting: paragraph dom nodes
                nextParagraph = null, startParagraph = null, endParagraph = null,
                // whether the iteration reached the end paragraph
                reachedEndParagraph = false;

            // handling height of implicit paragraphs, but not for insertText operations or splitting of paragraphs
            // and also not in readonly mode (28563)
            if (! insertOperation && ! splitOperation && ! readOnly) {

                startPosition = selection.getStartPosition();
                nodeInfo = Position.getDOMPosition(editdiv, startPosition);

                // if this is the last row of a table, a following implicit paragraph has to get height zero or auto
                if (Position.isPositionInTable(editdiv, startPosition)) {
                    domNode = nodeInfo.node;
                    paraNode = $(domNode).closest(DOM.PARAGRAPH_NODE_SELECTOR);
                    rowNode = $(paraNode).closest(DOM.TABLE_ROWNODE_SELECTOR);
                    tableNode = $(rowNode).closest(DOM.TABLE_NODE_SELECTOR);

                    if (paraNode !== null) {
                        if (paraNode.get(0) === increasedParagraphNode) {
                            increaseParagraph = true;  // avoid, that height is set to 0, if cursor is in this paragraph
                            if ((DOM.isImplicitParagraphNode(paraNode)) && (paraNode.prev().length === 0)) {  // maybe the previous table was just removed
                                paraNode.css('height', '');  // the current paragraph always has to be increased
                                increasedParagraphNode = null;
                            }
                        } else if ((DOM.isImplicitParagraphNode(paraNode)) && (DOM.isFinalParagraphBehindTable(paraNode))) {
                            if (increasedParagraphNode !== null) { $(increasedParagraphNode).css('height', 0); } // useful for tables in tables in tables in ...
                            paraNode.css('height', '');
                            increaseParagraph = true;   // cursor travelling from right/button directly into the explicit paragraph
                            increasedParagraphNode = paraNode.get(0);
                            // in Internet Explorer it is necessary to restore the browser selection, so that no frame is drawn around the paragraph (28132)
                            if (_.browser.IE) { selection.restoreBrowserSelection(); }
                        } else if ((DOM.isTableInTableNode(tableNode)) && ($(rowNode).next().length === 0) && (DOM.isImplicitParagraphNode($(tableNode).next()))) {
                            if (increasedParagraphNode !== null) { $(increasedParagraphNode).css('height', 0); } // useful for tables in tables in tables in ...
                            $(tableNode).next().css('height', '');
                            increaseParagraph = true;
                            increasedParagraphNode = $(tableNode).next().get(0);
                        } else if ((DOM.isImplicitParagraphNode(paraNode)) && (paraNode.prev().length === 0) && (paraNode.height() === 0))  {
                            // if the cursor was not in the table, when the inner table was deleted -> there was no increasedParagraphNode behind the table
                            paraNode.css('height', '');  // the current paragraph always has to be increased, maybe a previous table just was deleted (undo)
                        }
                    }
                } else {
                    if (_.browser.IE) {
                        // in Internet Explorer it is necessary to restore the browser selection, so that no frame is drawn around the paragraph (28132)
                        paraNode = $(nodeInfo.node).closest(DOM.PARAGRAPH_NODE_SELECTOR);
                        if ((DOM.isImplicitParagraphNode(paraNode)) && (DOM.isFinalParagraphBehindTable(paraNode))) {
                            paraNode.css('height', '');
                            selection.restoreBrowserSelection();
                        }
                    }
                }

                // if the implicit paragraph no longer has to get an increase height
                if ((! increaseParagraph) && (increasedParagraphNode)) {
                    $(increasedParagraphNode).css('height', 0);
                    increasedParagraphNode = null;
                }

            }

            // in Internet Explorer it is necessary to restore the browser selection, so that no frame is drawn around the paragraph (28132),
            // this part for the read only mode
            if (readOnly && _.browser.IE && ! Position.isPositionInTable(editdiv, selection.getStartPosition())) {
                nodeInfo = Position.getDOMPosition(editdiv, selection.getStartPosition());
                paraNode = $(nodeInfo.node).closest(DOM.PARAGRAPH_NODE_SELECTOR);
                if ((DOM.isImplicitParagraphNode(paraNode)) && (DOM.isFinalParagraphBehindTable(paraNode))) {
                    paraNode.css('height', '');
                    selection.restoreBrowserSelection();
                }
            }

            // clear preselected attributes when selection changes
            if (! keepPreselectedAttributes) {
                preselectedAttributes = null;
            }

            // Special behaviour for cursor positions directly behind fields.
            // -> the character attributes of the field have to be used
            // -> using preselectedAttributes
            if ((! insertOperation) && (nodeInfo) && (nodeInfo.offset === 0) && (nodeInfo.node) && (nodeInfo.node.parentNode) && (DOM.isFieldNode(nodeInfo.node.parentNode.previousSibling))) {
                explicitAttributes = StyleSheets.getExplicitAttributes(nodeInfo.node.parentNode.previousSibling.firstChild);
                self.addPreselectedAttributes(explicitAttributes);
            }

            // removing highlighting of selection of empty paragraphs in Firefox and IE
            if (!_.isEmpty(highlightedParagraphs)) { removeArtificalHighlighting(); }

            // highlighting selection of empty paragraphs in Firefox and IE
            if ((_.browser.Firefox || _.browser.IE) && selection.hasRange() && selection.getSelectionType() !== 'cell') {

                startNodeInfo = Position.getDOMPosition(editdiv, selection.getStartPosition());
                endNodeInfo = Position.getDOMPosition(editdiv, selection.getEndPosition());

                if (startNodeInfo && startNodeInfo.node && endNodeInfo && endNodeInfo.node) {
                    startParagraph = Utils.getDomNode($(startNodeInfo.node).closest(DOM.PARAGRAPH_NODE_SELECTOR));
                    endParagraph = Utils.getDomNode($(endNodeInfo.node).closest(DOM.PARAGRAPH_NODE_SELECTOR));

                    if (startParagraph && endParagraph && startParagraph !== endParagraph) {
                        nextParagraph = startParagraph;

                        while (nextParagraph && !reachedEndParagraph) {

                            if (nextParagraph) {

                                // only highlight empty paragraphs
                                if (Position.getParagraphNodeLength(nextParagraph) === 0) {
                                    // collecting all modified paragraphs in an array
                                    highlightedParagraphs.push({ node: nextParagraph, backgroundColor: $(nextParagraph).css('background-color') });
                                    // modify the background color of the empty paragraph to simulate selection
                                    $(nextParagraph).css('background-color', selectionHighlightColor);
                                }

                                if (nextParagraph === endParagraph) {
                                    reachedEndParagraph = true;
                                } else {
                                    nextParagraph = Utils.findNextNode(editdiv, nextParagraph, DOM.PARAGRAPH_NODE_SELECTOR);
                                }
                            }
                        }
                    }
                }
            }

            // forward selection change events to own listeners (view and controller)
            self.trigger('selection', selection, { insertOperation: insertOperation });
        });

        // handle user data update (user selection, etc)
        app.on('docs:users', userDataUpdateHandler);

        // Register selectionchange handler to get notification if user changes
        // selection via handles. No other way to detect these selection changes on
        // mobile Safari.
        if (Utils.IPAD) {
            app.registerGlobalEventHandler(document, 'selectionchange', function (event) {
                var sel = window.getSelection();

                if (sel && sel.rangeCount >= 1) {
                    // Check that the selection is inside our editdiv node. We are listing to
                    // the whole document!
                    if (Utils.containsNode(editdiv, sel.getRangeAt(0).startContainer)) {
                        selection.processBrowserEvent(event);
                    }
                }
            });
        }

        // registering keypress handler at document for Chrome to avoid problems with direct keypress to document
        // -> using app.registerGlobalEventHandler the handler is registered with 'show' and deregistered
        // with 'hide' event of the app automatically and does not influence other apps.
        if (_.browser.WebKit) {
            app.registerGlobalEventHandler(document, 'keypress', processKeypressOnDocument);
        }

        // hybrid edit mode
        editdiv.on({
            keydown: processKeyDown,
            keypress: processKeyPressed,
            compositionstart: processCompositionStart,
            compositionupdate: processCompositionUpdate,
            compositionend: processCompositionEnd,
            textInput: processTextInput,
            input: processInput,
            'mousedown touchstart': processMouseDown,
            'mouseup touchend': processMouseUp,
            dragstart: processDragStart,
            drop: processDrop,
            'dragenter dragexit dragover dragleave': false,
            contextmenu: false,
            cut: _.bind(this.cut, this),
            copy: _.bind(this.copy, this),
            'paste beforepaste': _.bind(this.paste, this)     // for IE we need to handle 'beforepaste', on all other browsers 'paste'
        });

        // Fix for 29751: IE support double click for word selection
        if (_.browser.IE) {
            editdiv.on({
                dblclick: processDoubleClick
            });
        }

        // mouseup events can be anywhere -> binding to $(document)
        $(document).on({'mouseup touchend': processMouseUpOnDocument});

        // destroy all class members on destruction
        this.registerDestructor(function () {
            selection.destroy();
            selection = null;
            // remove event handler from document
            $(document).off('mouseup touchend', processMouseUpOnDocument);
            $(document).off('keypress', processKeypressOnDocument);
        });

    } // class Editor

    // exports ================================================================

    // derive this class from class EditModel
    return EditModel.extend({ constructor: Editor });

});
