/**
 * All content on this website (including text, images, source code and any
 * other original works), unless otherwise noted, is licensed under a Creative
 * Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2013 Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/hyperlinkmixin',
    ['io.ox/office/editframework/view/hyperlinkutil'
    ], function (HyperlinkUtil) {

    'use strict';

    // mix-in class HyperlinkMixin ============================================

    /**
     * Mix-in class for the class SpreadsheetView that provides extensions for
     * dealing with hyperlinks in cells.
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     */
    function HyperlinkMixin(app) {

        var // self reference (the spreadsheet view)
            self = this,

            // the inline pop-up node containing the URL and a few buttons
            popupNode = HyperlinkUtil.createPopupNode(app).css({ marginTop: 3, minWidth: 130 });

        // private methods ----------------------------------------------------

        function setHyperlink(text, url) {

            var attributes = {};

            if (url && url.length) {

                var oldURL = self.getCellContents().attributes.character.url;

                attributes.character = { url: url };

                if (!oldURL) {
                    attributes.character.underline = true;
                    attributes.character.color = { type: 'scheme', value: 'hyperlink' };
                }
            } else {
                attributes.character = { url: null, underline : null, color: null };
            }

            self.fillCellRanges(text, attributes);
        }

        function updatePopupNode() {

            if (self.hasDrawingSelection()) {
                popupNode.hide();
                return;
            }

            if (!popupNode.children().length) {
                //FIXME: this is a workaround for the internet explorer, it seems, the IE deletes the children,w hen you hide an element
                popupNode = HyperlinkUtil.createPopupNode(app).css({ marginTop: 3, minWidth: 130 });
            }

            var address = self.getActiveCell();
            var cellData = self.getActiveSheetModel().getCellCollection().getCellEntry(address);

            var url = cellData.attributes.character.url;
            if (url && url.length > 0 && HyperlinkUtil.isLegalUrl(url)) {
                HyperlinkUtil.updatePopup(url, popupNode);

                var parentnode = self.getActiveGridPane().getNode().find('.active-cell');
                popupNode.css('top', parentnode.height() + 'px');
                parentnode.append(popupNode);

                popupNode.show();
            } else {
                popupNode.hide();
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Opens a Hyperlink Dialog Frame with the data from current Selection
         *
         * @returns {Jquery.Deferred}
         * this object can be used to observe the Dialog Frame, when url is changed
         */
        this.editHyperlink = function () {

            var item = this.getCellContents(),
                display = item.display,
                url = item.attributes.character.url;

            if (!url || !url.length) {
                url = HyperlinkUtil.checkForHyperlink(display);
            }

            var def = HyperlinkUtil.showDialog(display, url, app);

            return def.done(function (data) {
                var text;
                if (_.isString(data.text) && (data.text.length > 0) && (data.text !== display)) {
                    text = data.text;
                }
                setHyperlink(text, data.url);
            });
        };

        this.updateHyperlinkPopup = updatePopupNode;

        /**
         * removes the url and its style from the selected cell.
         * is called by the hyperlink-info-popup
         * @param {Baseapplication} app
         */
        this.removeHyperlink = function () {
            setHyperlink(undefined, null);
        };

        // initialization -----------------------------------------------------

        // initialize class members
        app.on('docs:init', function () {
            HyperlinkUtil.addEditModeListener(app.getModel(), popupNode, $.noop);
        });

        // register event handlers for view events
        this.on('change:layoutdata celledit:leave', updatePopupNode);
        this.on('celledit:enter', function () { popupNode.hide(); });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            popupNode.remove();
            app = self = popupNode = null;
        });

    } // class HyperlinkMixin

    // exports ================================================================

    return HyperlinkMixin;

});
