/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/mixin/celleditmixin',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/popup/tooltip',
     'io.ox/office/tk/popup/listmenu',
     'io.ox/office/editframework/model/format/color',
     'io.ox/office/editframework/view/hyperlinkutil',
     'io.ox/office/spreadsheet/utils/paneutils',
     'io.ox/office/spreadsheet/model/tokenarray',
     'io.ox/office/spreadsheet/model/cellcollection',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, KeyCodes, ToolTip, ListMenu, Color, HyperlinkUtil, PaneUtils, TokenArray, CellCollection, gt) {

    'use strict';

    // mix-in class CellEditMixin =============================================

    /**
     * Mix-in class for the class SpreadsheetView that provides extensions for
     * the cell in-place edit mode, and for showing and manipulating
     * highlighted cell ranges based on formula token arrays.
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     */
    function CellEditMixin(app) {

        var // self reference (spreadsheet view instance)
            self = this,

            // the spreadsheet model, and other model objects
            model = null,
            undoManager = null,
            documentStyles = null,
            fontCollection = null,
            numberFormatter = null,

            // the model and collections of the active sheet
            sheetModel = null,
            cellCollection = null,

            // the text area control
            textArea = Utils.createTextArea().attr({ tabindex: 0, 'data-focus-role': 'textarea' }),
            // the underlay node containing highlighting for formulas
            textAreaUnderlay = $('<div>').addClass('textarea-underlay'),

            // tooltip attached to the text area (for function/parameter help)
            textAreaToolTip = new ToolTip({ anchor: textArea, classes: 'textarea-tooltip' }),
            // drop-down list attached to the text area (for function names)
            textAreaListMenu = new ListMenu({ anchor: textArea }),
            // tooltip attached to the text area drop-down list (for the selected function)
            textAreaListItemToolTip = new ToolTip({ anchor: getTextAreaListButtonNode, border: textAreaListMenu.getNode(), position: 'right left', align: 'center' }),

            // whether the in-place cell edit mode is active
            editActive = false,
            // the index of the sheet containing the edited cell (active sheet may change during edit mode)
            editSheet = null,
            // the cell address of the edited cell
            editAddress = null,
            // the grid pane instance where in-place cell edit mode is active
            editGridPane = null,
            // the descriptor of the edited cell (result, attributes, etc.)
            editCellData = null,
            // the token array representing the current formula in the text area
            editTokenArray = null,
            // current character attributes for the text area
            editAttributes = null,

            // local undo/redo information while cell edit mode is active
            undoStack = null,
            // pointer into the local undo/redo stack
            undoIndex = 0,

            // initial text at the beginning of the edit mode
            originalText = null,
            // the value at the time the text area has been updated the last time
            lastValue = null,

            // quick edit mode (true), or text cursor mode (false)
            quickEditMode = null,
            // whether the current value of the text area is a formula
            isFormula = false,
            // whether to show auto-completion list for a function or name
            autoComplete = false,
            // token info for function auto-completion
            autoTokenInfo = null,
            // how many closing parenthesis entered manually will be ignored
            autoCloseData = [],
            // the position of the current range address while in range selection mode
            rangeSelection = null,

            // the token arrays currently used for highlighted ranges
            highlightTokenArrays = [],
            // all options used to highlight cell ranges
            highlightOptions = null;

        // private methods ----------------------------------------------------

        /**
         * Updates the CSS formatting of the text area control used for cell
         * in-place edit mode according to the current formatting of the active
         * cell.
         */
        function updateTextAreaStyle() {

            var // the address of the active cell
                address = editAddress,
                // the cell attributes of the cell
                cellAttributes = null,
                // the character attributes of the cell
                charAttributes = null,
                // text area node and the underlay node, as one jQuery collection
                targetNodes = textArea.add(textAreaUnderlay);

            // applies a transformation to the passed color, and returns the CSS color value
            function getModifiedCssColor(color, type, value) {
                if (Color.isAutoColor(color)) { return ''; }
                color = _.copy(color, true);
                (color.transformations || (color.transformations = [])).push({ type: type, value: value });
                return documentStyles.getCssColor(color);
            }

            // receive cell contents from cell collection
            editCellData = _.copy(cellCollection.getCellEntry(address), true);
            cellAttributes = editCellData.attributes.cell;
            charAttributes = editCellData.attributes.character;

            // overwrite with current edit attributes
            _(cellAttributes).extend(editAttributes.cell);
            _(charAttributes).extend(editAttributes.character);

            // update line height and font size according to current zoom factor
            charAttributes.lineHeight = self.getEffectiveLineHeight(charAttributes);
            charAttributes.fontSize = self.getEffectiveFontSize(charAttributes.fontSize);

            // set CSS attributes according to formatting of active cell
            targetNodes.css({
                padding: '0 ' + self.getEffectiveCellPadding() + 'px',
                lineHeight: charAttributes.lineHeight + 'px',
                fontFamily: documentStyles.getCssFontFamily(charAttributes.fontName),
                fontSize: Math.max(6, charAttributes.fontSize) + 'pt',
                fontWeight: charAttributes.bold ? 'bold' : 'normal',
                fontStyle: charAttributes.italic ? 'italic' : 'normal',
                textDecoration: PaneUtils.getCssTextDecoration(editCellData),
                textAlign: PaneUtils.getCssTextAlignment(editCellData, true)
            });

            // set cell text color and fill color to the appropriate nodes
            textArea.css('color', getModifiedCssColor(charAttributes.color, 'shade', 30000));
            textAreaUnderlay.css('background-color', getModifiedCssColor(cellAttributes.fillColor, 'tint', 30000));
        }

        /**
         * Updates the position and size of the text area control for the cell
         * in-place edit mode, according to the current length of the text in
         * the text area control, and the scroll position of the grid pane.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.direct=false]
         *      If set to true, the position of the text area will be updated
         *      immediately. By default, updating the text area will be done
         *      debounced.
         *  @param {Boolean} [options.size=true]
         *      If set to false, the size of the text area will not be changed,
         *      only its position. Used in situations where the content of the
         *      text area does not change (e.g. while scrolling).
         */
        var updateTextAreaPosition = (function () {

            var // whether updating the position is pending
                updatePosition = false,
                // whether updating the size is pending
                updateSize = false;

            // direct callback, invoked immediately when calling updateTextAreaTokenArray()
            function directCallback(options) {

                // update the formula flag immediately
                // TODO: accept plus and minus after fixing bug 32821
                //isFormula = /^[-+=]/.test(textArea.val());
                isFormula = /^=/.test(textArea.val());

                // set the pending update flags for the deferred callback
                updatePosition = true;
                updateSize = updateSize || Utils.getBooleanOption(options, 'size', true);

                Utils.log('height=' + textArea.outerHeight() + ' inner=' + textArea[0].scrollHeight);
                if (textArea.outerHeight() < textArea[0].scrollHeight) { Utils.log('forced immediate update'); }

                // invoke deferred callback immediately if specified, or if the text has caused a new line break
                if (Utils.getBooleanOption(options, 'direct', false) || (textArea.outerHeight() < textArea[0].scrollHeight)) {
                    deferredCallback();
                }
            }

            // deferred callback, called once after the specified delay
            function deferredCallback() {

                // do nothing if in-place edit mode is not active
                if (!editActive || (!updatePosition && !updateSize)) { return; }
                Utils.log('updating text area');

                var // the position of the edited cell
                    position = null,
                    // the visible area of this grid pane
                    visiblePosition = null,
                    // the current character attributes (with zoomed font size)
                    charAttributes = editCellData.attributes.character,
                    // whether the text in the cell wraps automatically
                    isWrapped = CellCollection.isWrappedText(editCellData),
                    // the effective horizontal CSS text alignment
                    textAlign = textArea.css('text-align'),
                    // text area node and the underlay node, as one jQuery collection
                    targetNodes = textArea.add(textAreaUnderlay),
                    // round up width to multiples of the line height
                    blockWidth = Math.max(32, charAttributes.lineHeight),
                    // the new width and height of the text area
                    targetWidth = 0, targetHeight = 0;

                // returns required width needed to show all text lines in a single line
                function getMinimumWidth() {

                    var // required width for all text lines
                        requiredWidth = _(textArea.val().split(/\n/)).reduce(function (memo, textLine) {
                            return Math.max(memo, fontCollection.getTextWidth(textLine, charAttributes));
                        }, 0);

                    // round up to block width, enlarge to cell width
                    return Math.max(position.width, Utils.roundUp(requiredWidth, blockWidth) + blockWidth);
                }

                // returns maximum width according to horizontal alignment and width of visible area
                function getMaximumWidth() {

                    var // the distance from left visible border to right cell border (for right alignment)
                        leftWidth = Utils.minMax(position.left + position.width - visiblePosition.left, 0, visiblePosition.width),
                        // the distance from left cell border to right visible border (for left alignment)
                        rightWidth = Utils.minMax(visiblePosition.left + visiblePosition.width - position.left, 0, visiblePosition.width),
                        // effective available width
                        availWidth = 0;

                    // get available width according to text alignment
                    switch (textAlign) {
                    case 'right':
                        availWidth = leftWidth;
                        break;
                    case 'center':
                        availWidth = Math.min(leftWidth, rightWidth) * 2 - position.width;
                        break;
                    default:
                        availWidth = rightWidth;
                    }

                    // enlarge to at least half of the width of the visible area
                    return Math.max(availWidth, Math.round(visiblePosition.width / 2));
                }

                // get horizontal and vertical cell position and size
                position = sheetModel.getCellRectangle(editAddress, { expandMerged: true });
                // reduce width by one pixel to exclude the grid line
                position.width = Math.max(blockWidth, position.width - 1);
                // reduce height by one pixel to exclude the grid line
                position.height -= 1;

                // get position and size of the visible area in the grid pane
                visiblePosition = editGridPane.getVisibleRectangle();

                // calculate and set the new size of the text area
                if (updateSize) {

                    var // find best size, stick to cell width in auto-wrapping mode
                        minWidth = isWrapped ? position.width : getMinimumWidth(),
                        maxWidth = isWrapped ? visiblePosition.width : getMaximumWidth();

                    // set resulting width to the text area (hide scroll bars temporarily to get the best height)
                    targetWidth = Math.min(minWidth, maxWidth);
                    textArea.css({ overflowY: 'hidden', width: targetWidth, height: 1 });

                    // width is optimal, get resulting height, restricted to visible area
                    targetHeight = Math.min(Math.max(textArea[0].scrollHeight, position.height), visiblePosition.height);
                    targetNodes.css({ width: targetWidth, height: targetHeight, overflowY: '' });

                } else {
                    targetWidth = textArea.outerWidth();
                    targetHeight = textArea.outerHeight();
                }

                // calculate and set the new position of the text area
                if (updatePosition) {

                    var // left and top position of the text area
                        leftOffset = position.left,
                        topOffset = position.top;

                    // calculate horizontal position according to alignment
                    switch (textAlign) {
                    case 'right':
                        leftOffset += position.width - targetWidth;
                        break;
                    case 'center':
                        leftOffset += Math.round((position.width - targetWidth) / 2);
                        break;
                    }

                    // move text area into visible area, set effective position
                    leftOffset = Utils.minMax(leftOffset, visiblePosition.left, visiblePosition.left + visiblePosition.width - targetWidth);
                    topOffset = Utils.minMax(topOffset, visiblePosition.top, visiblePosition.top + visiblePosition.height - targetHeight);
                    targetNodes.css({ left: leftOffset - visiblePosition.left, top: topOffset - visiblePosition.top });
                }

                updatePosition = updateSize = false;
            }

            // return the debounced method updateTextAreaPosition()
            return app.createDebouncedMethod(directCallback, deferredCallback, { delay: 250, maxDelay: 500 });
        }());

        /**
         * Updates and shows or hides the tooltip attached to the text area,
         * containing the name, parameter list, and description of the current
         * function.
         */
        function updateFunctionSignatureToolTip() {

            var // the cursor position in the text area (off by one to exclude the equality sign)
                textPos = textArea[0].selectionStart - 1,
                // function data for the cursor position
                functionInfo = editTokenArray.getFunctionAtPosition(textPos),
                // the help descriptor of the function, containing the description text
                functionHelp = functionInfo ? app.getFunctionHelp(functionInfo.name) : null,
                // current list separator
                SEP = app.getListSeparator();

            // do nothing if cursor is not inside a function, or no help is available for the function
            if (!functionHelp) {
                textAreaToolTip.hide();
                return;
            }

            // build function name link to OX AppSuite static help pages
            // TODO: add URL of real help files when we have it
            var functionHelpLink = $('<a>', { href: '/appsuite/help/en_US/index.html', target: '_blank' })
                .toggleClass('param-active', functionInfo.paramIndex < 0)
                .text(functionInfo.name);

            var varargIndex = functionHelp.varargs,
                optargIndex = functionHelp.optargs,
                isVararg = _.isNumber(varargIndex),
                isOptarg = _.isNumber(optargIndex),
                paramsArray = _.clone(functionHelp.params),
                isPair = isVararg && _.isString(paramsArray[varargIndex + 1]),
                varargArray = [],
                paramsNode = $('<span>'),
                bracketStartIndex = 0;

            // prepare parameters array plus generate custom parameters if we have an infinite parameter
            if (isVararg) {
                // handle special case with infinite parameter pairs like parameters from the function 'SUMIFS'
                var varargName = paramsArray[varargIndex].trim(),
                    secondVarargName = isPair ? paramsArray[varargIndex + 1].trim() : '',
                    varargParamCount = 1,
                    sliceCount = isPair ? 2 : 1;
                // calculate number of variable parameters to generate
                if (functionInfo.paramCount > paramsArray.length) {
                    varargParamCount = isPair ? (functionInfo.paramCount - varargIndex) / 2 : functionInfo.paramCount - varargIndex;
                }
                // generate variable parameters
                var i = 0;
                while (i <= varargParamCount) {
                    varargArray.push(varargName + (i + 1));
                    if (isPair) varargArray.push(secondVarargName + (i + 1));
                    i++;
                }
                // join initial parameter array with generated variable parameters array
                paramsArray = paramsArray.slice(0, paramsArray.length - sliceCount).concat(varargArray);
                bracketStartIndex = isPair ? varargIndex + 1 : varargIndex;
            }

            if (isOptarg) { bracketStartIndex = optargIndex - 1; }

            // build mark-up and highlight active parameter (draw brackets, add highlight class, add delimiters)
            _(paramsArray).each(function (elem, index, list) {
                var elemMarkup = elem,
                    paramNode = $('<span>');
                // add brackets where necessary
                if (index > bracketStartIndex) {
                    if (isPair) {
                        elemMarkup = (index % 2) ? ('[' + elem) : (elem + ']');
                    } else {
                        elemMarkup = (isVararg || isOptarg) ? ('[' + elem + ']') : elem;
                    }
                }
                paramNode.html(elemMarkup);
                if (index === functionInfo.paramIndex) paramNode.addClass('param-active');
                paramsNode.append(paramNode);
                if (index !== (list.length - 1)) paramsNode.append(SEP + ' ');
            });

            // add parentheses around the parameters and ellipsis in the end if we have infinite parameters
            paramsNode.prepend('( ');
            if (isVararg) paramsNode.append(SEP + ' \u2026');
            paramsNode.append(' )');
            textAreaToolTip.clearContents().appendContentNodes(functionHelpLink, paramsNode);

            // check if we have parameter help data, get it and build its mark-up
            if (_.isArray(functionHelp.paramshelp)) {
                // show right help in case of variable parameters
                var paramHelpIndex = 0;
                if (isVararg && (functionInfo.paramIndex > varargIndex)) {
                    paramHelpIndex = varargIndex + (isPair ? ((functionInfo.paramIndex - varargIndex) % 2) : 0);
                } else {
                    paramHelpIndex = functionInfo.paramIndex;
                }
                var activeParamHelp = functionHelp.paramshelp[paramHelpIndex] || '';
                if (activeParamHelp.length > 0) {
                    textAreaToolTip.appendContentNodes($('<div>').addClass('param-help').text(activeParamHelp));
                }
            }

            // and finally, show the tooltip
            textAreaToolTip.show();
        }

        /**
         * Returns the selected list item node in the drop-down menu element
         * for the text area.
         *
         * @returns {jQuery}
         *  The selected menu item element in the text area drop-down menu.
         */
        function getTextAreaListButtonNode() {
            return textAreaListMenu.getSelectedItemNodes().first();
        }

        /**
         * Updates and shows or hides the tooltip attached to the text area,
         * containing the name, parameter list, and description of the current
         * function.
         */
        function updateFunctionDescriptionToolTip() {

            var // the selected button element in the auto-completion menu
                buttonNode = getTextAreaListButtonNode(),
                // the name and type of the selected list item
                itemValue = Utils.getControlValue(buttonNode),
                // the help descriptor for the selected function (nothing for defined names)
                functionHelp = (_.isObject(itemValue) && (itemValue.type === 'func')) ? app.getFunctionHelp(itemValue.name) : null;

            // only show tooltips for supported functions (not for defined names)
            if (functionHelp) {
                textAreaListItemToolTip.setText(functionHelp.description).show();
            } else {
                textAreaListItemToolTip.hide();
            }
        }

        /**
         * Hides the function signature tooltip and the function
         * auto-completion pop-up menu.
         */
        function hideTextAreaPopups() {
            textAreaToolTip.hide();
            textAreaListMenu.hide();
        }

        /**
         * Shows the function signature tooltip or the function auto-completion
         * pop-up menu if possible.
         */
        function updateTextAreaPopups() {

            var // the cursor position in the text area (off by one to exclude the equality sign)
                textPos = textArea[0].selectionStart - 1,
                // token info for function auto-completion
                tokenInfo = null,
                // the text to be used for function auto-completion
                autoText = null;

            // show pop-ups only in formula mode, do not show them while dragging
            // highlight ranges, or selecting new ranges for the formula
            if (!isFormula ||
                _.isNumber(sheetModel.getViewAttribute('highlightIndex')) ||
                _.isObject(sheetModel.getViewAttribute('activeSelection'))
            ) {
                hideTextAreaPopups();
                return;
            }

            // prefer function auto-completion over function tooltip
            autoTokenInfo = null;

            // auto completion of names: cursor must be located exactly at the end of a function or name token
            if (autoComplete && (tokenInfo = editTokenArray.getTokenAtPosition(textPos)) && (tokenInfo.end === textPos)) {

                // get the text entered before the text cursor (may be a name token or a function token)
                switch (tokenInfo.token.getType()) {
                case 'name':
                    if (!tokenInfo.token.getSheetRef()) { autoText = tokenInfo.token.getName(); }
                    break;
                case 'func':
                    autoText = tokenInfo.token.getName();
                    break;
                }

                // store token info (needed to replace the entered text with the full function name)
                autoTokenInfo = tokenInfo;
            }

            // fill pop-up menu for auto-completion
            if (_.isString(autoText)) {

                // prepare the menu sections
                textAreaListMenu.clearContents();
                textAreaListMenu.createSectionNode('funcs');
                textAreaListMenu.createSectionNode('names');

                // insert all matching function names
                autoText = autoText.toUpperCase();
                _(app.getLocalizedFunctionNames()).each(function (funcName) {
                    if (funcName.substr(0, autoText.length) === autoText) {
                        textAreaListMenu.createItemNode('funcs', { name: funcName, type: 'func' }, { label: funcName });
                    }
                });

                // show the menu, if it contains at least one entry
                if (textAreaListMenu.getItemNodes().length > 0) {
                    textAreaToolTip.hide();
                    textAreaListMenu.show().selectItem(0, { scroll: true });
                    updateFunctionDescriptionToolTip();
                    return;
                }
            }

            // try to show the signature (name and parameters) of the current function
            textAreaListMenu.hide();
            updateFunctionSignatureToolTip();
        }

        /**
         * Updates the function tooltip, and other settings according to the
         * text selection in the text area control.
         */
        function updateTextAreaAfterSelection() {

            // do not open the auto-completion menu even if the new cursor position is valid
            autoComplete = false;

            // do not ignore closing parentheses entered manually anymore (function auto-complete)
            autoCloseData = [];

            // defer execution to let the browser process pending key events
            _.defer(function () {
                // fail-safe check that cell edit mode is still active
                if (editActive) { updateTextAreaPopups(); }
            });
        }

        /**
         * Updates the highlighting of cell references in the current formula
         * while cell in-place edit mode is active.
         */
        function updateTextAreaHighlighting() {

            var // info objects for all ranges in the formula
                rangeInfos = editTokenArray.extractRanges({
                    filterSheet: self.getActiveSheet(),
                    refAddress: editAddress,
                    targetAddress: editAddress,
                    resolveNames: true
                }),
                // unified token indexes of all ranges
                rangeIndexes = _.chain(rangeInfos).pluck('index').unique().value(),
                // the mark-up of the underlay node (span elements containing token texts)
                markup = '';

            // generate the new mark-up for the underlay node
            if (rangeIndexes.length > 0) {
                markup += '<span>=</span>';
                editTokenArray.iterateTokens(function (token, index) {
                    var styleIndex = _(rangeIndexes).indexOf(index, true);
                    markup += '<span data-token-type="' + token.getType() + '"';
                    if (styleIndex >= 0) { markup += ' data-style="' + ((styleIndex % Utils.SCHEME_COLOR_COUNT) + 1) + '"'; }
                    markup += '>' + Utils.escapeHTML(token.getText()) + '</span>';
                });
            }
            textAreaUnderlay[0].innerHTML = markup;
        }

        /**
         * Updates the formula token array according to the current contents of
         * the text area control used for cell in-place edit mode. The token
         * array will trigger a 'change:tokens' event that will cause to redraw
         * the highlighted ranges in all grid panes.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {Boolean} [options.direct=false]
         *      If set to true, the token array and all other dependent
         *      settings will be updated immediately. By default, updating the
         *      token array will be debounced.
         */
        var updateTextAreaTokenArray = (function () {

            // direct callback, invoked immediately when calling updateTextAreaTokenArray()
            function directCallback(options) {

                // update the formula flag immediately (TODO: accept plus and minus after fixing bug 32821)
                //isFormula = /^[-+=]/.test(textArea.val());
                isFormula = /^=/.test(textArea.val());

                // invoke deferred callback immediately if specified
                if (Utils.getBooleanOption(options, 'direct', false)) {
                    deferredCallback();
                }
            }

            // deferred callback, called once after the specified delay
            function deferredCallback() {

                var // the current input string in the text area
                    value = textArea.val();

                // the cell edit mode may have been finished already
                if (!editActive) { return; }

                // parse the formula string
                if (isFormula) {
                    editTokenArray.parseFormula(value.slice(1));
                } else {
                    editTokenArray.clear();
                }

                // insert highlighting mark-up for the underlay node
                updateTextAreaHighlighting();

                // update pop-up nodes and other settings dependent on the text selection
                updateTextAreaPopups();
            }

            // return the debounced method updateTextAreaTokenArray()
            return app.createDebouncedMethod(directCallback, deferredCallback, { delay: 100, maxDelay: 300 });
        }());

        /**
         * Changes the contents and selection of the text area, and updates its
         * position and size.
         *
         * @param {String} text
         *  The new contents of the text area.
         *
         * @param {Number} [start]
         *  The new start position of the selection. If omitted, the text
         *  cursor will be moved to the end of the new text.
         *
         * @param {Number} [end=start]
         *  The new end position of the selection. If omitted, a simple text
         *  cursor will be shown at the specified start position.
         */
        function setTextAreaContents(text, start, end) {
            textArea.val(text);
            lastValue = text;
            Utils.setTextFieldSelection(textArea, _.isNumber(start) ? start : text.length, end);
            updateTextAreaPosition({ direct: true });
            updateTextAreaTokenArray();
        }

        /**
         * Replaces the token before the text cursor with the selected entry
         * from the auto-complete menu.
         */
        function applyAutoCompleteText(itemValue) {

            var // the text to be inserted into the formula, and the new cursor position
                newText = null,
                newStart = 0;

            // always hide the pop-up menu
            textAreaListMenu.hide();

            // fail-safety check
            if (!autoTokenInfo || !_.isObject(itemValue)) { return; }

            // get new text and new cursor position
            newText = itemValue.name;
            newStart = autoTokenInfo.start + newText.length + 1;

            // insert parentheses after function name (unless the existing token is
            // a function token which means it already has an opening parenthesis)
            if (itemValue.type === 'func') {
                if (autoTokenInfo.token.getType() === 'name') {
                    newText += '()';
                    // ignore the next closing parenthesis entered manually
                    autoCloseData.push(true);
                }
                // move cursor behind the opening parenthesis
                newStart += 1;
            }

            // insert the new text into the formula string
            setTextAreaContents(Utils.replaceSubString(textArea.val(), autoTokenInfo.start + 1, autoTokenInfo.end + 1, newText), newStart);
            saveUndoState({ update: true });
        }

        /**
         * Changes the quick edit mode while in-place cell edit mode is active,
         * and shows a status label at the bottom of the application pane.
         *
         * @param {Boolean} newQuickEdit
         *  The new value of the quick edit mode.
         */
        function setQuickEditMode(newQuickEditMode) {

            // change the quick edit state
            if (quickEditMode === newQuickEditMode) { return; }
            quickEditMode = newQuickEditMode;

            // show the status label in the view
            self.showStatusLabel(quickEditMode ?
                //#. Special edit mode in spreadsheet document: quickly type text into cell, use cursor keys to move to next cell
                gt('Insert mode') :
                //#. Standard edit mode in spreadsheet document: edit existing cell text, cursor keys move text cursor inside cell
                gt('Edit mode')
            );
        }

        /**
         * Returns an object that represents the current state of the cell
         * in-place edit mode, intended to be inserted into the undo stack.
         */
        function createUndoStackEntry(fixed) {
            return {
                text: textArea.val(),
                sel: Utils.getTextFieldSelection(textArea),
                attrs: _.copy(editAttributes, true),
                fixed: fixed
            };
        }

        /**
         * Saves the current state of the cell edit mode on top of the undo
         * stack. If the current undo action is not on top of the stack, all
         * following actions will be deleted first.
         */
        function saveUndoState(options) {

            var // whether to try to update the current stack entry with current text
                update = Utils.getBooleanOption(options, 'update', false);

            if (update && (undoIndex + 1 === undoStack.length) && !undoStack[undoIndex].fixed) {
                undoStack[undoIndex].text = textArea.val();
                undoStack[undoIndex].sel = Utils.getTextFieldSelection(textArea);
            } else {
                undoStack.splice(undoIndex + 1);
                undoStack.push(createUndoStackEntry(!update));
            }

            undoIndex = undoStack.length - 1;
            self.trigger('celledit:change');
        }

        /**
         * Restores the current state in the undo stack.
         */
        function restoreUndoState() {

            var // the entry from the undo/redo stack to be restored
                entry = undoStack[undoIndex];

            editAttributes = _.copy(entry.attrs, true);
            setTextAreaContents(entry.text, entry.sel.start, entry.sel.end);
            updateTextAreaStyle();
            self.trigger('celledit:change');
        }

        /**
         * Removes the activated ranges from the view that have been inserted
         * into the formula while cell edit mode is active.
         */
        function resetActiveSelection() {
            sheetModel.setViewAttribute('activeSelection', null);
        }

        /**
         * Updates the position of the text area control after the grid pane
         * has been scrolled (or modified in any other way, e.g. zoom).
         */
        function gridPaneScrollHandler() {
            updateTextAreaPosition({ direct: true, size: false });
        }

        /**
         * Handles 'keydown' events in the text input control, while in-place
         * editing is active.
         */
        function textAreaKeyDownHandler(event) {

            var // the list item currently selected in the auto-completion pop-up menu
                itemNode = textAreaListMenu.getSelectedItemNodes().first(),
                // the value of the item node
                itemValue = Utils.getControlValue(itemNode);

            // first, try to cancel active tracking mode on ESCAPE key
            if ((event.keyCode === KeyCodes.ESCAPE) && $.cancelTracking()) {
                return false;
            }

            // short-cuts in visible pop-up menu for auto-completion
            if (textAreaListMenu.isVisible()) {

                // close pop-up menu on ESCAPE key
                if (event.keyCode === KeyCodes.ESCAPE) {
                    textAreaListMenu.hide();
                    return false;
                }

                // insert selected list item into the formula
                if (KeyCodes.matchKeyCode(event, 'TAB') || KeyCodes.matchKeyCode(event, 'ENTER')) {

                    // Bug 31999: For performance reasons, the token array and the auto-completion pop-up
                    // menu will be updated debounced. To prevent that ENTER/TAB keys insert a function
                    // name unintentionally, the pop-up menu must be updated before applying a list item.
                    updateTextAreaTokenArray({ direct: true });

                    // list menu may have been hidden by the updateTextAreaPopups() call,
                    // do not apply any list item in this case, but leave cell edit mode
                    if (textAreaListMenu.isVisible()) {
                        applyAutoCompleteText(itemValue);
                        return false;
                    }
                }

                // otherwise, try to move the selected list item
                itemNode = textAreaListMenu.getItemNodeForKeyEvent(event, itemNode);
                if (itemNode.length > 0) {
                    textAreaListMenu.selectItemNode(itemNode, { scroll: true });
                    updateFunctionDescriptionToolTip();
                    return false;
                }
            }

            // quit in-place edit mode on ESCAPE key regardless of any modifier keys
            if (event.keyCode === KeyCodes.ESCAPE) {
                self.leaveCellEditMode();
                return false;
            }

            // toggle quick edit mode
            if (KeyCodes.matchKeyCode(event, 'F2')) {
                setQuickEditMode(!quickEditMode);
                // forget last cell range selected in the sheet
                resetActiveSelection();
                return false;
            }

            // ENTER without modifier keys (but ignoring SHIFT): set value, move cell down/up
            if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null })) {
                self.leaveCellEditMode('cell');
                return; // let key event bubble up to move the cell cursor
            }

            // ENTER with CTRL or META key: fill value to all cells (no SHIFT), or insert array formula (SHIFT)
            if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null, ctrlOrMeta: true })) {
                self.leaveCellEditMode(event.shiftKey ? 'array' : 'fill');
                // do not let key event bubble up (no change of selection)
                return false;
            }

            // ENTER with ALT key without any other modifier keys: insert line break
            if (KeyCodes.matchKeyCode(event, 'ENTER', { alt: true })) {
                Utils.replaceTextInTextFieldSelection(textArea, '\n');
                textAreaInputHandler();
                return false;
            }

            // ENTER with any other modifier keys: discard the entire event
            if (event.keyCode === KeyCodes.ENTER) {
                return false;
            }

            // TAB key without modifier keys (but ignoring SHIFT): set value, move cell to next/previous cell
            if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                self.leaveCellEditMode('cell');
                return; // let key event bubble up to move the cell cursor
            }

            // any cursor navigation key
            if (_([KeyCodes.LEFT_ARROW, KeyCodes.RIGHT_ARROW, KeyCodes.UP_ARROW, KeyCodes.DOWN_ARROW, KeyCodes.PAGE_UP, KeyCodes.PAGE_DOWN, , KeyCodes.HOME, KeyCodes.END]).contains(event.keyCode)) {
                if (quickEditMode) {
                    // quick edit mode: do not let the text area process the key events
                    event.preventDefault();
                } else {
                    // standard edit mode: do not process the key events (no change of cell selection)
                    event.stopPropagation();
                    // update pop-up nodes and other settings dependent on the new text selection
                    updateTextAreaAfterSelection();
                    // forget last cell range selected in the sheet
                    resetActiveSelection();
                }
                // let key event bubble up to move the cell cursor, or text cursor in the text area
                return;
            }

            // all other events will be processed by the text field
        }

        /**
         * Handles 'keypress' events in the text input control, while in-place
         * editing is active.
         */
        function textAreaKeyPressHandler(event) {

            // the entered Unicode character
            switch (String.fromCharCode(event.charCode)) {

            // Register an opening parenthesis entered manually. The related closing parenthesis
            // will be inserted even if it occurs among other closing parentheses inserted via
            // function auto-completion (see below).
            case '(':
                autoCloseData.push(false);
                break;

            // When a function name has been inserted from the auto-completion pop-up menu, skip
            // a closing parenthesis because it has been inserted already by the auto-completion).
            // But the text cursor needs to be moved one character forward.
            case ')':
                if (autoCloseData.pop() === true) {
                    Utils.setTextFieldSelection(textArea, Utils.getTextFieldSelection(textArea).start + 1);
                    updateTextAreaPopups();
                    return false;
                }
                break;
            }
        }

        /**
         * Handles any changes in the text area.
         */
        function textAreaInputHandler() {

            var // the current value of the text area
                currValue = textArea.val(),
                // the text selection in the text area control
                textSelection = Utils.getTextFieldSelection(textArea),
                // whether the selection in the text area is a cursor only
                isCursor = textSelection.start === textSelection.end,
                // whether to show type-ahead suggestion in the text area
                typeAhead = (lastValue.length < currValue.length) && isCursor && (textSelection.end === currValue.length);

            // do nothing if the edit text did not change at all
            if (lastValue === currValue) { return; }

            // try to show drop-down list with function names and defined names
            autoComplete = isCursor;

            // update text area settings (position, formula range highlighting)
            lastValue = currValue;
            updateTextAreaPosition();
            updateTextAreaTokenArray();
            saveUndoState({ update: true });

            // Mobile devices may add auto-correction text which modifies the selection after a text change
            // event has been processed. Recalculate the autoComplete flag after the browser has set the final selection.
            _.defer(function () {
                var textSelection = Utils.getTextFieldSelection(textArea);
                autoComplete = textSelection.start === textSelection.end;
            });

            // forget last cell range selected in the sheet
            resetActiveSelection();

            /* Start-of autocompleteText */
            // TODO rework needed
            var columnText = [], suggestions = [];

            if (typeAhead && !isFormula) { //if user writes(adds) new letter -> auto suggest new text

                var address = self.getActiveCell(),
                    boundRange = { start: [address[0], Math.max(0, address[1] - 256)], end: [address[0], Math.min(model.getMaxRow(), address[1] + 256)] };

                cellCollection.iterateCellsInLine(address, 'up down', function (cellData) {
                    if (CellCollection.isText(cellData) && (cellData.result.length > 0)) {
                        columnText.push(cellData.result);
                    }
                }, { type: 'content', hidden: 'all', boundRange: boundRange, skipStartCell: true });

                //if first letter in currValue is LowerCase
                if (currValue.charAt(0) === currValue.charAt(0).toLowerCase() && currValue.charAt(0) !== currValue.charAt(0).toUpperCase()) {
                   //charAt() - returns the character at the specified index in a string.

                    for (var i = 0; i < columnText.length; i++) {
                        columnText[i] = columnText[i].charAt(0).toLowerCase() + columnText[i].substring(1); //make first letter in columText LowerCase
                    }
                    //search user input for matching element in array "columnText"
                    for (var i = 0; i < columnText.length; i++) {
                        if (columnText[i].indexOf(currValue) === 0) { //loop through each element in "columnText" array, to see if it begins with the text in textArea. indexOf() returns 0 if this is the case.
                            suggestions.push(columnText[i]); //fill an array "suggestions" with all possible suggestions
                        }
                    }
                } //end-of if LowerCase
                else {
                    for (var i = 0; i < columnText.length; i++) {
                        columnText[i] = columnText[i].charAt(0).toUpperCase() + columnText[i].substring(1); //make first letter in columText UpperCase
                    }
                    //search user input for matching element in array "columnText"
                    for (var i = 0; i < columnText.length; i++) {
                        if (columnText[i].indexOf(currValue) === 0) {
                            suggestions.push(columnText[i]);
                        }
                    }
                } //end-of else

                if (suggestions.length === 0) { //for mix of LowerCase/UpperCase suggestion
                    for (var i = 0; i < columnText.length; i++) {
                        columnText[i] = columnText[i].toLowerCase();
                    }
                    for (var i = 0; i < columnText.length; i++) {
                        if (columnText[i].indexOf(currValue) === 0) {
                            suggestions.push(columnText[i]);
                        }
                    }
                }
            }

            //sort suggestions by length-ascending
            suggestions.sort(function (a, b) {
                return a.length - b.length;
            });

            //make sure there is at least one suggestion in array "suggestions"
            if (suggestions.length > 0) {
                textArea.focus();
                // suggest new text to user, first best only
                setTextAreaContents(suggestions[0], currValue.length, suggestions[0].length);
                saveUndoState({ update: true });
            }
        }

        /**
         * Handles mousedown or touchstart events in the text area.
         */
        function textAreaMouseTouchHandler() {
            // leave quick-edit mode (cursor keys will move the text cursor)
            setQuickEditMode(false);
            // update pop-up nodes and other settings dependent on the new text selection
            updateTextAreaAfterSelection();
            // forget last cell range selected in the sheet
            resetActiveSelection();
        }

        /**
         * Updates the scroll position of the underlay node, after the scroll
         * position of the text area node has been changed.
         */
        function textAreaScrollHandler() {
            textAreaUnderlay.scrollTop(textArea.scrollTop());
        }

        /**
         * Hides the pop-up nodes if the focus leaves the text area control,
         * unless the auto-completion menu is being clicked.
         */
        var textAreaFocusHandler = (function () {

            var // whether to ignore focus event in the text area (no update of pop-up nodes)
                ignoreFocusEvent = false;

            return function (event) {
                switch (event.type) {

                // text area focused: rebuild all active pop-up nodes
                case 'focus':
                    if (!ignoreFocusEvent) {
                        updateTextAreaPopups();
                    }
                    break;

                // text area lost focus: hide pop-up nodes (unless the auto-complete menu
                // has been clicked, in this case, move focus back to text area immediately)
                case 'blur':
                    _.defer(function () {
                        if (editActive && textAreaListMenu.hasFocus()) {
                            ignoreFocusEvent = true;
                            textArea.focus();
                            // IE sends focus event deferred
                            _.defer(function () { ignoreFocusEvent = false; });
                        } else {
                            hideTextAreaPopups();
                        }
                    });
                }
            };
        }());

        /**
         * Inserts the name clicked in the auto-completion pop-up menu into the
         * current formula text.
         */
        function textAreaListClickHandler(event) {
            var itemValue = Utils.getControlValue($(event.currentTarget));
            applyAutoCompleteText(itemValue);
        }

        /**
         * Returns whether the current text selection in the text area control
         * is valid to start selecting new ranges for the current formula. The
         * text area must contain a formula string (starting with an equality
         * sign), and the start position of the text selection must be located
         * behind the leading equality sign, an operator, a list separator, or
         * an opening parenthesis).
         */
        function isValidRangeSelectionPosition() {

            var // the cursor position in the token array (without leading equality sign)
                textPos = textArea[0].selectionStart - 1,
                // the information for the token at the cursor position
                tokenInfo = null;

            // invalid if not in cell in-place edit mode with a formula
            if (!editActive || !isFormula) { return false; }

            // existing view attribute, or cursor at the beginning of the formula is always valid
            if ((textPos === 0) || _.isObject(sheetModel.getViewAttribute('activeSelection'))) { return true; }

            // get data of the token located before the text cursor
            tokenInfo = editTokenArray.getTokenAtPosition(textPos);

            // cursor must be located between tokens; leading token must be of correct
            // type (or cursor must be located at the beginning of the formula)
            return tokenInfo && (textPos === tokenInfo.end) && (/^(op|sep|open)$/).test(tokenInfo.token.getType());
        }

        /**
         * When in cell edit mode with a formula, returns the active selection
         * which will be used for keyboard navigation and mouse/touch tracking.
         */
        function getCellSelectionHandler() {

            var // the current value of the 'activeSelection' view attribute
                activeSelection = sheetModel.getViewAttribute('activeSelection');

            // always return existing view attribute
            if (_.isObject(activeSelection)) { return activeSelection; }

            // check the text selection in the text area control
            if (!isValidRangeSelectionPosition()) { return null; }

            // start with an empty active selection
            return { ranges: [], activeRange: 0, activeCell: [0, 0] };
        }

        /**
         * When in cell edit mode with a formula, selects a single range and
         * inserts the range address into the formula string.
         */
        function setCellSelectionHandler(selection) {

            var // a helper token array to generate reference addresses for the active ranges
                tokenArray = null,
                // the string representation of the active ranges
                references = null,
                // the new formula text
                formulaText = null;

            // do nothing if ranges cannot be selected with the current text selection
            if (!isValidRangeSelectionPosition()) { return false; }

            // store current text selection, if no range are activated yet
            // (selected text will be replaced with initial range addresses)
            if (!_.isObject(sheetModel.getViewAttribute('activeSelection'))) {
                rangeSelection = Utils.getTextFieldSelection(textArea);
            }

            // build the new formula term to be inserted into the text
            tokenArray = new TokenArray(app, sheetModel, { silent: true, grammar: 'ui' });
            tokenArray.appendRangeList(selection.ranges);
            references = tokenArray.getFormula();
            tokenArray.destroy();
            formulaText = Utils.replaceSubString(textArea.val(), rangeSelection.start, rangeSelection.end, references);

            // insert the new formula text, set cursor behind the range (no real selection)
            rangeSelection.end = rangeSelection.start + references.length;
            setTextAreaContents(formulaText, rangeSelection.end);
            saveUndoState({ update: true });

            // update the view attribute
            sheetModel.setViewAttribute('activeSelection', selection);

            // do not perform standard sheet selection
            return true;
        }

        /**
         * Handles change events from any of the token arrays, and notifies the
         * own event listeners.
         */
        function changeTokenArrayHandler() {

            var // options passed to the extractRanges method
                extractOptions = Utils.extendOptions(highlightOptions, { filterSheet: self.getActiveSheet() }),
                // array of arrays of range info objects from the token arrays
                allRangeInfos = _(highlightTokenArrays).invoke('extractRanges', extractOptions),
                // whether dragging the cell ranges is enabled
                draggable = Utils.getBooleanOption(highlightOptions, 'draggable', false),
                // the flattened highlighted range addresses
                highlightRanges = [];

            _(allRangeInfos).each(function (rangeInfos, arrayIndex) {
                _(rangeInfos).each(function (rangeInfo) {
                    highlightRanges.push(_({
                        // the range identifier containing the index of the token array
                        id: arrayIndex + ',' + rangeInfo.index,
                        // ranges of defined names are not draggable
                        draggable: draggable && (rangeInfo.type === 'ref')
                    }).extend(rangeInfo.range));
                });
            });

            // set the highlighted ranges at the model instance of the active sheet
            sheetModel.setViewAttribute('highlightRanges', highlightRanges);
        }

        /**
         * Registers or unregisters the change listener at all current token
         * arrays.
         */
        function registerTokenArrayListeners(type) {
            _(highlightTokenArrays).invoke(type, 'triggered', changeTokenArrayHandler);
        }

        /**
         * Keeps the reference of the active sheet model up-to-date.
         */
        function changeActiveSheetHandler(event, activeSheet, activeSheetModel) {
            sheetModel = activeSheetModel;
            cellCollection = sheetModel.getCellCollection();
        }

        // methods ------------------------------------------------------------

        /**
         * Returns whether the in-place cell edit mode is currently active.
         *
         * @returns {Boolean}
         *  Whether the in-place cell edit mode is currently active.
         */
        this.isCellEditMode = function () {
            return editActive;
        };

        /**
         * Starts in-place cell edit mode in the active grid pane.
         *
         * @param {Object} [options]
         *  Additional optional parameters:
         *  @param {String} [options.text]
         *      The initial text to insert into the edit area. If omitted, the
         *      current value of the active cell will be inserted.
         *  @param {Number} [options.pos]
         *      The initial position of the text cursor. If omitted, the text
         *      cursor will be placed after the entire text.
         *  @param {Boolean} [options.quick=false]
         *      If set to true, the quick edit mode will be initiated. All
         *      cursor navigation keys will not move the text cursor in the
         *      text area control but will behave as in the normal sheet
         *      selection mode.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.enterCellEditMode = function (options) {

            var // the initial text for the text area
                initialText = Utils.getStringOption(options, 'text'),
                // the initial cursor position for the text area
                initialPos = Utils.getIntegerOption(options, 'pos'),
                // whether the active sheet is protected
                sheetLocked = sheetModel.isLocked();

            // first, check document edit mode
            if (!this.requireEditMode()) { return this; }

            // check whether in-place edit mode is already active
            if (editActive) { return this; }

            // cancel current tracking
            $.cancelTracking();

            // store location of edited cell (sheet may change during range selection mode for formulas)
            editAddress = this.getActiveCell();
            editSheet = this.getActiveSheet();
            editGridPane = this.getActiveGridPane();
            editAttributes = {};

            // initialize CSS formatting of the text area (this initializes the variable editCellData)
            updateTextAreaStyle();

            // scroll to the active cell (before checking edit mode)
            editGridPane.scrollToCell(editAddress);

            // give user some info and quit if cell is locked
            if (sheetLocked && !editCellData.attributes.cell.unlocked) {
                this.yell('info', gt('Protected cells cannot be modified.'));
                return this;
            }

            // initial state of quick edit mode and other settings
            editActive = true;
            setQuickEditMode(Utils.getBooleanOption(options, 'quick', false));

            // the token array stores the parsed formula for range highlighting
            editTokenArray = new TokenArray(app, sheetModel, { grammar: 'ui' });
            this.startRangeHighlighting(editTokenArray, {
                draggable: true,
                refAddress: editAddress,
                targetAddress: editAddress,
                resolveNames: true
            });

            // update formula string in text area while dragging the highlighted ranges
            editTokenArray.on('change:token', function (event, token, index) {
                var tokenPos = editTokenArray.getTokenPosition(index);
                setTextAreaContents(textArea.val()[0] + editTokenArray.getFormula(), tokenPos.end + 1);
                saveUndoState({ update: true });
                // immediately update the highlighted cell references
                updateTextAreaHighlighting();
                resetActiveSelection();
            });

            // insert the text area control into the DOM of the active grid pane, and activate it
            editGridPane.initializeCellEditMode(textArea, textAreaUnderlay);
            editGridPane.on('change:scrollpos', gridPaneScrollHandler);

            // IE9 does not trigger 'input' events when deleting characters or
            // pasting text, use a timer interval as a workaround
            if (Utils.IE9) {
                textArea.data('updateTimer', app.repeatDelayed(textAreaInputHandler, { delay: 250 }));
            }

            // calculate initial value for the text area (formula wins over display string)
            if (_.isString(editCellData.formula)) {
                // hide formula if the cell contains the hidden attribute
                originalText = (sheetLocked && editCellData.attributes.cell.hidden) ? '' : editCellData.formula;
            } else if (_.isObject(editCellData)) {
                // use appropriate representation according to number format category
                originalText = numberFormatter.formatValueForEditMode(editCellData.result, editCellData.format.cat);
            } else {
                // undefined cells
                originalText = '';
            }
            textAreaUnderlay.empty();

            // set initial contents, position, size, and selection of the text area
            setTextAreaContents(_.isString(initialText) ? initialText : originalText, initialPos);

            // initialize local undo/redo stack
            undoStack = [{ text: originalText, sel: _.isString(initialText) ? { start: 0, end: originalText.length } : Utils.getTextFieldSelection(textArea), attrs: {}, fixed: true }];
            undoIndex = 0;

            // selection handler that changes cell ranges in formaulas in cell edit mode
            this.registerCellSelectionHandlers(getCellSelectionHandler, setCellSelectionHandler);

            // set last value and invoke input handler to handle custom initial text
            lastValue = originalText;
            textAreaInputHandler();

            // notify listeners
            return this.trigger('celledit:enter');
        };

        /**
         * Leaves in-place cell edit mode and commits the current text. See
         * method 'GridPane.leaveCellEditMode()' for details.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.leaveCellEditMode = function (commitMode) {

            var // the new cell text
                editText = textArea.val(),
                // the trimmed URL extracted from the value
                url = HyperlinkUtil.checkForHyperlink(editText);

            function addAttribute(family, name, value) {
                editAttributes[family] = editAttributes[family] || {};
                editAttributes[family][name] = value;
            }

            // nothing to do, if in-place edit mode is not active
            if (!editActive) { return this; }

            // deinitialize range highlighting
            editActive = false;
            this.endRangeHighlighting();
            this.unregisterCellSelectionHandlers();
            resetActiveSelection();
            editTokenArray.destroy();
            editTokenArray = null;

            // IE9 does not trigger 'input' events when deleting characters or pasting text,
            // remove the timer interval that has been started as a workaround
            if (textArea.data('updateTimer')) {
                textArea.data('updateTimer').abort();
                textArea.data('updateTimer', null);
            }

            // back to cell selection mode
            quickEditMode = null;
            hideTextAreaPopups();

            // remove the text area control from the DOM of the active grid pane
            editGridPane.off('change:scrollpos', gridPaneScrollHandler);
            editGridPane.cleanupCellEditMode();
            editGridPane = null;
            textArea.detach();
            textAreaUnderlay.detach();

            // notify listeners
            this.trigger('celledit:leave');

            // add auto-wrapping attribute, if the text contains an explicit line break
            if (editText.indexOf('\n') >= 0) {
                addAttribute('cell', 'wrapText', true);
            }

            // add hyperlink specific attributes
            if (_.isString(url) && (url.length > 0)) {
                addAttribute('character', 'url', url);
                addAttribute('character', 'underline', true);
                addAttribute('character', 'color', { type: 'scheme', value: 'hyperlink' });
            }

            // do not pass empty attributes object
            if (_.isEmpty(editAttributes)) { editAttributes = null; }

            // send current value to document model after returning to selection mode
            switch (model.getEditMode() ? commitMode : 'discard') {
            case 'cell':
                // single cell mode: do not send operations if text and formatting did not change
                if ((originalText !== editText) || _.isObject(editAttributes)) {
                    this.setCellContents(editText, editAttributes, { parse: true });
                }
                break;
            case 'fill':
                // fill mode: always fill all cells in the selection
                this.fillCellRanges(editText, editAttributes, { parse: true });
                break;
            case 'array':
                // TODO
                break;
            default:
                commitMode = 'discard';
            }

            return this;
        };

        /**
         * Returns the additional formatting attributes used in in-place cell
         * edit mode.
         *
         * @return {Object|Null}
         *  The new attributes applied while the in-place cell edit mode is
         *  active.
         */
        this.getCellEditAttributes = function () {
            return editActive ? _.copy(editAttributes, true) : null;
        };

        /**
         * Modifies formatting attributes while the in-place cell edit mode is
         * active.
         *
         * @param {Object} [attributes]
         *  The new attributes to be applied to the text area. If omitted, the
         *  call of this method will be ignored.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.setCellEditAttributes = function (attributes) {

            function extendAttributeFamily(family) {
                if (_.isObject(attributes[family])) {
                    _(editAttributes[family] || (editAttributes[family] = {})).extend(attributes[family]);
                }
            }

            if (editActive && _.isObject(attributes)) {

                // extend the edit attribute set
                extendAttributeFamily('cell');
                extendAttributeFamily('character');

                // update the formatting of the text area, and the undo/redo stack
                updateTextAreaStyle();
                updateTextAreaPosition({ direct: true });
                saveUndoState();
            }

            return this;
        };

        // extended undo/redo -------------------------------------------------

        /**
         * Returns whether at least one undo action is available on the undo
         * stack. In cell in-place edit mode, returns whether the cell contents
         * are changed currently.
         *
         * @returns {Boolean}
         *  Whether at least one undo action is available on the stack.
         */
        this.isUndoAvailable = function () {
            return editActive ? (undoIndex > 0) : (undoManager.getUndoCount() > 0);
        };

        /**
         * Applies the topmost undo action on the undo stack. In cell in-place
         * edit mode, restores the original contents of the cell.
         *
         * @returns {jQuery.Promise}
         *  A Promise that will be resolved after the undo action has been
         *  applied.
         */
        this.undo = function () {
            if (editActive) {
                if (undoIndex > 0) {
                    undoIndex -= 1;
                    restoreUndoState();
                }
                return $.when();
            }
            return undoManager.undo(1);
        };

        /**
         * Returns whether at least one redo action is available on the redo
         * stack. In cell in-place edit mode, returns whether the cell contents
         * have been changed, but are not changed anymore (after undo).
         *
         * @returns {Boolean}
         *  Whether at least one redo action is available on the stack.
         */
        this.isRedoAvailable = function () {
            return editActive ? (undoIndex + 1 < undoStack.length) : (undoManager.getRedoCount() > 0);
        };

        /**
         * Applies the topmost redo action on the redo stack. In cell in-place
         * edit mode, restores the last changed contents of the cell, after
         * these changes have been undone.
         *
         * @returns {jQuery.Promise}
         *  A Promise that will be resolved after the redo action has been
         *  applied.
         */
        this.redo = function () {
            if (editActive) {
                if (undoIndex + 1 < undoStack.length) {
                    undoIndex += 1;
                    restoreUndoState();
                }
                return $.when();
            }
            return undoManager.redo(1);
        };

        // range highlighting -------------------------------------------------

        /**
         * Registers token arrays used to render highlighted cell ranges in the
         * active sheet, and updates the 'highlightRanges' view property of the
         * active sheet whenever the token arrays are changing.
         *
         * @param {Array|TokenArray} tokenArrays
         *  An array of token arrays, or a single token array instance.
         *
         * @param {Boolean} [options]
         *  A map with additional options for this method. The following
         *  options are supported:
         *  @param {Boolean} [options.draggable=false]
         *      If set to true, the cell ranges representing reference tokens
         *      will be marked draggable in the 'highlightRanges' sheet view
         *      attribute.
         *  @param {Number[]} [options.refAddress]
         *      The source reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as source reference cell.
         *  @param {Number[]} [options.targetAddress]
         *      The target reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as target reference cell.
         *  @param {Boolean} [options.wrapReferences=false]
         *      If set to true, relocated range addresses that are located
         *      outside the sheet range will be wrapped at the sheet borders.
         *  @param {Boolean} [options.resolveNames=false]
         *      If set to true, the cell ranges contained in defined names will
         *      be highlighted too.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.startRangeHighlighting = function (tokenArrays, options) {

            // store tracking options for usage in event handlers
            highlightOptions = _.clone(options);

            // unregister event handlers at old token arrays
            registerTokenArrayListeners('off');

            // store the token arrays
            highlightTokenArrays = _.chain(tokenArrays).getArray().clone().value();
            registerTokenArrayListeners('on');

            // notify initial ranges to listeners
            changeTokenArrayHandler();
            return this;
        };

        /**
         * Unregisters all token arrays currently registered.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.endRangeHighlighting = function () {
            return this.startRangeHighlighting([]);
        };

        /**
         * Modifies the cell range address of a single reference token inside a
         * single token array currently registered for range highlighting.
         *
         * @param {String} tokenId
         *  The unique identifier of the reference token to be modified, as
         *  contained in the 'highlightRanges' view attribute.
         *
         * @param {Object} range
         *  The new logical range address to be inserted into the reference
         *  token. The absolute/relative state of the reference components will
         *  not be changed.
         *
         * @returns {CellEditMixin}
         *  A reference to this instance.
         */
        this.modifyReferenceToken = function (tokenId, range) {

            var // the token array index, and the token index
                indexes = _(tokenId.split(/,/)).map(function (id) { return parseInt(id, 10); }),
                // the reference token to be modified
                refToken = null;

            // modify the reference token, if the passed identifier is valid
            if ((indexes.length === 2) && (0 <= indexes[0]) && (indexes[0] < highlightTokenArrays.length)) {
                if ((refToken = highlightTokenArrays[indexes[0]].getToken(indexes[1], 'ref'))) {
                    refToken.setRange(range);
                }
            }
            return this;
        };

        // initialization -----------------------------------------------------

        // initialize class members
        app.on('docs:init', function () {

            // resolve reference to document model objects
            model = app.getModel();
            undoManager = model.getUndoManager();
            documentStyles = model.getDocumentStyles();
            fontCollection = model.getFontCollection();
            numberFormatter = model.getNumberFormatter();

            // keep the reference of the active sheet model up-to-date
            self.on('change:activesheet', changeActiveSheetHandler);
        });

        // events for in-place edit mode
        textArea.on({
            keydown: textAreaKeyDownHandler,
            keypress: textAreaKeyPressHandler,
            input: textAreaInputHandler,
            mousedown: textAreaMouseTouchHandler,
            touchstart: textAreaMouseTouchHandler,
            mouseup: textAreaMouseTouchHandler,
            touchend: textAreaMouseTouchHandler,
            scroll: textAreaScrollHandler,
            focus: textAreaFocusHandler,
            blur: textAreaFocusHandler
        });

        // handle click events in the auto-completion pop-up menu
        textAreaListMenu.getNode().on('click', Utils.BUTTON_SELECTOR, textAreaListClickHandler);

        // hide function description tooltip when closing the auto-completion menu
        textAreaListMenu.on('popup:hide', function () { textAreaListItemToolTip.hide(); });

        // destroy all class members on destruction
        this.registerDestructor(function () {

            registerTokenArrayListeners('off');

            textArea.remove();
            textAreaUnderlay.remove();
            textAreaToolTip.destroy();
            textAreaListMenu.destroy();
            textAreaListItemToolTip.destroy();

            app = self = model = undoManager = documentStyles = fontCollection = numberFormatter = null;
            sheetModel = cellCollection = null;
            textArea = textAreaUnderlay = textAreaToolTip = textAreaListMenu = textAreaListItemToolTip = null;
            undoStack = null;
            highlightTokenArrays = highlightOptions = null;
        });

    } // class CellEditMixin

    // exports ================================================================

    return CellEditMixin;

});
