/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/gridpane',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/popup/listmenu',
     'io.ox/office/tk/object/triggerobject',
     'io.ox/office/baseframework/view/baseview',
     'io.ox/office/editframework/model/format/color',
     'io.ox/office/drawinglayer/model/drawingutils',
     'io.ox/office/drawinglayer/view/drawingframe',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/utils/paneutils',
     'io.ox/office/spreadsheet/utils/clipboard',
     'io.ox/office/spreadsheet/model/cellcollection',
     'io.ox/office/spreadsheet/view/mixin/gridtrackingmixin',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, KeyCodes, ListMenu, TriggerObject, BaseView, Color, DrawingUtils, DrawingFrame, SheetUtils, PaneUtils, Clipboard, CellCollection, GridTrackingMixin, gt) {

    'use strict';

    var // maximum number of cells with any contents to be rendered at once in the background loop
        MAX_RENDER_CELL_COUNT = Modernizr.touch ? 75 : _.browser.IE ? 100 : 150;

    // class GridPane =========================================================

    /**
     * Represents a single scrollable grid pane in the spreadsheet view. If the
     * view has been split or frozen at a specific cell position, the view will
     * consist of up to four panes (top-left, top-right, bottom-left, and
     * bottom-right pane).
     *
     * Instances of this class trigger the following events:
     * - 'change:scrollpos':
     *      After the scroll position of the grid pane has been changed.
     *      Listeners may want to update the position of additional DOM content
     *      inserted into the grid pane.
     * - 'select:start'
     *      After selection tracking has been started. Event handlers receive
     *      the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number[]} address
     *          The address of the start cell.
     *      (3) {String} mode
     *          The selection mode according to the keyboard modifier keys:
     *          - 'select': Standard selection without modifier keys.
     *          - 'append': Append new range to current selection (CTRL/META).
     *          - 'extend': Extend current active range (SHIFT).
     * - 'select:move'
     *      After the address of the tracked cell has changed while selection
     *      tracking is active. Event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number[]} address
     *          The address of the current cell.
     * - 'select:end'
     *      After selection tracking has been finished (either successfully or
     *      not). Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number[]} address
     *          The address of the last cell.
     * - 'select:change'
     *      While selection tracking on touch devices is active, and a
     *      selection range has been modified by tracking the resizer handles.
     *      Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} rangeIndex
     *          The array index of the modified range in the current selection.
     *      (3) {Object} range
     *          The new address of the specified range.
     *      (4) {Number[]} [activeCell]
     *          The address of the active cell. If omitted, the current
     *          active cell will not be changed, unless the new active range
     *          does not contain it anymore.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this grid pane.
     *
     * @param {String} panePos
     *  The position of this grid pane.
     */
    function GridPane(app, panePos) {

        var // self reference
            self = this,

            // the spreadsheet model and view
            model = null,
            view = null,

            // the style sheet container of the document
            documentStyles = null,
            fontCollection = null,
            numberFormatter = null,

            // the model and collections of the active sheet
            sheetModel = null,
            cellCollection = null,
            colCollection = null,
            rowCollection = null,
            mergeCollection = null,
            validationCollection = null,
            drawingCollection = null,

            // the column and row header pane associated to the grid pane
            colHeaderPane = null,
            rowHeaderPane = null,

            // the container node of this grid pane (the 'tabindex' attribute makes the node focusable)
            rootNode = $('<div>', { 'data-pos': panePos }).addClass('abs grid-pane f6-target'),

            // the scrollable node of this pane (embed into rootNode to allow to hide scroll bars if needed)
            // (attribute unselectable='on' is needed for IE to prevent focusing the node when clicking on a scroll bar)
            scrollNode = $('<div>', { unselectable: 'on' }).addClass('grid-scroll').appendTo(rootNode),

            // the node controlling the size and position of the scroll bars
            scrollSizeNode = $('<div>').addClass('grid-scroll-size').appendTo(scrollNode),

            // the root node for all layer nodes, positioned dynamically in the scrollable area
            layerRootNode = $('<div>').addClass('grid-layer-root noI18n').appendTo(scrollNode),

            // the grid layer (grid lines below cell nodes)
            gridCanvasNode = $('<canvas>').addClass('grid-canvas').appendTo(layerRootNode),

            // the cell layer (container for the cell nodes)
            cellLayerNode = $('<div>').addClass('grid-layer cell-layer').appendTo(layerRootNode),

            // the collaborative layer node for displaying cursors of other users etc
            collaborativeLayerNode = $('<div>').addClass('grid-layer collab-layer').appendTo(layerRootNode),

            // the selection layer (container for the selected ranges)
            selectionLayerNode = $('<div>').addClass('grid-layer selection-layer').appendTo(layerRootNode),

            // the drawing layer (container for the drawing frames)
            drawingLayerNode = $('<div>').addClass('grid-layer drawing-layer').appendTo(layerRootNode),

            // the range highlight layer (container for the highlighted ranges of formulas)
            highlightLayerNode = $('<div>').addClass('grid-layer highlight-layer').appendTo(layerRootNode),

            // the active range layer (container for the active range selected in cell edit mode)
            activeRangeLayerNode = $('<div>').addClass('grid-layer active-layer').appendTo(layerRootNode),

            // drop-down menu for auto-completion and list validation of active cell
            cellListMenu = null,

            // the clipboard node that carries the browser focus for copy/paste events
            clipboardNode = null,
            clipboardFocusMethod = null,

            // the target node for the browser focus (either clipboard node, or the text area in cell edit mode)
            focusTargetNode = null,

            // the current column and row intervals of the cells shown in this grid pane
            colInterval = null,
            rowInterval = null,

            // the column and row indexes of the cells shown in this grid pane
            colIndexes = null,
            rowIndexes = null,

            // the ratio between (public) absolute scroll position, and internal scroll bars
            scrollLeftRatio = 1, scrollTopRatio = 1,

            // distance between cell A1 and the top-left visible cell of this pane
            hiddenWidth = 0, hiddenHeight = 0,

            // the cell range and absolute position covered by the layer nodes in the sheet area
            layerRange = null,
            layerRectangle = null,

            // all cells to be rendered in a background loop
            pendingCells = {};

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);
        GridTrackingMixin.call(this, app, panePos, layerRootNode);

        // private methods ----------------------------------------------------

        /**
         * Refreshes the appearance of this grid pane according to the position
         * of the grid pane that is currently active.
         */
        function updateFocusDisplay() {

            var // which header pane is currently focused (while selection tracking is active)
                focusPaneSide = sheetModel.getViewAttribute('activePaneSide'),
                // which grid pane is currently focused
                focusPanePos = sheetModel.getViewAttribute('activePane'),
                // whether this grid pane appears focused
                focused = false;

            if (view.isCellEditMode()) {
                // never highlight selection in cell in-place edit mode
                focused = false;
            } else if (view.hasFrozenSplit()) {
                // always highlight all panes in frozen split mode
                focused = true;
            } else if (_.isString(focusPaneSide)) {
                // a header pane is active (selection tracking): focus this
                // grid pane if it is related to the active header pane
                focused = (PaneUtils.getColPaneSide(panePos) === focusPaneSide) || (PaneUtils.getRowPaneSide(panePos) === focusPaneSide);
            } else {
                // a single grid pane is focused
                focused = focusPanePos === panePos;
            }

            rootNode.toggleClass(Utils.FOCUSED_CLASS, focused);
        }

        /**
         * Updates the position of the auto-fill handle and the resize handles
         * for cell selection, when entire columns or rows are selected.
         */
        function updateResizeHandlePositions() {

            var // the auto-fill resizer handle node
                autoFillHandleNode = selectionLayerNode.find('.autofill.resizers>[data-pos]'),
                // the range node containing the auto-fill handle
                autoFillRangeNode = autoFillHandleNode.closest('.range');

            // adjust position of the auto-fill handle node for entire column or row selection
            switch (autoFillHandleNode.attr('data-pos')) {
            case 'r':
                autoFillHandleNode.css({ top: -(layerRootNode.position().top + autoFillRangeNode.position().top) });
                break;
            case 'b':
                autoFillHandleNode.css({ left: -(layerRootNode.position().left + autoFillRangeNode.position().left) });
                break;
            }

            // adjust position of the selection handle node for entire column or row selection
            selectionLayerNode.find('.select.resizers>[data-pos]').each(function () {

                var handleNode = $(this),
                    rangeNode = handleNode.closest('.range');

                switch (handleNode.attr('data-pos')) {
                case 'l':
                case 'r':
                    handleNode.css({ top: -(layerRootNode.position().top + rangeNode.position().top) + (scrollNode[0].clientHeight / 2) });
                    break;
                case 't':
                case 'b':
                    handleNode.css({ left: -(layerRootNode.position().left + rangeNode.position().left) + (scrollNode[0].clientWidth / 2) });
                    break;
                }
            });
        }

        /**
         * Handles changed size of the scroll area received from the column or
         * row header pane.
         */
        function updateScrollAreaSize() {

            var // the available size according to existence of scroll bars
                visibleWidth = scrollNode[0].clientWidth,
                visibleHeight = scrollNode[0].clientHeight,
                // difference of scroll area size between header pane and this grid pane
                widthCorrection = rootNode.width() - visibleWidth,
                heightCorrection = rootNode.height() - visibleHeight,
                // the new absolute width and height of the scrollable area
                scrollWidth = Math.max(0, colHeaderPane.getScrollSize() - widthCorrection),
                scrollHeight = Math.max(0, rowHeaderPane.getScrollSize() - heightCorrection),
                // the real size of the scroll size node (may be smaller due to browser limits)
                effectiveSize = null;

            // initialize the size of the scroll sizer node, the resulting size may be smaller
            effectiveSize = Utils.setContainerNodeSize(scrollSizeNode, scrollWidth, scrollHeight);

            // recalculate ratio between absolute and effective scroll area size
            scrollLeftRatio = (effectiveSize.width > visibleWidth) ? ((scrollWidth - visibleWidth) / (effectiveSize.width - visibleWidth)) : 0;
            scrollTopRatio = (effectiveSize.height > visibleHeight) ? ((scrollHeight - visibleHeight) / (effectiveSize.height - visibleHeight)) : 0;
        }

        /**
         * Handles changed scroll position received from the column or row
         * header pane.
         */
        function updateScrollPosition() {

            var // the new DOM scroll position
                scrollLeft = (scrollLeftRatio > 0) ? Math.round(colHeaderPane.getScrollPos() / scrollLeftRatio) : 0,
                scrollTop = (scrollTopRatio > 0) ? Math.round(rowHeaderPane.getScrollPos() / scrollTopRatio) : 0,
                // the effective DOM node offsets according to the scroll position
                offsetLeft = 0,
                offsetTop = 0;

            // Only set new DOM scroll position if it differs from current position. This test
            // MUST be done, otherwise Firefox scrolls extremely slow when using the mouse wheel!
            if (scrollLeft !== scrollNode[0].scrollLeft) { scrollNode[0].scrollLeft = scrollLeft; }
            if (scrollTop !== scrollNode[0].scrollTop) { scrollNode[0].scrollTop = scrollTop; }

            // calculate the effective offsets according to current scroll position
            offsetLeft = Math.max(-layerRectangle.width * 2, layerRectangle.left - colHeaderPane.getScrollPos() + scrollLeft - hiddenWidth);
            offsetTop = Math.max(-layerRectangle.height * 2, layerRectangle.top - rowHeaderPane.getScrollPos() + scrollTop - hiddenHeight);

            // workaround IE positioning limitations (not possible to set an
            // absolute left/top position of more than Utils.MAX_NODE_SIZE pixels)
            Utils.setPositionInContainerNode(scrollNode, layerRootNode, offsetLeft, offsetTop);

            // set size of the layer root node (should always be inside the node size limits)
            layerRootNode.css({ width: layerRectangle.width, height: layerRectangle.height });

            // update the position of resizer handles (auto fill, touch selection) for column/row selection
            updateResizeHandlePositions();

            // notify listeners
            self.trigger('change:scrollpos');
        }

        /**
         * Returns whether the passed validation settings as returned by the
         * method ValidationCollection.getValidationSettings() describe a cell
         * that will show a drop-down list menu.
         *
         * @param {Object} [settings]
         *  The validation settings received from the validation collection.
         *
         * @returns {Boolean}
         *  Whether the passed validation settings are valid and describe a
         *  cell with source/list validation with a drop-down list menu.
         */
        function isDropDownValidation(settings) {
            return _.isObject(settings) && /^(source|list)$/.test(settings.attributes.type) && settings.attributes.showDropDown;
        }

        /**
         * Invokes the passed iterator function for all cell ranges that are
         * located inside the visible area of this grid pane. The iterator
         * function will receive additional data needed for rendering those
         * ranges in some way.
         *
         * @param {Object|Array}
         *  The logical address of a single cell range, or an array of cell
         *  range addresses.
         *
         * @param {Function} iterator
         *  The iterator function that will be invoked for each range that is
         *  inside the visible area of this grid pane. Receives
         *  the following parameters:
         *  (1) {Object} range
         *      The logical address of the cell range to be rendered.
         *  (2) {Number} index
         *      The array index of the current cell range.
         *  (3) {Object} rectangle
         *      The location of the current range, relative to the layer root
         *      node of this grid pane.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        function iterateRangesForRendering(ranges, iterator) {

            var // the bounding rectangle for range nodes (prevent oversized DOM nodes)
                boundRectangle = { width: Utils.MAX_NODE_SIZE, height: Utils.MAX_NODE_SIZE };

            // get start position of the bounding rectangle for range nodes, to prevent oversized DOM nodes
            boundRectangle.left = Math.max(0, layerRectangle.left - (boundRectangle.width - layerRectangle.width) / 2);
            boundRectangle.top = Math.max(0, layerRectangle.top - (boundRectangle.height - layerRectangle.height) / 2);

            // restrict to ranges in the current cell area to prevent oversized DOM nodes
            // that would collapse to zero size, e.g. when entire columns or rows are selected
            // (for details, see the comments for the Utils.MAX_NODE_SIZE constant)
            return Utils.iterateArray(_(ranges).getArray(), function (range, index) {

                var // position and size of the selection range, restricted to the bounding rectangle
                    rectangle = DrawingUtils.getIntersectionRectangle(sheetModel.getRangeRectangle(range), boundRectangle);

                // skip ranges completely outside the bounding rectangle
                if (!rectangle) { return; }

                // convert range position relative to layer node
                rectangle.left -= layerRectangle.left;
                rectangle.top -= layerRectangle.top;

                // invoke the iterator
                return iterator.call(self, range, index, rectangle);
            });
        }

        /**
         * Renders all visible selection ranges according to the current pane
         * layout data received from a view layout update notification.
         */
        function renderCellSelection() {

            var // the entire cell selection
                selection = view.getSelection(),
                // the range occupied by the active cell (will not be filled)
                activeRange = null,
                // the position of the active cell, extended to the merged range
                activeRectangle = null,
                // additional data for active auto-fill tracking
                autoFillData = sheetModel.getViewAttribute('autoFillData'),
                // whether the active cell touches the borders of selection ranges
                activeBorders = { left: false, top: false, right: false, bottom: false },
                // hide left/top border of selection ranges, if first column/row is hidden
                firstColHidden = !colCollection.isEntryVisible(0),
                firstRowHidden = !rowCollection.isEntryVisible(0),
                // the HTML mark-up for all selection ranges
                markup = '';

            // returns the mark-up of a resizer handler according to the type of the passed range
            function createResizerMarkup(range, leadingCorner) {
                var hPos = leadingCorner ? 'l' : 'r', vPos = leadingCorner ? 't' : 'b';
                return '<div data-pos="' + (model.isColRange(range) ? hPos : model.isRowRange(range) ? vPos : (vPos + hPos)) + '"></div>';
            }

            function enlargeLeft(rectangle, size) {
                rectangle.left -= size;
                rectangle.width += size;
            }

            function enlargeTop(rectangle, size) {
                rectangle.top -= size;
                rectangle.height += size;
            }

            // add active flag to range object
            selection.ranges[selection.activeRange].active = true;

            // adjust single range for auto-fill tracking
            if ((selection.ranges.length === 1) && _.isObject(autoFillData)) {

                var // the selected cell range
                    firstRange = selection.ranges[0],
                    // whether to expand/shrink the leading or trailing border
                    leading = /^(left|top)$/.test(autoFillData.border),
                    // whether to expand/shrink columns or rows
                    columns = /^(left|right)$/.test(autoFillData.border),
                    // the array index in the cell address
                    addrIndex = columns ? 0 : 1;

                activeRange = _.copy(firstRange, true);
                if (autoFillData.count >= 0) {
                    // adjust range for auto-fill mode
                    if (leading) {
                        firstRange.start[addrIndex] -= autoFillData.count;
                    } else {
                        firstRange.end[addrIndex] += autoFillData.count;
                    }
                } else {
                    // adjust range for deletion mode
                    firstRange.collapsed = true;
                    if (-autoFillData.count < (columns ? SheetUtils.getColCount(firstRange) : SheetUtils.getRowCount(firstRange))) {
                        // range partly covered
                        if (leading) {
                            activeRange.start[addrIndex] -= autoFillData.count;
                        } else {
                            activeRange.end[addrIndex] += autoFillData.count;
                        }
                    } else {
                        // remove active range for deletion of the entire range
                        activeRange = {};
                    }
                }

                // add tracking style effect
                firstRange.tracking = true;
            }

            // the range covered by the active cell (or any other range in auto-fill tracking mode)
            activeRange = _.isObject(activeRange) ? (_.isEmpty(activeRange) ? null : activeRange) :
                mergeCollection.expandRangeToMergedRanges({ start: selection.activeCell, end: selection.activeCell });

            // convert active range to a rectangle relative to the layer root node
            if (_.isObject(activeRange)) {
                iterateRangesForRendering(activeRange, function (range, index, rectangle) {
                    activeRectangle = rectangle;
                });
            }

            // render all ranges that are visible in this grid pane
            iterateRangesForRendering(selection.ranges, function (range, index, rectangle) {

                var // position of active range, relative to current range
                    relActiveRect = null,
                    // the style attribute
                    styleAttr = '',
                    // mark-up for the fill elements and other helper nodes
                    fillMarkup = '';

                // enlarge rectangle by one pixel to the left and top, to get matching
                // border line positions in multi-selection
                enlargeLeft(rectangle, 1);
                enlargeTop(rectangle, 1);

                // hide left/top border of the range, if the first column/row is hidden
                if (firstColHidden && (range.start[0] === 0)) {
                    enlargeLeft(rectangle, 5);
                }
                if (firstRowHidden && (range.start[1] === 0)) {
                    enlargeTop(rectangle, 5);
                }

                // insert the semi-transparent fill elements (never cover the active range with the fill elements)
                if (activeRectangle && SheetUtils.rangesOverlap(range, activeRange)) {

                    // initialize position of active cell, relative to selection range
                    relActiveRect = _.clone(activeRectangle);
                    relActiveRect.left -= rectangle.left;
                    relActiveRect.top -= rectangle.top;
                    relActiveRect.right = rectangle.width - relActiveRect.left - relActiveRect.width;
                    relActiveRect.bottom = rectangle.height - relActiveRect.top - relActiveRect.height;

                    // insert fill element above active cell
                    if (relActiveRect.top > 0) {
                        fillMarkup += '<div class="fill" style="left:0;right:0;top:0;height:' + relActiveRect.top + 'px;"></div>';
                    }

                    // insert fill element left of active cell
                    if (relActiveRect.left > 0) {
                        fillMarkup += '<div class="fill" style="left:0;width:' + relActiveRect.left + 'px;top:' + relActiveRect.top + 'px;height:' + relActiveRect.height + 'px;"></div>';
                    }

                    // insert fill element right of active cell
                    if (relActiveRect.right > 0) {
                        fillMarkup += '<div class="fill" style="left:' + (relActiveRect.left + relActiveRect.width) + 'px;right:0;top:' + relActiveRect.top + 'px;height:' + relActiveRect.height + 'px;"></div>';
                    }

                    // insert fill element below active cell
                    if (relActiveRect.bottom > 0) {
                        fillMarkup += '<div class="fill" style="left:0;right:0;top:' + (relActiveRect.top + relActiveRect.height) + 'px;bottom:0;"></div>';
                    }

                    // update border flags for the active cell
                    activeBorders.left = activeBorders.left || (range.start[0] === activeRange.start[0]);
                    activeBorders.top = activeBorders.top || (range.start[1] === activeRange.start[1]);
                    activeBorders.right = activeBorders.right || (range.end[0] === activeRange.end[0]);
                    activeBorders.bottom = activeBorders.bottom || (range.end[1] === activeRange.end[1]);

                } else {
                    fillMarkup += '<div class="abs fill"></div>';
                }

                // build the style attribute for the root node of the selection range
                styleAttr += PaneUtils.getRectangleStyleMarkup(rectangle);

                // generate the HTML mark-up for the selection range
                markup += '<div class="range';
                if (range.active) { markup += ' active'; }
                if (range.tracking) { markup += ' tracking-active'; }
                if (range.collapsed) { markup += ' collapsed'; }
                markup += '" style="' + styleAttr + '" data-index="' + index + '">' + fillMarkup + '<div class="border"></div>';

                // additional mark-up for single-range selection
                if (selection.ranges.length === 1) {
                    if (Modernizr.touch && !_.isObject(autoFillData)) {
                        // add resize handlers for selection on touch devices
                        markup += '<div class="select resizers">' + createResizerMarkup(range, true) + createResizerMarkup(range, false) + '</div>';
                    } else {
                        // add the auto-fill handler in the bottom right corner of the selection
                        markup += '<div class="autofill resizers">' + createResizerMarkup(range, false) + '</div>';
                    }
                }

                markup += '</div>';
            });

            // additions for the active cell (or active range in auto-fill)
            if (activeRectangle) {

                // add thin border for active cell on top of the selection
                enlargeLeft(activeRectangle, 1);
                enlargeTop(activeRectangle, 1);
                markup += '<div class="active-cell';
                _(activeBorders).each(function (isBorder, borderName) { if (!isBorder) { markup += ' ' + borderName; } });
                markup += '" style="' + PaneUtils.getRectangleStyleMarkup(activeRectangle) + '">';

                // add drop-down button for list validation
                if (!_.isObject(autoFillData) && model.getEditMode() && isDropDownValidation(validationCollection.getValidationSettings(selection.activeCell))) {
                    markup += '<div class="cell-dropdown-button skip-tracking"';
                    if (activeBorders.right) { markup += ' style="margin-left:2px;"'; }
                    markup += '>' + Utils.createIconMarkup('fa-caret-down') + '</div>';
                }

                // close the active cell node
                markup += '</div>';
            }

            // insert entire HTML mark-up into the selection container node
            selectionLayerNode[0].innerHTML = markup;

            // update the position of resizer handles (auto fill, touch selection) for column/row selection
            updateResizeHandlePositions();
            view.updateHyperlinkPopup();
        }

        /**
         * Renders all visible highlighted cell ranges.
         */
        function renderHighlightedRanges() {

            var // all ranges currently highlighted
                highlightRanges = sheetModel.getViewAttribute('highlightRanges'),
                // unified range identifiers
                rangeIds = _.chain(highlightRanges).pluck('id').unique().value(),
                // array index of the range currently tracked
                highlightIndex = sheetModel.getViewAttribute('highlightIndex'),
                // the HTML mark-up for all highlight ranges
                markup = '';

            // render all ranges that are visible in this grid pane
            if (_.isArray(highlightRanges)) {
                iterateRangesForRendering(highlightRanges, function (range, index, rectangle) {

                    var // array index of the range identifier (multiple ranges may origin from the same defined name)
                        idIndex = rangeIds.indexOf(range.id),
                        // CSS class name for range color style
                        styleId = (idIndex % Utils.SCHEME_COLOR_COUNT) + 1;

                    // generate the HTML mark-up for the highlighted range
                    markup += '<div class="range ' + ((highlightIndex === index) ? ' tracking-active' : '') + '" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"';
                    markup += ' data-style="' + styleId + '" data-index="' + index + '"><div class="fill"></div>';
                    if (model.getEditMode() && Utils.getBooleanOption(range, 'draggable', false)) {
                        markup += '<div class="borders">';
                        markup += _(['l', 'r', 't', 'b']).map(function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                        markup += '</div><div class="resizers">';
                        markup += _(['tl', 'tr', 'bl', 'br']).map(function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                        markup += '</div>';
                    } else {
                        markup += '<div class="static-border"></div>';
                    }
                    markup += '</div>';
                });
            }

            // insert entire HTML mark-up into the container node
            highlightLayerNode[0].innerHTML = markup;
        }

        /**
         * Renders the activated cell ranges. These ranges are used for example
         * while selecting new ranges for a formula while in cell edit mode.
         */
        var renderActiveSelection = (function () {

            var // the timer for the wireframe animation
                timer = null;

            return function () {

                var // the active ranges
                    selection = sheetModel.getViewAttribute('activeSelection'),
                    // the HTML mark-up for the range
                    markup = '';

                // generate the HTML mark-up for the ranges
                if (_.isObject(selection)) {
                    iterateRangesForRendering(selection.ranges, function (range, index, rectangle) {
                        markup += '<div class="range" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"><div class="borders">';
                        markup += _(['l', 'r', 't', 'b']).map(function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                        markup += '</div></div>';
                    });
                    if (!timer) {
                        timer = app.repeatDelayed(function (index) {
                            activeRangeLayerNode.attr('data-frame', index % 6);
                        }, { delay: 100 });
                    }
                } else if (timer) {
                    timer.abort();
                    timer = null;
                }

                // insert entire HTML mark-up into the container node
                activeRangeLayerNode[0].innerHTML = markup;
            };
        }());

        /**
         * Renders collaborative user selections.
         */
        function renderCollaborativeSelections() {

            var // current user selections to render
                collabSelections = sheetModel.getViewAttribute('collaborativeSelections'),
                // the style color index (different per user)
                styleIndex = 0,
                // the HTML mark-up for all user selections
                markup = '';

            // iterate all collaborating users
            _(collabSelections).each(function (collabSelection) {

                // ignore own cursor data
                if ((collabSelection.userId === app.getClientId()) || !collabSelection.selection) { return; }

                var // the escaped user name, ready to be inserted into HTML mark-up
                    userName = Utils.escapeHTML(collabSelection.userName);

                // render all ranges of the user
                iterateRangesForRendering(collabSelection.selection.ranges, function (range, index, rectangle) {

                    // adjust left and top border to fit the range to the grid lines
                    rectangle.left -= 1;
                    rectangle.top -= 1;
                    rectangle.width += 1;
                    rectangle.height += 1;

                    // create the mark-up for the range
                    markup += '<div class="range" data-style="' + ((styleIndex % Utils.SCHEME_COLOR_COUNT) + 1) + '" data-username="' + userName + '"';
                    markup += ' style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"><div class="borders">';
                    markup += _(['l', 'r', 't', 'b']).map(function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                    markup += '</div></div>';
                });

                // new color for next user
                styleIndex += 1;
            });

            // insert entire HTML mark-up into the container node
            collaborativeLayerNode[0].innerHTML = markup;
        }

        /**
         * Updates the position of the passed drawing frame according to the
         * anchor attributes in the passed attribute set.
         */
        function updateDrawingFramePosition(drawingModel, drawingFrame) {

            var // the effective drawing rectangle (null if the drawing is hidden)
                rectangle = drawingModel.getRectangle();

            if (rectangle) {
                // convert position relative to layer node
                rectangle.left -= layerRectangle.left;
                rectangle.top -= layerRectangle.top;
                // show the drawing frame, and set its position and size
                drawingFrame.show().css(rectangle);
            } else {
                // hide the drawing frame if it is located in hidden columns or rows
                drawingFrame.hide();
            }
        }

        /**
         * Creates and inserts a new DOM drawing frame node that represents the
         * passed drawing model.
         */
        function createDrawingFrame(drawingModel, position) {

            var // the new drawing frame, as jQuery object
                drawingFrame = DrawingFrame.createDrawingFrame(drawingModel),
                // the merged drawing attributes
                attributes = drawingModel.getMergedAttributes(),
                // all existing drawing frames, as plain DOM collection
                drawingNodes = drawingLayerNode[0].childNodes,
                // the DOM child index (Z order) specified by the logical position
                index = position[0];

            // add the unique identifier of the drawing model for DOM look-up
            drawingFrame.attr('data-model-uid', drawingModel.getUid());

            // insert the drawing frame with the correct Z order
            if (drawingNodes.length === index) {
                drawingLayerNode.append(drawingFrame);
            } else {
                drawingFrame.insertBefore(drawingNodes[index]);
            }

            // update position and formatting
            updateDrawingFramePosition(drawingModel, drawingFrame);
            DrawingFrame.updateFormatting(app, drawingFrame, attributes);
        }

        /**
         * Creates DOM drawing frames for all existing drawing objects in the
         * current active sheet.
         */
        function createDrawingFrames() {
            drawingCollection.iterateModelsByPosition(createDrawingFrame);
            renderDrawingSelection();
        }

        /**
         * Updates the position and formatting of the drawing frame that
         * represents the passed drawing model.
         */
        function updateDrawingFrame(drawingModel) {

            var // the existing DOM drawing frame
                drawingFrame = self.getDrawingFrame(drawingModel.getUid());

            // update position and formatting of frames inside the visible area
            if (SheetUtils.rangesOverlap(layerRange, drawingModel.getRange())) {
                updateDrawingFramePosition(drawingModel, drawingFrame);
                DrawingFrame.updateFormatting(app, drawingFrame, drawingModel.getMergedAttributes());
            } else {
                drawingFrame.hide();
            }
        }

        /**
         * Updates position and formatting of all drawing frames in the current
         * visible area or the passed bounding range.
         */
        function updateDrawingFrames() {
            drawingCollection.iterateModelsByPosition(updateDrawingFrame);
        }

        /**
         * Updates the selection of all drawing frames.
         */
        function renderDrawingSelection() {

            var // all drawing frames currently selected in the DOM (also embedded drawings)
                oldSelectedFrames = drawingLayerNode.find(DrawingFrame.NODE_SELECTOR + Utils.SELECTED_SELECTOR),
                // all existing drawing frames, as plain DOM collection
                drawingNodes = drawingLayerNode[0].childNodes,
                // disable tracking in read-only mode, or if sheet is protected
                editMode = model.getEditMode() && !sheetModel.isLocked();

            // process all drawings to be selected
            _(view.getSelectedDrawings()).each(function (position) {

                var // the root drawing frame addressed by the current logical position
                    drawingFrame = drawingNodes[position[0]];

                // drawing frame may be missing, e.g. while remotely inserting new drawings
                // TODO: find embedded drawing frames
                if (drawingFrame && (position.length === 1)) {
                    DrawingFrame.drawSelection(drawingFrame, { movable: editMode, resizable: editMode });
                    oldSelectedFrames = oldSelectedFrames.not(drawingFrame);
                }
            });

            // remove selection border from all drawing frames not selected anymore
            oldSelectedFrames.each(function () { DrawingFrame.clearSelection(this); });
        }

        /**
         * Processes a new drawing object inserted into the drawing collection.
         */
        function insertDrawingHandler(event, drawingModel, position) {
            createDrawingFrame(drawingModel, position);
            renderDrawingSelection();
        }

        /**
         * Processes a drawing object removed from the drawing collection.
         */
        function deleteDrawingHandler(event, drawingModel) {
            self.getDrawingFrame(drawingModel.getUid()).remove();
            renderDrawingSelection();
        }

        /**
         * Processes a drawing object that has been changed in any way.
         */
        function changeDrawingHandler(event, drawingModel) {
            updateDrawingFrame(drawingModel);
        }

        /**
         * Renders the grid lines between the cells.
         */
        function renderCellGrid() {

            var // whether grid lines are currently visible
                showGrid = sheetModel.getViewAttribute('showGrid'),
                // the current grid color
                gridColor = sheetModel.getEffectiveGridColor(),
                // the canvas renderer context
                context = null,
                // all merged ranges in the visible area
                mergedRanges = null;

            // update the visibility of the grid lines
            rootNode.toggleClass('grid-hidden', !showGrid);

            // performance: do not render anything if grid is invisible
            if (!showGrid) { return; }

            // draws a vertical line at the specified position
            function drawVertLine(x, y1, y2) {
                context.moveTo(x, y1);
                context.lineTo(x, y2);
            }

            // draws a horizontal line at the specified position
            function drawHorLine(y, x1, x2) {
                context.moveTo(x1, y);
                context.lineTo(x2, y);
            }

            // renders either all vertical or all horizontal grid lines
            function renderGridLines(columns) {

                var // the array index into cell addresses
                    addrIndex = columns ? 0 : 1,
                    // the column/row collection for the entries
                    collection1 = columns ? colCollection : rowCollection,
                    // the column/row collection in the other direction (for intervals of merged ranges)
                    collection2 = columns ? rowCollection : colCollection,
                    // the column/row interval
                    interval = columns ? SheetUtils.getColInterval(layerRange) : SheetUtils.getRowInterval(layerRange),
                    // the merged ranges spanning over multiple entries in the current direction
                    spanningMergedRanges = _(mergedRanges).filter(function (range) {
                        return range.start[addrIndex] < range.end[addrIndex];
                    }),
                    // the function to convert merged ranges into column/row intervals
                    getIntervalsFunc = columns ? SheetUtils.getRowIntervals : SheetUtils.getColIntervals,
                    // the layer offset in the current direction
                    layerOffset = columns ? layerRectangle.top : layerRectangle.left,
                    // the layer size in the opposite direction
                    layerSize = columns ? layerRectangle.height : layerRectangle.width,
                    // the function to draw a line into the canvas
                    drawLineFunc = columns ? drawVertLine : drawHorLine;

                // draw the grid lines for all columns/rows
                collection1.iterateEntries(interval, function (entry) {

                    var // the position of the lines for the current column/row
                        offset = entry.offset + entry.size - 1 - (columns ? layerRectangle.left : layerRectangle.top),
                        // all merged ranges that will break the grid line
                        coveredMergedRanges = _(spanningMergedRanges).filter(function (range) {
                            return (range.start[addrIndex] <= entry.index) && (entry.index < range.end[addrIndex]);
                        }),
                        // the merged column/row intervals of the merged ranges
                        mergedIntervals = getIntervalsFunc(coveredMergedRanges),
                        // the coordinates of the line segments, as flat array
                        lineCoords = null;

                    // calculate the line start/end coordinates
                    lineCoords = _.chain(mergedIntervals)
                        // get the pixel positions of the merged column/row intervals
                        .map(_.bind(collection2.getIntervalPosition, collection2))
                        // filter hidden merged ranges
                        .filter(function (position) { return position.size > 0; })
                        // adjust layer offset in positions and convert to (nested) integer array
                        .map(function (position) {
                            var offset = position.offset - layerOffset;
                            return [offset - 1, offset + position.size - 1];
                        })
                        // convert to a flat array, add start and end position of the layer rectangle
                        .flatten().unshift(-1).push(layerSize)
                        // get the resulting array
                        .value();

                    // draw all line segments
                    for (var index = 0, length = lineCoords.length; index < length; index += 2) {
                        if (lineCoords[index] < lineCoords[index + 1]) {
                            drawLineFunc(offset, lineCoords[index], lineCoords[index + 1]);
                        }
                    }
                });
            }

            // special CSS formatting for automatic grid color
            gridCanvasNode.toggleClass('auto-color', sheetModel.isAutoGridColor());

            // set the size of the canvas element as element attributes
            gridCanvasNode.attr({ width: layerRectangle.width, height: layerRectangle.height });

            // get the renderer context
            context = gridCanvasNode[0].getContext('2d');
            // get all merged ranges in the visible area
            mergedRanges = mergeCollection.getMergedRanges(layerRange);

            // initialize canvas
            context.translate(0.5, 0.5);
            context.clearRect(0, 0, layerRectangle.width, layerRectangle.height);
            context.lineWidth = 1;
            context.strokeStyle = documentStyles.getCssColor(gridColor, 'line');

            // render all horizontal and vertical grid lines
            context.beginPath();
            renderGridLines(false);
            renderGridLines(true);
            context.stroke();
        }

        /**
         * Returns the next available non-empty cell inside the layer range.
         *
         * @param {Number[]} address
         *  The address of the cell whose nearest adjacent content cell will be
         *  searched.
         *
         * @param {String} direction
         *  The direction to look for the content cell. Must be one of the
         *  values 'left', 'right', 'up', or 'down'.
         *
         * @returns {Object|Null}
         *  The descriptor of an existing content cell; or null, if no content
         *  cell has been found.
         */
        function findNearestContentCell(address, direction) {

            var // the resulting cell entry
                cellData = null;

            // iterate to the first available content cell
            cellCollection.iterateCellsInLine(address, direction, function (currCellData) {
                cellData = currCellData;
                return Utils.BREAK;
            }, { type: 'content', boundRange: layerRange, skipStartCell: true });

            return cellData;
        }

        /**
         * Returns the nearest column/row index interval in the passed sorted
         * (!) array of index intervals for the specified column/row index, by
         * performing a fast binary search on the interval array.
         *
         * @param {Array} intervals
         *  The sorted and unified array of column/row index intervals.
         *
         * @param {Number} index
         *  A single column/row index to be checked.
         *
         * @param {Boolean} forward
         *  Whether to look for an interval following (true) or preceding
         *  (false) the passed index.
         *
         * @returns {Object|Null}
         *  An interval from the passed array, that precedes or follows the
         *  specified index, but does not contain that index; or null, if no
         *  such interval exists in the array.
         */
        function findNearestInterval(intervals, index, forward) {

            var // the interval found in the array
                nextInterval = forward ?
                    Utils.findFirst(intervals, function (interval) { return index < interval.first; }, { sorted: true }) :
                    Utils.findLast(intervals, function (interval) { return interval.last < index; }, { sorted: true });

            return nextInterval ? nextInterval : null;
        }

        /**
         * Returns the clipping and overflow settings for the specified cell
         * (the additional space at the left and right side of the specified
         * cell that can be used for visible overlapping text contents).
         *
         * @param {Object} cellData
         *  The cell descriptor, as returned e.g. by the cell collection. If
         *  the cell is part of a merged range, this object MUST contain the
         *  additional property 'mergedRange' with the logical address of the
         *  merged range.
         *
         * @returns {Object}
         *  The available space at the left and right side of the cell, in the
         *  following properties:
         *  - {Number} clipLeft
         *      The additional space left of the cell that can be used to show
         *      the text contents.
         *  - {Number} clipRight
         *      The additional space right of the cell that can be used to show
         *      the text contents.
         *  - {Number} hiddenLeft
         *      Additional width left of the visible clip region needed to have
         *      enough space for very long text without reaching the content
         *      node borders.
         *  - {Number} hiddenRight
         *      Additional width right of the visible clip region needed to
         *      have enough space for very long text without reaching the
         *      content node borders.
         */
        function calculateCellClipping(cellData) {

            var // the address of the passed cell
                address = cellData.address,
                // the effective CSS text alignment
                textAlign = PaneUtils.getCssTextAlignment(cellData),
                // no overflow left of cell needed, if cell is left-aligned or justified
                overflowLeft = /^(right|center)$/.test(textAlign),
                // no overflow right of cell needed, if cell is right-aligned or justified
                overflowRight = /^(left|center)$/.test(textAlign),
                // the column intervals of all merged ranges covering the row
                mergedColIntervals = cellData.mergedColIntervals,
                // the nearest content cell left or right of the cell
                nextCellData = null,
                // the nearest merged column interval left or right of the cell
                nextMergedInterval = null,
                // column index of the empty cell next to the content cell or merged range
                clipCol = 0,
                // the resulting clip settings
                clippingData = { clipLeft: 0, clipRight: 0, hiddenLeft: 0, hiddenRight: 0 };

            // calculate clipping distance left and right of the cell (but not
            // for merged ranges, they will always be cropped at cell boders)
            if (!_.isObject(cellData.mergedRange) && (overflowLeft || overflowRight)) {

                // calculate the column intervals of all merged ranges covering the row
                // (if not contained in the passed cell data)
                if (!mergedColIntervals) {
                    mergedColIntervals = SheetUtils.getColIntervals(mergeCollection.getMergedRanges(model.makeRowRange(address[1])));
                }

                // calulate left clipping size (width of empty cells between the preceding
                // content cell or merged range, and the cell passed to this method)
                if (overflowLeft) {
                    clipCol = layerRange.start[0];
                    if ((nextCellData = findNearestContentCell(address, 'left'))) {
                        clipCol = Math.max(clipCol, nextCellData.address[0] + 1);
                    }
                    if ((nextMergedInterval = findNearestInterval(mergedColIntervals, address[0], false))) {
                        clipCol = Math.max(clipCol, nextMergedInterval.last + 1);
                    }
                    if (clipCol < address[0]) {
                        clippingData.clipLeft = colCollection.getIntervalPosition({ first: clipCol, last: address[0] - 1 }).size;
                    }
                }

                // calulate right clipping size (width of empty cells between the cell
                // passed to this method, and the following content cell or merged range)
                if (overflowRight) {
                    clipCol = layerRange.end[0];
                    if ((nextCellData = findNearestContentCell(address, 'right'))) {
                        clipCol = Math.min(clipCol, nextCellData.address[0] - 1);
                    }
                    if ((nextMergedInterval = findNearestInterval(mergedColIntervals, address[0], true))) {
                        clipCol = Math.min(clipCol, nextMergedInterval.first - 1);
                    }
                    if (address[0] < clipCol) {
                        clippingData.clipRight = colCollection.getIntervalPosition({ first: address[0] + 1, last: clipCol }).size;
                    }
                }
            }

            // reserve 1 pixel at right border for grid lines
            clippingData.clipRight -= 1;

            // calculate size of additional hidden space for the entire text contents
            if (overflowLeft) { clippingData.hiddenLeft = 500000 - clippingData.clipLeft; }
            if (overflowRight) { clippingData.hiddenRight = 500000 - clippingData.clipRight; }

            return clippingData;
        }

        /**
         * Renders the specified number of cells from the 'pendingCells'
         * collection, and removes the information of the rendered cells from
         * that collection.
         *
         * @param {Number} contentCount
         *  The maximum number of cells with any text contents to be rendered
         *  during this invokation. The total number of rendered cells (empty
         *  or with text contents) will be restricted to the fourfold of this
         *  value.
         */
        function renderPendingCells(contentCount) {

            // Bug 31125: to prevent performance problems when rendering cells,
            // modifying the DOM (insert contents, change node size and position),
            // and accessing measument properties of DOM elements (size, position)
            // MUST NOT be performed alernatingly. After any modification of the
            // DOM the browser will update its internal rendering tree before it
            // returns any measurement properties. That update may take a few dozen
            // milliseconds (especially Chrome and Internet Explorer), thus doing
            // that for each cell will result in a massive performance fall-off.

            // This method renders the specified number of cells, and performs each
            // step (create HTML mark-up, insert mark-up, post-process cells) for
            // all rendered cells at once, before continuing with the next step.

            // Cell nodes will be ordered according to whether they have text contents.
            // This is needed to work-around a Safari bug on old iPads which does not
            // respect the z-order settings of the cell content nodes correctly and
            // draws cells with background color above overflowing text nodes from
            // preceding cells in the DOM, regardless of the increased z-order of the
            // text spans.

            var // all pending cells, in an array sorted by cell address
                sortedCells = _(pendingCells).values(),
                // all cells to be rendered in this invocation
                allCells = [],
                // all cells with any text contents to be rendered in this invocation
                contentCells = [],
                // maximum number of cells to be rendered (empty or with contents)
                renderCount = contentCount * 4,
                // maximum number of cells to be visited from the queue
                maxCount = renderCount * 2,
                // the inner padding for text content nodes (according to zoom)
                cellPadding = sheetModel.getEffectiveCellPadding(),
                // difference between cell width and maximum content width (cell padding left/right, 1px grid line)
                totalPadding = 2 * cellPadding + 1,
                // grid color used for 'automatic' border colors
                gridColor = sheetModel.getEffectiveGridColor(),
                // the HTML mark-up of all rendered cells without text contents
                leadingMarkup = '',
                // the HTML mark-up of all rendered cells with text contents
                trailingMarkup = '';

            // STEP 1: sort the array of all pending cells by cell address (bug 31123)
            sortedCells.sort(function (cellData1, cellData2) {
                return SheetUtils.compareCells(cellData1.address, cellData2.address);
            });

            // STEP 2: extact the specified number of cells from the cache of pending cells,
            // calculate DOM position of the cell, prepare a few CSS attribute value needed to
            // decide whether the cell has to be rendered at all (text content, fill, borders)
            _(sortedCells).any(function (cellData) {

                var // the address of the cell
                    address = cellData.address,
                    // the merged range that starts at the current cell
                    mergedRange = cellData.mergedRange,
                    // the absolute position and size of the cell in the sheet, in pixels
                    rectangle = mergedRange ? sheetModel.getRangeRectangle(mergedRange) : sheetModel.getCellRectangle(address),
                    // whether the cell has any visible text contents
                    hasContent = cellData.display.length > 0,
                    // CSS attribute value for fill color
                    fillColor = documentStyles.getCssColor(cellData.attributes.cell.fillColor, 'fill'),
                    // CSS styles for border lines
                    borderStyle = '',
                    // whether any border line is visible
                    hasBorder = false;

                // creates and adds CSS attributes for the specified border
                function processBorder(attrName, position, roundUp) {

                    var // the border attribute value
                        border = cellData.attributes.cell[attrName],
                        // the effective CSS attributes
                        cssAttrs = documentStyles.getCssBorderAttributes(border, { autoColor: gridColor }),
                        // the number of pixels the border will be moved out of the cell
                        offset = roundUp ? Math.ceil(cssAttrs.width / 2) : Math.floor(cssAttrs.width / 2);

                    borderStyle += position + ':-' + offset + 'px;';
                    if (cssAttrs.width > 0) {
                        hasBorder = true;
                        borderStyle += 'border-' + position + ':' + cssAttrs.style + ' ' + cssAttrs.width + 'px ' + cssAttrs.color + ';';
                    }
                }

                // remove the cell from the DOM and the cache of pending cells
                self.getCellNode(address).remove();
                delete pendingCells[SheetUtils.getCellKey(address)];
                maxCount -= 1;

                // silently skip cells without size (contained in hidden columns/rows)
                if ((rectangle.width > 0) && (rectangle.height > 0)) {

                    // add the pixel rectangle to the cell data
                    cellData.rectangle = rectangle;

                    // prepare additional formatting information for the cell
                    cellData.cssFillColor = (fillColor !== 'transparent') ? fillColor : null;
                    // offset to left/top cell border: round down (keep middle line for odd line widths on left/top pixel of the PREVIOUS cell)
                    processBorder('borderLeft', 'left', true);
                    processBorder('borderTop', 'top', true);
                    // offset to right/bottom cell border: round up (keep middle line for odd line widths on left/top pixel of the OWN cell)
                    processBorder('borderRight', 'right', false);
                    processBorder('borderBottom', 'bottom', false);
                    // insert border styles into cell data, if at least one border line is visible
                    cellData.cssBorderStyle = hasBorder ? borderStyle : null;

                    // silently skip cells without any visible contents (display value, border, fill)
                    if (hasContent || cellData.cssFillColor || cellData.cssBorderStyle) {

                        // add cell to the array of cells to be rendered in this iteration
                        allCells.push(cellData);
                        renderCount -= 1;

                        // additional processing for cells with text contents
                        if (hasContent) {
                            contentCells.push(cellData);
                            contentCount -= 1;
                        }
                    }
                }

                // repeat until the specified number of cells has been found
                return (maxCount === 0) || (renderCount === 0) || (contentCount === 0);
            });

            // STEP 3: create HTML mark-up of the clip node, content node, and text contents (this part
            // uses fontCollection.getTextWidht(), modifications of the DOM are not allowed here!)
            _(contentCells).each(function (cellData) {

                var // style attribute value for the cell clip node
                    clipStyle = '',
                    // style attribute value for the cell content node (parent of the text spans)
                    contentStyle = 'padding-left:' + cellPadding + 'px;padding-right:' + cellPadding + 'px;',
                    // style attribute value for the text span (for now, only one span for entire text)
                    spanStyle = '',

                    // attribute map for the 'cell' attribute family
                    cellAttributes = cellData.attributes.cell,
                    // attribute map for the 'character' attribute family
                    charAttributes = cellData.attributes.character,

                    // the available width for the cell text
                    availableWidth = Math.max(2, cellData.rectangle.width - totalPadding),
                    // whether to show negative numbers in red text color
                    redText = cellData.format.red && CellCollection.isNumber(cellData) && (cellData.result < 0),
                    // CSS text color, calculated from text color and fill color attributes
                    textColor = documentStyles.getCssTextColor(redText ? Color.RED : charAttributes.color, [cellAttributes.fillColor]),
                    // CSS text decoration, calculated from underline and strike attributes
                    textDecoration = PaneUtils.getCssTextDecoration(cellData),
                    // additional width for the left/right border of the clip node
                    clippingData = null,
                    // the mark-up for the final text contents
                    textMarkup = null;

                // generates the HTML mark-up of a single text span
                function getSpanMarkup(text) {
                    return '<span style="' + spanStyle + '">' + Utils.escapeHTML(text) + '</span>';
                }

                // cache character attibutes with effective font size and line height for current zoom factor in cell data
                charAttributes = cellData.charAttributes = _.clone(charAttributes);
                charAttributes.fontSize = view.getEffectiveFontSize(charAttributes.fontSize);
                charAttributes.lineHeight = view.getEffectiveLineHeight(cellData.attributes.character);

                // horizontal alignment will be set later (calculate optimal
                // width of cells with justified alignment before setting it)
                cellData.contentCss = { textAlign: PaneUtils.getCssTextAlignment(cellData) };

                // add vertical alignment
                switch (cellAttributes.alignVert) {
                case 'top':
                    contentStyle += 'top:0;';
                    break;
                case 'middle':
                    // alignment will be updated dynamically below (needs content node in DOM)
                    break;
                case 'justify':
                    // alignment will be updated dynamically below (needs content node in DOM)
                    contentStyle += 'top:0;';
                    break;
                default:
                    // 'bottom' alignment, and fall-back for unknown alignments
                    contentStyle += 'bottom:-1px;';
                }

                // create CSS formatting for character attributes
                spanStyle += 'font-family:' + documentStyles.getCssFontFamily(charAttributes.fontName).replace(/"/g, '\'') + ';';
                spanStyle += 'line-height:' + charAttributes.lineHeight + 'px;';
                if (charAttributes.fontSize !== 11) { spanStyle += 'font-size:' + charAttributes.fontSize + 'pt;'; }
                if (charAttributes.bold) { spanStyle += 'font-weight:bold;'; }
                if (charAttributes.italic) { spanStyle += 'font-style:italic;'; }
                if (textColor !== '#000000') { spanStyle += 'color:' + textColor + ';'; }
                if (textDecoration !== 'none') { spanStyle += 'text-decoration:' + textDecoration + ';'; }

                // create mark-up for the text contents
                cellData.wrappedText = CellCollection.isWrappedText(cellData);
                if (cellData.wrappedText) {
                    // automatic text wrapping: CSS's 'white-space:pre-wrap' does not work together with
                    // 'text-align:justify', need to process white-space dynamically and use 'white-space:normal' instead
                    contentStyle += 'white-space:normal;word-wrap:break-word;';
                    // create a span element for each text line
                    textMarkup = _(cellData.display.split(/\n/)).map(function (textLine) {
                        textLine = Utils.cleanString(textLine).replace(/\s/g, ' ').replace(/^ /, '\xa0').replace(/ {2}/g, ' \xa0').replace(/ $/, '\xa0');
                        return getSpanMarkup(textLine);
                    }).join('<br>');
                } else if (CellCollection.isText(cellData)) {
                    // text cells without wrapping: calculate available space for the cell text
                    // over preceding/following empty cells (but always crop merged ranges)
                    clippingData = calculateCellClipping(cellData);
                    clipStyle += 'left:' + (-clippingData.clipLeft) + 'px;right:' + (-clippingData.clipRight) + 'px;';
                    contentStyle += 'left:' + (-clippingData.hiddenLeft) + 'px;right:' + (-clippingData.hiddenRight) + 'px;';
                    // remove line breaks from text, get optimal width
                    textMarkup = cellData.display.replace(/\n/g, '');
                    cellData.contentWidth = fontCollection.getTextWidth(textMarkup, charAttributes);
                    // create a single span element for the final text
                    textMarkup = getSpanMarkup(textMarkup);
                } else {
                    // number cells with 'General' number format
                    if (CellCollection.isNumber(cellData) && _.isFinite(cellData.result) && (cellData.format.cat === 'standard') && (cellData.display !== '\u2026')) {
                        // get the pixel width of the number without any restriction (used for automatic sizing of columns)
                        var formatResult = numberFormatter.formatStandardNumber(cellData.result, SheetUtils.MAX_LENGTH_STANDARD_CELL, charAttributes);
                        textMarkup = formatResult.text;
                        cellData.contentWidth = formatResult.width;
                        // get a text representation that fits into the cell
                        if (availableWidth < cellData.contentWidth) {
                            // may become null, if no valid display string can be found for the available width
                            formatResult = numberFormatter.formatStandardNumber(cellData.result, SheetUtils.MAX_LENGTH_STANDARD_CELL, charAttributes, availableWidth);
                            textMarkup = formatResult ? formatResult.text : null;
                        }
                    } else {
                        // other numbers, Boolean values, and error codes: get width of display string
                        cellData.contentWidth = fontCollection.getTextWidth(cellData.display, charAttributes);
                        if (cellData.contentWidth <= availableWidth) {
                            textMarkup = cellData.display;
                        }
                    }
                    // set railroad track error for cells where display strng does not fit
                    if (!_.isString(textMarkup)) {
                        var hashCharWidth = fontCollection.getTextWidth('#', charAttributes);
                        textMarkup = Utils.repeatString('#', Math.max(1, Math.floor(availableWidth / hashCharWidth)));
                    }
                    // create a single span element for the final text
                    textMarkup = getSpanMarkup(textMarkup);
                }

                // create the clip node and the content node
                cellData.contentMarkup = '<div class="clip" style="' + clipStyle + '"><div class="content" style="' + contentStyle + '">' + textMarkup + '</div></div>';
            });

            // STEP 4: create HTML mark-up for all cells to be rendered
            _(allCells).each(function (cellData) {

                var // the logical address of the cell
                    address = cellData.address,
                    // the merged range the cell is part of
                    mergedRange = cellData.mergedRange,
                    // the location of the cell in the sheet, in pixels
                    rectangle = cellData.rectangle,
                    // style attribute value for the root node of the cell
                    cellStyle = '',
                    // the effective horizontal alignment
                    textAlign = PaneUtils.getCssTextAlignment(cellData),
                    // the HTML mark-up of the cell
                    cellMarkup = '';

                // convert cell position relative to layer node
                rectangle = _.clone(rectangle);
                rectangle.left -= layerRectangle.left;
                rectangle.top -= layerRectangle.top;

                // restrict width of oversized cells to browser limit, according to horizontal alignment
                if (rectangle.width > Utils.MAX_NODE_SIZE) {
                    var widthDiff = rectangle.width - Utils.MAX_NODE_SIZE,
                        reducedLeftInside = rectangle.left + widthDiff > 0,
                        reducedRightInside = rectangle.left + Utils.MAX_NODE_SIZE < layerRectangle.width;
                    switch (textAlign) {
                    case 'left':
                        // Reduce node width to keep cell contents left-aligned
                        // correctly. If the new right border will be located inside
                        // the layer rectangle after width reduction, adjust left border
                        // (which is far outside the layer rectangle in that case).
                        rectangle.left += reducedRightInside ? widthDiff : 0;
                        break;
                    case 'right':
                        // Adjust left offset of the node to keep cell contents
                        // right-aligned correctly, unless the new left border of
                        // the cell will be located inside the layer rectangle after
                        // width reduction (right border is far outside the layer
                        // rectangle in that case).
                        rectangle.left += reducedLeftInside ? 0 : widthDiff;
                        break;
                    default:
                        // Adjust left offset by half of the width correction to
                        // keep the centered text at its position, unless the left
                        // or right border of the cell will be located inside the
                        // layer rectangle after adjustment or width reduction.
                        rectangle.left += reducedRightInside ? widthDiff : reducedLeftInside ? 0 : Math.round(widthDiff / 2);
                    }
                    rectangle.width = Utils.MAX_NODE_SIZE;
                }

                // restrict height of oversized cells to browser limit, according to vertical alignment
                if (rectangle.height > Utils.MAX_NODE_SIZE) {
                    var heightDiff = rectangle.height - Utils.MAX_NODE_SIZE,
                        reducedTopInside = rectangle.top + heightDiff > 0,
                        reducedBottomInside = rectangle.top + Utils.MAX_NODE_SIZE < layerRectangle.height;
                    switch (cellData.attributes.cell.alignVert) {
                    case 'top':
                        // Reduce node height to keep cell contents top-aligned
                        // correctly. If the new bottom border will be located inside
                        // the layer rectangle after height reduction, adjust top border
                        // (which is far outside the layer rectangle in that case).
                        rectangle.top += reducedBottomInside ? heightDiff : 0;
                        break;
                    case 'bottom':
                        // Adjust top offset of the node to keep cell contents
                        // bottom-aligned correctly, unless the new top border of
                        // the cell will be located inside the layer rectangle after
                        // height reduction (bottom border is far outside the layer
                        // rectangle in that case).
                        rectangle.top += reducedTopInside ? 0 : heightDiff;
                        break;
                    default:
                        // Adjust top offset by half of the height correction to
                        // keep the centered text at its position, unless the top
                        // or bottom border of the cell will be located inside the
                        // layer rectangle after adjustment or height reduction.
                        rectangle.top += reducedBottomInside ? heightDiff : reducedTopInside ? 0 : Math.round(heightDiff / 2);
                    }
                    rectangle.height = Utils.MAX_NODE_SIZE;
                }

                // add effective position and size of the cell to the style attribute
                cellStyle += PaneUtils.getRectangleStyleMarkup(rectangle);

                // generate the HTML mark-up for the cell node
                cellMarkup += '<div class="cell" data-col="' + address[0] + '" data-row="' + address[1] + '" data-address="' + SheetUtils.getCellKey(address) + '"';
                if (mergedRange) {
                    if (mergedRange.start[0] < mergedRange.end[0]) { cellMarkup += ' data-col-span="' + SheetUtils.getColCount(mergedRange) + '"'; }
                    if (mergedRange.start[1] < mergedRange.end[1]) { cellMarkup += ' data-row-span="' + SheetUtils.getRowCount(mergedRange) + '"'; }
                    if (_.isNumber(cellData.visCol)) { cellMarkup += ' data-vis-col="' + cellData.visCol + '"'; }
                    if (_.isNumber(cellData.visRow)) { cellMarkup += ' data-vis-row="' + cellData.visRow + '"'; }
                }
                if (cellData.wrappedText && _.isString(cellData.contentMarkup)) { cellMarkup += ' data-wrapped="true"'; }
                cellMarkup += ' style="' + cellStyle + '">';

                // add background style node
                if (cellData.cssFillColor) {
                    cellMarkup += '<div class="fill" style="background-color:' + cellData.cssFillColor + ';border-color:' + cellData.cssFillColor + ';"></div>';
                }

                // add border node, if any border is visible
                if (cellData.cssBorderStyle) {
                    cellMarkup += '<div class="border" style="' + cellData.cssBorderStyle + '"></div>';
                }

                // add text contents created in the previous step
                if (_.isString(cellData.contentMarkup)) {
                    cellMarkup += cellData.contentMarkup;
                }

                // close the root cell node
                cellMarkup += '</div>';

                // append the cell mark-up to the correct mark-up list, according to existence of text contents
                if (_.isString(cellData.contentMarkup)) {
                    trailingMarkup += cellMarkup;
                } else {
                    leadingMarkup += cellMarkup;
                }
            });

            // STEP 5: insert the entire HTML mark-up into the cell layer node, extract DOM nodes
            // (this part modifies the DOM, reading node size properties is not allowed here!)
            cellLayerNode.prepend(leadingMarkup).append(trailingMarkup);
            _(allCells).each(function (cellData) { cellData.cellNode = self.getCellNode(cellData.address); });
            _(contentCells).each(function (cellData) { cellData.contentNode = cellData.cellNode.find('>.clip>.content'); });

            // STEP 6: post-processing: update vertical alignment, calculate optimal width of justified cells
            // (this part reads the size of DOM nodes, modifications of the DOM are not allowed here!)
            _(contentCells).each(function (cellData) {

                // update vertical alignment 'middle'
                function updateMiddleAlignment() {
                    cellData.contentCss.top = (cellData.rectangle.height - cellData.contentNode.height()) / 2;
                }

                // update vertical alignment 'justify'
                function updateJustifyAlignment() {

                    var // the total height of the content node (line height has been set to 100% above)
                        contentHeight = cellData.contentNode.height(),
                        // the effective font size of the cell, in pixels
                        fontSize = Utils.convertLength(cellData.charAttributes.fontSize, 'pt', 'px'),
                        // the effective line height, in pixels
                        normalLineHeight = cellData.charAttributes.lineHeight,
                        // the free space between text line border and characters, in pixels
                        normalLinePadding = (normalLineHeight - fontSize) / 2,
                        // the number of text lines
                        lineCount = Math.round(contentHeight / normalLineHeight),
                        // the free space between text line border and characters with target line height, in pixels
                        targetLinePadding = 0,
                        // the effective line height, in pixels
                        targetLineHeight = 0;

                    if (lineCount > 1) {
                        targetLinePadding = (cellData.rectangle.height - 2 * normalLinePadding - lineCount * fontSize) / (2 * lineCount - 2);
                        targetLineHeight = 2 * targetLinePadding + fontSize;
                    }
                    targetLineHeight = Math.max(targetLineHeight, normalLineHeight);
                    (cellData.spanCss || (cellData.spanCss = {})).lineHeight = Math.floor(targetLineHeight) + 'px';
                    cellData.contentCss.top = Math.round(normalLinePadding - targetLinePadding);
                    // store real (unexpanded) content height in 1/100 mm (used to calculate optimal row height)
                    cellData.optimalHeightHmm = documentStyles.getLineHeight(cellData.attributes.character) * lineCount;
                }

                // update vertical alignment dynamically
                switch (cellData.attributes.cell.alignVert) {
                case 'middle':
                    updateMiddleAlignment();
                    break;
                case 'justify':
                    if (cellData.wrappedText) { updateJustifyAlignment(); }
                    break;
                }

                // write the optimal width of wrapped text cells as data attribute into the cell
                if (cellData.wrappedText) {
                    cellData.contentWidth = _(cellData.contentNode[0].childNodes).reduce(function (memo, textSpan) {
                        return Math.max(memo, textSpan.offsetWidth);
                    }, 0);
                }
            });

            // STEP 7: insert CSS formatting created in the post-processing step
            // (this part modifies the DOM, reading node size properties is not allowed here!)
            _(contentCells).each(function (cellData) {

                // add content width (used to calculate optimal width of columns)
                if (_.isNumber(cellData.contentWidth) && (cellData.contentWidth > 0)) {
                    var optimalWidth = sheetModel.convertPixelToHmm(cellData.contentWidth + totalPadding + 0.5);
                    cellData.cellNode.attr('data-optimal-width', optimalWidth);
                }

                // add content height (used to calculate optimal height of rows)
                if (_.isNumber(cellData.optimalHeightHmm) && (cellData.optimalHeightHmm > 0)) {
                    cellData.cellNode.attr('data-optimal-height', cellData.optimalHeightHmm);
                }

                // additional CSS formatting attributes
                cellData.contentNode.css(cellData.contentCss);
                if (cellData.spanCss) {
                    cellData.contentNode.find('>span').css(cellData.spanCss);
                }
            });
        }

        /**
         * Renders all cells in the passed cell ranges according to the current
         * column, row, and cell collections. The cells will be stored in the
         * internal cache 'pendingCells' for being rendered in a background
         * loop.
         *
         * @param {Object|Array} ranges
         *  The logical address of a single cell range, or an array with cell
         *  range addresses.
         */
        function renderCellRanges(ranges) {

            var // all merged ranges covered by the rendered cells
                mergedRanges = null,
                // additional cells whose clip regions need to be updated
                updateClipCells = {},
                // all merged ranges not yet rendered, mapped by their reference address
                pendingMergedRanges = {},
                // cached column intervals of merged ranges per row (mapped by row index)
                mergedColIntervalsCache = {};

            // registers a cell for background rendering, and starts the background loop
            var registerCellForRendering = (function () {

                var // the timer representing the background loop
                    timer = null;

                // registers a new cell for debounced rendering
                function registerCell(cellData) {
                    var key = SheetUtils.getCellKey(cellData.address);
                    // map cells by their address key (in case cells are registered repeatedly)
                    pendingCells[key] = cellData;
                    // remove the cell from the list of existing cells whose clip region will be updated
                    delete updateClipCells[key];
                }

                // starts a background loop that renders a few cells at a time
                function renderCells() {
                    // check whether the background loop already runs
                    if (timer || _.isEmpty(pendingCells)) { return; }
                    // create a new background loop
                    timer = app.repeatDelayed(function () {
                        renderPendingCells(MAX_RENDER_CELL_COUNT);
                        return _.isEmpty(pendingCells) ? Utils.BREAK : undefined;
                    });
                    // clean up after all cells have been rendered
                    timer.always(function () { timer = null; });
                }

                // create and return the actual debounced method registerCellForRendering()
                return app.createDebouncedMethod(registerCell, renderCells);

            }()); // local scope of registerCellForRendering()

            // prepares the cell data of a single cell for rendering
            function prepareCellForRendering(cellData, mergedRange) {

                var // the cell address
                    address = cellData.address,
                    // the map key of the cell
                    mapKey = SheetUtils.getCellKey(address),
                    // column/row collection entry, needed for merged ranges
                    entryData = null;

                // remove existing outdated entry from the cache of pending cells
                delete pendingCells[mapKey];

                // store column intervals of merged ranges in cell data (from cache, or update cache)
                cellData.mergedColIntervals = mergedColIntervalsCache[address[1]];
                if (!cellData.mergedColIntervals) {
                    cellData.mergedColIntervals = mergedColIntervalsCache[address[1]] =
                        SheetUtils.getColIntervals(SheetUtils.getIntersectionRanges(mergedRanges, model.makeRowRange(address[1])));
                }

                // additional processing for merged ranges
                if (mergedRange) {
                    if (_.isEqual(mergedRange.start, address)) {
                        // this is the reference cell of a merged range
                        delete pendingMergedRanges[mapKey];
                    } else {
                        // this is a cell hidden by a merged range, do not render it
                        self.getCellNode(address).remove();
                        return;
                    }
                }

                // add the merged range address to the cell data
                cellData.mergedRange = mergedRange;
                // add information for merged ranges whose first column/row is hidden
                if (mergedRange) {
                    if ((entryData = colCollection.getNextVisibleEntry(address[0])) && (address[0] < entryData.index)) { cellData.visCol = entryData.index; }
                    if ((entryData = rowCollection.getNextVisibleEntry(address[1])) && (address[1] < entryData.index)) { cellData.visRow = entryData.index; }
                }

                // register cell for rendering in background task
                registerCellForRendering(cellData);
            }

            // finds an additional cell outside the passed ranges whose clip region needs to be updated
            function registerUpdateClipCell(cellData, forward) {

                var // the next available non-empty cell
                    nextCellData = findNearestContentCell(cellData.address, forward ? 'right' : 'left'),
                    // an adjacent merged column interval
                    nextMergedInterval = null,
                    // the unique key of the cell to be updated
                    key = null;

                // only text cells with overflowing text contents are of interest here
                if (!CellCollection.isOverflowText(nextCellData)) { return; }

                // do not update clip area of cells that will be rendered
                key = SheetUtils.getCellKey(nextCellData.address);
                if (key in pendingCells) { return; }

                // find the nearest preceding or following merged column interval
                nextMergedInterval = findNearestInterval(cellData.mergedColIntervals, cellData.address[0], forward);

                // do nothing, if a merged range starts before or at the next available content cell
                // (forward mode); or if a merged range ends after or at the next available content cell
                // (backward mode), because merged ranges always clip their contents by themselves
                if (nextMergedInterval && (forward ? (nextMergedInterval.first <= nextCellData.address[0]) : (nextCellData.address[0] <= nextMergedInterval.last))) {
                    return;
                }

                // an existing text cell that needs to be updated has been found, store it
                updateClipCells[key] = nextCellData;
            }

            // restrict to the layer range of this grid pane
            ranges = SheetUtils.getIntersectionRanges(ranges, layerRange);
            if (ranges.length === 0) { return; }
            ranges = SheetUtils.getUnifiedRanges(ranges);

            // find all merged ranges covered by the rendered cells
            mergedRanges = mergeCollection.getMergedRanges(ranges);

            // initialize the map of merged ranges not yet rendered
            _(mergedRanges).each(function (mergedRange) {
                pendingMergedRanges[SheetUtils.getCellKey(mergedRange.start)] = mergedRange;
            });

            // generate the HTML mark-up for all cells in the passed ranges
            cellCollection.iterateCellsInRanges(ranges, function (cellData, origRange) {

                var // the cell address
                    address = cellData.address,
                    // the merged range containing the cell
                    mergedRange = SheetUtils.findFirstRange(mergedRanges, address);

                // remove the old cell node from the DOM, register the cell for background rendering
                prepareCellForRendering(cellData, mergedRange);

                // If cell is located at the left or right border of a rendered range, find the
                // preceding/following overlapping text cells which need to be rendered again
                // (its clip region may change due to the new contents of the current cell).
                if (address[0] === origRange.start[0]) { registerUpdateClipCell(cellData, false); }
                if (address[0] === origRange.end[0]) { registerUpdateClipCell(cellData, true); }
            });

            // render remaining merged ranges (e.g. starting outside the changed ranges/visible area)
            _(pendingMergedRanges).each(function (mergedRange) {
                if (cellCollection.containsCell(mergedRange.start)) {
                    prepareCellForRendering(cellCollection.getCellEntry(mergedRange.start), mergedRange);
                }
            });

            // update the clip region of existing cells found while rendering the ranges
            _(updateClipCells).each(function (cellData) {
                var cellNode = self.getCellNode(cellData.address), // cell node may be missing (if covered by passed cell ranges)
                    clippingData = (cellNode.length === 0) ? null : calculateCellClipping(cellData);
                if (clippingData) {
                    cellNode.find('>.clip').css({ left: -clippingData.clipLeft, right: -clippingData.clipRight })
                        .find('>.content').css({ left: -clippingData.hiddenLeft, right: -clippingData.hiddenRight });
                }
            });

            // render the first 5 cells synchronously (looks better when
            // entering multiple single cells manually one-by-one)
            renderPendingCells(5);
        }

        /**
         * Updates the DOM cell layer in either horizontal or vertical
         * direction, after the visible interval of this grid pane has changed,
         * or an operation has inserted, deleted, or changed columns or rows.
         * Removes DOM cell nodes of cells not visible anymore, and returns the
         * addresses of the cell ranges to be rendered.
         *
         * @returns {Array}
         *  The addresses of all ranges to be rendered.
         */
        function updateCellLayer(columns, operation) {

            var // the column/row collection
                collection = columns ? colCollection : rowCollection,
                // the old column/row indexes
                oldIndexes = columns ? colIndexes : rowIndexes,
                // the old column/row interval
                oldInterval = (oldIndexes && (oldIndexes.length > 0)) ? { first: oldIndexes[0], last: _.last(oldIndexes) } : null,
                // the visible interval in the specified direction
                newInterval = columns ? colInterval : rowInterval,
                // the new column/row indexes
                newIndexes = collection.getVisibleEntryIndexes(newInterval),
                // the column/row interval of cells to be moved
                moveInterval = null,
                // all cell ranges to be rendered
                renderRanges = [],

                // the model method to get the column/row range from an interval
                makeRangeFunc = _.bind(columns ? model.makeColRange : model.makeRowRange, model),
                // the name of the column/row data attribute
                indexAttrName = columns ? 'data-col' : 'data-row',
                // the name of the additional column/row data attribute for merged ranges with leading hidden columns/rows
                visibleIndexAttrName = columns ? 'data-vis-col' : 'data-vis-row',
                // helper function to update the cell address data attribute of a cell node
                updateCellAddress = columns ?
                    function (cellNode, col) { cellNode.attr('data-address', SheetUtils.getCellKey([col, Utils.getElementAttributeAsInteger(cellNode, 'data-row')])); } :
                    function (cellNode, row) { cellNode.attr('data-address', SheetUtils.getCellKey([Utils.getElementAttributeAsInteger(cellNode, 'data-col'), row])); },
                // the array index for cell addresses
                addrIndex = columns ? 0 : 1,
                // the name of the cell data property containing the visible column/row index in partly hidden merged ranges
                visibleIndexPropName = columns ? 'visCol' : 'visRow',
                // the name of the CSS position attribute for cell nodes
                posAttrName = columns ? 'left' : 'top',
                // the leading position of the layer rectangle
                layerOffset = layerRectangle[posAttrName],

                 // whether a 'refresh' operation has been passed
                refreshAll = _.isObject(operation) && (operation.name === 'refresh'),
                // the name of the passed operation
                operationName = (!refreshAll && _.isObject(operation)) ? operation.name : null,
                // the number of columns/rows affected by the operation
                operationSize = _.isString(operationName) ? SheetUtils.getIntervalSize(operation) : null,

                // loop variables
                arrayIndex1 = 0, arrayIndex2 = 0, arraySize = 0;

            // removes all cell nodes in the specified columns/rows from the cell layer
            function removeCellNodes(indexes) {
                _(indexes).each(function (index) {

                    // remove cell nodes from the DOM
                    cellLayerNode.find('>[' + indexAttrName + '="' + index + '"]').remove();

                    // remove entries from the cache of pending cells
                    // (iterate entire cache, assuming that it is quite empty most of the time)
                    _(pendingCells).each(function (cellData, mapKey) {
                        if (cellData.address[addrIndex] === index) {
                            delete pendingCells[mapKey];
                        }
                    });
                });
            }

            // updates the address data attributes of all cells in a column/row
            function updateCellNodeAttributes(oldIndex, diff) {

                var // the target column/row index
                    newIndex = oldIndex + diff;

                // update data attributes of cells existing in the DOM
                cellLayerNode.find('>[' + indexAttrName + '="' + oldIndex + '"]')
                    .attr(indexAttrName, newIndex)
                    .each(function () { updateCellAddress($(this), newIndex); });
                cellLayerNode.find('>[' + visibleIndexAttrName + '="' + oldIndex + '"]')
                    .attr(visibleIndexAttrName, newIndex)
                    .each(function () { $(this).attr(indexAttrName, Utils.getElementAttributeAsInteger(this, indexAttrName) + diff); });

                // re-register pending cells for rendering at new position
                // (iterate entire cache, assuming that it is quite empty most of the time)
                _(pendingCells).each(function (cellData, mapKey) {
                    if ((cellData.address[addrIndex] === oldIndex) || (cellData[visibleIndexPropName] === oldIndex)) {
                        delete pendingCells[mapKey];
                        renderRanges.push({ start: cellData.address, end: cellData.address });
                    }
                });
            }

            // set the new column/row indexes
            if (columns) { colIndexes = newIndexes; } else { rowIndexes = newIndexes; }

            // handle 'insert columns/rows' operation
            if ((operationName === 'insert') && oldInterval && (operation.first <= oldInterval.last)) {

                // insert indexes for new columns/rows
                arrayIndex1 = _(oldIndexes).sortedIndex(operation.first);
                oldIndexes.splice.apply(oldIndexes, [arrayIndex1, 0].concat(collection.getVisibleEntryIndexes(operation)));

                // increase column/row indexes in old index array and cell nodes
                arrayIndex1 += operationSize;
                for (arrayIndex2 = oldIndexes.length - 1; arrayIndex2 >= arrayIndex1; arrayIndex2 -= 1) {
                    updateCellNodeAttributes(oldIndexes[arrayIndex2], operationSize);
                    oldIndexes[arrayIndex2] += operationSize;
                }

                // update the old interval
                oldInterval = { first: oldIndexes[0], last: _.last(oldIndexes) };
            }

            // handle 'delete columns/rows' operation
            if ((operationName === 'delete') && oldInterval && (operation.first <= oldInterval.last)) {

                // remove all old indexes and cell nodes in the operation interval
                arrayIndex1 = _(oldIndexes).sortedIndex(operation.first);
                arraySize = _(oldIndexes).sortedIndex(operation.last + 1) - arrayIndex1;
                removeCellNodes(oldIndexes.splice(arrayIndex1, arraySize));

                // decrease column/row indexes in old index array and cell nodes
                for (; arrayIndex1 < oldIndexes.length; arrayIndex1 += 1) {
                    updateCellNodeAttributes(oldIndexes[arrayIndex1], -operationSize);
                    oldIndexes[arrayIndex1] -= operationSize;
                }

                // update the old interval
                oldInterval = { first: oldIndexes[0], last: _.last(oldIndexes) };
            }

            // remove all old cell nodes not visible anymore
            if (oldIndexes) {
                removeCellNodes(_(oldIndexes).difference(newIndexes));
            }

            // move existing cell nodes in the DOM, if cells at the leading border have
            // been removed, or if a column/row operation has caused the change event
            if (oldInterval && !refreshAll) {
                if (oldInterval.first !== newInterval.first) {
                    moveInterval = oldInterval;
                } else if (_.isString(operationName)) {
                    moveInterval = SheetUtils.getIntersectionInterval({ first: operation.last + 1, last: columns ? model.getMaxCol() : model.getMaxRow() }, oldInterval);
                }
                if (moveInterval && (moveInterval = SheetUtils.getIntersectionInterval(moveInterval, newInterval))) {
                    collection.iterateEntries(moveInterval, function (entryData) {
                        var selector = '>[' + indexAttrName + '="' + entryData.index + '"],>[' + visibleIndexAttrName + '="' + entryData.index + '"]';
                        // move existing cell nodes in the DOM
                        cellLayerNode.find(selector).css(posAttrName, (entryData.offset - layerOffset) + 'px');
                    });
                }
            }

            // get the range addresses of the cells in all new columns/rows
            if (oldInterval && !refreshAll) {
                // new cells moved into the leading border of the bounding range
                if (newInterval.first < oldInterval.first) {
                    renderRanges.push(makeRangeFunc({ first: newInterval.first, last: Math.min(newInterval.last, oldInterval.first - 1) }));
                }
                // new cells moved into the trailing border of the bounding range
                if (oldInterval.last < newInterval.last) {
                    renderRanges.push(makeRangeFunc({ first: Math.max(oldInterval.last + 1, newInterval.first), last: newInterval.last }));
                }
                // operation applied, render all affected cells
                if (_.isString(operationName)) {
                    renderRanges.push(makeRangeFunc(operation));
                }
            } else {
                // render all cells
                renderRanges.push(makeRangeFunc(newInterval));
                cellLayerNode.empty();
                pendingCells = {};
            }

            // return addresses of all changed ranges for background rendering loop
            return renderRanges;
        }

        /**
         * Updates the layer range and the DOM layer nodes. Removes DOM cell
         * nodes of all cells not visible anymore, and renders new cell nodes
         * now visible in this grid pane.
         *
         * @param {Object|Null} interval
         *  The new column/row interval to be inserted into the layer range. If
         *  set to null, the grid pane will be hidden, and all layer nodes will
         *  be cleared.
         *
         * @param {Boolean} columns
         *  Whether the modified layer range was caused by a change in the
         *  column header pane (true) or row header pane (false).
         *
         * @param {Object} [operation]
         *  A pseudo operation that caused the changed layer range, as passed
         *  by the respective events of the header panes.
         */
        function updateLayerRange(interval, columns, operation) {

            var // addresses of all ranges to be rendered
                renderRanges = null;

            // store the passed column/row interval in the internal members
            if (columns) { colInterval = _.clone(interval); } else { rowInterval = _.clone(interval); }

            // the new layer range according to the current column and row interval
            layerRange = (colInterval && rowInterval) ? SheetUtils.makeRangeFromIntervals(colInterval, rowInterval) : null;

            // column or row header pane hidden: this grid pane is hidden too
            if (layerRange) {

                // ensure that all cells in the visible range exist in the cell collection
                cellCollection.requestCellRange(layerRange, { visible: true, merged: true });

                // calculate the new position of the layer nodes in the entire sheet
                layerRectangle = sheetModel.getRangeRectangle(layerRange);

                // update the scroll position (exact position of the layer root node in the DOM)
                updateScrollPosition();

                // update existing DOM cell nodes according to the operation
                renderRanges = updateCellLayer(false, columns ? null : operation);
                renderRanges = renderRanges.concat(updateCellLayer(true, columns ? operation : null));

                // update all DOM layer nodes
                renderCellGrid();
                renderCellRanges(renderRanges);
                renderCellSelection();
                renderHighlightedRanges();
                renderActiveSelection();
                renderCollaborativeSelections();

                // repaint all visible drawing frames
                updateDrawingFrames();

            } else {

                // safely destroy all image nodes to prevent memory problems on iPad
                app.destroyImageNodes(drawingLayerNode.find('img'));

                // clear all DOM layers
                layerRootNode.children('.grid-layer').empty();

                // reset all other variables related to the bounding range
                colIndexes = rowIndexes = layerRectangle = null;
                pendingCells = {};
            }
        }

        /**
         * Handles 'change:cells' events containing the range addresses of all
         * changed cells. Initiates debounced rendering of all affected cells.
         */
        var changeCellsHandler = (function () {

            var // the cell ranges to be rendered after collection updates
                changedRanges = [];

            // direct callback: stores the changed ranges in an internal array
            function registerRanges(event, ranges) {
                if (layerRange) { changedRanges = changedRanges.concat(ranges); }
            }

            // deferred callback: renders all changed cells, updates other layout settings of the grid pane
            function renderChangedRanges() {
                if (layerRange) { renderCellRanges(changedRanges); }
                changedRanges = [];
            }

            // create and return the actual debounced 'registerChangedRanges()' method
            return app.createDebouncedMethod(registerRanges, renderChangedRanges);
        }());

        /**
         * Updates the selection and highlight layers after specific sheet view
         * attributes have been changed.
         */
        function changeSheetViewAttributesHandler(event, attributes) {

            // render cell and drawing selection
            if ('selection' in attributes) {
                cellListMenu.hide();
                renderCellSelection();
                renderDrawingSelection();
            }

            // display collaborative user cursors
            if ('collaborativeSelections' in attributes) {
                renderCollaborativeSelections();
            }

            // update cell grid if visibility or grid color has changed
            if (Utils.hasProperty(attributes, /^(showGrid|gridColor)$/)) {
                renderCellGrid();
            }

            // render all cells if grid color has changed (border line color may change)
            if ('gridColor' in attributes) {
                renderCellRanges(layerRange);
            }

            // refresh focus display
            if (Utils.hasProperty(attributes, /^activePane/)) {
                cellListMenu.hide();
                updateFocusDisplay();
            }

            // changed auto-fill tracking data
            if ('autoFillData' in attributes) {
                cellListMenu.hide();
                renderCellSelection();
            }

            // changed highlight ranges
            if (Utils.hasProperty(attributes, /^highlight/)) {
                renderHighlightedRanges();
            }

            // changed active ranges (used while selecting ranges in cell edit mode)
            if ('activeSelection' in attributes) {
                renderActiveSelection();
            }
        }

        /**
         * Initializes this grid pane after the active sheet has been changed.
         */
        function changeActiveSheetHandler(event, activeSheet, activeSheetModel) {
            sheetModel = activeSheetModel;
            cellCollection = sheetModel.getCellCollection();
            colCollection = sheetModel.getColCollection();
            rowCollection = sheetModel.getRowCollection();
            mergeCollection = sheetModel.getMergeCollection();
            validationCollection = sheetModel.getValidationCollection();
            drawingCollection = sheetModel.getDrawingCollection();
        }

        /**
         * Prepares this grid pane for entering the in-place cell edit mode.
         */
        function cellEditEnterHandler() {
            cellListMenu.hide();
            // disable global F6 focus traveling into this grid pane while in-place cell edit mode is active in another grid pane
            rootNode.toggleClass('f6-target', self === view.getActiveGridPane());
            updateFocusDisplay();
        }

        /**
         * Prepares this grid pane for leaving the in-place cell edit mode.
         */
        function cellEditLeaveHandler() {
            rootNode.addClass('f6-target');
            updateFocusDisplay();
        }

        /**
         * Renders the cell grid, the selection, and the highlighted ranges,
         * after the cell geometry has been changed.
         */
        var changeCellLayoutHandler = app.createDebouncedMethod($.noop, function () {
            renderCellGrid();
            renderCellSelection();
            renderHighlightedRanges();
        });

        /**
         * Handles DOM 'scroll' events of the scrollable node.
         */
        function scrollHandler() {
            colHeaderPane.scrollTo(Math.round(scrollNode[0].scrollLeft * scrollLeftRatio));
            rowHeaderPane.scrollTo(Math.round(scrollNode[0].scrollTop * scrollTopRatio));
        }

        /**
         * Shows the pop-up menu for list validation.
         */
        var cellDropDownClickHandler = (function () {

            var // the server request currently running
                runningRequest = null;

            // return the actual 'cellDropDownClickHandler()' method
            return function cellDropDownClickHandler() {

                var // the current cell address (will be checked in server response)
                    address = view.getActiveCell(),
                    // the server request started by this method invokation
                    localRequest = null;

                // leave cell edit mode, or set focus back into clipboard node of this grid pane
                view.leaveCellEditMode('cell');
                self.grabFocus();

                // hide visible pop-up menu without any action
                if (cellListMenu.isVisible()) {
                    cellListMenu.hide();
                    return;
                }

                // cancel server request currently running, to prevent showing the menu
                // after the drop-down button has clicked again before the response arrives
                if (runningRequest) {
                    runningRequest.abort();
                    runningRequest = null;
                    return;
                }

                // receive source data from server
                localRequest = runningRequest = validationCollection.queryListValues(address);

                // fill list menu with values received from server
                localRequest.done(function (values) {

                    // check that this is still the last request (with slow connections,
                    // the response may arrive after selecting another cell with drop-down
                    // menu, and trying to open the dop-down menu of that cell)
                    if (runningRequest !== localRequest) { return; }

                    // check that the cell address of current cell is still the same
                    // (response may arrive after selecting another cell)
                    if (_.isEqual(address, view.getActiveCell())) {
                        cellListMenu.clearContents();
                        _(values).each(function (value) {
                            cellListMenu.createItemNode('', value, { label: _.noI18n(value) });
                        });
                        cellListMenu.show();
                    }
                });

                // reset the 'runningRequest' variable, if the last response is available
                localRequest.always(function () {
                    if (runningRequest === localRequest) {
                        runningRequest = null;
                    }
                });
            };
        }());

        /**
         * Inserts the value represented by the passed list item from the cell
         * pop-up menu into the cell.
         */
        function applyCellListValue(buttonNode) {

            var // the value of the selected list item
                itemValue = (buttonNode.length > 0) ? Utils.getControlValue(buttonNode) : null;

            // always hide the pop-up menu
            cellListMenu.hide();

            // insert the value into the cell
            if (_.isString(itemValue)) { view.setCellContents(itemValue); }
        }

        /**
         * Updates the layout of the cell pop-up menu. Hides the menu if the
         * anchor cell is not visible anymore, otherwise adjusts the minimum
         * width of the menu node according to the width of the cell.
         */
        function cellListBeforeLayoutHandler() {

            var buttonNode = selectionLayerNode.find('.cell-dropdown-button'),
                cellNode = buttonNode.parent();

            if ((cellNode.length === 0) || (buttonNode.length === 0)) { return; }

            // check the position of the cell node in the scrollable area
            if (!Utils.isChildNodeVisibleInNode(scrollNode, cellNode)) {
                cellListMenu.hide();
                return;
            }

            // update minimum width of the menu sections according to cell width (plus width of drop-down button)
            cellListMenu.getSectionNodes().css({ minWidth: buttonNode.offset().left + buttonNode.outerWidth() - cellNode.offset().left });
        }

        /**
         * Handles clicks in the cell pop-up menu.
         */
        function cellListClickHandler(event) {
            applyCellListValue($(event.currentTarget));
        }

        /**
         * Handles 'keydown' events from the root node.
         */
        function keyDownHandler(event) {

            var // the list item currently selected in the cell pop-up menu
                itemNode = null;

            // ESCAPE key (only one action per key event)
            if (event.keyCode === KeyCodes.ESCAPE) {

                // cancel current tracking, but do nothing else
                if ($.cancelTracking()) {
                    return false;
                }

                // drawing selection: back to cell selection
                if (view.hasDrawingSelection()) {
                    view.removeDrawingSelection();
                    return false;
                }

                // close cell pop-up menu
                if (cellListMenu.isVisible()) {
                    cellListMenu.hide();
                    return false;
                }

                // close search bar
                if (app.getController().getItemValue('view/searchpane')) {
                    app.getController().executeItem('view/searchpane', false);
                    return false;
                }

                // otherwise: let ESCAPE key bubble up
                return;
            }

            // show cell pop-up menu
            if (KeyCodes.matchKeyCode(event, 'DOWN_ARROW', { alt: true })) {
                if (!cellListMenu.isVisible()) {
                    self.scrollToCell(view.getActiveCell());
                    layerRootNode.find('.cell-dropdown-button').click();
                }
                return false;
            }

            // handle keyboard shortcuts for opened cell pop-up menu
            if (cellListMenu.isVisible()) {
                itemNode = cellListMenu.getSelectedItemNodes().first();

                // apply the selected value
                if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null }) || KeyCodes.matchKeyCode(event, 'ENTER', { shift: null })) {
                    applyCellListValue(itemNode);
                    return; // let event bubble up for selection change
                }

                // change selection
                itemNode = cellListMenu.getItemNodeForKeyEvent(event, itemNode, { selectEmpty: true });
                if (itemNode.length > 0) {
                    cellListMenu.selectItemNode(itemNode, { scroll: true });
                    return false;
                }
            }
        }

        /**
         * Sets the browser focus to the clipboard node, and selects all its
         * contents.
         */
        function grabClipboardFocus() {

            var // the browser selection
                selection = window.getSelection(),
                // a browser selection range object
                docRange = null;

            // calls the native focus() method at the clipboard node
            function grabNativeFocus() {
                clipboardFocusMethod.call(clipboardNode[0]);
            }

            // bug 29948: prevent grabbing to clipboard node while in cell in-place edit mode
            if (view.isCellEditMode()) { return; }

            // on touch devices, selecting text nodes in a content-editable node will
            // always show the virtual keyboard
            // TODO: need to find a way to provide clipboard support
            if (Modernizr.touch) {
                grabNativeFocus();
                return;
            }

            // Bug 26283: Browsers are picky how to correctly focus and select text
            // in a node that is content-editable: Chrome and IE will change the
            // scroll position of the editable node, if it has been focused
            // explicitly while not having an existing browser selection (or even
            // if the selection is not completely visible). Furthermore, the node
            // will be focused automatically when setting the browser selection.
            // Firefox, on the other hand, wants to have the focus already set
            // before the browser selection can be changed, otherwise it may throw
            // exceptions. Additionally, changing the browser selection does NOT
            // automatically set the focus into the editable node.
            if (_.browser.Firefox) {
                grabNativeFocus();
            }

            // Clear the old browser selection.
            // Bug 28515, bug 28711: IE fails to clear the selection (and to modify
            // it afterwards), if it currently points to a DOM node that is not
            // visible anymore (e.g. the 'Show/hide side panel' button). Workaround
            // is to move focus to an editable DOM node which will cause IE to update
            // the browser selection object. The target container node cannot be used
            // for that, see comments above for bug 26283. Using another focusable
            // node (e.g. the body element) is not sufficient either. Interestingly,
            // even using the (editable) clipboard node does not work here. Setting
            // the new browser selection below will move the browser focus back to
            // the application pane.
            BaseView.clearBrowserSelection();

            // set the browser selection
            try {
                docRange = window.document.createRange();
                docRange.setStart(clipboardNode[0], 0);
                docRange.setEnd(clipboardNode[0], clipboardNode[0].childNodes.length);
                selection.addRange(docRange);
            } catch (ex) {
                Utils.error('GridPane.grabClipboardFocus(): failed to select clipboard node: ' + ex);
            }
        }

        /**
         * Updates the grid pane after the document edit mode has changed.
         */
        function editModeHandler(event, editMode) {

            // nothing to do, if this grid pane is currently not visible
            if (!layerRange) { return; }

            // leave cell in-place edit mode, if document changes to read-only mode
            if (!editMode && app.isImportFinished()) {
                view.leaveCellEditMode();
                cellListMenu.hide();
            }

            // redraw selection according to edit mode
            renderCellSelection();
            renderDrawingSelection();
            renderHighlightedRanges();
        }

        /**
         * Handles 'cut' events of the clipboard node.
         * Invokes the copyHandler, then deletes the values and clears
         * the attributes of the selected cells.
         */
        function cutHandler(event) {

            var // the clipboard event data
                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData;

            // check for cell protection
            if (!view.requireEditableSelection()) {
                event.preventDefault();
                return false;
            }

            // invoke clipboard copy handler first
            copyHandler(event);

            if (clipboardData) {
                // prevent default cut handling for desktop browsers, but not for touch devices
                if (!Modernizr.touch) {
                    event.preventDefault();
                }

                // delete values and clear attributes of the selected cells
                view.fillCellRanges(null, undefined, { clear: true });

            } else {
                app.executeDelayed(function () {
                    // delete values and clear attributes of the selected cells
                    view.fillCellRanges(null, undefined, { clear: true });
                });
            }
        }

        /**
         * Handles 'copy' events of the clipboard node.
         * Creates a HTML table of the active range of the current selection
         * and generates the client clipboard id. Adds both to the system clipboard.
         * And invokes the Calcengine copy operation.
         */
        function copyHandler(event) {

            var // current selection in the active sheet
                selection = view.getSelection(),
                // the active cell range in the selection
                activeRange = selection.ranges[selection.activeRange],
                // the clipboard event data
                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData,
                // the client clipboard id to identify the copy data when pasting
                clientClipboardId = view.createClientClipboardId(),
                // the html table string to export
                htmlTable,
                // the plain text to export
                plainText;

            // restrict the number of cells to copy at the same time
            if (SheetUtils.getCellCount(activeRange) > SheetUtils.MAX_FILL_CELL_COUNT) {
                view.yell('info', gt('It is not possible to copy more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)));
                event.preventDefault();
                return false;
            }

            // generate HTML table and plain text representation from the active range of the current selection
            htmlTable = Clipboard.getHTMLStringFromRange(app, sheetModel, activeRange, clientClipboardId);
            plainText = Clipboard.getPlainTextFromRange(sheetModel, activeRange);

            // initiate clipboard server copy call
            view.clipboardServerCopy(selection);
            Utils.log('clipboard - copy - client id: ' + clientClipboardId);

            // set clipboard debug pane content
            view.setClipboardDebugInfo(htmlTable);

            // if browser supports clipboard api add data to the event
            if (clipboardData) {
                // add plain text and html of the current browser selection
                clipboardData.setData('text/plain', plainText);
                clipboardData.setData('text/html', htmlTable);

                // prevent default copy handling for desktop browsers, but not for touch devices
                if (!Modernizr.touch) {
                    event.preventDefault();
                }
            } else {
                // add Excel XML name spaces for custom tag support
                $('html').attr({
                    'xmlns': 'http://www.w3.org/TR/REC-html40',
                    'xmlns:o': 'urn:schemas-microsoft-com:office:office',
                    'xmlns:x': 'urn:schemas-microsoft-com:office:excel'
                });

                // add content and focus
                app.destroyImageNodes(clipboardNode);
                clipboardNode.empty().append(htmlTable);
                grabClipboardFocus();

                // remove XML name spaces again
                app.executeDelayed(function () {
                    $('html').removeAttr('xmlns xmlns:o xmlns:x');
                });
            }
        }

        /**
         * Handles 'paste' events of the clipboard node.
         */
        function pasteHandler(event) {
            var // current selection in the active sheet
                selection = view.getSelection(),
                // the clipboard event data
                clipboardData = event && event.originalEvent && event.originalEvent.clipboardData,
                // the cleaned up html data
                htmlData,
                // the html data attached to the event
                htmlRawData,
                // the plain text data attached to the event
                textData;


            function pasteHtmlClipboard(html) {
                var // the client clipboard id to identify the copy data when pasting
                    clientClipboardId = Clipboard.getClientClipboardId(html),
                    // the server clipboard id to identify the copy data when pasting
                    serverClipboardId = view.getServerClipboardId(clientClipboardId);

                if (serverClipboardId) {
                    // set clipboard debug pane content
                    view.setClipboardDebugInfo('Server side paste with clipboard id: ' + serverClipboardId);
                    // server paste call
                    view.clipboardServerPaste(selection, serverClipboardId);
                    Utils.log('clipboard - server side paste');
                } else {
                    // parse HTML
                    handleParserResult(Clipboard.parseHTMLData(html, documentStyles));
                    Utils.log('clipboard - client side html paste');
                }
            }

            function pasteTextClipboard(text) {
                // set clipboard debug pane content
                view.setClipboardDebugInfo(text);
                // parse text
                handleParserResult(Clipboard.parseTextData(text));
                Utils.log('clipboard - client side text paste');
            }

            function handleParserResult(result) {

                var // the target range of the paste
                    pasteRange = Clipboard.createRangeFromParserResult(selection.activeCell, result),
                    // the number of cells to paste
                    pasteCellCount = SheetUtils.getCellCount(pasteRange);

                // restrict the number of cells to paste at the same time
                if (pasteCellCount > SheetUtils.MAX_FILL_CELL_COUNT) {
                    view.yell('info', gt('It is not possible to paste more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)));
                    return;
                }

                // check for protected cells
                view.areRangesEditable(pasteRange).done(function () {

                    // create a single undo action for all affected operations
                    model.getUndoManager().enterUndoGroup(function () {

                        var // the ranges to be merged before pasting
                            mergeRanges = [];

                        // unmerge cells if necessary
                        if ((pasteCellCount > 1) && mergeCollection.rangeOverlapsMergedRange(pasteRange)) {
                            sheetModel.mergeRanges(pasteRange, 'unmerge');
                        }

                        // merge cells
                        _(result.mergeCollection).each(function (range) {
                            range = model.getCroppedMovedRange(range, selection.activeCell[0], selection.activeCell[1]);
                            if (range) { mergeRanges.push(range); }
                        });
                        if (mergeRanges.length > 0) {
                            sheetModel.mergeRanges(mergeRanges, 'merge');
                        }

                        // set cell contents
                        sheetModel.setCellContents(view.getActiveCell(), result.contents, { parse: true });
                    });
                })
                .fail(function () {
                    view.yell('info', gt('Pasting into protected cells is not allowed.'));
                });
            }

            /**
             * Cleans up script elements and event handlers to prevent Cross Site Scripting.
             * And removes position:fixed CSS style to avoid clipboard content being visible.
             */
            function cleanUpHtml(html) {
                var result;

                if (!html) { return null; }

                // remove event handler to prevent script injection and XSS
                result = html.replace(/<[^>]*\s(onload|onunload|onerror|onerrorupdate|onchange)=.*>/gi, '');

                // remove script elements
                result = $.parseHTML(result, null, false /*keepScripts*/);

                // remove position fixed style to avoid clipboard content being visible
                $(result).find('*').andSelf().css('position', '');

                return result;
            }

            htmlRawData = clipboardData && clipboardData.getData('text/html');
            htmlData = cleanUpHtml(htmlRawData);

            // check if clipboard event data contains html
            if (Clipboard.containsHtmlTable(htmlData)) {
                // prevent default paste handling for desktop browsers, but not for touch devices
                if (!Modernizr.touch) {
                    event.preventDefault();
                }

                // set clipboard debug pane content
                view.setClipboardDebugInfo(htmlRawData);
                // render the html to apply css styles provided by Excel or Calc
                app.destroyImageNodes(clipboardNode);
                clipboardNode.empty().append(htmlData);
                // parse and create operations
                pasteHtmlClipboard(clipboardNode);
                // clear clipboard
                clipboardNode.text('\xa0');

                return false;
            }

            // focus and select the clipboard node
            grabClipboardFocus();

            // check if clipboard event data contains plain text
            // to be used in case the clipboard div contains no html table to parse
            textData = (_.browser.IE) ? window.clipboardData && window.clipboardData.getData('text') : clipboardData && clipboardData.getData('text/plain');

            // read and parse pasted data
            app.executeDelayed(function () {
                if (Clipboard.containsHtmlTable(clipboardNode)) {
                    // set clipboard debug pane content
                    view.setClipboardDebugInfo(clipboardNode);
                    pasteHtmlClipboard(clipboardNode);
                } else if (textData) {
                    pasteTextClipboard(textData);
                }
                // clear clipboard
                clipboardNode.text('\xa0');
            });
        }

        /**
         * Registers an event handler at the specified instance that will only
         * be invoked when this grid pane is visible.
         */
        function registerEventHandler(source, type, handler) {
            source.on(type, function () {
                if (layerRange) {
                    handler.apply(self, _.toArray(arguments));
                }
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the root DOM node of this grid pane.
         *
         * @returns {jQuery}
         *  The root node of this grid pane, as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Returns whether the root node of this grid pane is currently focused
         * or contains a DOM node that is currently focused (e.g. the text area
         * node while edit mode is active).
         *
         * @returns {Boolean}
         *  Whether this grid pane is currently focused.
         */
        this.hasFocus = function () {
            return focusTargetNode.is(window.document.activeElement);
        };

        /**
         * Sets the browser focus into this grid pane.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.grabFocus = function () {
            focusTargetNode.focus();
            return this;
        };

        /**
         * Initializes the settings and layout of this grid pane.
         *
         * @param {Object} colSettings
         *  The view settings of the horizontal pane side (left or right).
         *  Supports the following properties:
         *  @param {Boolean} colSettings.visible
         *      Whether the column header pane is visible.
         *  @param {Number} colSettings.offset
         *      The absolute horizontal offset of the grid pane in the view
         *      root node, in pixels.
         *  @param {Number} colSettings.size
         *      The total outer width of the grid pane, in pixels.
         *  @param {Boolean} colSettings.frozen
         *      Whether the grid pane is frozen (not scrollable) horizontally.
         *  @param {Boolean} colSettings.showOppositeScroll
         *      Whether the vertical scroll bar will be visible or hidden
         *      outside the pane root node.
         *  @param {Number} colSettings.hiddenSize
         *      The total width of the hidden columns in front of the sheet in
         *      frozen view mode.
         *
         * @param {Object} rowSettings
         *  The view settings of the vertical pane side (top or bottom).
         *  Supports the following properties:
         *  @param {Boolean} rowSettings.visible
         *      Whether the row header pane is visible.
         *  @param {Number} rowSettings.offset
         *      The absolute vertical offset of the grid pane in the view root
         *      node, in pixels.
         *  @param {Number} rowSettings.size
         *      The total outer height of the grid pane, in pixels.
         *  @param {Boolean} rowSettings.frozen
         *      Whether the grid pane is frozen (not scrollable) vertically.
         *  @param {Boolean} rowSettings.showOppositeScroll
         *      Whether the horizontal scroll bar will be visible or hidden
         *      outside the pane root node.
         *  @param {Number} rowSettings.hiddenSize
         *      The total height of the hidden rows in front of the sheet in
         *      frozen view mode.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.initializePaneLayout = function (colSettings, rowSettings) {

            // initialize according to visibility
            if (colSettings.visible && rowSettings.visible) {

                // show pane root node and initialize auto-scrolling
                rootNode.show();
                layerRootNode.enableTracking(Utils.extendOptions(PaneUtils.DEFAULT_TRACKING_OPTIONS, {
                    autoScroll: colSettings.frozen ? (rowSettings.frozen ? false : 'vertical') : (rowSettings.frozen ? 'horizontal' : true),
                    borderNode: rootNode
                }));

                // position and size
                rootNode.css({ left: colSettings.offset, top: rowSettings.offset, width: colSettings.size, height: rowSettings.size });

                // initialize the scroll node containing the scroll bars
                scrollNode.css({
                    overflowX: colSettings.frozen ? 'hidden' : '',
                    overflowY: rowSettings.frozen ? 'hidden' : '',
                    bottom: (!colSettings.frozen && !rowSettings.showOppositeScroll) ? -Math.max(10, Utils.SCROLLBAR_HEIGHT) : 0,
                    right: (!rowSettings.frozen && !colSettings.showOppositeScroll) ? -Math.max(10, Utils.SCROLLBAR_WIDTH) : 0
                });

                // other layout options
                hiddenWidth = colSettings.hiddenSize;
                hiddenHeight = rowSettings.hiddenSize;

                // initialize the layer nodes
                layerRange = SheetUtils.makeRangeFromIntervals(colHeaderPane.getInterval(), rowHeaderPane.getInterval());
                layerRectangle = sheetModel.getRangeRectangle(layerRange);
                updateScrollAreaSize();
                updateScrollPosition();

                // update CSS classes at root node for focus display
                updateFocusDisplay();

                // create all drawing frames if not done yet (e.g. enabled split in current sheet)
                if (!drawingLayerNode[0].hasChildNodes()) {
                    createDrawingFrames();
                }

            } else {

                // hide pane root node and deinitialize auto-scrolling
                rootNode.hide();
                layerRootNode.disableTracking();
            }

            return this;
        };

        /**
         * Returns whether this grid pane is currently visible (depending on
         * its position, and the view split settings of the active sheet).
         *
         * @returns {Boolean}
         *  Whether this grid pane is currently visible.
         */
        this.isVisible = function () {
            return _.isObject(layerRange);
        };

        /**
         * Returns the address of the layer cell range.
         *
         * @returns {Object|Null}
         *  The address of the layer cell range, if this grid pane is visible;
         *  otherwise null.
         */
        this.getLayerRange = function () {
            return _.copy(layerRange, true);
        };

        /**
         * Returns the offset and size of the entire sheet rectangle covered by
         * this grid pane, including the ranges around the visible area.
         *
         * @returns {Object|Null}
         *  The offset and size of the sheet area covered by this grid pane, in
         *  pixels, if this grid pane is visible; otherwise null.
         */
        this.getLayerRectangle = function () {
            return _.copy(layerRectangle, true);
        };

        /**
         * Returns the effective offset and size of the visible sheet area in
         * this grid pane.
         *
         * @returns {Object}
         *  The offset and size of the sheet area visible in this grid pane, in
         *  pixels.
         */
        this.getVisibleRectangle = function () {
            return {
                left: colHeaderPane.getVisiblePosition().offset,
                top: rowHeaderPane.getVisiblePosition().offset,
                width: scrollNode[0].clientWidth,
                height: scrollNode[0].clientHeight
            };
        };

        /**
         * Returns the logical address of the top-left cell visible in this
         * grid pane. The cell is considered visible, it it's column and row
         * are at least half visible.
         *
         * @returns {Number[]}
         *  The logical address of the top-left cell in this grid pane.
         */
        this.getTopLeftAddress = function () {
            return [colHeaderPane.getFirstVisibleIndex(), rowHeaderPane.getFirstVisibleIndex()];
        };

        /**
         * Returns the DOM cell nodes matching the specified selector.
         *
         * @param {String} selector
         *  The jQuery selector used to filter the existing cell nodes.
         *
         * @returns {jQuery}
         *  All matching cell nodes, as jQuery collection.
         */
        this.findCellNodes = function (selector) {
            return cellLayerNode.find(selector);
        };

        /**
         * Returns the DOM cell node at the specified logical address.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @returns {jQuery}
         *  The DOM node of the cell, as jQuery object (empty, if the cell node
         *  does not exist).
         */
        this.getCellNode = function (address) {
            return cellLayerNode.find('>[data-address="' + SheetUtils.getCellKey(address) + '"]');
        };

        /**
         * Returns all DOM cell nodes in the specified column.
         *
         * @param {Number} col
         *  The zero-based column index.
         *
         * @returns {jQuery}
         *  The DOM nodes of all cells in the specified column, as jQuery
         *  collection.
         */
        this.getCellNodesInCol = function (col) {
            return cellLayerNode.find('>[data-col="' + col + '"]');
        };

        /**
         * Returns all DOM cell nodes in the specified row.
         *
         * @param {Number} row
         *  The zero-based row index.
         *
         * @returns {jQuery}
         *  The DOM nodes of all cells in the specified row, as jQuery
         *  collection.
         */
        this.getCellNodesInRow = function (row) {
            return cellLayerNode.find('>[data-row="' + row + '"]');
        };

        /**
         * Returns the DOM drawing frame node that represents the passed unique
         * drawing model identifier.
         *
         * @param {String} uid
         *  The unique identifier of the drawing model.
         *
         * @returns {jQuery}
         *  The DOM node of the drawing frame, as jQuery object (empty, if the
         *  drawing frame does not exist).
         */
        this.getDrawingFrame = function (uid) {
            return drawingLayerNode.find('>[data-model-uid="' + uid + '"]');
        };

        /**
         * Changes the scroll position of this grid pane relative to the
         * current scroll position.
         *
         * @param {Number} leftDiff
         *  The difference for the horizontal scroll position, in pixels.
         *
         * @param {Number} topDiff
         *  The difference for the vertical scroll position, in pixels.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollRelative = function (leftDiff, topDiff) {
            colHeaderPane.scrollRelative(leftDiff);
            rowHeaderPane.scrollRelative(topDiff);
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified cell visible.
         *
         * @param {Number[]} address
         *  The logical cell position to be made visible.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToCell = function (address) {
            var position = sheetModel.getCellRectangle(address, { expandMerged: true });
            colHeaderPane.scrollToPosition(position.left, position.width);
            rowHeaderPane.scrollToPosition(position.top, position.height);
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified DOM node visible.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM node to be made visible. If this object is a jQuery
         *  collection, uses the first node it contains. Must be a descendant
         *  node of the layer root node.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToNode = function (node) {

            var // the position of the node in the layer root node
                position = Utils.getChildNodePositionInNode(layerRootNode, node);

            colHeaderPane.scrollToPosition(position.left + layerRectangle.left, position.width);
            rowHeaderPane.scrollToPosition(position.top + layerRectangle.top, position.height);
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified drawing frame visible.
         *
         * @param {Number[]} position
         *  The logical document position of the drawing frame.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToDrawingFrame = function (position) {
            // TODO: scroll to embedded drawing frames
            return this.scrollToNode(drawingLayerNode[0].childNodes[position[0]]);
        };

        // in-place cell edit mode --------------------------------------------

        /**
         * Inserts the passed elements into the DOM of this grid pane in
         * preparation of the cell in-place edit mode.
         *
         * @param {jQuery} textArea
         *  The <textarea> element used to enter the text.
         *
         * @param {jQuery} textAreaUnderlay
         *  An additional helper element used to highlight specific portions of
         *  the edited text (used for formulas).
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.initializeCellEditMode = function (textArea, textAreaUnderlay) {
            rootNode.append(textAreaUnderlay, textArea);
            focusTargetNode = textArea;
            textArea.focus();
            clipboardNode.detach();
            return this;
        };

        /**
         * Cleans up after cell in-place edit mode has ended.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.cleanupCellEditMode = function () {
            rootNode.append(clipboardNode);
            focusTargetNode = clipboardNode;
            grabClipboardFocus();
            return this;
        };

        // initialization -----------------------------------------------------

        // marker for touch devices and browser types
        Utils.addDeviceMarkers(rootNode);

        // initialize class members
        app.on('docs:init', function () {

            // the spreadsheet model and view
            model = app.getModel();
            view = app.getView();

            // the style sheet container of the document
            documentStyles = model.getDocumentStyles();
            fontCollection = model.getFontCollection();
            numberFormatter = model.getNumberFormatter();

            // references to column and row header pane for this grid pane
            colHeaderPane = view.getColHeaderPane(panePos);
            rowHeaderPane = view.getRowHeaderPane(panePos);

            // drop-down menu for auto-completion and list validation of active cell
            cellListMenu = new ListMenu({ anchor: function () { return selectionLayerNode.find('.active-cell'); } });

            // the clipboard node that carries the browser focus for copy/paste events
            clipboardNode = view.createClipboardNode().attr('tabindex', 0).text('\xa0').appendTo(rootNode);
            clipboardFocusMethod = clipboardNode[0].focus;

            // by default, focus the clipboard node
            focusTargetNode = clipboardNode;

            // on touch devices, disable the content-editable mode of the clipboard node,
            // to prevent showing the virtual keyboard all the time
            if (Modernizr.touch) { clipboardNode.removeAttr('contenteditable'); }

            // listen to events from the view
            view.on({
                'change:activesheet': changeActiveSheetHandler,
                'celledit:enter': cellEditEnterHandler,
                'celledit:leave': cellEditLeaveHandler
            });

            // update DOM after interval changes of the column and row header pane
            colHeaderPane.on({
                'change:interval': function (event, interval, operation) { updateLayerRange(interval, true, operation); },
                'hide:interval': function () { updateLayerRange(null, true); }
            });
            rowHeaderPane.on({
                'change:interval': function (event, interval, operation) { updateLayerRange(interval, false, operation); },
                'hide:interval': function () { updateLayerRange(null, false); }
            });

            // update DOM after any cells have changed
            registerEventHandler(view, 'change:cells', changeCellsHandler);

            // update selection after the column, row, or cell geometry has changed
            registerEventHandler(view, 'insert:columns delete:columns change:columns insert:rows delete:rows change:rows insert:merged delete:merged', changeCellLayoutHandler);

            // update drawing layer node (only, if this grid pane is visible)
            registerEventHandler(view, 'insert:drawing', insertDrawingHandler);
            registerEventHandler(view, 'delete:drawing', deleteDrawingHandler);
            registerEventHandler(view, 'change:drawing', changeDrawingHandler);

            // update layers according to changed sheet view attributes (only, if this grid pane is visible)
            registerEventHandler(view, 'change:sheet:viewattributes', changeSheetViewAttributesHandler);

            // listen to DOM scroll events
            scrollNode.on('scroll', scrollHandler);

            // listen to changed scroll position/size from header panes
            registerEventHandler(colHeaderPane, 'change:scrollsize', updateScrollAreaSize);
            registerEventHandler(colHeaderPane, 'change:scrollpos', updateScrollPosition);
            registerEventHandler(rowHeaderPane, 'change:scrollsize', updateScrollAreaSize);
            registerEventHandler(rowHeaderPane, 'change:scrollpos', updateScrollPosition);

            // activate this pane on focus change
            rootNode.on('focusin', function () {
                // do not change active pane while in cell edit mode (range selection mode in formulas)
                if (!view.isCellEditMode()) {
                    sheetModel.setViewAttribute('activePane', panePos);
                }
            });

            // handle key events
            rootNode.on('keydown', keyDownHandler);

            // show drop-down menu for active cell
            selectionLayerNode.on('click', '.cell-dropdown-button', cellDropDownClickHandler);
            // suppress double-clicks for cell drop-down button
            selectionLayerNode.on('dblclick', '.cell-dropdown-button', false);

            // bug 31479: suppress double-clicks for drawings
            drawingLayerNode.on('dblclick', false);

            // cut, copy & paste events in clipboard node
            clipboardNode.on({
                cut: cutHandler,
                copy: copyHandler,
                paste: pasteHandler
            });

            // handle DOM focus requests at the clipboard node (select clipboard contents)
            clipboardNode[0].focus = grabClipboardFocus;

            // adjust minimum width of the cell pop-up menu
            cellListMenu.on('popup:beforelayout', cellListBeforeLayoutHandler);

            // handle click events in the cell pop-up menu
            cellListMenu.getNode().on('click', Utils.BUTTON_SELECTOR, cellListClickHandler);

            // hide pop-up menu when starting to select (including click on current cell with open menu)
            self.on('select:start', function () { cellListMenu.hide(); });

            // update grid pane when document edit mode changes
            model.on('change:editmode', editModeHandler);

            // hide pop-up menu when application is not active
            app.getWindow().on('beforehide', function () { cellListMenu.hide(); });
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {

            cellListMenu.destroy();
            clipboardNode[0].focus = clipboardFocusMethod;

            app.destroyImageNodes(drawingLayerNode);
            app.destroyImageNodes(clipboardNode);

            model = view = documentStyles = fontCollection = numberFormatter = null;
            sheetModel = cellCollection = colCollection = rowCollection = mergeCollection = validationCollection = drawingCollection = null;
            colHeaderPane = rowHeaderPane = null;

            rootNode = scrollNode = scrollSizeNode = layerRootNode = null;
            gridCanvasNode = cellLayerNode = collaborativeLayerNode = selectionLayerNode = drawingLayerNode = highlightLayerNode = activeRangeLayerNode = null;

            cellListMenu = clipboardNode = clipboardFocusMethod = null;

            colInterval = rowInterval = colIndexes = rowIndexes = pendingCells = null;
        });

    } // class GridPane

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: GridPane });

});
