/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/numberformatter',
    ['io.ox/core/date',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/io',
     'io.ox/office/tk/object/baseobject',
     'io.ox/office/spreadsheet/utils/sheetutils'
    ], function (CoreDate, Utils, IO, BaseObject, SheetUtils) {

    'use strict';

    var // the exponent character
        EXP = 'E',

        // zero characters, used for formatting numbers with leading zeros
        ZEROS = '0000000000000000';

    // class ParsedSection ====================================================

    /**
     * Contains the parsed information of a single section in a format code.
     * Sections will be separated with semicolons in the format code and can be
     * used to define different number formats for specific intervals of
     * numbers, e.g. for positive and negative numbers.
     *
     * @constructor
     */
    function ParsedSection() {

        /**
         * An array containing the format code tokens of this code section.
         */
        this.tokens = [];

        /**
         * The number of percent tokens contained in the code section.
         */
        this.percent = 0;

        /**
         * Whether the format code contains any date/time tokens.
         */
        this.dateTime = false;

        /**
         * Whether the format code contains an AM/PM token that switches the
         * time to 12-hours format.
         */
        this.isAmPm = false;

    } // class ParsedSection

    // methods ----------------------------------------------------------------

    /**
     * Parses the first section of the passed format code snippet, until
     * reaching the next semicolon character, or to the end of the string. All
     * parsed format tokens will be appended to the 'tokens' property of this
     * instance, and other information about the code section will be updated.
     *
     * @param {String} formatCode
     *  The format code snippet to be parsed.
     *
     * @returns {String}
     *  The remainder of the passed format code following the semicolon
     *  character that separates different code sections.
     */
    ParsedSection.prototype.parseSection = function (formatCode) {

        var // self reference
            self = this,
            // the matches of the regular expressions
            matches = null;

        // add a new token to the current section
        function pushToken(token) {
            self.tokens.push(token);
        }

        // add a new date/time token
        function pushDateTimeToken(type, length, total) {
            self.tokens.push({ type: type, length: length, total: !!total });
            self.dateTime = true;
        }

        // add a new string literal token, or extends the last existing token
        function pushLiteralToken(text) {
            var token = _.last(self.tokens);
            if (_.isObject(token) && (token.type === 'lit')) {
                token.text += text;
            } else {
                self.tokens.push({ type: 'lit', text: text });
            }
        }

        // stop if the entire format code has been consumed
        while (formatCode.length > 0) {

            // try/finally to be able to continue the while-loop at any place
            try {

                // 'General' format token
                if ((matches = /^GENERAL/i.exec(formatCode))) {
                    pushToken({ type: 'general' });
                    continue;
                }

                // date/time: AM/PM marker
                if ((matches = /^AM\/PM/i.exec(formatCode))) {
                    pushToken({ type: 'ampm' });
                    this.dateTime = this.isAmPm = true;
                    continue;
                }

                // date/time category: year
                if ((matches = /^Y+/i.exec(formatCode))) {
                    pushDateTimeToken('year', Math.min(4, Utils.roundUp(matches[0].length, 2))); // only YY and YYYY allowed
                    continue;
                }

                // date/time category: month
                if ((matches = /^M+/.exec(formatCode))) {
                    pushDateTimeToken('month', Math.min(5, matches[0].length)); // M to MMMMM allowed
                    continue;
                }

                // date/time category: day/weekday
                if ((matches = /^D+/i.exec(formatCode))) {
                    pushDateTimeToken('day', Math.min(4, matches[0].length)); // D to DDDD allowed
                    continue;
                }

                // date/time category: total number of hours
                if ((matches = /^\[(h+)\]/i.exec(formatCode))) {
                    pushDateTimeToken('hour', Math.min(2, matches[1].length), true); // [h] or [hh] allowed
                    continue;
                }

                // date/time category: total number of minutes
                if ((matches = /^\[(m+)\]/.exec(formatCode))) {
                    pushDateTimeToken('minute', Math.min(2, matches[1].length), true); // [m] or [mm] allowed
                    continue;
                }

                // date/time category: total number of seconds
                if ((matches = /^\[(s+)\]/i.exec(formatCode))) {
                    pushDateTimeToken('second', Math.min(2, matches[1].length), true); // [s] or [ss] allowed
                    continue;
                }

                // date/time category: hour
                if ((matches = /^h+/i.exec(formatCode))) {
                    pushDateTimeToken('hour', Math.min(2, matches[0].length)); // h or hh allowed
                    continue;
                }

                // date/time category: minute
                if ((matches = /^m+/.exec(formatCode))) {
                    pushDateTimeToken('minute', Math.min(2, matches[0].length)); // m or mm allowed
                    continue;
                }

                // date/time category: second
                if ((matches = /^s+/i.exec(formatCode))) {
                    pushDateTimeToken('second', Math.min(matches[0].length, 2)); // s or ss allowed
                    continue;
                }

                // percent category
                if ((matches = /^%/.exec(formatCode))) {
                    pushLiteralToken('%');
                    this.percent += 1;
                    continue;
                }

                // string literal, enclosed in double-quotes
                if ((matches = /^"([^"]*)"/.exec(formatCode))) {
                    pushLiteralToken(matches[1]);
                    continue;
                }

                // character literal, preceded by backspace
                if ((matches = /^\\(.)/.exec(formatCode))) {
                    pushLiteralToken(matches[1]);
                    continue;
                }

                // section separator: break while-loop (finally block will remove the semicolon)
                if ((matches = /^;/.exec(formatCode))) {
                    break;
                }

                // any non-matching token is considered a character literal
                matches = [formatCode[0]];
                pushLiteralToken(matches[0]);
            }

            // remove the current match from the formula string
            finally {
                formatCode = formatCode.slice(matches[0].length);
            }
        }

        // return the remaining format code
        return formatCode;
    };

    // class NumberFormatter ==================================================

    function NumberFormatter(app) {

        var // self reference
            self = this,

            // the font collection used to calculate text widths
            fontCollection = null,

            // cache for parsed code sections, mapped by format code
            parsedCodesCache = {},

            // all predefined number format codes, mapped by category
            categoryCodesDatabase = {},

            // all format codes for in-place edit mode
            editCodesDatabase = { date: 'DD/MM/YYYY', time: 'hh:mm:ss' },

            // null date (corresponding to cell value zero) TODO: use null date from document settings
            nullDate = new Date(1899, 11, 30),

            // whether to support negative date values (-1 = one day before null date)
            negativeDates = false;

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Parses the specified format code, and returns a descriptor with
         * arrays of code tokens, separated into several sections.
         *
         * @param {String} formatCode
         *  The format code to be parsed.
         *
         * @returns {Array|Null}
         *   An array representing the parsed code sections contained in the
         *   passed format code. Each array element is an instance of the class
         *   ParsedSection.
         */
        function parseFormatCode(formatCode) {

            // first check whether the code has been parsed already
            if (formatCode in parsedCodesCache) {
                return parsedCodesCache[formatCode];
            }

            var // all format code sections, as array
                parsedCode = parsedCodesCache[formatCode] = [];

            while (formatCode.length > 0) {
                parsedCode.push(new ParsedSection());
                formatCode = _.last(parsedCode).parseSection(formatCode);
            }

            // return the parsed format code
            return parsedCode;
        }

        /**
         * Returns the decimal string representation of the passed number.
         *
         * @constructor
         *
         * @extends BaseObject
         *
         * @param {Number} number
         *  The number whose decimal string representation will be returned.
         *
         * @param {Number} digits
         *  The number of digits after the decimal separator the passed number
         *  will be rounded to.
         *
         * @returns {Object}
         *  A result object with the following properties:
         *  - {Number} number
         *      The resulting rounded number.
         *  - {String} text
         *      The complete decimal string representation of the number.
         *  - {String} abs
         *      The decimal string representation of the absolute number.
         *  - {Number} digits
         *      The number of significant decimal places in the integral and
         *      fractional part in the string representation of the rounded
         *      number.
         */
        function formatAsDecimal(number, digits) {

            var // the absolute number, rounded to the specified digit count after decimal separator
                round = Utils.round(Math.abs(number), Math.pow(10, -digits)),
                // the integral part of the absolute number
                int = Math.floor(round),
                // the fractional part of the absolute number
                frac = Math.round((round - int) * Math.pow(10, digits)),
                // the text representation of the integral part
                intText = String(int),
                // the text representation of the significant fractional part (without leading zeros)
                fracText = String(frac),
                // the zeros between decimal separator and significant fractional part
                fillText = null,
                // the result returned by this method
                result = { abs: intText, digits: (int === 0) ? 0 : intText.length };

            // add the rounded number to the result
            result.number = (number < 0) ? -round : round;

            // no fraction available: do not add decimal separator
            if (frac > 0) {

                // leading zeros for the fractional part
                fillText = Utils.repeatString('0', digits - fracText.length);
                // remove trailing zeros
                fracText = fracText.replace(/0+$/, '');
                // concatenate integer and fractional part, remove trailing zeros
                result.abs += app.getDecimalSeparator() + fillText + fracText;
                // update number of significant digits
                if (int > 0) { result.digits += fillText.length; }
                result.digits += fracText.length;
            }

            // add final string representation (with sign)
            result.text = (number < 0) ? ('-' + result.abs) : result.abs;

            return result;
        }

        /**
         * Returns the scientific string representation of the passed number.
         *
         * @param {Number} mant
         *  The mantissa of the number whose scientific string representation
         *  will be returned. Must be in the half-open intervals +-[1, 10).
         *
         * @param {Number} exp
         *  The exponent of the number whose scientific string representation
         *  will be returned.
         *
         * @param {Number} digits
         *  The number of digits after the decimal separator the passed
         *  mantissa will be rounded to.
         *
         * @returns {Object}
         *  A result object with the following properties:
         *  - {Number} number
         *      The resulting rounded number.
         *  - {String} text
         *      The complete scientific string representation of the number.
         *  - {String} abs
         *      The scientific string representation of the absolute number.
         *  - {Number} digits
         *      The number of significant decimal places in the integral and
         *      fractional part in the string representation of the rounded
         *      mantissa.
         */
        function formatAsScientific(mant, exp, digits) {

            var // get the formatted mantissa
                result = formatAsDecimal(mant, digits);

            // if mantissa has been rounded up to 10, increase the exponent
            if (Math.abs(result.number) === 10) {
                result.number /= 10;
                result.abs = '1';
                exp += 1;
            }

            // update the rounded number in the result
            result.number *= Math.pow(10, exp);

            // add the exponent (at least two digits, but may be three digits)
            result.abs += EXP + ((exp < 0) ? '-' : '+');
            exp = Math.abs(exp);
            if (exp < 10) { result.abs += '0'; }
            result.abs += String(exp);

            // add final string representation (with sign)
            result.text = (mant < 0) ? ('-' + result.abs) : result.abs;

            return result;
        }

        /**
         * Formats a number with increasing number of digits following the
         * decimal separator, as long as the resulting number fits according to
         * the provided result validator.
         *
         * @param {Number} minDigits
         *  The minimum number of digits after the decimal separator.
         *
         * @param {Number} maxDigits
         *  The maximum number of digits after the decimal separator.
         *
         * @param {Function} formatCallback
         *  The callback function that receives the current number of digits
         *  following the decimal separator, and must return a result object
         *  containing the properties 'number', 'text', 'abs', and 'digits'.
         *  See methods formatAsDecimal() and formatAsScientific() above for a
         *  detailed description about this result object.
         *
         * @param {Function} validateCallback
         *  The callback function that checks the current result object it
         *  receives. Must return a Boolean value specifying whether the result
         *  object is valid.
         *
         * @returns {Object}
         *  The last matching result object returned by the callback function.
         */
        function findLongestFormat(minDigits, maxDigits, formatCallback, validateCallback) {

            var currDigits = minDigits,
                currResult = null,
                lastResult = null;

            do {
                currResult = formatCallback(currDigits);
                if (!validateCallback(currResult)) { break; }
                lastResult = currResult;
                currDigits += 1;
            } while (currDigits <= maxDigits);

            return lastResult;
        }

        // methods ------------------------------------------------------------

        /**
         * Returns all available number format codes for the specified format
         * category and current UI language.
         *
         * @param {String} category
         *  The identifier of the number format category.
         *
         * @returns {Array|Null}
         *  The format codes for the specified category, or null if no format
         *  codes have been defined for the category. Each element in the
         *  returned array is an object with the following properties:
         *  - {String} value
         *      The actual format code.
         *  - {String} label
         *      A caption label with a formatted example number, intended to be
         *      used in GUI form controls.
         *  - {Boolean} [red=false]
         *      If set to true, the format code contains a [RED] color modifier
         *      for negative numbers.
         */
        this.getCategoryCodes = function (category) {
            var categoryFormats = categoryCodesDatabase[category.toLocaleLowerCase()];
            return _.isArray(categoryFormats) ? _.copy(categoryFormats, true) : null;
        };

        /**
         * Returns the default number format code for the specified format
         * category and current UI language.
         *
         * @param {String} category
         *  The identifier of the number format category.
         *
         * @returns {String|Null}
         *  The default format code for the specified category, or null if no
         *  format code has been found.
         */
        this.getCategoryDefaultCode = function (category) {
            var categoryCodes = categoryCodesDatabase[category.toLocaleLowerCase()];
            return (_.isArray(categoryCodes) && _.isObject(categoryCodes[0])) ? categoryCodes[0].value : null;
        };

        /**
         * Returns the display string of the passed number formatted with the
         * 'General' number format, fitting into the specified text width.
         *
         * @param {Number} number
         *  The number whose string representation in 'General' number format
         *  will be returned.
         *
         * @param {Number} maxLength
         *  The maximum number of characters allowed for the absolute part of
         *  the passed number, including the decimal separator and the complete
         *  exponent in scientific notation.
         *
         * @param {Object} [attributes]
         *  Character formatting attributes influencing the text width. If
         *  omitted, the result object returned by this method will not contain
         *  the effective pixel width of the formatted number. MUST be
         *  specified, if the parameter 'maxWidth' (see below) is set.
         *  @param {String} attributes.fontName
         *      The name of the original font family (case-insensitive).
         *  @param {Number} attributes.fontSize
         *      The font size, in points.
         *  @param {Boolean} [attributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [attributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         *
         * @param {Number} [maxWidth]
         *  The maximum width of the resulting string representation, in
         *  pixels. If omitted, the number will be formatted without a pixel
         *  width restriction (but still with the restriction to a maximum
         *  number of text characters specified in the 'maxLength' parameter).
         *
         * @returns {Object|Null}
         *  A result object containing the formatted number and other data, in
         *  the following properties:
         *  - {String} text
         *      The resulting display text of the number, formatted with the
         *      'General' number format.
         *  - {Number} digits
         *      The number of significant decimal places in the integral and
         *      fractional part in the string representation of the rounded
         *      number.
         *  - {Number} [width]
         *      The width of the display text, in pixels. Will be omitted, if
         *      no character attributes have been passed (see parameter
         *      'attributes' above).
         *  If the passed maximum pixel width is too small to take the passed
         *  number in either decimal or scientific notation, the null value
         *  will be returned instead.
         */
        this.formatStandardNumber = function (number, maxLength, attributes, maxWidth) {

            // Bug 31125: This method uses a local cache containing the widths of all
            // characters that will appear in a number formatted with the 'General'
            // number format, and will add up these widths to calculate the total
            // width of a formatted number. This works twice as fast as simply using
            // the method FontCollection.getTextWidth() that inserts the text into a
            // DOM element and reads its resulting width. A very little drawback is
            // that the result might not be correct to the last pixel, as it misses
            // advanced font features such as kerning, but this should not be a
            // problem for numbers as hardly any font uses kerning to render digits.

            var // exponent (will be -INF for value zero)
                exp = Math.floor(Math.log(Math.abs(number)) / Math.LN10),
                // mantissa in the half-open intervals +-[1, 10) (will be NaN for zero)
                mant = number / Math.pow(10, exp),
                // whether to calculate text widths in pixels
                hasAttributes = _.isObject(attributes),
                // whether to restrict the result to a pixel width
                useMaxWidth = hasAttributes && _.isNumber(maxWidth),
                // cached widths of all characters that can appear in the display string
                charWidths = hasAttributes ? getCharacterWidths() : null,
                // the maximum pixel width of a digit
                digitWidth = useMaxWidth ? Math.max(1, fontCollection.getDigitWidth(attributes)) : null,
                // the pixel width of the decimal separator character
                decWidth = useMaxWidth ? charWidths[app.getDecimalSeparator()] : null,
                // the pixel width of the exponent with two digits
                exp2Width = useMaxWidth ? (charWidths[EXP] + charWidths['+'] + 2 * digitWidth) : null,
                // available, minimum, and maximum number of digits for the passed width
                availDigits = 0, minDigits = 0, maxDigits = 0,
                // the formatter callback function (decimal or scientific)
                formatterFunc = null,
                // the validator callback function (depending on pixel mode)
                validateFunc = null,
                // conversion result in decimal notation
                decResult = null,
                // conversion result in scientific notation
                scientResult = null,
                // the result to be used (either decimal or scientific)
                finalResult = null;

            // returns an object with the widths of all characters that can appear in a result
            function getCharacterWidths() {
                return fontCollection.getCustomFontMetrics(attributes, function () {
                    return _('0123456789+-' + EXP + app.getDecimalSeparator()).reduce(function (memo, char) {
                        memo[char] = fontCollection.getTextWidth(char, attributes);
                        return memo;
                    }, {});
                }, 'standard.widths');
            }

            // returns the width of the passed formatted number
            function getTextWidth(text) {
                var textWidth = _(text).reduce(function (memo, char) { return memo + charWidths[char]; }, 0);
                if (_.isNumber(textWidth)) { return textWidth; }
                Utils.warn('NumberFormatter.getTextWidth(): unexpected character in formatted number: ' + text);
                return fontCollection.getTextWidth(text, attributes);
            }

            // validates the passed result by character count only
            function validateLength(result) {
                return result.abs.length <= maxLength;
            }

            // validates the passed result by character count and pixel width
            function validateLengthAndWidth(result) {
                return (result.abs.length <= maxLength) && (getTextWidth(result.text) <= maxWidth);
            }

            // set the the validator callback function, depending on pixel mode
            validateFunc = useMaxWidth ? validateLengthAndWidth : validateLength;

            // find the best decimal representation
            if (exp < 2 - maxLength) {
                // start with zero representation for numbers equal to or very close to zero
                decResult = { number: number, abs: '0', text: (number < 0) ? '-0' : '0', digits: 0 };
                if (!validateFunc(decResult)) { decResult = null; }
            } else if (exp <= maxLength - 1) {
                // try to find the best explicit representation in a specific interval of exponents
                availDigits = useMaxWidth ? Math.floor((maxWidth - decWidth) / digitWidth) : maxLength;
                minDigits = Math.max(0, -(exp + 2));
                maxDigits = Math.min(15, Math.max(0, 14 - exp), maxLength - Math.max(2, exp), availDigits - Math.max(0, exp) - 1);
                formatterFunc = _.bind(formatAsDecimal, null, number);
                decResult = findLongestFormat(minDigits, maxDigits, formatterFunc, validateFunc);
            }

            // find the best scientific representation (exclude zero)
            if (_.isFinite(exp) && ((exp <= -4) || (exp >= 4))) {
                availDigits = maxLength - 5;
                if (useMaxWidth) { availDigits = Math.min(availDigits, Math.floor((maxWidth - decWidth - exp2Width) / digitWidth)); }
                minDigits = Utils.minMax(availDigits - 2, 0, 13);
                maxDigits = Utils.minMax(availDigits - 1, 0, 14);
                formatterFunc = _.bind(formatAsScientific, null, mant, exp);
                scientResult = findLongestFormat(minDigits, maxDigits, formatterFunc, validateFunc);
            }

            // prefer decimal notation, if it shows at least as much significant digits as the scientific notation
            finalResult = (decResult && (!scientResult || (decResult.digits >= scientResult.digits))) ? decResult : scientResult;
            return finalResult ? _({ text: finalResult.text, digits: finalResult.digits }).extend(hasAttributes ? { width: getTextWidth(finalResult.text) } : null) : null;
        };

        /**
         * Returns the passed cell value, formatted with the specified number
         * format.
         *
         * @param {Number|String|Boolean|Null} value
         *  The typed cell value to be formatted.
         *
         * @param {String} formatCode
         *  The format code used to format the passed cell value.
         *
         * @returns {String|Null}
         *  The formatted value; or null, if the passed format code is invalid,
         *  or if the value cannot be formatted with the format code (for
         *  example, a number is too large to be formatted as date/time).
         */
        this.formatValue = function (value, formatCode) {

            var // the parsed format code, either from cache or freshly parsed
                parsedCode = parseFormatCode(formatCode),
                // the section of the format code to be used for the passed value
                parsedSection = null,
                // the formatted string representation
                result = '',
                // the passed value, as Date object
                date = null;

            // if resulting text is shorter, fills with leading zeros
            function fillZeros(int, length) {
                return (ZEROS + int).substr(-Math.max(String(int).length, length));
            }

            // if resulting text is shorter, fills with leading zeros; if longer, truncates to length
            function fillZerosOrTruncate(int, length) {
                return (ZEROS + int).substr(-length);
            }

            // bail out if the passed format code is invalid
            if (!_.isArray(parsedCode) || (parsedCode.length === 0)) {
                return null;
            }

            // TODO: currently, only numbers are supported
            if (!_.isNumber(value) || !isFinite(value)) {
                return null;
            }

            // TODO: select parsed section according to value
            parsedSection = parsedCode[0];

            // correct value for percentage formats
            if (parsedSection.percent > 0) {
                value *= Math.pow(100, parsedSection.percent);
                if (!isFinite(value)) { return null; }
            }

            // additional preparations for date/time formats
            if (parsedSection.dateTime) {
                // error, if negative numbers as date are not supported
                if (!negativeDates && (value < 0)) { return null; }
                // get correct date/time for current null date
                date = new Date(nullDate.getTime() + Math.floor(value * 86400000));
                // error, if number is too large or too small to be formatted as date
                if (!isFinite(date.getTime()) || (date.getFullYear() < 0) || (date.getFullYear() > 9999)) { return null; }
            }

            // process all format code tokens
            _(parsedSection.tokens).each(function (token) {
                switch (token.type) {
                case 'lit':
                    result += token.text;
                    break;
                case 'year':
                    result += fillZerosOrTruncate(date.getFullYear(), token.length);
                    break;
                case 'month':
                    // TODO: month names
                    result += fillZeros(date.getMonth() + 1, token.length);
                    break;
                case 'day':
                    // TODO: weekday names
                    result += fillZeros(date.getDate(), token.length);
                    break;
                case 'hour':
                    result += fillZeros(token.total ? Math.floor(value * 24) : parsedSection.isAmPm ? (((date.getHours() + 11) % 12) + 1) : date.getHours(), token.length);
                    break;
                case 'minute':
                    result += fillZeros(token.total ? Math.floor(value * 1440) : date.getMinutes(), token.length);
                    break;
                case 'second':
                    result += fillZeros(token.total ? Math.round(value * 86400) : date.getSeconds(), token.length);
                    break;
                case 'ampm':
                    result += CoreDate.locale.dayPeriods[(date.getHours() < 12) ? 'am' : 'pm'];
                    break;
                default:
                    Utils.warn('NumberFormatter.formatValue(): unsupported token type "' + token.type + '"');
                }
            });

            return result;
        };

        /**
         * Returns the display string of the passed cell value, formatted with
         * the appropriate number format for the specified format code
         * category, intended for in-place cell edit mode.
         *
         * @param {Number|String|Boolean|Null} value
         *  The typed cell value to be formatted for in-place cell edit mode.
         *
         * @param {String} category
         *  The number format category the passed value will be formatted to.
         *
         * @returns {String}
         *  The resulting display text of the cell value, formatted with the
         *  appropriate number format for the specified category.
         */
        this.formatValueForEditMode = function (value, category) {

            // strings: use plain unformatted string for editing
            // error codes: use plain error code literal for editing
            if (_.isString(value)) {
                return (value[0] === '#') ? app.getLocalizedErrorCode(value) : value;
            }

            // Booleans: use plain Boolean literal for editing
            if (_.isBoolean(value)) {
                return app.getBooleanLiteral(value);
            }

            // numbers: use appropriate number representation according to number format category
            if (_.isNumber(value) && isFinite(value)) {

                var // the resulting formatted value
                    result = null;

                switch (category) {

                // percent: multiply by 100, add percent sign without whitespace
                case 'percent':
                    value *= 100; // may become infinite
                    result = (isFinite(value) ? this.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_EDIT).text : '') + '%';
                    break;

                // automatically show date and/or time, according to the number
                case 'date':
                case 'time':
                case 'datetime':

                    var // the integral part of the value (date)
                        date = Math.floor(value),
                        // the fractional part of the value (time), converted to seconds
                        time = Math.floor((value - date) * 86400),
                        // whether to add the date and time part to the result
                        showDate = (category !== 'time') || (date !== 0),
                        showTime = (category !== 'date') || (time !== 0),
                        // the resulting format code
                        formatCode = (showDate ? editCodesDatabase.date : '') + ((showDate && showTime) ? '\\ ' : '') + (showTime ? editCodesDatabase.time : '');

                    // the resulting formatted value (may be null for invalid dates)
                    result = this.formatValue(value, formatCode);
                    break;
                }

                // use 'General' number format for all other format codes
                return _.isString(result) ? result : this.formatStandardNumber(value, SheetUtils.MAX_LENGTH_STANDARD_EDIT).text;
            }

            // empty cells
            return '';
        };

        // initialization -----------------------------------------------------

        // initialize class members
        app.on('docs:init', function () {
            fontCollection = app.getModel().getFontCollection();
        });

        // get localized predefined format codes
        IO.loadResource('io.ox/office/spreadsheet/resource/numberformats').done(function (data) {
            if (self && !self.destroyed) {
                _(categoryCodesDatabase).extend(data.categories);
                _(editCodesDatabase).extend(data.edit);
            }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            fontCollection = categoryCodesDatabase = editCodesDatabase = null;
        });

    } // class NumberFormatter

    // exports ================================================================

    return NumberFormatter;

});
