/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/chartmodel',
        ['io.ox/office/tk/utils',
         'io.ox/office/editframework/model/format/color',
         'io.ox/office/drawinglayer/model/drawingmodel',
         'io.ox/office/spreadsheet/utils/sheetutils',
         'io.ox/office/spreadsheet/model/operations',
         'io.ox/office/spreadsheet/model/tokenarray',
         'io.ox/office/spreadsheet/model/drawing/drawingmodelmixin',
         'io.ox/office/spreadsheet/model/drawing/manualcolorchart',
         'io.ox/office/spreadsheet/model/drawing/axismodel',
         'io.ox/office/spreadsheet/model/drawing/dataseriesmodel',
         'io.ox/office/spreadsheet/model/drawing/legendmodel',
         'io.ox/office/spreadsheet/model/drawing/axistitlemodel',
         'io.ox/office/drawinglayer/view/chartstyleutil',
         'gettext!io.ox/office/spreadsheet'
        ], function (Utils, Color, DrawingModel, SheetUtils, Operations, TokenArray, DrawingModelMixin, ManualColor, AxisModel, DataSeriesModel, LegendModel, AxisTitleModel, ChartStyleUtil, gt) {

    'use strict';

    var TYPES = {
        'column standard' : 'column',
        'column clustered' : 'column',
        'column stacked' : 'stackedColumn',
        'column percentStacked' : 'stackedColumn100',
        'bar standard' : 'bar',
        'bar clustered' : 'bar',
        'bar stacked' : 'stackedBar',
        'bar percentStacked' : 'stackedBar100',
        'area standard' : 'area',
        'area stacked' : 'stackedArea',
        'area percentStacked' : 'stackedArea100',
        'line standard' : 'line',
        'line clustered' : 'line',
        'line stacked' : 'line',
        'line percentStacked' : 'spline',
        'line standard curved' : 'spline',
        'line clustered curved' : 'spline',
        'line stacked curved' : 'spline',
        'line percentStacked curved' : 'spline',
        'scatter standard' : 'line',
        'scatter standard curved' : 'spline',
        'bubble standard' : 'bubble',
        'pie standard' : 'pie',
        'donut standard' : 'doughnut'
    };

    var MARKER_TYPES = {
        'circle': 'circle',
        'dot': 'circle',
        'square': 'square',
        'triangle' : 'triangle',
        'x' : 'cross',
        'none': 'none'
    };
    var MARKER_LIST = ['circle', 'square', 'triangle', 'cross'];


    //'io.ox/office/spreadsheet/model/numberformatter'
    // null date (corresponding to cell value zero) TODO: use null date from document settings
    var nullDate = new Date(1899, 11, 30).getTime();

    // class SheetChartModel ==================================================

    /**
     * A class that implements a chart model to be inserted into a sheet of a
     * spreadsheet document.
     *
     * Additionally to the events triggered by the base class DrawingModel,
     * instances of this class trigger the following 'change:drawing' events:
     * - 'insert:series'
     *      After a new data series has been inserted into this chart. Event
     *      handlers receive the logical index of the new data series.
     * - 'delete:series'
     *      After a new data series has been deleted from this chart. Event
     *      handlers receive the last logical index of the data series.
     *
     * @constructor
     *
     * @extends DrawingModel
     * @extends DrawingModelMixin
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this chart model.
     *
     * @param {SheetModel} sheetModel
     *  The model instance of the sheet that contains this chart object.
     *
     * @param {Object} [initAttributes]
     *  Initial formatting attribute set for this chart model.
     */
    function SheetChartModel(app, sheetModel, initAttributes) {

        var self = this;

        var docModel = app.getModel();

        var controllerToken = null;

        var data = {
            culture:  'en',
//            zoomEnabled: true,
            bg: 'white',
            backgroundColor: 'transparent',
            axisX: {},
            axisY: {stripLines: []},
            creditHref: '',
            creditText: '',
            title: {
                text: ''
                    /*
                    ,
                text: 'invisible',
                verticalAlign: 'center',
                horizontalAlign: 'right',
                fontColor: 'transparent',
                fontSize: 10
                */
            },
            legend: {},
            data: []
        };


        data.axisY.stripLines[0] = {
            value: 0,
            color: 'black',
            label : ''
        };

        var indexLabelAttrs = {
            fontSize: 10,
            color: {type: 'auto'},
            fontName: 'Arial'
        };

        var colorHandler = new ManualColor(app, self, data);

        // base constructors --------------------------------------------------

        DrawingModel.call(this, app, 'chart', initAttributes, { additionalFamilies: ['chart', 'fill'] });
        DrawingModelMixin.call(this, app, sheetModel);


        this.getApp = function() {
            return app;
        };

        var axes = {
            x : new AxisModel(this, {}, data.axisX, new AxisTitleModel(this, {}, data.axisX), data.axisY.stripLines),
            y : new AxisModel(this, {}, data.axisY, new AxisTitleModel(this, {}, data.axisY), data.axisX.stripLines),
            z : new AxisModel(this, {})
        };

        var mainName = null;
        var mainTitleModel = new AxisTitleModel(this, {}, null);

        var legendModel = new LegendModel(this, {}, data.legend);


        // private methods ----------------------------------------------------

        function colRowChangeHandler(event, sheet, interval) {
            switch (event.type) {
            case 'insert:columns':
                checkDataInsideInterval(sheet, interval, true, true);
                break;
            case 'delete:columns':
                checkDataInsideInterval(sheet, interval, false, true);
                break;
            case 'insert:rows':
                checkDataInsideInterval(sheet, interval, true, false);
                break;
            case 'delete:rows':
                checkDataInsideInterval(sheet, interval, false, false);
                break;
            }
        }

        function checkDataInsideInterval(sheetModelIndex, interval, insert, columns) {
            var res = null;

            for (var i = 0; i < data.data.length; i++) {
                var dataSeries = data.data[i];
                var model = dataSeries.model;

                var change = model.checkRangeInsideInterval(sheetModelIndex, interval, insert, columns);
                if (change) {
                    dataSeries.needUpdate = true;
                    res = true;
                }
            }
            return res;
        }

        /**
         * Handles 'docs:update:cells' notifications. If this chart refers to
         * changed cells, the respective data series will be updated.
         */
        function updateNotificationHandler(changedData, allChangedRanges) {

            var // whether any data series overlaps with the changed ranges
                changed = false;

            // check passed ranges if they overlap with the source ranges of the data series
            _(data.data).each(function (dataSeries) {
                if (dataSeries.needUpdate) {
                    changed = true;
                } else if (dataSeries.model.rangesOverlap(allChangedRanges)) {
                    changed = true;
                    dataSeries.needUpdate = true;
                }
            });

            if (changed) { self.refresh(); }
        }


        function changeAttributes(evt, mergedAttributes, oldMergedAttributes) {
            var mc = mergedAttributes.chart;
            var oc = oldMergedAttributes.chart;
            var mf = mergedAttributes.fill;
            var of = oldMergedAttributes.fill;

            if (!_.isEqual(mc, oc) || !_.isEqual(mf, of)) {

                updateChartType();

                changeDrawing(evt, 'series');
            }
        }

        function isXYType() {
            var attributes = self.getMergedAttributes();
            return attributes.chart.type.indexOf('bubble') === 0 || attributes.chart.type.indexOf('scatter') === 0;
        }


        function changeDrawing(evt, subtype) {
            var lAttrs = legendModel.getMergedAttributes();
            var attributes = self.getMergedAttributes();

            var xyChart = isXYType();
            var cjst = getCJSType();

            var biggestY = 0;
            var smallestY = 9999999999999;
            //collect data to keep to small values visible

            _(data.data).each(function (dataSeries, index) {
                if (subtype === 'series' || !index) {
                    colorHandler.handleColor(dataSeries.model.getMergedAttributes(), index);
                }
                if (cjst === 'pie' || cjst === 'doughnut') {
                    dataSeries.startAngle = attributes.chart.rotation - 90;
                }
                if (lAttrs.legend.pos === 'off') {
                    dataSeries.showInLegend = false;
                } else {
                    dataSeries.showInLegend = true;
                    if (cjst === 'pie' || cjst === 'doughnut') {
                        if (index > 0) {
                            //we only see the first series
                            dataSeries.showInLegend = false;
                        } else if (dataSeries.dataPoints.length > 50) {
                            //canvasjs has rendering problems with to big legend-sizes
                            dataSeries.showInLegend = false;
                        }
                    }
                }

                ChartStyleUtil.handleCharacterProps(self, indexLabelAttrs, dataSeries, 'indexLabel');

                if (attributes.chart.dataLabel) {
                    dataSeries.indexLabel = '{name}';
                    dataSeries.indexLabelLineColor = null;
                } else {
                    dataSeries.indexLabel = ' ';
                    dataSeries.indexLabelLineColor = 'transparent';
                }

                var unclean2d = false;


                if (cjst === 'line' || cjst === 'spline') {
                    dataSeries.legendMarkerType = dataSeries.marker;
                } else {
                    dataSeries.legendMarkerType = 'square';
                }

                _.each(dataSeries.dataPoints, function (dataPoint, n) {
                    if (cjst === 'line' || cjst === 'spline') {
                        dataPoint.markerType = dataSeries.marker;
                    } else {
                        dataPoint.markerType = null;
                    }

                    //bubble & scatter are 2d-coordinate oriented, they want real numbers,b ut we dont always have real numbers!
                    if (xyChart) {
                        dataPoint.x = dataPoint.realX;
                        if (dataPoint.x === undefined || dataPoint.x === null || (!_.isDate(dataPoint.x) && isNaN(dataPoint.x))) {
                            unclean2d = true;
                        }
                    } else {
                        dataPoint.x = n + 1;
                        biggestY = Math.max(Math.abs(dataPoint.realY), biggestY);
                        smallestY = Math.min(Math.abs(dataPoint.realY), smallestY);
                    }
                });

                if (unclean2d) {
                    _.each(dataSeries.dataPoints, function (dataPoint, n) {
                        dataPoint.x = n + 1;
                    });
                }
            });

            var thresholdY = 0.005;
            //workaround that too small values stay visible
            if (!xyChart && (smallestY / biggestY) < thresholdY) {
                _(data.data).each(function (dataSeries) {
                    _.each(dataSeries.dataPoints, function (dataPoint) {
                        var rel = dataPoint.realY / biggestY;
                        if (Math.abs(rel)<thresholdY) {
                            if (rel<0) {
                                dataPoint.y = -thresholdY * biggestY;
                            } else {
                                dataPoint.y = thresholdY * biggestY;
                            }
                        }
                    });
                });
            }

            var mnt = mainTitleModel.getMergedAttributes();
            data.title.text = mnt.text.link[0] || '';
            if (!data.title.text && data.data.length === 1) {
                data.title.text = mainName || '';
            }
            if (data.title.text) {
                ChartStyleUtil.handleCharacterProps(self, mnt.character, data.title);
            }

            legendModel.refreshInfo();
            refreshAxis();
        }

        function getCJSType() {
            var type = self.getChartType();
            var res = TYPES[type];

            if (!res) {
                Utils.warn('no correct charttype found', type);
                res = 'column';
            }

            if (res === 'line' && self.getMergedAttributes().chart.curved === true) {
                res = 'spline';
            }
            return res;
        }

        function updateChartType() {
            var cjs = getCJSType();
            for (var i = 0; i < data.data.length; i++) {
                var series = data.data[i];
                series.type = cjs;
            }
            self.trigger('change:drawing');
        }


        function optCellString(cell, def) {
            if (cell && cell.display) {
                var trim = cell.display.trim();
                if (trim.length) {
                    return trim;
                }
            }
            return def;
        }

        /**
         * checks for assigned numberformat,
         * if there is only a code it looks in the clipboard-api for the code's numberformat
         *
         * if there is a 'm' in the numberformat, the code returns true because minutes or months are in Dates
         */
        function isDate(values) {
            if (!values || !values.length) {
                return;
            }
            return ChartStyleUtil.isDate(values[0]);
        }

        // public methods -----------------------------------------------------

        /**
         * is called by the DrawingController for a selected chart
         */
        this.getChartType = function () {
            var attributes = self.getMergedAttributes();

            var type = attributes.chart.type;
            type = type.replace('2d', '').replace('3d', '');

            var stacking = attributes.chart.stacking;
            if (stacking.length) {
                type = type + ' ' + stacking;
            }
            var curved = attributes.chart.curved;
            if (curved && (type === 'line' || type === 'scatter')) {
                type = type + ' curved';
            }
            return type;
        };

        this.getChartTypeForGui = function() {
            var chartType = this.getChartType();
            return chartType.replace('clustered', 'standard');
        };

        /**
         * this is only called by the Drawingframe to initialize the REAL CanvasJS-chart
         */
        this.getModelData = function () {
            return data;
        };

        /**
         * @param operation
         */
        this.deleteDataSeries = function (operation) {
            var removed = data.data.splice(operation.series, 1);
            removed[0].model.destroy();
            this.trigger('change:drawing');
            return true;
        };

        /**
         * refresh checks if something is changed in the series,
         * if true it calls the 'updateview'-request to the server
         */
        this.refresh = app.createDebouncedMethod($.noop, function () {

            var // all ranges to be querried from server, as array of arrays of range addresses
                sourceRanges = null,
                // maps series indexes and source link types to array elements in 'sourceRanges'
                collectInfo = null;

            // debounced method: chart may have been deleted in the meantime
            if (!self || self.destroyed) { return; }

            _(data.data).each(function (seriesData, series) {
                if (!seriesData.needUpdate) { return; }

                if (!sourceRanges) {
                    sourceRanges = [];
                    collectInfo = [];
                }

                var info = { series: series, indexes: {} };


                seriesData.model.iterateTokens(function (token, name) {
                    var entryRanges = token.resolveRangeList();
                    if (_.isArray(entryRanges)) {
                        info.indexes[name] = sourceRanges.length;
                        sourceRanges.push(entryRanges);
                    }
                });
                collectInfo.push(info);

                seriesData.needUpdate = false;
            });

            if (!_.isArray(sourceRanges)) {
                changeDrawing();
                return;
            }

            if (sourceRanges.length > 0) {

                var // query cell contents for the data series
                    request = app.queryCellResults(sourceRanges, {attributes: true});

                request.done(function (allContents) {

                    // asynchronous server request: chart may have been deleted in the meantime
                    if (!self || self.destroyed) { return; }

                    _(collectInfo).each(function (info, seriesIndex) {

                        var seriesIndex = info.series,
                            dataSeries = data.data[seriesIndex];

                        // returns the cell contents for the specified source link type
                        function getCellContents(key) {
                            return _.isNumber(info.indexes[key]) ? allContents[info.indexes[key]] : [];
                        }

                        // series values must exist
                        var valuesSource = getCellContents('values');
                        if (valuesSource.length === 0) { return; }

                        // all other source data may be missing
                        var titleSource = getCellContents('title');
                        var namesSource = getCellContents('names');
                        var bubblesSource = getCellContents('bubbles');


                        var nameIsDate = isDate(namesSource);

                        // build series title
                        if (titleSource.length > 0) {
                            // TODO: join title originating from multiple cells
                            dataSeries.name = optCellString(titleSource[0]);
                            if (!seriesIndex) {
                                mainName = dataSeries.name;
                            }
                        } else {
                            if (!seriesIndex) {
                                mainName = null;
                            }
                        }
                        if (!dataSeries.name) {
                            dataSeries.name =
                            //#. A data series in a chart object (a sequence of multiple data points with a title)
                            //#. %1$d is the numeric index of the series (e.g. "Series 1", "Series 2", etc.)
                            //#. This label should match the default title for data series in other spreadsheet
                            //#. applications (OpenOffice/LibreOffice Calc, Microsoft Excel).
                            //#, c-format
                            gt('Series %1$d', _.noI18n(seriesIndex + 1));
                        }

                        dataSeries.dataPoints = [];
                        _.each(valuesSource, function (valueCell, n) {
                            var dataPoint = {};
                            dataSeries.dataPoints.push(dataPoint);

                            // add value of data point (Y value)

                            dataPoint.y = isNaN(valueCell.result) ? 0 : valueCell.result;

                            // add title of data point (category name), add X value of data point (in scatter charts)
                            var nameCell = namesSource[n];
                            dataPoint.label = optCellString(nameCell, String(n + 1));
                            if (dataPoint.label.length > 23) {
                                dataPoint.label = dataPoint.label.substring(0, 20) + '...';
                            }
                            dataPoint.legendText = dataPoint.label;

                            var display = optCellString(valueCell, String(n + 1));
                            dataPoint.name = display;
                            if (dataPoint.name && dataPoint.name.length > 23) {
                                dataPoint.name = dataPoint.name.substring(0, 20) + '...';
                            }


                            dataPoint.toolTipContent = '{label}: {name}';


                            dataPoint.realX = (nameCell && !isNaN(nameCell.result)) ? nameCell.result : null;
                            if (nameIsDate && !isNaN(dataPoint.realX)) {
                                dataPoint.realX = new Date(nullDate + Math.floor(dataPoint.realX * 86400000));
                            }

                            if (dataPoint.y === undefined || dataPoint.y === null || isNaN(dataPoint.y)) {
                                dataPoint.y = 0;
                            }

                            dataPoint.realY = dataPoint.y;

                            // add bubble sizes (bubble charts only)
                            var sizeCell = bubblesSource[n];
                            if (sizeCell) {
                                dataPoint.z = sizeCell.result;
                            } else {
                                dataPoint.z = 1;
                            }
                        });
                    });
                    refreshAxis();
                    self.trigger('change:drawing', 'series');
                });
            } else {
                _(data.data).each(function (seriesData) {
                    seriesData.needUpdate = false;
                });
                self.trigger('change:drawing', 'series');
            }
        }, { delay: 50, maxDelay: 250 });

        /**
         * saves the Operation-Series in the model, but fetches the correct Data only if its visible
         * via trigger the Events 'change:drawing' and 'insert:series' and mark the current series that ists changed
         * @param operation
         *  @param {Object} [options.attrs]
         *   @param {Object} [options.attrs.series]
         *    the content of the series could be a formula {String}, or explicit Values {Array}.
         *    the values can be taken directy. the formula calls a request to server, the next time the chart visible
         *    @param {String|Array} [options.attrs.series.title]
         *    @param {String|Array} [options.attrs.series.names]
         *    @param {String|Array} [options.attrs.series.bubbles]
         *    @param {String|Array} [options.attrs.series.values]
         *
         * @returns {Boolean}
         */
        this.insertDataSeries = function (operation) {
            var dataSeries = {
                type: getCJSType()
            };
            data.data.splice(operation.series, 0, dataSeries);

            var series = operation.attrs.series;

            dataSeries.marker = MARKER_TYPES[series.marker];
            if (!dataSeries.marker) {
                var index = operation.series % MARKER_LIST.length;
                dataSeries.marker = MARKER_LIST[index];
            }

            var model = new DataSeriesModel(app, sheetModel, operation.attrs);

            dataSeries.model = model;

            dataSeries.needUpdate = true;

            dataSeries.dataPoints = [];
            dataSeries.name = '';

            this.trigger('change:drawing');

            if (controllerToken) {
                updateTokenArrays();
            }

            return true;
        };

        this.setAxisAttributes = function (operation) {
            var axisModel = this.getAxisModel(operation.axis);

            axisModel.setAttributes(operation.attrs);
            self.trigger('change:drawing');
            return true;
        };

        this.setGridlineAttributes = function (operation) {
            var gridModel = this.getAxisModel(operation.axis).getGrid();
            gridModel.setAttributes(operation.attrs);
            self.trigger('change:drawing');
            return true;
        };

        this.setTitleAttributes = function (operation) {
            if (operation.axis === 'main') {
                mainTitleModel.setAttributes(operation.attrs);
            } else {
                var titleModel = this.getAxisModel(operation.axis).getTitle();
                titleModel.setAttributes(operation.attrs);
            }
            self.trigger('change:drawing');
            return true;
        };

        this.setLegendAttributes = function (operation) {
            legendModel.setAttributes(operation.attrs);
            self.trigger('change:drawing', 'series');
        };

        this.getLegendModel = function () {
            return legendModel;
        };

        this.getAxisModel = function (axis) {
            return axes[axis];
        };

        this.getTitleModel = function (type) {
            if (type === 'main') {
                return mainTitleModel;
            } else {
                return this.getAxisModel(type).getTitle();
            }
        };

        this.setChartDataSeriesAttributes = function (operation) {
            var index = operation.series;
            var attrs = operation.attrs;

            var dataSeries = data.data[index];
            dataSeries.model.setAttributes(attrs);
            dataSeries.needUpdate = true;
            updateTokenArrays();
            self.trigger('change:drawing', 'series');
            return true;
        };

        /**
         *  Adds formula tokens to the given containing the source ranges of all data
         *  series of this chart object.
         *
         *  @param {Array} array where tokenarrays are pushed in
         */
        this.getTokenArrays = function (res) {
            updateTokenArrays();
            res.push(controllerToken);
        };

        function updateTokenArrays() {
            if (!controllerToken) {
                controllerToken = new TokenArray(app);
            }

            var tmpList = {};
            _.each(data.data, function (dataSerie) {
                dataSerie.model.iterateTokens(function (tkn) {
                    var formula = tkn.getFormula();
                    if (formula.length) {
                        tmpList[formula] = true;
                    }
                });
            });

            var form = _(tmpList).keys().join(',');
            controllerToken.parseFormula(form);
        }

        function makeMin(min, max) {
            if (Math.abs(max - min) < 0.1 || (min / max < 5 / 6)) {
                return 0;
            } else {
                return min - ((max - min) / 2);
            }
        }

        function makeMax(min, max) {
            var res = max + 0.2 * (max - min);
            if (res < max) {
                res = max;
            }
            return res;
        }

        function refreshAxis() {
            refreshAxisX();
            refreshAxisY();
        }

        function refreshAxisX() {
            if (!data.data.length) {
                return;
            }
            axes.x.refreshInfo();

            if (isXYType() || self.getMergedAttributes().chart.type.indexOf('area') === 0) {
                var xMin = +9999999999999;
                var xMax = -9999999999999;

                _.each(data.data, function (dd) {
                    _.each(dd.dataPoints, function (dataPoint) {
                        var x = dataPoint.x;
                        if (_.isDate(x)) {
                            x = x.getTime();
                        }
                        xMin = Math.min(xMin, x);
                        xMax = Math.max(xMax, x);
                    });
                });

				//TODO: the random is only here, because canvasjs ignores the values otherwise
                if (isXYType()) {
                    data.axisX.minimum = xMin - 1.5 - Math.random() * 0.001;
                    data.axisX.maximum = xMax + 1.5 + Math.random() * 0.001;
                } else {
                    data.axisX.minimum = xMin - 0.25 - Math.random() * 0.001;
                    data.axisX.maximum = xMax + 0.25 + Math.random() * 0.001;
                }
            } else {
                data.axisX.minimum = null;
                data.axisX.maximum = null;
            }
        }

        function refreshAxisY() {
            if (!data.data.length) {
                return;
            }

            var type = getCJSType();

            var givenMin = null;
            var givenMax = null;

            var axisInfo = axes.y.getMergedAttributes();
            if (axisInfo) {
                if (axisInfo.min !== 'auto') {
                    givenMin = axes.y.min;
                }
                if (axisInfo.max !== 'auto') {
                    givenMax = axes.y.max;
                }
            }

            axes.y.refreshInfo();

            var yMin = 9999999999999;
            var yMax = -9999999999999;

            if (type.indexOf('100') > 0) {
                var first = data.data[0];
                _.times(first.dataPoints.length, function (j) {
                    var y = 0;
                    _.each(data.data, function (dd) {
                        y += dd.dataPoints[j].y;
                    });
                    yMin = Math.min(yMin, y);
                    yMax = Math.max(yMax, y);
                });
            } else {
                _.each(data.data, function (dd) {
                    _.each(dd.dataPoints, function (dataPoint) {
                        var y = dataPoint.y;
                        yMin = Math.min(yMin, y);
                        yMax = Math.max(yMax, y);
                    });
                });
            }

            var intervalCount = 8;

            if (isXYType()) {
                var diff = yMax - yMin;
                yMax += 0.5 + diff / intervalCount;
                yMin -= 0.5 + diff / intervalCount;
            } else if (type.indexOf('100') > 0) {
                //Trial & error...
                if (yMax >= 0 && yMin >= 0) {
                    //Szenario 1
                    yMin = givenMin || 0;
                    yMax = givenMax || 100;
                } else if (yMax <= 0 && yMin <= 0) {
                    //Szenario 2
                    yMax = givenMax || 0;
                    yMin = givenMin || -100;
                } else {
                    //Szenario 3
                    yMax =  givenMax || 100;
                    yMin =  givenMin || -100;
                }
            } else {
                if (yMax >= 0 && yMin >= 0) {
                    //Szenario 1
                    yMin = givenMin || makeMin(yMin, yMax);
                    yMax = givenMax || makeMax(yMin, yMax);
                } else if (yMax <= 0 && yMin <= 0) {
                    //Szenario 2
                    yMax = givenMax || -makeMin(-yMax, -yMin);
                    yMin = givenMin || -makeMax(-yMax, -yMin);
                } else {
                    //Szenario 3
                    var ma = yMax;
                    var mi = yMin;
                    yMax =  givenMax || makeMax(mi, ma);
                    yMin =  givenMin || -makeMax(-ma, -mi);
                }
            }

            yMin = Math.round(yMin.toPrecision(2));
            yMax = Math.round(yMax.toPrecision(2));

            if (yMin === yMax) {
                yMax = yMin + 1;
            }

            if (yMin < yMax) {
                //TODO: the random is only here, because canvasjs ignores the values otherwise
                var interval = (yMax - yMin) / intervalCount;
                interval = Math.round(interval.toPrecision(1));

                if (interval === 0) {
                    data.axisY.minimum = yMin - Math.random() * 0.001;
                    data.axisY.maximum = yMax + Math.random() * 0.001;
                    data.axisY.interval = null;
                } else {
                    var intCount = Math.round((yMax - yMin) / interval);

                    if (Math.abs(yMax) < Math.abs(yMin)) {
                        data.axisY.minimum = yMax - (interval * intCount) + Math.random() * 0.001;
                        data.axisY.maximum = yMax + Math.random() * 0.001;
                        data.axisY.interval = interval;
                    } else {
                        data.axisY.minimum = yMin + Math.random() * 0.001;
                        data.axisY.maximum = yMin + (interval * intCount) + Math.random() * 0.001;
                        data.axisY.interval = interval;
                    }


                }
            }
        }

        function getStyleId() {
            return self.getMergedAttributes().chart.chartStyleId;
        }

        function clearSeriesColor(generator, position) {
            _(data.data).each(function (dataSeries, index) {
                var at = dataSeries.model.getMergedAttributes();
                if (at.fill.color.type !== 'auto') {
                    generator.generateDrawingOperation(Operations.SET_CHART_DATASERIES_ATTRIBUTES, position[0], position[1], { series: index, attrs: { fill: { color: Color.AUTO } } });
                }
            });
        }

        function createBackgroundColorOp(styleId) {
            var drAttrs = { chart: {chartStyleId: styleId } };

            var bgColor = colorHandler.getBackgroundColor(styleId);
            if (bgColor) {
                drAttrs.fill = { color: bgColor };
            }
            return drAttrs;
        }

        /**
         * return true if the chart has only one series or is type of bubble or donut (because there you see only the first series)
         * @returns {Boolean}
         */
        this.isVaryColorEnabled = function () {
            return colorHandler.isVaryColorEnabled();
        };

        this.isAxesEnabled = function () {
            var type = getCJSType();
            return type !== 'pie' && type !== 'doughnut';
        };

        this.isDataLabelEnabled = function () {
            var type = getCJSType();
            return type !== 'bubble';
        };

        /**
         * it generates a unique-color-ID out of the internal chartStyleId
         * for describing the color patterns
         * return null, if one or more series have a unique fill color
         * @returns {String|null}
         */
        this.getColorSet = function () {
            var indiv = _(data.data).find(function (dataSeries) {
                var at = dataSeries.model.getMergedAttributes();
                if (at.fill.color.type !== 'auto') {
                    return true;
                }
            });

            if (indiv) {
                return null;
            }

            var styleId = getStyleId() - 1;
            var colorSet = styleId % 8;
            return 'cs' + colorSet;
        };

        /**
         * it generates a unique-style-ID out of the internal chartStyleId
         * for describing the background and the highlight of the chart
         * it ignores unique fill colors of the series
         * @returns {String}
         */
        this.getStyleSet = function () {
            var styleId = getStyleId() - 1;
            var styleSet = (styleId / 8) | 0;

            if (styleSet < 4) {
                return 'ss0';
            } else {
                return 'ss' + styleSet;
            }
        };

        /**
         * updates the interal chartStyleId by the unique ID
         * deletes all series colors if existing
         * @param colorSet
         */
        this.changeColorSet = function (colorSet) {
            var styleId = getStyleId() - 1;
            var csId = colorSet.replace('cs', '') | 0;
            var styleSet = (styleId / 8) | 0;

            styleId = styleSet * 8 + csId + 1;


            var position = self.getPosition();
            var generator = docModel.createOperationsGenerator();

            clearSeriesColor(generator, position);

            var drAttrs = createBackgroundColorOp(styleId);
            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position[0], position[1], { attrs: drAttrs });
            docModel.applyOperations(generator.getOperations());

        };

        /**
         * updates the interal chartStyleId by the unique ID
         * does not touch the series colors
         * @param colorSet
         */
        this.changeStyleSet = function (styleSet) {
            var styleId = getStyleId() - 1;
            var ssId = styleSet.replace('ss', '') | 0;
            var colorSet = styleId % 8;

            styleId = ssId * 8 + colorSet + 1;

            var position = self.getPosition();

            var generator = docModel.createOperationsGenerator();

            var drAttrs = createBackgroundColorOp(styleId);
            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position[0], position[1], { attrs: drAttrs });
            docModel.applyOperations(generator.getOperations());

        };

        /**
         * changes the varycolors flag
         * deletes all series colors if existing, so the colors only orient on the patterns
         * @param colorSet
         */
        this.changeVaryColors = function (state) {
            var position = self.getPosition();
            var generator = docModel.createOperationsGenerator();

            clearSeriesColor(generator, position);

            generator.generateDrawingOperation(Operations.SET_DRAWING_ATTRIBUTES, position[0], position[1], { attrs: { chart: {varyColors: state} } });

            docModel.applyOperations(generator.getOperations());
        };

        this.getSeriesCount = function () {
            return data.data.length;
        };

        /**
         * must be called for having correct behavior mit "varyColors" in combination with more than one series
         */
        this.firstInit = function () {
            updateChartType();
        };

        /**
         * iterates all sources and calculates min and max positions,
         * direction of the series-values and the sheetindex
         *
         * @returns {object}
         * [res.warn] reutrns if sources are not on the same sheet,
         * or if they have different direction
         * (these are symptoms that the sources have no definite relations)
         * [res.range.start] start has the smallest position of all sources
         * [res.range.end] start has the biggest position of all sources
         * [res.axis] detected main-axis of the source-values
         * [res.sheet] sheet-index of the sources
         *
         */
        this.getExchangeInfo = function () {
            var start = [docModel.getMaxCol() + 1, docModel.getMaxRow() + 1];
            var end = [-1, -1];

            var mainAxis = null;
            var sheet = null;
            var warn = null;

            _.each(data.data, function (dataSeries) {
                dataSeries.model.iterateTokens(function (token, name) {
                    var range = token.resolveRangeList()[0];
                    if (range) {
                        if (sheet === null) {
                            sheet = range.sheet;
                        } else if (sheet !== range.sheet) {
                            warn = 'sheets';
                        }
                        if (name === 'values') {
                            var axis = null;
                            var col = SheetUtils.getColCount(range);
                            var row = SheetUtils.getRowCount(range);
                            if (col > row) {
                                axis = 0;
                            } else if (row > 1) {
                                axis = 1;
                            }
                            if (_.isNumber(axis)) {
                                if (mainAxis === null) {
                                    mainAxis = axis;
                                } else if (mainAxis !== axis) {
                                    warn = 'directions';
                                }
                            }
                        }
                        _.times(2, function (ax) {
                            start[ax] = Math.min(start[ax], range.start[ax]);
                            end[ax] = Math.max(end[ax], range.end[ax]);
                        });
                    }
                });
            });
            if (warn) {
                return {warn: warn};
            } else if (sheet === null) {
                return {warn: 'nodata'};
            } else {
                return {range: {start: start, end: end}, axis: mainAxis, sheet: sheet};
            }
        };

        // initialization -----------------------------------------------------

        this.on('change:attributes', changeAttributes);
        this.on('change:drawing', changeDrawing);
        this.listenTo(app, 'docs:update:cells', updateNotificationHandler);
        this.listenTo(docModel, 'insert:columns delete:columns insert:rows delete:rows', colRowChangeHandler);

        // constructor called from BaseObject.clone()
        this.registerCloneConstructor(function (newSheetModel, newSheet) {

            var cloneData = { series: [], axes: axes, oldSheet: sheetModel.getIndex(), newSheet: newSheet, legend: legendModel, title: mainTitleModel },
                tokenArray = new TokenArray(app, sheetModel);

            _.each(data.data, function (dataSeries) {
                var series = {};

                dataSeries.model.iterateTokens(function (token, name) {
                    tokenArray.parseFormula(token.getFormula());
                    tokenArray.relocateSheet(cloneData.oldSheet, newSheet);
                    series[name] = tokenArray.getFormula();
                });
                var nsp = dataSeries.model.getExplicitAttributes();
                nsp.series = series;
                cloneData.series.push(nsp);
            });
            tokenArray.destroy();
            tokenArray = null;

            // pass private data as hidden arguments to the constructor
            return new SheetChartModel(app, newSheetModel, this.getExplicitAttributes(), cloneData);
        });

        // clone private data passed as hidden argument to the c'tor
        (function (args) {
            var cloneData = args[SheetChartModel.length];
            if (_.isObject(cloneData)) {
                app.executeDelayed(function () {
                    if (!self || self.destroyed) { return; }
                    _.each(cloneData.axes, function (axisModel, axisType) {
                        self.setAxisAttributes({ axis: axisType, attrs: axisModel.getExplicitAttributes() });
                        self.setGridlineAttributes({ axis: axisType, attrs: axisModel.getGrid().getExplicitAttributes() });
                        var axTitle = axisModel.getTitle();
                        if (axTitle) {
                            self.setTitleAttributes({ axis: axisType, attrs: axTitle.getExplicitAttributes() });
                        }
                    });
                    _.each(cloneData.series, function (series, index) {
                        self.insertDataSeries({ series: index, attrs: series });
                    });
                    self.setTitleAttributes({ axis: 'main', attrs: cloneData.title.getExplicitAttributes() });
                    self.setLegendAttributes({ attrs: cloneData.legend.getExplicitAttributes() });
                });
            }
        }(arguments));

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _(axes).invoke('destroy');
            _.chain(data.data).pluck('model').invoke('destroy');
            mainTitleModel.destroy();
            legendModel.destroy();

            mainTitleModel = legendModel = docModel = self = data = axes = null;

        });

    } // class SheetChartModel

    // exports ================================================================

    // derive this class from class DrawingModel
    return DrawingModel.extend({ constructor: SheetChartModel });

});
