/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/cellmodel',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/attributedmodel',
     'io.ox/office/editframework/model/format/border'
    ], function (Utils, AttributedModel, Border) {

    'use strict';

    // class CellModel ========================================================

    /**
     * Base class for objects containing cell attributes. Provides a
     * specialized implementation of the method 'setAttributes()' that is able
     * to update single border properties of visible border attributes.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SpreadsheetApplication} app
     *  The application instance containing this cell model object.
     *
     * @param {Object} [initAttributes]
     *  An attribute set with initial formatting attributes for the cell.
     *
     * @param {Object} [initOptions]
     *  A map with additional options controlling the behavior of this
     *  instance. Supports all attributes also supported by the base class
     *  AttributedModel. The following attributes will be set to fixed values
     *  and cannot be changed:
     *  - initOptions.silent will be set to true (no change events).
     *  - initOptions.styleFamily will be set to 'cell'.
     */
    function CellModel(app, initAttributes, initOptions) {

        // base constructor ---------------------------------------------------

        // do not trigger any events (collection entries are potential mass objects)
        AttributedModel.call(this, app, initAttributes, Utils.extendOptions(initOptions, { silent: true, styleFamily: 'cell' }));

        // methods ------------------------------------------------------------

        /**
         * Changes and/or removes specific explicit formatting attributes, or
         * the style sheet reference of this model object. In addition to the
         * base class method 'AttributedModel.setAttributes()', this method
         * adds special treatment for border attributes (see options below).
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method.
         *  Additionally to the options supported by the base class method
         *  'AttributedModel.setAttributes()', the following options are
         *  supported:
         *  @param {Boolean} [options.innerLeft=false]
         *      If set to true, the border attribute 'borderInsideVert' will be
         *      used for the left border of the cell, instead of the attribute
         *      'borderLeft'.
         *  @param {Boolean} [options.innerRight=false]
         *      If set to true, the border attribute 'borderInsideVert' will be
         *      used for the right border of the cell, instead of the attribute
         *      'borderRight'.
         *  @param {Boolean} [options.innerTop=false]
         *      If set to true, the border attribute 'borderInsideHor' will be
         *      used for the top border of the cell, instead of the attribute
         *      'borderTop'.
         *  @param {Boolean} [options.innerBottom=false]
         *      If set to true, the border attribute 'borderInsideHor' will be
         *      used for the bottom border of the cell, instead of the
         *      attribute 'borderBottom'.
         *  @param {Boolean} [options.visibleBorders=false]
         *      If set to true, border attributes may be incomplete. Specified
         *      border properties (line style, line width, line color) will be
         *      applied for visible borders only.
         *
         * @returns {Boolean}
         *  Whether any attributes have actually been changed.
         */
        this.setCellAttributes = function (attributes, options) {

            var // the current merged attributes of this cell
                mergedAttributes = this.getMergedAttributes();

            // copies the inner border attribute to an outer border attribute
            function copyBorderAttribute(cellAttributes, optionName, innerBorderName, outerBorderName) {
                if (Utils.getBooleanOption(options, optionName, false)) {
                    if (innerBorderName in cellAttributes) {
                        cellAttributes[outerBorderName] = cellAttributes[innerBorderName];
                    } else {
                        delete cellAttributes[outerBorderName];
                    }
                }
            }

            // updates a visible border attribute with the passed incomplete border (in 'visibleBorders' mode)
            function updateVisibleBorderAttribute(cellAttributes, borderName) {
                if (_.isObject(cellAttributes[borderName]) && Border.isVisibleBorder(mergedAttributes.cell[borderName])) {
                    cellAttributes[borderName] = _({}).extend(mergedAttributes.cell[borderName], cellAttributes[borderName]);
                    if (!Border.isVisibleBorder(cellAttributes[borderName])) {
                        cellAttributes[borderName] = Border.NONE;
                    }
                }
            }

            // process border attributes
            if (_.isObject(attributes.cell)) {

                // update the border attributes in a deep copy of the passed attribute set
                attributes = _.copy(attributes, true);

                // move inner border attributes to outer border attributes if specified in the options
                copyBorderAttribute(attributes.cell, 'innerLeft', 'borderInsideVert', 'borderLeft');
                copyBorderAttribute(attributes.cell, 'innerRight', 'borderInsideVert', 'borderRight');
                copyBorderAttribute(attributes.cell, 'innerTop', 'borderInsideHor', 'borderTop');
                copyBorderAttribute(attributes.cell, 'innerBottom', 'borderInsideHor', 'borderBottom');

                // cells cannot contain inner border attributes
                delete attributes.cell.borderInsideVert;
                delete attributes.cell.borderInsideHor;

                // update existing borders in 'visibleBorders' mode
                if (Utils.getBooleanOption(options, 'visibleBorders', false)) {
                    updateVisibleBorderAttribute(attributes.cell, 'borderLeft');
                    updateVisibleBorderAttribute(attributes.cell, 'borderRight');
                    updateVisibleBorderAttribute(attributes.cell, 'borderTop');
                    updateVisibleBorderAttribute(attributes.cell, 'borderBottom');
                }
            }

            // apply the attributes
            return this.setAttributes(attributes, options);
        };

    } // class CellModel

    // exports ================================================================

    // derive this class from class AttributedModel
    return AttributedModel.extend({ constructor: CellModel });

});
