/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/app/controller',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/border',
     'io.ox/office/editframework/model/format/mixedborder',
     'io.ox/office/editframework/app/editcontroller',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/app/config',
     'io.ox/office/spreadsheet/controller/drawingcontroller'
    ], function (Utils, Border, MixedBorder, EditController, SheetUtils, Config, DrawingController) {

    'use strict';

    // class SpreadsheetController ============================================

    /**
     * The controller of a OX Spreadsheet application.
     *
     * @constructor
     *
     * @extends EditController
     */
    function SpreadsheetController(app) {

        var // self reference
            self = this,

            // the spreadsheet model
            model = null,

            // the spreadsheet view
            view = null,

            // all the little controller items
            items = {

                // document ---------------------------------------------------

                'document/search/search': {
                    // enabled in read-only mode
                    set: function (searchQuery) { view.search(searchQuery); },
                    focus: 'never'
                },

                'document/search/nextResult': {
                    // enabled in read-only mode
                    set: function () {
                        if (app.getWindow().search.active) {
                            view.searchIterate('next');
                        }
                    },
                    shortcut: [{ keyCode: 'G', ctrlOrMeta: true }, { keyCode: 'F3' }],
                    focus: 'never' // do not grab focus after setter
                },

                'document/search/previousResult': {
                    // enabled in read-only mode
                    set: function () {
                        if (app.getWindow().search.active) {
                            view.searchIterate('prev');
                        }
                    },
                    shortcut: [{ keyCode: 'G', ctrlOrMeta: true, shift: true }, { keyCode: 'F3', shift: true }],
                    focus: 'never'
                },

                'document/search/replaceAll': {
                    parent: 'document/editable',
                    set: function () { view.searchReplace({all: true}); },
                    focus: 'never' // do not grab focus after setter
                },

                'document/search/replaceSelected': {
                    parent: 'document/editable',
                    set: function () { view.searchReplace(); },
                    focus: 'never'
                },

                'document/search/close': {
                    // enabled in read-only mode
                    set: function () { view.searchClear({clearInputs: true}); }
                },

                // selection mode ---------------------------------------------

                // enabled if the document is editable, and no drawing objects are selected
                'document/editable/cell': {
                    parent: 'document/editable',
                    enable: function () { return !view.hasDrawingSelection(); }
                },

                // view -------------------------------------------------------

                // enabled if the in-place cell edit mode is NOT active
                'view/cell/editmode/off': {
                    parent: 'app/valid',
                    enable: function () { return !view.isCellEditMode(); }
                },

                'view/sheet/active': {
                    get: function () { return view.getActiveSheet({ visible: true }); },
                    set: function (index) { view.setActiveSheet(index, { visible: true }); }
                },

                'view/sheet/prev': {
                    enable: function () { return view.getActiveSheet({ visible: true }) > 0; },
                    set: function () { view.activatePreviousSheet(); },
                    shortcut: { keyCode: 'PAGE_UP', ctrlOrMeta: true, alt: true }
                },

                'view/sheet/next': {
                    enable: function () { return view.getActiveSheet({ visible: true }) + 1 < view.getVisibleSheetCount(); },
                    set: function () { view.activateNextSheet(); },
                    shortcut: { keyCode: 'PAGE_DOWN', ctrlOrMeta: true, alt: true }
                },

                'view/subtotals/type': {
                    parent: 'app/valid',
                    get: function () { return app.getUserSettingsValue('subtotalType', 'sum'); },
                    set: function (subtotalType) { app.setUserSettingsValue('subtotalType', subtotalType); }
                },

                'view/grid/show': {
                    parent: 'app/valid',
                    get: function () { return view.getActiveSheetModel().getViewAttribute('showGrid'); },
                    set: function (state) { view.getActiveSheetModel().setViewAttribute('showGrid', state); }
                },

                'view/split/dynamic': {
                    parent: 'view/cell/editmode/off',
                    get: function () { return view.hasDynamicSplit(); },
                    set: function (state) { view.setDynamicSplit(state); }
                },

                'view/split/frozen': {
                    parent: 'view/cell/editmode/off',
                    get: function () { return view.hasFrozenSplit(); },
                    set: function (value) {
                        if (_.isBoolean(value)) {
                            view.setFrozenSplit(value);
                        } else if (value === 'toggle') {
                            view.setFrozenSplit(!this.getValue());
                        } else if (_.isObject(value)) {
                            view.setFixedFrozenSplit(value.cols, value.rows);
                        }
                    }
                },

                'view/zoom/dec': {
                    parent: 'app/valid',
                    enable: function () { return view.getZoom() > SheetUtils.MIN_ZOOM; },
                    set: function () { view.decZoom(); }
                },

                'view/zoom/inc': {
                    parent: 'app/valid',
                    enable: function () { return view.getZoom() < SheetUtils.MAX_ZOOM; },
                    set: function () { view.incZoom(); }
                },

                // sheet operations -------------------------------------------

                'sheet/unlocked': {
                    parent: 'document/editable',
                    enable: function () { return !view.getActiveSheetModel().isLocked(); }
                },

                'sheet/attributes': {
                    parent: 'document/editable',
                    get: function () { return view.getSheetAttributes().sheet; }
                },

                'sheet/visible': {
                    parent: 'sheet/attributes',
                    enable: function () { return view.getVisibleSheetCount() > 1; },
                    get: function (attributes) { return attributes.visible; },
                    set: function (visible) { view.setSheetAttributes({ sheet: { visible: visible } }); }
                },

                'sheet/showall': {
                    parent: 'document/editable',
                    enable: function () { return view.getHiddenSheetCount() > 0; },
                    set: function () { view.showAllSheets(); }
                },

                'sheet/insert': {
                    parent: 'document/editable',
                    enable: function () { return model.getSheetCount() < Config.getMaxSheetCount(); },
                    set: function () { view.insertSheet(); }
                },

                'sheet/delete': {
                    parent: 'document/editable',
                    enable: function () { return view.getVisibleSheetCount() > 1; },
                    set: function () { view.deleteSheet(); }
                },

                'sheet/name': {
                    parent: 'document/editable',
                    get: function () { return view.getSheetName(); },
                    set: function (sheetName) { view.setSheetName(sheetName); }
                },

                'sheet/rename/dialog': {
                    parent: 'sheet/name',
                    set: function () { return view.showRenameSheetDialog(); },
                    focus: 'wait' // wait for the dialog before returning focus to application
                },

                'sheet/copy/dialog': {
                    parent: 'sheet/insert',
                    set: function () { return view.showCopySheetDialog(); },
                    focus: 'wait' // wait for the dialog before returning focus to application
                },

                'sheet/reorder/dialog': {
                    parent: 'document/editable',
                    enable: function () { return view.getVisibleSheetCount() > 1; },
                    set: function () { return view.showReorderSheetsDialog(); },
                    focus: 'wait'
                },

                // column operations ------------------------------------------

                'column/operation': {
                    parent: ['document/editable/cell', 'sheet/unlocked', 'view/cell/editmode/off']
                },

                'column/insert': {
                    parent: 'column/operation',
                    enable: function () { return view.columnsInsertable(); },
                    set: function () { view.insertColumns(); }
                },

                'column/delete': {
                    parent: 'column/operation',
                    enable: function () { return view.columnsDeleteable(); },
                    set: function () { view.deleteColumns(); }
                },

                'column/attributes': {
                    parent: 'column/operation',
                    get: function () { return view.getColumnAttributes(); }
                },

                'column/visible': {
                    parent: 'column/attributes',
                    get: function (attributes) { return attributes.column.visible; },
                    set: function (state) { view.setColumnAttributes({ column: { visible: state } }); }
                },

                'column/hide': {
                    parent: 'column/visible',
                    enable: function () { return this.getValue() !== false; },
                    set: function () { view.setColumnAttributes({ column: { visible: false } }); }
                },

                'column/show': {
                    parent: 'column/visible',
                    enable: function () { return this.getValue() !== true; },
                    set: function () { view.setColumnAttributes({ column: { visible: true } }); }
                },

                'column/width/optimal': {
                    parent: 'column/hide', // only enabled when visible columns are selected
                    set: function () { view.setOptimalColumnWidth(); }
                },

                'column/attributes/active': {
                    parent: 'column/operation',
                    get: function () { return view.getActiveColumnAttributes(); }
                },

                'column/width/active': {
                    parent: 'column/attributes/active',
                    get: function (attributes) { return attributes.visible ? attributes.width : 0; },
                    set: function (value) { view.setColumnWidth(value); }
                },

                // row operations ---------------------------------------------

                'row/operation': {
                    parent: ['document/editable/cell', 'sheet/unlocked', 'view/cell/editmode/off']
                },

                'row/insert': {
                    parent: 'row/operation',
                    enable: function () { return view.rowsInsertable(); },
                    set: function () { view.insertRows(); }
                },

                'row/delete': {
                    parent: 'row/operation',
                    enable: function () { return view.rowsDeleteable(); },
                    set: function () { view.deleteRows(); }
                },

                'row/attributes': {
                    parent: 'row/operation',
                    get: function () { return view.getRowAttributes(); }
                },

                'row/visible': {
                    parent: 'row/attributes',
                    get: function (attributes) { return attributes.row.visible; },
                    set: function (state) { view.setRowAttributes({ row: { visible: state } }); }
                },

                'row/hide': {
                    parent: 'row/visible',
                    enable: function () { return this.getValue() !== false; },
                    set: function () { view.setRowAttributes({ row: { visible: false } }); }
                },

                'row/show': {
                    parent: 'row/visible',
                    enable: function () { return this.getValue() !== true; },
                    set: function () { view.setRowAttributes({ row: { visible: true } }); }
                },

                'row/height/optimal': {
                    parent: 'row/hide', // only enabled when visible rows are selected
                    set: function () { view.setOptimalRowHeight(); }
                },

                'row/attributes/active': {
                    parent: 'row/operation',
                    get: function () { return view.getActiveRowAttributes(); }
                },

                'row/height/active': {
                    parent: 'row/attributes/active',
                    get: function (attributes) { return attributes.visible ? attributes.height : 0; },
                    set: function (value) { view.setRowHeight(value); }
                },

                // cell operations --------------------------------------------

                'cell/operation': {
                    parent: ['document/editable/cell', 'view/cell/editmode/off']
                },

                'cell/merge': {
                    parent: 'cell/operation',
                    enable: function () { return view.hasAnyRangeSelected(); },
                    get: function () { return view.hasMergedRangeSelected(); },
                    set: function (type) { view.mergeRanges(type); }
                },

                'cell/autoformula': {
                    parent: 'cell/operation',
                    set: function (type) { view.insertAutoSum(type); }
                },

                'cell/painter': {
                    parent: 'cell/operation',
                    get: function () { return view.applyPainterState(); },
                    set: function (state) { view.applyPainter(state); }
                },

                // cell attributes --------------------------------------------

                'cell/attributes': {
                    parent: 'document/editable/cell',
                    get: function () { return view.getCellAttributes(); }
                },

                'cell/stylesheet': {
                    parent: ['cell/attributes', 'view/cell/editmode/off'],
                    get: function (attributes) { return attributes.styleId; },
                    set: function (styleId) { view.fillCellRanges(undefined, { styleId: styleId }, { clear: true }); }
                },

                'cell/reset': {
                    parent: ['cell/attributes', 'view/cell/editmode/off'],
                    set: function () { view.clearAttributes(); }
                },

                // cell attributes (family 'cell') ----------------------------

                'cell/attributes/cell': {
                    parent: ['cell/attributes', 'view/cell/editmode/off'],
                    get: function (attributes) { return attributes.cell; }
                },

                'cell/fillcolor': {
                    parent: 'cell/attributes/cell',
                    get: function (attributes) { return attributes.fillColor; },
                    set: function (color) { view.setCellAttribute('fillColor', color); }
                },

                'cell/linebreak': {
                    parent: 'cell/attributes/cell',
                    get: function (attributes) { return attributes.wrapText; },
                    set: function (wrap) { view.setCellAttribute('wrapText', wrap); }
                },

                'cell/alignhor': {
                    parent: 'cell/attributes/cell',
                    get: function (attributes) { return attributes.alignHor; },
                    set: function (alignment) { view.setCellAttribute('alignHor', alignment); }
                },

                'cell/alignvert': {
                    parent: 'cell/attributes/cell',
                    get: function (attributes) { return attributes.alignVert; },
                    set: function (alignment) { view.setCellAttribute('alignVert', alignment); }
                },

                // border mode, as Boolean map with position keys (left, top, ...)
                'cell/border/mode': {
                    parent: ['document/editable/cell', 'view/cell/editmode/off'],
                    get: function () { return view.getBorderMode(); },
                    set: function (borderMode) { view.setBorderMode(borderMode); }
                },

                // all mixed border attributes of the selection
                'cell/border/attributes': {
                    parent: 'document/editable/cell',
                    get: function () { return view.getBorderAttributes(); }
                },

                // enabled if at least one border in the selection is visible
                'cell/border/visible': {
                    parent: 'cell/border/attributes',
                    enable: function (borderAttributes) { return _(borderAttributes).any(MixedBorder.isVisibleBorder); }
                },

                // a single mixed border for all borders in the selection
                'cell/border/value': {
                    parent: ['cell/border/visible', 'view/cell/editmode/off'],
                    get: function (borderAttributes) { return MixedBorder.mixBorders(_(borderAttributes).values()); }
                },

                'cell/border/style': {
                    parent: 'cell/border/value',
                    get: function (border) { return border.style; },
                    set: function (value) { view.changeVisibleBorders({ style: value }); }
                },

                // width of the border lines, in points (rounded to 1/2 points)
                'cell/border/width': {
                    parent: 'cell/border/value',
                    get: function (border) { return _.isNumber(border.width) ? Utils.convertHmmToLength(border.width, 'pt', 0.5) : null; },
                    set: function (value) { view.changeVisibleBorders({ width: Utils.convertLengthToHmm(value, 'pt') }); }
                },

                'cell/border/color': {
                    parent: 'cell/border/value',
                    get: function (border) { return border.color; },
                    set: function (value) { view.changeVisibleBorders({ color: value }); }
                },

                'cell/numberformat/category': {
                    parent: ['document/editable/cell', 'view/cell/editmode/off'],
                    get: function () { return view.getNumberFormatCategory(); },
                    set: function (category) { view.setNumberFormatCategory(category); },
                    shortcut: [
                        { keyCode: '1', shift: true, ctrlOrMeta: true, value: 'number' },
                        { keyCode: '2', shift: true, ctrlOrMeta: true, value: 'scientific' },
                        { keyCode: '3', shift: true, ctrlOrMeta: true, value: 'date' },
                        { keyCode: '4', shift: true, ctrlOrMeta: true, value: 'currency' },
                        { keyCode: '5', shift: true, ctrlOrMeta: true, value: 'percent' },
                        { keyCode: '6', shift: true, ctrlOrMeta: true, value: 'standard' }
                    ]
                },

                'cell/numberformat/code': {
                    parent: 'cell/numberformat/category',
                    enable: function (category) { return !_(['standard', 'custom']).contains(category); },
                    get: function () { return view.getNumberFormatCode(); },
                    set: function (formatCode) { view.setCellAttribute('numberFormat', { code: formatCode }); }
                },

                // cell attributes (family 'character') -----------------------

                // also enabled in in-place cell edit mode
                'cell/attributes/character': {
                    parent: 'cell/attributes',
                    get: function (attributes) { return attributes.character; }
                },

                'character/bold': {
                    parent: 'cell/attributes/character',
                    get: function (attributes) { return attributes.bold; },
                    set: function (state) { view.setCharacterAttribute('bold', state); },
                    shortcut: { keyCode: 'B', ctrlOrMeta: true, value: function (state) { return !state; } }
                },

                'character/italic': {
                    parent: 'cell/attributes/character',
                    get: function (attributes) { return attributes.italic; },
                    set: function (state) { view.setCharacterAttribute('italic', state); },
                    shortcut: { keyCode: 'I', ctrlOrMeta: true, value: function (state) { return !state; } }
                },

                'character/underline': {
                    parent: 'cell/attributes/character',
                    get: function (attributes) { return attributes.underline; },
                    set: function (state) { view.setCharacterAttribute('underline', state); },
                    shortcut: { keyCode: 'U', ctrlOrMeta: true, value: function (state) { return !state; } }
                },

                'character/strike': {
                    parent: 'cell/attributes/character',
                    get: function (attributes) { return _.isString(attributes.strike) ? (attributes.strike !== 'none') : null; },
                    set: function (state) { view.setCharacterAttribute('strike', state ? 'single' : 'none'); }
                },

                'character/color': {
                    parent: 'cell/attributes/character',
                    get: function (attributes) { return attributes.color; },
                    set: function (color) { view.setCharacterAttribute('color', color); }
                },

                'character/fontname': {
                    parent: 'cell/attributes/character',
                    get: function (attributes) { return attributes.fontName; },
                    set: function (fontName) { view.setCharacterAttribute('fontName', fontName); }
                },

                'character/fontsize': {
                    parent: 'cell/attributes/character',
                    get: function (attributes) { return attributes.fontSize; },
                    set: function (fontSize) { view.setCharacterAttribute('fontSize', fontSize); }
                },

                'character/url': {
                    parent: 'cell/attributes/character',
                    get: function (attributes) { return attributes.url; },
                    set: function (url) { view.setCharacterAttribute('url', url); }
                },

                'character/hyperlink/dialog': {
                    parent: ['character/url', 'view/cell/editmode/off'],
                    set: function () { return view.editHyperlink(); },
                    focus: 'wait' // wait for the dialog before returning focus to application
                },

                'character/hyperlink/remove': {
                    parent: ['character/url', 'view/cell/editmode/off'],
                    set: function () { view.removeHyperlink(); }
                },

                // drawing ---------------------------------------------------
                // drawing comes now from the DrawingController

                // debug ------------------------------------------------------

                'debug/view/update': {
                    parent: 'debug/enabled',
                    set: function (type) { view.debugRequestUpdate(type); }
                },

                'debug/insert/drawing': {
                    parent: 'debug/enabled',
                    enable: function () { return model.getEditMode(); },
                    set: function (type) { view.debugInsertDrawing(type); }
                }
            };

        // base constructor ---------------------------------------------------

        EditController.call(this, app);
        DrawingController.call(this, app);

        // initialization -----------------------------------------------------

        // register item definitions
        this.registerDefinitions(items);

        // initialization after construction
        app.on('docs:init', function () {

            // model and view are not available at construction time
            model = app.getModel();
            view = app.getView();

            // update GUI after changed selection, and after receiving view updates
            view.on('change:sheet:viewattributes change:layoutdata celledit:enter celledit:change celledit:leave', function () { self.update(); });
        });

    } // class SpreadsheetController

    // exports ================================================================

    // derive this class from class EditController
    return EditController.extend({ constructor: SpreadsheetController });

});
