/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/app/application',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/io',
     'io.ox/office/editframework/app/editapplication',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/model',
     'io.ox/office/spreadsheet/view/view',
     'io.ox/office/spreadsheet/app/controller',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, IO, EditApplication, SheetUtils, SpreadsheetModel, SpreadsheetView, SpreadsheetController, gt) {

    'use strict';

    var // default settings for locale data
        DEFAULT_LOCALE_DATA = { functions: {}, errorCodes: {}, decSeparator: '.', groupSeparator: ',', listSeparator: ',', trueLiteral: 'TRUE', falseLiteral: 'FALSE' },

        // global cache for locale data, mapped by locale identifier
        localeDataCache = {};

    // private static functions ===============================================

    /**
     * Inserts additional information into the passed locale data object.
     */
    function postProcessLocaleData(localeData) {

        // build inverted function name map (translated to native), and arrays of function names
        localeData.invFunctions = _.invert(localeData.functions);
        localeData.nativeFunctionNames = _.chain(localeData.functions).keys().sort().value();
        localeData.localizedFunctionNames = _.chain(localeData.invFunctions).keys().sort().value();
        if (localeData.nativeFunctionNames.length !== localeData.localizedFunctionNames.length) {
            Utils.error('SpreadsheetApplication.postProcessLocaleData(): list of translated function names contains duplicates');
        }

        // build inverted error code map (translated to native), and arrays of error codes
        localeData.invErrorCodes = _.invert(localeData.errorCodes);
        localeData.nativeErrorCodes = _.chain(localeData.errorCodes).keys().sort().value();
        localeData.localizedErrorCodes = _.chain(localeData.invErrorCodes).keys().sort().value();
        if (localeData.nativeErrorCodes.length !== localeData.localizedErrorCodes.length) {
            Utils.error('SpreadsheetApplication.postProcessLocaleData(): list of translated error codes contains duplicates');
        }
    }

    // class SpreadsheetApplication ===========================================

    /**
     * The OX Spreadsheet application.
     *
     * Triggers the following additional events:
     * - 'docs:update:cells'
     *      After the application has received a 'docs:update' event which
     *      contains the contents and/or addresses of changed cells in the
     *      document. Event handlers receive the following parameter:
     *      (1) {Object} changedData
     *          A map of change descriptors for each changed sheet, mapped by
     *          the zero-based sheet index. Each change descriptor object
     *          contains the following properties:
     *          - {Number} sheet
     *              The zero-based sheet index. Equal to the map key of this
     *              object, but as number for convenience.
     *          - {Array} [rangeContents]
     *              An array with cell range descriptors. Each descriptor
     *              object contains the same properties as also sent in the
     *              response data of view update requests. If omitted, no cell
     *              content data is available for the sheet. If omitted, no
     *              cells have been changed in the sheet, or no explicit cell
     *              content data is available yet (see property 'dirtyRanges').
     *          - {Array} [dirtyRanges]
     *              An array with range addresses of additional cells that are
     *              not included in the 'rangeContents' property, but whose
     *              contents have been changed in the sheet. These ranges need
     *              to be updated with further server requests. If omitted, the
     *              sheet does not contain additional dirty cell ranges.
     *          - {Array} [changedRanges]
     *              An array containing the unified range addresses from the
     *              properties 'rangeContents' and 'dirtyRanges'. Will exist,
     *              if either of these properties exists too.
     *          - {Number} [usedCols]
     *              The number of used columns in the sheet. If omitted, the
     *              number of used columns has not changed.
     *          - {Number} [usedRows]
     *              The number of used rows in the sheet. If omitted, the
     *              number of used rows has not changed.
     *      (2) {Array} allChangedRanges
     *          An array containing the range addresses of all changed ranges
     *          in all sheets. This is the concatenation of all cell ranges
     *          contained in the 'changedRanges' properties for all sheets in
     *          the 'changedData' event parameter. Each range object in this
     *          array contains the additional property 'sheet' with the
     *          zero-based sheet index of the range.
     *
     * @constructor
     *
     * @extends EditApplication
     *
     * @param {Object} [appOptions]
     *  A map of static application options, that have been passed to the
     *  static method BaseApplication.createLauncher().
     *
     * @param {Object} [launchOptions]
     *  A map of options to control the properties of the application. Supports
     *  all options supported by the base class EditApplication.
     */
    var SpreadsheetApplication = EditApplication.extend({ constructor: function (appOptions, launchOptions) {

        var // self reference
            self = this,

            // the spreadsheet document model and view
            model = null,
            view = null,

            // the locale data (translated function names etc.)
            localeData = _.copy(DEFAULT_LOCALE_DATA, true),

            // the function help resource
            functionHelp = {};

        // base constructor ---------------------------------------------------

        EditApplication.call(this, SpreadsheetModel, SpreadsheetView, SpreadsheetController, appOptions, launchOptions, {
            newDocumentParams: { initial_sheetname: SheetUtils.generateSheetName(0) },
            preProcessHandler: preProcessDocument,
            preProcessProgressSize: 0.1,
            postProcessHandler: postProcessDocument,
            postProcessProgressSize: 0,
            importFailedHandler: importFailedHandler,
            prepareFlushHandler: prepareFlushDocument,
            prepareLoseEditRightsHandler: prepareLoseEditRights,
            realTimeDelay: 10
        });

        // private methods ----------------------------------------------------

        /**
         * Preparation of the document, before its operations will be applied.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been prepared successfully; or rejected on any error.
         */
        function preProcessDocument() {

            var // load function help data asynchronously
                functionHelpPromise = IO.loadResource('io.ox/office/spreadsheet/resource/functions');

            // store function help data, but ignore the success/failure result
            // (application may run even without function help)
            functionHelpPromise.done(function (data) { functionHelp = data; });

            // try to receive cached locale data already downloaded before
            if (Utils.LOCALE in localeDataCache) {
                localeData = localeDataCache[Utils.LOCALE];
                return $.when();
            }

            // request locale data from server
            return self.sendFilterRequest({
                method: 'POST',
                params: {
                    action: 'query',
                    requestdata: JSON.stringify({
                        type: 'localeData',
                        locale: Utils.LOCALE
                    })
                }
            })
            .then(function (result) {

                // print console warnings for missing properties
                if (Utils.DEBUG) {
                    if (!result.decSeparator) { Utils.error('SpreadsheetApplication.preProcessDocument(): missing decimal separator'); }
                    if (!result.groupSeparator) { Utils.error('SpreadsheetApplication.preProcessDocument(): missing group separator'); }
                    if (!result.listSeparator) { Utils.error('SpreadsheetApplication.preProcessDocument(): missing list separator'); }
                    if (!result.trueLiteral) { Utils.error('SpreadsheetApplication.preProcessDocument(): missing literal for TRUE'); }
                    if (!result.falseLiteral) { Utils.error('SpreadsheetApplication.preProcessDocument(): missing literal for FALSE'); }
                }

                // extract locale data settings from response data object
                localeData = {};
                localeData.functions = Utils.getObjectOption(result, 'functions');
                localeData.errorCodes = Utils.getObjectOption(result, 'errorCodes');
                localeData.decSeparator = Utils.getStringOption(result, 'decSeparator', DEFAULT_LOCALE_DATA.decSeparator, true)[0];
                localeData.groupSeparator = Utils.getStringOption(result, 'groupSeparator', DEFAULT_LOCALE_DATA.groupSeparator, true)[0];
                localeData.listSeparator = Utils.getStringOption(result, 'listSeparator', DEFAULT_LOCALE_DATA.listSeparator, true)[0];
                localeData.trueLiteral = Utils.getStringOption(result, 'trueLiteral', DEFAULT_LOCALE_DATA.trueLiteral, true).toUpperCase();
                localeData.falseLiteral = Utils.getStringOption(result, 'falseLiteral', DEFAULT_LOCALE_DATA.falseLiteral, true).toUpperCase();
                localeData.engineVersion = Utils.getStringOption(result, 'engineVersion');
                postProcessLocaleData(localeData);
                Utils.info('SpreadsheetApplication.preProcessDocument(): new locale data:', localeData);

                // check that the translated function names exist in the function help
                functionHelpPromise.done(function (data) {
                    var supportedFuncs = _(localeData.functions).values(),
                        annotatedFuncs = _(data).keys(),
                        missingFuncs = _(supportedFuncs).difference(annotatedFuncs),
                        unknownFuncs = _(annotatedFuncs).difference(supportedFuncs);
                    if (missingFuncs.length > 0) {
                        missingFuncs.sort();
                        Utils.warn('SpreadsheetApplication.preProcessDocument(): missing help texts for functions: ' + missingFuncs.join(', '));
                    }
                    if (unknownFuncs.length > 0) {
                        unknownFuncs.sort();
                        Utils.info('SpreadsheetApplication.preProcessDocument(): help available for unknown functions: ' + unknownFuncs.join(', '));
                    }
                });

                // cache complete locale data object in the global cache
                localeDataCache[Utils.LOCALE] = localeData;

            }, function () {
                Utils.error('SpreadsheetApplication.preProcessDocument(): could not download locale data for "' + Utils.LOCALE + '"');
                return { headline: gt('Server Error'), message: gt('The engine used for calculating formulas is not available.') };
            });
        }

        /**
         * Post-processing of the document, after all its import operations
         * have been applied successfully.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  document has been post-processed successfully; or rejected when the
         *  document is invalid, or an error has occurred.
         */
        function postProcessDocument() {
            // document must contain at least one sheet (returning a rejected Deferred
            // object will cause invocation of the prepareInvalidDocument() method)
            return (model.getSheetCount() === 0) ? $.Deferred().reject() : model.postProcessDocument();
        }

        /**
         * Will be called when importing the document fails for any reason.
         */
        function prepareInvalidDocument() {
            // document must contain at least one sheet, to prevent if-else
            // statements for empty documents at thousand places in the view code
            model.prepareInvalidDocument();
        }

        /**
         * Handler will be called by base class if importing the document
         * failed. The functions handles Spreadsheet specific errors.
         */
        function importFailedHandler(response) {

            var // specific error code sent by the server
                error = Utils.getStringOption(response, 'error');

            switch (error) {
            case 'LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR':
                response.message = gt('This document exceeds the spreadsheet size and complexity limits.');
                break;
            case 'LOADDOCUMENT_CALCENGINE_NOT_WORKING_ERROR':
                response.headline = gt('Server Error');
                response.message = gt('The engine used for calculating formulas is not available.');
                break;
            }

            if (!self.isInQuit()) {
                prepareInvalidDocument();
            }
        }

        /**
         * Preprocessing before the document will be flushed for downloading,
         * printing, sending as mail, or closing.
         */
        function prepareFlushDocument() {
            // commit current text of the cell in-place edit mode
            view.leaveCellEditMode('cell');
            // send all changed view settings before flushing editable document
            view.sendChangedViewAttributes();
        }

        /**
         * Preparations before the edit mode will be switched off.
         */
        function prepareLoseEditRights() {
            // commit current text of the cell in-place edit mode
            view.leaveCellEditMode('cell');
        }

        /**
         * Handles 'docs:update' notifications. If the message data contains
         * information about changed cells in the document, this method
         * triggers a 'docs:update:cells' event with preprocessed user data.
         *
         * @param {Object} data
         *  The message data of the 'docs:update' application event.
         */
        function updateNotificationHandler(data) {

            var // complete changed data
                changedData = Utils.getObjectOption(data, 'changed'),
                // the locale used to format the cell contents
                locale = Utils.getStringOption(data, 'locale'),
                // all changed cell information for the own 'docs:update:cells' event
                eventData = null,
                // all changed range addresses, with sheet index
                allChangedRanges = null;

            // update message may not contain any data about changed cells
            if (!_.isObject(changedData)) { return; }
            Utils.info('SpreadsheetApplication.updateNotificationHandler(): notification received:', changedData);

            // extract the map of changed data per sheet
            changedData = Utils.getObjectOption(changedData, 'sheets');
            if (!changedData) {
                Utils.warn('SpreadsheetApplication.updateNotificationHandler(): no sheets defined');
                return;
            }

            // process all changed sheets
            eventData = {};
            allChangedRanges = [];
            _(changedData).each(function (changedSheetData, sheetKey) {

                var sheet = parseInt(sheetKey, 10),
                    rangeContents = Utils.getArrayOption(changedSheetData, 'contents', []),
                    dirtyRanges = Utils.getArrayOption(changedSheetData, 'cells', []),
                    colIntervals = Utils.getArrayOption(changedSheetData, 'columns'),
                    rowIntervals = Utils.getArrayOption(changedSheetData, 'rows'),
                    usedCols = Utils.getIntegerOption(changedSheetData, 'usedCols'),
                    usedRows = Utils.getIntegerOption(changedSheetData, 'usedRows'),
                    eventSheetData = {},
                    changedRanges = null;

                if (!isFinite(sheet) || (sheet < 0) || (sheet >= model.getSheetCount())) {
                    Utils.warn('SpreadsheetApplication.updateNotificationHandler(): invalid sheet index: "' + sheetKey + '"');
                    return;
                }

                // insert cell contents
                if (_.isArray(rangeContents) && (rangeContents.length > 0)) {
                    _(rangeContents).each(function (rangeData) {
                        if (!rangeData.end) { rangeData.end = _.clone(rangeData.start); }
                    });
                    // bug 32624: use content data only if formatted with the own locale; otherwise store as 'dirty ranges'
                    if (locale === Utils.LOCALE) {
                        eventSheetData.rangeContents = rangeContents;
                    } else {
                        dirtyRanges = dirtyRanges.concat(rangeContents);
                    }
                }

                // insert addresses of dirty ranges
                _(dirtyRanges).each(function (range) {
                    if (!range.end) { range.end = _.clone(range.start); }
                });
                _(colIntervals).each(function (interval) {
                    dirtyRanges.push(model.makeColRange(('last' in interval) ? interval : interval.first));
                });
                _(rowIntervals).each(function (interval) {
                    dirtyRanges.push(model.makeRowRange(('last' in interval) ? interval : interval.first));
                });
                if (dirtyRanges.length > 0) {
                    eventSheetData.dirtyRanges = SheetUtils.getUnifiedRanges(dirtyRanges);
                }

                // insert number of used columns/rows
                if (_.isNumber(usedCols)) { eventSheetData.usedCols = usedCols; }
                if (_.isNumber(usedRows)) { eventSheetData.usedRows = usedRows; }

                // insert sheet entry into the own event data
                if (!_.isEmpty(eventSheetData)) {
                    eventSheetData.sheet = sheet;
                    eventData[sheetKey] = eventSheetData;
                }

                // extend array of all changed ranges
                if (eventSheetData.rangeContents && eventSheetData.dirtyRanges) {
                    changedRanges = SheetUtils.getUnifiedRanges(eventSheetData.rangeContents.concat(eventSheetData.dirtyRanges));
                } else if (eventSheetData.rangeContents) {
                    changedRanges = SheetUtils.getUnifiedRanges(eventSheetData.rangeContents);
                } else if (eventSheetData.dirtyRanges) {
                    changedRanges = _.copy(dirtyRanges, true);
                }
                _(changedRanges).each(function (range) { range.sheet = sheet; });
                allChangedRanges = allChangedRanges.concat(changedRanges);
            });

            // trigger the event, if the event data object contains any useful information
            if (!_.isEmpty(eventData)) {
                self.trigger('docs:update:cells', eventData, allChangedRanges);
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the sorted list with the names of all supported functions
         * used in formulas, translated for the current UI language.
         *
         * @returns {Array}
         *  The sorted list with the names of all supported functions.
         */
        this.getLocalizedFunctionNames = function () {
            return localeData.localizedFunctionNames;
        };

        /**
         * Returns the name of the specified function, translated for the
         * current UI language.
         *
         * @param {String} nativeFuncName
         *  The English name of the function.
         *
         * @returns {String}
         *  The translated name of the specified function if available,
         *  otherwise the passed function name.
         */
        this.getLocalizedFunctionName = function (nativeFuncName) {
            nativeFuncName = nativeFuncName.toUpperCase();
            return (nativeFuncName in localeData.functions) ? localeData.functions[nativeFuncName] : nativeFuncName;
        };

        /**
         * Returns the sorted list with the native (English) names of all
         * supported functions used in formulas.
         *
         * @returns {Array}
         *  The sorted list with the native names of all supported functions.
         */
        this.getNativeFunctionNames = function () {
            return localeData.nativeFunctionNames;
        };

        /**
         * Returns the native (English) name of the specified function name in
         * the current UI language.
         *
         * @param {String} localFuncName
         *  The translated name of the function.
         *
         * @returns {String}
         *  The native name of the specified function if available, otherwise
         *  the passed function name.
         */
        this.getNativeFunctionName = function (localFuncName) {
            localFuncName = localFuncName.toUpperCase();
            return (localFuncName in localeData.invFunctions) ? localeData.invFunctions[localFuncName] : localFuncName;
        };

        /**
         * Returns a help text descriptor for the specified translated function
         * name.
         *
         * @param {String} localFuncName
         *  The translated name of the function.
         *
         * @returns {Object|Null}
         *  A help text descriptor if available, with the following properties:
         *  - {String} description
         *      A text describing the entire formula.
         *  - {String[]} params
         *      An array with the names of all function parameters. Functions
         *      with trailing repeating parameters will be mentioned only once
         *      in this array.
         *  - {String[]} paramshelp
         *      An array describing each parameter more detailed. This array
         *      MUST have the same length as the 'params' array.
         *  - {Number} [varargs]
         *      If contained, the zero-based index of the first parameter that
         *      can be used repeatedly. MUST be less than the length of the
         *      'params' array. If this index does not point to the last array
         *      element, the function expects pairs or even larger tuples of
         *      repeating parameters.
         */
        this.getFunctionHelp = function (localFuncName) {
            localFuncName = localFuncName.toUpperCase();
            return (localFuncName in functionHelp) ? functionHelp[localFuncName] : null;
        };

        /**
         * Returns the sorted list with the names of all supported error codes
         * used in formulas, translated for the current UI language.
         *
         * @returns {Array}
         *  The sorted list with the names of all supported error codes.
         */
        this.getLocalizedErrorCodes = function () {
            return localeData.localizedErrorCodes;
        };

        /**
         * Returns the name of the specified error code, translated for the
         * current UI language.
         *
         * @param {String} nativeErrorCode
         *  The English name of the error code, e.g. '#REF!', '#NUM!' etc.
         *
         * @returns {String}
         *  The translated name of the specified error code if available,
         *  otherwise the unmodified passed string.
         */
        this.getLocalizedErrorCode = function (nativeErrorCode) {
            nativeErrorCode = nativeErrorCode.toUpperCase();
            return (nativeErrorCode in localeData.errorCodes) ? localeData.errorCodes[nativeErrorCode] : nativeErrorCode;
        };

        /**
         * Returns the sorted list with the internal values of all supported
         * error codes used in formulas.
         *
         * @returns {Array}
         *  The sorted list with the internal values of all supported error
         *  codes.
         */
        this.getNativeErrorCodes = function () {
            return localeData.nativeErrorCodes;
        };

        /**
         * Returns the internal value of the specified translated error code
         * literal.
         *
         * @param {String} localErrorCode
         *  The translated name of the error code.
         *
         * @returns {String}
         *  The internal English value of the specified error code if
         *  available, otherwise the unmodified passed string.
         */
        this.getNativeErrorCode = function (localErrorCode) {
            localErrorCode = localErrorCode.toUpperCase();
            return (localErrorCode in localeData.invErrorCodes) ? localeData.invErrorCodes[localErrorCode] : localErrorCode;
        };

        /**
         * Returns the decimal separator for the current UI language.
         *
         * @returns {String}
         *  The decimal separator for the current UI language.
         */
        this.getDecimalSeparator = function () {
            return localeData.decSeparator;
        };

        /**
         * Returns the group separator (usually the character that separates
         * groups of 3 digits in the integral part of a number, but different
         * rules may apply in specific locales) for the current UI language.
         *
         * @returns {String}
         *  The group separator for the current UI language.
         */
        this.getGroupSeparator = function () {
            return localeData.groupSeparator;
        };

        /**
         * Returns the list separator for the current UI language. The list
         * separator is used to separate parameters of functions, and for the
         * range list operator.
         *
         * @returns {String}
         *  The list separator for the current UI language.
         */
        this.getListSeparator = function () {
            return localeData.listSeparator;
        };

        /**
         * Returns the string literal of the specified Boolean value for the
         * current UI language.
         *
         * @param {Boolean} value
         *  The Boolean value.
         *
         * @returns {String}
         *  The string literal of the specified Boolean value.
         */
        this.getBooleanLiteral = function (value) {
            return localeData[value ? 'trueLiteral' : 'falseLiteral'];
        };

        /**
         * Returns the version of the server-side Calc Engine.
         *
         * @returns {String}
         *  The version of the Calc Engine.
         */
        this.getEngineVersion = function () {
            return localeData.engineVersion;
        };

        /**
         * Queries the contents of cells referred by the specified cell ranges.
         * A single server request will be sent debounced after collecting
         * multiple queries.
         *
         * @param {Array} ranges
         *  An array of arrays (!) with logical range addresses. Each inner
         *  array of ranges will be resolved to a single list of cell contents.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.attributes=false]
         *      If set to true, the result will contain formatting attributes
         *      of all cells.
         *  @param {Number} [options.maxCount=1000]
         *      The maximum number of cell values allowed per result list
         *      (originating from an inner array of range addresses, see
         *      parameter 'ranges' above).
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  cell contents referred by the specified ranges. The result will be
         *  an array with the same number of elements as range lists are
         *  contained in the parameter 'ranges', and in the same order. Each
         *  array element is an array of cell content objects, each object will
         *  contain the following properties:
         *  - {String} display
         *      The display string of the cell result.
         *  - {Number|String|Boolean|Null} result
         *      The typed result value. Error codes are represented by strings
         *      starting with the hash character.
         *  - {Object} [attrs]
         *      The explicit formatting attributes of the cell. Will only be
         *      set, if the option 'attributes' has been set (see above), and
         *      the cell contains at least one explicit formatting attribute.
         */
        this.queryCellResults = (function () {

            var // all pending queries (one element per call of this method)
                pendingQueries = [],
                // all range lists to be actually queried from the server
                queryDescriptors = [],
                // maps range list names to array indexes in 'queryDescriptors' to prevent duplicates
                queryDescriptorIndexMap = {};

            // returns the name of the passed range with sheet index
            function getRangeName(range) {
                return range.sheet + '!' + SheetUtils.getCellName(range.start) + ':' + SheetUtils.getCellName(range.end || range.start);
            }

            // returns the names of the passed range list with sheet indexes
            function getRangesName(ranges) {
                return _(ranges).map(getRangeName).join(',');
            }

            // registers a new source data query
            function registerQuery(ranges, options) {

                var // whether to query for formatting attributes
                    attrs = Utils.getBooleanOption(options, 'attributes', false),
                    // the entry in the 'pendingQueries' array for this invocation
                    pendingQuery = {
                        def: $.Deferred(),
                        attrs: attrs,
                        maxCount: Utils.getIntegerOption(options, 'maxCount', 1000, 1, 1000),
                        indexes: []
                    };

                // add all range lists to 'queryDescriptors' and update 'queryDescriptorIndexMap'
                Utils.info('SpreadsheetApplication.queryCellResults(): registering ranges:', '[' + _(ranges).map(getRangesName).join('] [') + ']');
                _(ranges).each(function (rangeList) {

                    var // string representation of the range list, used as map key
                        rangesKey = getRangesName(rangeList),
                        // the array index of query descriptors already registered before
                        queryDescIndex = queryDescriptorIndexMap[rangesKey];

                    if (_.isNumber(queryDescIndex)) {
                        // range list exists already: update 'attrs' flag in the query descriptor (the exact same
                        // ranges with and without 'attrs' flag will be fetched with a single query with attributes)
                        if (attrs) { queryDescriptors[queryDescIndex].attrs = true; }
                    } else {
                        // range list does not exist yet: create a new entry
                        queryDescriptorIndexMap[rangesKey] = queryDescIndex = queryDescriptors.length;
                        queryDescriptors.push({ ranges: _.copy(rangeList, true), attrs: attrs });
                    }

                    // store array index of the query descriptor in the pending query data
                    pendingQuery.indexes.push(queryDescIndex);
                });

                // store the new query entry, and return a promise that will be resolved with the cell contents
                pendingQueries.push(pendingQuery);

                // create a promise that will be aborted automatically when application is closed
                return self.createAbortablePromise(pendingQuery.def);
            }

            // sends a single server request for all cached source data queries
            function executeQuery() {

                var // local copy of the queries
                    localQueries = pendingQueries,
                    // the request data to be sent to the server
                    requestData = { type: 'sourceData', locale: Utils.LOCALE, queries: queryDescriptors },
                    // the server request
                    request = null;

                // reset cache variables (collect new queries while server request is running)
                pendingQueries = [];
                queryDescriptors = [];
                queryDescriptorIndexMap = {};

                // do nothing if no requests available, or the application shuts down or is in error state
                if ((localQueries.length === 0) || self.isInQuit() || (self.getState() === 'error')) { return; }

                // send the server request
                Utils.info('SpreadsheetApplication.queryCellResults(): sending server request:', requestData);
                request = self.sendFilterRequest({
                    method: 'POST',
                    params: { action: 'query', requestdata: JSON.stringify(requestData) },
                    resultFilter: function (data) {
                        Utils.info('SpreadsheetApplication.queryCellResults(): server response received:', data);
                        var contents = Utils.getArrayOption(data, 'contents');
                        return (_.isArray(contents) && (contents.length === requestData.queries.length)) ? contents : undefined;
                    }
                });

                // process response data on success
                request.done(function (allContents) {

                    // resolve each registered query with the correct result arrays
                    _(localQueries).each(function (localQuery) {

                        var // the complete result (array of arrays) to be passed to the Deferred object of this query
                            queryResults = [];

                        // check whether the request has been aborted already
                        if (localQuery.def.state() !== 'pending') { return; }

                        // localQuery.indexes contains the array indexes into the 'allContents' array
                        _(localQuery.indexes).each(function (index) {

                            var // the unpacked result array for a single range list
                                result = [];

                            // unpack a result array from the server result into 'result'
                            _(allContents[index]).any(function (cellContents) {

                                var // repetition count of the current cell entry
                                    repeat = Math.min(Utils.getIntegerOption(cellContents, 'repeat', 1), localQuery.maxCount - result.length),
                                    // the cell entry data to be returned
                                    entry = {
                                        display: _.isString(cellContents.display) ? cellContents.display : '',
                                        result: _.isUndefined(cellContents.result) ? null : cellContents.result
                                    };

                                // only add formatting attributes if the original query has asked for them
                                if (localQuery.attrs && ('attrs' in cellContents)) {
                                    entry.attrs = cellContents.attrs;
                                }

                                // clone equal cell entry to the specified number
                                result.push(entry);
                                _(repeat - 1).times(function () { result.push(_.copy(entry, true)); });

                                // break the _.any() loop if the maximum number of entries has been reached
                                return result.length === localQuery.maxCount;
                            });

                            queryResults.push(result);
                        });

                        // resolve the Deferred object of the current query
                        localQuery.def.resolve(queryResults);
                    });
                });

                // reject all Deferred objects on error
                request.fail(function (response) {
                    Utils.warn('SpreadsheetApplication.queryCellResults(): server request failed:', response);
                    _(localQueries).each(function (query) {
                        query.def.reject(response);
                    });
                });
            }

            // return the actual debounced queryCellResults() method
            return self.createDebouncedMethod(registerQuery, executeQuery);
        }());

        // initialization -----------------------------------------------------

        // initialization of class members
        this.on('docs:init', function () {
            model = self.getModel();
            view = self.getView();
            // initialize default locale data
            postProcessLocaleData(localeData);
        });

        // application notifies changed contents/results of cells create an own 'docs:update:cells' event
        this.on('docs:update', updateNotificationHandler);

        // destroy all class members
        this.on('docs:destroy', function () {
            appOptions = launchOptions = null;
            model = view = localeData = functionHelp = null;
        });

    }}); // class SpreadsheetApplication

    // static methods ---------------------------------------------------------

    /**
     * Replacement for the generic method EditApplication.createLauncher()
     * without parameters, to launch spreadsheet applications.
     */
    SpreadsheetApplication.createLauncher = function () {
        return EditApplication.createLauncher('io.ox/office/spreadsheet', SpreadsheetApplication, { icon: 'fa-table', search: true });
    };

    // exports ================================================================

    return SpreadsheetApplication;

});
