/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/preview/view/controls',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/view/basecontrols',
     'gettext!io.ox/office/preview'
    ], function (Utils, BaseControls, gt) {

    'use strict';

    var // class name shortcuts
        Label = BaseControls.Label,
        Button = BaseControls.Button,
        TextField = BaseControls.TextField,
        RadioList = BaseControls.RadioList;

    // static class PreviewControls ===========================================

    /**
     * Additional classes defining specialized GUI controls for the OX Viewer
     * application.
     *
     * @extends BaseControls
     */
    var PreviewControls = _.clone(BaseControls);

    // constants --------------------------------------------------------------

    /**
     * Standard options for the 'Previous page' button.
     *
     * @constant
     */
    PreviewControls.PREV_OPTIONS = { icon: 'fa-arrow-up', tooltip: gt('Show previous page') };

    /**
     * Standard options for the 'Next page' button.
     *
     * @constant
     */
    PreviewControls.NEXT_OPTIONS = { icon: 'fa-arrow-down', tooltip: gt('Show next page') };

    // class EditDocumentButton ===============================================

    /**
     * The button to start editing the current document.
     *
     * @constructor
     *
     * @extends Button
     */
    PreviewControls.EditDocumentButton = Button.extend({ constructor: function () {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        Button.call(this, { icon: 'fa-pencil', tooltip: gt('Edit document') });

        // initialization -----------------------------------------------------

        // hide the button completely, if it is disabled
        this.on('group:enable', function (event, state) { self.toggle(state); });

    }}); // class EditDocumentButton

    // class PagePicker =======================================================

    /**
     * The drop-down list for additional options to select a page.
     *
     * @constructor
     *
     * @extends RadioList
     */
    PreviewControls.PagePicker = BaseControls.PopupMenuButton.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        BaseControls.PopupMenuButton.call(this, app, 'previewpages', Utils.extendOptions({
            icon: 'fa-ellipsis-v',
            tooltip: gt('Select page'),
            caret: false
        }, initOptions));

        // initialization -----------------------------------------------------

        this.getMenuNode().addClass('app-preview page-picker');

        // create the text input field for the page number, when page count is known after import
        app.on('docs:import:success', function () {
            var pageCount = app.getModel().getPageCount();
            // create the first/last list entries
            self.addGroup('pages/first', new Button({ label: gt('Show first page') }))
                .addGroup('pages/last', new Button({ label: gt('Show last page') }))
                .addSeparator()
                .addGroup('pages/current', new Label({ label: gt('Go to page') }))
                .addGroup('pages/current', new TextField({ tooltip: gt('Page number'), width: 45, appendInline: true, css: { textAlign: 'right' }, keyboard: 'number', validator: new TextField.NumberValidator({ min: 1, max: pageCount }) }));
//                .addGroup('pages/current', new SpinField({ tooltip: gt('Page number'), width: 45, appendInline: true, css: { textAlign: 'right' }, keyboard: 'number', min: 1, max: pageCount }));
        });

    }}); // class PagePicker

    // class ZoomTypePicker ===================================================

    /**
     * The selector for additional zoom types.
     *
     * @constructor
     *
     * @extends RadioList
     */
    PreviewControls.ZoomTypePicker = RadioList.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            icon: 'fa-ellipsis-v',
            tooltip: gt('More zoom settings'),
            caret: false,
            updateCaptionMode: 'none'
        }, initOptions));

        // initialization -----------------------------------------------------

        this.createMenuSection('fix')
            .createOptionButton('fix',  50, { label: gt('50%') })
            .createOptionButton('fix',  75, { label: gt('75%') })
            .createOptionButton('fix', 100, { label: gt('100%') })
            .createOptionButton('fix', 150, { label: gt('150%') })
            .createOptionButton('fix', 200, { label: gt('200%') })
            .createMenuSection('fit')
            .createOptionButton('fit', 'width', { label: gt('Fit to screen width') })
            .createOptionButton('fit', 'page',  { label: gt('Fit to screen size') });

    }}); // class ZoomTypePicker

    // exports ================================================================

    return PreviewControls;

});
