/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/editview',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/baseframework/view/baseview',
     'io.ox/office/baseframework/view/pane',
     'io.ox/office/baseframework/view/toolbox',
     'io.ox/office/editframework/view/editsidepane',
     'io.ox/office/editframework/view/editcontrols',
     'io.ox/office/editframework/app/rtconnection',
     'gettext!io.ox/office/editframework',
     'less!io.ox/office/editframework/view/editstyle'
    ], function (Utils, KeyCodes, BaseView, Pane, ToolBox, EditSidePane, EditControls, RTConnection, gt) {

    'use strict';

    var // class name shortcuts
        Button = EditControls.Button;

    // class EditView =========================================================

    /**
     * @constructor
     *
     * @extends BaseView
     *
     * @param {EditApplication} app
     *  The application containing this view instance.
     *
     * @param {Object} [initOptions]
     *  Additional options to control the appearance of the view. Supports all
     *  options that are supported by the base class BaseView.
     */
    function EditView(app, initOptions) {

        var // self reference
            self = this,

            // the main side pane
            sidePane = null,

            // the debug pane
            debugPane = null,

            // tool pane floating over the top of the application pane
            overlayPane = null,

            // tool box in the upper overlay pane
            overlayToolBox = null,

            // operations node
            operationsNode = null,

            // the clipboard debug pane
            clipboardPane = null,

            // the data node of the clipboard debug pane
            clipboardDataNode = null;

        // base constructor ---------------------------------------------------

        BaseView.call(this, app, Utils.extendOptions(initOptions, { initHandler: initHandler, deferredInitHandler: deferredInitHandler }));

        // private methods ----------------------------------------------------

        /**
         * Creates and initializes the debug pane.
         */
        function initDebugMode() {

            var // number of operations already logged
                operationsCount = 0,
                // operations not yet logged
                pendingOperations = [],
                // background loop logging all pending operations
                logOperationsTimer = null,
                // the debug info table
                infoNode = $('<table>').css('table-layout', 'fixed').append($('<colgroup>').append($('<col>', { width: '40px' }), $('<col>'))),
                // the drop-down menu with debug actions
                actionMenu = null;

            // initialize the operations log
            operationsNode = $('<table>');

            // scrolls the operations output down to its end
            function scrollDownOperationsNode() {
                var scrollNode = operationsNode.parent();
                scrollNode.scrollTop(scrollNode[0].scrollHeight);
            }

            // direct callback for logOperations(): called every time when logOperations() is called
            function registerOperations(event, operations, external) {
                // append the new operations to the pending operations queue
                pendingOperations = pendingOperations.concat(_(operations).map(function (operation) {
                    if (app.isPrepareLosingEditRights()) {
                        RTConnection.log('registerOperation: ' + JSON.stringify(operation));
                    }
                    return { operation: operation, external: external };
                }));
            }

            // deferred callback for logOperations(): called once after current script execution ends
            function printSomeOperations() {

                // check if the background loop is already running
                if (logOperationsTimer) { return; }

                // create a new background loop that processes all pending operations
                logOperationsTimer = app.repeatDelayed(function () {

                    var // the operations to insert into the table
                        operations = pendingOperations.splice(0, 20),
                        // the number of operations applied successfully by the document model
                        successCount = app.getModel().getOperationsCount(),
                        // checking the operation length, so that operations removed from the
                        // 'optimizeOperationHandler' will not be displayed in the debug pane.
                        oplCounter = 1;

                    // abort the background loop, if all operations have been logged
                    if (operations.length === 0) { return Utils.BREAK; }

                    // log the next operations chunk
                    _(operations).each(function (entry) {

                        var operation = _.clone(entry.operation),
                            name = operation.name,
                            osn = ('osn' in operation) ? operation.osn : '',
                            opl = ('opl' in operation) ? operation.opl : '',
                            // skipping operations, that were 'optimized' in 'sendActions' in the
                            // optimizeOperationHandler. Otherwise the debug pane will be irritating, because
                            // 'merged operations' are followed by the operations, that will not be sent to
                            // the server, because they were removed by the 'optimizeOperationHandler'.
                            skipOperation = false;

                        // checking if this operation was merged by the 'optimizeOperationHandler'. This can
                        // only happen for internal operations (task 29601)
                        if (! entry.external) {
                            if (_.isNumber(opl) && (opl > 1)) {
                                oplCounter = opl;
                            } else if (oplCounter > 1) {
                                oplCounter--;
                                skipOperation = true;
                            }
                        }

                        if (! skipOperation) {
                            delete operation.name;
                            delete operation.osn;
                            delete operation.opl;
                            operation = JSON.stringify(operation).replace(/^\{(.*)\}$/, '$1').replace(/"(\w+)":/g, '$1:').replace(/ /g, '\xb7');
                            operationsCount += 1;
                            operationsNode.append($('<tr>')
                                .toggleClass('error', successCount < operationsCount)
                                .append(
                                    $('<td>').text(_.noI18n(operationsCount)),
                                    $('<td>').text(_.noI18n(entry.external ? 'E' : 'I')).attr('title', _.noI18n((entry.external ? 'Ex' : 'In') + 'ternal operation')),
                                    $('<td>').text(_.noI18n(osn)).css('text-align', 'right'),
                                    $('<td>').text(_.noI18n(opl)).css('text-align', 'right'),
                                    $('<td>').text(_.noI18n(name)),
                                    $('<td>').text(_.noI18n(operation))
                                ));
                        }
                    });
                    scrollDownOperationsNode();

                }, { delay: 50 });

                // forget reference to the timer, when all operations have been logged
                logOperationsTimer.always(function () { logOperationsTimer = null; });
            }

            // extend public interface
            self.addDebugInfoHeader = function (header, options) {
                var tooltip = Utils.getStringOption(options, 'tooltip', '');
                infoNode.append($('<tr>').attr({ 'data-header': header, title: _.noI18n(tooltip) }).append($('<th>', { colspan: 2 }).text(_.noI18n(Utils.capitalizeWords(header)))));
                return this;
            };

            self.addDebugInfoNode = function (label, options) {
                var header = Utils.getStringOption(options, 'header', infoNode.find('tr').last().attr('data-header')),
                    tooltip = Utils.getStringOption(options, 'tooltip', ''),
                    node = $('<tr>').attr({ 'data-header': header, title: _.noI18n(tooltip) }).append($('<td>').text(_.noI18n(label)), $('<td>').attr('data-id', label));
                node.insertAfter(infoNode.find('tr[data-header="' + header + '"]').last());
                return this;
            };

            self.setDebugInfoText = function (header, label, text) {
                infoNode.find('tr[data-header="' + header + '"]>td[data-id="' + label + '"]').text(_.noI18n(text));
                return this;
            };

            self.createDebugToolBox = function () {

                // create the drop-down list for the debug actions
                actionMenu = new EditControls.PopupMenuButton(app, 'debugactions', { icon: 'fa-wrench', label: _.noI18n('Actions'), tooltip: _.noI18n('Additional debug actions') });

                // register realtime debug actions
                actionMenu
                    .addGroup('debug/realtime/trigger', new Button({ icon: 'fa-unlink',      label: _.noI18n('Switch to offline mode'),      value: 'offline' }))
                    .addGroup('debug/realtime/trigger', new Button({ icon: 'fa-link',        label: _.noI18n('Switch to online mode'),       value: 'online' }))
                    .addSeparator()
                    .addGroup('debug/realtime/trigger', new Button({ icon: 'fa-exclamation', label: _.noI18n('Simulate connection reset'),   value: 'reset' }))
                    .addGroup('debug/realtime/trigger', new Button({ icon: 'fa-clock-o',     label: _.noI18n('Simulate connection timeout'), value: 'timeout' }))
                    .addGroup('debug/realtime/trigger', new Button({ icon: 'fa-refresh',     label: _.noI18n('Simulate connection hangup'),  value: 'hangup' }));

                // register debug actions for operations
                actionMenu.addSeparator()
                    .addGroup('debug/realtime/sendinvalidoperation', new Button({ icon: 'fa-exclamation',  label: _.noI18n('Apply operation with invalid OSN') }))
                    .addGroup('debug/operations/apply',              new Button({ icon: 'fa-minus-circle', label: _.noI18n('Apply operation with invalid name'), value: { name: '_invalid_operation_' } }));

                // register debug actions for application quit
                actionMenu.addSeparator()
                    .addGroup('debug/quit', new Button({ icon: 'fa-warning',      label: _.noI18n('Close document with unsaved changes'), value: 'unsaved' }))
                    .addGroup('debug/quit', new Button({ icon: 'fa-times-circle', label: _.noI18n('Close document with server error'),    value: 'error' }));

                // create and return the debug tool box
                return self.createToolBox('debug', { label: _.noI18n('Debug'), visible: 'debug/enabled' })
                    .addGroup('debug/toggle',    new Button({ icon: 'fa-bug', tooltip: _.noI18n('Toggle debug panel'),     toggle: true }))
                    .addGroup('debug/highlight', new Button({ icon: 'fa-eye', tooltip: _.noI18n('Highlight DOM elements'), toggle: true }))
                    .addGroup('debug/clipboard', new Button({ icon: 'fa-calendar-o', tooltip: _.noI18n('Toggle clipboard debug panel'), toggle: true }))
                    .addRightTab()
                    .addPrivateGroup(actionMenu)
                    .newLine();
            };

            self.getDebugActionMenu = function () {
                return actionMenu;
            };

            self.isDebugPaneVisible = function () {
                return debugPane.isVisible();
            };

            self.toggleDebugPane = function (state) {
                debugPane.toggle(state);
                if (state) { scrollDownOperationsNode(); }
                return this;
            };

            self.isDebugHighlight = function () {
                return app.getWindowNode().hasClass('debug-highlight');
            };

            self.toggleDebugHighlight = function (state) {
                app.getWindowNode().toggleClass('debug-highlight', state);
                return this;
            };

            // create the debug pane
            debugPane = new Pane(app, 'debug', { position: 'bottom', classes: 'debug user-select-text' });

            // add the operations table and the info table to the debug pane
            self.addPane(debugPane.hide());
            debugPane.hide().getNode().append(
                $('<table>').append(
                    $('<colgroup>').append($('<col>', { width: '70%' })),
                    $('<tr>').append(
                        $('<td>').append($('<div>', { tabindex: 0 }).append(operationsNode)),
                        $('<td>').append($('<div>', { tabindex: 0 }).append(infoNode))
                    )
                )
            );

            // create the output nodes in the debug pane
            self.addDebugInfoHeader('application', { tooltip: 'Global application state' })
                .addDebugInfoNode('state', { tooltip: 'Application state' })
                .addDebugInfoNode('osn', { tooltip: 'Operation state number' });

            // log the application state
            self.listenTo(app, 'docs:state', function (state) {
                self.setDebugInfoText('application', 'state', state);
            });

            // log the operation state number of the document
            self.listenTo(app.getModel(), 'operations:after', function () {
                self.setDebugInfoText('application', 'osn', app.getModel().getOperationStateNumber());
            });

            // log all applied operations
            operationsNode.append($('<tr>').append(
                $('<th>'),
                $('<th>').text(_.noI18n('T')).attr('title', _.noI18n('Operation type')),
                $('<th>').text(_.noI18n('OSN')).attr('title', _.noI18n('Operation State Number')).css('font-size', '70%'),
                $('<th>').text(_.noI18n('OPL')).attr('title', _.noI18n('Operation Length')).css('font-size', '70%'),
                $('<th>').text(_.noI18n('Name')),
                $('<th>').text(_.noI18n('Parameters'))));

            // simply collect all operations while importing the document
            self.listenTo(app.getModel(), 'operations:after', registerOperations);

            // log operations after importing the document
            app.on('docs:import:after', function () {
                // create the debounced logOperations() method
                var logOperations = app.createDebouncedMethod(registerOperations, printSomeOperations);
                // reconnect the 'operations:after' event to the debounced method
                app.getModel().off('operations:after', registerOperations);
                self.listenTo(app.getModel(), 'operations:after', logOperations);
                // trigger deferred logging by calling 'logOperations' once without new operations
                logOperations({}, []);
            });

            // keyboard handling for debug pane
            self.listenTo(operationsNode.add(infoNode).parent(), 'keydown', function (event) {
                if (KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                    var docRange = window.document.createRange();
                    docRange.setStart(event.delegateTarget, 0);
                    docRange.setEnd(event.delegateTarget, 1);
                    window.getSelection().removeAllRanges();
                    window.getSelection().addRange(docRange);
                    return false;
                }
                if (event.keyCode === KeyCodes.ESCAPE) {
                    self.grabFocus();
                    return false;
                }
            });


            // create the clipboard debug pane
            clipboardPane = new Pane(app, 'clipboard-debug', { position: 'bottom', classes: 'debug user-select-text' });

            // init clipboard data node
            clipboardDataNode = $('<div>', { tabindex: 0 }).addClass('user-select-text').css('line-height', 1).append(
                    $('<span>').text('Clipboard debug window')
            );

            // add the clipboard debug pane and the data node
            self.addPane(clipboardPane.hide());
            clipboardPane.hide().getNode().append(
                    $('<table>').append(
                            $('<colgroup>').append($('<col>', { width: '100%' })),
                            $('<tr>').append(
                                    $('<td>').append(clipboardDataNode)
                            )
                    )
            );

            // keyboard handling for the clipboard debug pane
            clipboardDataNode.on('keydown', function (event) {
                if (KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                    var docRange = window.document.createRange();
                    docRange.setStart(event.delegateTarget, 0);
                    docRange.setEnd(event.delegateTarget, 1);
                    window.getSelection().removeAllRanges();
                    window.getSelection().addRange(docRange);
                    return false;
                }
                if (KeyCodes.matchKeyCode(event, 'DELETE')) {
                    clipboardDataNode.empty();
                }
                if (event.keyCode === KeyCodes.ESCAPE) {
                    self.grabFocus();
                    return false;
                }
            });

            // public funtions of the clipboard debug pane
            self.isClipboardPaneVisible = function () {
                return clipboardPane.isVisible();
            };

            self.toggleClipboardPane = function (state) {
                clipboardPane.toggle(state);
                return this;
            };

            self.setClipboardDebugInfo = function (content) {

                var node;

                function createOperationsTableNode (operations) {
                    var table = $('<table>').append('<tbody>');

                    // add header
                    table.append($('<tr>').append(
                            $('<th>'),
                            $('<th>').text(_.noI18n('Name')),
                            $('<th>').text(_.noI18n('Parameters'))));

                    // add content
                    _.each(operations, function (entry, index) {
                        var operation = _.clone(entry),
                            name = operation.name;

                        delete operation.name;
                        delete operation.osn;
                        delete operation.opl;
                        operation = JSON.stringify(operation).replace(/^\{(.*)\}$/, '$1').replace(/"(\w+)":/g, '$1:').replace(/ /g, '\xb7');

                        table.append($('<tr>').append(
                                $('<td>').text(_.noI18n(index)),
                                $('<td>').text(_.noI18n(name)),
                                $('<td>').text(_.noI18n(operation))
                        ));
                    });
                    return table;
                }

                if ((_.isArray(content)) && (content.length > 0)) {
                    node = createOperationsTableNode(content);
                }
                else if ((content instanceof $) && (content.contents().length > 0)) {
                    node = $('<span>').text(content.html());
                } else if ((_.isString(content)) && (content.length > 0)) {
                    node = $('<span>').text(content);
                } else {
                    node = $('<span>').text('Clipboard contained no data');
                }
                // clear old and set new clipboard content
                clipboardDataNode.empty().append(node);
                return this;
            };

        }

        /**
         * Updates the visibility of control groups in the overlay pane
         * according to the current document edit mode.
         */
        function updateOverlayVisibility() {

            var // whether application is in internal error mode
                internalError = app.getState() === 'error',
                // current document edit mode
                editMode = app.getModel().getEditMode(),
                // all visible group nodes
                visibleGroupNodes = $(),
                // all hidden group nodes
                hiddenGroupNodes = $();

            // select hidden and visible groups according to application state
            overlayToolBox.iterateGroups(function (group) {

                var visibility = ' ' + Utils.getStringOption(group.getOptions(), 'visibility', 'default') + ' ',
                    visible = (internalError ? /\s(always|error)\s/ : app.isLocked() ? /\s(always)\s/ : editMode ? /\s(always|default|editmode)\s/ : /\s(always|default|readonly)\s/).test(visibility);

                if (visible) {
                    visibleGroupNodes = visibleGroupNodes.add(group.getNode());
                } else {
                    hiddenGroupNodes = hiddenGroupNodes.add(group.getNode());
                }
            });

            // hide and show all groups according to the passed edit mode
            visibleGroupNodes.closest('.group-container').removeClass('hidden');
            hiddenGroupNodes.closest('.group-container').addClass('hidden');
        }

        /**
         * Builds generic contents of this edit view instance: the side pane,
         * the overlay panes, and the debug pane. After that, calls the passed
         * callback handler to build the application-specific view elements.
         */
        function initHandler() {

            var // the application status label
                statusLabel = new EditControls.StatusLabel(app);

            // Disable browser table controls after edit mode has changed. Execute
            // delayed to be sure to run after other DOM changes (e.g. changing the
            // content-editable mode will enable the edit handlers again).
            self.listenTo(app.getModel(), 'change:editmode', function () { self.disableBrowserEditControls(); });
            app.on('docs:import:before', function () { self.disableBrowserEditControls(); });

            // create the side pane (initializes itself internally)
            self.addPane(sidePane = new EditSidePane(app));

            // create the overlay pane
            self.addPane(overlayPane = new Pane(app, 'overlaytop', { position: 'top', classes: 'inline right', overlay: true, transparent: true, hoverEffect: true }));
            overlayPane
                .addViewComponent(overlayToolBox = new ToolBox(app, 'overlaymain')
                    .addGroup('view/sidepane', new Button(EditControls.SHOW_SIDEPANE_OPTIONS))
                    .addGap()
                );

            // create the second overlay pane containing the status label
            self.addPane(new Pane(app, 'statuslabel', { position: 'top', classes: 'inline right', overlay: true, transparent: true })
                .addViewComponent(new ToolBox(app, 'statuslabel', { landmark: false }).addPrivateGroup(statusLabel))
            );

            // initialize the status label
            self.listenTo(app, 'docs:state', function (state) {
                switch (state) {
                case 'sending':
                    statusLabel.setValue(gt('Saving changes'), { type: 'warning', delay: 1000 });
                    break;
                case 'ready':
                    statusLabel.setValue(gt('All changes saved'), { type: 'success', fadeOut: true });
                    break;
                case 'error':
                    statusLabel.setValue(null);
                    self.lockPaneLayout(function () {
                        self.toggleSidePane(false);
                        self.toggleDebugPane(true);
                        app.getWindow().setChromeless(true);
                        updateOverlayVisibility();
                    });
                    break;
                default:
                    statusLabel.setValue(null);
                }
            });

            // show/hide controls in overlay pane according to edit mode
            self.listenTo(app.getModel(), 'change:editmode', updateOverlayVisibility);

            // initially, show the side pane if there is enough room
            self.toggleSidePane(window.innerWidth >= 1000);

            // additions for debug mode
            if (Utils.DEBUG) { initDebugMode(); }

            // call initialization handler passed by sub class
            Utils.getFunctionOption(initOptions, 'initHandler', $.noop).call(self);
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         */
        function deferredInitHandler() {

            // call initialization handler passed by sub class
            Utils.getFunctionOption(initOptions, 'deferredInitHandler', $.noop).call(self);

            // add remaining controls to the overlay tool box for edit mode
            overlayToolBox
                .addGap()
                .addGroup('document/acquireedit', new EditControls.AcquireEditButton(app))
                .addGap()
                .addGroup('document/reload', new Button(EditControls.RELOAD_OPTIONS))
                .addGap()
                .addGroup('document/undo', new Button(EditControls.UNDO_OPTIONS))
                .addGroup('document/redo', new Button(EditControls.REDO_OPTIONS))
                .addGap()
                .addGroup('app/quit', new Button(Utils.extendOptions(EditControls.QUIT_OPTIONS, { visibility: 'always' })));

            // initialize visibility of the overlay groups
            updateOverlayVisibility();
        }

        // methods ------------------------------------------------------------

        /**
         * Returns whether at least one undo action is available on the undo
         * stack.
         *
         * @returns {Boolean}
         *  Whether at least one undo action is available on the stack.
         */
        this.isUndoAvailable = function () {
            return app.getModel().getUndoManager().getUndoCount() > 0;
        };

        /**
         * Applies the topmost undo action on the undo stack.
         *
         * @returns {jQuery.Promise}
         *  A Promise that will be resolved after the undo action has been
         *  applied.
         */
        this.undo = function () {
            return app.getModel().getUndoManager().undo(1);
        };

        /**
         * Returns whether at least one redo action is available on the redo
         * stack.
         *
         * @returns {Boolean}
         *  Whether at least one redo action is available on the stack.
         */
        this.isRedoAvailable = function () {
            return app.getModel().getUndoManager().getRedoCount() > 0;
        };

        /**
         * Applies the topmost redo action on the redo stack.
         *
         * @returns {jQuery.Promise}
         *  A Promise that will be resolved after the redo action has been
         *  applied.
         */
        this.redo = function () {
            return app.getModel().getUndoManager().redo(1);
        };

        /**
         * Disables the internal resize handles shown for of tables, drawing
         * objects, etc. provided by the browser edit mode (content-editable
         * elements).
         *
         * @returns {EditView}
         *  A reference to this instance.
         */
        this.disableBrowserEditControls = function () {
            // execute delayed to prevent conflicts with ongoing event handling
            app.executeDelayed(function () {
                try {
                    // disable FireFox table manipulation handlers
                    window.document.execCommand('enableObjectResizing', false, false);
                    window.document.execCommand('enableInlineTableEditing', false, false);
                } catch (ex) {
                }
            });
            return this;
        };

        /**
         * Returns the side pane of this application.
         *
         * @returns {EditSidePane}
         *  The side pane of this application.
         */
        this.getSidePane = function () {
            return sidePane;
        };

        /**
         * Creates a tool box in the side pane, and registers it for automatic
         * visibility handling.
         *
         * @param {String} id
         *  The identifier for the new tool box. Must be unique across all view
         *  components in the application.
         *
         * @param {Object} [options]
         *  A map of options to control the properties of the new tool box.
         *  Supports all options supported by the method
         *  SidePane.createToolBox().
         *
         * @returns {ToolBox}
         *  The new tool box instance.
         */
        this.createToolBox = function (id, options) {
            return sidePane.createToolBox(id, options);
        };

        /**
         * Returns the customizable tool box in the overlay pane.
         *
         * @returns {ToolBox}
         *  The tool box in the overlay pane.
         */
        this.getOverlayToolBox = function () {
            return overlayToolBox;
        };

        /**
         * Returns whether the main side pane is currently visible.
         *
         * @returns {Boolean}
         *  Whether the main side pane is currently visible.
         */
        this.isSidePaneVisible = function () {
            return sidePane.isVisible();
        };

        /**
         * Changes the visibility of the main side pane and the overlay tool
         * box. If the side pane is visible, the overlay tool box is hidden,
         * and vice versa.
         *
         * @param {Boolean} state
         *  Whether to show or hide the main side pane.
         *
         * @returns {EditView}
         *  A reference to this instance.
         */
        this.toggleSidePane = function (state) {
            this.lockPaneLayout(function () {
                sidePane.toggle(state);
                overlayPane.toggle(!state);
                //app.getWindow().setChromeless(!state);
            });
            return this;
        };

        /**
         * Returns whether the debug pane is currently visible.
         *
         * @returns {Boolean}
         *  Whether the debug pane is currently visible.
         */
        this.isDebugPaneVisible = function () {
            // will be replaced with actual implementation in initDebugMode()
            return false;
        };

        /**
         * Changes the visibility of the debug pane.
         *
         * @param {Boolean} state
         *  Whether to show or hide the debug pane.
         *
         * @returns {EditView}
         *  A reference to this instance.
         */
        this.toggleDebugPane = function (/*state*/) {
            // will be replaced with actual implementation in initDebugMode()
            return this;
        };

        /**
         * Returns whether the debug highlighting mode is currently active.
         *
         * @returns {Boolean}
         *  Whether the debug highlighting mode is currently active.
         */
        this.isDebugHighlight = function () {
            // will be replaced with actual implementation in initDebugMode()
            return false;
        };

        /**
         * Toggles the debug highlighting of specific DOM elements.
         *
         * @param {Boolean} state
         *  Whether to highlight specific DOM elements.
         *
         * @returns {EditView}
         *  A reference to this instance.
         */
        this.toggleDebugHighlight = function (/*state*/) {
            // will be replaced with actual implementation in initDebugMode()
            return this;
        };

        /**
         * Returns whether the clipboard debug pane is currently visible.
         *
         * @returns {Boolean}
         *  Whether the debug pane is currently visible.
         */
        this.isClipboardPaneVisible = function () {
            // will be replaced with actual implementation in initDebugMode()
            return false;
        };

        /**
         * Changes the visibility of the clipboard debug pane.
         *
         * @param {Boolean} state
         *  Whether to show or hide the debug pane.
         *
         * @returns {EditView}
         *  A reference to this instance.
         */
        this.toggleClipboardPane = function (/*state*/) {
            // will be replaced with actual implementation in initDebugMode()
            return this;
        };

        /**
         * Set content to display in the clipboard debug pane.
         *
         * @param {jQuery|String} content
         *  The jQuery or HTML String to display in the clipboard debug pane.
         *
         *  @returns {EditView}
         *  A reference to this instance.
         */
        this.setClipboardDebugInfo = function (/*content*/) {
            // will be replaced with actual implementation in initDebugMode()
            return this;
        };

        /**
         * Creates a new editable container node for clipboard functionality,
         * and inserts it into the hidden container node of the application.
         *
         * @returns {jQuery}
         *  The new empty clipboard container node.
         */
        this.createClipboardNode = function () {
            // clipboard node must be content-editable to allow system copy-to-clipboard
            // clipboard node must contain 'user-select-text' to be focusable in Chrome
            var clipboardNode = $('<div>', { contenteditable: true, 'data-focus-role': 'clipboard' }).addClass('clipboard user-select-text');
            this.insertHiddenNodes(clipboardNode);
            return clipboardNode;
        };

        // destroy all class members on destruction
        this.registerDestructor(function () {
            if (operationsNode) { operationsNode.remove(); }
            if (clipboardDataNode) { clipboardDataNode.remove(); }
            self = sidePane = debugPane = overlayPane = overlayToolBox = operationsNode = clipboardDataNode = null;
        });

        this.log = $.noop;

    } // class EditView

    // exports ================================================================

    // derive this class from class BaseView
    return BaseView.extend({ constructor: EditView });

});
