/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/editcontrols',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/io',
     'io.ox/office/tk/popup/basemenu',
     'io.ox/office/tk/popup/tooltip',
     'io.ox/office/tk/control/menumixin',
     'io.ox/office/baseframework/view/basecontrols',
     'io.ox/office/editframework/model/format/color',
     'io.ox/office/editframework/model/format/border',
     'io.ox/office/drawinglayer/view/drawingcontrols',
     'gettext!io.ox/office/editframework'
    ], function (Utils, KeyCodes, IO, BaseMenu, Tooltip, MenuMixin, BaseControls, Color, Border, DrawingControls, gt) {

    'use strict';

    var // class name shortcuts
        Button = BaseControls.Button,
        RadioList = BaseControls.RadioList,
        ComboField = BaseControls.ComboField,

        // predefined color definitions
        BUILTIN_COLOR_DEFINITIONS = [
            { name: 'dark-red',    label: gt('Dark red'),    color: { type: 'rgb', value: 'C00000' } },
            { name: 'red',         label: gt('Red'),         color: { type: 'rgb', value: 'FF0000' } },
            { name: 'orange',      label: gt('Orange'),      color: { type: 'rgb', value: 'FFC000' } },
            { name: 'yellow',      label: gt('Yellow'),      color: { type: 'rgb', value: 'FFFF00' } },
            { name: 'light-green', label: gt('Light green'), color: { type: 'rgb', value: '92D050' } },
            { name: 'green',       label: gt('Green'),       color: { type: 'rgb', value: '00B050' } },
            { name: 'light-blue',  label: gt('Light blue'),  color: { type: 'rgb', value: '00B0F0' } },
            { name: 'blue',        label: gt('Blue'),        color: { type: 'rgb', value: '0070C0' } },
            { name: 'dark-blue',   label: gt('Dark blue'),   color: { type: 'rgb', value: '002060' } },
            { name: 'purple',      label: gt('Purple'),      color: { type: 'rgb', value: '7030A0' } }
        ],

        // definitions for the theme color table (in order as shown in the GUI color picker)
        // transformation table: 0 = pure color; negative values = 'shade'; positive values = 'tint'
        SCHEME_COLOR_DEFINITIONS = [
            { name: 'background1', label: getBackgroundColorName(1), transformations: [    0, -242, -217, -191, -166, -128 ] },
            { name: 'text1',       label: getTextColorName(1),       transformations: [  128,  166,  191,  217,  242,    0 ] },
            { name: 'background2', label: getBackgroundColorName(2), transformations: [    0, -230, -191, -128,  -64,  -26 ] },
            { name: 'text2',       label: getTextColorName(2),       transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent1',     label: getAccentColorName(1),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent2',     label: getAccentColorName(2),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent3',     label: getAccentColorName(3),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent4',     label: getAccentColorName(4),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent5',     label: getAccentColorName(5),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent6',     label: getAccentColorName(6),     transformations: [   51,  102,  153,    0, -191, -128 ] }
        ],

        LOCALE_DEFINITIONS = [
            { locale: 'en-US', label: gt('English (US)') },
            { locale: 'de-DE', label: gt('German') },
            { locale: 'fr-FR', label: gt('French') },
            { locale: 'es-ES', label: gt('Spanish') },
            { locale: 'cs-CZ', label: gt('Czech') },
            { locale: 'da-DK', label: gt('Danish') },
            { locale: 'nl-NL', label: gt('Dutch (Netherlands)') },
            { locale: 'fi-FI', label: gt('Finnish') },
            { locale: 'el-GR', label: gt('Greek') },
            { locale: 'hu-HU', label: gt('Hungarian') },
            { locale: 'it-IT', label: gt('Italian (Italy)') },
            { locale: 'pl-PL', label: gt('Polish') },
            { locale: 'pt-PT', label: gt('Portuguese (Portugal)') },
            { locale: 'ro-RO', label: gt('Romanian') },
            { locale: 'ru-RU', label: gt('Russian') },
            { locale: 'sv-SE', label: gt('Swedish (Sweden)') }
        ],

        // all available entries for the border picker control
        BORDER_ENTRIES = [

            { name: 'left',    value: { left:    true }, icon: 'docs-border-l', label: /*#. in paragraphs and tables cells */ gt('Left border') },
            { name: 'right',   value: { right:   true }, icon: 'docs-border-r', label: /*#. in paragraphs and tables cells */ gt('Right border') },
            { name: 'top',     value: { top:     true }, icon: 'docs-border-t', label: /*#. in paragraphs and tables cells */ gt('Top border') },
            { name: 'bottom',  value: { bottom:  true }, icon: 'docs-border-b', label: /*#. in paragraphs and tables cells */ gt('Bottom border') },
            { name: 'insidev', value: { insidev: true }, icon: 'docs-border-v', label: /*#. in paragraphs and tables cells */ gt('Inner vertical borders') },
            { name: 'insideh', value: { insideh: true }, icon: 'docs-border-h', label: /*#. in paragraphs and tables cells */ gt('Inner horizontal borders') },

            { name: 'none',               value: { left: false, right: false, insidev: false, top: false, bottom: false, insideh: false }, icon: 'docs-border',        label: /*#. in paragraphs and tables cells */ gt('No borders') },
            { name: 'outer',              value: { left: true,  right: true,  insidev: false, top: true,  bottom: true,  insideh: false }, icon: 'docs-border-tblr',   label: /*#. in paragraphs and tables cells */ gt('Outer borders') },
            { name: 'inner',              value: { left: false, right: false, insidev: true,  top: false, bottom: false, insideh: true },  icon: 'docs-border-hv',     label: /*#. in paragraphs and tables cells */ gt('Inner borders') },
            { name: 'all',                value: { left: true,  right: true,  insidev: true,  top: true,  bottom: true,  insideh: true },  icon: 'docs-border-tbhlrv', label: /*#. in paragraphs and tables cells */ gt('All borders') },
            { name: 'left-right',         value: { left: true,  right: true,  insidev: false, top: false, bottom: false, insideh: false }, icon: 'docs-border-lr',     label: /*#. in paragraphs and tables cells */ gt('Left and right borders') },
            { name: 'left-right-insidev', value: { left: true,  right: true,  insidev: true,  top: false, bottom: false, insideh: false }, icon: 'docs-border-lrv',    label: /*#. in paragraphs and tables cells */ gt('All vertical borders') },
            { name: 'outer-insidev',      value: { left: true,  right: true,  insidev: true,  top: true,  bottom: true,  insideh: false }, icon: 'docs-border-tblrv',  label: /*#. in paragraphs and tables cells */ gt('Inner vertical and outer borders') },
            { name: 'top-bottom',         value: { left: false, right: false, insidev: false, top: true,  bottom: true,  insideh: false }, icon: 'docs-border-tb',     label: /*#. in paragraphs and tables cells */ gt('Top and bottom borders') },
            { name: 'top-bottom-insideh', value: { left: false, right: false, insidev: false, top: true,  bottom: true,  insideh: true },  icon: 'docs-border-tbh',    label: /*#. in paragraphs and tables cells */ gt('All horizontal borders') },
            { name: 'outer-insideh',      value: { left: true,  right: true,  insidev: false, top: true,  bottom: true,  insideh: true },  icon: 'docs-border-tbhlr',  label: /*#. in paragraphs and tables cells */ gt('Inner horizontal and outer borders') }
        ],

        // all available border line styles
        BORDER_LINE_STYLES = [
            { value: 'single', label: /*#. Border line style: a solid single line */ gt('Single') },
            { value: 'double', label: /*#. Border line style: a solid double line */ gt('Double') },
            { value: 'dotted', label: /*#. Border line style: a dotted single line */ gt('Dotted') },
            { value: 'dashed', label: /*#. Border line style: a dashed single line */ gt('Dashed') }
        ],

        ACCESS_RIGHTS_DEFINITIONS = [
            { mail: 'user1@example.com', name: 'User 1' },
            { mail: 'user2@example.com', name: 'User 2' },
            { mail: 'user3@example.com', name: 'User 3' },
            { mail: 'user4@example.com', name: 'User 4' }
        ];

    // private global functions ===============================================

    /**
     * Returns the localized name of a text color in a color scheme.
     */
    function getTextColorName(index) {
        var label =
            //#. The name of a text color in a color scheme (a color scheme consists
            //#. of two text colors, two background colors, and six accent colors).
            //#. Example result: "Text 1", "Text 2"
            //#. %1$d is the index of the text color
            //#, c-format
            gt('Text %1$d', _.noI18n(index));
        return label;
    }

    /**
     * Returns the localized name of a background color in a color scheme.
     */
    function getBackgroundColorName(index) {
        var label =
            //#. The name of a background color in a color scheme (a color scheme consists
            //#. of two text colors, two background colors, and six accent colors).
            //#. Example result: "Background 1", "Background 2"
            //#. %1$d is the index of the background color
            //#, c-format
            gt('Background %1$d', _.noI18n(index));
        return label;
    }

    /**
     * Returns the localized name of an accented color in a color scheme.
     */
    function getAccentColorName(index) {
        var label =
            //#. The name of an accent color in a color scheme (a color scheme consists
            //#. of two text colors, two background colors, and six accent colors).
            //#. Example result: "Accent 1", "Accent 2"
            //#. %1$d is the index of the accent color
            //#, c-format
            gt('Accent %1$d', _.noI18n(index));
        return label;
    }

    // static class EditControls ==============================================

    /**
     * Additional classes defining specialized GUI controls for the OX editor
     * applications.
     *
     * @extends BaseControls
     * @extends DrawingControls
     */
    var EditControls = _(BaseControls).extend(DrawingControls);

    // constants --------------------------------------------------------------

    /**
     * Standard options for the 'Reload' button.
     *
     * @constant
     */
    EditControls.RELOAD_OPTIONS = { icon: 'fa-repeat', tooltip: gt('Reload document'), visibility: 'error' };

    /**
     * Standard options for the 'Undo' button.
     *
     * @constant
     */
    EditControls.UNDO_OPTIONS = { icon: 'docs-undo', tooltip: gt('Revert last operation'), visibility: 'editmode' };

    /**
     * Standard options for the 'Redo' button.
     *
     * @constant
     */
    EditControls.REDO_OPTIONS = { icon: 'docs-redo', tooltip: gt('Restore last operation'), visibility: 'editmode' };

    /**
     * Standard options for a 'Bold' toggle button.
     *
     * @constant
     */
    EditControls.BOLD_OPTIONS = { icon: 'docs-font-bold', tooltip: gt('Bold'), toggle: true, visibility: 'editmode' };

    /**
     * Standard options for an 'Italic' toggle button.
     *
     * @constant
     */
    EditControls.ITALIC_OPTIONS = { icon: 'docs-font-italic', tooltip: gt('Italic'), toggle: true, visibility: 'editmode' };

    /**
     * Standard options for an 'Underline' toggle button.
     *
     * @constant
     */
    EditControls.UNDERLINE_OPTIONS = { icon: 'docs-font-underline', tooltip: gt('Underline'), toggle: true, visibility: 'editmode' };

    /**
     * Standard options for a 'Strike through' toggle button.
     *
     * @constant
     */
    EditControls.STRIKEOUT_OPTIONS = { icon: 'docs-font-strikeout', tooltip: gt('Strike through'), toggle: true, visibility: 'editmode' };

    /**
     * Standard options for a 'Clear formatting' button.
     *
     * @constant
     */
    EditControls.CLEAR_FORMAT_OPTIONS = { icon: 'docs-reset-formatting', tooltip: /*#. clear manual formatting of text or table cells */ gt('Clear formatting'), visibility: 'editmode' };

    // class ColorPicker ======================================================

    /**
     * Creates a control with a drop-down menu used to choose a color from a
     * set of color items. Shows a selection of standard colors, and a table of
     * shaded scheme colors from the current document theme.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {EditApplication} app
     *  The application containing this control group.
     *
     * @param {String|Object} autoColor
     *  Additional information needed to resolve the automatic color. Can be a
     *  string for predefined automatic colors, or an explicit color object
     *  with a value other than the automatic color. See method
     *  Color.getCssColor() for more details.
     *
     * @param {Object} [initOptions]
     *  A map of options to control the properties of the drop-down button and
     *  menu. Supports all options of the RadioList base class.
     */
    EditControls.ColorPicker = RadioList.extend({ constructor: function (app, autoColor, initOptions) {

        var // self reference
            self = this,

            // the color box in the drop-down button
            colorBox = $('<div>').addClass('color-box'),

            // the style sheet containers of the document
            documentStyles = app.getModel().getDocumentStyles();

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            position: 'bottom top left',
            itemDesign: 'grid',
            gridColumns: 10,
            updateCaptionMode: 'none',
            equality: sameColors
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Converts the passed color value to a CSS color string.
         */
        function resolveCssColor(color) {
            var cssColor = _.isObject(color) ? documentStyles.getCssColor(color, autoColor) : 'transparent';
            // show 'white' for transparent colors in dark GUI design
            return (cssColor === 'transparent') ? 'white' : cssColor;
        }

        /**
         * Initializes a new color item button.
         */
        function createItemHandler(event, buttonNode, sectionId, color) {
            buttonNode.prepend($('<div>').addClass('color-button').css('background-color', resolveCssColor(color)));
        }

        /**
         * Updates the color box in the drop-down menu button.
         */
        function updateHandler(color) {
            colorBox.css('background-color', (_.isUndefined(color) || _.isNull(color)) ? 'transparent' : resolveCssColor(color));
        }

        /**
         * Returns whether the passed color objects are considered being equal.
         */
        function sameColors(color1, color2) {

            var color1Value = _.isString(color1.value) ? color1.value.toLowerCase() : null,
                color2Value = _.isString(color2.value) ? color2.value.toLowerCase() : null;

            // returns whether type and value are considered being equal
            function sameTypeAndValue() {

                // exactly the same type and value
                if ((color1.type === color2.type) && ((color1.type === 'auto') || (color1Value === color2Value))) { return true; }

                // scheme colors: match 'textN' and 'darkN', as well as 'backgroundN' and 'lightN'
                // TODO: really always, also in dark color schemes where background would be dark?
                return (color1.type === 'scheme') && (color2.type === 'scheme') && (
                    ((color1Value === 'text1') && (color2Value === 'dark1')) ||
                    ((color1Value === 'dark1') && (color2Value === 'text1')) ||
                    ((color1Value === 'text2') && (color2Value === 'dark2')) ||
                    ((color1Value === 'dark2') && (color2Value === 'text2')) ||
                    ((color1Value === 'background1') && (color2Value === 'light1')) ||
                    ((color1Value === 'light1') && (color2Value === 'background1')) ||
                    ((color1Value === 'background2') && (color2Value === 'light2')) ||
                    ((color1Value === 'light2') && (color2Value === 'background2')));
            }

            // returns whether transformations are considered being equal
            function sameTransformations() {

                // transformations may be missing in both colors
                if (_.isUndefined(color1.transformations) && _.isUndefined(color2.transformations)) { return true; }

                // arrays must be equally sized, and transformations must be similar enough (bug 30085)
                return _.isArray(color1.transformations) && _.isArray(color2.transformations) &&
                    (color1.transformations.length === color2.transformations.length) &&
                    _(color1.transformations).all(function (tf1, index) {
                        var tf2 = color2.transformations[index];
                        return _.isString(tf1.type) && (tf1.type === tf2.type) &&
                            _.isNumber(tf1.value) && _.isNumber(tf2.value) &&
                            // may not be completely equal (bug 30085)
                            (Math.abs(tf1.value - tf2.value) <= 256);
                    });
            }

            // compare colors directly
            if (sameTypeAndValue() && sameTransformations()) {
                return true;
            }

            // special treatment for fallback values in ODF mode
            return app.isODF() &&
                ((color1.type === 'scheme') && _.isString(color1.fallbackValue) && (color1.fallbackValue.toLowerCase() === color2Value)) ||
                ((color2.type === 'scheme') && _.isString(color2.fallbackValue) && (color2.fallbackValue.toLowerCase() === color1Value));
        }

        /**
         * Creates the option buttons for one table row of scheme colors.
         */
        function fillSchemeColorRow(rowIndex) {
            _(SCHEME_COLOR_DEFINITIONS).each(function (definition) {

                var // the encoded transformation
                    encoded = definition.transformations[rowIndex],
                    // decoded tint/shade value
                    tint = encoded > 0,
                    value = Math.round(Math.abs(encoded) / 255 * 100000),
                    // the theme color object
                    color = { type: 'scheme', value: definition.name },
                    // the data value added at the button
                    dataValue = definition.name,
                    // description of the color (tool tip)
                    label = definition.label,
                    // shade/tint value as integral percentage
                    percent = (100 - Math.round(value / 1000)),
                    // fallback value for format without theme support
                    fallbackValue = null;

                if (value !== 0) {
                    color.transformations = [{ type: tint ? 'tint' : 'shade', value: value }];
                    dataValue += (tint ? '-lighter' : '-darker') + percent;
                    label = tint ?
                        //#. The full name of a light theme color (a base color lightened by a specific percentage value)
                        //#. Example result: "Green, lighter 20%"
                        //#. %1$s is the name of the base color
                        //#. %2$d is the percentage value, followed by a literal percent sign
                        //#, c-format
                        gt('%1$s, lighter %2$d%', label, _.noI18n(percent)) :
                        //#. The full name of a dark theme color (a base color darkened by a specific percentage value)
                        //#. Example result: "Green, darker 20%"
                        //#. %1$s is the name of the base color
                        //#. %2$d is the percentage value, followed by a literal percent sign
                        //#, c-format
                        gt('%1$s, darker %2$d%', label, _.noI18n(percent));
                }

                // store the resulting RGB color in the color value object
                if (Color.isThemeColor(color)) {
                    fallbackValue = documentStyles.getCssColor(color, 'fill');
                    if (fallbackValue[0] === '#') {
                        color.fallbackValue = fallbackValue.slice(1);
                    } else {
                        Utils.warn('ColorPicker.fillSchemeColorRow(): unexpected CSS fallback color: ', fallbackValue);
                    }
                }

                self.createOptionButton('theme', color, { tooltip: label, dataValue: dataValue });
            });
        }

        /**
         * Returns the button label for the automatic color.
         */
        function getAutoColorLabel() {
            return Color.isTransparentColor(Color.AUTO, autoColor) ?
                /*#. no fill color, transparent */ gt('No color') :
                /*#. automatic text color (white on dark backgrounds, otherwise black) */ gt('Automatic color');
        }

        /**
         * Inserts all available colors into the drop-down menu.
         */
        var initializeColorTable = app.createDebouncedMethod($.noop, function () {

            self.clearMenu();

            // add automatic color
            self.createMenuSection('auto', { gridColumns: 1 })
                .createOptionButton('auto', Color.AUTO, { label: getAutoColorLabel(), dataValue: 'auto' });

            // add scheme colors
            if (documentStyles.hasSchemeColors()) {
                self.createMenuSection('theme', { label: gt('Theme Colors') });
                _(SCHEME_COLOR_DEFINITIONS[0].transformations.length).times(fillSchemeColorRow);
            }

            // add predefined colors
            self.createMenuSection('standard', { label: gt('Standard Colors') });
            _(BUILTIN_COLOR_DEFINITIONS).each(function (definition) {
                self.createOptionButton('standard', definition.color, { tooltip: definition.label, dataValue: definition.name });
            });
        });

        // methods ------------------------------------------------------------

        /**
         * Changes the information needed to resolve the automatic color, and
         * repaints the menu list item for the automatic color.
         *
         * @param {String|Object} newAutoColor
         *   Can be a string for predefined automatic colors, or an explicit
         *   color object with a value other than the automatic color. See
         *   method Color.getCssColor() for more details.
         *
         * @returns {ColorPicker}
         *  A reference to this instance.
         */
        this.setAutoColor = function (newAutoColor) {
            if (!_.isEqual(autoColor, newAutoColor)) {
                autoColor = newAutoColor;
                this.getMenu().findItemNodes(Color.AUTO).find('.color-button').css('background-color', resolveCssColor(Color.AUTO));
            }
            return this;
        };

        // initialization -----------------------------------------------------

        // add marker class for additional CSS formatting
        this.getMenuNode().addClass('color-picker');

        // add the color preview box to the menu button
        this.getMenuButton().append(colorBox);

        // register an update handler that updates the color box
        this.registerUpdateHandler(updateHandler);

        // register a handler that inserts a color box into each list item
        this.getMenu().on('create:item', createItemHandler);

        // insert color buttons in the drop-down menu after import and changed themes
        app.on('docs:import:success', initializeColorTable);
        this.listenTo(app.getModel().getThemeCollection(), 'triggered', initializeColorTable);

    }}); // class ColorPicker

    // class StyleSheetPicker =================================================

    /**
     * A drop-down list control used to select a style sheet from a list. The
     * drop-down list entries will visualize the formatting attributes of the
     * style sheet if possible.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {EditApplication} app
     *  The application containing this control group.
     *
     * @param {String} family
     *  The attribute family of the style sheets visualized by this control.
     *
     * @param {Object} [initOptions]
     *  Additional options passed to the RadioList constructor. Supports all
     *  options of the RadioList class, and the following additional options:
     *  @param {String|String[]} [initOptions.previewFamilies]
     *      The attribute families used to get the formatting options of the
     *      list items representing the style sheets. If omitted, only
     *      attributes of the family specified by the 'family' parameter will
     *      be used.
     *  @param {String[]} [initOptions.sections]
     *      An array specifying the order of all predefined section
     *      identifiers.
     *  @param {String} [initOptions.i18nModulePath]
     *      The module path to the localized JSON object containing the
     *      configuration of translated style sheet names. Each property of the
     *      map must be a string with the English style sheet name as key.
     *      These keys may contain regular expressions which must be placed in
     *      capturing groups (in parentheses). The values of the map are the
     *      translated style sheet names. The placeholders '$1', '$2', etc.
     *      will match the capturing groups defined in the key patterns.
     */
    EditControls.StyleSheetPicker = RadioList.extend({ constructor: function (app, family, initOptions) {

        var // self reference
            self = this,

            // the model instance
            model = app.getModel(),

            // the style sheet container
            styleSheets = model.getStyleSheets(family),

            // the path to the translated style sheet names
            modulePath = Utils.getStringOption(initOptions, 'i18nModulePath', ''),

            // the configuration for translated style sheet names
            translationDatabase = {},

            // the cache with translated style sheet names
            translationCache = {};

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            itemDesign: 'grid',
            sortItems: Utils.getControlUserData,
            updateCaptionMode: 'label'
        }, initOptions));

        // private methods ----------------------------------------------------

        /**
         * Creates a new button in the drop-down menu for the specified style
         * sheet.
         */
        function createOptionButton(styleName, styleId) {

            var // the section identifier for the style sheet
                category = styleSheets.getUICategory(styleId) || '',
                // sorting priority
                priority = styleSheets.getUIPriority(styleId),
                // the sort index stored at the button for lexicographical sorting
                sortIndex = String((priority < 0) ? (priority + 0x7FFFFFFF) : priority);

            // translate and adjust style name
            styleName = self.translateStyleName(styleName);

            // build a sorting index usable for lexicographical comparison:
            // 1 digit for priority sign, 10 digits positive priority,
            // followed by lower-case translated style sheet name
            while (sortIndex.length < 10) { sortIndex = '0' + sortIndex; }
            sortIndex = ((priority < 0) ? '0' : '1') + sortIndex + styleName.toLowerCase();

            // create the list item, pass sorting index as user data
            self.createOptionButton(category, styleId, { label: styleName, tooltip: styleName, userData: sortIndex });
        }

        /**
         * Fills the drop-down list with all known style names, and adds
         * preview CSS formatting to the list items.
         */
        var fillList = app.createDebouncedMethod($.noop, function () {
            self.clearMenu();
            // configure the sections
            _(Utils.getArrayOption(initOptions, 'sections')).each(function (sectionId) {
                self.createMenuSection(sectionId);
            });
            // insert the style sheets
            _(styleSheets.getStyleSheetNames()).each(createOptionButton);
        });

        // public methods -----------------------------------------------------

        /**
         * Returns the translation of the passed style sheet name for the
         * current locale. Additionally, the passed style name will be adjusted
         * further for GUI display. All words will be capitalized, automatic
         * line breaks before numbers will be prevented by converting the
         * preceding space characters to NBSP characters.
         *
         * @param {String} styleName
         *  The original (English) style sheet name.
         *
         * @returns {String}
         *  The translated and adjusted style sheet name. If no translation is
         *  available, the passed style sheet name will be adjusted and
         *  returned.
         */
        this.translateStyleName = function (styleName) {

            var // the lower-case style name, used as database key
                styleKey = styleName.toLowerCase(),
                // the translated style name
                translatedName = null;

            // first, check the cache of the translation database
            if (styleKey in translationCache) { return translationCache[styleKey]; }

            // try to find a database entry for the base style name
            _(translationDatabase).any(function (value, key) {

                var // the matches for the pattern of the current entry
                    matches = new RegExp('^' + key + '$', 'i').exec(styleName);

                // continue with next database entry, if key does not match
                if (!_.isArray(matches)) { return; }

                // matching entry found, translate it and replace the placeholders
                translatedName = value;
                for (var index = matches.length - 1; index > 0; index -= 1) {
                    translatedName = translatedName.replace(new RegExp('\\$' + index, 'g'), matches[index]);
                }

                // exit the _.any() loop
                return true;
            });

            // adjust the resulting style name for GUI display
            translatedName = (translatedName || Utils.capitalizeWords(styleName)).replace(/ ([0-9])/g, '\xa0$1');

            // put the translated name into the cache
            return (translationCache[styleKey] = _.noI18n(translatedName));
        };

        // initialization -----------------------------------------------------

        // add CSS class to menu button and drop-down list for special formatting
        this.getNode().addClass('style-picker family-' + styleSheets.getStyleFamily());
        this.getMenuNode().addClass('app-' + app.getDocumentType() + ' style-picker family-' + styleSheets.getStyleFamily());

        // add all known style sheets after import and changed style sheets
        app.on('docs:import:success', fillList);
        self.listenTo(styleSheets, 'triggered', fillList);
        // also reinitialize the preview if theme settings have been changed
        self.listenTo(model.getThemeCollection(), 'triggered', fillList);

        // load the translated style names if specified
        if (modulePath.length > 0) {
            IO.loadResource(modulePath).done(function (data) {
                // prevent invoking anything if this control has been destroyed already
                if (self && !self.destroyed) {
                    translationDatabase = data;
                    translationCache = {};
                    fillList();
                }
            });
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    }}); // class StyleSheetPicker

    // class FontFamilyPicker =================================================

    /**
     * A combobox control used to select a font family.
     *
     * @constructor
     *
     * @extends ComboField
     *
     * @param {Object} [initOptions]
     *  A map with options controlling the appearance of the control. Supports
     *  all options of the base class ComboField.
     */
    EditControls.FontFamilyPicker = ComboField.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // the collection of fonts of the edited document
            fontCollection = app.getModel().getFontCollection();

        // base constructor ---------------------------------------------------

        ComboField.call(this, Utils.extendOptions({
            width: 100,
            tooltip: gt('Font name'),
            sortItems: true,
            typeAhead: true
        }, initOptions));

        // methods ------------------------------------------------------------

        /**
         * Fills the drop-down list with all known font names.
         */
        var fillList = app.createDebouncedMethod($.noop, function () {
            self.getMenu().clearContents().createSectionNode('values');
            _(fontCollection.getFontNames()).each(function (fontName) {
                self.createListEntry(fontName, {
                    label: _.noI18n(fontName),
                    labelCss: { fontFamily: fontCollection.getCssFontFamily(fontName), fontSize: '115%' }
                });
            });
        });

        // initialization -----------------------------------------------------

        // add all known fonts after import and changed fonts
        app.on('docs:import:success', fillList);
        self.listenTo(fontCollection, 'triggered', fillList);


    }}); // class FontFamilyPicker

    // class FontHeightPicker =================================================

    /**
     * A combobox control used to select a font height.
     *
     * @constructor
     *
     * @extends ComboField
     *
     * @param {Object} [initOptions]
     *  A map with options controlling the appearance of the control. Supports
     *  all options of the base class ComboField.
     */
    EditControls.FontHeightPicker = ComboField.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        ComboField.call(this, Utils.extendOptions({
            width: 30,
            tooltip: gt('Font size'),
            css: { textAlign: 'right' },
            keyboard: 'number',
            validator: new ComboField.NumberValidator({ min: 1, max: 999.9, precision: 0.1 })
        }, initOptions));

        // initialization -----------------------------------------------------

        _([6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 18, 20, 22, 24, 26, 28, 32, 36, 40, 44, 48, 54, 60, 66, 72, 80, 88, 96]).each(function (size) {
            this.createListEntry(size, { label: _.noI18n(String(size)), css: { textAlign: 'right', paddingRight: 20 } });
        }, this);

    }}); // class FontHeightPicker

    // class LanguagePicker ===================================================

    (function () {

        var // a Promise that will resolve when the supported locales have been received
            supportedLocales = IO.sendRequest({
                method: 'POST',
                module: 'spellchecker',
                params: {
                    action: 'supportedlocales'
                }
            })
            .done(function (data) {
                var supportedLocales = Utils.getArrayOption(data, 'SupportedLocales', []);
                _(LOCALE_DEFINITIONS).each(function (definition) {
                    var localeName = definition.locale.replace('-', '_');
                    definition.supported = _(supportedLocales).contains(localeName);
                });
            });

        EditControls.LanguagePicker = RadioList.extend({ constructor: function (initOptions) {

            var // self reference
                self = this;

            // base constructor -----------------------------------------------

            RadioList.call(this, Utils.extendOptions(initOptions, {
                tooltip: gt('Text language'),
                css: { textAlign: 'left' },
                sortItems: true
            }));

            // initialization -------------------------------------------------

            this.getMenuNode().addClass('language-picker');

            supportedLocales.always(function () {
                if (self.destroyed) { return; }
                _(LOCALE_DEFINITIONS).each(function (definition) {
                    self.createOptionButton('', definition.locale, {
                        label: definition.label,
                        icon: definition.supported ? 'fa-check-square-o' : 'fa-square-o'
                    });
                });
            });

        }}); // class LanguagePicker

    }());

    // class BorderPicker =====================================================

    /**
     * A drop-down list control with different settings for single or multiple
     * components supporting outer and inner border lines. The list entries are
     * divided into two sections. The first section contains entries for single
     * border lines that can be toggled. The second section contains list
     * entries for complex border settings for multiple component borders.
     *
     * The method 'BorderPicker.setValue()' expects object values with the
     * optional properties 'top', 'bottom', 'left', 'right', insideh', and
     * 'insidev' of type Boolean or Null specifying the visibility state of the
     * respective border line. The property value null represents an ambiguous
     * state for the border line. Additionally, the method accepts an options
     * map as second parameter. The two options 'showInsideHor' and
     * 'showInsideVert' will override the respective options passed to the
     * constructor of this instance.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Object} [initOptions]
     *  A map with options controlling the appearance of the drop-down list.
     *  Supports all options of the base class RadioList. Additionally, the
     *  following options are supported:
     *  @param {Boolean|Function} [initOptions.showInsideHor=false]
     *      If set to true, the drop-down list will provide list entries for
     *      horizontal inner borders between multiple components that are
     *      stacked vertically. If set to a function, the visibility of the
     *      affected list items will be evaluated dynamically every time this
     *      control will be updated, according to the return value.
     *  @param {Boolean|Function} [initOptions.showInsideVert=false]
     *      If set to true, the drop-down list will provide list entries for
     *      vertical inner borders between multiple components that are located
     *      side-by-side. If set to a function, the visibility of the affected
     *      list items will be evaluated dynamically every time this control
     *      will be updated, according to the return value.
     */
    EditControls.BorderPicker = RadioList.extend({ constructor: function (initOptions) {

        var // self reference
            self = this,

            // whether list entries with inner horizontal borders will be shown
            showInsideHor = Utils.getFunctionOption(initOptions, 'showInsideHor', Utils.getBooleanOption(initOptions, 'showInsideHor', false)),

            // whether list entries with inner vertical borders will be shown
            showInsideVert = Utils.getFunctionOption(initOptions, 'showInsideVert', Utils.getBooleanOption(initOptions, 'showInsideVert', false));

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            icon: 'docs-border-tblr',
            tooltip: /*#. in paragraphs and tables cells */ gt('Borders')
        }, initOptions, {
            equality: matchBorder,
            updateCaptionMode: 'none'
        }));

        // private methods ----------------------------------------------------

        /**
         * Returns whether to show list items with horizontal inner borders.
         */
        function showInsideHorItems(value) {
            return _.isFunction(showInsideHor) ? showInsideHor.call(self, value) : showInsideHor;
        }

        /**
         * Returns whether to show list items with vertical inner borders.
         */
        function showInsideVertItems(value) {
            return _.isFunction(showInsideVert) ? showInsideVert.call(self, value) : showInsideVert;
        }

        /**
         * Returns whether to show list items with both inner borders.
         */
        function showInsideBothItems(value) {
            return showInsideHorItems(value) && showInsideVertItems(value);
        }

        /**
         * Returns whether the passed borders matches one of the drop-down list
         * entries.
         *
         * @param {Object} value
         *  The border value to be matched against a list entry.
         *
         * @param {Object} entryValue
         *  The value of one of the list entries of this control.
         *
         * @returns {Boolean}
         *  Whether the border value matches the list entry. All properties
         *  contained in the list entry must be equal in the border value.
         *  Other existing properties of the border value are ignored.
         */
        function matchBorder(value, entryValue) {
            return _.isObject(value) && _(entryValue).all(function (visible, propName) {
                // use false for missing properties in value, but do not match existing null values
                return visible === ((propName in value) ? value[propName] : false);
            });
        }

        /**
         * Updates the icon in the drop-down menu button according to the
         * current value of this control.
         */
        function updateHandler(value) {
            var icon = '';
            if (_.isObject(value)) {
                if (value.top) { icon += 't'; }
                if (value.bottom) { icon += 'b'; }
                if (value.insideh) { icon += 'h'; }
                if (value.left) { icon += 'l'; }
                if (value.right) { icon += 'r'; }
                if (value.insidev) { icon += 'v'; }
                if (icon.length > 0) { icon = '-' + icon; }
                icon = 'docs-border' + icon;
            }
            Utils.setControlCaption(self.getMenuButton(), icon ? { icon: icon } : self.getOptions());
        }

        // initialization -----------------------------------------------------

        // prepare the sections for the toggle buttons and the complex border buttons
        self.createMenuSection('single').createMenuSection('compound');

        // insert all supported list items
        _(BORDER_ENTRIES).each(function (entry) {
            var sectionId = (_.size(entry.value) === 1) ? 'single' : 'compound',
                visible = entry.value.insideh ? (entry.value.insidev ? showInsideBothItems : showInsideHorItems) : (entry.value.insidev ? showInsideVertItems : true);
            self.createOptionButton(sectionId, entry.value, { icon: entry.icon, label: entry.label, dataValue: entry.name, visible: visible });
        });

        // register custom update handler to select the correct border icon
        this.registerUpdateHandler(updateHandler);

    }}); // class BorderPicker

    // class BorderStylePicker ================================================

    EditControls.BorderStylePicker = RadioList.extend({ constructor: function (initOptions) {

        var // the caption element in the drop-down menu button
            captionNode = $('<div>').addClass('caption').css({ width: '100%' }).append(
                $('<p>').css({ borderTop: '3px black none', marginTop: '15px', marginRight: '10px' })
            );

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            tooltip: /*#. line style (solid, dashed, dotted) in paragraphs and tables cells */ gt('Border style'),
            css: { textAlign: 'left' }
        }, initOptions, {
            updateCaptionMode: 'none'
        }));

        // private methods ----------------------------------------------------

        /**
         * Initializes a new list item according to the border styles.
         */
        function createItemHandler(event, buttonNode, sectionId, value) {
            buttonNode.prepend($('<p>').css({ borderTop: '3px black ' + Border.getCssBorderStyle(value), marginTop: '12px' }));
        }

        /**
         * Updates the border style in the drop-down menu button.
         */
        function updateHandler(value) {
            if (value === null) { value = 'none'; }
            captionNode.children('p').css('border-top-style', Border.getCssBorderStyle(value));
        }

        // initialization -----------------------------------------------------

        this.getMenuButton().prepend(captionNode);
        this.registerUpdateHandler(updateHandler);

        // register a handler that inserts a border style element into each list item
        this.getMenu().on('create:item', createItemHandler);

        _(BORDER_LINE_STYLES).each(function (entry) {
            this.createOptionButton('', entry.value, { tooltip: entry.label });
        }, this);

    }}); // class BorderStylePicker

    // class TableBorderWidthPicker ===========================================

    EditControls.TableBorderWidthPicker = ComboField.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        ComboField.call(this, Utils.extendOptions({
            width: 40,
            tooltip: gt('Border width'),
            css: { textAlign: 'right' },
            keyboard: 'number',
            validator: new ComboField.NumberValidator({ min: 0.5, max: 10, precision: 0.1 })
        }, initOptions));

        // initialization -----------------------------------------------------

        _([0.5, 1, 1.5, 2, 2.5, 3, 4, 6]).each(function (size) {
            this.createListEntry(size, {
                label: gt.format(
                    //#. A single list entry of a GUI border width picker.
                    //#. Example result: "5 points"
                    //#. %1$d is the width of the border line.
                    //#, c-format
                    gt.ngettext('%1$d point', '%1$d points', size),
                    _.noI18n(size)
                ),
                css: { textAlign: 'center' }
            });
        }, this);

    }}); // class TableBorderWidthPicker

    // class CellBorderWidthPicker ============================================

    EditControls.CellBorderWidthPicker = ComboField.extend({ constructor: function (initOptions) {

        // base constructor ---------------------------------------------------

        ComboField.call(this, Utils.extendOptions({
            width: 40,
            tooltip: gt('Border width'),
            css: { textAlign: 'right' },
            keyboard: 'number',
            validator: new ComboField.NumberValidator({ min: 0.5, max: 3, precision: 0.1 })
        }, initOptions));

        // initialization -----------------------------------------------------

        _([0.5, 1, 2, 3]).each(function (size) {
            this.createListEntry(size, {
                label: gt.format(
                    //#. A single list entry of a GUI border width picker.
                    //#. Example result: "5 points"
                    //#. %1$d is the width of the border line.
                    //#, c-format
                    gt.ngettext('%1$d point', '%1$d points', size),
                    _.noI18n(size)
                ),
                css: { textAlign: 'center' }
            });
        }, this);

    }}); // class CellBorderWidthPicker

    // class TableSizePicker ==================================================

    /**
     * A drop-down button and a drop-down menu containing a resizable grid
     * allowing to select a specific size.
     *
     * @constructor
     *
     * @extends Button
     * @extends MenuMixin
     *
     * @param {Object} [initOptions]
     *  A map with options controlling the appearance of this control. Supports
     *  all options that are supported by the base class Button. Additionally,
     *  the following options are supported:
     *  @param {Integer} [initOptions.maxCols=15]
     *      The maximal number of columns to be displayed in the control.
     *  @param {Integer} [initOptions.maxRows=15]
     *      The maximal number of rows to be displayed in the control.
     */
    EditControls.TableSizePicker = Button.extend({ constructor: function (app, initOptions) {

        var // self referemce
            self = this,

            // minimum size allowed to choose
            MIN_SIZE = { width: 1, height: 1 },
            // maximum size allowed to choose (defaulting to 15 columns and rows)
            MAX_SIZE = { width: Utils.getIntegerOption(initOptions, 'maxCols', 15, 1), height: Utils.getIntegerOption(initOptions, 'maxRows', 15, 1) },
            // minimum size visible even if selected size is smaller
            MIN_VIEW_SIZE = { width: Math.min(MAX_SIZE.width, 6), height: Math.min(MAX_SIZE.height, 4) },

            // current grid size
            gridSize = { width: 0, height: 0 },

            // the drop-down menu instance (must be created after Button base constructor!)
            menu = null,

            // the button control to be inserted into the drop-down menu
            gridButton = Utils.createButton().addClass(Utils.FOCUSABLE_CLASS),

            // container for the dynamically generated grid mark-up
            gridContainer = $('<div>').addClass('grid-container'),

            // the tooltip node showing the current grid size
            sizeTooltip = null,

            // whether to grow to the left or right or to the top or bottom
            growTop = false, growLeft = false,

            // last applied page position
            lastPagePos = null;

        // base constructors --------------------------------------------------

        Button.call(this, Utils.extendOptions({ icon: 'docs-table-insert', tooltip: gt('Insert table') }, initOptions));
        menu = new BaseMenu({ classes: 'table-size-picker', autoLayout: false, anchor: this.getNode() });
        MenuMixin.call(this, menu, { button: this.getButtonNode() });

        // private methods ----------------------------------------------------

        /**
         * Returns the current size of the grid, as object with 'width' and
         * 'height' attributes.
         */
        function getViewGridSize() {
            var rowNodes = gridContainer[0].childNodes;
            return { width: rowNodes[0].childNodes.length, height: rowNodes.length };
        }

        /**
         * Changes the current size of the grid, and updates the badge labels.
         */
        function setGridSize(newSize) {

            var // new total size to be shown in the view grid
                newViewSize = {},
                // HTML mark-up of the grid cells
                rowCellsMarkup = '', gridMarkup = '';

            // validate passed size
            gridSize.width = Utils.minMax(newSize.width, MIN_SIZE.width, MAX_SIZE.width);
            gridSize.height = Utils.minMax(newSize.height, MIN_SIZE.height, MAX_SIZE.height);
            newViewSize.width = Math.max(gridSize.width, MIN_VIEW_SIZE.width);
            newViewSize.height = Math.max(gridSize.height, MIN_VIEW_SIZE.height);

            // generate mark-up for all cells in one row
            _(newViewSize.width).times(function (col) {
                var selected = growLeft ? (col >= newViewSize.width - gridSize.width) : (col < gridSize.width);
                rowCellsMarkup += '<div data-width="' + (growLeft ? (newViewSize.width - col) : (col + 1)) + '"';
                if (selected) { rowCellsMarkup += ' class="selected"'; }
                rowCellsMarkup += '></div>';
            });

            // generate markup for all rows
            _(newViewSize.height).times(function (row) {
                var selected = growTop ? (row >= newViewSize.height - gridSize.height) : (row < gridSize.height);
                gridMarkup += '<div data-height="' + (growTop ? (newViewSize.height - row) : (row + 1)) + '"';
                if (selected) { gridMarkup += ' class="selected"'; }
                gridMarkup += '>' + rowCellsMarkup + '</div>';
            });
            gridContainer[0].innerHTML = gridMarkup;

            // update the tooltip
            Utils.log('set text: w=' + gridSize.width + ', h=' + gridSize.height);
            sizeTooltip.setText(_.noI18n(gt.format(
                //#. %1$d is the number of columns in the drop-down grid of a table size picker control.
                //#, c-format
                gt.ngettext('%1$d column', '%1$d columns', gridSize.width),
                _.noI18n(gridSize.width)
            ) + ' \xd7 ' + gt.format(
                //#. %1$d is the number of rows in the drop-down grid of a table size picker control.
                //#, c-format
                gt.ngettext('%1$d row', '%1$d rows', gridSize.height),
                _.noI18n(gridSize.height)
            )));
        }

        /**
         * Sets the position and growing directions of the menu node according
         * to the available space around the group node.
         */
        function initializeMenuNodePosition() {

            var // position and size of the group node
                groupPosition = Utils.getNodePositionInPage(self.getNode()),
                // sizes available for the drop-down menu per group side
                availableSizes = menu.getAvailableSizes(),
                // decide where to grow the grid to
                newGrowTop = (availableSizes.bottom.height < 200) && (availableSizes.top.height > availableSizes.bottom.height),
                newGrowLeft = availableSizes.left.width > availableSizes.right.width;

            // rerender the grid (position of selected cells) if orientation has changed
            if ((growTop !== newGrowTop) || (growLeft !== newGrowLeft)) {
                growTop = newGrowTop;
                growLeft = newGrowLeft;
                setGridSize(gridSize);
            }

            // initialize grid position
            menu.getNode().toggleClass('grow-top', growTop).toggleClass('grow-left', growLeft);
            menu.setNodePosition({
                top: growTop ? null : (groupPosition.top + groupPosition.height + 1),
                bottom: growTop ? (groupPosition.bottom + groupPosition.height + 1) : null,
                left: growLeft ? null : groupPosition.left,
                right: growLeft ? groupPosition.right : null
            });
        }

        /**
         * Updates the grid size according to the passed screen coordinates.
         */
        function updateGridSizeForPosition(pageX, pageY) {

            var // current size of the view grid
                viewSize = getViewGridSize(),
                // width and height of one cell
                cellWidth = gridButton.outerWidth() / viewSize.width,
                cellHeight = gridButton.outerHeight() / viewSize.height,
                // position/size of the grid button node
                buttonDim = Utils.extendOptions(gridButton.offset(), { width: gridButton.outerWidth(), height : gridButton.outerHeight() }),
                // mouse position relative to grid origin
                mouseX = growLeft ? (buttonDim.left + buttonDim.width - pageX) : (pageX - buttonDim.left),
                mouseY = growTop ? (buttonDim.top + buttonDim.height - pageY) : (pageY - buttonDim.top),
                // new grid size (enlarge if the last column/row is covered more than 80% of its width/height)
                newWidth = (cellWidth > 0) ? Math.floor(mouseX / cellWidth + 1.2) : 1,
                newHeight = (cellHeight > 0) ? Math.floor(mouseY / cellHeight + 1.2) : 1;

            // set grid size if changed
            if ((newWidth !== gridSize.width) || (newHeight !== gridSize.height)) {
                setGridSize({ width: newWidth, height: newHeight });
            }

            // store position, needed to refresh grid if its position changes from outside
            lastPagePos = { x: pageX, y: pageY };

            return false;
        }

        /**
         * Handles 'menu:open' events and initializes the drop-down grid.
         * Registers a 'mouseenter' handler at the drop-down menu that starts
         * a 'mousemove' listener when the mouse first hovers the grid element.
         */
        function menuOpenHandler() {

            // unbind all running global event handlers
            unbindGlobalEventHandlers();

            // always show the tooltip
            sizeTooltip.show();

            // initialize grid
            lastPagePos = null;
            setGridSize(MIN_SIZE);
            initializeMenuNodePosition();

            // wait for mouse to enter the grid before listening globally to mousemove events
            gridButton.off('mouseenter').one('mouseenter', function () {
                $(document).on('mousemove', gridMouseMoveHandler);
            });
        }

        /**
         * Handles 'menu:close' events.
         */
        function menuCloseHandler() {
            sizeTooltip.hide();
            unbindGlobalEventHandlers();
            lastPagePos = null;
        }

        /**
         * Handles 'menu:layout' events.
         */
        function menuLayoutHandler() {
            if (lastPagePos) {
                updateGridSizeForPosition(lastPagePos.x, lastPagePos.y);
            } else {
                setGridSize(gridSize);
            }
            initializeMenuNodePosition();
        }

        /**
         * Unbind the global event listener registered at the document.
         */
        function unbindGlobalEventHandlers() {
            $(document).off('mousemove', gridMouseMoveHandler);
        }

        /**
         * Handles 'mousemove' events in the open drop-down menu element.
         */
        function gridMouseMoveHandler(event) {
            updateGridSizeForPosition(event.pageX, event.pageY);
            return false;
        }

        /**
         * Handles keyboard events in the open drop-down menu element.
         */
        function gridKeyDownHandler(event) {

            var // new size of the grid
                newGridSize = _.clone(gridSize);

            switch (event.keyCode) {
            case KeyCodes.LEFT_ARROW:
                newGridSize.width += (growLeft ? 1 : -1);
                setGridSize(newGridSize);
                return false;
            case KeyCodes.UP_ARROW:
                newGridSize.height += (growTop ? 1 : -1);
                setGridSize(newGridSize);
                return false;
            case KeyCodes.RIGHT_ARROW:
                newGridSize.width += (growLeft ? -1 : 1);
                setGridSize(newGridSize);
                return false;
            case KeyCodes.DOWN_ARROW:
                newGridSize.height += (growTop ? -1 : 1);
                setGridSize(newGridSize);
                return false;
            }
        }

        // initialization -----------------------------------------------------

        // convert ENTER, SPACE, and TAB keys to click events
        Utils.setButtonKeyHandler(gridButton, { tab: true });

        // initialize the button control and embed it in the drop-down menu
        // Bug 28583: add absolutely positioned overlay node to enable tracking on touch devices
        gridButton.append(gridContainer.append($('<div>').append($('<div>'))), $('<div>').addClass('abs grid-overlay'));
        menu.appendContentNodes(gridButton);

        // create the tooltip that shows the table size currently selected
        sizeTooltip = new Tooltip({ anchor: menu.getNode() });

        // register event handlers
        this.registerChangeHandler('tracking:end', { source: gridButton, valueResolver: function () { return gridSize; } })
            // bug 31170: restrict to 'click' events that originate from keyboard shortcuts (prevent double trigger from 'tracking:end' and 'click')
            .registerChangeHandler('click', { source: gridButton, eventFilter: function (event) { return _.isNumber(event.keyCode); }, valueResolver: function () { return gridSize; } })
            .on({ 'menu:open': menuOpenHandler, 'menu:close': menuCloseHandler, 'menu:layout': menuLayoutHandler });
        gridButton.on({ 'tracking:move': gridMouseMoveHandler, 'keydown': gridKeyDownHandler });

        // Bug 28583: enable trcking for touch devices
        gridButton.enableTracking();

        // destroy all class members on destruction
        this.registerDestructor(function () {
            unbindGlobalEventHandlers();
            gridContainer.remove();
            gridButton.remove();
            sizeTooltip.destroy();

            app = menu = gridButton = gridContainer = sizeTooltip = null;
        });

    }}); // class TableSizePicker

    // class AcquireEditButton ================================================

    /**
     * Standard button for the 'Acquire edit rights' action.
     */
    EditControls.AcquireEditButton = Button.extend({ constructor: function (app, initOptions) {

        // base constructor ---------------------------------------------------

        Button.call(this, Utils.extendOptions({
            icon: 'fa-pencil',
            tooltip: gt('Acquire edit rights'),
            visibility: 'readonly'
        }, initOptions));

    }}); // class AcquireEditButton

    // class AccessRightsPicker ===============================================

    EditControls.AccessRightsPicker = RadioList.extend({ constructor: function (app, initOptions) {

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions(initOptions, {
            label: _.noI18n('Access Rights'),
            tooltip: _.noI18n('Access rights'),
            sortItems: true
        }));

        // initialization -----------------------------------------------------

        _(ACCESS_RIGHTS_DEFINITIONS).each(function (definition) {
            this.createOptionButton('', definition.mail, { label: _.noI18n(definition.name), tooltip: _.noI18n(definition.mail) });
        }, this);

    }}); // class AccessRightsPicker

    // exports ================================================================

    return EditControls;

});
