/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/format/fontcollection',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/model/modelobject'
    ], function (Utils, ModelObject) {

    'use strict';

    var // all known fonts, mapped by font name
        PREDEFINED_FONTS = {

            'Helvetica':            { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Arial'] } },
            'Arial':                { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Helvetica'] } },
            'Arial Black':          { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Arial'] }, hidden: true },
            'Verdana':              { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Arial'] } },
            'Tahoma':               { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Arial'] } },
            'Calibri':              { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Carlito', 'Arial'] } },
            'Carlito':              { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Calibri', 'Arial'] }, hidden: true },
            'Albany':               { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Arial'] }, hidden: true },
            'Impact':               { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Arial'] } },
            'Helvetica Neue':       { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Helvetica'] }, hidden: true },
            'Open Sans':            { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Helvetica Neue'] }, hidden: true },
            'Open Sans Light':      { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Open Sans'] }, hidden: true },
            'Open Sans Semibold':   { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Open Sans'] }, hidden: true },
            'Open Sans Extrabold':  { cssDefault: 'sans-serif', description: { pitch: 'variable', altNames: ['Open Sans Semibold'] }, hidden: true },

            'Times':                { cssDefault: 'serif', description: { pitch: 'variable', altNames: ['Times New Roman'] }, hidden: true },
            'Times New Roman':      { cssDefault: 'serif', description: { pitch: 'variable', altNames: ['Times'] } },
            'Georgia':              { cssDefault: 'serif', description: { pitch: 'variable', altNames: ['Times New Roman'] } },
            'Palatino':             { cssDefault: 'serif', description: { pitch: 'variable', altNames: ['Times New Roman'] } },
            'Cambria':              { cssDefault: 'serif', description: { pitch: 'variable', altNames: ['Caladea', 'Times New Roman'] } },
            'Caladea':              { cssDefault: 'serif', description: { pitch: 'variable', altNames: ['Cambria', 'Times New Roman'] }, hidden: true },
            'Thorndale':            { cssDefault: 'serif', description: { pitch: 'variable', altNames: ['Times New Roman'] }, hidden: true },
            'Book Antiqua':         { cssDefault: 'serif', description: { pitch: 'variable', altNames: ['Palatino'] } },

            'Courier':              { cssDefault: 'monospace', description: { pitch: 'fixed', altNames: ['Courier New'] }, hidden: true },
            'Courier New':          { cssDefault: 'monospace', description: { pitch: 'fixed', altNames: ['Courier'] } },
            'Andale Mono':          { cssDefault: 'monospace', description: { pitch: 'fixed', altNames: ['Courier New'] } },
            'Consolas':             { cssDefault: 'monospace', description: { pitch: 'fixed', altNames: ['Courier New'] } },
            'Cumberland':           { cssDefault: 'monospace', description: { pitch: 'fixed', altNames: ['Courier New'] }, hidden: true }
        },

        // a regular expression that matches valid font names
        RE_VALID_FONTNAME = new RegExp('^[^<>[\\]()"\'&/\\\\]+$'),

        // static cache for calculated font metrics (mapped by font name)
        fontMetricsCache = {},

        // the dummy element used to calculate font metrics
        helperNode = $('<div>').appendTo($('<div>').css({ position: 'absolute', left: 0, top: 0, width: 1, height: 1, overflow: 'hidden' }).prependTo('body'));

    // class FontCollection ===================================================

    /**
     * Contains the definitions of all known fonts in a document.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {EditApplication} app
     *  The root application instance.
     */
    function FontCollection(app) {

        var // self reference
            self = this,

            // all registered fonts, mapped by capitalized font name
            fonts = {};

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Removes the calculated CSS font families from all registered fonts.
         */
        function deleteAllCssFontFamilies() {
            _(fonts).each(function (font) { delete font.cssFontFamily; });
        }

        /**
         * Calculates and returns the CSS font family attribute containing the
         * specified font name and all alternative font names.
         */
        function calculateCssFontFamily(fontName) {

            var // the resulting collection of CSS font names
                cssFontNames = [fontName],
                // the last found CSS default font family
                lastCssDefault = null;

            // collects all alternative font names of the fonts in the passed array
            function collectAltNames(fontNames) {

                var // alternative font names of the passed fonts
                    altFontNames = [];

                // collect all alternative font names
                _(fontNames).each(function (fontName2) {
                    if ((fontName2 in fonts) && _.isArray(fonts[fontName2].description.altNames)) {
                        altFontNames = altFontNames.concat(fonts[fontName2].description.altNames);
                        if (_.isString(fonts[fontName2].cssDefault)) {
                            lastCssDefault = fonts[fontName2].cssDefault;
                        }
                    }
                });

                // unify the new alternative font names and remove font names
                // already collected in cssFontNames (prevent cyclic references)
                altFontNames = _.chain(altFontNames).unique().difference(cssFontNames).value();

                // insert the new alternative font names into result list, and
                // call recursively until no new font names have been found
                if (altFontNames.length > 0) {
                    cssFontNames = cssFontNames.concat(altFontNames);
                    collectAltNames(altFontNames);
                }
            }

            // collect alternative font names by starting with passed font name
            collectAltNames([fontName]);

            // finally, add the CSS default family
            if (_.isString(lastCssDefault)) {
                cssFontNames.push(lastCssDefault);
            }

            // remove invalid/dangerous characters
            cssFontNames = _(cssFontNames).filter(_.bind(RE_VALID_FONTNAME.test, RE_VALID_FONTNAME));

            // enclose all fonts with white-space into quote characters
            _(cssFontNames).each(function (cssFontName, index) {
                if (/\s/.test(cssFontName)) {
                    cssFontNames[index] = '"' + cssFontName.replace(/\s+/g, ' ') + '"';
                }
            });

            // return the CSS font-family attribute value
            return cssFontNames.join(',');
        }

        /**
         * Prepares the helper DOM node used to calculate a specific font
         * metrics setting, and invokes the passed callback function.
         *
         * @param {Object} attributes
         *  Character formatting attributes influencing the font metrics.
         *  @param {String} attributes.fontName
         *      The name of the original font family (case-insensitive).
         *  @param {Number} attributes.fontSize
         *      The font size, in points.
         *  @param {Boolean} [attributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [attributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         *
         * @param {Function} callback
         *  The callback function that will be invoked after preparing the
         *  helper DOM node. Receives the prepared helper DOM node, already
         *  inserted into the DOM, as first parameter. Must return the
         *  resulting font metrics setting.
         *
         * @param {String} [cacheKey]
         *  If specified, a map key for the font metrics cache that will be
         *  checked first. An existing cache entry will be returned without
         *  invoking the callback function. Otherwise, the new value returned
         *  by the callback function will be inserted into the cache.
         *
         * @return {Number}
         *  The effective width of the passed text, in pixels.
         */
        function calculateFontMetrics(attributes, callback, cacheKey) {

            var // capitalized font name (used as cache key)
                fontName = Utils.capitalizeWords(attributes.fontName),
                // font size, rounded to tenths of points
                fontSize = Utils.round(attributes.fontSize, 0.1),
                // font weight
                bold = Utils.getBooleanOption(attributes, 'bold', false),
                // font slant
                italic = Utils.getBooleanOption(attributes, 'italic', false),

                // cache entry for the font (cache not used for long texts)
                fontEntry = _.isString(cacheKey) ? (fontMetricsCache[fontName] || (fontMetricsCache[fontName] = {})) : null,
                // map key for the font entry with all font attributes
                attrKey = fontEntry ? (fontSize + ',' + bold + ',' + italic) : null,
                // subsequent cache entry for the passed font attributes
                attrEntry = attrKey ? (fontEntry[attrKey] || (fontEntry[attrKey] = {})) : null,

                // the result of the callback function
                result = 0;

            // look for an existing value in the cache
            if (attrEntry && (cacheKey in attrEntry)) {
                return attrEntry[cacheKey];
            }

            // set text formatting at the helper DOM node
            helperNode.empty().css({
                display: 'inline-block',
                width: 'auto',
                padding: 0,
                border: 'none',
                lineHeight: 'normal',
                whiteSpace: 'pre',
                fontFamily: self.getCssFontFamily(fontName),
                fontSize: fontSize + 'pt',
                fontWeight: bold ? 'bold' : 'normal',
                fontStyle: italic ? 'italic' : 'normal'
            });

            // fetch result from the specified callback function
            result = callback.call(self, helperNode);

            // insert the result into the cache entry
            if (attrEntry) { attrEntry[cacheKey] = result; }
            return result;
        }

        // methods ------------------------------------------------------------

        /**
         * Inserts a new font description into this container. An existing font
         * with the specified name will be replaced.
         *
         * @param {String} fontName
         *  The name of of the new font.
         *
         * @param {Object} description
         *  The font description of the font.
         *
         * @returns {FontCollection}
         *  A reference to this instance.
         */
        this.insertFont = function (fontName, description) {

            // fonts are keyed by capitalized names
            fontName = Utils.capitalizeWords(fontName);

            // store passed font attributes
            description = _.copy(description, true);
            fonts[fontName] = { description: description, hidden: false };

            // validate alternative names
            if (_.isArray(description.altNames)) {
                description.altNames = _.chain(description.altNames).without('').map(Utils.capitalizeWords).value();
            } else {
                description.altNames = [];
            }

            // add predefined alternative names
            if ((fontName in PREDEFINED_FONTS) && _.isArray(PREDEFINED_FONTS[fontName].description.altNames)) {
                description.altNames = description.altNames.concat(PREDEFINED_FONTS[fontName].description.altNames);
            }

            // unify alternative names, remove own font name
            description.altNames = _.chain(description.altNames).unique().without(fontName).value();

            // add default CSS font family if known
            if (fontName in PREDEFINED_FONTS) {
                fonts[fontName].cssDefault = PREDEFINED_FONTS[fontName].cssDefault;
            }

            // delete old font metrics from the cache
            delete fontMetricsCache[fontName];

            // remove calculated CSS font families of all fonts, notify listeners
            deleteAllCssFontFamilies();
            this.trigger('insert:font', fontName);

            return this;
        };

        /**
         * Returns the names of all predefined and registered fonts. Predefined
         * fonts with the hidden flag that have not been registered will not be
         * included.
         *
         * @returns {String[]}
         *  The names of all predefined and registered fonts, as array.
         *
         */
        this.getFontNames = function () {
            var fontNames = [];
            _(fonts).each(function (fontEntry, fontName) {
                if (!Utils.getBooleanOption(fontEntry, 'hidden', false)) {
                    fontNames.push(fontName);
                }
            });
            return fontNames;
        };

        /**
         * Returns the value of the CSS 'font-family' attribute containing the
         * specified font name and all alternative font names.
         *
         * @param {String} fontName
         *  The name of of the font (case-insensitive).
         *
         * @returns {String}
         *  The value of the CSS 'font-family' attribute containing the
         *  specified font name and all alternative font names.
         */
        this.getCssFontFamily = function (fontName) {

            // fonts are keyed by capitalized names
            fontName = Utils.capitalizeWords(fontName);

            // default for unregistered fonts: just the passed font name
            if (!(fontName in fonts)) {
                return fontName;
            }

            // calculate and cache the CSS font family
            if (!_.isString(fonts[fontName].cssFontFamily)) {
                fonts[fontName].cssFontFamily = calculateCssFontFamily(fontName);
            }

            return fonts[fontName].cssFontFamily;
        };

        /**
         * Calculates the normal line height for the passed character
         * attributes, as reported by the current browser. Note that this value
         * may vary for the same font settings in different browsers.
         *
         * @param {Object} attributes
         *  Character formatting attributes influencing the normal line height.
         *  @param {String} attributes.fontName
         *      The name of the original font family (case-insensitive).
         *  @param {Number} attributes.fontSize
         *      The font size, in points.
         *  @param {Boolean} [attributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [attributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         *
         * @returns {Number}
         *  The 'normal' line height for the passed character attributes, in
         *  pixels (with a precision of 1/10 pixel for browsers using
         *  fractional line heights internally).
         */
        this.getNormalLineHeight = (function () {

            var // the complete HTML mark-up for ten text lines (IE uses fractional line heights internally)
                NODE_MARKUP = Utils.repeatString('<span>X</span>', 10, '<br>');

            function getNormalLineHeight(attributes) {
                return calculateFontMetrics(attributes, function (node) {
                    return node.html(NODE_MARKUP).height() / 10;
                }, 'int.lineheight');
            }

            return getNormalLineHeight;
        }());

        /**
         * Returns the effective width of the passed text, in pixels.
         *
         * @param {String} text
         *  The text whose width will be calculated. If the text consists of
         *  one or two characters only, the calculated text width will be
         *  cached internally for subsequent invocations with the same text and
         *  character attributes.
         *
         * @param {Object} attributes
         *  Character formatting attributes influencing the text width.
         *  @param {String} attributes.fontName
         *      The name of the original font family (case-insensitive).
         *  @param {Number} attributes.fontSize
         *      The font size, in points.
         *  @param {Boolean} [attributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [attributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         *
         * @return {Number}
         *  The effective width of the passed text, in pixels.
         */
        this.getTextWidth = function (text, attributes) {

            var // the key for the font metrics cache (only for short text)
                cacheKey = (text.length <= 2) ? ('int.width.' + text) : null;

            return calculateFontMetrics(attributes, function (node) {
                switch (text.length) {
                case 0:
                    return 0;
                case 1:
                    return node.text(Utils.repeatString(text, 10)).width() / 10;
                default:
                    return Math.ceil(node.text(text).width());
                }
            }, cacheKey);
        };

        /**
         * Returns the largest text width of the digits '0' to '9', in pixels.
         *
         * @param {Object} attributes
         *  Character formatting attributes influencing the text width of the
         *  digits.
         *  @param {String} attributes.fontName
         *      The name of the original font family (case-insensitive).
         *  @param {Number} attributes.fontSize
         *      The font size, in points.
         *  @param {Boolean} [attributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [attributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         *
         * @return {Number}
         *  The largest text width of the digits '0' to '9', in pixels.
         */
        this.getDigitWidth = function (attributes) {
            return calculateFontMetrics(attributes, function () {
                var digitWidths = _(10).times(function (index) {
                    return self.getTextWidth(String(index), attributes);
                });
                return Math.max.apply(Math.max, digitWidths);
            }, 'int.digitwidth');
        };

        /**
         * Provides the ability to calculate and cache custom font metrics
         * settings.
         *
         * @param {Object} attributes
         *  Character formatting attributes influencing the font metrics.
         *  @param {String} attributes.fontName
         *      The name of the original font family (case-insensitive).
         *  @param {Number} attributes.fontSize
         *      The font size, in points.
         *  @param {Boolean} [attributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  @param {Boolean} [attributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         *
         * @param {Function} callback
         *  The callback function that will be invoked after preparing the
         *  helper DOM node. Receives the prepared helper DOM node, already
         *  inserted into the DOM, as first parameter. Must return the
         *  resulting font metrics setting to be cached.
         *
         * @param {String} [cacheKey]
         *  If specified, a unique map key for the font metrics cache that will
         *  be checked first. An existing cache entry will be returned without
         *  invoking the callback function. Otherwise, the new value returned
         *  by the callback function will be inserted into the cache. If
         *  omitted, the callback function will always be invoked, and the
         *  result will not be inserted into the cache.
         *
         * @return {Any}
         *  The font metric setting from the cache.
         */
        this.getCustomFontMetrics = function (attributes, callback, cacheKey) {
            cacheKey = _.isString(cacheKey) ? ('ext.' + cacheKey) : null;
            return calculateFontMetrics(attributes, callback, cacheKey);
        };

        // initialization -----------------------------------------------------

        // start with all predefined fonts
        fonts = _.copy(PREDEFINED_FONTS, true);

    } // class FontCollection

    // static methods ---------------------------------------------------------

    /**
     * Returns the first font name specified in the passed font family string
     * received from a CSS font-family attribute. The CSS font family may
     * consist of a comma separated list of font names, where the font names
     * may be enclosed in quote characters.
     *
     * @param {String} cssFontFamily
     *  The value of a CSS font-family attribute.
     *
     * @returns {String}
     *  The first font name from the passed string, without quote characters.
     */
    FontCollection.getFirstFontName = function (cssFontFamily) {

        var // extract first font name from comma separated list
            fontName = $.trim(cssFontFamily.split(',')[0]);

        // Remove leading and trailing quote characters (be tolerant and allow
        // mixed quotes such as "Font' or even a missing quote character on one
        // side of the string). After that, trim again and return.
        return Utils.capitalizeWords($.trim(fontName.match(/^["']?(.*?)["']?$/)[1]));
    };

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: FontCollection });

});
