/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 */

define('io.ox/office/editframework/model/format/color', ['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    var COLOR_NAMES = {};
    var SYSTEM_COLORS = {};

    (function () {
        var tmpCN = {  aliceBlue: 'f0f8ff',         antiqueWhite: 'faebd7',      aqua: '00ffff',                  aquamarine: '7fffd4',
                       azure: 'f0ffff',             beige: 'f5f5dc',             bisque: 'ffe4c4',                black: '000000',
                       blanchedAlmond: 'ffebcd',    blue: '0000ff',              blueViolet: '8a2be2',            brown: 'a52a2a',
                       burlyWood: 'deb887',         cadetBlue: '5f9ea0',         chartreuse: '7fff00',            chocolate: 'd2691e',
                       coral: 'ff7f50',             cornflowerBlue: '6495ed',    cornsilk: 'fff8dc',              crimson: 'dc143c',
                       cyan: '00ffff',              darkBlue: '00008b',          darkCyan: '008b8b',              darkGoldenrod: 'b8860b',
                       darkGray: 'a9a9a9',          darkGreen: '006400',         darkKhaki: 'bdb76b',             darkMagenta: '8b008b',
                       darkOliveGreen: '556b2f',    darkOrange: 'ff8c00',        darkOrchid: '9932cc',            darkRed: '8b0000',
                       darkSalmon: 'e9967a',        darkSeaGreen: '8fbc8f',      darkSlateBlue: '483d8b',         darkSlateGray: '2f4f4f',
                       darkTurquoise: '00ced1',     darkViolet: '9400d3',        deepPink: 'ff1493',              deepSkyBlue: '00bfff',
                       dimGray: '696969',           dodgerBlue: '1e90ff',        firebrick: 'b22222',             floralWhite: 'fffaf0',
                       forestGreen: '228b22',       fuchsia: 'ff00ff',           gainsboro: 'dcdcdc',             ghostWhite: 'f8f8ff',
                       gold: 'ffd700',              goldenrod: 'daa520',         gray: '808080',                  green: '008000',
                       greenYellow: 'adff2f',       honeydew: 'f0fff0',          hotPink: 'ff69b4',               indianRed : 'cd5c5c',
                       indigo: '4b0082',            ivory: 'fffff0',             khaki: 'f0e68c',                 lavender: 'e6e6fa',
                       lavenderBlush: 'fff0f5',     lawnGreen: '7cfc00',         lemonChiffon: 'fffacd',          lightBlue: 'add8e6',
                       lightCoral: 'f08080',        lightCyan: 'e0ffff',         lightGoldenrodYellow: 'fafad2',  lightGray: 'd3d3d3',
                       lightGreen: '90ee90',        lightPink: 'ffb6c1',         lightSalmon: 'ffa07a',           lightSeagreen: '20b2aa',
                       lightSkyBlue: '87cefa',      lightSlateGray: '778899',    lightSteelBlue: 'b0c4de',        lightYellow: 'ffffe0',
                       lime: '00ff00',              limeGreen: '32cd32',         linen: 'faf0e6',                 magenta: 'ff00ff',
                       maroon: '800000',            mediumAquamarine: '66cdaa',  mediumBlue: '0000cd',            mediumOrchid: 'ba55d3',
                       mediumPurple: '9370d8',      mediumSeagreen: '3cb371',    mediumSlateBlue: '7b68ee',       mediumSpringGreen: '00fa9a',
                       mediumTurquoise: '48d1cc',   mediumVioletred: 'c71585',   midnightBlue: '191970',          mintCream: 'f5fffa',
                       mistyRose: 'ffe4e1',         moccasin: 'ffe4b5',          navajoWhite: 'ffdead',           navy: '000080',
                       oldLace: 'fdf5e6',           olive: '808000',             oliveDrab: '6b8e23',             orange: 'ffa500',
                       orangeRed: 'ff4500',         orchid: 'da70d6',            paleGoldenrod: 'eee8aa',         paleGreen: '98fb98',
                       paleTurquoise: 'afeeee',     paleVioletRed: 'd87093',     papayaWhip: 'ffefd5',            peachPuff: 'ffdab9',
                       peru: 'cd853f',              pink: 'ffc0cb',              plum: 'dda0dd',                  powderBlue: 'b0e0e6',
                       purple: '800080',            red: 'ff0000',               rosyBrown: 'bc8f8f',             royalBlue: '4169e1',
                       saddleBrown: '8b4513',       salmon: 'fa8072',            sandyBrown: 'f4a460',            seaGreen: '2e8b57',
                       seaShell: 'fff5ee',          sienna: 'a0522d',            silver: 'c0c0c0',                skyBlue: '87ceeb',
                       slateBlue: '6a5acd',         slateGray: '708090',         snow: 'fffafa',                  springGreen: '00ff7f',
                       steelBlue: '4682b4',         tan: 'd2b48c',               teal: '008080',                  thistle: 'd8bfd8',
                       tomato: 'ff6347',            turquoise: '40e0d0',         violet: 'ee82ee',                wheat: 'f5deb3',
                       white: 'ffffff',             whiteSmoke: 'f5f5f5',        yellow: 'ffff00',                yellowGreen: '9acd32'
        };


        for (var key in tmpCN) {
            COLOR_NAMES[key.toLowerCase()] = {
                name: key,
                rgb: tmpCN[key]
            };
        }



        var tmpSC = ['ActiveBorder', 'ActiveCaption', 'AppWorkspace', 'Background',
            'ButtonFace', 'ButtonHighlight', 'ButtonShadow',
            'ButtonText', 'CaptionText', 'GrayText', 'Highlight',
            'HighlightText', 'InactiveBorder', 'InactiveCaption',
            'InactiveCaptionText', 'InfoBackground', 'InfoText',
            'Menu', 'MenuText', 'Scrollbar', 'threeDDarkShadow',
            'ThreeDFace', 'ThreeDHighlight', 'ThreeDLightShadow',
            'ThreeDShadow', 'Window', 'WindowFrame', 'WindowText'];
        for (var i = 0; i < tmpSC.length; i++) {
            var key = tmpSC[i];
            var name = key.substring(0, 1).toLowerCase() + key.substring(1, key.length);
            SYSTEM_COLORS[key.toLowerCase()] = {
                name: name
            };
        }
    })();

    var TRANSFORM_HANDLERS = {
            /**
             * Darkens the passed CRGB color value with the provided shade value.
             *
             * @param {Object} crgb
             *  The CRGB color value.
             *
             * @param {String} shade
             *  An integer value in the interval [0, 100000] to darken the provided
             *  color. A value of 0 will change the color to black, a value of 100000
             *  will not modify the color, values in between will darken the color
             *  accordingly.
             *
             * @returns {Object}
             *  The darkened CRGB color Object.
             */
            shade: function (crgb, shade) {
                var hsl = convertCrgbToHsl(crgb);
                var shadeValue = shade / 100000;
                hsl.l = hsl.l * shadeValue;
                return convertHslToCrgb(hsl);
            },
            /**
             * Lightens the passed CRGB color value with the provided tint value.
             *
             * @param {Object} crgb
             *  The CRGB color value.
             *
             * @param {Number} tint
             *  An integer value in the interval [0, 100000] to lighten the provided
             *  color. A value of 0 will change the color to white, a value of 100000
             *  will not modify the color, values in between will lighten the color
             *  accordingly.
             *
             * @returns {Object}
             *  The lightened CRGB color Object.
             */
            tint: function (crgb, tint) {
                var hsl = convertCrgbToHsl(crgb);
                var tintValue = tint / 100000;
                hsl.l = hsl.l * tintValue + (1 - tintValue);
                return convertHslToCrgb(hsl);
            },
            /**
             * Shift the passed CRGB Luminance with the provided lumOff value.
             *
             * @param {Object} crgb
             *  The CRGB color value.
             *
             * @param {String} lumOff
             *  An integer value in the interval [-100000, 100000] to shift the provided
             *  luminance. The value 0 leaves the channel unmodified.
             *
             * @returns {Object}
             *  The CRGB color Object value restricted between 0 and 100000
             */
            lumOff: function (crgb, lumOff) {
                var hsl = convertCrgbToHsl(crgb);
                var shiftValue = lumOff / 100000;
                hsl.l = Math.min(100000, Math.max(0, hsl.l + shiftValue));
                return convertHslToCrgb(hsl);
            },
            /**
             * Modulates the passed CRGB Luminance with the provided lumMod value.
             *
             * @param {Object} crgb
             *  The CRGB color value.
             *
             * @param {String} lumMod
             *  An integer value in the interval [0, *] to modulate the provided
             *  luminance. The value 100000 leaves the channel unmodified.
             *
             * @returns {Object}
             *  The CRGB color Object value restricted between 0 and 100000
             */
            lumMod: function (crgb, lumMod) {
                var hsl = convertCrgbToHsl(crgb);
                var modValue = lumMod / 100000;
                hsl.l = Math.min(100000, hsl.l * modValue);
                return convertHslToCrgb(hsl);
            },

            /**
             * Modulates the passed CRGB Saturation with the provided satMod value.
             *
             * @param {Object} crgb
             *  The CRGB color value.
             *
             * @param {String} satMod
             *  An integer value in the interval [0, *] to modulate the provided
             *  satuaration. The value 0 leaves the channel unmodified.
             *
             * @returns {Object}
             *  The CRGB color Object value restricted between 0 and 100000
             */
            satMod: function (crgb, satMod) {
                var hsl = convertCrgbToHsl(crgb);
                var modValue = satMod / 100000;
                hsl.s = Math.min(100000, hsl.s * modValue);
                return convertHslToCrgb(hsl);
            },

            /**
             * Set the passed CRGB Alpha with the provided alpha value.
             *
             * @param {Object} crgb
             *  The CRGB color value.
             *
             * @param {String} alpha
             *  An integer value in the interval [0, 100000] to set the provided
             *  alpha.
             *
             * @returns {Object}
             *  The CRGB color Object value restricted between 0 and 100000
             */
            alpha: function (crgb, alpha) {
                var res = _.copy(crgb);
                res.a = alpha;
                return res;
            },
        };


    // private static functions ===============================================

    /**
     * Converts SystemColor Name to its rgb color value
     *
     * @param {String} sysColor as name
     *
     * @returns {String}
     * the correct rgb hex string.
     *
     */
    function convertSysColorToRGB(sysColor) {
        var lower = sysColor.toLowerCase();
        var entry = SYSTEM_COLORS[lower];
        if (entry) {
            if (!entry.rgb) {
                var div = $('div');
                div.css('color', lower);
                entry.rgb = div.css('color');
            }
            return entry.rgb;
        }
        Utils.warn('System Color "' + sysColor + '" is not defined in Color.SYSTEM_COLORS');
        return '9E9E9E';
    }

    //Presets not supported
    /*
    function convertPresetColorToRGB(preColor) {
        var holder = COLOR_NAMES[preColor.toLowerCase()];
        Utils.warn('convertPresetColorToRGB', preColor, holder);
        if (holder) {
            return holder.rgb;
        }
        Utils.warn('Preset Color "' + preColor + '" is not defined in Color.COLOR_NAMES');
        return '9E9E9E';
    }
    */
    /**
     * Converts a hex based rgb color value to a
     * number.
     *
     * @param {String} rgbColor
     * A hex based rgb color value (RRGGBB).
     *
     * @returns {Number}
     * The integer based value of the hex string.
     */
    function convertRgbColorToNumber(rgbColor) {
        return parseInt(rgbColor, 16);
    }

    /**
     * Converts a RGB color value to a HSL color object
     * which contains the attributes h (hue),
     * s (saturation) and l (luminance).
     *
     * @param {Object} crgb
     * The crgb color object to be converted to the HSL
     * color model.
     *
     * @returns {Object}
     * The HSL color object which contains h,s,l attributes
     * based on the provided rgb color value.
     */
    function convertCrgbToHsl(crgb) {
        var r, g, b, min, max, h, s, l;

        r = crgb.r / 100000;
        g = crgb.g / 100000;
        b = crgb.b / 100000;

        min = Math.min(r, g, b);
        max = Math.max(r, g, b);

        l = (min + max) / 2;
        s = (min === max) ? 0 : (l < 0.5) ? (max - min) / (max + min) : (max - min) / (2.0 - max - min);
        h = 0;
        if (s !== 0) {
            h = ((max === r) ? (g - b) / (max - min) : (max === g) ? 2.0 + (b - r) / (max - min) : 4.0 + (r - g) / (max - min)) * 60;
            h = ((h < 0) ? h + 360 : h) / 360;
        }

        return { h: h, s: s, l: l, a: crgb.a };
    }

    /**
     * Converts a RGB color hex string to a HSL color object
     * which contains the attributes h (hue),
     * s (saturation) and l (luminance)
     * with gamma-correction.
     *
     * @param {String} rgb
     * The rgb color hex string to be converted to the HSL
     * color model.
     *
     * @returns {Object}
     * The HSL color object which contains h,s,l attributes
     * based on the provided rgb hex string.
     */
    function convertRgbToCrgb(rgb) {

        var rgbValue = convertRgbColorToNumber(rgb);

        var r = ((rgbValue & 16711680) >> 16) / 255;
        var g = ((rgbValue & 65280) >> 8) / 255;
        var b = (rgbValue & 255) / 255;

        return { r: invGammaChannel(r) * 100000, g: invGammaChannel(g) * 100000, b: invGammaChannel(b) * 100000, a: 100000 };
    }

    /**
     * Converts a CRGB color object to a CSS compatible string
     * with gamma-correction
     *
     * @param {Object} crgb
     * The crgb color object to be converted to css string
     *
     * @returns {String}
     * The css color string in form of '#hex',
     * 'rgba()' or 'transparent'
     */
    function convertCrgbToCss(crgb) {
        var r = Math.round(gammaChannel(crgb.r / 100000) * 255);
        var g = Math.round(gammaChannel(crgb.g / 100000) * 255);
        var b = Math.round(gammaChannel(crgb.b / 100000) * 255);

        switch (crgb.a) {
        case 0:
            return 'transparent';
        case 100000:
            var rgbNr = ((r << 16) + (g << 8) + b);
            return '#' + convertNumberToRgbColor(rgbNr);
        default:
            return 'rgba(' + r + ',' + g + ',' + b + ',' + (crgb.a / 100000) + ')';
        }
    }

    /**
     * corrects the given channel from crgb to rgb
     */
    function gammaChannel(channel) {
        return channel; //TODO: gamma correction is disabled Math.pow(channel, 1 / 2.3);
    }
    /**
     * corrects the given channel from rgb to crgb
     */
    function invGammaChannel(channel) {
        return channel;  //TODO: gamma correction is disabled Math.pow(channel, 2.3);
    }

    /**
     * Converts a HSL color object to a rgb color value.
     *
     * @param {Object} hsl
     * The HSL color object which contains the color
     * attributes h, s, l
     *
     * @returns {Object}
     * The rgb color object calculated from the hsl color object.
     */
    function convertHslToCrgb(hsl) {
        var r, g, b;

        if (hsl.s === 0) {
            r = g = b = Math.round(hsl.l * 100000);
        }
        else {
            var t1, t2, tr, tg, tb;

            t1 = (hsl.l < 0.5) ? hsl.l * (1.0 + hsl.s) : hsl.l + hsl.s - hsl.l * hsl.s;
            t2 = 2 * hsl.l - t1;

            tr = hsl.h + (1 / 3);
            tg = hsl.h;
            tb = hsl.h - (1 / 3);

            tr = (tr < 0) ? tr + 1 : (tr > 1) ? tr - 1 : tr;
            tg = (tg < 0) ? tg + 1 : (tg > 1) ? tg - 1 : tg;
            tb = (tb < 0) ? tb + 1 : (tb > 1) ? tb - 1 : tb;

            var convChannel = function conv(ch) {
                if ((6 * ch) < 1)
                    return t2 + (t1 - t2) * 6 * ch;
                else if ((2 * ch) < 1)
                    return t1;
                else if ((3 * ch) < 2)
                    return t2 + (t1 - t2) * ((2 / 3) - ch) * 6;
                else
                    return t2;
            };

            r = Math.round(convChannel(tr) * 100000);
            g = Math.round(convChannel(tg) * 100000);
            b = Math.round(convChannel(tb) * 100000);
        }

        return { r: r, g: g, b: b, a: hsl.a };
    }

    /**
     * Calculates the resulting RGB color from a source RGB color value
     * and a transformation array defining the transformation rules.
     *
     * @param {String} rgbColor
     *  The source color as a hexadecimal string (RRGGBB).
     *
     * @param {Array} transformations
     *  An array with transformation objects describing the
     *  transformation rules to be applied to the provided rgbColor.
     *
     * @returns {String}
     *  The resulting RGB color as a HEX string.
     */
    function transformRGBColor(rgbColor, transformations) {
        var crgb = convertRgbToCrgb(rgbColor);
        return transformCRGBColor(crgb, transformations);
    }

    /**
     * Calculates the resulting CRGB color from a source CRGB color value
     * and a transformation array defining the transformation rules.
     *
     * @param {Object} crgbColor
     *  The source color as a object
     *
     * @param {Array} transformations
     *  An array with transformation objects describing the
     *  transformation rules to be applied to the provided rgbColor.
     *
     * @returns {Object}
     *  The resulting CRGB color object.
     */
    function transformCRGBColor(crgbColor, transformations) {
        var crgb = crgbColor;
        if (transformations && (transformations.length > 0)) {
            _(transformations).each(function (transformation) {
                var tHandler = TRANSFORM_HANDLERS[transformation.type];
                if (!tHandler) {
                    Utils.warn('Color.transformRGBColor(): unknown color transformation: ' + transformation.type);
                } else {
                    crgb = tHandler(crgb, transformation.value);
                }
            });
        }
        return crgb;
    }

    /**
     * Calculates the resulting CRGB color from a source CRGB color value
     * and a transformation array defining the transformation rules.
     *
     * @param {Object} hslColor
     *  The source color as a object
     *
     * @param {Array} transformations
     *  An array with transformation objects describing the
     *  transformation rules to be applied to the provided rgbColor.
     *
     * @returns {Object}
     *  The resulting HSL color object.
     */
    function transformHSLColor(hslColor, transformations) {
        var crgb = convertHslToCrgb(hslColor);
        return transformCRGBColor(crgb, transformations);
    }

    /**
     * Converts a number to a hex based rgb color string.
     *
     * @param {Number} colorValue
     * A rgb color value.
     *
     * @returns {String}
     * Hex based color string (RRGGBB).
     */
    function convertNumberToRgbColor(colorValue) {
        var color = colorValue.toString(16);
        // MUST be 6-digits with leading zeros, otherwise, CSS *appends* zeros instead of prepending them
        while (color.length < 6) { color = '0' + color; }
        return color;
    }

    // static class Color =====================================================

    /**
     * Predefined color objects.
     */
    var Color = {
            BLACK: { type: 'rgb', value: '000000' },
            WHITE: { type: 'rgb', value: 'FFFFFF' },
            RED:   { type: 'rgb', value: 'FF0000' },
            AUTO:  { type: 'auto' }
        };

    /**
     * Converts the passed attribute color object to a CSS color value.
     *
     * @param {Object} color
     *  The color object as used in operations.
     *
     * @param {String|Object} autoColor
     *  Additional information needed to resolve the automatic color. Can be a
     *  string for predefined automatic colors, or an explicit color object
     *  with a value other than the automatic color. Predefined identifiers are
     *  - 'text': for text colors (mapped to to black),
     *  - 'line': for line colors (e.g. borders, maps to black),
     *  - 'fill': for fill colors (e.g. table cells, maps to transparent).
     *
     * @param {Theme} theme
     *  The theme object used to map scheme color names to color values.
     *
     * @returns {String}
     *  The CSS color value converted from the passed color object.
     */
    Color.getCssColor = function (color, autoColor, theme) {

        var type = Utils.getStringOption(color, 'type', 'auto'),
            transformations = Utils.getArrayOption(color, 'transformations', []),
            cRgbColor = null;

        switch (type) {
        case 'system':
            cRgbColor = transformRGBColor(convertSysColorToRGB(color.value), transformations);
            break;
        //Presets not supported
        /*
        case 'preset':
            rgbColor = transformRGBColor(convertPresetColorToRGB(color.value), transformations);
            break;
        */
        case 'crgb':
            cRgbColor = transformCRGBColor(color.value, transformations);
            break;
        case 'hsl':
            cRgbColor = transformHSLColor(color.value, transformations);
            break;
        case 'rgb':
            cRgbColor = transformRGBColor(color.value, transformations);
            break;
        case 'scheme':
            if (theme) {
                if (color.value === 'style') {
                    Utils.warn('Color.getCssColor(): scheme color "style" handling is not implemented yet. current behavior is BLACK');
                    cRgbColor = convertRgbToCrgb(Color.BLACK.value);
                } else {
                    cRgbColor = transformRGBColor(theme.getSchemeColor(color.value), transformations);
                }
            } else {
                cRgbColor = convertRgbToCrgb(color.fallbackValue);
            }
            break;
        case 'auto':
            if (_.isObject(autoColor)) {
                return Color.getCssColor(autoColor, 'fill');
            }
            switch (autoColor) {
            case 'text':
            case 'line':
                cRgbColor = convertRgbToCrgb(Color.BLACK.value);
                break;
            case 'fill':
                // transparent: keep rgbColor empty
                break;
            default:
                Utils.warn('Color.getCssColor(): unknown replacement for auto color: ' + autoColor);
                cRgbColor = convertRgbToCrgb(Color.BLACK.value);
            }
            break;
        default:
            cRgbColor = _.isString(color.fallbackValue) ? convertRgbToCrgb(color.fallbackValue) : null;
            Utils.error('Color.getCssColor(): unknown color type: ' + type);
        }

        if (_.isObject(cRgbColor)) {
            return convertCrgbToCss(cRgbColor);
        }

        return 'transparent';
    };

    /**
     * Determine if provided color has theme based color value
     *
     * @param color
     * A color object
     *
     * @returns {Boolean}
     * true if color is theme based otherwise false
     */
    Color.isThemeColor = function (color) {
        var type = Utils.getStringOption(color, 'type', 'none');
        return (type === 'scheme');
    };

    /**
     * Determine if provided color has type auto
     *
     * @param color
     * A color object
     *
     * @returns {Boolean}
     * true if color has type auto otherwise false
     */
    Color.isAutoColor = function (color) {
        var type = Utils.getStringOption(color, 'type', 'auto');
        return (type === 'auto');
    };

    /**
     * Returns whether the passed color resolves to full transparency.
     *
     * @param {Object} color
     *  The color object.
     *
     * @param {String} context
     *  The context needed to resolve the color type 'auto'.
     *
     * @returns {Boolean}
     *  Whether the passed color represents full transparency.
     */
    Color.isTransparentColor = function (color, context) {
        return Color.getCssColor(color, context) === 'transparent';
    };

    /**
     * Determine if the provided color is dark or light
     *
     * @param {String} rgbColor
     * The rgb color as hex value string (RRGGBB) with/without trailing #
     *
     * @returns {Boolean}
     * True if the color is dark otherwise false.
     */
    Color.isDark = function (rgbColor) {
        if (rgbColor && (6 <= rgbColor.length) && (rgbColor.length <= 7)) {
            var hexString = rgbColor.length === 6 ? rgbColor : rgbColor.substring(1);
            var rgbColorValue = convertRgbColorToNumber(hexString);
            var luminance = ((((rgbColorValue & 0x00ff0000) >> 16) * 5036060) +
                             (((rgbColorValue & 0x0000ff00) >> 8) * 9886846) +
                              ((rgbColorValue & 0x000000ff) * 1920103)) / (1 << 24);

            return luminance <= 60;
        }

        return false;
    };

    /**
     * Comparing two color values. These are equal if:
     * - both color parameter are Color objects
     * - both have the same 'type'
     * - both have no property 'value' or both have a property 'value' and the value of 'value' is equal
     * - both have no property 'transformations' or both have a property 'transformations' and the value of 'transformations' is equal
     *
     * @param {Color} color1
     * The first color to compare
     *
     * @param {Color} color2
     * The second color to compare
     *
     * @returns {Boolean}
     * Whether the two colors are equal.
     */
    Color.isEqual = function (color1, color2) {
        return (color1 &&
                color2 &&
                ('type' in color1) &&
                ('type' in color2) &&
                (color1.type === color2.type) &&
                ((!('value' in color1) && (!('value' in color2))) || (('value' in color1) && ('value' in color2) && (color1.value === color2.value))) &&
                ((!('transformations' in color1) && (!('transformations' in color2))) || (('transformations' in color1) && ('transformations' in color2) && _.isEqual(color1.transformations, color2.transformations))));
    };

    /**
     * Converts a CSS color string into a Color object.
     * Accepts the following CSS colors as input:
     *  - predefined color names e.g. 'red'
     *  - hex  colors e.g. #RRGGBB
     *  - rgb  colors e.g. rgb(R, G, B)
     *  - rgba colors e.g. rgb(R, G, B, A)
     * Returns a Color of type auto if a rgba color defines a transparent alpha.
     *
     * @param {String} cssColor
     *  The CSS color string.
     *
     * @returns {Object|null}
     *  A Color object or null.
     */
    Color.convertCssColorToRgbColor = function (cssColor) {
        var parts;

        if (!_.isString(cssColor)) { return null; }

        // predefined color names

        var preset = COLOR_NAMES[cssColor.toLowerCase()];
        if (preset) {
            //Presets not supported
            //return { type: 'preset', value: preset.name };
            return { type: 'rgb', value: preset.rgb };
        }

        // hex color
        parts = cssColor.match(/^#([0-9,A-F]{6})$/i);
        if (_.isArray(parts) && parts[1]) {
            return { type: 'rgb', value: parts[1] };
        }

        // rgb(a) color
        parts = cssColor.match(/^rgba?\(\s*(\d+)\s*,\s*(\d+)\s*,\s*(\d+)(?:\s*,\s*(1|0|0?\.\d+))?\s*\)$/);
        if (!_.isArray(parts)) { return null; }

        // transparent rgba color
        if (parts[4] === '0') { return Color.AUTO; }

        for (var i = 1; i <= 3; ++i) {
            parts[i] = parseInt(parts[i], 10).toString(16).toUpperCase();
            if (parts[i].length === 1) { parts[i] = '0' + parts[i]; }
        }

        return { type: 'rgb', value: parts[1] + parts[2] + parts[3] };
    };

    // exports ================================================================

    return Color;

});
