/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/format/attributedmodel',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/model/modelobject',
     'io.ox/office/editframework/model/format/documentstyles'
    ], function (Utils, ModelObject, DocumentStyles) {

    'use strict';

    // class AttributedModel ==================================================

    /**
     * An abstract model object containing formatting attributes and the
     * reference to a style sheet of a specific attribute family.
     *
     * Triggers the following events:
     * - 'change:attributes': After the explicit attributes or the style sheet
     *      identifier have been changed, either directly by using the method
     *      'AttributedModel.setAttributes()', or indirectly after the style
     *      sheet has been changed. The event handlers receive the new
     *      resulting merged attribute set, and the old merged attribute set.
     * No events will be generated while the application imports the document.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {EditApplication} app
     *  The application instance containing this attributed model object.
     *
     * @param {Object} [initAttributes]
     *  An attribute set with initial formatting attributes for the attributed
     *  model object.
     *
     * @param {Object} [initOptions]
     *  A map with additional options controlling the behavior of this
     *  instance. Supports all attributes also supported by the base class
     *  ModelObject, and the following additional options:
     *  @param {String} [initOptions.styleFamily]
     *      The attribute family of style sheets that can be referred by this
     *      attributed model object. If omitted, this object supports explicit
     *      attributes only. All attribute families that are supported by these
     *      style sheets will also be supported by this model object.
     *  @param {String|Array} [initOptions.additionalFamilies]
     *      The attribute families of explicit attributes supported by this
     *      attributed model object. The style family specified in the
     *      'styleFamily' option, and all additional attribute families
     *      supported by the style sheet container do not need to be repeated
     *      here.
     *  @param {AttributedModel} [initOptions.parentModel]
     *      A parent model object (instance of AttributedModel) that supports
     *      all or a few of the attribute families supported by this instance.
     *      The merged attributes of that parent model will be added to the own
     *      merged attribute set before adding the own style attributes and
     *      explicit attributes. This works also recursively (the parent model
     *      may define its own parent model).
     */
    function AttributedModel(app, initAttributes, initOptions) {

        var // self reference
            self = this,

            // the attribute family of the style sheets
            styleFamily = Utils.getStringOption(initOptions, 'styleFamily', null, true),

            // the parent model used to build the merged attribute set
            parentModel = Utils.getObjectOption(initOptions, 'parentModel'),

            // the global style sheets container of the document model
            documentStyles = app.getModel().getDocumentStyles(),

            // the style sheets container for the style family
            styleSheets = styleFamily ? documentStyles.getStyleSheets(styleFamily) : null,

            // all attribute families supported by instances of this class
            supportedFamilies = null,

            // the explicit attributes of this model object
            explicitAttributes = {},

            // the cached merged attributes of this model object
            mergedAttributes = null;

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Recalculates the merged attribute set.
         */
        function calculateMergedAttributes(options) {

            var // the identifier of the style sheet
                styleId = styleSheets ? Utils.getStringOption(explicitAttributes, 'styleId', '') : null,
                // remember the old merged attributes
                oldMergedAttributes = mergedAttributes;

            // start with defaults and style sheet attributes
            mergedAttributes = documentStyles.getDefaultAttributes(supportedFamilies);

            // add merged attributes of parent model object
            if (parentModel) {
                documentStyles.extendAttributes(mergedAttributes, parentModel.getMergedAttributes());
                _.chain(mergedAttributes).keys().difference(supportedFamilies).each(function (family) {
                    delete mergedAttributes[family];
                });
            }

            // add style sheet attributes
            if (_.isString(styleId)) {
                documentStyles.extendAttributes(mergedAttributes, styleSheets.getStyleSheetAttributes(styleId));
            }

            // add the explicit attributes (protect resulting style sheet identifier)
            styleId = mergedAttributes.styleId;
            documentStyles.extendAttributes(mergedAttributes, explicitAttributes);
            mergedAttributes.styleId = styleId;

            // notify all listeners
            if (Utils.getBooleanOption(options, 'notify', false)) {
                self.trigger('change:attributes', mergedAttributes, oldMergedAttributes);
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the explicit attribute set of this model object.
         *
         * @returns {Object}
         *  An (incomplete) attribute set (a map of attribute maps with
         *  name/value pairs, keyed by the attribute families, optionally
         *  containing the style sheet reference in the 'styleId' property),
         *  containing values for all attributes that have been explicitly set.
         */
        this.getExplicitAttributes = function () {
            return _.copy(explicitAttributes, true);
        };

        /**
         * Returns the merged attribute set of this model object.
         *
         * @returns {Object}
         *  A complete attribute set (a map of attribute maps with name/value
         *  pairs, keyed by the attribute families, additionally containing the
         *  style sheet reference in the 'styleId' property), containing values
         *  for all known attributes, collected from the document default
         *  values, the style sheet currently referenced by this model object,
         *  and the explicit attributes of this model object.
         */
        this.getMergedAttributes = function () {
            return mergedAttributes;
        };

        /**
         * Changes and/or removes specific explicit formatting attributes, or
         * the style sheet reference of this model object.
         *
         * @param {Object} attributes
         *  An (incomplete) attribute set (a map of attribute maps with
         *  name/value pairs, keyed by the attribute families, optionally
         *  containing the style sheet reference in the 'styleId' property)
         *  with all formatting attributes to be changed. To clear an explicit
         *  attribute value (thus defaulting to the current style sheet), the
         *  respective value in this attribute set has to be set explicitly to
         *  the null value. Attributes missing in this attribute set will not
         *  be modified.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {String} [options.notify='auto']
         *      Specifies whether to trigger the event 'change:attributes'. If
         *      set to 'auto' or omitted, the event will be triggered if at
         *      least one formatting attribute has changed its value. If set to
         *      'always', the event will always be triggered without comparing
         *      the old and new attribute values. If set to 'never', no event
         *      will be triggered.
         *
         * @returns {Boolean}
         *  Whether any attributes have actually been changed.
         */
        this.setAttributes = function (attributes, options) {

            var // whether to trigger the event
                notify = Utils.getStringOption(options, 'notify', 'auto'),
                // whether any attribute has been changed
                changed = false;

            // adds or removes an attribute value, updates the changed flag
            function changeValue(values, name, value) {
                if (_.isNull(value)) {
                    if (name in values) {
                        delete values[name];
                        changed = true;
                    }
                } else {
                    if (!_.isEqual(values[name], value)) {
                        values[name] = _.isObject(value) ? _.copy(value, true) : value;
                        changed = true;
                    }
                }
            }

            // add/remove the style sheet identifier
            if (_.isString(attributes.styleId) || _.isNull(attributes.styleId)) {
                changeValue(explicitAttributes, 'styleId', attributes.styleId);
            }

            // add/remove the passed explicit attributes of all supported attribute families
            _(supportedFamilies).each(function (family) {

                var // attribute definitions of the current family
                    definitions = documentStyles.getAttributeDefinitions(family),
                    // target attribute map for the current family
                    values = null;

                // check if new attributes exist in the passed attribute set
                if (_.isObject(attributes[family])) {

                    // add an attribute map for the current family
                    values = explicitAttributes[family] || (explicitAttributes[family] = {});

                    // update the attribute map with the new attribute values
                    _(attributes[family]).each(function (value, name) {
                        if (DocumentStyles.isRegisteredAttribute(definitions, name)) {
                            changeValue(values, name, value);
                        }
                    });

                    // remove empty attribute value maps completely
                    if (_.isEmpty(values)) {
                        delete explicitAttributes[family];
                    }
                }
            });

            // recalculate merged attributes and notify all listeners
            if (changed || !mergedAttributes) {
                calculateMergedAttributes({ notify: (changed && (notify === 'auto')) || (notify === 'always') });
            }

            return changed;
        };

        // initialization -----------------------------------------------------

        // get all supported attribute families
        supportedFamilies = Utils.getStringOption(initOptions, 'additionalFamilies') || Utils.getArrayOption(initOptions, 'additionalFamilies', []);
        supportedFamilies = _.getArray(supportedFamilies);
        if (styleSheets) { supportedFamilies = supportedFamilies.concat(styleSheets.getSupportedFamilies()); }
        supportedFamilies = _.unique(supportedFamilies);

        // set the attributes passed to the constructor (filters by supported attributes)
        this.setAttributes(_.isObject(initAttributes) ? initAttributes : {}, { notify: 'never' });

        // listen to changes in the parent object, update the merged attributes and notify own listeners
        if (parentModel) {
            this.listenTo(parentModel, 'triggered', function () {
                calculateMergedAttributes({ notify: true });
            });
        }

        // listen to style sheet changes, update the merged attributes and notify own listeners
        if (styleSheets) {
            this.listenTo(styleSheets, 'triggered', function () {
                calculateMergedAttributes({ notify: true });
            });
        }

        // update merged attributes silently once after import
        if (!app.isImportFinished()) {
            // use listenTo(), this object may be destroyed before import finishes
            this.listenTo(app, 'docs:import:success', function () {
                calculateMergedAttributes();
            });
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = parentModel = documentStyles = styleSheets = null;
        });

    } // class AttributedModel

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: AttributedModel });

});
