/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2011
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 */

define('io.ox/office/editframework/app/topbaraction',
    ['io.ox/core/extPatterns/stage',
     'io.ox/office/tk/utils',
     'gettext!io.ox/office/files'
    ], function (Stage, Utils, gt) {

    'use strict';

    // static class TopbarAction ==============================================

    var TopbarAction = {};

    /**
     * Creates a top-bar entry for the specific OX Documents application.
     *
     * @param {String} appBaseName
     *  The base name of the application (name of the application source code
     *  directory).
     *
     * @param {Object} [initOptions]
     *  A map with options controlling the behavior of the created extensions
     *  in the Files application. The following options are supported:
     *  @param {Number} [initOptions.index=0]
     *      The index is used to specify the order of the new top-bar entry.
     */
    TopbarAction.create = function (appBaseName, initOptions) {

        var // the module name of the edit application
            MODULE_NAME = 'io.ox/office/' + appBaseName;

        // private methods ----------------------------------------------------

        /**
         * Launches a new OX Documents application with the passed options.
         */
        function launchApplication(launchOptions) {
            ox.launch(MODULE_NAME + '/main', launchOptions);
        }

        /**
         * Provides the translated application titles.
         *
         * @attention:
         *  This function must not be called directly while executing the
         *  TopbarAction.create() method. The translation system is not
         *  correctly initialized yet.
         *
         * @param {String} appBaseName
         *  The base name of the application.
         *
         * @param {String}
         *  The translated title of the application.
         */
        function getApplicationTitle() {
            switch (appBaseName) {
            case 'text':
                return gt('Text');
            case 'spreadsheet':
                return gt('Spreadsheet');
            case 'presentation':
                return gt('Presentation');
            }
            Utils.warn('TopbarAction.getApplicationTitle(): invalid application base name: "' + appBaseName + '"');
            return '';
        }

        // register stage listener that gets called when the provided index is reached
        new Stage('io.ox/core/stages', {
            id: 'ox_documents_' + appBaseName,
            index: 700 + Utils.getNumberOption(initOptions, 'index', 0),
            run: function () {
                require('io.ox/core/main').addLauncher(
                'left',
                getApplicationTitle(),
                function () {
                    launchApplication({ action: 'new', folderId: require('settings!io.ox/core').get('folder/infostore').toString() });
                });
            }
        });
    };

    // exports ================================================================

    return TopbarAction;
});
