/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * © 2012 Open-Xchange Inc., Tarrytown, NY, USA. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/app/fileactions',
    ['io.ox/core/extensions',
     'io.ox/core/extPatterns/links',
     'io.ox/office/tk/utils',
     'io.ox/office/baseframework/app/extensionregistry',
     'io.ox/core/api/folder',
     'gettext!io.ox/office/files'
    ], function (ext, links, Utils, ExtensionRegistry, FolderAPI, gt) {

    'use strict';

    // static private functions ===============================================

    function disableActionForInvalidFiles(actionId) {
        new links.Action(actionId, {
            id: 'disable_action',
            index: 'first',
            requires: function (data) {
                if (ExtensionRegistry.isError(data.baton.data.filename)) {
                    data.stopPropagation();
                    return false;
                }
            }
        });
    }

    // static class FileActions ===============================================

    var FileActions = {};

    // methods ----------------------------------------------------------------

    /**
     * Creates all required actions and links in the OX Files application for a
     * specific OX Documents application.
     *
     * @param {String} appBaseName
     *  The base name of the application (name of the application source code
     *  directory).
     *
     * @param {Object} options
     *  A map with options controlling the behavior of the created extensions
     *  in the Files application. The following options are supported:
     *  @param {String} options.newDocumentLabel
     *      The string shown for the 'Create new document' action link inserted
     *      into the Files tool bar.
     *  @param {Number} options.newDocumentIndex
     *      The index of the 'Create new document' action link that will be
     *      inserted into the Files tool bar. The 'Create new document' action
     *      links of the various OX Document applications will be ordered
     *      increasing by this index.
     */
    FileActions.create = function (appBaseName, options) {

        var // root for extension points in Files application
            ACTION_POINT = 'io.ox/files/actions/office/' + appBaseName,
            // the module name of the edit application
            MODULE_NAME = 'io.ox/office/' + appBaseName;

        // private methods ----------------------------------------------------

        /**
         * Launches a new OX Documents application with the passed options.
         */
        function launchApplication(launchOptions) {
            ox.launch(MODULE_NAME + '/main', launchOptions);
        }


        /**
         * calls data of the folder in the baton,
         * returns false if the folder is the trash folder
         * if data is not loaded yet it return a deferred/promise with the correct check inside
         */
        function isBatonEditable(baton) {
            function isBatonDataEditable(data) {
                return !FolderAPI.is('trash', data);//no new files in trash folders
            }

            if (baton.data.length) {
                return isBatonDataEditable(baton.data);
            } else {
                if (!baton.app) {
                    Utils.warn('could not check, if baton is in Trash folder, because baton has no app entry!', baton);
                    return true;
                } else {
                    return baton.app.folder.getData().then(isBatonDataEditable);
                }
            }
        }

        // initialization -----------------------------------------------------

        // Creates a new empty document in the current folder of the Files application.
        if (ExtensionRegistry.supportsEditMode(appBaseName)) {
            new links.Action(ACTION_POINT + '/new', {
                requires: function (e) {
                	if (e.collection.has('create')) { // bug 32367
	                    return isBatonEditable(e.baton);
	                }
                },
                action: function (baton) {
                    launchApplication({ action: 'new', folderId: baton.app.folder.get() });
                }
            });
        }

        // Loads an existing document from the current file in the Files application.
        // If the document requires conversion to a supported file format, creates the
        // converted file and loads it.
        new links.Action(ACTION_POINT + '/edit', {
            requires: function (e) {
                var filename = e.context.filename;
                if (e.collection.has('one', 'read', 'modify') && ExtensionRegistry.isEditable(filename, MODULE_NAME) && !ExtensionRegistry.isTemplate(filename, MODULE_NAME)) {
                    return isBatonEditable(e.baton);
                }
            },
            filter: function (obj) {
                return ExtensionRegistry.isEditable(obj.filename, MODULE_NAME);
            },
            action: function (baton) {
                if (ExtensionRegistry.isNative(baton.data.filename)) {
                    launchApplication({ action: 'load', file: baton.data });
                } else {
                    launchApplication({ action: 'convert', folderId: baton.data.folder_id, templateFile: baton.data, preserveFileName: true });
                }
            }
        });

        new links.Action(ACTION_POINT + '/newfromtemplate', {
            requires: function (e) {
                var filename = e.context.filename;
                if (e.collection.has('one', 'read') && !ExtensionRegistry.isError(filename) && ExtensionRegistry.isTemplate(filename, MODULE_NAME)) {
                    return isBatonEditable(e.baton);
                }
            },
            filter: function (obj) {
                return ExtensionRegistry.isTemplate(obj.filename, MODULE_NAME);
            },
            action: function (baton) {
                launchApplication({ action: 'convert', template: true, folderId: baton.data.folder_id, templateFile: baton.data, initial_filename: gt('unnamed') });
            }
        });

        // Creates a new document as copy of the current file in the Files application.
        if (ExtensionRegistry.supportsEditMode(appBaseName)) {
            new links.Action(ACTION_POINT + '/editasnew', {
                requires: function (e) {
                    var filename = e.context.filename;
                    if (e.collection.has('one', 'read') && !ExtensionRegistry.isError(filename) && ExtensionRegistry.isEditable(filename, MODULE_NAME) && !ExtensionRegistry.isTemplate(filename, MODULE_NAME)) {
                        return isBatonEditable(e.baton);
                    }
                },
                filter: function (obj) {
                    return ExtensionRegistry.isEditable(obj.filename, MODULE_NAME);
                },
                action: function (baton) {
                    var convert = ExtensionRegistry.isConvertible(baton.data.filename);
                    launchApplication({ action: convert ? 'convert' : 'new', folderId: baton.data.folder_id, templateFile: baton.data });
                }
            });

            new links.Action(ACTION_POINT + '/edittemplate', {
                requires: function (e) {
                    var filename = e.context.filename;
                    if (e.collection.has('one', 'read', 'modify') && !ExtensionRegistry.isError(filename) && ExtensionRegistry.isTemplate(filename, MODULE_NAME) && !ExtensionRegistry.isTemplateLocked(filename, MODULE_NAME) && ExtensionRegistry.isEditable(filename, MODULE_NAME)) {
                        return isBatonEditable(e.baton);
                    }
                },
                filter: function (obj) {
                    return ExtensionRegistry.isEditable(obj.filename, MODULE_NAME);
                },
                action: function (baton) {
                    if (ExtensionRegistry.isNative(baton.data.filename)) {
                        launchApplication({ action: 'load', file: baton.data });
                    } else {
                        launchApplication({ action: 'convert', folderId: baton.data.folder_id, templateFile: baton.data, preserveFileName: true });
                    }
                }
            });
        }

        // An action link in the main tool bar of the Files application used
        // to create a new empty document.
        new links.ActionLink('io.ox/files/links/toolbar/default', {
            index: 200 + Utils.getIntegerOption(options, 'newDocumentIndex', 0),
            id: appBaseName + '_new',
            label: Utils.getStringOption(options, 'newDocumentLabel', ''),
            ref: ACTION_POINT + '/new'
        });

        // An action link in the details view of a file used to load the file.
        ext.point('io.ox/files/links/inline').extend(new links.Link({
            id: appBaseName + '_edit',
            index: 150,
            prio: 'hi',
            mobile: 'hi',
            section: 'edit',
            label: gt('Edit'),
            ref: ACTION_POINT + '/edit'
        }));

        // An action link in the details view of a file used to create a new
        // document based on that file.
        ext.point('io.ox/files/links/inline').extend(new links.Link({
            id: appBaseName + '_editasnew',
            prio: 'lo',
            mobile: 'lo',
            index: 'first',
            section: 'edit',
            label: gt('Edit as new'),
            ref: ACTION_POINT + '/editasnew'
        }));

        ext.point('io.ox/files/links/inline').extend(new links.Link({
            id: appBaseName + '_newfromtemplate',
            prio: 'hi',
            mobile: 'hi',
            index: 150,
            label: gt('New from template'),
            ref: ACTION_POINT + '/newfromtemplate'
        }));

        ext.point('io.ox/files/links/inline').extend(new links.Link({
            id: appBaseName + '_edittemplate',
            prio: 'lo',
            mobile: 'lo',
            index: 'first',
            section: 'edit',
            label: gt('Edit template'),
            ref: ACTION_POINT + '/edittemplate'
        }));

    };

    // static initialization ==================================================

    disableActionForInvalidFiles('io.ox/files/actions/add-to-portal');
    disableActionForInvalidFiles('io.ox/files/actions/publish');

    // exports ================================================================

    return FileActions;

});
