/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/drawinglayer/view/drawingcontrols',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/view/basecontrols',
     'io.ox/office/drawinglayer/view/chartcolorsetpicker',
     'io.ox/office/drawinglayer/view/charttypepicker',
     'gettext!io.ox/office/drawing'
    ], function (Utils, BaseControls, ChartColorsetPicker, ChartTypePicker, gt) {

    'use strict';

    var CHART_LEGEND_VALUES = {
            off: /*#. Chart legend is hidden */ gt('Off'),
            top: /*#. legend on top of the chart */ gt('Top'),
            bottom: /*#. legend on bottom of the chart */ gt('Bottom'),
            left: /*#. legend on left of the chart */ gt('Left'),
            right: /*#. legend on right of the chart */ gt('Right'),
            topright: /*#. legend on top-right of the chart */ gt('Top right')
        };

    // private constants --------------------------------------------------------------

    var // class name shortcuts
        RadioList = BaseControls.RadioList;

    // static class DrawingControls ===========================================

    /**
     * Additional classes defining specialized GUI controls for drawing
     * objects.
     */
    var DrawingControls = {};

    // constants --------------------------------------------------------------

    /**
     * Standard options for the 'Delete drawing' button in Category Drawing.
     *
     * @constant
     */
    DrawingControls.DELETE_DRAWING_OPTIONS = { icon: 'docs-drawing-remove', tooltip: gt('Delete drawing object') };

    /**
     * Combines all possible Chart-attribute-constellations to map on compatible CanvasJS-Types
     * CJS-Type are generated in the function
     *
     */
    DrawingControls.CHART_TYPES = (function () {
        var res = {};

        res['bar standard']             = {cat: 1, title: /*#. Type of a chart object: horizontal filled bars, all bars lay over each other*/                                               gt('Bar'),              type: 'bar2d',      stacking: 'standard',          cjs: 'bar'                               };
        res['bar stacked']              = {cat: 1, title: /*#. Type of a chart object: horizontal filled bars, all bars are stacked on each other */                                        gt('Bar stacked'),      type: 'bar2d',      stacking: 'stacked',           cjs: 'stackedBar'                        };
        res['bar percentStacked']       = {cat: 1, title: /*#. Type of a chart object: horizontal filled bars, all bars are stacked on each other and have sizes relative to each other */  gt('Bar percent'),      type: 'bar2d',      stacking: 'percentStacked',    cjs: 'stackedBar100'                     };
        res['column standard']          = {cat: 1, title: /*#. Type of a chart object: vertical filled bars, all columns are stacked on each other */                                       gt('Column'),           type: 'column2d',   stacking: 'standard',          cjs: 'column'                            };
        res['column stacked']           = {cat: 1, title: /*#. Type of a chart object: vertical filled bars, all columns are stacked on each other */                                       gt('Column stacked'),   type: 'column2d',   stacking: 'stacked',           cjs: 'stackedColumn'                     };
        res['column percentStacked']    = {cat: 1, title: /*#. Type of a chart object: vertical filled bars, all columns are stacked on each other and have sizes relative to each other */ gt('Column percent'),   type: 'column2d',   stacking: 'percentStacked',    cjs: 'stackedColumn100'                  };
        res['line standard']            = {cat: 2, title: /*#. Type of a chart object: data points connected with lines */                                                                  gt('Line'),             type: 'line2d',     stacking: 'standard',          cjs: 'line'                              };
        res['line standard curved']     = {cat: 2, title: /*#. Type of a chart object: data points connected with curved lines */                                                           gt('Line curved'),      type: 'line2d',     stacking: 'standard',          cjs: 'spline',          curved: true     };
        res['scatter standard']         = {cat: 2, title: /*#. Type of a chart object: data points with X/Y coordinates connected with lines */                                             gt('Scatter'),          type: 'scatter2d',  stacking: 'standard',          cjs: 'line'                              };
        res['scatter standard curved']  = {cat: 2, title: /*#. Type of a chart object: data points with X/Y coordinates connected with curved lines */                                      gt('Scatter curved'),   type: 'scatter2d',  stacking: 'standard',          cjs: 'spline',          curved: true     };
        res['bubble standard']          = {cat: 3, title: /*#. Type of a chart object: data points with X/Y coordinates drawn as circles */                                                 gt('Bubble'),           type: 'bubble2d',   stacking: 'standard',          cjs: 'bubble'                            };
        res['pie standard']             = {cat: 3, title: /*#. Type of a chart object: single pie */                                                                                        gt('Pie'),              type: 'pie2d',      stacking: 'standard',          cjs: 'pie',             varyColors: true };
        res['donut standard']           = {cat: 3, title: /*#. Type of a chart object: one or multiple concentric circles */                                                                gt('Donut'),            type: 'donut2d',    stacking: 'standard',          cjs: 'doughnut',        varyColors: true };
        res['area standard']            = {cat: 3, title: /*#. Type of a chart object: filled area between X axis and data points, all areas lay over each other */                         gt('Area'),             type: 'area2d',     stacking: 'standard',          cjs: 'area'                              };
        res['area stacked']             = {cat: 3, title: /*#. Type of a chart object: filled area between X axis and data points, all bars are stacked on each other */                    gt('Area stacked'),     type: 'area2d',     stacking: 'stacked',           cjs: 'stackedArea'                       };
        res['area percentStacked']      = {cat: 3, title: /*#. Type of a chart object: filled area between X axis and data points, all bars are stacked on each other and have sizes relative to each other */ gt('Area percent'), type: 'area2d', stacking: 'percentStacked', cjs: 'stackedArea100'            };

        _.each(res, function(type) {
            type.curved = type.curved || false;
            type.varyColors = type.varyColors || false;
        });

        return res;
    })();

    /**
     * These warning texts will be triggered when the chart sources try to switch between columns and rows
     *
     * @constant
     */
    DrawingControls.WARN_DIFF_SHEETS = /*#. Warning text: A chart object in a spreadsheet contains complex data source (cells from multiple sheets). */ gt('Data references are located on different sheets.');

    /**
     * These warning texts will be triggered when the chart sources try to switch between columns and rows
     *
     * @constant
     */
    DrawingControls.WARN_DIFF_DIRS = /*#. Warning text: A chart object in a spreadsheet contains complex data source (not a simple cell range). */ gt('Data references are too complex for this operation.');

    DrawingControls.CHART_TITLE_OPTIONS = { fullWidth: true,  tooltip: gt('Chart title'), placeholder: gt('Chart title'), select: true };

    DrawingControls.CHART_SOURCE_OPTIONS = { css: {textAlign: 'center'}, fullWidth: true, label: /*#. Change source data for a chart object in a spreadsheet */ gt('Edit data references'), toggle: true };

    DrawingControls.CHART_SOURCE_EXCHANGE_OPTIONS = { css: { textAlign: 'center' }, fullWidth: true, label: /*#. Switch orientation of data series in rectangular source data of a chart object */ gt('Switch rows/columns') };

    //TODO
    DrawingControls.SEARCH_CHART_SERIES_OPTIONS = { icon: 'fa-search', /*#. select the cells containing the current source data of a chart object */ tooltip: gt('Select source data') };

    // class ChartColorsetPicker ==============================================

    DrawingControls.ChartColorsetPicker = ChartColorsetPicker.extend({ constructor: function (app, options) {

        // base constructor ---------------------------------------------------

        ChartColorsetPicker.call(this, app, Utils.extendOptions({ label: _.noI18n('Color'), tooltip: _.noI18n('Chart color set'), updateCaptionMode: 'none' }, options));

    }}); // class ChartColorsetPicker

    // class ChartStylesetPicker ==============================================

    DrawingControls.ChartStylesetPicker = RadioList.extend({ constructor: function (options) {

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({ label: _.noI18n('Style'), tooltip: _.noI18n('Chart style set'), updateCaptionMode: 'none' }, options));

        // initialization -----------------------------------------------------

        this.createOptionButton('', 'ss0', { label: _.noI18n('Light') });
        this.createOptionButton('', 'ss4', { label: _.noI18n('Colored') });
        this.createOptionButton('', 'ss5', { label: _.noI18n('Dark') });

    }}); // class ChartStylesetPicker

    // class ChartTypePicker ==================================================

    /**
     * A drop-down list with all supported chart types.
     *
     * @constructor
     *
     * @extends ChartTypePicker
     */
    DrawingControls.ChartTypePicker = ChartTypePicker.extend({ constructor: function (app, options) {

        // base constructor ---------------------------------------------------

        ChartTypePicker.call(this, app, DrawingControls.CHART_TYPES, options);

    }}); // class ChartTypePicker

    // class ChartLegendPicker ================================================

    DrawingControls.ChartLegendPicker = RadioList.extend({ constructor: function (options) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({ label: _.noI18n('Legend'), tooltip: _.noI18n('Legend position'), updateCaptionMode: 'none' }, options));

        // initialization -----------------------------------------------------

        _.each(CHART_LEGEND_VALUES, function (value, id) {
            self.createOptionButton('', id, { label: value });
        });

    }}); // class ChartLegendPicker

    // exports ================================================================

    return DrawingControls;

});
