/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/component',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/object/triggerobject'
    ], function (Utils, KeyCodes, TriggerObject) {

    'use strict';

    // class Component ========================================================

    /**
     * Base class for view components containing instances of Group objects
     * (controls or groups of controls).
     *
     * Instances of this class trigger the following events:
     * - 'component:show': After the view component has been shown or hidden.
     *      The event handler receives the new visibility state.
     * - 'component:layout': After the size of the view component has been
     *      changed, by manipulating (showing, hiding, changing) the control
     *      groups it contains.
     * - 'component:focus': After a control group in this component has been
     *      focused, by initially focusing any of its focusable child nodes.
     * - 'component:blur': After a control group in this component has lost the
     *      browser focus, after focusing any other DOM node outside the group.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {BaseApplication} app
     *  The application containing this view component instance.
     *
     * @param {String} id
     *  The identifier for this view component. Must be unique across all view
     *  components in the application.
     *
     * @param {Object} [initOptions]
     *  A map of options to control the properties of the new view component.
     *  The following options are supported:
     *  @param {String} [initOptions.classes]
     *      Additional CSS classes that will be set at the root DOM node of
     *      this view component.
     *  @param {Object} [initOptions.css]
     *      Additional CSS formatting that will be set at the root DOM node of
     *      this view component.
     *  @param {Boolean} [initOptions.landmark=true]
     *      If set to true or omitted, the view component will be inserted into
     *      the chain of focusable landmark nodes reachable with specific
     *      global keyboard shortcuts (usually the F6 key with platform
     *      dependent modifier keys).
     *  @param {Boolean} [initOptions.hoverEffect=false]
     *      If set to true, all control groups in this view component will be
     *      displayed half-transparent as long as the mouse does not hover the
     *      view component. Has no effect, if the current device is a touch
     *      device.
     *  @param {Function} [initOptions.groupInserter]
     *      A function that will implement inserting the root DOM node of a new
     *      group into this view component. The function receives the reference
     *      to the new group instance as first parameter. Will be called in the
     *      context of this view component instance. If omitted, groups will be
     *      appended to the root node of this view component.
     */
    function Component(app, id, initOptions) {

        var // self reference
            self = this,

            // create the DOM root element representing the view component
            node = Utils.createContainerNode('view-component', initOptions),

            // the last cached size of the root node, used to detect layout changes
            nodeSize = null,

            // all control groups, as plain array
            groups = [],

            // all control groups, mapped by key
            groupsByKey = {},

            // handler called to insert a new group into this view component
            groupInserter = Utils.getFunctionOption(initOptions, 'groupInserter');

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Returns the current outer size of the root node of this view
         * component. If the node is currently invisible, returns the last
         * cached size.
         */
        function getNodeSize() {
            return self.isReallyVisible() ? { width: node.outerWidth(), height: node.outerHeight() } : nodeSize;
        }

        /**
         * Handles 'group:cancel' events. Returns browser focus to the
         * application pane.
         */
        function groupCancelHandler(event, options) {
            app.getController().update();
            if (!Utils.getBooleanOption(options, 'preserveFocus', false)) {
                app.getView().grabFocus();
            }
        }

        /**
         * Handles 'group:show' and 'group:layout' events. Triggers a
         * 'component:layout' event to all listeners, if the size of this view
         * component has been changed due to the changed control group.
         */
        function groupLayoutHandler() {
            var newNodeSize = getNodeSize();
            if (!_.isEqual(nodeSize, newNodeSize)) {
                nodeSize = newNodeSize;
                Utils.LayoutLogger.log('type="component:layout" component-id="' + id + '"');
                self.trigger('component:layout');
            }
        }

        /**
         * Handles 'group:focus' and 'group:blur' events from all registered
         * group instances. Updates the CSS marker class at the root node of
         * this view component, and forwards the event to all listeners.
         */
        function groupFocusHandler(event) {
            var focused = event.type === 'group:focus',
                type = focused ? 'component:focus' : 'component:blur';
            node.toggleClass(Utils.FOCUSED_CLASS, focused);
            Utils.FocusLogger.log('type="' + type + '" component-id="' + id + '"');
            self.trigger(type);
        }

        /**
         * Inserts the passed control group into this view component, either by
         * calling the handler function passed to the constructor, or by
         * appending the root node of the group to the children of the own root
         * node.
         *
         * @param {Group} group
         *  The group instance to be inserted into this view component.
         */
        function insertGroup(group) {

            // remember the group object
            groups.push(group);

            // insert the group into this view component
            if (_.isFunction(groupInserter)) {
                groupInserter.call(self, group);
            } else {
                node.append(group.getNode());
            }

            // forward various group events, update focusability depending on
            // the group's state, forward other layout events
            self.listenTo(group, 'group:cancel', groupCancelHandler);
            self.listenTo(group, 'group:show group:layout', groupLayoutHandler);
            self.listenTo(group, 'group:focus group:blur', groupFocusHandler);

            // make this view component focusable, if it contains any groups
            groupLayoutHandler();
        }

        /**
         * Returns all visible group objects that contain focusable controls as
         * array.
         */
        function getFocusableGroups() {
            return _(groups).filter(function (group) {
                return group.isReallyVisible() && group.hasFocusableControls();
            });
        }

        /**
         * Keyboard handler for the entire view component.
         *
         * @param {jQuery.Event} event
         *  The jQuery keyboard event object.
         *
         * @returns {Boolean}
         *  True, if the event has been handled and needs to stop propagating.
         */
        function keyHandler(event) {
            if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                if (event.type === 'keydown') {
                    self.moveFocus(event.shiftKey ? 'prev' : 'next');
                }
                return false;
            }
        }

        /**
         * Handler for application controller 'change:items' events.
         */
        function changeItemsHandler(event, changedItems) {
            _(changedItems).each(function (itemState, key) {
                if (key in groupsByKey) {
                    _.chain(groupsByKey[key])
                        .invoke('enable', itemState.enable)
                        .invoke('setValue', itemState.value);
                }
            });
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the root element containing this view component as jQuery
         * object.
         */
        this.getNode = function () {
            return node;
        };

        /**
         * Returns the options map that has been passed to the constructor.
         */
        this.getOptions = function () {
            return initOptions;
        };

        /**
         * Adds the passed control group as a 'private group' to this view
         * component. The 'group:change' events triggered by the group will not
         * be handled. Instead, the caller has to register a change listener at
         * the group by itself.
         *
         * @param {Group} group
         *  The control group object to be inserted.
         *
         * @returns {Component}
         *  A reference to this view component.
         */
        this.addPrivateGroup = function (group) {
            insertGroup(group);
            return this;
        };

        /**
         * Adds the passed control group to this view component. The component
         * listens to 'change:items' events of the application controller and
         * forwards changed values to all registered control groups, and it
         * listens to 'group:change' events of all registered control groups
         * and executes the respective controller items.
         *
         * @param {String} key
         *  The unique key of the control group.
         *
         * @param {Group} group
         *  The control group object to be inserted.
         *
         * @returns {Component}
         *  A reference to this view component.
         */
        this.addGroup = function (key, group) {

            // insert the group object into this view component
            insertGroup(group);

            // execute controller item for 'group:change' events
            (groupsByKey[key] || (groupsByKey[key] = [])).push(group);
            self.listenTo(group, 'group:change', function (event, value, options) {
                app.getController().executeItem(key, value, options);
            });

            // create unique DOM identifier for Selenium testing, set the key as data attribute
            group.getNode().attr({
                id: node.attr('id') + '-group-' + key.replace(/[^a-zA-Z0-9]+/g, '-') + '-' + groupsByKey[key].length,
                'data-key': key
            });

            return this;
        };

        /**
         * Returns whether this view component is visible.
         */
        this.isVisible = function () {
            return !node.hasClass(Utils.HIDDEN_CLASS);
        };

        /**
         * Returns whether this view component is effectively visible (it must
         * not be hidden by itself, it must be inside the DOM tree, and all its
         * parent nodes must be visible too).
         */
        this.isReallyVisible = function () {
            return node.is(Utils.REALLY_VISIBLE_SELECTOR);
        };

        /**
         * Displays this view component, if it is currently hidden.
         *
         * @returns {Component}
         *  A reference to this view component.
         */
        this.show = function () {
            return this.toggle(true);
        };

        /**
         * Hides this view component, if it is currently visible.
         *
         * @returns {Component}
         *  A reference to this view component.
         */
        this.hide = function () {
            return this.toggle(false);
        };

        /**
         * Toggles the visibility of this view component.
         *
         * @param {Boolean} [state]
         *  If specified, shows or hides the view component depending on the
         *  boolean value. If omitted, toggles the current visibility of the
         *  view component.
         *
         * @returns {Component}
         *  A reference to this view component.
         */
        this.toggle = function (state) {
            var visible = (state === true) || ((state !== false) && !this.isVisible());
            if (this.isVisible() !== visible) {
                node.toggleClass(Utils.HIDDEN_CLASS, !visible);
                Utils.LayoutLogger.log('type="component:show" component-id="' + id + '" state=' + visible);
                this.trigger('component:show', visible);
                nodeSize = getNodeSize();
            }
            return this;
        };

        /**
         * Returns whether this view component contains the control that is
         * currently focused. Searches in all registered group objects.
         */
        this.hasFocus = function () {
            return _(groups).any(function (group) { return group.hasFocus(); });
        };

        /**
         * Sets the focus to the first enabled group object in this view
         * component, unless it already contains a focused group.
         *
         * @returns {Component}
         *  A reference to this view component.
         */
        this.grabFocus = function () {

            var // all visible group objects with focusable controls
                focusableGroups = this.hasFocus() ? [] : getFocusableGroups();

            // set focus to first focusable group
            if (focusableGroups.length > 0) {
                focusableGroups[0].grabFocus();
            }

            return this;
        };

        /**
         * Moves the focus to the first, previous, next, or last enabled
         * control in the view component.
         *
         * @param {String} mode
         *  Specifies how to move the focus. The following modes are supported:
         *  - 'first': Moves focus to the first control in this component.
         *  - 'prev': Moves focus to the control before the focused control.
         *  - 'next': Moves focus to the control after the focused control.
         *  - 'last': Moves focus to the last control in this component.
         */
        this.moveFocus = function (mode) {

            var // all visible group objects with focusable controls
                focusableGroups = getFocusableGroups(),
                // extract all focusable controls from the groups
                controls = _(focusableGroups).reduce(function (memo, group) { return memo.add(group.getFocusableControls()); }, $()),
                // focused control
                control = controls.filter(window.document.activeElement),
                // index of focused control in all enabled controls
                oldIndex = controls.index(control),
                // index of the control to be focused
                newIndex = oldIndex;

            // move focus to next/previous control
            if (controls.length > 0) {
                switch (mode) {
                case 'first':
                    newIndex = 0;
                    break;
                case 'prev':
                    newIndex = (oldIndex >= 0) ? ((oldIndex - 1 + controls.length) % controls.length) : (controls.length - 1);
                    break;
                case 'next':
                    newIndex = (oldIndex >= 0) ? ((oldIndex + 1) % controls.length) : 0;
                    break;
                case 'last':
                    newIndex = controls.length - 1;
                    break;
                }
                if (oldIndex !== newIndex) {
                    controls.eq(newIndex).focus();
                }
            }

            return this;
        };

        /**
         * Visits all control groups registered at this view component.
         *
         * @param {Function} iterator
         *  The iterator called for each control group. Receives the reference
         *  to the control group, and its insertion index. If the iterator
         *  returns the Utils.BREAK object, the iteration process will be
         *  stopped immediately.
         *
         * @param {Object} [context]
         *  If specified, the iterator will be called with this context (the
         *  symbol 'this' will be bound to the context inside the iterator
         *  function).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateGroups = function (iterator, context) {
            return _(groups).any(function (group, index) {
                return iterator.call(context, group, index) === Utils.BREAK;
            }) ? Utils.BREAK : undefined;
        };

        // initialization -----------------------------------------------------

        // create unique DOM identifier for Selenium testing
        node.attr('id', app.getWindowId() + '-component-' + id);

        // marker for touch devices
        node.toggleClass('touch', Modernizr.touch);

        // whether the pane will be a land mark for F6 focus traveling
        node.toggleClass('f6-target', Utils.getBooleanOption(initOptions, 'landmark', true));

        // hover effect for groups embedded in the view component
        node.toggleClass('hover-effect', Utils.getBooleanOption(initOptions, 'hoverEffect', false));

        // listen to key events for keyboard focus navigation
        self.listenTo(node, 'keydown keypress keyup', keyHandler);

        // update all registered groups after change events of the application controller
        app.getController().on('change:items', changeItemsHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            node.remove();

            _.invoke(groups, 'destroy');

            node = groups = groupsByKey = null;
        });

    } // class Component

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: Component });

});
