/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/view/basecontrols',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/popup/basemenu',
     'io.ox/office/tk/control/group',
     'io.ox/office/tk/control/label',
     'io.ox/office/tk/control/button',
     'io.ox/office/tk/control/checkbox',
     'io.ox/office/tk/control/radiogroup',
     'io.ox/office/tk/control/radiolist',
     'io.ox/office/tk/control/textfield',
     'io.ox/office/tk/control/unitfield',
     'io.ox/office/tk/control/combofield',
     'io.ox/office/tk/control/spinfield',
     'io.ox/office/tk/control/menumixin',
     'io.ox/office/baseframework/view/component',
     'gettext!io.ox/office/main'
    ], function (Utils, KeyCodes, BaseMenu, Group, Label, Button, CheckBox, RadioGroup, RadioList, TextField, UnitField, ComboField, SpinField, MenuMixin, Component, gt) {

    'use strict';

    // static class BaseControls ==============================================

    /**
     * Provides different classes for GUI controls. Collects all standard
     * controls defined in the toolkit, and adds more sophisticated controls.
     */
    var BaseControls = {
            Group: Group,
            Label: Label,
            Button: Button,
            CheckBox: CheckBox,
            RadioGroup: RadioGroup,
            RadioList: RadioList,
            TextField: TextField,
            UnitField: UnitField,
            ComboField: ComboField,
            SpinField: SpinField
        };

    // constants --------------------------------------------------------------

    /**
     * Standard options for the 'Close' button.
     *
     * @constant
     */
    BaseControls.QUIT_OPTIONS = { icon: 'fa-times', tooltip: gt('Close document') };

    /**
     * Standard options for the 'Hide side panel' button.
     *
     * @constant
     */
    BaseControls.HIDE_SIDEPANE_OPTIONS = { icon: 'docs-hide-sidepane', tooltip: gt('Hide side panel'), value: false };

    /**
     * Standard options for the 'Show side panel' button.
     *
     * @constant
     */
    BaseControls.SHOW_SIDEPANE_OPTIONS = { icon: 'docs-show-sidepane', tooltip: gt('Show side panel'), value: true };

    /**
     * Standard options for the 'Zoom out' button.
     *
     * @constant
     */
    BaseControls.ZOOMOUT_OPTIONS = { icon: 'docs-zoom-out', tooltip: gt('Zoom out') };

    /**
     * Standard options for the 'Zoom in' button.
     *
     * @constant
     */
    BaseControls.ZOOMIN_OPTIONS = { icon: 'docs-zoom-in', tooltip: gt('Zoom in') };

    // class StatusLabel ======================================================

    /**
     * A status label with special appearance and fade-out animation.
     *
     * The method StatusLabel.setValue() accepts an options map containing the
     * same options as described for the 'options' constructor parameter,
     * allowing to override the default options of the status label.
     *
     * @param {Object} [initOptions]
     *  A map with options controlling the default behavior of the status
     *  label. The following options are supported:
     *  @param {String} [initOptions.type='info']
     *      The label type ('success', 'warning', 'error', or 'info').
     *  @param {Boolean} [initOptions.fadeOut=false]
     *      Whether to fade out the label automatically after a short delay.
     *  @param {Number} [initOptions.delay=0]
     *      The delay time after the status label will be actually updated.
     *  @param {Number} [initOptions.changed=false]
     *      If set to true, a status label currently faded out will only be
     *      shown if its label has really changed (and not with every update
     *      call with the same label).
     */
    BaseControls.StatusLabel = Label.extend({ constructor: function (app, initOptions) {

        var // self reference
            self = this,

            // the new type of the label (colors)
            type = Utils.getStringOption(initOptions, 'type', 'info'),

            // whether to fade out the label automatically
            fadeOut = Utils.getBooleanOption(initOptions, 'fadeOut', false),

            // delay time after the label will be updated
            delay = Utils.getIntegerOption(initOptions, 'delay', 0, 0),

            // whether to show the label only after the value has changed
            changed = Utils.getBooleanOption(initOptions, 'changed', false),

            // current initial delay timer before the state of the label will be changed
            initialTimer = null,

            // current animation delay timer (jQuery.delay() does not work well with jQuery.stop())
            animationTimer = null;

        // base constructor ---------------------------------------------------

        Label.call(this, { classes: 'status-label' });

        // private methods ----------------------------------------------------

        /**
         * Stops the running jQuery fade animation and removes all explicit CSS
         * attributes from the group node.
         */
        function stopNodeAnimation() {
            self.getNode().stop(true).css({ display: '', opacity: '' });
        }

        /**
         * Callback called from the Label.setValue() method.
         */
        function updateHandler(caption, options) {

            // do nothing if the label will not be updated when the text does not change
            if (Utils.getBooleanOption(options, 'changed', changed) && (self.getLabelText() === caption)) {
                return;
            }

            if (initialTimer) { initialTimer.abort(); }
            initialTimer = app.executeDelayed(function () {

                // stop running fade-out and remove CSS attributes added by jQuery
                if (animationTimer) { animationTimer.abort(); }
                stopNodeAnimation();

                // update the status label
                if (_.isString(caption) && (caption.length > 0)) {
                    self.setLabelText(caption).show().getNode().attr('data-type', Utils.getStringOption(options, 'type', type));
                    if (Utils.getBooleanOption(options, 'fadeOut', fadeOut)) {
                        animationTimer = app.executeDelayed(function () {
                            self.getNode().fadeOut(function () { self.hide(); });
                        }, { delay: 2000 });
                    }
                } else {
                    self.hide();
                }

            }, { delay: Utils.getIntegerOption(options, 'delay', delay, 0) });
        }

        // initialization -----------------------------------------------------

        this.registerUpdateHandler(updateHandler).setValue(null);

        // destroy all class members
        this.registerDestructor(function () {
            stopNodeAnimation();
        });

    }}); // class StatusLabel

    // class PopupMenu ========================================================

    /**
     * A pop-up menu element that contains a complete self-contained instance
     * of a view component (class Component) with independent controls bound to
     * different controller items.
     *
     * @constructor
     *
     * @extends BaseMenu
     *
     * @param {BaseApplication} app
     *  The application containing this pop-up menu.
     *
     * @param {String} componentId
     *  The identifier for the view component contained in the pop-up menu.
     *  Must be unique across all view components in the application.
     *
     * @param {Object} [initOptions]
     *  A map of options with properties for the pop-up menu. Supports all
     *  options also supported by the base class BaseMenu.
     */
    BaseControls.PopupMenu = BaseMenu.extend({ constructor: function (app, componentId, initOptions) {

        var // the view component in the drop-down menu
            component = new Component(app, componentId, { groupInserter: groupInserter, landmark: false });

        // base constructors --------------------------------------------------

        BaseMenu.call(this, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Inserts the passed group into the menu component.
         */
        function groupInserter(group) {

            var // the root node of the passed group
                groupNode = group.getNode(),
                // the target node for the passed group
                targetNode = component.getNode(),
                // the last child in the menu component
                lastNode = targetNode.children().last();

            if (Utils.getBooleanOption(group.getOptions(), 'appendInline', false)) {
                if (lastNode.is('.inline-container')) {
                    targetNode = lastNode;
                } else {
                    targetNode = $('<div>').addClass('inline-container').appendTo(targetNode);
                    if (lastNode.is('.group')) { targetNode.append(lastNode); }
                }
            }

            // insert the group node
            targetNode.append(groupNode);
        }

        /**
         * Handles 'keydown' events in the drop-down menu.
         */
        function menuKeyDownHandler(event) {
            switch (event.keyCode) {
            case KeyCodes.UP_ARROW:
                component.moveFocus('prev');
                return false;
            case KeyCodes.DOWN_ARROW:
                component.moveFocus('next');
                return false;
            case KeyCodes.HOME:
                component.moveFocus('first');
                return false;
            case KeyCodes.END:
                component.moveFocus('last');
                return false;
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Adds the passed control group as a 'private group' to the pop-up
         * menu. The 'group:change' events triggered by the group will not be
         * forwarded to the listeners of the view component.
         *
         * @param {Group} group
         *  The control group object to be inserted into the pop-up menu.
         *
         * @returns {PopupMenu}
         *  A reference to this instance.
         */
        this.addPrivateGroup = function (group) {
            component.addPrivateGroup(group);
            return this;
        };

        /**
         * Adds the passed control group to the pop-up menu. The view component
         * contained in the menu listens to 'change:items' events of the
         * application controller and forwards changed values to all registered
         * control groups.
         *
         * @param {String} key
         *  The unique key of the control group.
         *
         * @param {Group} group
         *  The control group object to be inserted into the pop-up menu.
         *
         * @returns {PopupMenu}
         *  A reference to this instance.
         */
        this.addGroup = function (key, group) {
            component.addGroup(key, group);
            return this;
        };

        /**
         * Adds a separator line after the last control group in the pop-up
         * menu.
         *
         * @returns {PopupMenu}
         *  A reference to this instance.
         */
        this.addSeparator = function () {
            component.getNode().append($('<div>').addClass('separator-line'));
            return this;
        };

        // initialization -----------------------------------------------------

        // insert the root node of the view component into the drop-down menu
        this.appendContentNodes(component.getNode());

        // initially, move browser focus into the menu without focusing a menu item
        this.on('popup:show', function () { component.getNode().focus(); });

        // additional keyboard shortcuts for focus navigation (make the root node
        // of the view component focusable to be able to provide keyboard support
        // without actually focusing an existing menu item)
        component.getNode().attr('tabindex', 0).on('keydown', menuKeyDownHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            component.destroy();
            app = component = null;
        });

    }}); // class PopupMenu

    // class PopupMenuButton ==================================================

    /**
     * A generic drop-down button control that shows a complete self-contained
     * view component instance with completely independent controls in its
     * drop-down menu.
     *
     * @constructor
     *
     * @extends Button
     * @extends MenuMixin
     *
     * @param {BaseApplication} app
     *  The application containing this control.
     *
     * @param {String} componentId
     *  The identifier for the view component contained in the drop-down menu.
     *  Must be unique across all view components in the application.
     *
     * @param {Object} [initOptions]
     *  A map of options with properties for the drop-down button. Supports all
     *  options also supported by the base class Button, the mix-in class
     *  MenuMixin (visibility of caret icon), and the class
     *  BaseControls.PopupMenu (settings for the drop-down menu).
     */
    BaseControls.PopupMenuButton = Button.extend({ constructor: function (app, componentId, initOptions) {

        var // the pop-up menu instance (must be created after Button base constructor!)
            menu = null;

        // base constructors --------------------------------------------------

        Button.call(this, initOptions);
        menu = new BaseControls.PopupMenu(app, componentId, Utils.extendOptions(initOptions, { anchor: this.getNode() }));
        MenuMixin.call(this, menu, Utils.extendOptions(initOptions, { button: this.getButtonNode() }));

        // methods ------------------------------------------------------------

        // export methods of the menu
        _(['addPrivateGroup', 'addGroup', 'addSeparator']).each(function (methodName) {
            this[methodName] = function () {
                menu[methodName].apply(menu, arguments);
                return this;
            };
        }, this);

        // destroy all class members
        this.registerDestructor(function () {
            app = menu = null;
        });

    }}); // class PopupMenuButton

    // class ContextMenu ======================================================

    /**
     * An instance of this class represents a context menu bound to a specific
     * DOM element. Right-clicking that DOM element, or pressing specific keys
     * (CONTEXT key) or key combinations (e.g. Shift+F10) while the element is
     * focused or contains the browser focus will show this context menu.
     *
     * @constructor
     *
     * @extends BaseControls.PopupMenu
     *
     * @param {BaseApplication} app
     *  The application containing this context menu.
     *
     * @param {String} componentId
     *  The identifier for the view component contained in the context menu.
     *  Must be unique across all view components in the application.
     *
     * @param {HTMLObject|jQuery} anchorNode
     *  The DOM element that 'contains' this context menu. If this object is a
     *  jQuery collection, uses the first DOM element it contains.
     *
     * @param {Object} [initOptions]
     *  Initial options for this context menu.
     *  @param {Function} [initOptions.prepareShowHandler]
     *      A callback function that will be invoked before showing the context
     *      menu after a user action (right mouse click, or keyboard). Receives
     *      the original event caused opening the context menu (as jQuery.Event
     *      instance).
     *  @param {String|String[]} [initOptions.enableKey]
     *      The key of a controller item, or an array of controller item keys,
     *      that specify whether the context menu is enabled. Uses the enabled
     *      state of the controller items registered for the specified keys.
     *      All specified items must be enabled to enable the context menu. If
     *      omitted, the context menu will always be enabled.
     *  @param {Boolean} [initOptions.mouseDown=false]
     *      If set to true, the context menu will be shown instantly after the
     *      'mousedown' event for the right mouse button. If omitted or set to
     *      false, the context menu will be opened after the 'mouseup' and its
     *      related 'contextmenu' event.
     */
    BaseControls.ContextMenu = BaseControls.PopupMenu.extend({ constructor: function (app, componentId, anchorNode, initOptions) {

        var // self reference
            self = this,

            // preprocesses browser events before showing the context menu
            prepareShowHandler = Utils.getFunctionOption(initOptions, 'prepareShowHandler', $.noop),

            // controller keys specifying whether the context menu is enabled
            enableKeys = Utils.getOption(initOptions, 'enableKey'),

            // whether to show the context menu instantly after the 'mousedown' event
            mouseDown = Utils.getBooleanOption(initOptions, 'mouseDown', false),

            // the position to open the context menu (initialized from mouse/keyboard events)
            anchorPosition = null,

            // all event handlers registered at the document while the context  menu is open
            DOCUMENT_EVENT_MAP = { mousedown: globalClickHandler, focusin: globalFocusInHandler };

        // base constructor ---------------------------------------------------

        BaseControls.PopupMenu.call(this, app, componentId, { anchor: getAnchorPosition, anchorPadding: 0 });

        // methods ------------------------------------------------------------

        /**
         * Returns the target position of the context menu.
         */
        function getAnchorPosition() {
            return anchorPosition;
        }

        /**
         * Handles mouse clicks everywhere on the page, and closes the context
         * menu automatically.
         */
        function globalClickHandler(event) {
            var rootNode = self.getNode();
            if (!rootNode.is(event.target) && (rootNode.has(event.target).length === 0)) {
                self.hide();
            }
        }

        /**
         * Closes the context menu as soon as any DOM element outside the menu
         * gets the browser focus.
         */
        function globalFocusInHandler(event) {
            if (!self.getNode().is(event.target) && !Utils.containsNode(self.getNode(), event.target)) {
                self.hide();
            }
        }

        /**
         * Initializes the context menu after it has been opened.
         */
        function popupShowHandler() {
            // Add global click handler that closes the menu automatically when
            // clicking somewhere in the page. The event handler must be attached
            // deferred, otherwise it would close the context menu immediately
            // after it has been opened with a mouse click while the event bubbles
            // up to the document root.
            _.defer(function () {
                if (self.isVisible()) {
                    $(document).on(DOCUMENT_EVENT_MAP);
                }
            });
        }

        /**
         * Deinitializes the context menu before it will be closed.
         */
        function popupBeforeHideHandler() {
            // unregister the global click handler
            $(document).off(DOCUMENT_EVENT_MAP);
        }

        /**
         * Invokes the callback function passed to the constructor, and shows
         * the context menu at the correct position.
         */
        function showContextMenu(event) {
            var position = prepareShowHandler.call(self, event);
            if ((position !== Utils.BREAK) && self.isEnabled()) {
                anchorPosition = _.isObject(position) ? position : { left: event.pageX, top: event.pageY, width: 1, height: 1 };
                self.hide().show();
            }
        }

        /**
         * Handles 'mousedown' events of the anchor node. Shows the context
         * menu, if the option 'mouseDown' has been passed to the constructor.
         */
        function anchorMouseDownHandler(event) {
            if (mouseDown && (event.button === 2)) {
                showContextMenu(event);
                event.preventDefault();
            }
        }

        /**
         * Handles 'contextmenu' events of the anchor node, triggered after
         * right-clicking the anchor node, or pressing specific keys that cause
         * showing a context menu.
         */
        function anchorContextMenuHandler(event) {
            showContextMenu(event);
            event.preventDefault();
        }

        /**
         * Handles 'change:items' events of the controller. Hides the context
         * menu if any of the controller items specified with the constructor
         * option 'enableKey' has been disabled.
         */
        function controllerChangeHandler(event, changedItems) {

            // nothing to do, if the context menu is not visible
            if (!self.isVisible()) { return; }

            var // whether to hide the context menu due to changed controller item states
                hideMenu = _(enableKeys).any(function (enableKey) {
                    return (enableKey in changedItems) && !changedItems[enableKey].enable;
                });

            if (hideMenu) {
                self.hide();
                app.getView().grabFocus();
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns whether this context menu is currently enabled (whether it
         * will be shown after right-clicking the anchor node, or using the
         * keyboard).
         *
         * @returns {Boolean}
         *  Whether this context menu is currently enabled.
         */
        this.isEnabled = function () {
            var controller = app.getController();
            return _(enableKeys).all(_.bind(controller.isItemEnabled, controller));
        };

        // initialization -----------------------------------------------------

        // register event handlers for the pop-up menu
        this.on({
            'popup:show': popupShowHandler,
            'popup:beforehide': popupBeforeHideHandler
        });

        // listen to relevant context events on the anchor node
        anchorNode.on({
            mousedown: anchorMouseDownHandler,
            contextmenu: anchorContextMenuHandler
        });

        // always use an array of parent keys (as strings)
        enableKeys =
            _.isArray(enableKeys) ? _(enableKeys).filter(_.isString) :
            _.isString(enableKeys) ? [enableKeys] :
            null;

        // hide displayed context menu, if controller item changes to disabled state
        if (_.isArray(enableKeys) && (enableKeys.length > 0)) {
            this.listenTo(app.getController(), 'change:items', controllerChangeHandler);
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = anchorNode = null;
        });

    }}); // class ContextMenu

    // exports ================================================================

    return BaseControls;

});
