/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/utils/driveutils'
], function (DriveUtils) {

    'use strict';

    // static class DriveUtils ================================================

    describe('Toolkit module DriveUtils', function () {

        it('should exist', function () {
            expect(DriveUtils).to.be.an('object');
        });

        // constants ----------------------------------------------------------

        describe('constant "GUARD_EXT"', function () {
            it('should exist', function () {
                expect(DriveUtils).to.have.a.property('GUARD_EXT').that.equals('.pgp');
            });
        });

        // static methods -----------------------------------------------------

        describe('method "hasGuardExt"', function () {
            it('should exist', function () {
                expect(DriveUtils).itself.to.respondTo('hasGuardExt');
            });
            it('should return whether a file name has the PGP extension', function () {
                expect(DriveUtils.hasGuardExt('')).to.equal(false);
                expect(DriveUtils.hasGuardExt('txt')).to.equal(false);
                expect(DriveUtils.hasGuardExt('pgp')).to.equal(false);
                expect(DriveUtils.hasGuardExt('.txt')).to.equal(false);
                expect(DriveUtils.hasGuardExt('.pgp')).to.equal(false);
                expect(DriveUtils.hasGuardExt('example.')).to.equal(false);
                expect(DriveUtils.hasGuardExt('example.txt')).to.equal(false);
                expect(DriveUtils.hasGuardExt('example.pgp')).to.equal(true);
                expect(DriveUtils.hasGuardExt('example.txt.pgp')).to.equal(true);
                expect(DriveUtils.hasGuardExt('example.pgp.txt')).to.equal(false);
            });
            it('should ignore case', function () {
                expect(DriveUtils.hasGuardExt('example.PGP')).to.equal(true);
                expect(DriveUtils.hasGuardExt('example.TXT.PGP')).to.equal(true);
            });
        });

        describe('method "removeGuardExt"', function () {
            it('should exist', function () {
                expect(DriveUtils).itself.to.respondTo('removeGuardExt');
            });
            it('should return whether a file name has the PGP extension', function () {
                expect(DriveUtils.removeGuardExt('')).to.equal('');
                expect(DriveUtils.removeGuardExt('txt')).to.equal('txt');
                expect(DriveUtils.removeGuardExt('pgp')).to.equal('pgp');
                expect(DriveUtils.removeGuardExt('.txt')).to.equal('.txt');
                expect(DriveUtils.removeGuardExt('.pgp')).to.equal('.pgp');
                expect(DriveUtils.removeGuardExt('example.')).to.equal('example.');
                expect(DriveUtils.removeGuardExt('example.txt')).to.equal('example.txt');
                expect(DriveUtils.removeGuardExt('example.pgp')).to.equal('example');
                expect(DriveUtils.removeGuardExt('example.txt.pgp')).to.equal('example.txt');
                expect(DriveUtils.removeGuardExt('example.pgp.txt')).to.equal('example.pgp.txt');
            });
            it('should ignore case', function () {
                expect(DriveUtils.removeGuardExt('example.PGP')).to.equal('example');
                expect(DriveUtils.removeGuardExt('example.TXT.PGP')).to.equal('example.TXT');
            });
        });

        describe('method "getFileExtension"', function () {
            it('should exist', function () {
                expect(DriveUtils).itself.to.respondTo('getFileExtension');
            });
            it('should return the file extension', function () {
                expect(DriveUtils.getFileExtension('')).to.equal('');
                expect(DriveUtils.getFileExtension('txt')).to.equal('');
                expect(DriveUtils.getFileExtension('pgp')).to.equal('');
                expect(DriveUtils.getFileExtension('.txt')).to.equal('txt');
                expect(DriveUtils.getFileExtension('.pgp')).to.equal('pgp');
                expect(DriveUtils.getFileExtension('example.')).to.equal('');
                expect(DriveUtils.getFileExtension('example.txt')).to.equal('txt');
                expect(DriveUtils.getFileExtension('example.pgp')).to.equal('pgp');
                expect(DriveUtils.getFileExtension('example.txt.pgp')).to.equal('txt');
                expect(DriveUtils.getFileExtension('example.pgp.txt')).to.equal('txt');
            });
            it('should ignore case', function () {
                expect(DriveUtils.getFileExtension('example.TXT')).to.equal('txt');
                expect(DriveUtils.getFileExtension('example.TXT.PGP')).to.equal('txt');
            });
        });

        describe('method "replaceFileExtension"', function () {
            it('should exist', function () {
                expect(DriveUtils).itself.to.respondTo('replaceFileExtension');
            });
            it('should replace the file extension', function () {
                expect(DriveUtils.replaceFileExtension('example.txt', 'dat')).to.equal('example.dat');
                expect(DriveUtils.replaceFileExtension('example.txt.pgp', 'dat')).to.equal('example.dat.pgp');
                expect(DriveUtils.replaceFileExtension('example.TXT.PGP', 'dat')).to.equal('example.dat.pgp');
            });
        });

        describe('method "getFileBaseName"', function () {
            it('should exist', function () {
                expect(DriveUtils).itself.to.respondTo('getFileBaseName');
            });
            it('should return the file base name', function () {
                expect(DriveUtils.getFileBaseName('')).to.equal('');
                expect(DriveUtils.getFileBaseName('txt')).to.equal('txt');
                expect(DriveUtils.getFileBaseName('pgp')).to.equal('pgp');
                expect(DriveUtils.getFileBaseName('.txt')).to.equal('');
                expect(DriveUtils.getFileBaseName('.pgp')).to.equal('');
                expect(DriveUtils.getFileBaseName('example.')).to.equal('example.');
                expect(DriveUtils.getFileBaseName('example.txt')).to.equal('example');
                expect(DriveUtils.getFileBaseName('example.pgp')).to.equal('example');
                expect(DriveUtils.getFileBaseName('example.txt.pgp')).to.equal('example');
                expect(DriveUtils.getFileBaseName('example.pgp.txt')).to.equal('example.pgp');
            });
            it('should ignore case', function () {
                expect(DriveUtils.getFileBaseName('example.TXT')).to.equal('example');
                expect(DriveUtils.getFileBaseName('example.TXT.PGP')).to.equal('example');
            });
        });
    });

    // ========================================================================
});
