/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author York Richter <york.richter@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/model/model',
    'globals/apphelper',
    'globals/sheethelper'
], function (SpreadsheetModel, AppHelper, SheetHelper) {

    'use strict';

    // convenience shortcuts
    var a = SheetHelper.a;

    // class SpreadsheetModel =================================================

    describe('Spreadsheet class SpreadsheetModel', function () {

        it('should exist', function () {
            expect(SpreadsheetModel).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        function testChartModel(chartModel, dataSeries1, dataSeries2) {
            expect(chartModel).to.be.an('object');
            var count = 0;
            chartModel.iterateTokenArrays(function (seriesIndex, tokenArrays) {

                _.each(tokenArrays, function (tokenArray, name) {
                    count++;
                    var formula = tokenArray.getFormula('op');
                    var expectFormula = seriesIndex === 0 ? dataSeries1[name] : dataSeries2[name];
                    expect(formula).to.equal(expectFormula);
                });

            });
            expect(count).to.equal(6);
        }

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { fileFormat: 'ooxml', cols: 16384, rows: 1048576, activeSheet: 1 } } },
            { name: 'insertName', label: 'ObstPreisGlobal', formula: 'Sheet1!$B$5:$B$7' },
            { name: 'insertName', label: 'ObstGlobal', formula: 'Sheet1!$A$5:$A$7' },

            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1', attrs: { sheet: { zoom: 1 }, column: { width: 1611.81 }, row: { height: 529 } } },
            { name: 'insertName', sheet: 0, label: 'ObstPreis', formula: 'Sheet1!$B$5:$B$7' },
            { name: 'insertName', sheet: 0, label: 'Obst', formula: 'Sheet1!$A$5:$A$7' },
            { name: 'changeCells', sheet: 0, start: 'A1', contents: [['cell A1']] },
            { name: 'changeCells', sheet: 0, start: 'A4', contents: [
                [null,    'price', 'weight'],
                ['apple', 1.9,     1],
                ['pear',  2,       1],
                ['grape', 8,       1]
            ] },

            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2', attrs: { sheet: { zoom: 1, selectedRanges: [{ start: [10, 14], end: [10, 14] }], activeIndex: 0, activeCell: [10, 14] }, column: { width: 1611.81 }, row: { height: 529 } } },
            // cell formula
            { name: 'changeCells', sheet: 1, start: 'A1', contents: [
                [{ f: 'Sheet1!$A$1', v: 'Zelle Sheet1 A1' }, null, { f: 'COUNT(ObstPreisGlobal)', v: 3 }, { r: 3 }, 1],
                [{ r: 6 }, 6, { r: 7 }, 'Preis', 'Kilo'],
                ['Zelle Sheet2A3', null, { f: 'COUNT(Sheet1!ObstPreis)', v: 3 }, null, 'Kurz', null, 10, { r: 6 }, 'Apfel', 1.9, 1],
                [{ r: 4 }, 'Lang', { r: 8 }, 'Birne', 2, 1],
                [{ f: 'Sheet2!$A$3', v: 'Zelle Sheet2A3' }, null, 1, { r: 3 }, 1, { r: 6 }, 'Trauben', 8, 1],
                [{ r: 2 }, 2, { r: 3 }, 100, null, 'Sheet1!$A$4:$C$7', null, 'Sheet2!$R$2:$T$5'],
                [],
                [{ r: 2 }, { f: 'COUNT(Sheet2!TestRangeSheet2)', v: 2 }, { r: 3 }, 5],
                [{ r: 6 }, 80]
            ] },
            // Named Ranges
            { name: 'insertName', sheet: 1, label: 'TestRangeSheet2', formula: 'Sheet2!$C$5:$C$6' },
            // Chart Drawing
            { name: 'insertDrawing', start: [1, 0], type: 'chart', attrs: { drawing: { startCol: 8, startColOffset: 132, startRow: 0, startRowOffset: 0, endCol: 8, endColOffset: 4101, endRow: 4, endRowOffset: 370, anchorType: 'twoCell' }, chart: { chartStyleId: 2, stacking: 'clustered', chartColors: { id: 10, meth: 'cycle', schemeClr: [{ color: { type: 'scheme', value: 'accent1' } }, { color: { type: 'scheme', value: 'accent2' } }, { color: { type: 'scheme', value: 'accent3' } }, { color: { type: 'scheme', value: 'accent4' } }, { color: { type: 'scheme', value: 'accent5' } }, { color: { type: 'scheme', value: 'accent6' } }] }, varyColors: false }, line: { type: 'solid', width: 26, color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 15000 }, { type: 'lumOff', value: 85000 }] } }, fill: { type: 'solid', color: { type: 'scheme', value: 'background1' } } } },
            { name: 'insertChartDataSeries', start: [1, 0], series: 0, attrs: { series: { axisXIndex: 0, axisYIndex: 1, type: 'column2d', title: 'Sheet1!$B$4', names: 'Sheet1!$A$5:$A$7', values: 'Sheet1!$B$5:$B$7' }, line: { type: 'none' }, fill: { type: 'solid', color: { type: 'scheme', value: 'accent1' } } } },
            { name: 'insertChartDataSeries', start: [1, 0], series: 1, attrs: { series: { axisXIndex: 0, axisYIndex: 1, type: 'column2d', title: 'Sheet1!$C$4', names: 'Sheet1!$A$5:$A$7', values: 'Sheet1!$C$5:$C$7' }, line: { type: 'none' }, fill: { type: 'solid', color: { type: 'scheme', value: 'accent2' } } } },
            { name: 'setChartAxisAttributes', start: [1, 0], axis: 0, axPos: 'b', crossAx: 1, attrs: { line: { type: 'solid', width: 26, color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 15000 }, { type: 'lumOff', value: 85000 }] } }, fill: { type: 'none' }, axis: { label: true }, character: { color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 65000 }, { type: 'lumOff', value: 35000 }] }, fontSize: 9 } } },
            { name: 'setChartAxisAttributes', start: [1, 0], axis: 1, axPos: 'l', crossAx: 0, attrs: { line: { type: 'none', width: 18 }, fill: { type: 'none' }, axis: { label: true }, character: { color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 65000 }, { type: 'lumOff', value: 35000 }] }, fontSize: 9 } } },
            { name: 'setChartGridlineAttributes', start: [1, 0], axis: 1, attrs: { line: { type: 'solid', width: 26, color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 15000 }, { type: 'lumOff', value: 85000 }] } }, fill: { type: 'none' } } },
            { name: 'setChartLegendAttributes', start: [1, 0], attrs: { legend: { pos: 'off' } } },
            { name: 'insertDrawing', start: [1, 1], type: 'chart', attrs: { drawing: { startCol: 9, startColOffset: 1667, startRow: 0, startRowOffset: 0, endCol: 11, endColOffset: 79, endRow: 4, endRowOffset: 344, anchorType: 'twoCell' }, chart: { chartStyleId: 2, stacking: 'clustered', chartColors: { id: 10, meth: 'cycle', schemeClr: [{ color: { type: 'scheme', value: 'accent1' } }, { color: { type: 'scheme', value: 'accent2' } }, { color: { type: 'scheme', value: 'accent3' } }, { color: { type: 'scheme', value: 'accent4' } }, { color: { type: 'scheme', value: 'accent5' } }, { color: { type: 'scheme', value: 'accent6' } }] }, varyColors: false }, line: { type: 'solid', width: 26, color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 15000 }, { type: 'lumOff', value: 85000 }] } }, fill: { type: 'solid', color: { type: 'scheme', value: 'background1' } } } },
            { name: 'insertChartDataSeries', start: [1, 1], series: 0, attrs: { series: { axisXIndex: 0, axisYIndex: 1, type: 'column2d', title: 'Sheet2!$O$2', names: 'Sheet2!$N$3:$N$5', values: 'Sheet2!$O$3:$O$5' }, line: { type: 'none' }, fill: { type: 'solid', color: { type: 'scheme', value: 'accent1' } } } },
            { name: 'insertChartDataSeries', start: [1, 1], series: 1, attrs: { series: { axisXIndex: 0, axisYIndex: 1, type: 'column2d', title: 'Sheet2!$P$2', names: 'Sheet2!$N$3:$N$5', values: 'Sheet2!$P$3:$P$5' }, line: { type: 'none' }, fill: { type: 'solid', color: { type: 'scheme', value: 'accent2' } } } },
            { name: 'setChartAxisAttributes', start: [1, 1], axis: 0, axPos: 'b', crossAx: 1, attrs: { line: { type: 'solid', width: 26, color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 15000 }, { type: 'lumOff', value: 85000 }] } }, fill: { type: 'none' }, axis: { label: true }, character: { color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 65000 }, { type: 'lumOff', value: 35000 }] }, fontSize: 9 } } },
            { name: 'setChartAxisAttributes', start: [1, 1], axis: 1, axPos: 'l', crossAx: 0, attrs: { line: { type: 'none', width: 18 }, fill: { type: 'none' }, axis: { label: true }, character: { color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 65000 }, { type: 'lumOff', value: 35000 }] }, fontSize: 9 } } },
            { name: 'setChartGridlineAttributes', start: [1, 1], axis: 1, attrs: { line: { type: 'solid', width: 26, color: { type: 'scheme', value: 'text1', transformations: [{ type: 'lumMod', value: 15000 }, { type: 'lumOff', value: 85000 }] } }, fill: { type: 'none' } } },
            { name: 'setChartLegendAttributes', start: [1, 1], attrs: { legend: { pos: 'off' } } },
            // Data validation
            { name: 'insertValidation', sheet: 1, index: 0, ranges: [{ start: [4, 5] }], type: 'source', value1: '$E$3:$E$4' },
            { name: 'insertValidation', sheet: 1, index: 1, ranges: [{ start: [4, 0] }], type: 'source', value1: 'Sheet1!$A$5:$A$7' }
        ];

        // initialize test document
        var docModel = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
        });

        // public methods -----------------------------------------------------

        describe('method "getNumberFormatter"', function () {
            it('should exist', function () {
                expect(docModel).to.respondTo('getNumberFormatter');
            });
            it('should return the number formatter', function () {
                expect(docModel.getNumberFormatter()).to.be.an('object');
            });
        });

        describe('method "getFormulaResource"', function () {
            it('should exist', function () {
                expect(docModel).to.respondTo('getFormulaResource');
            });
            it('should return the formula resources', function () {
                expect(docModel.getFormulaResource()).to.be.an('object');
                expect(docModel.getFormulaResource().getFileFormat()).to.equal('ooxml');
            });
        });

        describe('method "getFormulaGrammar"', function () {
            it('should exist', function () {
                expect(docModel).to.respondTo('getFormulaGrammar');
            });
            it('should return the formula grammar', function () {
                var opGrammar = docModel.getFormulaGrammar('op');
                expect(opGrammar).to.be.an('object');
                expect(opGrammar).to.have.a.property('GRAMMAR', 'op');
                expect(opGrammar).to.have.a.property('REF_SYNTAX', 'ooxml');
                var uiGrammar = docModel.getFormulaGrammar('ui');
                expect(uiGrammar).to.be.an('object');
                expect(uiGrammar).to.have.a.property('GRAMMAR', 'ui');
                expect(uiGrammar).to.have.a.property('REF_SYNTAX', 'ooxml');
                var opOoxGrammar = docModel.getFormulaGrammar('op:ooxml');
                expect(opOoxGrammar).to.be.an('object');
                expect(opOoxGrammar).to.equal(opGrammar);
                var opOdfGrammar = docModel.getFormulaGrammar('op:odf');
                expect(opOdfGrammar).to.be.an('object');
                expect(opOdfGrammar).to.have.a.property('GRAMMAR', 'op');
                expect(opOdfGrammar).to.have.a.property('REF_SYNTAX', 'of');
            });
        });

        describe('method "getFormulaParser"', function () {
            it('should exist', function () {
                expect(docModel).to.respondTo('getFormulaParser');
            });
            it('should return the formula parser', function () {
                expect(docModel.getFormulaParser()).to.be.an('object');
            });
        });

        describe('method "getFormulaCompiler"', function () {
            it('should exist', function () {
                expect(docModel).to.respondTo('getFormulaCompiler');
            });
            it('should return the formula compiler', function () {
                expect(docModel.getFormulaCompiler()).to.be.an('object');
            });
        });

        describe('method "getFormulaInterpreter"', function () {
            it('should exist', function () {
                expect(docModel).to.respondTo('getFormulaInterpreter');
            });
            it('should return the formula interpreter', function () {
                expect(docModel.getFormulaInterpreter()).to.be.an('object');
            });
        });

        describe('method "getNameCollection"', function () {
            it('should exist', function () {
                expect(docModel).to.respondTo('getNameCollection');
            });
            it('should return the name collection', function () {
                expect(docModel.getNameCollection()).to.be.an('object');
            });
        });

        describe('method "getListCollection"', function () {
            it('should exist', function () {
                expect(docModel).to.respondTo('getListCollection');
            });
            it('should return the list collection', function () {
                expect(docModel.getListCollection()).to.be.an('object');
            });
        });

        describe('method "getDependencyManager"', function () {
            it('should exist', function () {
                expect(docModel).to.respondTo('getDependencyManager');
            });
            it('should return the dependency manager', function () {
                expect(docModel.getDependencyManager()).to.be.an('object');
            });
        });

        describe('method "getMaxCol"', function () {
            it('should exist', function () {
                expect(docModel).to.respondTo('getMaxCol');
            });
            it('should return the maximum column index', function () {
                expect(docModel.getMaxCol()).to.equal(16383);
            });
        });

        describe('method "getMaxRow"', function () {
            it('should exist', function () {
                expect(docModel).to.respondTo('getMaxRow');
            });
            it('should return the maximum row index', function () {
                expect(docModel.getMaxRow()).to.equal(1048575);
            });
        });

        describe('method "getMaxIndex"', function () {
            it('should exist', function () {
                expect(docModel).to.respondTo('getMaxIndex');
            });
            it('should return the maximum column/row index', function () {
                expect(docModel.getMaxIndex(true)).to.equal(16383);
                expect(docModel.getMaxIndex(false)).to.equal(1048575);
            });
        });

        describe('method "getMaxAddress"', function () {
            it('should exist', function () {
                expect(docModel).to.respondTo('getMaxAddress');
            });
            it('should return the maximum address', function () {
                expect(docModel.getMaxAddress()).to.stringifyTo('XFD1048576');
            });
        });

        describe('method "getSheetRange"', function () {
            it('should exist', function () {
                expect(docModel).to.respondTo('getSheetRange');
            });
            it('should return the sheet range', function () {
                expect(docModel.getSheetRange()).to.stringifyTo('A1:XFD1048576');
            });
        });

        describe('method "generateDeleteSheetOperations"', function () {
            it('should exist', function () {
                expect(docModel).to.respondTo('generateDeleteSheetOperations');
            });

            it('should generate correct cell operations', function () {

                var sheetModel = docModel.getSheetModel(1);
                var cellCollection = sheetModel.getCellCollection();

                // cell formula
                expect(cellCollection.getFormula(a('A1'), 'op')).to.equal('Sheet1!$A$1');
                expect(cellCollection.getFormula(a('A5'), 'op')).to.equal('Sheet2!$A$3');

                // named range
                var nameCollection = docModel.getNameCollection();
                expect(cellCollection.getFormula(a('C1'), 'op')).to.equal('COUNT(ObstPreisGlobal)');
                expect(nameCollection.getName('ObstGlobal').getFormula('op', a('C1'))).to.equal('Sheet1!$A$5:$A$7');
                expect(nameCollection.getName('ObstPreisGlobal').getFormula('op', a('C1'))).to.equal('Sheet1!$B$5:$B$7');

                nameCollection = docModel.getSheetModel(0).getNameCollection();
                expect(cellCollection.getFormula(a('C3'), 'op')).to.equal('COUNT(Sheet1!ObstPreis)');
                expect(nameCollection.getName('ObstPreis').getFormula('op', a('C1'))).to.equal('Sheet1!$B$5:$B$7');

                // named range on sheet2
                nameCollection = docModel.getSheetModel(1).getNameCollection();
                expect(cellCollection.getFormula(a('C8'), 'op')).to.equal('COUNT(Sheet2!TestRangeSheet2)');
                expect(nameCollection.getName('TestRangeSheet2').getFormula('op', a('C1'))).to.equal('Sheet2!$C$5:$C$6');

                // chart Drawing
                var drawingCollection = sheetModel.getDrawingCollection();
                var chartModel = drawingCollection.getModel([0], { type: 'chart' });
                testChartModel(chartModel, { title: 'Sheet1!$B$4', names: 'Sheet1!$A$5:$A$7', values: 'Sheet1!$B$5:$B$7' }, { title: 'Sheet1!$C$4', names: 'Sheet1!$A$5:$A$7', values: 'Sheet1!$C$5:$C$7' });

                chartModel = drawingCollection.getModel([1], { type: 'chart' });
                testChartModel(chartModel, { title: 'Sheet2!$O$2', names: 'Sheet2!$N$3:$N$5', values: 'Sheet2!$O$3:$O$5' }, { title: 'Sheet2!$P$2', names: 'Sheet2!$N$3:$N$5', values: 'Sheet2!$P$3:$P$5' });

                // data validation
                var validationCollection = sheetModel.getValidationCollection();
                expect(validationCollection.getValidationSettings(a('E1')).attributes.value1).to.equal('Sheet1!$A$5:$A$7');
                expect(validationCollection.getValidationSettings(a('E6')).attributes.value1).to.equal('$E$3:$E$4');

                // delete Sheet1
                var generator = docModel.createSheetOperationGenerator({ applyImmediately: true });
                var promise = docModel.generateDeleteSheetOperations(generator, 0);

                return promise.then(function () {

                    // cell formula
                    expect(cellCollection.getFormula(a('A1'), 'op')).to.equal('#REF!');
                    expect(cellCollection.getFormula(a('A5'), 'op')).to.equal('Sheet2!$A$3');

                    // named range
                    nameCollection = docModel.getNameCollection();
                    expect(cellCollection.getFormula(a('C1'), 'op')).to.equal('COUNT(ObstPreisGlobal)');
                    expect(nameCollection.getName('ObstGlobal').getFormula('op', a('C1'))).to.equal('#REF!');
                    expect(nameCollection.getName('ObstPreisGlobal').getFormula('op', a('C1'))).to.equal('#REF!');

                    expect(cellCollection.getFormula(a('C3'), 'op')).to.equal('COUNT([0]!ObstPreis)');

                    // named range on sheet2
                    nameCollection = docModel.getSheetModel(0).getNameCollection();
                    expect(cellCollection.getFormula(a('C8'), 'op')).to.equal('COUNT(Sheet2!TestRangeSheet2)');
                    expect(nameCollection.getName('TestRangeSheet2').getFormula('op', a('C1'))).to.equal('Sheet2!$C$5:$C$6');

                    // chart Drawing
                    drawingCollection = sheetModel.getDrawingCollection();
                    chartModel = drawingCollection.getModel([0], { type: 'chart' });
                    testChartModel(chartModel, { title: '#REF!', names: '#REF!', values: '#REF!' }, { title: '#REF!', names: '#REF!', values: '#REF!' });

                    chartModel = drawingCollection.getModel([1], { type: 'chart' });
                    testChartModel(chartModel, { title: 'Sheet2!$O$2', names: 'Sheet2!$N$3:$N$5', values: 'Sheet2!$O$3:$O$5' }, { title: 'Sheet2!$P$2', names: 'Sheet2!$N$3:$N$5', values: 'Sheet2!$P$3:$P$5' });

                    // data validation
                    validationCollection = sheetModel.getValidationCollection();
                    expect(validationCollection.getValidationSettings(a('E1')).attributes.value1).to.equal('#REF!');
                    expect(validationCollection.getValidationSettings(a('E6')).attributes.value1).to.equal('$E$3:$E$4');
                });
            });
        });
    });

    // ========================================================================
});
