/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/deps/dependencymanager'
], function (AppHelper, SheetHelper, SheetUtils, DependencyManager) {

    'use strict';

    // convenience shortcuts
    var a = SheetHelper.a;
    var r = SheetHelper.r;
    var ErrorCode = SheetUtils.ErrorCode;
    var Direction = SheetUtils.Direction;

    // class DependencyManager ================================================

    describe('Spreadsheet class DependencyManager', function () {

        it('should exist', function () {
            expect(DependencyManager).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertAutoStyle', styleId: 'a0', attrs: {}, default: true },

            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'changeCells', sheet: 0, start: 'B6', contents: [[0]] },
            { name: 'changeCells', sheet: 0, start: 'A9', contents: [[{ v: 0, f: 'B6*2' }], [{ v: 1, f: 'A9+1' }], [{ v: 1, f: 'B6+A10' }], [{ v: 2, f: 'SUM(A9:A11)' }]] },
            { name: 'changeCells', sheet: 0, start: 'A100', contents: [[1, 2], [3, 4]] },
            { name: 'insertCFRule', sheet: 0, id: '0', ranges: 'A100:B101', type: 'equal', value1: 'Sheet2!A100', priority: 1, attrs: { character: { bold: true } } },

            // bug 51453: recalculate formulas with wrong result types after import
            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
            { name: 'changeCells', sheet: 1, start: 'A1', contents: [[{ v: null, f: '1+1' }, { v: '2', f: '1+1' }]] },

            // bug 51455: recalculate formulas with table references after undoing deleted table
            { name: 'insertSheet', sheet: 2, sheetName: 'Sheet3' },
            { name: 'changeCells', sheet: 2, start: 'A1', contents: [['Col1', 'Col2'], [{ v: 0, f: 'Sheet1!B6' }, 2]] }, // table range
            { name: 'changeCells', sheet: 2, start: 'B3', contents: [[{ v: 2, f: 'SUM(Table1[])', r: 2 }]] },
            { name: 'insertTable', sheet: 2, table: 'Table1', start: 'A1', end: 'B2', attrs: { table: { headerRow: true } } }
        ];

        // initialize test document
        var docModel = null, depManager = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
            depManager = docModel.getDependencyManager();
        });

        function waitForEvent(source, type) {
            return docModel.waitForEvent(source, type, 1000);
        }

        function waitForRecalcEnd() {
            return waitForEvent(depManager, 'recalc:end');
        }

        function getSheetModel(sheet) {
            return docModel.getSheetModel(sheet);
        }

        function getCellCollection(sheet) {
            return getSheetModel(sheet).getCellCollection();
        }

        function getCellModel(sheet, address) {
            return getCellCollection(sheet).getCellModel(a(address));
        }

        function expectCellValue(sheet, address, value) {
            var cellModel = getCellModel(sheet, address);
            expect(cellModel).to.have.a.property('v', value);
            return cellModel;
        }

        // changes the value of a single cell
        function changeCell(sheet, address, value) {
            docModel.invokeOperationHandler({ name: 'changeCells', sheet: sheet, start: a(address).toJSON(), contents: [{ c: [{ v: value }] }] });
        }

        // changes the formula expression of a single cell
        function changeFormula(sheet, address, formula) {
            docModel.invokeOperationHandler({ name: 'changeCells', sheet: sheet, start: a(address).toJSON(), contents: [{ c: [{ f: formula }] }] });
        }

        // indirect tests -----------------------------------------------------

        it('should be contained in a document', function () {
            expect(depManager).to.be.an.instanceof(DependencyManager);
        });

        describe('method "recalcFormulas"', function () {
            it('should wait for initial update cycle', function () {
                return depManager.recalcFormulas({ volatile: false, timeout: 1000 }).then(function () {
                    // US DOCS-650: recalculate formulas with wrong cell value type
                    expectCellValue(1, 'A1', 2);
                    expectCellValue(1, 'B1', 2);
                });
            });
        });

        describe('background update task', function () {

            it('should update formula results after changing a cell', function () {
                changeCell(0, 'B6', 1);
                return waitForRecalcEnd().then(function () {
                    expectCellValue(0, 'A9', 2);
                    expectCellValue(0, 'A10', 3);
                    expectCellValue(0, 'A11', 4);
                    expectCellValue(0, 'A12', 9);
                    expectCellValue(2, 'B3', 3);
                    expectCellValue(2, 'C3', 3);
                });
            });

            it('should update formatting rules after changing a cell', function () {
                changeCell(1, 'A100', 1);
                return waitForEvent(getSheetModel(0), 'refresh:ranges').then(function (args) {
                    expect(args[1]).to.stringifyTo('A100:B101');
                });
            });

            it('should detect circular references', function () {
                changeFormula(0, 'A1000', 'A1000+1');
                return waitForRecalcEnd().then(function (args) {
                    var cycles = args[1];
                    expect(cycles).to.be.an('array').with.length(1);
                    expect(cycles[0]).to.stringifyTo('A1000');
                    expect(cycles[0][0]).to.have.a.property('sheetUid', docModel.getSheetModel(0).getUid());
                    expect(cycles[0][0]).to.have.a.property('sheet', 0);
                });
            });

            it('should update formulas in a copied sheet', function () { // bug 51453
                var toSheet = docModel.getSheetCount();
                docModel.invokeOperationHandler({ name: 'copySheet', sheet: 0, to: toSheet, sheetName: 'CopyOfSheet1' });
                changeCell(toSheet, 'B6', 0);
                return waitForRecalcEnd().then(function () {
                    expectCellValue(toSheet, 'A9', 0);
                    expectCellValue(toSheet, 'A10', 1);
                    expectCellValue(toSheet, 'A11', 1);
                    expectCellValue(toSheet, 'A12', 2);
                });
            });

            it.skip('should update formulas after undoing a deleted table range', function () { // bug 51455
                var sheetModel = getSheetModel(2);
                var cellCollection = sheetModel.getCellCollection();
                var generator = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise = cellCollection.generateMoveCellsOperations(generator, r('A1:B1048576'), Direction.LEFT);
                promise = promise.then(waitForRecalcEnd).then(function () {
                    expectCellValue(2, 'A3', ErrorCode.REF);
                });
                promise = promise.then(function () {
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                });
                return promise.then(waitForRecalcEnd).then(function () {
                    expectCellValue(2, 'B3', 3);
                    expectCellValue(2, 'C3', 3);
                });
            });
        });
    });

    // ========================================================================
});
