/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/presentation/components/drawing/drawingresize',
    'io.ox/office/presentation/utils/presentationutils',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/tk/utils'
], function (AppHelper, DrawingResize, PresentationUtils, DOM, Utils) {

    'use strict';

    // class DrawingResize ==============================================

    describe('Presentation class DrawingResize', function () {

        // private helpers ----------------------------------------------------

        var slide1Id = 'slide_1',
            slide2Id = 'slide_2',
            slide3Id = 'slide_3',
            layoutId_1 = 'layout1',
            masterId_1 = 'master1',
            ctrTitleLeft = 1900,
            subTitleLeft = 3800,
            ctrTitleTop = 5900,
            subTitleTop = 10800,
            ctrTitleHeight = 4000,
            subTitleHeight = 5000,
            ctrTitleWidth = 21600,
            subTitleWidth = 17800,

            drawZeroLeft = 2903,
            drawZeroTop = 2573,
            drawOneLeft = 7553,
            drawOneTop = 2573,
            drawTwoLeft = 2903,
            drawTwoTop = 8573,

            drawThreeLeft = 1000,
            drawThreeTop = 1000,
            drawFourLeft = 4000,
            drawFourTop = 4000,

            drawFiveLeft = 1000,
            drawFiveTop = 1000,
            drawSixLeft = 1000,
            drawSixTop = 1000,

            arrowOneLeft = 10000,
            arrowOneTop = 10000,

            // the operations to be applied by the document model
            OPERATIONS = [
                {
                    name: 'setDocumentAttributes',
                    attrs: { page: { width: 33866, height: 19050, orientation: 'landscape' } }
                },
                { name: 'insertMasterSlide', id: masterId_1 },

                { name: 'insertLayoutSlide', id: layoutId_1, target: masterId_1 },
                { name: 'insertDrawing', start: [0, 0], target: layoutId_1, type: 'shape', attrs: { presentation: { phType: 'ctrTitle' }, drawing: { name: 'Titel 1', left: ctrTitleLeft, top: ctrTitleTop, width: ctrTitleWidth, height: ctrTitleHeight } } },
                { name: 'insertParagraph', start: [0, 0, 0], target: layoutId_1 },
                { name: 'insertText', start: [0, 0, 0, 0], target: layoutId_1, text: 'Mastertitelformat bearbeiten' },
                { name: 'insertDrawing', start: [0, 1], target: layoutId_1, type: 'shape', attrs: { presentation: { phType: 'subTitle', phIndex: 1 }, drawing: { name: 'Untertitel 2', left: subTitleLeft, top: subTitleTop, width: subTitleWidth, height: subTitleHeight } } },
                { name: 'insertParagraph', start: [0, 1, 0], target: layoutId_1 },
                { name: 'insertText', start: [0, 1, 0, 0], target: layoutId_1, text: 'Master-Untertitelformat bearbeiten' },

                { name: 'insertSlide', start: [0], target: layoutId_1 },
                { name: 'insertDrawing', start: [0, 0], type: 'shape', attrs: { drawing: { name: 'TextBox 3', left: drawZeroLeft, top: drawZeroTop, width: 3958, height: 1026 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: 'FF0000' } } } },
                { name: 'insertParagraph', start: [0, 0, 0] },
                { name: 'insertText', start: [0, 0, 0, 0], text: 'test' },
                { name: 'insertDrawing', start: [0, 1], type: 'shape', attrs: { drawing: { name: 'TextBox 4', left: drawOneLeft, top: drawOneTop, width: 3958, height: 1026 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: 'FF0000' } } } },
                { name: 'insertParagraph', start: [0, 1, 0] },
                { name: 'insertText', start: [0, 1, 0, 0], text: 'test2' },
                // drawing without filling
                { name: 'insertDrawing', start: [0, 2], type: 'shape', attrs: { drawing: { name: 'TextBox 5', left: drawTwoLeft, top: drawTwoTop, width: 3958, height: 1026 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'none' }, line: { type: 'solid', style: 'solid', width: 26, color: { type: 'rgb', value: 'FF0000' } } } },
                { name: 'insertParagraph', start: [0, 2, 0] },
                { name: 'insertText', start: [0, 2, 0, 0], text: 'test3' },
                // drawing 'rightArrow'
                { name: 'insertDrawing', start: [0, 3], type: 'shape', attrs: { drawing: { name: 'RightArrow 1', left: arrowOneLeft, top: arrowOneTop, width: 3958, height: 1026 }, geometry: { presetShape: 'rightArrow', avList: {} }, fill: { type: 'none' }, line: { type: 'solid', style: 'solid', width: 26, color: { type: 'rgb', value: 'FF0000' } } } },
                { name: 'insertParagraph', start: [0, 3, 0] },
                { name: 'insertText', start: [0, 3, 0, 0], text: 'right arrow' },

                // inserting a second slide with a non-transparent and a transparent diamond drawing
                { name: 'insertSlide', start: [1], target: layoutId_1 },
                // not transparent diamond drawing
                { name: 'insertDrawing', start: [1, 0], type: 'shape', attrs: { drawing: { name: 'Diamond 1', left: drawThreeLeft, top: drawThreeTop, width: 2000, height: 2000 }, geometry: { presetShape: 'diamond', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: 'FFFF00' } }, line: { type: 'solid', style: 'solid', width: 26, color: { type: 'rgb', value: 'FF0000' } } } },
                { name: 'insertParagraph', start: [1, 0, 0] },
                { name: 'insertText', start: [1, 0, 0, 0], text: 'slide 2 drawing 1' },
                // transparent diamond drawing (even without paragraph)
                { name: 'insertDrawing', start: [1, 1], type: 'shape', attrs: { drawing: { name: 'Diamond 2', left: drawFourLeft, top: drawFourTop, width: 2000, height: 2000 }, geometry: { presetShape: 'diamond', avList: {} }, fill: { type: 'none' }, line: { type: 'solid', style: 'solid', width: 26, color: { type: 'rgb', value: 'FF0000' } } } },

                // inserting a third slide with overlapping drawings (underlying drawing is filled, upper drawing has only border)
                { name: 'insertSlide', start: [2], target: layoutId_1 },
                // not transparent diamond drawing
                { name: 'insertDrawing', start: [2, 0], type: 'shape', attrs: { drawing: { name: 'Diamond 1', left: drawFiveLeft, top: drawFiveTop, width: 2000, height: 2000 }, geometry: { presetShape: 'diamond', avList: {} }, fill: { type: 'solid', color: { type: 'rgb', value: 'FFFF00' } }, line: { type: 'solid', style: 'solid', width: 26, color: { type: 'rgb', value: 'FF0000' } } } },
                { name: 'insertParagraph', start: [2, 0, 0] },
                { name: 'insertText', start: [2, 0, 0, 0], text: 'slide 2 drawing 1' },
                // transparent diamond drawing at exact the same position as the not transparent drawing
                { name: 'insertDrawing', start: [2, 1], type: 'shape', attrs: { drawing: { name: 'Diamond 2', left: drawSixLeft, top: drawSixTop, width: 2000, height: 2000 }, geometry: { presetShape: 'diamond', avList: {} }, fill: { type: 'none' }, line: { type: 'solid', style: 'solid', width: 26, color: { type: 'rgb', value: 'FF0000' } } } }

            ];

        var app = null;
        var model = null;
        var selection = null;

        function waitForEvent(source, type) {
            return model.waitForEvent(source, type, 1000);
        }

        AppHelper.createPresentationApp('ooxml', OPERATIONS).done(function (theApp) {
            app = theApp;
            model = app.getModel();
            selection = model.getSelection();

            selection.setDrawingResizeHandler(DrawingResize.drawDrawingSelection);
        });

        function triggerTrackingEvent(node, type, x, y, options) {
            var isCopy = Utils.getBooleanOption(options, 'copy', false);
            var positionNode = Utils.getObjectOption(options, 'drawingNode', node);
            var startX = parseFloat(positionNode.css('left'));
            var startY = parseFloat(positionNode.css('top'));
            var event = new $.Event(type, {
                which: 1,
                startX: startX,
                startY: startY,
                pageX: x + startX,
                pageY: y + startY,
                ctrlKey: isCopy
            });
            node.trigger(event);
        }

        function createMouseDownEvent(triggerNode, positionNode, x, y) {
            var event = new $.Event('mousedown', {
                which: 1,
                button: 0,
                target: triggerNode[0],
                pageX: parseFloat(positionNode.css('left')) + x,
                pageY: parseFloat(positionNode.css('top')) + y
            });

            return event;
        }

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(DrawingResize).to.be.a('object');
        });

        // public methods -----------------------------------------------------

        describe('method "DrawingResize.drawDrawingSelection"', function () {

            it('should exist', function () {
                expect(DrawingResize).to.respondTo('drawDrawingSelection');
            });

            it('move drawing with "tracking.js" should work fine (for drawing with filling)', function () {

                selection.setTextSelection([0, 0], [0, 1]);

                var firstDrawing = selection.getSelectedDrawing();
                var oldSetLeft = Utils.convertCssLengthToHmm(firstDrawing.css('left'));
                var oldSetTop = Utils.convertCssLengthToHmm(firstDrawing.css('top'));

                expect(oldSetLeft).to.be.closeTo(drawZeroLeft, 20);
                expect(oldSetTop).to.be.closeTo(drawZeroTop, 20);

                triggerTrackingEvent(firstDrawing, 'tracking:start', 0, 0);
                triggerTrackingEvent(firstDrawing, 'tracking:move', 10, 35);
                triggerTrackingEvent(firstDrawing, 'tracking:end', 10, 35);

                var newSetLeft = Utils.convertCssLengthToHmm(firstDrawing.css('left'));
                var newSetTop = Utils.convertCssLengthToHmm(firstDrawing.css('top'));

                // move in unit tests always ends at position 0, because of movebox
                // -> because the movebox is not attached to the DOM, the offset() function returns (0,0),
                //    independent from the used values.
                expect(newSetLeft).to.be.closeTo(0, 20);
                expect(newSetTop).to.be.closeTo(0, 20);

                var newAttributes = firstDrawing.data('attributes');
                expect(newAttributes.drawing.left).to.be.closeTo(0, 20);
                expect(newAttributes.drawing.top).to.be.closeTo(0, 20);

            });

            it('should not move drawing without filling when clicking into content', function () {

                selection.setTextSelection([0, 2], [0, 3]);

                var thirdDrawing = selection.getSelectedDrawing();
                var oldSetLeft = Utils.convertCssLengthToHmm(thirdDrawing.css('left'));
                var oldSetTop = Utils.convertCssLengthToHmm(thirdDrawing.css('top'));

                expect(oldSetLeft).to.be.closeTo(drawTwoLeft, 20);
                expect(oldSetTop).to.be.closeTo(drawTwoTop, 20);

                triggerTrackingEvent(thirdDrawing, 'tracking:start', 10, 10); // position in area
                triggerTrackingEvent(thirdDrawing, 'tracking:move', 10, 35);
                triggerTrackingEvent(thirdDrawing, 'tracking:end', 10, 35);

                var newSetLeft = Utils.convertCssLengthToHmm(thirdDrawing.css('left'));
                var newSetTop = Utils.convertCssLengthToHmm(thirdDrawing.css('top'));

                expect(newSetLeft).to.be.equal(oldSetLeft);
                expect(newSetTop).to.be.equal(oldSetTop);
            });

            it('should move drawing without filling when clicking onto border', function () {

                selection.setTextSelection([0, 2], [0, 3]);

                var thirdDrawing = selection.getSelectedDrawing();
                var oldSetLeft = Utils.convertCssLengthToHmm(thirdDrawing.css('left'));
                var oldSetTop = Utils.convertCssLengthToHmm(thirdDrawing.css('top'));

                expect(oldSetLeft).to.be.closeTo(drawTwoLeft, 20);
                expect(oldSetTop).to.be.closeTo(drawTwoTop, 20);

                triggerTrackingEvent(thirdDrawing, 'tracking:start', 0, 0); // position on border
                triggerTrackingEvent(thirdDrawing, 'tracking:move', 10, 35);
                triggerTrackingEvent(thirdDrawing, 'tracking:end', 10, 35);

                var newSetLeft = Utils.convertCssLengthToHmm(thirdDrawing.css('left'));
                var newSetTop = Utils.convertCssLengthToHmm(thirdDrawing.css('top'));

                // move in unit tests always ends at position 0, because of movebox
                // -> because the movebox is not attached to the DOM, the offset() function returns (0,0),
                //    independent from the used values.
                expect(newSetLeft).to.be.closeTo(0, 10);
                expect(newSetTop).to.be.closeTo(0, 10);
            });

            it('copy drawing with "tracking.js" should work fine', function () {

                selection.setTextSelection([0, 1], [0, 2]);

                var activeSlide = model.getSlideById(model.getActiveSlideId());
                var drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                var secondDrawing = selection.getSelectedDrawing();

                expect(drawingsOnSlide.length).to.equal(4);

                var oldSetLeft = Utils.convertCssLengthToHmm(secondDrawing.css('left'));
                var oldSetTop = Utils.convertCssLengthToHmm(secondDrawing.css('top'));
                var oldAttributes = secondDrawing.data('attributes');

                expect(oldSetLeft).to.be.closeTo(drawOneLeft, 20);
                expect(oldSetTop).to.be.closeTo(drawOneTop, 20);

                triggerTrackingEvent(secondDrawing, 'tracking:start', 0, 0, { copy: true });
                triggerTrackingEvent(secondDrawing, 'tracking:move', 10, 35, { copy: true });
                triggerTrackingEvent(secondDrawing, 'tracking:end', 10, 35, { copy: true });

                drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                expect(drawingsOnSlide.length).to.equal(5); // one new drawing generated

                var newDrawing = drawingsOnSlide.last();

                var newSetLeft = Utils.convertCssLengthToHmm(newDrawing.css('left'));
                var newSetTop = Utils.convertCssLengthToHmm(newDrawing.css('top'));

                // move in unit tests always ends at position 0, because of movebox
                // -> because the movebox is not attached to the DOM, the offset() function returns (0,0),
                //    independent from the used values.
                expect(newSetLeft).to.be.closeTo(0, 20);
                expect(newSetTop).to.be.closeTo(0, 20);

                var newAttributes = newDrawing.data('attributes');

                expect(newAttributes.drawing.left).to.be.closeTo(0, 20);
                expect(newAttributes.drawing.top).to.be.closeTo(0, 20);

                expect(newAttributes.drawing.width).to.be.equal(oldAttributes.drawing.width);
                expect(newAttributes.drawing.height).to.be.equal(oldAttributes.drawing.height);

            });

            // working on right arrow at position [0,3]
            it('should flip drawing horizontally on resize', function () {

                selection.setTextSelection([0, 3], [0, 4]);

                var lastDrawing =  selection.getSelectedDrawing();
                var oldAttributes = lastDrawing.data('attributes');

                expect(oldAttributes.drawing.flipH).not.to.be.equal(true);

                var selectionDrawingNode = model.getNode().children('.drawingselection-overlay').children('.selectiondrawing').first();
                var leftResizer = selectionDrawingNode.find('.resizers > [data-pos="l"]');

                triggerTrackingEvent(leftResizer, 'tracking:start', 0, 0, { drawingNode: lastDrawing });
                triggerTrackingEvent(leftResizer, 'tracking:move', 500, 0, { drawingNode: lastDrawing });
                triggerTrackingEvent(leftResizer, 'tracking:end', 500, 0, { drawingNode: lastDrawing });
                var newAttributes = lastDrawing.data('attributes');

                expect(newAttributes.drawing.height).to.be.closeTo(oldAttributes.drawing.height, 20);
                expect(newAttributes.drawing.width).not.to.be.equal(oldAttributes.drawing.width);
                expect(newAttributes.drawing.width).to.be.closeTo(Utils.convertLengthToHmm(500, 'px') - oldAttributes.drawing.width, 20);
                expect(newAttributes.drawing.flipH).to.be.equal(true).and.not.to.be.equal(oldAttributes.drawing.flipH);
            });

            // working on right arrow at position [0,3]
            it('should flip drawing vertically on resize', function () {

                selection.setTextSelection([0, 3], [0, 4]);

                var lastDrawing =  selection.getSelectedDrawing();
                var oldAttributes = lastDrawing.data('attributes');

                expect(oldAttributes.drawing.flipV).not.to.be.equal(true);

                var selectionDrawingNode = model.getNode().children('.drawingselection-overlay').children('.selectiondrawing').first();
                var topResizer = selectionDrawingNode.find('.resizers > [data-pos="t"]');

                triggerTrackingEvent(topResizer, 'tracking:start', 0, 0, { drawingNode: lastDrawing });
                triggerTrackingEvent(topResizer, 'tracking:move', 0, 500, { drawingNode: lastDrawing });
                triggerTrackingEvent(topResizer, 'tracking:end', 0, 500, { drawingNode: lastDrawing });
                var newAttributes = lastDrawing.data('attributes');

                expect(newAttributes.drawing.width).to.be.closeTo(oldAttributes.drawing.width, 20);
                expect(newAttributes.drawing.height).not.to.be.equal(oldAttributes.drawing.height);
                expect(newAttributes.drawing.height).to.be.closeTo(Utils.convertLengthToHmm(500, 'px') - oldAttributes.drawing.height, 20);
                expect(newAttributes.drawing.flipV).to.be.equal(true).and.not.to.be.equal(oldAttributes.drawing.flipV);
            });

            // working on right arrow at position [0,3]
            it('should change shape appearence on adjusting', function () {

                selection.setTextSelection([0, 3], [0, 4]);

                var lastDrawing =  selection.getSelectedDrawing();
                var oldAttributes = lastDrawing.data('attributes');
                var newAttributes = null, selectionDrawingNode = null, adjHandle;

                expect(oldAttributes.geometry.avList).to.be.an('object');
                expect(oldAttributes.geometry.avList).to.be.empty;
                selectionDrawingNode = model.getNode().children('.drawingselection-overlay').children('.selectiondrawing').first();
                adjHandle = selectionDrawingNode.find('.adj-handle').last();

                triggerTrackingEvent(adjHandle, 'tracking:start', 0, 0, { drawingNode: lastDrawing });
                triggerTrackingEvent(adjHandle, 'tracking:move', 500, 50, { drawingNode: lastDrawing });
                triggerTrackingEvent(adjHandle, 'tracking:end', 500, 50, { drawingNode: lastDrawing });
                newAttributes = lastDrawing.data('attributes');

                expect(newAttributes.geometry.avList).to.be.an('object');
                expect(newAttributes.geometry.avList).not.to.be.empty;
                expect(newAttributes.geometry.avList.adj1).to.equal(50000);
                expect(newAttributes.geometry.avList.adj2).to.equal(100000);

                selectionDrawingNode = model.getNode().children('.drawingselection-overlay').children('.selectiondrawing').first();
                adjHandle = selectionDrawingNode.find('.adj-handle').last();
                triggerTrackingEvent(adjHandle, 'tracking:start', 0, 0, { drawingNode: lastDrawing });
                triggerTrackingEvent(adjHandle, 'tracking:move', -1000, 0, { drawingNode: lastDrawing });
                triggerTrackingEvent(adjHandle, 'tracking:end', -1000, 0, { drawingNode: lastDrawing });
                newAttributes = lastDrawing.data('attributes');

                expect(newAttributes.geometry.avList).not.to.be.empty;
                expect(newAttributes.geometry.avList.adj1).to.equal(50000);
                expect(newAttributes.geometry.avList.adj2).to.equal(0);
            });

        });

        // adding tests for selecting and moving of drawings with and without filling
        // -> this scenario contains two diamond drawings at position (1cm, 1cm) and (4cm, 4cm) and with
        //    width and height of 2cm, so that they do not intersect.
        //    The drawing at (1cm, 1cm) has a filling, the other drawing has no filling.
        describe('method "DrawingResize.drawDrawingSelection" for transparent drawings', function () {

            it('should change to the following slide (the second)', function () {

                var activeSlideId = model.getActiveSlideId();

                expect(activeSlideId).to.equal(slide1Id);

                model.changeToNeighbourSlide(); // selecting the second slide (for example using right scroll bar)

                return waitForEvent(model, 'change:activeslide:done').then(function (args) {
                    expect(args[1]).to.equal(slide2Id);

                    activeSlideId = model.getActiveSlideId();
                    expect(activeSlideId).to.equal(slide2Id);

                    var activeSlide = model.getSlideById(slide2Id);
                    var drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);

                    expect(drawingsOnSlide.length).to.equal(2);

                    expect(selection.isSlideSelection()).to.equal(true);
                });
            });

            it('should select the diamond drawing on the slide, if the canvas contains content', function () {

                selection.setSlideSelection();

                // no drawing selected
                var allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });

                expect(allSelectedDrawings.length).to.equal(0);

                var activeSlide = model.getSlideById(model.getActiveSlideId());
                var drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);

                expect(drawingsOnSlide.length).to.equal(2);

                var diamond = $(drawingsOnSlide[0]);
                var contentNode = diamond.children('.content');

                // clicking into the center of the diamond (with width and height of 76px)
                var event = createMouseDownEvent(contentNode, diamond, 38, 38);

                var handleEvent = PresentationUtils.checkEventPosition(app, event, diamond);
                expect(handleEvent).to.equal(true);

                contentNode.trigger(event);

                allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });
                expect(allSelectedDrawings.length).to.equal(1);
                expect(Utils.getDomNode(allSelectedDrawings[0])).to.equal(Utils.getDomNode(diamond));
                expect(selection.isSlideSelection()).to.equal(false);

                // resetting selection, unfortunately clicking next to slide does not work in
                // unit test, because it starts selection box
                selection.setSlideSelection();

                allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });
                expect(allSelectedDrawings.length).to.equal(0);
                expect(selection.isSlideSelection()).to.equal(true);

                // clicking on the drawing, but next to the diamond

                event = createMouseDownEvent(contentNode, diamond, 6, 6);
                handleEvent = PresentationUtils.checkEventPosition(app, event, diamond);
                expect(handleEvent).to.equal(false);

                contentNode.trigger(event);

                allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });
                expect(allSelectedDrawings.length).to.equal(0);
                expect(selection.isSlideSelection()).to.equal(true);
            });

            it('should not move the diamond drawing with filling, if the canvas contains no content', function () {

                selection.setSlideSelection();

                // no drawing selected
                var allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });

                expect(allSelectedDrawings.length).to.equal(0);

                var activeSlide = model.getSlideById(model.getActiveSlideId());
                var drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);

                expect(drawingsOnSlide.length).to.equal(2);

                var diamond = $(drawingsOnSlide[0]);
                var contentNode = diamond.children('.content');

                var oldSetLeft = Utils.convertCssLengthToHmm(diamond.css('left'));
                var oldSetTop = Utils.convertCssLengthToHmm(diamond.css('top'));

                expect(oldSetLeft).to.be.closeTo(drawThreeLeft, 10);
                expect(oldSetTop).to.be.closeTo(drawThreeTop, 10);

                // clicking into the center of the diamond (with width and height of 76px) to select the drawing
                var mouseDownEvent = createMouseDownEvent(contentNode, diamond, 38, 38);

                var handleEvent = PresentationUtils.checkEventPosition(app, mouseDownEvent, diamond);
                expect(handleEvent).to.equal(true);

                contentNode.trigger(mouseDownEvent);

                allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });
                expect(allSelectedDrawings.length).to.equal(1);
                expect(Utils.getDomNode(allSelectedDrawings[0])).to.equal(Utils.getDomNode(diamond));
                expect(selection.isSlideSelection()).to.equal(false);

                // click into shape and canvas, but not into canvas content
                triggerTrackingEvent(diamond, 'tracking:start', 6, 6);
                triggerTrackingEvent(diamond, 'tracking:move', 10, 35); // arbitrary numbers
                triggerTrackingEvent(diamond, 'tracking:end', 10, 35);

                var newSetLeft = Utils.convertCssLengthToHmm(diamond.css('left'));
                var newSetTop = Utils.convertCssLengthToHmm(diamond.css('top'));

                // no movement of drawing
                expect(newSetLeft).to.be.closeTo(drawThreeLeft, 10);
                expect(newSetTop).to.be.closeTo(drawThreeTop, 10);
            });

            it('should move the diamond drawing with filling, if the canvas contains content', function () {

                selection.setSlideSelection();

                // no drawing selected
                var allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });

                expect(allSelectedDrawings.length).to.equal(0);

                var activeSlide = model.getSlideById(model.getActiveSlideId());
                var drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);

                expect(drawingsOnSlide.length).to.equal(2);

                var diamond = $(drawingsOnSlide[0]);
                var contentNode = diamond.children('.content');

                var oldSetLeft = Utils.convertCssLengthToHmm(diamond.css('left'));
                var oldSetTop = Utils.convertCssLengthToHmm(diamond.css('top'));

                expect(oldSetLeft).to.be.closeTo(drawThreeLeft, 10);
                expect(oldSetTop).to.be.closeTo(drawThreeTop, 10);

                // clicking into the center of the diamond (with width and height of 76px) to select the drawing
                var mouseDownEvent = createMouseDownEvent(contentNode, diamond, 38, 38);

                var handleEvent = PresentationUtils.checkEventPosition(app, mouseDownEvent, diamond);
                expect(handleEvent).to.equal(true);

                contentNode.trigger(mouseDownEvent);

                allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });
                expect(allSelectedDrawings.length).to.equal(1);
                expect(Utils.getDomNode(allSelectedDrawings[0])).to.equal(Utils.getDomNode(diamond));
                expect(selection.isSlideSelection()).to.equal(false);

                // click into shape and canvas and into canvas content
                triggerTrackingEvent(diamond, 'tracking:start', 38, 38); // click on into diamond
                triggerTrackingEvent(diamond, 'tracking:move', 10, 35); // arbitrary numbers
                triggerTrackingEvent(diamond, 'tracking:end', 10, 35);

                var newSetLeft = Utils.convertCssLengthToHmm(diamond.css('left'));
                var newSetTop = Utils.convertCssLengthToHmm(diamond.css('top'));

                // move in unit tests always ends at position 0, because of movebox
                // -> because the movebox is not attached to the DOM, the offset() function returns (0,0),
                //    independent from the used values.
                expect(newSetLeft).to.be.closeTo(0, 10);
                expect(newSetTop).to.be.closeTo(0, 10);
            });

            it('should select the empty diamond drawing on the slide, only by clicking on the visible border', function () {

                selection.setSlideSelection();

                // no drawing selected
                var allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });

                expect(allSelectedDrawings.length).to.equal(0);

                var activeSlide = model.getSlideById(model.getActiveSlideId());
                var drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);

                expect(drawingsOnSlide.length).to.equal(2);

                var diamond = $(drawingsOnSlide[1]); // using the empty second diamond on this slide
                var contentNode = diamond.children('.content');

                // clicking into the center of the diamond (with width and height of 76px) -> should not select the diamond
                var event = createMouseDownEvent(contentNode, diamond, 38, 38);

                var handleEvent = PresentationUtils.checkEventPosition(app, event, diamond);
                expect(handleEvent).to.equal(false);

                contentNode.trigger(event);

                allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });
                expect(allSelectedDrawings.length).to.equal(0);
                expect(selection.isSlideSelection()).to.equal(true);

                // resetting selection
                selection.setSlideSelection();

                allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });
                expect(allSelectedDrawings.length).to.equal(0);
                expect(selection.isSlideSelection()).to.equal(true);

                // clicking on the visible border of the diamond drawing (with width and height of 38)
                event = createMouseDownEvent(contentNode, diamond, 19, 19);
                handleEvent = PresentationUtils.checkEventPosition(app, event, diamond);
                expect(handleEvent).to.equal(true);

                contentNode.trigger(event);

                allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });
                expect(allSelectedDrawings.length).to.equal(1);
                expect(Utils.getDomNode(allSelectedDrawings[0])).to.equal(Utils.getDomNode(diamond));
                expect(selection.isSlideSelection()).to.equal(false);
            });

        });

        // adding tests for selecting and moving of underlying drawings
        // -> this scenario contains two diamond drawings at position (1cm, 1cm) with width and height of 2cm.
        //    The upper drawing has no filling, the lower drawing has a filling.
        describe('method "DrawingResize.drawDrawingSelection" for underlying drawings', function () {

            it('should change to the following slide (the third)', function () {

                var activeSlideId = model.getActiveSlideId();

                expect(activeSlideId).to.equal(slide2Id);

                model.changeToNeighbourSlide(); // selecting the second slide (for example using right scroll bar)

                return waitForEvent(model, 'change:activeslide:done').then(function (args) {
                    expect(args[1]).to.equal(slide3Id);

                    activeSlideId = model.getActiveSlideId();
                    expect(activeSlideId).to.equal(slide3Id);

                    var activeSlide = model.getSlideById(slide3Id);
                    var drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);

                    expect(drawingsOnSlide.length).to.equal(2);

                    expect(selection.isSlideSelection()).to.equal(true);
                });
            });

            it('should select the correct drawing, dependent from canvas content', function () {

                selection.setSlideSelection();

                // no drawing selected
                var allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });

                expect(allSelectedDrawings.length).to.equal(0);

                var activeSlide = model.getSlideById(model.getActiveSlideId());
                var drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);

                expect(drawingsOnSlide.length).to.equal(2);

                var filledDiamond = $(drawingsOnSlide[0]);
                var emptyDiamond = $(drawingsOnSlide[1]);
                var emptyContentNode = emptyDiamond.children('.content');

                // first case: clicking into the center of the upper empty diamond (with width and height of 76px)
                var event = createMouseDownEvent(emptyContentNode, emptyDiamond, 38, 38); // trigger node is the upper drawing!

                var handleEvent = PresentationUtils.checkEventPosition(app, event, emptyDiamond);
                expect(handleEvent).to.equal(false);

                var validNode = PresentationUtils.findValidDrawingNodeBehindDrawing(app, event, emptyDiamond);
                expect(Utils.getDomNode(validNode.drawing)).to.equal(Utils.getDomNode(filledDiamond));

                emptyContentNode.trigger(event); // triggering mousedown on upper drawing

                allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });
                expect(allSelectedDrawings.length).to.equal(1);
                // the underlying filled diamond is selected!
                expect(Utils.getDomNode(allSelectedDrawings[0])).to.equal(Utils.getDomNode(filledDiamond));
                expect(selection.isSlideSelection()).to.equal(false);

                // resetting selection
                selection.setSlideSelection();

                allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });
                expect(allSelectedDrawings.length).to.equal(0);
                expect(selection.isSlideSelection()).to.equal(true);

                // second case: clicking at the border of the upper empty diamond (with width and height of 76px)
                event = createMouseDownEvent(emptyContentNode, emptyDiamond, 19, 19); // trigger node is the upper drawing!

                handleEvent = PresentationUtils.checkEventPosition(app, event, emptyDiamond);
                expect(handleEvent).to.equal(true);

                emptyContentNode.trigger(event); // triggering mousedown on upper drawing

                allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });
                expect(allSelectedDrawings.length).to.equal(1);
                // the upper empty diamond is selected!
                expect(Utils.getDomNode(allSelectedDrawings[0])).to.equal(Utils.getDomNode(emptyDiamond));
                expect(selection.isSlideSelection()).to.equal(false);

                // resetting selection
                selection.setSlideSelection();

                allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });
                expect(allSelectedDrawings.length).to.equal(0);
                expect(selection.isSlideSelection()).to.equal(true);

                // third case: clicking into the upper empty diamond, but out of the visible diamond border
                // -> in this case, both drawings must not be selected
                event = createMouseDownEvent(emptyContentNode, emptyDiamond, 6, 6); // trigger node is the upper drawing!

                handleEvent = PresentationUtils.checkEventPosition(app, event, emptyDiamond);
                expect(handleEvent).to.equal(false);

                validNode = PresentationUtils.findValidDrawingNodeBehindDrawing(app, event, emptyDiamond);
                expect(Utils.getDomNode(validNode)).to.equal(null);

                emptyContentNode.trigger(event); // triggering mousedown on upper drawing

                allSelectedDrawings = selection.getSelectedDrawings({ forceArray: true });
                expect(allSelectedDrawings.length).to.equal(0);
                expect(selection.isSlideSelection()).to.equal(true);
            });

        });
    });

    // ========================================================================
});
