/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define.async('io.ox/office/tk/locale/localedata', [
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'io.ox/office/tk/container/valuemap',
    'gettext!io.ox/office/tk/main'
], function (Config, Utils, IO, ValueMap, gt) {

    'use strict';

    // the locale data cache for all locales, mapped by locale code
    var localeDataByCode = null;

    // the locale data cache for all locales, mapped by MS locale identifier
    var localeDataByLcid = null;

    // locale data of current UI locale
    var localeData = null;

    // static cache for JSON resources already loaded, mapped by resource module name and locale name
    var resourceCache = new ValueMap();

    // regular expression to split language and optional region from a locale identifier
    var LOCALE_ID_RE = /^([a-z]+)(?:_([A-Z]+))?$/;

    // the translated names of some supported locales, mapped by locale identifier
    var LANGUAGE_NAMES = {
        cs_CZ: gt('Czech'),
        da_DK: gt('Danish'),
        de_DE: gt('German'),
        el_GR: gt('Greek'),
        en_GB: gt('English (UK)'),
        en_US: gt('English (US)'),
        es_ES: gt('Spanish'),
        fi_FI: gt('Finnish'),
        fr_FR: gt('French'),
        hu_HU: gt('Hungarian'),
        it_IT: gt('Italian (Italy)'),
        ja_JP: gt('Japanese'),
        nl_NL: gt('Dutch (Netherlands)'),
        pl_PL: gt('Polish'),
        pt_PT: gt('Portuguese (Portugal)'),
        ro_RO: gt('Romanian'),
        ru_RU: gt('Russian'),
        sv_SE: gt('Swedish (Sweden)')
    };

    // private global functions ===============================================

    /**
     * Returns the settings for the specified UI locale.
     *
     * @param {String|Number} [locale]
     *  The locale code with underscore character (e.g. 'en_GB'), or a numeric
     *  MS locale identifier (as contained in the property 'lcid' of the locale
     *  data entries). If omitted, the locale data for the current UI locale
     *  will be returned.
     *
     * @returns {LocaleDataEntry|Null}
     *  The locale data entry for the specified locale. If no locale data is
     *  available for the passed locale code or MS locale identifier, null will
     *  be returned instead.
     */
    function getLocaleData(locale) {
        switch (typeof locale) {
            case 'undefined':
                return localeData;
            case 'string':
                return localeDataByCode[locale] || null;
            case 'number':
                return localeDataByLcid[locale] || null;
        }
        return null;
    }

    // class LocaleDataEntry ==================================================

    /**
     * Represents the locale data of a single locale.
     *
     * @constructor
     *
     * @property {String} lc
     *  The locale code, e.g. 'en_GB'.
     *
     * @property {String} language
     *  The lower-case language identifier of the locale code, e.g. 'en'.
     *
     * @property {String} country
     *  The upper-case region identifier of the locale code, e.g. 'GB'.
     *
     * @property {Number} lcid
     *  The MS locale identifier (positive integer).
     *
     * @property {String|Null} name
     *  The name of the language, translated to the current UI language; or
     *  null, if no translated language name is available.
     *
     * @property {String} dec
     *  The decimal separator used in numbers (single character).
     *
     * @property {String} group
     *  The group separator (a.k.a. thousands separator in most locales) used
     *  in numbers (single character).
     *
     * @property {String} dir
     *  The default writing direction (either 'ltr' for left-to-right, or 'rtl'
     *  for right-to-left).
     *
     * @property {Boolean} cjk
     *  Whether the locale represents one of the languages Chinese, Japanese,
     *  or Korean.
     *
     * @property {String} unit
     *  The default measurement unit (one of 'cm', 'mm', or 'in').
     *
     * @property {String} currency
     *  The default currency symbol (non-empty string).
     *
     * @property {String} isoCurrency
     *  The default ISO currency code (three uppercase letters).
     *
     * @property {String} shortDate
     *  The number format code for a short date (month as number).
     *
     * @property {String} longDate
     *  The number format code for a long date (with month name).
     *
     * @property {String} shortTime
     *  The number format code for a short time (without seconds).
     *
     * @property {String} longTime
     *  The number format code for a time (with seconds).
     *
     * @property {Boolean} leadingMonth
     *  Whether short dates will be formatted with the month in front of the
     *  day (regardless of the year's position).
     *
     * @property {Boolean} leadingYear
     *  Whether short dates will be formatted with the year in front of the
     *  day and month.
     *
     * @property {Boolean} hours24
     *  Whether the 24-hours time format is preferred over the 12-hours AM/PM
     *  time format.
     *
     * @property {String} stdToken
     *  The translated number format token for the 'General' format code.
     *
     * @property {Object<String,String>} dateTokens
     *  The translated number format tokens for date/time format codes. Each
     *  property value of this map is a single character. The map contains the
     *  following properties:
     *  - {String} Y
     *      The translated format token for years.
     *  - {String} M
     *      The translated format token for months.
     *  - {String} D
     *      The translated format token for days.
     *  - {String} h
     *      The translated format token for hours.
     *  - {String} m
     *      The translated format token for minutes.
     *  - {String} s
     *      The translated format token for seconds.
     *  - {String} b
     *      The translated format token for buddhist years.
     *  - {String} g
     *      The translated format token for Japanese era names.
     *  - {String} e
     *      The translated format token for Japanese era years.
     *  - {String} a
     *      The translated format token for the name of the day (OOXML).
     *
     * @property {Array<String>} shortWeekdays
     *  The translated short names of all days in a week, starting with Sunday.
     *
     * @property {Array<String>} longWeekdays
     *  The translated full names of all days in a week, starting with Sunday.
     *
     * @property {Array<String>} shortMonths
     *  The translated short names of all months in a year.
     *
     * @property {Array<String>} longMonths
     *  The translated full names of all months in a year.
     *
     * @property {Array<String>} shortQuarters
     *  The translated short names of the four quarters of the year.
     *
     * @property {Array<String>} longQuarters
     *  The translated full names of the four quarters of the year.
     */
    function LocaleDataEntry(lc, jsonData) {

        // add all properties of the raw JSON data
        _.extend(this, jsonData);

        // add defaults for optional properties
        this.cjk = !!this.cjk;

        // add locale identifier, and split language/region codes
        var parts = lc.split('_');
        this.lc = lc;
        this.language = parts[0];
        this.country = parts[1];
        this.name = LANGUAGE_NAMES[lc] || null;

        // convert date/time format code tokens from string to map
        this.dateTokens = (function (t) {
            return { Y: t[0], M: t[1], D: t[2], h: t[3], m: t[4], s: t[5], b: t[6], g: t[7], e: t[8], a: t[9] };
        }(this.dateTokens));

        // add other properties
        this.leadingMonth = /M.*D/.test(this.shortDate);
        this.leadingYear = /Y.*M/.test(this.shortDate);
        this.hours24 = !/AM\/PM/.test(this.longTime);

    } // class LocaleDataEntry

    // static class LocaleData ================================================

    /**
     * Provides various settings for the current UI locale, or for any other
     * supported locale.
     *
     * @property {String} LOCALE
     *  The locale code of the current UI locale, e.g. 'en_GB', provided as a
     *  convenience shortcut for 'LocaleData.get().lc'.
     *
     * @property {String} LANGUAGE
     *  The lower-case language identifier of the current UI locale, e.g. 'en',
     *  provided as a convenience shortcut for 'LocaleData.get().language'.
     *
     * @property {String} COUNTRY
     *  The upper-case country identifier of the current locale, e.g. 'GB',
     *  provided as a convenience shortcut for 'LocaleData.get().country'.
     *
     * @property {Number} LCID
     *  The MS locale identifier of the current UI locale (positive integer),
     *  provided as a convenience shortcut for 'LocaleData.get().lcid'.
     *
     * @property {String|Null} NAME
     *  The translated name of the current UI language; or null, if the
     *  translated language name is not available. Provided as a convenience
     *  shortcut for 'LocaleData.get().name'.
     *
     * @property {String} DEC
     *  The decimal separator of the current UI locale (single character),
     *  provided as a convenience shortcut for 'LocaleData.get().dec'.
     *
     * @property {String} GROUP
     *  The group separator (a.k.a. thousands separator in most locales) of the
     *  current UI locale (single character), provided as a convenience
     *  shortcut for 'LocaleData.get().group'.
     *
     * @property {String} DIR
     *  The default writing direction of the current UI language (either 'ltr'
     *  for left-to-right, or 'rtl' for right-to-left), provided as a
     *  convenience shortcut for 'LocaleData.get().dir'.
     *
     * @property {Boolean} CJK
     *  Whether the locale represents one of the languages Chinese, Japanese,
     *  or Korean, provided as a convenience shortcut for
     *  'LocaleData.get().cjk'.
     *
     * @property {String} UNIT
     *  The default measurement unit to be used in the current UI language
     *  (one of 'cm', 'mm', or 'in'), provided as a convenience shortcut for
     *  'LocaleData.get().unit'.
     *
     * @property {String} CURRENCY
     *  The default currency symbol of the current UI locale, provided as a
     *  convenience shortcut for 'LocaleData.get().currency'.
     *
     * @property {String} ISO_CURRENCY
     *  The default ISO currency code of the current UI locale, provided as a
     *  convenience shortcut for 'LocaleData.get().isoCurrency'.
     *
     * @property {String} SHORT_DATE
     *  The number format code for a short date (month as number) of the
     *  current UI locale, provided as a convenience shortcut for
     *  'LocaleData.get().shortDate'.
     *
     * @property {String} LONG_DATE
     *  The number format code for a long date (with month name) of the current
     *  UI locale, provided as a convenience shortcut for
     *  'LocaleData.get().longDate'.
     *
     * @property {String} SHORT_TIME
     *  The number format code for a short time (without seconds) of the
     *  current UI locale, provided as a convenience shortcut for
     *  'LocaleData.get().shortTime'.
     *
     * @property {String} LONG_TIME
     *  The number format code for a time (with seconds) of the current UI
     *  locale, provided as a convenience shortcut for
     *  'LocaleData.get().longTime'.
     *
     * @property {Boolean} LEADING_MONTH
     *  Whether short dates will be formatted with the month in front of the
     *  day (regardless of the year's position) in the current UI locale.
     *  Provided as a convenience shortcut for 'LocaleData.get().leadingMonth'.
     *
     * @property {Boolean} LEADING_YEAR
     *  Whether short dates will be formatted with the year in front of the
     *  day and month in the current UI locale. Provided as a convenience
     *  shortcut for 'LocaleData.get().leadingYear'.
     *
     * @property {Boolean} HOURS24
     *  Whether the 24-hours time format is preferred over the 12-hours AM/PM
     *  time format in the current UI locale. Provided as a convenience
     *  shortcut for 'LocaleData.get().hours24'.
     *
     * @property {String} STD_TOKEN
     *  The translated number format token for the 'General' format code.
     *  Provided as a convenience shortcut for 'LocaleData.get().stdToken'.
     *
     * @property {Object<String,String>} DATE_TOKENS
     *  The translated number format tokens for date/time format codes. Each
     *  property value of this map is a single character. See description of
     *  the property LocaleDataEntry.dateTokens for detail. Provided as a
     *  convenience shortcut for 'LocaleData.get().dateTokens'.
     *
     * @property {Array<String>} SHORT_WEEKDAYS
     *  The translated short names of all days in a week, starting with Sunday.
     *  Provided as a convenience shortcut for 'LocaleData.get().shortWeekdays'.
     *
     * @property {Array<String>} LONG_WEEKDAYS
     *  The translated full names of all days in a week, starting with Sunday.
     *  Provided as a convenience shortcut for 'LocaleData.get().longWeekdays'.
     *
     * @property {Array<String>} SHORT_MONTHS
     *  The translated short names of all months in a year. Provided as a
     *  convenience shortcut for 'LocaleData.get().shortMonths'.
     *
     * @property {Array<String>} LONG_MONTHS
     *  The translated full names of all months in a year. Provided as a
     *  convenience shortcut for 'LocaleData.get().longMonths'.
     *
     * @property {Array<String>} SHORT_QUARTERS
     *  The translated short names of the four quarters of the year. Provided
     *  as a convenience shortcut for 'LocaleData.get().shortQuarters'.
     *
     * @property {Array<String>} LONG_QUARTERS
     *  The translated full names of the four quarters of the year. Provided as
     *  a convenience shortcut for 'LocaleData.get().longQuarters'.
     */
    var LocaleData = {};

    // constants --------------------------------------------------------------

    /**
     * The values of all available roman digits, mapped by the respective roman
     * digit letter.
     *
     * @constant
     * @type Object<String,Number>
     */
    LocaleData.ROMAN_DIGITS = {
        M: 1000,
        D: 500,
        C: 100,
        L: 50,
        X: 10,
        V: 5,
        I: 1
    };

    // static methods ---------------------------------------------------------

    /**
     * Returns the map with the locale data entries of all supported locales.
     *
     * @returns {Object}
     *  The map with the locale data entries of all supported locales, mapped
     *  by locale identifier.
     */
    LocaleData.getAll = function () {
        return localeDataByCode;
    };

    /**
     * Returns the settings for the current UI locale, or for any locale
     * specified in the parameter.
     *
     * @param {String|Number} [locale]
     *  The locale code with underscore character (e.g. 'en_GB'), or a numeric
     *  MS locale identifier (as contained in the property 'lcid' of the locale
     *  data entries). If omitted, the locale data for the current UI locale
     *  will be returned.
     *
     * @returns {LocaleDataEntry}
     *  The locale data entry for the specified locale. If no locale data is
     *  available for the passed locale code or MS locale identifier, the
     *  locale data for 'en_US' will be returned instead.
     */
    LocaleData.get = function (locale) {
        var data = getLocaleData(locale);
        return data ? data : localeDataByCode.en_US;
    };

    /**
     * Returns the translated language name for the passed locale.
     *
     * @param {String|Number} [locale]
     *  The locale code with underscore character (e.g. 'en_GB'), or a numeric
     *  MS locale identifier (as contained in the property 'lcid' of the locale
     *  data entries). If omitted, the translated language name for the current
     *  UI locale will be returned.
     *
     * @returns {String|Null}
     *  The language name for the specified locale; or null if no translated
     *  name exists for the passed locale.
     */
    LocaleData.getLanguageName = function (locale) {
        var data = getLocaleData(locale);
        return data ? data.name : null;
    };

    /**
     * Loads the specified JSON resource module for the current UI locale. If
     * there is no resource module available for the current language, tries to
     * fall back to the language code 'en'. All JSON resources loaded once will
     * be cached internally, and will be returned immediately when calling this
     * method again.
     *
     * @param {String} modulePath
     *  The module path to the directory containing the JSON resource modules.
     *  The resource modules must be stored as JSON source file containing the
     *  plain language code as file name: '[modulePath]/[LANGUAGE].json'. The
     *  file must contain a JSON object with locale names as property keys, and
     *  the resource data as property values, e.g.:
     *      {
     *          "en": {...},
     *          "en_US": {...},
     *          "en_GB": {...}
     *      }
     *  The exact locale identifier (e.g. "en_GB") will be preferred over data
     *  mapped by a generic language name (e.g. "en"). The latter will be used
     *  as fall-back in case no data exists for the exact locale identifier
     *  (e.g., the data mapped by "en" will be loaded for the locale "en_CA" in
     *  the example above).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean} [options.merge=false]
     *      If set to true, the data mapped by the generic language name (e.g.
     *      "en"), and the data mapped by the exact locale identifier (e.g.
     *      "en_GB") will be merged.
     *  - {String} [options.locale]
     *      If specified, the resource data for the specified locale will be
     *      loaded instead for the current UI locale. MUST be a complete locale
     *      identifier with language and region, separated by an underscore
     *      character (e.g. "en_GB").
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved with the JSON object loaded from the
     *  specified resource file, or that will be rejected on any error (e.g.,
     *  resource file not found, or invalid JSON data).
     */
    LocaleData.loadResource = function (modulePath, options) {

        // tries to load the resource file for the passed locale
        function requireResource(language, country) {

            // load the JSON source file for the passed language
            return IO.loadJSON(modulePath + '/' + language).then(function (jsonData) {

                // if the file consists of a simple string, use it as fall-back locale
                var matches = null;
                if (_.isString(jsonData) && (matches = LOCALE_ID_RE.exec(jsonData))) {
                    return requireResource(matches[1], matches[2]);
                }

                // IO.loadJSON() accepts any JSON data type
                if (!_.isObject(jsonData)) {
                    Utils.error('LocaleData.loadResource(): invalid JSON data');
                    return $.Deferred().reject(jsonData);
                }

                // validate resource data (must be a map with language and/or local identifiers
                var keyPattern = new RegExp(_.escapeRegExp(language) + '(_[A-Z]{2,})?');
                _.each(jsonData, function (map, key) {
                    if (!keyPattern.test(key)) {
                        Utils.error('LocaleData.loadResource(): invalid map key "' + key + '"');
                    }
                });

                // no country code available: return data for language code only
                if (!country) { return jsonData[language]; }

                // merge country specific resources (e.g. 'en_GB') over generic language resources (e.g. 'en')
                return Utils.getBooleanOption(options, 'merge', false) ?
                    Utils.extendOptions(jsonData[language], jsonData[language + '_' + country]) :
                    (jsonData[language + '_' + country] || jsonData[language]);
            },
            function (response) {
                // on error, fall back to en_US (unless English is already the requested language)
                return (language === 'en') ? response : requireResource('en', 'US');
            });
        }

        // get the name of the locale used to load the resource data
        var locale = Utils.getStringOption(options, 'locale', LocaleData.LOCALE);
        var matches = LOCALE_ID_RE.exec(locale);
        if (!matches || !matches[2]) {
            Utils.error('LocaleData.loadResource(): invalid locale identifier "' + locale + '"');
            return $.Deferred().reject();
        }

        // the (pending, resolved, or rejected) promise is stored in the cache
        var key = locale + ':' + modulePath;
        return resourceCache.getOrCreate(key, function () {
            return requireResource(matches[1], matches[2]);
        });
    };

    // UI locale data =========================================================

    var modulePromise = IO.loadJSON('io.ox/office/tk/locale/localedata').then(function (jsonDataMap) {

        // store the map by locale code, add locale/language/country codes to the entries
        localeDataByCode = _.mapObject(jsonDataMap, function (jsonData, lc) {
            return new LocaleDataEntry(lc, jsonData);
        });

        // create the map by LCID
        localeDataByLcid = {};
        _.each(localeDataByCode, function (entry) {
            localeDataByLcid[entry.lcid] = entry;
        });

        // store data of current locale
        if (!(Config.LOCALE in localeDataByCode)) {
            Utils.error('LocaleData: unknown UI language: ' + Config.LOCALE);
        }
        localeData = LocaleData.get(Config.LOCALE);

        // convenience shortcuts for current locale
        LocaleData.LOCALE = localeData.lc;
        LocaleData.LANGUAGE = localeData.language;
        LocaleData.COUNTRY = localeData.country;
        LocaleData.LCID = localeData.lcid;
        LocaleData.NAME = localeData.name;
        LocaleData.DEC = localeData.dec;
        LocaleData.GROUP = localeData.group;
        LocaleData.DIR = localeData.dir;
        LocaleData.CJK = localeData.cjk;
        LocaleData.UNIT = localeData.unit;
        LocaleData.CURRENCY = localeData.currency;
        LocaleData.ISO_CURRENCY = localeData.isoCurrency;
        LocaleData.SHORT_DATE = localeData.shortDate;
        LocaleData.LONG_DATE = localeData.longDate;
        LocaleData.SHORT_TIME = localeData.shortTime;
        LocaleData.LONG_TIME = localeData.longTime;
        LocaleData.LEADING_MONTH = localeData.leadingMonth;
        LocaleData.LEADING_YEAR = localeData.leadingYear;
        LocaleData.HOURS24 = localeData.hours24;
        LocaleData.STD_TOKEN = localeData.stdToken;
        LocaleData.DATE_TOKENS = localeData.dateTokens;
        LocaleData.SHORT_WEEKDAYS = localeData.shortWeekdays;
        LocaleData.LONG_WEEKDAYS = localeData.longWeekdays;
        LocaleData.SHORT_MONTHS = localeData.shortMonths;
        LocaleData.LONG_MONTHS = localeData.longMonths;
        LocaleData.SHORT_QUARTERS = localeData.shortQuarters;
        LocaleData.LONG_QUARTERS = localeData.longQuarters;
        return LocaleData;
    });

    // exports ================================================================

    return modulePromise;

});
