/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/dialog/folderpickerdialog', [
    'io.ox/core/folder/api',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/tk/dialog/basedialog',
    'gettext!io.ox/office/tk/main'
], function (FolderAPI, DriveUtils, BaseDialog, gt) {

    'use strict';

    // class FolderPickerDialog ===============================================

    /**
     * A dialog that allows to pick a Drive folder.
     *
     * @constructor
     *
     * @extends BaseDialog
     *
     * @param {String|Object} windowId
     *  The identifier of the root window of the context application that is
     *  creating the dialog, or an object with a method getWindowId() that
     *  returns such a window identifier. Used for debugging and logging of
     *  running timers in automated test environments.
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. Supports all options also supported by the base class
     *  PickerDialog. Additionally, the following options are supported:
     *  - {String} [initOptions.folderId]
     *      The identifier of a folder to be selected in the folder picker.
     *  - {Boolean} [initOptions.flatPicker=false]
     *      Whether to create a FlatFolderPicker instance instead of a regular
     *      FolderPicker instance.
     *  - {Boolean} [initOptions.createFolderButton=false]
     *      If set to true, the dialog will provide an additional push button
     *      that allows to create new subfolders.
     */
    var FolderPickerDialog = BaseDialog.extend(function (windowId, initOptions) {

        // base constructor
        BaseDialog.call(this, windowId, initOptions);

        // the folder picker list control embedded in this dialog
        var folderPicker = this._folderPicker = (initOptions && initOptions.flatPicker) ?
            new DriveUtils.FlatFolderPicker(windowId, initOptions) :
            new DriveUtils.FolderPicker(windowId, initOptions);

        // add the folder picker to the body of this dialog
        this.append(folderPicker.getNode());

        // add a "create folder" button
        if (initOptions && initOptions.createFolderButton) {

            var container = this.getBody().closest('.io-ox-dialog-wrapper');

            this.createButton('create', gt('Create folder'), {
                alignLeft: true,
                actionHandler: function () {
                    container.busy();

                    var folderId = folderPicker.getFolderId();
                    var folderModel = FolderAPI.pool.getModel(folderId);

                    // request and open create-folder-dialog
                    var promise = require(['io.ox/core/folder/actions/add'], function (addAction) {
                        return addAction(folderId, { module: folderModel.get('module') }).always(function () {
                            container.show();
                        });
                    });

                    // add additonal 5ms to tree nodes debounced onSort handler
                    promise = promise.then(function (newFolder) {
                        return new $.Deferred(function (def) {
                            _.delay(function () {
                                folderPicker.selectFolder(newFolder.id);
                                def.resolve();
                            }, 15);
                        });
                    });

                    return promise.always(function () { container.hide(); });
                },
                keepOpen: true
            });
        }

        // validate after changing the folder selection
        this.addOkValidator(function () { return folderPicker.getFolderId(); });
        this.validateOn(folderPicker, 'change');

        // pass the folder identifier as result for the OK button
        this.setOkHandler(function () { return folderPicker.getFolderId(); });

        this.registerDestructor(function () {
            folderPicker.destroy();
            folderPicker = null;
        });

    }); // class FolderPickerDialog

    // public methods ---------------------------------------------------------

    /**
     * Returns the folder picker control contained in this dialog.
     *
     * @returns {FolderPicker|FlatFolderPicker}
     *  The folder picker control contained in this dialog.
     */
    FolderPickerDialog.prototype.getFolderPicker = function () {
        return this._folderPicker;
    };

    /**
     * Returns the identifier of the selected folder.
     *
     * @returns {String}
     *  The identifier of the selected folder.
     */
    FolderPickerDialog.prototype.getFolderId = function () {
        return this._folderPicker.getFolderId();
    };

    // exports ================================================================

    return FolderPickerDialog;

});
