/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/dialog/basedialog', [
    'io.ox/core/tk/dialogs',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/utils/class',
    'io.ox/office/tk/utils/scheduler',
    'gettext!io.ox/office/tk/main'
], function (CoreDialogs, Utils, Forms, KeyCodes, Class, Scheduler, gt) {

    'use strict';

    // constants ==============================================================

    var OK_ACTION = 'ok';
    var CANCEL_ACTION = 'cancel';
    var CLOSE_ACTION = 'close';

    var OK_LABEL = /*#. button label in a OK/Cancel dialog box */ gt('OK');
    var CANCEL_LABEL = /*#. button label in a OK/Cancel dialog box */ gt('Cancel');

    // private global functions ===============================================

    /**
     * Returns a dialog width suitable for the current device type, and for the
     * specified desired width.
     *
     * @param {Object} [initOptions]
     *  Optional parameters passed to the constructor of a dialog. The value of
     *  the option "width" will be used as preferred dialog width.
     *
     * @returns {Number}
     *  The effective width of the dialog, in pixels. Will not be greater than
     *  95% of the current screen width. On small devices, the dialog width
     *  returned by this method will be fixed to 95% of the screen width,
     *  regardless of the passed width.
     */
    function getBestDialogWidth(initOptions) {
        var width = Utils.getIntegerOption(initOptions, 'width', 500);
        var maxWidth = Math.round(window.innerWidth * 0.95);
        return Utils.SMALL_DEVICE ? maxWidth : Math.min(width, maxWidth);
    }

    // class BaseDialog =======================================================

    /**
     * Creates an empty modal dialog. Conceptually, a dialog instance is a
     * one-way object. After creating an instance, it can be shown once, and
     * will destroy itself afterwards.
     *
     * @constructor
     *
     * @extends CoreDialogs.ModalDialog
     *
     * @param {String|Object} windowId
     *  The identifier of the root window of the context application that is
     *  creating the dialog, or an object with a method getWindowId() that
     *  returns such a window identifier. Used for debugging and logging of
     *  running timers in automated test environments.
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. The following options are supported:
     *  - {String} [initOptions.title]
     *      The title of the dialog window that will be shown in a larger font.
     *  - {Number} [initOptions.width=500]
     *      The width of the dialog, in pixels.
     *  - {String} [initOptions.okLabel=gt('OK')]
     *      The label of the primary button that triggers the default action of
     *      the dialog. If omitted, the button will be created with the
     *      translated default label "OK".
     *  - {String} [initOptions.cancelLabel=gt('Cancel')]
     *      The label of the cancel button that closes the dialog without
     *      performing any action. If omitted, the button will be created with
     *      the translated default label "Cancel".
     *  - {String} [initOptions.classes]
     *      Additional CSS classes that will be set at the root DOM node of
     *      this view component.
     */
    var BaseDialog = Class.extend(CoreDialogs.ModalDialog, function (windowId, initOptions) {

        // base constructor
        CoreDialogs.ModalDialog.call(this, {
            width: getBestDialogWidth(initOptions),
            enter: this._inputEnterHandler.bind(this),
            async: true,
            focus: false
        });

        // deferred object representing the life cycle of this dialog
        var deferred = Scheduler.createDeferred(windowId, 'BaseDialog.show');
        // all destructor callbacks
        var destructors = [];
        // bound version of the global event handler
        var globalKeyDownHandler = this._globalKeyDownHandler.bind(this);
        // bound version of the button action handler
        var buttonActionHandler = this._buttonActionHandler.bind(this);

        // add own CSS marker class
        var rootNode = this.getPopup();
        var additionalClass = Utils.getStringOption(initOptions, 'classes', '');
        rootNode.addClass('io-ox-office-main io-ox-office-dialog').addClass(additionalClass);
        Forms.addDeviceMarkers(rootNode);

        // add a header element for the title
        var dialogTitle = Utils.getStringOption(initOptions, 'title', null);
        var headerNode = dialogTitle ? $('<h4 id="dialog-title">').text(dialogTitle) : $();
        this.header(headerNode);

        // create the default action buttons
        this.createButton(CANCEL_ACTION, Utils.getStringOption(initOptions, 'cancelLabel', CANCEL_LABEL));
        this.createButton(OK_ACTION, Utils.getStringOption(initOptions, 'okLabel', OK_LABEL), { buttonStyle: 'primary' });

        // event handler for all events triggered by this dialog, filters by button actions,
        // resolves or rejects the passed deferred object according to the result of the action
        this.on('triggered', buttonActionHandler);

        // listen to key events for additional keyboard handling
        this.getBody().on('keydown', function (event) {
            if (event.keyCode === KeyCodes.ENTER) {
                this._inputEnterHandler();
                return false;
            }
        }.bind(this));

        // override the existing show() method to add specific behavior
        // (the base method is not contained in the class prototype!)
        this.show = _.wrap(this.show, function (showMethod) {

            // Bug 41397: When focus leaves a node with browser selection (e.g. a clipboard
            // node, or a content-editable node), the browser selection MUST be destroyed.
            // Otherwise, all keyboard events bubbling to the document body will cause to
            // focus that previous element having the browser selection regardless whether
            // it is actually focused. This may cause for example to be able to edit a
            // document while a modal dialog is open.
            Utils.clearBrowserSelection();

            // register global key handler in case the browser focus leaves the dialog frame
            document.addEventListener('keydown', globalKeyDownHandler);

            // show the dialog, but ignore the promise it returns
            showMethod.call(this);

            // return the own promise that will be resolved/rejected after finishing the dialog
            return deferred.promise();
        });

        // set focus to an element when dialog is shown
        this.on('show', function () { this.validate().grabFocus(); }.bind(this));

        // clean-up after closing the dialog
        deferred.always(function () {

            // do not process any more actions (especially not a following "close")
            this.off('triggered', buttonActionHandler);
            // unregister all global (external) event handlers
            document.removeEventListener('keydown', globalKeyDownHandler);

            // close the dialog frame
            this.close();

            // invoke all destructor callbacks, release all local variables
            destructors.forEach(function (destructor) { destructor.call(this); }, this);
            deferred = destructors = null;
            globalKeyDownHandler = buttonActionHandler = null;
        }.bind(this));

        // instance properties used in class methods
        this.__handlers = {};
        this.__validators = {};
        this.__def = deferred;
        this.__dtors = destructors;
        this.__header = headerNode;
        this.__focus = this.getOkButton();
        this.__busy = false;

    }); // class BaseDialog

    // constants --------------------------------------------------------------

    /**
     * The action identifier of the primary button (the OK button).
     *
     * @constant
     * @type String
     */
    BaseDialog.OK_ACTION = OK_ACTION;

    /**
     * The action identifier of the Cancel button.
     *
     * @constant
     * @type String
     */
    BaseDialog.CANCEL_ACTION = CANCEL_ACTION;

    // private methods --------------------------------------------------------

    /**
     * Event handler that triggers the primary button of the dialog, after
     * the ENTER key has been pressed in a text input control.
     */
    BaseDialog.prototype._inputEnterHandler = function () {
        var buttonEnabled = this.isOkButtonEnabled();
        if (buttonEnabled) { this.invoke(OK_ACTION); }
        return !buttonEnabled;
    };

    /**
     * Event handler for global key events triggered from outside the dialog
     * frame. The TAB key will return the browser focus to the dialog, and the
     * ESCAPE key will close the dialog even without having the focus inside.
     */
    BaseDialog.prototype._globalKeyDownHandler = function (event) {

        // handle key events from document body only
        if (Utils.getActiveElement() !== document.body) { return; }

        // handle TAB and ESCAPE keys
        if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
            Forms.grabFocus(this.getPopup());
            event.preventDefault();
        } else if (KeyCodes.matchKeyCode(event, 'ESCAPE')) {
            this.close();
        }
    };

    /**
     * Event handler for all events triggered by this dialog instance. The
     * events will be filtered by button action identifiers, and will resolve
     * or reject the deferred object returned by the show() method according to
     * the result of the action.
     */
    BaseDialog.prototype._buttonActionHandler = function (event, action) {

        // the deferred object to be resolved/rejected
        var deferred = this.__def;

        // bug 32244: handle explicit "dialog.close()" calls
        if (action === CLOSE_ACTION) {
            this.close = _.noop; // prevent recursive calls from promise callbacks
            deferred.reject(action);
            return;
        }

        // do not invoke action handler recursively, check that a button exists for the action identifier
        if (this.__busy || (this.getButton(action).length === 0)) { return; }

        // switch dialog to busy mode
        this.__busy = true;
        this.__header.busy();

        // whether the Cancel button has been activated
        var isCancel = action === CANCEL_ACTION;
        // the callback handler and configuration of the action
        var entry = this.__handlers[action];
        // keep open mode (not for Cancel action)
        var keepOpen = (!isCancel && entry && entry.options && entry.options.keepOpen) || false;

        var returnToIdle = function () {
            this.__busy = false;
            this.__header.idle();
            this.validate().idle().grabFocus();
        }.bind(this);

        function handleSuccess(value) {
            if ((keepOpen === true) || (keepOpen === 'done')) {
                returnToIdle();
            } else if (isCancel) {
                deferred.reject(action);
            } else {
                deferred.resolve(value);
            }
        }

        function handleFailure(err) {
            if ((keepOpen === true) || (keepOpen === 'fail')) {
                returnToIdle();
            } else {
                deferred.reject(isCancel ? action : err);
            }
        }

        // invoke the action handler callback
        try {
            var result = entry ? entry.handler.call(this, action) : null;
            if (Utils.isPromise(result)) {
                result.done(handleSuccess).fail(handleFailure);
            } else {
                handleSuccess(result);
            }
        } catch (err) {
            handleFailure(err);
        }
    };

    // public methods ---------------------------------------------------------

    /**
     * Registers a destructor callback function that will be invoked when the
     * dialog has been closed and will be destroyed.
     *
     * @param {Function} destructor
     *  A destructor callback function. The dialog will invoke all registered
     *  destructor callbacks in reverse order of their registration. The
     *  callback functions will be invoked in the context of this instance.
     *
     * @returns {BaseDialog}
     *  A reference to this instance.
     */
    BaseDialog.prototype.registerDestructor = function (destructor) {
        this.__dtors.unshift(destructor);
        return this;
    };

    /**
     * Creates an action button and inserts it into the footer area of this
     * dialog.
     *
     * @param {String} action
     *  The action identifier for the button. Can be any non-empty string but
     *  the identifier of the primary button (BaseDialog.OK_ACTION), or of the
     *  Cancel button (BaseDialog.CANCEL_ACTION).
     *
     * @param {String} label
     *  The caption label to be inserted into the button.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean} [options.alignLeft=false]
     *      If set to true, the button will be inserted at the left border of
     *      the footer. By default, the button will be inserted at the right
     *      border behind all existing buttons (except when using the option
     *      "insertBefore").
     *  - {String} [options.insertBefore]
     *      The action identifier of an existing button that should be located
     *      right of the new button.
     *  - {String} [options.buttonStyle]
     *      The appearance of the button. Must be one of the following values:
     *      - "primary": The button will appear as the primary (OK) button.
     *      - "success": The button will appear as a success button (green).
     *      - "warning": The button will appear as a warning button (yellow).
     *      - "danger": The button will appear as a danger button (red).
     *      If omitted, the button will appear with its default style.
     *  - {Function} [options.actionHandler]
     *      An action handler callback function. This option is a convenience
     *      shortcut and calls method BaseDialog.setActionHandler() for the
     *      button action internally.
     *  - {Boolean|String} [options.keepOpen=false]
     *      Specifies when to keep open the dialog after activating the action
     *      button. Will be used together with the action handler passed in the
     *      option "actionHandler". See method BaseDialog.setActionHandler()
     *      for details.
     *
     * @returns {jQuery}
     *  The <button> element, as jQuery collection.
     */
    BaseDialog.prototype.createButton = function (action, label, options) {

        // add the button using base class functionality (the method does not return the button!)
        var alignLeft = Utils.getBooleanOption(options, 'alignLeft', false);
        if (alignLeft) {
            this.addAlternativeButton(action, label, action);
        } else {
            this.addButton(action, label, action);
        }

        // fetch the new button from the dialog
        var buttonNode = this.getButton(action);

        // place the button in front of another button, or append new button to the right border
        var insertBefore = Utils.getStringOption(options, 'insertBefore', null);
        var beforeButton = insertBefore ? this.getButton(insertBefore) : $();
        if (beforeButton.length > 0) {
            buttonNode.insertBefore(beforeButton);
        } else if (!alignLeft) {
            this.getFooter().append(buttonNode);
        }

        // set the Bootstrap style class
        var buttonStyle = Utils.getStringOption(options, 'buttonStyle', 'default');
        buttonNode.addClass('btn-' + buttonStyle);

        // register action handler
        var actionHandler = Utils.getFunctionOption(options, 'actionHandler', null);
        if (actionHandler) { this.setActionHandler(action, actionHandler, options); }

        return buttonNode;
    };

    /**
     * Returns the button element from the footer of this dialog bound to the
     * specified action.
     *
     * @param {String} action
     *  The identifier of the action the button is bound to.
     *
     * @returns {jQuery}
     *  The specified button element, as jQuery object.
     */
    BaseDialog.prototype.getButton = function (action) {
        return this.getFooter().find('.btn[data-action="' + action + '"]');
    };

    /**
     * Returns the OK button of this dialog, as jQuery object.
     *
     * @returns {jQuery}
     *  The button element of the OK button.
     */
    BaseDialog.prototype.getOkButton = function () {
        return this.getButton(OK_ACTION);
    };

    /**
     * Returns the Cancel button of this dialog, as jQuery object.
     *
     * @returns {jQuery}
     *  The button element of the Cancel button.
     */
    BaseDialog.prototype.getCancelButton = function () {
        return this.getButton(CANCEL_ACTION);
    };

    /**
     * Returns whether the specified action button is enabled.
     *
     * @param {String} action
     *  The identifier of the action the button is bound to.
     *
     * @returns {Boolean}
     *  Whether the specified action button is enabled.
     */
    BaseDialog.prototype.isButtonEnabled = function (action) {
        return Forms.isBSButtonEnabled(this.getButton(action));
    };

    /**
     * Returns whether the primary button (the OK button) is enabled.
     *
     * @returns {Boolean}
     *  Whether the primary button is enabled.
     */
    BaseDialog.prototype.isOkButtonEnabled = function () {
        return this.isButtonEnabled(OK_ACTION);
    };

    /**
     * Enables or disables the specified action button.
     *
     * @param {String} action
     *  The identifier of the action the button is bound to.
     *
     * @param {Boolean} state
     *  Whether to enable (true) or disable (false) the button.
     *
     * @returns {BaseDialog}
     *  A reference to this instance.
     */
    BaseDialog.prototype.enableButton = function (action, state) {
        Forms.enableBSButton(this.getButton(action), state);
        return this;
    };

    /**
     * Enables or disables the primary button (the OK button).
     *
     * @param {Boolean} state
     *  Whether to enable (true) or disable (false) the button.
     *
     * @returns {BaseDialog}
     *  A reference to this instance.
     */
    BaseDialog.prototype.enableOkButton = function (state) {
        return this.enableButton(OK_ACTION, state);
    };

    /**
     * Registers a callback function that will be invoked when pressing the
     * button with the specified action identifier.
     *
     * @param {String} action
     *  The identifier of the action the callback function is bound to.
     *
     * @param {Function} handler
     *  The callback function that will invoked when the action button has been
     *  activated. Will be called in the context of this dialog instance,
     *  receives the action identifier as first parameter (to be able to reuse
     *  the same handler function for similar actions), and must return the
     *  final result of the dialog.
     *  - Synchronous mode: The callback function may return a result value
     *     synchronously (anything but a promise), or may throw to signal any
     *     kind of failure.
     *  - Asynchronous mode: The callback function may return a promise that
     *     will fulfil or reject later.
     *  According to the result of the callback function, the promise returned
     *  by the method show() will fulfil or reject with the result value.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  - {Boolean|String} [options.keepOpen=false]
     *      Specifies when to keep open the dialog after activating the action
     *      button.
     *      - By default (value false), the dialog will be closed when clicking
     *          the action button (after the action handler has finished).
     *      - If set to the boolean value true, the dialog will be kept open
     *          regardless of the result of the action handler (success or
     *          failure).
     *      - If set to the string "done", the dialog will be kept open if the
     *          action succeeds (synchronous return value, or resolved
     *          promise).
     *      - If set to the string "fail", the dialog will be kept open if the
     *          action fails (exception thrown, or rejected promise).
     *      If the dialog remains in opened state, the promise returned by the
     *      method show() remains in pending state too.
     *
     * @returns {BaseDialog}
     *  A reference to this instance.
     */
    BaseDialog.prototype.setActionHandler = function (action, handler, options) {
        this.__handlers[action] = { handler: handler, options: options };
        return this;
    };

    /**
     * Registers a callback function that will be invoked when pressing the
     * primary button (the OK button) of the dialog.
     *
     * @param {Function} handler
     *  The callback function that will invoked when the primary button has
     *  been activated. See method BaseDialog.setActionHandler() for details.
     *
     * @param {Object} [options]
     *  Optional parameters. See method BaseDialog.setActionHandler() for
     *  details.
     *
     * @returns {BaseDialog}
     *  A reference to this instance.
     */
    BaseDialog.prototype.setOkHandler = function (handler, options) {
        return this.setActionHandler(OK_ACTION, handler, options);
    };

    /**
     * Registers a callback function that will be invoked when pressing the
     * Cancel button of the dialog. The dialog cannot be kept open when
     * pressing the Cancel button.
     *
     * @param {Function} handler
     *  The callback function that will be invoked when the primary button has
     *  been activated. See method BaseDialog.setActionHandler() for details.
     *
     * @returns {BaseDialog}
     *  A reference to this instance.
     */
    BaseDialog.prototype.setCancelHandler = function (handler) {
        return this.setActionHandler(CANCEL_ACTION, handler);
    };

    /**
     * Adds a validator predicate for the specified action button. If at least
     * one of the registered validators fails, the respective action button
     * will be disabled.
     *
     * @param {String} action
     *  The identifier of the action button the validator is bound to.
     *
     * @param {Function} validator
     *  The validator predicate. Will be called in the context of this dialog
     *  instance. Receives the action identifier as first parameter. If the
     *  function returns a falsy value or throws, the action button will be
     *  disabled.
     *
     * @returns {BaseDialog}
     *  A reference to this instance.
     */
    BaseDialog.prototype.addValidator = function (action, validator) {
        var validators = this.__validators[action] || (this.__validators[action] = []);
        validators.push(validator);
        return this;
    };

    /**
     * Adds a validator predicate for the primary button (the OK button). If at
     * least one of the registered validators fails, the primary button will be
     * disabled.
     *
     * @param {Function} validator
     *  The validator predicate. See method BaseDialog.addValidator() for
     *  details.
     *
     * @returns {BaseDialog}
     *  A reference to this instance.
     */
    BaseDialog.prototype.addOkValidator = function (validator) {
        return this.addValidator(OK_ACTION, validator);
    };

    /**
     * Runs all registered validators, and updates the state of all action
     * buttons.
     *
     * @returns {BaseDialog}
     *  A reference to this instance.
     */
    BaseDialog.prototype.validate = function () {
        _.forEach(this.__validators, function (validators, action) {
            try {
                var state = validators.every(function (validator) {
                    return validator.call(this, action);
                }, this);
                this.enableButton(action, state);
            } catch (err) {
                this.enableButton(action, false);
            }
        }, this);
        return this;
    };

    /**
     * Registers an event handler at the specified event source that validates
     * the action buttons.
     *
     * @param {TriggerObject|HTMLElement|jQuery} eventSource
     *  The event source to listen to.
     *
     * @param {String} eventType
     *  The name of the event to listen to. Can be a space-separated list of
     *  event names.
     *
     * @returns {BaseDialog}
     *  A reference to this instance.
     */
    BaseDialog.prototype.validateOn = function (eventSource, eventType) {
        if (eventSource instanceof HTMLElement) { eventSource = $(eventSource); }
        eventSource.on(eventType, this.validate.bind(this));
        return this;
    };

    /**
     * Registers a DOM node that will get the browser focus when the dialog
     * will be shown. In some circumstances, another node may be focused
     * instead, e.g. on mobile devices, the primary button (the OK button) will
     * be focued to prevent showing the virtual keyboard too early.
     *
     * @param {HTMLElement|jQuery|Function} focusNode
     *  The node to be focused when the dialog opens; or a callback function
     *  that will be invoked after the dialog became visible, and must return a
     *  node to be focused (as HTMLElement or as jQuery collection).
     *
     * @returns {BaseDialog}
     *  A reference to this instance.
     */
    BaseDialog.prototype.setFocusNode = function (focusNode) {
        this.__focus = focusNode;
        return this;
    };

    /**
     * Sets the browser focus to the default focus element of the dialog.
     *
     * @returns {BaseDialog}
     *  A reference to this instance.
     */
    BaseDialog.prototype.grabFocus = function () {

        // resolve callback function
        var focusNode = (this.__focus instanceof Function) ? this.__focus() : this.__focus;

        // convert DOM node to a jQuery object
        var $focusNode = $(focusNode);
        // the OK button (default focus target)
        var okButton = this.getOkButton();

        // US 102078426: Dialogs on small devices should come up without keyboard,
        // so the focus should not be in an input element.
        // Bug 46886: Also for iPad because since iOS 9, body is not scrolled when
        // code triggers focus on input element.
        if (Utils.SMALL_DEVICE || Utils.IOS) {
            $focusNode = okButton;
        }

        // if the OK button is diabled, set focus to Cancel button
        if ($focusNode.is(okButton) && !this.isOkButtonEnabled()) {
            $focusNode = this.getCancelButton();
        }

        // set focus to the node, select text contents of text input controls
        Utils.setFocus($focusNode);
        if ($focusNode.is('input')) { $focusNode.select(); }

        return this;
    };

    // exports ================================================================

    return BaseDialog;

});
