/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/selectionrenderer', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/view/render/renderutils',
    'io.ox/office/spreadsheet/view/render/rendererbase'
], function (Utils, Config, SheetUtils, PaneUtils, RenderUtils, RendererBase) {

    'use strict';

    // convenience shortcuts
    var Range = SheetUtils.Range;

    // bug 41205: IE/Edge: navigating via tab key in last row results in rendering errors
    var renderHelperNode = $('<div id="io-ox-office-spreadsheet-ie-render-helper" style="position:absolute;">');

    // private global functions ===============================================

    var setIERenderHelperNode = _.browser.IE ? function () {
        renderHelperNode.remove();
        window.setTimeout(function () {
            $('body').append(renderHelperNode);
        }, 20);
    } : _.noop;

    // class SelectionRenderer ================================================

    /**
     * Renders the own selection, and the selections of remote users into the
     * DOM layers of a single grid pane.
     *
     * @constructor
     *
     * @extends RendererBase
     *
     * @param {GridPane} gridPane
     *  The grid pane instance that owns this selection layer renderer.
     */
    var SelectionRenderer = RendererBase.extend({ constructor: function (gridPane) {

        // self reference
        var self = this;

        // the spreadsheet model and view
        var docView = gridPane.getDocView();
        var docModel = docView.getDocModel();

        // the remote layer node for displaying selections of other users
        var remoteLayerNode = gridPane.createLayerNode('remote-layer');

        // the selection layer (container for the selected ranges)
        var selectionLayerNode = gridPane.createLayerNode('selection-layer');

        // the cell range and absolute position covered by the layer nodes in the sheet area
        var layerRectangle = null;

        // base constructor ---------------------------------------------------

        RendererBase.call(this, gridPane);

        // private methods ----------------------------------------------------

        /**
         * Updates the position of the auto-fill handle and the resize handles
         * for cell selection, when entire columns or rows are selected.
         */
        function updateResizeHandlePositions() {

            // the auto-fill resizer handle node
            var autoFillHandleNode = selectionLayerNode.find('.autofill.resizers>[data-pos]');
            // the range node containing the auto-fill handle
            var autoFillRangeNode = autoFillHandleNode.closest('.range');
            // visible area of the grid pane
            var visibleRect = gridPane.getVisibleRectangle();

            // converts the passed offset in the visible area to an offset relative to 'parentNode'
            function setRelativeOffset(targetNode, offsetName, parentNode, visibleOffset) {
                var offset = visibleRect[offsetName] - layerRectangle[offsetName] - parentNode.position()[offsetName] + visibleOffset;
                targetNode.css(offsetName, offset + 'px');
            }

            // adjust position of the auto-fill handle node for entire column or row selection
            switch (autoFillHandleNode.attr('data-pos')) {
                case 'r':
                    setRelativeOffset(autoFillHandleNode, 'top', autoFillRangeNode, 0);
                    break;
                case 'b':
                    setRelativeOffset(autoFillHandleNode, 'left', autoFillRangeNode, 0);
                    break;
            }

            // adjust position of the selection handle node for entire column or row selection
            selectionLayerNode.find('.select.resizers>[data-pos]').each(function () {

                var handleNode = $(this),
                    rangeNode = handleNode.closest('.range');

                switch (handleNode.attr('data-pos')) {
                    case 'l':
                    case 'r':
                        setRelativeOffset(handleNode, 'top', rangeNode, visibleRect.height / 2);
                        break;
                    case 't':
                    case 'b':
                        setRelativeOffset(handleNode, 'left', rangeNode, visibleRect.width / 2);
                        break;
                }
            });
        }

        /**
         * Renders all visible selection ranges according to the current pane
         * layout data received from a view layout update notification.
         */
        function renderCellSelection() {

            // nothing to do in chart sheets etc.
            var sheetModel = docView.getSheetModel();
            if (!sheetModel.isCellType()) { return; }

            // nothing to do, if all columns/rows are hidden (method may be called debounced)
            if (!gridPane.isVisible()) {
                selectionLayerNode.empty();
                return;
            }

            // the entire cell selection
            var selection = docView.getSelection();
            // the range occupied by the active cell (will not be filled)
            var activeRange = null;
            // the position of the active cell, extended to the merged range
            var activeRectangle = null;
            // additional data for active auto-fill tracking
            var autoFillData = sheetModel.getViewAttribute('autoFillData');
            // whether the active cell touches the borders of selection ranges
            var activeBorders = { left: false, top: false, right: false, bottom: false };
            // hide left/top border of selection ranges, if first column/row is hidden
            var firstColHidden = !sheetModel.getColCollection().isEntryVisible(0);
            var firstRowHidden = !sheetModel.getRowCollection().isEntryVisible(0);
            // the HTML mark-up for all selection ranges but the active range
            var rangesMarkup = '';
            // the HTML mark-up for the active selection range
            var activeMarkup = '';

            // returns the mark-up of a resizer handler according to the type of the passed range
            function createResizerMarkup(range, leadingCorner) {
                var hPos = leadingCorner ? 'l' : 'r', vPos = leadingCorner ? 't' : 'b';
                return '<div data-pos="' + (docModel.isColRange(range) ? hPos : docModel.isRowRange(range) ? vPos : (vPos + hPos)) + '"></div>';
            }

            // add active flag to range object
            selection.ranges[selection.active].active = true;

            // adjust single range for auto-fill tracking
            if ((selection.ranges.length === 1) && autoFillData) {

                // the selected cell range
                var firstRange = selection.ranges.first();
                // auto-fill direction and count
                var fillDir = autoFillData.direction;
                var fillCount = autoFillData.count;
                // whether to expand/shrink columns or rows
                var vertical = SheetUtils.isVerticalDir(fillDir);
                // whether to expand/shrink the leading or trailing border
                var leading = SheetUtils.isLeadingDir(fillDir);

                activeRange = firstRange.clone();
                if (fillCount >= 0) {
                    // adjust range for auto-fill mode
                    if (leading) {
                        firstRange.start.move(-fillCount, !vertical);
                    } else {
                        firstRange.end.move(fillCount, !vertical);
                    }
                } else {
                    // adjust range for deletion mode
                    firstRange.collapsed = true;
                    if (-fillCount < firstRange.size(!vertical)) {
                        // range partly covered
                        if (leading) {
                            activeRange.start.move(-fillCount, !vertical);
                        } else {
                            activeRange.end.move(fillCount, !vertical);
                        }
                    } else {
                        // special marker for deletion of the entire range
                        activeRange = Utils.BREAK;
                    }
                }

                // add tracking style effect
                firstRange.tracking = true;
            }

            // the range covered by the active cell (or any other range in auto-fill tracking mode)
            activeRange = (activeRange === Utils.BREAK) ? null : activeRange ? activeRange :
                docView.getMergeCollection().expandRangeToMergedRanges(new Range(selection.address));

            // convert active range to a rectangle relative to the layer root node
            if (_.isObject(activeRange)) {
                gridPane.iterateRangesForRendering(activeRange, function (range, rectangle) {
                    activeRectangle = rectangle;
                }, { alignToGrid: true });
            }

            // render all ranges that are visible in this grid pane
            gridPane.iterateRangesForRendering(selection.ranges, function (range, rectangle, index) {

                // hide left/top border of the range, if the first column/row is hidden
                var expandL = (firstColHidden && (range.start[0] === 0)) ? 5 : 0;
                var expandT = (firstRowHidden && (range.start[1] === 0)) ? 5 : 0;
                rectangle.expandSelf(expandL, expandT, 0, 0);

                // generate the HTML mark-up for the range
                var rangeMarkup = '<div class="range';
                if (range.active) { rangeMarkup += ' active'; }
                if (range.tracking) { rangeMarkup += ' tracking-active'; }
                if (range.collapsed) { rangeMarkup += ' collapsed'; }
                rangeMarkup += '" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '" data-index="' + index + '">';

                // insert the semi-transparent fill elements (never cover the active range with the fill elements)
                if (activeRectangle && range.overlaps(activeRange)) {

                    // initialize position of active cell, relative to selection range
                    var relActiveRect = activeRectangle.clone();
                    relActiveRect.left -= rectangle.left;
                    relActiveRect.top -= rectangle.top;
                    var rightDist = rectangle.width - relActiveRect.right();
                    var bottomDist = rectangle.height - relActiveRect.bottom();

                    // insert fill element above active cell
                    if (relActiveRect.top > 0) {
                        rangeMarkup += '<div class="fill" style="left:0;right:0;top:0;height:' + relActiveRect.top + 'px;"></div>';
                    }

                    // insert fill element left of active cell
                    if (relActiveRect.left > 0) {
                        rangeMarkup += '<div class="fill" style="left:0;width:' + relActiveRect.left + 'px;top:' + relActiveRect.top + 'px;height:' + relActiveRect.height + 'px;"></div>';
                    }

                    // insert fill element right of active cell
                    if (rightDist > 0) {
                        rangeMarkup += '<div class="fill" style="left:' + relActiveRect.right() + 'px;right:0;top:' + relActiveRect.top + 'px;height:' + relActiveRect.height + 'px;"></div>';
                    }

                    // insert fill element below active cell
                    if (bottomDist > 0) {
                        rangeMarkup += '<div class="fill" style="left:0;right:0;top:' + relActiveRect.bottom() + 'px;bottom:0;"></div>';
                    }

                    // update border flags for the active cell
                    activeBorders.left = activeBorders.left || (range.start[0] === activeRange.start[0]);
                    activeBorders.top = activeBorders.top || (range.start[1] === activeRange.start[1]);
                    activeBorders.right = activeBorders.right || (range.end[0] === activeRange.end[0]);
                    activeBorders.bottom = activeBorders.bottom || (range.end[1] === activeRange.end[1]);

                } else {
                    rangeMarkup += '<div class="abs fill"></div>';
                }

                // generate the HTML mark-up for the selection range
                rangeMarkup += '<div class="border"></div>';

                // additional mark-up for single-range selection
                if (selection.ranges.length === 1) {
                    if (Utils.TOUCHDEVICE && !autoFillData) {
                        // add resize handlers for selection on touch devices
                        rangeMarkup += '<div class="select resizers">' + createResizerMarkup(range, true) + createResizerMarkup(range, false) + '</div>';
                    } else {
                        // add the auto-fill handler in the bottom right corner of the selection
                        rangeMarkup += '<div class="autofill resizers">' + createResizerMarkup(range, false) + '</div>';
                    }
                }

                // close the range node
                rangeMarkup += '</div>';

                // and the generated mark-up to the appropriate mark-up list
                if (range.active) { activeMarkup += rangeMarkup; } else { rangesMarkup += rangeMarkup; }
            }, { alignToGrid: true });

            // additions for the active cell (or active range in auto-fill)
            if (activeRectangle) {

                // add thin border for active cell on top of the selection
                rangesMarkup += '<div class="active-cell';
                _.each(activeBorders, function (isBorder, borderName) { if (!isBorder) { rangesMarkup += ' ' + borderName; } });
                rangesMarkup += '" style="' + PaneUtils.getRectangleStyleMarkup(activeRectangle) + '"></div>';
            }

            // insert entire HTML mark-up into the selection container node
            selectionLayerNode[0].innerHTML = rangesMarkup + activeMarkup;

            // update the visibility/position of additional DOM elements
            updateResizeHandlePositions();

            // bug 41205: IE/Edge: navigating via tab key in last row results in rendering errors
            setIERenderHelperNode();

            // notify listeners for additional rendering depending on the selection
            self.trigger('render:cellselection');
        }

        /**
         * Debounced version of the method renderCellSelection().
         */
        var renderCellSelectionDebounced = this.createDebouncedMethod('SelectionRenderer.renderCellSelectionDebounced', null, renderCellSelection, { delay: 100 });

        /**
         * Renders selections of all remote users which have the same sheet
         * activated.
         */
        var renderRemoteSelection = Config.SHOW_REMOTE_SELECTIONS ? function () {

            // nothing to do, if all columns/rows are hidden (method may be called debounced)
            if (!gridPane.isVisible()) {
                remoteLayerNode.empty();
                return;
            }

            // top-left visible address of the grid pane, to detect if the name-layer outside of the view
            var firstRow = gridPane.getTopLeftAddress()[1];
            // the HTML mark-up for all user selections
            var markup = '';

            // generate selections of all remote users on the same sheet
            docView.iterateRemoteSelections(function (userName, colorIndex, ranges, drawings) {
                if (!ranges || drawings) { return; }

                // the escaped user name, ready to be inserted into HTML mark-up
                userName = Utils.escapeHTML(_.noI18n(userName));

                // render all ranges of the user
                gridPane.iterateRangesForRendering(ranges, function (range, rectangle) {

                    // the CSS classes to be set at the range (bug 35604: inside range if on top of the grid pane)
                    var classes = 'range' + ((range.start[1] === firstRow) ? '' : ' remote-badge-outside');

                    // create the mark-up for the range
                    markup += '<div class="' + classes + '" data-remote-user="' + userName + '" data-remote-color="' + colorIndex + '"';
                    markup += ' style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"><div class="fill"></div></div>';
                }, { alignToGrid: true });
            });

            // insert entire HTML mark-up into the container node
            remoteLayerNode[0].innerHTML = markup;

        } : _.noop;

        // protected methods --------------------------------------------------

        /**
         * Changes the layers according to the passed layer range.
         */
        this.setLayerRange = RenderUtils.profileMethod('SelectionRenderer.setLayerRange()', function (layerSettings) {

            // store new layer range and rectangle for convenience
            layerRectangle = layerSettings.rectangle;

            // render own selection, and selections of all remote editors
            renderCellSelection();
            renderRemoteSelection();
        });

        /**
         * Resets this renderer, clears the DOM layer nodes.
         */
        this.hideLayerRange = function () {
            layerRectangle = null;
            selectionLayerNode.empty();
            remoteLayerNode.empty();
        };

        // initialization -----------------------------------------------------

        // render cell selection again when document edit mode changes
        this.listenToWhenVisible(docView.getApp(), 'docs:editmode', renderCellSelection);

        // render cell selection after merging/unmerging cell ranges (size of active cell may change)
        this.listenToWhenVisible(docView, 'insert:merged delete:merged', renderCellSelectionDebounced);

        // update layers according to changed document view attributes
        this.listenToWhenVisible(docView, 'change:viewattributes', function (event, attributes) {
            if ('remoteClients' in attributes) {
                renderRemoteSelection();
            }
        });

        // update the layer nodes after specific sheet view attributes have been changed
        this.listenToWhenVisible(docView, 'change:sheet:viewattributes', function (event, attributes) {

            // render cell selection (triggers a 'render:cellselection' event)
            if (Utils.hasProperty(attributes, /^(selection$|split|autoFillData$)/)) {
                renderCellSelection();
            }

            // trigger a 'render:cellselection' event if active pane changes
            // (no actual rendering needed, border colors will change via CSS)
            if ('activePane' in attributes) {
                self.trigger('render:cellselection');
            }
        });

        // post-processing after scrolling
        this.listenTo(gridPane, 'change:scrollpos', updateResizeHandlePositions);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = gridPane = docModel = docView = null;
            remoteLayerNode = selectionLayerNode = null;
        });

    } }); // class SelectionRenderer

    // exports ================================================================

    return SelectionRenderer;

});
