/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/autostylecache', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/baseframework/app/appobjectmixin',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/view/render/renderutils'
], function (Utils, ValueMap, BaseObject, AppObjectMixin, Color, Border, SheetUtils, RenderUtils) {

    'use strict';

    // convenience shortcuts
    var StyleDescriptor = RenderUtils.StyleDescriptor;

    // constants ==============================================================

    // the attribute names of all outer borders
    var OUTER_BORDER_NAMES = 'tblr'.split('').map(SheetUtils.getBorderName);

    // class AutoStyleCache ===================================================

    /**
     * A cache that stores rendering information for all known auto-styles in
     * the document.
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {SpreadsheetView} docView
     *  The document view that contains this cache instance.
     */
    var AutoStyleCache = BaseObject.extend({ constructor: function (docView) {

        // self reference
        var self = this;

        // the spreadsheet model
        var docModel = docView.getDocModel();

        // the number formatter of the spreadsheet document
        var numberFormatter = docModel.getNumberFormatter();

        // the attribute pool for cell attribute handling
        var attributePool = docModel.getCellAttributePool();

        // the cell auto-style collection of the document
        var autoStyles = docModel.getCellAutoStyles();

        // map of all style descriptor caches (separate caches per zoom factor, and grid color)
        var styleCacheMap = new ValueMap();

        // the current style descriptor cache (according to grid color and zoom)
        var styleCache = null;

        // the current grid line color
        var gridColor = new Color('auto');

        // the current zoom factor
        var sheetZoom = 0;

        // base constructors --------------------------------------------------

        BaseObject.call(this, docView);
        AppObjectMixin.call(this, docView.getApp());

        // private methods ----------------------------------------------------

        /**
         * In debug mode, collects the use counts of multiple cache accesses,
         * and prints a message to the browser console.
         */
        var logUsage = RenderUtils.isLoggingActive() ? (function () {

            var counts = { total: 0, hit: 0, update: 0, miss: 0 };

            function collect(type) {
                counts[type] += 1;
                counts.total += 1;
            }

            function format(key) {
                var count = counts[key];
                return (count > 0) ? (', ' + key + '=' + count + ' (' + Math.round(count / counts.total * 1000) / 10 + '%)') : '';
            }

            function flush() {
                RenderUtils.info('AutoStyleCache access statistics: total=' + counts.total + format('hit') + format('update') + format('miss'));
                _.forEach(counts, function (count, key) { counts[key] = 0; });
            }

            return self.createDebouncedMethod('AutoStyleCache.logUsage', collect, flush, { delay: 500 });
        }()) : _.noop;

        /**
         * Returns whether the passed explicit attribute set contains the text
         * color attribute.
         */
        function hasExplicitTextColor(attributeSet) {
            return _.isObject(attributeSet.character) && ('color' in attributeSet.character);
        }

        /**
         * Updates the passed style descriptor, if its own age is less than the
         * age of the current style cache.
         *
         * @param {StyleDescriptor} styleDesc
         *  The style descriptor to be updated if needed.
         *
         * @returns {StyleDescriptor}
         *  The passed style descriptor, for convenience.
         */
        function updateStyleDescriptor(styleDesc, type) {

            // update the style descriptor, if its age is less than the age of this cache
            if (styleDesc.age < styleCache.age) {
                styleDesc.update(docModel, gridColor, sheetZoom);
                styleDesc.age = styleCache.age;
                if (!type) { type = 'update'; }
            }

            logUsage(type || 'hit');
            return styleDesc;
        }

        /**
         * Creates a new style descriptor for the formatting attributes of the
         * specified column, row, or cell descriptor.
         *
         * @param {String} styleId
         *  The identifier of an auto-style.
         *
         * @param {String|Null} borderStyleId
         *  The identifier of an additional auto-style that will be used to
         *  resolve the style settings of all outer border attributes; or null
         *  to use the border attributes of the real auto-style.
         *
         * @param {Object|Null} condAttrSet
         *  A non-empty incomplete attribute set with formatting attributes of
         *  one or more conditional formatting rules to be merged over the
         *  attributes of the auto-style; or null, if no explicit attributes
         *  will be merged over the auto-style.
         *
         * @param {Object|Null} tableAttrSet
         *  An incomplete attribute set with formatting attributes from a table
         *  style sheet to be merged with the attributes of the auto-style; or
         *  null, if no table attributes will be merged over the auto-style.
         *
         * @returns {StyleDescriptor}
         *  The new style descriptor for the passed column, row, or cell.
         */
        function createStyleDescriptor(styleId, borderStyleId, condAttrSet, tableAttrSet) {

            // the merged attribute set of the auto-style
            var mergedAttrSet = autoStyles.getMergedAttributeSet(styleId);
            // whether the number format of the auto-style has been overridden by explicit attributes
            var formatChanged = false;
            // whether the text color has been set by an explicit formatting attribute
            var explicitTextColor = false;

            // creates a deep clone of the merged attribute set on first call
            var ensureClonedAttributeSet = _.once(function () {
                mergedAttrSet = _.copy(mergedAttrSet, true);
            });

            // updates the rendering priority of all border lines in the rmerged attribute set
            function setBorderPrio(attributSet, borderPrio) {
                if (attributSet.cell) {
                    OUTER_BORDER_NAMES.forEach(function (borderName) {
                        var border = attributSet.cell[borderName];
                        var mergedBorder = mergedAttrSet.cell[borderName];
                        if (border && !mergedBorder.prio && Border.isEqual(border, mergedBorder)) {
                            mergedBorder.prio = borderPrio;
                        }
                    });
                }
            }

            // merge the outer borders of another auto-style
            if (_.isString(borderStyleId)) {
                ensureClonedAttributeSet();
                var borderAttrSet = autoStyles.getMergedAttributeSet(borderStyleId);
                OUTER_BORDER_NAMES.forEach(function (borderName) {
                    mergedAttrSet.cell[borderName] = borderAttrSet.cell[borderName];
                });
            }

            // merge the passed explicit attributes over the resulting attributes
            if (condAttrSet) {
                ensureClonedAttributeSet();

                // special handling for number formats: detect whether the format has been changed by the explicit attributes
                var oldParsedFormat = autoStyles.getParsedFormat(styleId);
                attributePool.extendAttributeSet(mergedAttrSet, condAttrSet);
                var newParsedFormat = numberFormatter.getParsedFormatForAttributes(mergedAttrSet.cell);
                formatChanged = oldParsedFormat !== newParsedFormat;

                // update border priority (conditional border always wins against adjacent cell border)
                setBorderPrio(condAttrSet, 1);

                // detect whether the text color has been set explicitly
                explicitTextColor = hasExplicitTextColor(condAttrSet);
            }

            // merge the passed table attributes over the resulting attributes
            if (tableAttrSet) {
                ensureClonedAttributeSet();
                docModel.extendWithTableAttributeSet(mergedAttrSet, mergedAttrSet, tableAttrSet);
                // update border priority (cell border always wins against table style border)
                setBorderPrio(tableAttrSet, -1);
            }

            // if the text color has not been set explicitly by the passed attributes, it may be part of the auto-style itself
            if (!explicitTextColor) {
                explicitTextColor = hasExplicitTextColor(autoStyles.getExplicitAttributeSet(styleId));
            }

            // create a new style descriptor
            var styleDesc = new StyleDescriptor(mergedAttrSet, formatChanged, explicitTextColor);
            styleDesc.age = 0;

            // update the formatting information of the style descriptor
            return updateStyleDescriptor(styleDesc, 'miss');
        }

        function storeStyleKey(styleId, styleKey) {
            var styleKeys = styleCache.keys[styleId] || (styleCache.keys[styleId] = []);
            styleKeys.push(styleKey);
        }

        function deleteCachedStyles(styleId) {
            styleCacheMap.forEach(function (cacheEntry) {
                var styleKeys = cacheEntry.keys[styleId];
                if (styleKeys) {
                    styleKeys.forEach(function (styleKey) {
                        delete cacheEntry.map[styleKey];
                    });
                }
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Returns a style descriptor with rendering information for the
         * specified cell auto-style.
         *
         * @param {String} styleId
         *  The identifier of an auto-style.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {String} [options.borderStyleId]
         *      The identifier of an additional auto-style that will be used to
         *      resolve the style settings of all outer border attributes.
         *  - {Object} [options.condAttrSet]
         *      An incomplete attribute set with formatting attributes of one
         *      or more conditional formatting rules to be merged over the
         *      attributes of the auto-style. MUST NOT contain a style sheet
         *      identifier.
         *  - {Object} [options.tableAttrSet]
         *      An incomplete attribute set with formatting attributes from a
         *      table style sheet to be merged with the attributes of the
         *      auto-style. MUST NOT contain a style sheet identifier.
         *
         * @returns {StyleDescriptor}
         *  The style descriptor for the specified auto-style.
         */
        this.getStyle = function (styleId, options) {

            // normalize the default auto-style identifier
            styleId = autoStyles.getEffectiveStyleId(styleId);

            // an optional auto-style identifier for outer border attributes
            var borderStyleId = Utils.getStringOption(options, 'borderStyleId', null);
            if (autoStyles.areEqualStyleIds(styleId, borderStyleId)) { borderStyleId = null; }

            // conditional attributes to be merged over the auto-style
            var condAttrSet = Utils.getObjectOption(options, 'condAttrSet', null);
            if (_.isEmpty(condAttrSet)) { condAttrSet = null; }

            // table style attributes to be merged with the auto-style
            var tableAttrSet = Utils.getObjectOption(options, 'tableAttrSet', null);
            if (_.isEmpty(tableAttrSet)) { tableAttrSet = null; }

            // the cache key for the auto-style, and the explicit attributes
            var styleKey = Utils.stringifyJSON([styleId, borderStyleId, condAttrSet, tableAttrSet]);

            // if style descriptor exists already in the cache, update it on demand
            var styleDesc = styleCache.map[styleKey];
            if (styleDesc) { return updateStyleDescriptor(styleDesc); }

            // cache the complete style keys separated by auto-style identifiers
            // (needed to clean cache entries for changed or deleted auto-styles)
            storeStyleKey(styleId, styleKey);
            if (_.isString(borderStyleId)) { storeStyleKey(borderStyleId, styleKey); }

            // create a new style descriptor for an auto-style, and store it in the cache
            return (styleCache.map[styleKey] = createStyleDescriptor(styleId, borderStyleId, condAttrSet, tableAttrSet));
        };

        /**
         * Updates this cache according to the grid color and zoom index of the
         * active sheet in the spreadsheet document, and marks the cached style
         * descriptors to be dirty, if the grid color or zoom factor has really
         * been changed (the current grid color is used to render cell border
         * lines with automatic line color). When accessing a dirty style
         * descriptor the next time with the method AutoStyleCache.getStyle(),
         * its formatting settings will be updated automatically.
         *
         * @returns {AutoStyleCache}
         *  A reference to this instance.
         */
        this.updateStyles = function () {

            // model of the active sheet in the document
            var sheetModel = docView.getSheetModel();
            // current grid color of the active sheet
            var newGridColor = sheetModel.getGridColor();
            // current grid color, as CSS color value
            var oldCssColor = docModel.resolveColor(gridColor, 'line').css;
            // new grid color, as CSS color value
            var newCssColor = docModel.resolveColor(newGridColor, 'line').css;
            // whether the grid color has been changed
            var gridColorChanged = oldCssColor !== newCssColor;
            // current zoom factor of the active sheet
            var newSheetZoom = sheetModel.getEffectiveZoom();
            // whether the zoom factor has been changed
            var sheetZoomChanged = sheetZoom !== newSheetZoom;

            // nothing to do, if neither effective CSS grid color nor zoom factor have changed
            if (!gridColorChanged && !sheetZoomChanged) { return this; }

            // store the new settings, mark the styles to be dirty, but do not touch the descriptors yet
            gridColor = newGridColor;
            sheetZoom = newSheetZoom;

            // use permanent cache for simple grid colors
            var permColorKey = /^#(00|FF){3}$/i.test(newCssColor) ? newCssColor.toUpperCase() : '';
            // use permanent cache for all zoom factors that are a whole multiple of 5%
            var permZoomKey = (sheetZoom === Utils.round(sheetZoom, 0.05)) ? Math.round(sheetZoom * 100) : '';
            // key of the current style descriptor cache
            var cacheKey = permColorKey + ':' + permZoomKey;

            // create the style descriptor cache if missing
            styleCache = styleCacheMap.getOrCreate(cacheKey, function () { return { map: {}, keys: {}, age: 1 }; });

            // update the age of non-permanent style caches, if grid color or zoom factor has changed
            if ((gridColorChanged && !permColorKey) || (sheetZoomChanged && !permZoomKey)) {
                RenderUtils.log('AutoStyleCache.updateStyles(): style cache "' + cacheKey + '" invalidated');
                styleCache.age += 1;
            } else {
                RenderUtils.log('AutoStyleCache.updateStyles(): style cache "' + cacheKey + '" activated');
            }

            return this;
        };

        // initialization -----------------------------------------------------

        // invalidate the cache after inserting new fonts (CSS font family chains may change)
        this.listenTo(docModel.getFontCollection(), 'triggered', function () {
            styleCacheMap.clear();
            self.updateStyles();
        });

        // remove all cached data of changed and deleted auto-styles
        this.listenTo(autoStyles, 'change:autostyle delete:autostyle', function (event, styleId) {
            deleteCachedStyles(styleId);
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = docModel = numberFormatter = null;
            attributePool = autoStyles = styleCacheMap = styleCache = null;
        });

    } }); // class AutoStyleCache

    // exports ================================================================

    return AutoStyleCache;

});
