/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/formulapane', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/keycodes',
    'io.ox/office/baseframework/view/toolbar',
    'io.ox/office/baseframework/view/toolpane',
    'io.ox/office/spreadsheet/view/controls',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Forms, KeyCodes, ToolBar, ToolPane, Controls, gt) {

    'use strict';

    var Button = Controls.Button;

    // class FormulaPane =======================================================

    /**
     * The formula pane in spreadsheet documents containing the cell values
     *
     * @constructor
     *
     * @extends ToolPane
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view containing this instance.
     */
    var FormulaPane = ToolPane.extend({ constructor: function (docView) {

        // self reference
        var self = this;

        // the application instance
        var app = docView.getApp();

        // the spreadsheet document model
        var docModel = docView.getDocModel();

        // some nodes
        var paneNode = null;
        var leadingNode = null;
        var centerNode = null;

        // the trailing opener-icon
        var opener = null;
        // formulaBar extended, or not
        var extended = false;
        // Cell-Watcher
        var cellAddressContainer = $('<span class="currentCell">');

        // the underlay node containing highlighting for formulas
        var textUnderlayNode = $('<div class="underlay">');
        // extra attributes to prevent autocomplete aso on mobile devices
        var mobileAttrs = (Utils.SMALL_DEVICE) ? { autocomplete: 'off', autocorrect: 'off', autocapitalize: 'off', spellcheck: 'false' } : {};
        // the text area control
        var textAreaNode = $(Forms.createTextAreaMarkup({ attributes: _.extend({ disabled: true }, mobileAttrs) }));
        // the container element for the text underlay node and the text area control
        var textAreaContainer = $('<div class="textarea-container" data-focus-role="textarea">').append(textUnderlayNode, textAreaNode);

        // base constructor ---------------------------------------------------

        ToolPane.call(this, docView, { position: 'top', classes: 'formula-pane standard-design' });

        // private methods ----------------------------------------------------

        /**
         * disables or enables the formulaBar, if it's allowed
         */
        function handleInputFieldAccess() {
            var enabled = !docView.hasDrawingSelection() && (docView.ensureUnlockedActiveCell({ sync: true }) === null);
            textAreaNode.prop('disabled', !enabled);
        }

        /**
         * handler for leaving or entering the cell edit mode,
         * respectively losing or getting the editrights.
         */
        function modeHandler() {

            if (!docView.isTextEditMode('cell')) {
                var text = docView.hasDrawingSelection() ? '' : docView.getCellCollection().getEditString(docView.getActiveCell(), { matrixBraces: !docView.isTextEditMode('cell') });
                textAreaNode.val(text);
            }

            // hanlde the enable/disable-state of the formulaBar
            handleInputFieldAccess();
        }

        /**
         * Updates the formula pane after the state of the current sheet
         * selection has been changed.
         */
        function activeSelectionHandler(event, selection, tracking) {

            // the resulting caption text for the address label
            var address = '';

            // during tracking, show the size of the tracked range
            var activeRange = tracking ? selection.activeRange() : null;
            if (activeRange && !activeRange.single()) {
                var cols = activeRange.cols(), rows = activeRange.rows();
                if (docModel.isColRange(activeRange)) {
                    //#. info label shown while selecting entire columns in a spreadsheet
                    address = gt.ngettext('%1$d column', '%1$d columns', cols, _.noI18n(cols));
                } else if (docModel.isRowRange(activeRange)) {
                    //#. info label shown while selecting entire rows in a spreadsheet
                    address = gt.ngettext('%1$d row', '%1$d rows', rows, _.noI18n(rows));
                } else {
                    address = _.noI18n(rows + ' \xd7 ' + cols);
                }
            } else if (selection.drawings.length === 0) {
                address = _.noI18n(selection.address.toString());
            }

            cellAddressContainer.text(address);

            centerNode.parent().scrollTop(0);
            textAreaContainer.scrollTop(0);
            textAreaNode.scrollTop(0);

            modeHandler();
        }

        /**
         * handler to enter the cell edit mode, when the formulaBar gets the focus
         */
        function focusHandler() {
            if (!docView.isTextEditMode('cell')) {
                if (Utils.SMALL_DEVICE) {
                    docView.enterTextEditMode('cell', { formulaBarMode: false });
                } else {
                    docView.enterTextEditMode('cell', { formulaBarMode: true });
                    textUnderlayNode.scrollTop(textAreaNode.scrollTop());
                }

            }
        }

        /**
         * handler to scroll the underlay-node similarly to the formulaBar-node
         */
        function scrollTextAreaHandler() {
            self.executeDelayed(function () {
                textUnderlayNode.scrollTop(textAreaNode.scrollTop());
            }, 'FormulaPane.scrollTextAreaHandler', 100);
        }

        /**
         * handler to open/close the formulaBar.
         */
        function toggleFormulaBarHeight() {

            var height = 20;
            var extHeight = 3 * height;
            var setHeight = extended ? height : extHeight;

            self.setSize(setHeight + 3);

            textAreaNode.css('height', setHeight);
            centerNode.css('height', setHeight);

            opener.setIcon(extended ? 'fa-caret-down' : 'fa-caret-up');

            leadingNode.toggleClass('open', !extended);

            extended = !extended;
        }

        // public methods -----------------------------------------------------

        this.getTextInputNode = function () {
            return textAreaNode;
        };

        this.getTextUnderlayNode = function () {
            return textUnderlayNode;
        };

        /**
         * handles the inputField access (enable or disable) and set the cursor
         * into the formulaBar
         */
        this.grabFocus = function () {
            handleInputFieldAccess();

            textAreaNode[0].focus();
            textAreaNode[0].setSelectionRange(0, 0);
            textAreaNode.scrollTop(0);
        };

        /**
         * Toggles the height of the formulaBar
         */
        this.toggleHeight = function () {
            toggleFormulaBarHeight();
        };

        // initialization -----------------------------------------------------

        // set/grab some html-nodes
        paneNode = self.getNode();
        leadingNode = paneNode.find('>.leading>.area-container');
        centerNode = paneNode.find('>.center>.area-container');

        if (Utils.SMALL_DEVICE) {
            cellAddressContainer.addClass('small_device');
        }

        // build the formulaPane-html
        leadingNode.append(cellAddressContainer);
        centerNode.append(textAreaContainer);

        this.addViewComponent(new ToolBar(docView)
            .addGroup('cell/autoformula', new Button(docView, { icon: 'docs-auto-sum', tooltip: gt('Sum'), value: 'SUM' }))
            .addGroup('function/insert/dialog', new Button(docView, { icon: 'docs-functions', tooltip: gt('Functions') })),
        { targetArea: 'leading' });

        this.addViewComponent(new ToolBar(docView)
            .addGroup('view/formulapane/toggle/height', opener = new Button(docView, { icon: 'fa-caret-down', tooltip: gt('Expand formula bar') })),
        { targetArea: 'trailing' });

        // event listeners
        textAreaNode.on({
            focus: focusHandler,
            scroll: scrollTextAreaHandler
        });

        self.listenTo(docView, 'change:activesheet', modeHandler);
        self.listenTo(docView, 'active:selection', activeSelectionHandler);

        self.listenOnceTo(docView, 'change:activesheet', function () {
            self.listenTo(app, 'docs:editmode', modeHandler);
            self.listenTo(docView, 'textedit:enter textedit:leave change:sheet:attributes change:cells', modeHandler);
            app.waitForImportSuccess(modeHandler);
        });

        // initialize visibility
        this.toggle(!app.isViewerMode() && app.getUserSettingsValue('showFormulaBar', true));

        // always hide the formula pane when an internal application error occurs
        this.listenTo(app, 'docs:state:error', function () { self.hide(); });

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = app = docView = docModel = null;
            paneNode = leadingNode = centerNode = null;
            opener = extended = null;
            textUnderlayNode = textAreaNode = textAreaContainer = null;
        });

    } }); // class FormulaPane

    // exports ================================================================

    return FormulaPane;

});
