/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/hyperlinkcollection', [
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/tk/container/extarray',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (Iterator, ExtArray, ModelObject, SheetUtils) {

    'use strict';

    // constants ==============================================================

    // convenience shortcuts
    var IndexIterator = Iterator.IndexIterator;
    var RangeArray = SheetUtils.RangeArray;
    var RangeSet = SheetUtils.RangeSet;
    var MoveDescriptor = SheetUtils.MoveDescriptor;

    // class HyperlinkCollection ==============================================

    /**
     * Collects information about all hyperlinks of a single sheet in a
     * spreadsheet document.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */

    var HyperlinkCollection = ModelObject.extend({ constructor: function (sheetModel) {

        // self reference
        var self = this;

        // the document model instance
        var docModel = sheetModel.getDocModel();

        // all hyperlink ranges (may overlap), as Range objects with additional "url" properties
        var linkRangeSet = new RangeSet();

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Generates the operations needed to remove parts of the hyperlink
         * ranges.
         */
        function generateReduceLinkRangesOperations(generator, removeRange) {

            // collect all affected hyperlink ranges before removing elements from the set
            linkRangeSet.findAll(removeRange).forEach(function (linkRange) {

                // delete the existing hyperlink range
                generator.generateHyperlinkOperation(linkRange, null);

                // regenerate remaining parts of hyperlink range
                new RangeArray(linkRange).difference(removeRange).forEach(function (remainRange) {
                    generator.generateHyperlinkOperation(remainRange, linkRange.url);
                });

                // undo: restore the hyperlink range (will remove the smaller range parts generated above automatically)
                generator.generateHyperlinkOperation(linkRange, linkRange.url, { undo: true });
            });
        }

        /**
         * Transforms the specified hyperlink range.
         *
         * @param {Range} range
         *  The hyperlink range to be transformed.
         *
         * @param {Array<MoveDescriptor>} moveDescs
         *  An array of move descriptors that specify how to transform the
         *  passed hyperlink range.
         *
         * @param {Boolean} [reverse=false]
         *  If set to true, the move descriptors will be processed in reversed
         *  order, and the opposite move operation will be used to transform
         *  the hyperlink range.
         *
         * @returns {Range|Null}
         *  The transformed hyperlink range if available, otherwise null.
         */
        function transformHyperlinkRange(range, moveDescs, reverse) {
            // transform the passed range without expanding the end of the range
            return MoveDescriptor.transformRange(range, moveDescs, { reverse: reverse });
        }

        /**
         * Recalculates the position of all hyperlink ranges, after cells have
         * been moved (including inserted/deleted columns or rows) in the
         * sheet.
         */
        function moveCellsHandler(event, moveDesc) {
            var newLinkRangeSet = new RangeSet();
            linkRangeSet.forEach(function (linkRange) {
                var newLinkRange = transformHyperlinkRange(linkRange, [moveDesc]);
                if (newLinkRange) {
                    newLinkRange.url = linkRange.url;
                    newLinkRangeSet.insert(newLinkRange);
                }
            });
            linkRangeSet = newLinkRangeSet;
        }

        /**
         * Returns the internal contents of this collection, needed for cloning
         * into another collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @returns {Object}
         *  The internal contents of this collection.
         */
        this._getCloneData = function () {
            return { linkRangeSet: linkRangeSet };
        };

        // operation implementations ------------------------------------------

        /**
         * Callback handler for the document operation "copySheet". Clones all
         * hyperlinks from the passed collection into this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the "copySheet" document operation.
         *
         * @param {HyperlinkCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyCopySheetOperation = function (context, collection) {

            // the internal contents of the source collection
            var cloneData = collection._getCloneData();

            // clone the contents of the source collection
            linkRangeSet = cloneData.linkRangeSet.clone(function (newRange, oldRange) {
                newRange.url = oldRange.url;
            });
        };

        /**
         * Callback handler for the document operation "insertHyperlink" that
         * attaches a URL to a cell range.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the "insertHyperlink" document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyInsertHyperlinkOperation = function (context) {

            // the new hyperlink range
            var newLinkRange = context.getJSONRange();

            // remove all hyperlink ranges completely covered by the new range
            linkRangeSet.findAll(newLinkRange, 'contain').forEach(linkRangeSet.remove, linkRangeSet);

            // insert the new hyperlink range
            newLinkRange.url = context.getStr('url');
            linkRangeSet.insert(newLinkRange, true);

            // TODO: notify all change listeners?
        };

        /**
         * Callback handler for the document operation "deleteHyperlink" that
         * removes all covered hyperlink ranges.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the "deleteHyperlink" document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyDeleteHyperlinkOperation = function (context) {

            // the range to be cleared
            var range = context.getJSONRange();

            // remove the hyperlink range overlapping with the passed range
            linkRangeSet.findAll(range).forEach(linkRangeSet.remove, linkRangeSet);

            // TODO: notify all change listeners?
        };

        // public methods -----------------------------------------------------

        /**
         * Returns wether the passed range overlaps with any hyperlink range.
         *
         * @param {Range} range
         *  The address of the cell range to checked.
         *
         * @param {String} [matchType]
         *  Specifies which hyperlink ranges from this collection will match.
         *  See method RangeSet.findIterator() for details.
         *
         * @returns {Boolean}
         *  Wether the passed range overlaps with any hyperlink range.
         */
        this.coversAnyLinkRange = function (range, matchType) {
            return linkRangeSet.findOne(range, matchType) !== null;
        };

        /**
         * Returns all hyperlink ranges that overlap with the passed range.
         *
         * @param {Range} range
         *  The address of the cell range to checked.
         *
         * @param {String} [matchType]
         *  Specifies which hyperlink ranges from this collection will match.
         *  See method RangeSet.findIterator() for details.
         *
         * @returns {RangeArray}
         *  All hyperlink ranges that overlap with the passed cell range. Each
         *  array element contains the additional property "url".
         */
        this.getLinkRanges = function (range, matchType) {
            return linkRangeSet.findAll(range, matchType);
        };

        /**
         * Returns the URL of a hyperlink range covering the specified address.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {String|Null}
         *  The URL of a hyperlink attached to the specified cell; or null, if
         *  the cell does not contain a hyperlink.
         */
        this.getCellURL = function (address) {
            var linkRange = linkRangeSet.findOneByAddress(address);
            return linkRange ? linkRange.url : null;
        };

        // generator methods --------------------------------------------------

        /**
         * Generates the operations and undo operations to insert or delete
         * hyperlinks for the specified cell ranges.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address to
         *  generate the operations for.
         *
         * @param {String} url
         *  The URL of the hyperlink to be attached to the cell range. If set
         *  to the empty string, existing hyperlinks will be removed instead.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateHyperlinkOperations = function (generator, ranges, url) {

            // whether to generate 'deleteHyperlink' operations
            var deleteHyperlinks = url.length === 0;

            // remove ranges that are completely covered by other ranges
            var targetRanges = RangeArray.get(ranges).filterCovered();

            // create the operations to insert or delete hyperlinks
            return this.iterateArraySliced(targetRanges, function (targetRange) {

                // search matching hyperlink range for deleting single cell
                if (deleteHyperlinks && targetRange.single()) {
                    targetRange = linkRangeSet.findOne(targetRange) || targetRange;
                }

                // undo: delete the new hyperlink range before restoring the old ranges
                if (!deleteHyperlinks) {
                    generator.generateHyperlinkOperation(targetRange, null, { undo: true });
                }

                // delete the covered parts of all affected hyperlink ranges
                generateReduceLinkRangesOperations(generator, targetRange);

                // create the new hyperlink range
                if (!deleteHyperlinks) {
                    generator.generateHyperlinkOperation(targetRange, url);
                }
            }, 'HyperlinkCollection.generateHyperlinkOperations');
        };

        /**
         * Generates the operations, and the undo operations, to repeatedly
         * copy the hyperlink ranges from the source range into the specified
         * direction.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Range} range
         *  The address of the source cell range to be copied.
         *
         * @param {Direction} direction
         *  The direction in which the specified cell range will be expanded.
         *
         * @param {Number} count
         *  The number of columns/rows to extend the cell range into the
         *  specified direction.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateAutoFillOperations = function (generator, range, direction, count) {

            // the target range to be filled
            var targetRange = SheetUtils.getAdjacentRange(range, direction, count);
            // get source hyperlink ranges (reduced to the source range)
            var linkRanges = linkRangeSet.findAll(range).map(function (linkRange) {
                var newLinkRange = linkRange.intersect(range);
                newLinkRange.url = linkRange.url;
                return newLinkRange;
            });

            // first step for undo: delete the merged ranges generated by auto-fill
            if (!linkRanges.empty()) {
                generator.generateHyperlinkOperation(targetRange, null, { undo: true });
            }

            // delete the covered parts of all existing hyperlink ranges in the target range
            generateReduceLinkRangesOperations(generator, targetRange);

            // nothing more to do, if no hyperlink ranges exist in the source range
            if (linkRanges.empty()) { return this.createResolvedPromise(); }

            // number of repetitions
            var columns = !SheetUtils.isVerticalDir(direction);
            var size = range.size(columns);
            var cycles = Math.ceil(count / size);

            // create all hyperlink ranges for the target range
            var sign = SheetUtils.isLeadingDir(direction) ? -1 : 1;
            var promise = this.iterateSliced(new IndexIterator(cycles, { offset: 1 }), function (cycle) {
                var diff = sign * cycle * size;
                linkRanges.forEach(function (linkRange) {
                    var newLinkRange = linkRange.clone().moveBoth(diff, columns).intersect(targetRange);
                    if (newLinkRange) {
                        generator.generateHyperlinkOperation(newLinkRange, linkRange.url);
                    }
                });
            }, 'HyperlinkCollection.generateAutoFillOperations');

            return promise;
        };

        /**
         * Generates the undo operations to restore the hyperlink in this
         * collection that would not be restored automatically with the reverse
         * operation of the passed move descriptor.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Array<MoveDescriptor>} moveDescs
         *  An array of move descriptors that specify how to transform the
         *  hyperlink in this collection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateBeforeMoveCellsOperations = function (generator, moveDescs) {

            // collect all hyperlinks that cannot be restored by applying the reversed move operation
            var restoreLinks = new ExtArray();
            var promise = this.iterateSliced(linkRangeSet, function (linkRange) {

                // transform the hyperlinks back and forth to decide whether it can be restored implicitly
                var transformRange = transformHyperlinkRange(linkRange, moveDescs);
                var restoredRange = transformRange ? transformHyperlinkRange(transformRange, moveDescs, true) : null;

                // collect all ranges that cannot be restored implicitly
                if (!restoredRange || restoredRange.differs(linkRange)) {
                    restoreLinks.push(linkRange);
                }
            }, 'HyperlinkCollection.generateBeforeMoveCellsOperations');

            // add all hyperlinks to the undo generator that need to be restored
            promise = promise.then(function () {
                return self.iterateArraySliced(restoreLinks, function (linkRange) {
                    generator.generateHyperlinkOperation(linkRange, linkRange.url, { undo: true });
                }, 'HyperlinkCollection.generateBeforeMoveCellsOperations');
            });

            return promise;
        };

        /**
         * Generates the operations and undo operations to merge cells which
         * covers or overlaps hyperlink ranges
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address to
         *  generate the operations for.
         *
         * @param {MergeMode} mergeMode
         *  The merge mode for the specified ranges.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateMergeCellsOperations = function (generator, ranges/*, mergeMode*/) {

            var cellCollection = sheetModel.getCellCollection();

            return this.iterateArraySliced(RangeArray.get(ranges), function (range) {

                // the address to extract a hyperlink URL from (prefer value cell that will be moved to the top-left corner)
                var address = cellCollection.findFirstCell(range, { type: 'value', covered: true }) || range.start;
                // use the first hyperlink covering the address
                var linkRange = linkRangeSet.findOneByAddress(address);
                // empty string as URl deletes the hyperlinks
                var url = linkRange ? linkRange.url : '';

                return self.generateHyperlinkOperations(generator, range, url);
            }, 'HyperlinkCollection.generateMergeCellsOperations');
        };

        // initialization -----------------------------------------------------

        // update hyperlinks after moving cells, or inserting/deleting columns or rows
        this.listenTo(sheetModel, 'move:cells', moveCellsHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = sheetModel = linkRangeSet = null;
        });

    } }); // class HyperlinkCollection

    // exports ================================================================

    return HyperlinkCollection;

});
