/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/utils/mathutils', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/errorcode'
], function (Utils, ErrorCode) {

    'use strict';

    // shortcuts to mathematical functions
    var floor = Math.floor;
    var ceil = Math.ceil;
    var abs = Math.abs;
    var pow = Math.pow;
    var sqrt = Math.sqrt;
    var exp = Math.exp;
    var log = Math.log;

    // mathematical constants
    var PI = Math.PI;
    var SQRT_PI = sqrt(PI);
    var LN10 = Math.LN10;

    // smallest positive normalized number
    var MIN_NORM = Utils.MIN_NUMBER;

    // Depth for the taylor expansion (exponent) and continued fraction expansion (number of fractions)
    // to calculate an approximation for erf() and erfc(), see functions erf_taylor() and erfc_cfrac().
    var ERF_DEPTH = 25;

    // The threshold for x values to select between the two expansion algorithms to calculate erf().
    // Using a depth of 25, the minimum distance between both expansions is about 6.6791e-13 at x=1.9785.
    // Therefore, for 0<=x<=1.9785, taylor expansion will be used, and for x>1.9785, the continued
    // fraction expansion will be used.
    var ERF_THRESHOLD = 1.9785;

    // static private functions ===============================================

    /**
     * Returns whether the passed number is considered to be equal to zero by
     * the formula engine. Beside positive and negative zero, all denormalized
     * numbers (all numbers with an absolute value less than 2^-1022) will be
     * handled as zero too.
     *
     * @param {Number} number
     *  The number to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed number is zero, or denormalized. Returns false for
     *  positive/negative infinite, or for NaN.
     */
    function isZero(number) {
        return abs(number) < MIN_NORM;
    }

    /**
     * Implementation of the taylor expansion for erf() with maximum depth 25.
     * The maximum error to the 'better' results of erf() as computed with
     * taylor expansion, depth 50, is about 2.35e-13 at x=1.9785 (the threshold
     * point between taylor expansion and continued fraction expansion).
     *
     * @param {Number} x
     *  The input parameter. MUST NOT be negative.
     *
     * @returns {Number}
     *  The approximated result of erf(x) for small positive x.
     */
    function erf_taylor(x) {
        var y = x;
        for (var i = 1, j = 3, fact = 1, sgn = -1, y2 = 0; (i <= ERF_DEPTH) && (y !== y2); i += 1, j += 2, fact *= i, sgn = -sgn) {
            y2 = y;
            y += sgn * pow(x, j) / j / fact;
        }
        return 2 * y / SQRT_PI;
    }

    /**
     * Implementation of the continued fraction expansion for erfc() with depth
     * 25. The maximum error to the 'better' results of erfc() as computed with
     * continued fraction expansion, depth 50, is about 4.3e-13 at x=1.9785
     * (the threshold point between taylor expansion and continued fraction
     * expansion).
     *
     * @param {Number} x
     *  The input parameter. MUST NOT be negative.
     *
     * @returns {Number}
     *  The approximated result of erfc(x) for large positive x.
     */
    function erfc_cfrac(x) {
        var y = 0;
        for (var i = ERF_DEPTH, j = (i % 2 + 1); i >= 1; i -= 1, j = 3 - j) {
            y = i / (j * x + y);
        }
        return exp(-x * x) / SQRT_PI / (x + y);
    }

    // static class MathUtils =================================================

    /**
     * A collection of low-level mathematical helper methods.
     */
    var MathUtils = {};

    // static methods ---------------------------------------------------------

    /**
     * Returns whether the passed number is considered to be equal to zero by
     * the formula engine. Beside positive and negative zero, all denormalized
     * numbers (all numbers with an absolute value less than 2^-1022) will be
     * handled as zero too.
     *
     * @param {Number} number
     *  The number to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed number is zero, or denormalized. returns false for
     *  positive/negative infinite, or for NaN.
     */
    MathUtils.isZero = isZero;

    /**
     * Removes the fractional part of the signed number, and leaves the integer
     * part, i.e. rounds the passed number towards zero to the next integer.
     *
     * @param {Number} number
     *  The number to be truncated.
     *
     * @returns {Number}
     *  The truncated number.
     */
    MathUtils.trunc = function (number) {
        return (number < 0) ? ceil(number) : floor(number);
    };

    /**
     * Returns the sum of the passed numbers (useful for callbacks).
     *
     * @param {Number} number1
     *  The first summand.
     *
     * @param {Number} number2
     *  The second summand.
     *
     * @returns {Number}
     *  The sum of the passed numbers.
     */
    MathUtils.add = function (number1, number2) {
        return number1 + number2;
    };

    /**
     * Returns the difference of the passed numbers (useful for callbacks).
     *
     * @param {Number} number1
     *  The minuend.
     *
     * @param {Number} number2
     *  The subtrahend.
     *
     * @returns {Number}
     *  The difference of the passed numbers.
     */
    MathUtils.sub = function (number1, number2) {
        return number1 - number2;
    };

    /**
     * Returns the product of the passed numbers (useful for callbacks).
     *
     * @param {Number} number1
     *  The first factor.
     *
     * @param {Number} number2
     *  The second factor.
     *
     * @returns {Number}
     *  The product of the passed numbers.
     */
    MathUtils.mul = function (number1, number2) {
        return number1 * number2;
    };

    /**
     * Returns the quotient of the passed numbers. If the divisor is zero, the
     * error code #DIV/0! will be thrown instead.
     *
     * @param {Number} number1
     *  The divident.
     *
     * @param {Number} number2
     *  The divisor.
     *
     * @returns {Number}
     *  The quotient of the passed numbers.
     *
     * @throws {ErrorCode}
     *  The #DIV/0! error code, if the divisor is zero.
     */
    MathUtils.div = function (number1, number2) {
        if (isZero(number2)) { throw ErrorCode.DIV0; }
        return number1 / number2;
    };

    /**
     * Returns the floating-point modulo of the passed numbers. If the divisor
     * is zero, the error code #DIV/0! will be thrown instead. If the passed
     * numbers have different signs, the result will be adjusted according to
     * the results of other spreadsheet applications.
     *
     * @param {Number} number1
     *  The divident.
     *
     * @param {Number} number2
     *  The divisor.
     *
     * @returns {Number}
     *  The modulo of the passed numbers.
     *
     * @throws {ErrorCode}
     *  The #DIV/0! error code, if the divisor is zero.
     */
    MathUtils.mod = function (number1, number2) {
        if (isZero(number2)) { throw ErrorCode.DIV0; }
        var result = number1 % number2;
        return (!isZero(result) && (number1 * number2 < 0)) ? (result + number2) : result;
    };

    /**
     * Returns the power of the passed numbers. If both numbers are zero, the
     * error code #NUM! will be thrown instead of returning 1. If the base is
     * zero, and the exponent is negative, the error code #DIV/0! will be
     * thrown instead of returning INF.
     *
     * @param {Number} base
     *  The base of the power to be calculated.
     *
     * @param {Number} expn
     *  The exponent of the power to be calculated.
     *
     * @returns {Number}
     *  The passed base raised to the exponent.
     *
     * @throws {ErrorCode}
     *  The error code #NUM!, if base and exponent are zero.
     *  The error code #DIV/0!, if base is zero, and exponent is negative.
     */
    MathUtils.powNaN = function (base, expn) {
        // Math.pow() returns 1 for 0^0; Excel returns #NUM! error
        if (isZero(base)) {
            if (isZero(expn)) { throw ErrorCode.NUM; }
            if (expn < 0) { throw ErrorCode.DIV0; }
        }
        return pow(base, expn);
    };

    /**
     * Returns the power of the passed numbers. If both numbers are zero, the
     * number 1 will be returned. If the base is zero, and the exponent is
     * negative, the error code #NUM! will be thrown instead of returning INF.
     *
     * @param {Number} base
     *  The base of the power to be calculated.
     *
     * @param {Number} expn
     *  The exponent of the power to be calculated.
     *
     * @returns {Number}
     *  The passed base raised to the exponent.
     *
     * @throws {ErrorCode}
     *  The error code #NUM!, if base is zero, and exponent is negative.
     */
    MathUtils.pow1 = function (base, expn) {
        if (isZero(base) && (expn < 0)) { throw ErrorCode.NUM; }
        return pow(base, expn);
    };

    /**
     * An object that contains resolver functions for calculating the power of
     * two numbers, mapped by file format identifiers.
     *
     * @constant
     */
    MathUtils.POW = { ooxml: MathUtils.powNaN, odf: MathUtils.pow1 };

    /**
     * Returns the passed number raised to the power of 10.
     *
     * @param {Number} expn
     *  The exponent.
     *
     * @returns {Number}
     *  The number 10 raised to the passed exponent.
     */
    MathUtils.pow10 = function (expn) {
        return pow(10, expn);
    };

    /**
     * Returns the decimal logarithm of the passed number.
     *
     * @param {Number} number
     *  The number to calculate the logarithm for.
     *
     * @returns {Number}
     *  The decimal logarithm of the passed number.
     */
    MathUtils.log10 = function (number) {
        return log(number) / LN10;
    };

    /**
     * Returns the two-parameter arc tangent of the passed coordinates. If both
     * numbers are zero, the error code #DIV/0! will be thrown instead. The
     * order of the parameters 'x' and 'y' is switched compared to the native
     * method Math.atan2, according to usage in other spreadsheet applications.
     *
     * @param {Number} x
     *  The X coordinate.
     *
     * @param {Number} y
     *  The Y coordinate.
     *
     * @returns {Number}
     *  The arc tangent of the passed coordinates.
     *
     * @throws {ErrorCode}
     *  The error code #DIV/0!, if both numbers are zero.
     */
    MathUtils.atan2 = function (x, y) {
        if (isZero(x) && isZero(y)) { throw ErrorCode.DIV0; }
        // Spreadsheet applications use (x,y); but Math.atan2() expects (y,x)
        return Math.atan2(y, x);
    };

    /**
     * Returns the hyperbolic sine of the passed number.
     */
    MathUtils.sinh = Math.sinh || function (number) {
        return (exp(number) - exp(-number)) / 2;
    };

    /**
     * Returns the hyperbolic cosine of the passed number.
     */
    MathUtils.cosh = Math.cosh || function (number) {
        return (exp(number) + exp(-number)) / 2;
    };

    /**
     * Returns the hyperbolic tangent of the passed number.
     */
    MathUtils.tanh = Math.tanh || function (number) {
        var pexp = exp(number), nexp = exp(-number);
        return (pexp - nexp) / (pexp + nexp);
    };

    /**
     * Returns the argument resulting in the passed hyperbolic sine.
     */
    MathUtils.asinh = Math.asinh || function (number) {
        return log(number + sqrt(number * number + 1));
    };

    /**
     * Returns the argument resulting in the passed hyperbolic cosine.
     */
    MathUtils.acosh = Math.acosh || function (number) {
        return log(number + sqrt(number * number - 1));
    };

    /**
     * Returns the argument resulting in the passed hyperbolic tangent.
     */
    MathUtils.atanh = Math.atanh || function (number) {
        return log((1 + number) / (1 - number)) / 2;
    };

    /**
     * Returns the argument resulting in the passed hyperbolic cotangent.
     */
    MathUtils.acoth = Math.acoth || function (number) {
        return log((number + 1) / (number - 1)) / 2;
    };

    /**
     * Returns the sum of the passed numbers.
     *
     * @param {Array<Numbers>} numbers
     *  The numbers to be added.
     *
     * @returns {Number}
     *  The sum of all passed numbers, or 0 if the passed array is empty.
     */
    MathUtils.sum = function (numbers) {
        return numbers.reduce(MathUtils.add, 0);
    };

    /**
     * Returns the product of the passed numbers.
     *
     * @param {Array<Numbers>} numbers
     *  The numbers to be multiplied.
     *
     * @returns {Number}
     *  The product of all passed numbers, or 1 if the passed array is empty.
     */
    MathUtils.product = function (numbers) {
        return numbers.reduce(MathUtils.mul, 1);
    };

    /**
     * Returns the factorial of the passed number. Throws the error code #NUM!,
     * if the passed number is negative or too large.
     *
     * @param {Number} number
     *  The number to calculate the factorial for.
     *
     * @returns {Number}
     *  The factorial of the passed number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed number is negative, or too large.
     */
    MathUtils.factorial = (function () {

        // create and return the cache for all available factorials from 0 to 170
        var getFactorials = _.once(function () {
            var FACTORIALS = [];
            var fact = 1;
            while (isFinite(fact)) {
                FACTORIALS.push(fact);
                fact *= FACTORIALS.length;
            }
            return FACTORIALS;
        });

        // public implementation, will cause to create the cached factorials on first call
        return function factorial(number) {
            var fact = getFactorials()[floor(number)];
            if (!fact) { throw ErrorCode.NUM; }
            return fact;
        };
    }());

    /**
     * Returns the double factorial of the passed number. Throws the error code
     * #NUM!, if the passed number is negative or too large.
     *
     * @param {Number} number
     *  The number to calculate the double factorial for.
     *
     * @returns {Number}
     *  The double factorial of the passed number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed number is negative, or too large.
     */
    MathUtils.factorial2 = (function () {

        // create and return the cache for all available factorials from 0 to 170
        var getFactorials = _.once(function () {
            var FACTORIALS = [];
            var fact1 = 1;
            var fact2 = 1;
            while (isFinite(fact1)) {
                FACTORIALS.push(fact1);
                fact1 *= (FACTORIALS.length + 1);
                if (isFinite(fact2)) {
                    FACTORIALS.push(fact2);
                    fact2 *= (FACTORIALS.length + 1);
                }
            }
            return FACTORIALS;
        });

        // public implementation, will cause to create the cached factorials on first call
        return function factorial2(number) {
            var fact = getFactorials()[floor(number)];
            if (!fact) { throw ErrorCode.NUM; }
            return fact;
        };
    }());

    /**
     * Returns the binomial coefficient of the passed input values.
     *
     * @param {Number} n
     *  The upper value of the binomial coefficient (size of the set to be
     *  chosen from).
     *
     * @param {Number} k
     *  The lower value of the binomial coefficient (number of choices from the
     *  set).
     *
     * @returns {Number}
     *  The binomial coefficient, if both values are non-negative, and 'n' is
     *  greater than or equal to 'k'.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed values are invalid, or if the
     *  resulting binomial coefficient is too large.
     */
    MathUtils.binomial = (function () {

        // cache for different combinations of input parameters
        var BINOM_CACHE = {};
        // maximum input parameters for cache usage
        var CACHE_MAX_NUM = 50;

        // core implementation without error checking
        function coreBinomial(n, k) {

            // the resulting binomial coefficient
            var result = n - k + 1;

            // The function isFinite() is quite expensive, by using two nested loops (the inner
            // loop runs up to 10 iterations without checking the result) the entire algorithm
            // becomes about 30% to 40% faster.
            for (var i = 2; (i <= k) && isFinite(result);) {
                for (var j = 0; (j < 10) && (i <= k); i += 1, j += 1) {
                    result /= i;
                    result *= (n - k + i);
                }
            }
            return result;
        }

        // public implementation with error checking and caching
        function binomial(n, k) {

            // check input values
            n = floor(n);
            k = floor(k);
            if ((n < 0) || (k < 0) || (n < k)) { throw ErrorCode.NUM; }

            // special cases: (n 0)=1 and (n n)=1
            if ((k === 0) || (k === n)) { return 1; }
            // special cases: (n 1)=n and (n n-1)=n
            if ((k === 1) || (k === n - 1)) { return n; }

            // binomial coefficient is symmetric: (n k) = (n n-k); use lower k for performance
            if (k * 2 > n) { k = n - k; }

            // use the cache, if both numbers are in a specific limit (k<=n, see above)
            var result = 0;
            if (n <= CACHE_MAX_NUM) {
                var key = n + ',' + k;
                result = BINOM_CACHE[key];
                // result is always valid (finite) in the cachable limit
                return result ? result : (BINOM_CACHE[key] = coreBinomial(n, k));
            }

            // do not cache larger numbers, check final result (may be infinite)
            result = coreBinomial(n, k);
            if (isFinite(result)) { return result; }
            throw ErrorCode.NUM;
        }

        return binomial;
    }());

    /**
     * Returns the result of the Gauss error function.
     *
     * @param {Number} x
     *  The input value.
     *
     * @returns {Number}
     *  The result value of the Gauss error function.
     */
    MathUtils.erf = (function () {

        // Implementation of erf() for non-negative numbers. For all x equal to or greater than 6, the result is
        // assumed to be 1. The real result of erf() is too close to 1 to be represented with the limited
        // precision of JavaScript floating-point numbers.
        function erf_abs(x) {
            return (x >= 6) ? 1 : (x > ERF_THRESHOLD) ? (1 - erfc_cfrac(x)) : erf_taylor(x);
        }

        // Public implementation of erf(). Use the fact that erf() is an odd function: erf(-x)=-erf(x), with a
        // shortcut for the function's zero: erf(0)=0.
        function erf(x) {
            return (x > 0) ? erf_abs(x) : (x < 0) ? -erf_abs(-x) : 0;
        }

        return erf;
    }());

    /**
     * Returns the result of the complementary Gauss error function.
     *
     * @param {Number} x
     *  The input value.
     *
     * @returns {Number}
     *  The result value of the complementary Gauss error function.
     */
    MathUtils.erfc = function (x) {
        // Prevent arithmetic underflow for large x: erf(6) returns exactly 1 due to precision limitations
        // of floating-point numbers; but erfc(6)=1-erf(6) is close to but greater than 0.
        return (x > ERF_THRESHOLD) ? erfc_cfrac(x) : (1 - MathUtils.erf(x));
    };

    /**
     * Returns the linear slope of a collection of Y values.
     *
     * @param {Array<Numbers>} numbers
     *  The Y values to be processed. The array indexes of the numbers are used
     *  as the X values associated to the array elements.
     *
     * @returns {Number}
     *  The linear slope of the passed numbers.
     *
     * @throws {ErrorCode}
     *  The #DIV/0! error code, if the passed array is empty.
     */
    MathUtils.slope = function (numbers) {
        var meanX = (numbers.length - 1) / 2;
        var meanY = MathUtils.div(MathUtils.sum(numbers), numbers.length);
        var sumDYX = numbers.reduce(function (sum, y, x) { return sum + (x - meanX) * (y - meanY); }, 0);
        var sumDX2 = meanX * (meanX + 1) * (2 * meanX + 1) / 3;
        return sumDYX / sumDX2;
    };

    /**
     * Returns the sum of the squares of the absolute distances of the passed
     * numbers to their arithmetic mean.
     *
     * @param {Array<Number>} numbers
     *  The numbers to be processed.
     *
     * @param {Number} sum
     *  The sum of the passed numbers. In most situations, the sum of these
     *  numbers is already avaiable (e.g. during aggregation of function
     *  operands), and is therefore expected to be passed to this method.
     *
     * @returns {Number}
     *  The sum of the squares of the absolute distances of the passed numbers
     *  to their arithmetic mean.
     *
     * @throws {ErrorCode}
     *  The #DIV/0! error code, if the passed array is empty.
     */
    MathUtils.devSq = function (numbers, sum) {
        var mean = MathUtils.div(sum, numbers.length);
        return numbers.reduce(function (sum, number) {
            var diff = number - mean;
            return sum + diff * diff;
        }, 0);
    };

    /**
     * Returns the variance or the standard deviation of the passed numbers.
     *
     * @param {Array<Number>} numbers
     *  The numbers to calculate the variance or standard deviation for.
     *
     * @param {Number} sum
     *  The sum of the passed numbers. In most situations, the sum of these
     *  numbers is already avaiable (e.g. during aggregation of function
     *  operands), and is therefore expected to be passed to this method.
     *
     * @param {String} method
     *  The result type, either 'dev' for the standard deviation, or 'var' for
     *  the variance.
     *
     * @param {Boolean} population
     *  If set to true, calculates the variance or standard deviation based on
     *  the entire population (the number of elements in the passed array). If
     *  set to false, calculates the result based on a sample (the number of
     *  elements reduced by one).
     *
     * @returns {Number}
     *  The variance of the passed numbers.
     *
     * @throws {ErrorCode}
     *  The #DIV/0! error code, if the passed array is empty; or if the passed
     *  array contains only one element, and the parameter 'population' is set
     *  to false (variance based on sample).
     */
    MathUtils.variance = function (numbers, sum, method, population) {
        var devSq = MathUtils.devSq(numbers, sum);
        var size = population ? numbers.length : (numbers.length - 1);
        var variance = MathUtils.div(devSq, size);
        return (method === 'dev') ? sqrt(variance) : variance;
    };

    // exports ================================================================

    return MathUtils;

});
