/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/colrowcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iterator',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/model/attributedmodel',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/operationcontext'
], function (Utils, Iterator, ModelObject, Border, AttributedModel, Operations, SheetUtils, SheetOperationContext) {

    'use strict';

    // convenience shortcuts
    var IndexIterator = Iterator.IndexIterator;
    var TransformIterator = Iterator.TransformIterator;
    var FilterIterator = Iterator.FilterIterator;
    var ReduceIterator = Iterator.ReduceIterator;
    var NestedIterator = Iterator.NestedIterator;
    var Interval = SheetUtils.Interval;
    var IntervalArray = SheetUtils.IntervalArray;

    // class ColRowModel ======================================================

    /**
     * Represents the model entries of a ColRowCollection instance: an interval
     * of columns or rows with equal size and default formatting for all cells.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @property {Interval} interval
     *  The column/row interval represented by this instance.
     *
     * @property {Boolean} columns
     *  The orientation of this instance: true for columns, false for rows.
     *
     * @property {Number} offsetHmm
     *  Absolute position of the first column/row represented by this instance,
     *  in 1/100 of millimeters.
     *
     * @property {Number} sizeHmm
     *  Size of a single column/row represented by this instance, in 1/100 of
     *  millimeters.
     *
     * @property {Number} offset
     *  Absolute position of the first column/row represented by this instance,
     *  in pixels.
     *
     * @property {Number} size
     *  Size of a single column/row represented by this instance, in pixels.
     *
     * @property {String} style
     *  The identifier of the cell auto-style used for all undefined cells.
     */
    var ColRowModel = AttributedModel.extend({ constructor: function (sheetModel, interval, columns, initAttributes, styleId, listenToParent) {

        // helper properties: names of attributes and attribute family
        this.FAMILY_NAME = columns ? 'column' : 'row';
        this.SIZE_ATTR_NAME = columns ? 'width' : 'height';

        // base constructor
        AttributedModel.call(this, sheetModel.getDocModel(), initAttributes, {
            families: this.FAMILY_NAME + ' group',
            parentModel: sheetModel,
            listenToParent: listenToParent,
            autoClear: true
        });

        // column/row interval covered by this model
        this.interval = interval.clone();
        this.columns = columns;

        // effective position/size of the columns/rows, in 1/100 mm
        this.offsetHmm = 0;
        this.sizeHmm = 0;

        // effective position/size of the columns/rows, in pixels
        this.offset = 0;
        this.size = 0;

        // the cell auto-style for default formatting of undefined cells
        this.style = styleId;

    } }); // class ColRowModel

    // protected methods ------------------------------------------------------

    /**
     * Creates and returns a cloned instance of this entry model for the
     * specified sheet.
     *
     * @internal
     *  Used by the class ColRowCollection during clone construction. DO NOT
     *  CALL from external code!
     *
     * @returns {ColRowModel}
     *  A clone of this entry model, initialized for ownership by the passed
     *  sheet model.
     */
    ColRowModel.prototype.clone = function (sheetModel) {
        var newModel = new ColRowModel(sheetModel, this.interval, this.columns, this.getExplicitAttributeSet(true), this.style, false);
        newModel.offsetHmm = this.offsetHmm;
        newModel.sizeHmm = this.sizeHmm;
        newModel.offset = this.offset;
        newModel.size = this.size;
        return newModel;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns the end offset of this entry model in 1/100 mm, independent from
     * the current sheet zoom factor.
     */
    ColRowModel.prototype.getEndOffsetHmm = function () {
        return this.offsetHmm + this.sizeHmm * this.interval.size();
    };

    /**
     * Returns the end offset of this entry model in pixels, according to the
     * current sheet zoom factor.
     */
    ColRowModel.prototype.getEndOffset = function () {
        return this.offset + this.size * this.interval.size();
    };

    /**
     * Returns the effective merged column/row attributes of this entry model.
     *
     * @returns {Object}
     *  The merged attribute values of the 'column' attribute family, if this
     *  model represents columns, otherwise the merged attribute values of the
     *  'row' attribute family.
     */
    ColRowModel.prototype.getMergedEntryAttributes = function () {
        return this.getMergedAttributeSet(true)[this.FAMILY_NAME];
    };

    /**
     * Returns the explicit column/row attributes of this entry model.
     *
     * @returns {Object}
     *  The explicit attribute values of the 'column' attribute family, if this
     *  model represents columns, otherwise the explicit attribute values of
     *  the 'row' attribute family.
     */
    ColRowModel.prototype.getExplicitEntryAttributes = function () {
        return this.getExplicitAttributeSet(true)[this.FAMILY_NAME] || {};
    };

    /**
     * Returns whether this model contains an active auto-style. The auto-style
     * of a column model is always active; the auto-style of a row model is
     * active, if the row attribute 'customFormat' is set to true.
     *
     * @returns {Boolean}
     *  Whether this model contains an active auto-style.
     */
    ColRowModel.prototype.hasAutoStyle = function () {
        return this.columns || this.getMergedEntryAttributes().customFormat;
    };

    // class ColRowDescriptor =================================================

    /**
     * A simple descriptor object for a column or row, returned by the public
     * methods of the class ColRowCollection.
     *
     * @constructor
     *
     * @property {Number} index
     *  The zero-based index of the column or row.
     *
     * @property {Interval} uniqueInterval
     *  The column/row interval containing this entry with the same formatting
     *  attributes and auto-style.
     *
     * @property {Number} offsetHmm
     *  The absolute position of the column/row, in 1/100 mm.
     *
     * @property {Number} sizeHmm
     *  The effective size (zero for hidden columns/rows), in 1/100 mm.
     *
     * @property {Number} offset
     *  The absolute position of the column/row, in pixels.
     *
     * @property {Number} size
     *  The effective size (zero for hidden columns/rows), in pixels.
     *
     * @property {String} style
     *  The identifier of the auto-style containing the character and cell
     *  formatting attributes for all undefined cells in the column/row.
     *
     * @property {Object} attributes
     *  The merged attribute set of the cell auto-style. MUST NOT be changed!
     *
     * @property {ParsedFormat} format
     *  The parsed number format code extracted from the cell auto-style.
     *
     * @property {Object} merged
     *  The effective merged column/row attributes, as simple key/value map
     *  (NOT mapped by the 'column' or 'row' family name). MUST NOT be changed!
     *
     * @property {Object} explicit
     *  The explicit column/row attributes, as simple key/value map (NOT mapped
     *  by the 'column' or 'row' family name). MUST NOT be changed!
     */
    function ColRowDescriptor(autoStyles, entryModel, index, uniqueInterval) {

        // column/row index, and unique formatting interval
        this.index = index;
        this.uniqueInterval = uniqueInterval || entryModel.interval.clone();

        // position and size
        var relIndex = index - entryModel.interval.first;
        this.offsetHmm = entryModel.offsetHmm + entryModel.sizeHmm * relIndex;
        this.sizeHmm = entryModel.sizeHmm;
        this.offset = entryModel.offset + entryModel.size * relIndex;
        this.size = entryModel.size;

        // the cell auto-style, and the merged attributes
        this.style = entryModel.style;
        this.attributes = autoStyles.getMergedAttributeSet(this.style);
        this.format = autoStyles.getParsedFormat(this.style);

        // the merged and explicit column/row attributes, as simple object
        this.merged = entryModel.getMergedEntryAttributes();
        this.explicit = entryModel.getExplicitEntryAttributes();

    } // class ColRowDescriptor

    // class ColRowCollection =================================================

    /**
     * Collects information about all columns or all rows of a single sheet in
     * a spreadsheet document.
     *
     * Triggers the following events:
     * - 'insert:entries'
     *      After new columns/rows have been inserted into the sheet. The event
     *      handler receives the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Interval} interval
     *          The index interval of the inserted entries.
     * - 'delete:entries'
     *      After columns/rows have been deleted from the sheet. The event
     *      handler receives the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Interval} interval
     *          The index interval of the deleted entries.
     * - 'change:entries'
     *      After the formatting attributes, or the auto-style, of a column/row
     *      interval have been changed. The event handler receives the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Interval} interval
     *          The index interval of the changed entries.
     *      (3) {Object} changeInfo
     *          Additional information about the changed index interval, in the
     *          following properties:
     *          - {Object|Boolean|Null} options.attributes
     *              The explicit column/row attributes applied at the interval;
     *              or the value true, if the entire collection has been
     *              changed (e.g. after changing the sheet default attributes);
     *              or the value null, if no explicit column/row attributes
     *              have been changed.
     *          - {String|Null} options.styleId
     *              The identifier of the new cell auto-style applied at the
     *              interval; or the value null, if the auto-style has not been
     *              changed.
     *          - {Boolean} options.sizeChanged
     *              Whether the effective size of a column/row in the interval
     *              has been changed, including changed visibility of the
     *              columns or rows (hidden columns/rows have effective size of
     *              zero).
     *          - {Boolean} options.visibilityChanged
     *              Whether the visibility of a column/row in the interval has
     *              been changed (the property 'size' of this object will also
     *              be true in this case, see above).
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     *
     * @param {Boolean} columns
     *  Whether this collection contains column headers (true) or row headers
     *  (false).
     */
    var ColRowCollection = ModelObject.extend({ constructor: function (sheetModel, columns) {

        // self reference
        var self = this;

        // the document model
        var docModel = sheetModel.getDocModel();

        // the attribute pool for column/row attributes
        var attributePool = docModel.getAttributePool();

        // the cell auto-style collection of the document model
        var autoStyles = docModel.getCellAutoStyles();

        // sorted sparse array of index ranges (instances of ColRowModel)
        var entryModels = [];

        // the largest valid column/row index
        var maxIndex = docModel.getMaxIndex(columns);

        // the current zoom factor of the sheet
        var effectiveZoom = 0;

        // the padding added to the inner size available for text contents at 100% zoom
        var totalPadding100 = 0;

        // the padding added to the inner size available for text contents at current zoom
        var totalPaddingZoom = 0;

        // default model for entries missing in the collection
        var defaultModel = new ColRowModel(sheetModel, new Interval(0, maxIndex), columns, null, '', true);

        // attribute family (column or row)
        var FAMILY_NAME = columns ? 'column' : 'row';

        // name of the size attribute according to the attribute family
        var SIZE_ATTR_NAME = columns ? 'width' : 'height';

        // the names of all formatting attributes that will change the effective size an entry of this collection
        var RESIZE_ATTRIBUTE_NAMES = attributePool.getRegisteredAttributes(FAMILY_NAME).reduce({}, function (set, definition, name) {
            if (definition.resize) { set[name] = true; }
            return set;
        });

        // the name of the document operation to insert new entries into the collection
        var INSERT_OPERATION_NAME = columns ? Operations.INSERT_COLUMNS : Operations.INSERT_ROWS;

        // the name of the document operation to insert new entries into the collection
        var DELETE_OPERATION_NAME = columns ? Operations.DELETE_COLUMNS : Operations.DELETE_ROWS;

        // the name of the document operation to change formatting attributes and auto-styles
        var CHANGE_OPERATION_NAME = columns ? Operations.CHANGE_COLUMNS : Operations.CHANGE_ROWS;

        // maximum number of columns/rows that can be changed simultaneously
        var MAX_CHANGE_COUNT = columns ? SheetUtils.MAX_CHANGE_COLS_COUNT : SheetUtils.MAX_CHANGE_ROWS_COUNT;

        // helper function that converts column/row size from operation units to 1/100mm
        var convertSizeFromUnit = (columns ? docModel.convertColWidthFromUnit : docModel.convertRowHeightFromUnit).bind(docModel);

        // helper function that converts column/row size from 1/100mm to operation units
        var convertSizeToUnit = (columns ? docModel.convertColWidthToUnit : docModel.convertRowHeightToUnit).bind(docModel);

        // special behavior for OOXML
        var ooxml = docModel.getApp().isOOXML();

        // base constructor ---------------------------------------------------

        // bug 38436: events must be triggered during import, in order to update position of drawing objects
        ModelObject.call(this, docModel, { trigger: 'always' });

        // private methods ----------------------------------------------------

        /**
         * Returns the array index of an entry model that contains, follows, or
         * precedes the passed column/row index.
         *
         * @param {Number} index
         *  The zero-based column/row index. Must be located in the valid range
         *  covered by this collection.
         *
         * @param {Boolean} [reverse=false]
         *  If set to true, and the passed index is located in a gap between
         *  two entry models, the array index of the preceding entry model will
         *  be returned, instead of the following entry model.
         *
         * @returns {Number}
         *  The array index of the entry model containing the passed index. If
         *  the index is located in a gap between two entries, returns the
         *  array index of the entry model starting after (in reversed mode:
         *  before) the passed index. This may also be an array index pointing
         *  after the last existing entry (in reversed mode: array index before
         *  the first model, i.e. -1).
         */
        function getModelArrayIndex(index, reverse) {
            var arrayIndex = reverse ?
                Utils.findLastIndex(entryModels, function (entryModel) { return entryModel.interval.first <= index; }, { sorted: true }) :
                Utils.findFirstIndex(entryModels, function (entryModel) { return index <= entryModel.interval.last; }, { sorted: true });
            return (arrayIndex < 0) ? (reverse ? -1 : entryModels.length) : arrayIndex;
        }

        /**
         * Updates internal settings according to the current effective sheet
         * zoom factor.
         */
        function updateZoomSettings() {
            effectiveZoom = sheetModel.getEffectiveZoom();
            totalPadding100 = SheetUtils.getTotalCellPadding(docModel.getDefaultDigitWidth(1));
            totalPaddingZoom = SheetUtils.getTotalCellPadding(sheetModel.getDefaultDigitWidth());
        }

        /**
         * Updates the size properties of the passed column/row model.
         *
         * @param {ColRowModel} model
         *  The column/row model to be updated.
         */
        function updateModelSize(model) {

            // the merged attributes of the correct family
            var attributes = model.getMergedEntryAttributes();

            // effective size of hidden columns/rows is zero (filtered rows are hidden too)
            if (!attributes.visible || (!columns && attributes.filtered)) {
                model.sizeHmm = model.size = 0;
                return;
            }

            // start with the effective size of a single column or row, in 1/100 mm
            model.sizeHmm = Math.max(0, convertSizeFromUnit(attributes[model.SIZE_ATTR_NAME]));

            // different calculation methods for column width and row height
            if (columns) {

                // if the column width is zero, use the default size provided by the sheet
                if (model.sizeHmm === 0) {
                    // the base column width (in number of digits) to be used for undefined columns
                    var baseWidth = sheetModel.getMergedAttributeSet(true).sheet.baseColWidth;
                    // the width of the digits of the default font
                    var digitWidth = sheetModel.getDefaultDigitWidth();
                    // the default text padding according to zoom and digit width
                    var textPadding = SheetUtils.getTextPadding(digitWidth);
                    // padding for base column width is a multiple of 8 pixels, depending on standard text padding
                    model.size = Math.ceil(digitWidth * baseWidth) + 8 * (Math.floor(textPadding / 4) + 1);
                    model.sizeHmm = sheetModel.convertPixelToHmm(model.size);
                } else {
                    // the inner size available for text contents at 100% zoom
                    var innerSize = Utils.convertHmmToLength(model.sizeHmm, 'px') - totalPadding100;
                    // scale the available inner size according to the current zoom factor, and add current effective padding
                    model.size = Math.ceil(innerSize * effectiveZoom) + totalPaddingZoom;
                }

            } else {

                // if the row height is zero, use the default height provided by the sheet
                if (model.sizeHmm === 0) {
                    // the character attributes of the default cell style sheet
                    var defCharAttrs = docModel.getCellStyles().getDefaultStyleAttributeSet().character;
                    // resolve the resulting row height from the default character attributes
                    model.sizeHmm = docModel.getRowHeightHmm(defCharAttrs);
                    model.size = sheetModel.getRowHeight(defCharAttrs);
                } else {
                    // simple conversion to pixels for row height
                    model.size = sheetModel.convertHmmToPixel(model.sizeHmm);
                }
            }

            // ensure a specific minimum pixel size to prevent rendering problems (e.g. clipping artifacts)
            model.size = Math.max(SheetUtils.MIN_CELL_SIZE, model.size);
        }

        /**
         * Updates the dirty entry model offsets and sizes up to the end of the
         * collection, after changing the collection, or the sheet attributes.
         *
         * @param {Number} startIndex
         *  The zero-based column/row index where updating the offsets starts.
         *
         * @param {Boolean} [updateSize=false]
         *  If set to true, the size of the entries will be updated too. By
         *  default, only the offset positions of the entries will be updated.
         */
        function updateModelGeometry(startIndex, updateSize) {

            // the array index of the first dirty entry model
            var arrayIndex = getModelArrayIndex(startIndex);
            // the last valid entry model
            var prevModel = entryModels[arrayIndex - 1];
            // the current entry model to be updated
            var currModel = null;

            // update start offsets of all dirty collection entries
            for (; arrayIndex < entryModels.length; arrayIndex += 1) {

                // set offset of current entry to the end offset of the previous entry
                currModel = entryModels[arrayIndex];
                currModel.offsetHmm = prevModel ? prevModel.getEndOffsetHmm() : 0;
                currModel.offset = prevModel ? prevModel.getEndOffset() : 0;

                // add size of the gap between previous and current entry
                var currFirst = currModel.interval.first,
                    prevLast = prevModel ? (prevModel.interval.last + 1) : 0;
                if ((prevLast === 0) || (prevLast < currFirst)) {
                    currModel.offsetHmm += defaultModel.sizeHmm * (currFirst - prevLast);
                    currModel.offset += defaultModel.size * (currFirst - prevLast);
                }

                // update size of the current entry model (first update the merged
                // attributes, parent sheet attributes may have been changed)
                if (updateSize) {
                    currModel.refreshMergedAttributeSet();
                    updateModelSize(currModel);
                }

                prevModel = currModel;
            }
        }

        /**
         * Updates the default column/row size according to the current
         * formatting attributes of the sheet model.
         */
        function updateDefaultSize() {

            // the old default size, in 1/100 mm, and in pixels
            var oldDefSizeHmm = defaultModel.sizeHmm;
            var oldDefSize = defaultModel.size;

            // calculate the new default size
            updateModelSize(defaultModel);

            // do nothing, if the effective size has not changed (check both sizes to catch any rounding errors)
            if ((defaultModel.sizeHmm !== oldDefSizeHmm) || (defaultModel.size !== oldDefSize)) {
                updateModelGeometry(0, true);
                var visibility = (oldDefSize === 0) !== (defaultModel.size === 0);
                var changeInfo = { attributes: true, styleId: null, sizeChanged: true, visibilityChanged: visibility };
                self.trigger('change:entries', self.getFullInterval(), changeInfo);
            }
        }

        /**
         * Returns whether the passed entry model is not formatted differently
         * than the sheet default.
         */
        function isDefaultFormatted(entryModel) {
            return autoStyles.areEqualStyleIds(entryModel.style, defaultModel.style) && !entryModel.hasExplicitAttributes();
        }

        /**
         * Returns whether both passed entry models contain the same formatting
         * attributes, and cell auto-style identifier.
         */
        function areEquallyFormatted(entryModel1, entryModel2) {
            return autoStyles.areEqualStyleIds(entryModel1.style, entryModel2.style) && entryModel1.hasEqualAttributes(entryModel2);
        }

        /**
         * Inserts the passed entry model into the internal array.
         *
         * @param {Number} arrayIndex
         *  The array index for the new entry model.
         *
         * @param {ColRowModel} entryModel
         *  The new entry model to be inserted.
         */
        function insertModel(arrayIndex, entryModel) {
            entryModels.splice(arrayIndex, 0, entryModel);
        }

        /**
         * Deletes the specified entry models from the array.
         *
         * @param {Number} arrayIndex
         *  The array index of the first entry model to be deleted.
         *
         * @param {Number} [count]
         *  The number of entry models to be deleted. If omitted, all entry
         *  models up to the end of the array will be deleted.
         */
        function deleteModels(arrayIndex, count) {
            // call destructor for all deleted entries
            _.invoke(entryModels.splice(arrayIndex, count), 'destroy');
        }

        /**
         * Splits the entry model at the passed array index, and inserts the
         * new entry model after the existing entry model.
         *
         * @param {Number} arrayIndex
         *  The array index of the entry model to be split.
         *
         * @param {Number} leadingCount
         *  The new interval size of the existing entry model. Must be less
         *  than the current interval size of the entry model.
         *
         * @returns {ColRowModel}
         *  The new entry model inserted after the existing entry model.
         */
        function splitModel(arrayIndex, leadingCount) {

            // the existing collection entry
            var oldModel = entryModels[arrayIndex];
            // the clone of the existing entry
            var newModel = oldModel.clone(sheetModel);

            // adjust start and end position of the entries
            oldModel.interval.last = oldModel.interval.first + leadingCount - 1;
            newModel.interval.first = oldModel.interval.first + leadingCount;
            newModel.offsetHmm = oldModel.offsetHmm + leadingCount * oldModel.sizeHmm;
            newModel.offset = oldModel.offset + leadingCount * oldModel.size;

            // insert the new entry model into the collection
            insertModel(arrayIndex + 1, newModel);
            return newModel;
        }

        /**
         * Tries to merge the entry model at the passed array index with its
         * predecessor and/or successor, if these entries are equal. If
         * successful, the preceding and/or following entry models will be
         * removed from this collection.
         *
         * @param {Number} arrayIndex
         *  The array index of the entry model to be merged with its
         *  predecessor and /or successor. May be an invalid index (no entry
         *  exists in the collection), or may refer to the first or last entry
         *  model (no predecessor or successor exists in the collection).
         *
         * @param {Object} [options]
         *  Optional parameters specifying which entry models will be merged
         *  with the specified entry model:
         *  - {Boolean} [options.prev=false]
         *      If set to true, the entry and its predecessor will be tried to
         *      merge.
         *  - {Boolean} [options.next=false]
         *      If set to true, the entry and its successor will be tried to
         *      merge.
         *
         * @returns {Object}
         *  An object containing the Boolean flags 'prev' and 'next' specifying
         *  whether the entry model has been merged with its predecessor and/or
         *  successor respectively.
         */
        function mergeModel(arrayIndex, options) {

            // tries to merge the entry with its predecessor, removes the preceding entry
            function tryMerge(currIndex) {

                // the preceding collection entry
                var prevModel = entryModels[currIndex - 1];
                // the current collection entry
                var thisModel = entryModels[currIndex];

                // check that the entries are equal and do not have a gap
                if (prevModel && thisModel && (prevModel.interval.last + 1 === thisModel.interval.first) && areEquallyFormatted(prevModel, thisModel)) {
                    thisModel.interval.first = prevModel.interval.first;
                    thisModel.offsetHmm = prevModel.offsetHmm;
                    thisModel.offset = prevModel.offset;
                    deleteModels(currIndex - 1, 1);
                    return true;
                }
                return false;
            }

            // try to merge the entry with its successor/predecessor
            var result = {};
            result.next = Utils.getBooleanOption(options, 'next', false) && tryMerge(arrayIndex + 1);
            result.prev = Utils.getBooleanOption(options, 'prev', false) && tryMerge(arrayIndex);
            return result;
        }

        /**
         * Creates an iterator that visits all entry models that cover the
         * specified column/row interval.
         *
         * @param {Interval} interval
         *  The column/row interval to be visited by the iterator.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.visible=false]
         *      If set to true, only visible entry models will be covered by
         *      the iterator. By default, all visible and hidden models will be
         *      visited.
         *  - {Boolean} [options.reverse=false]
         *      If set to true, the entry models will be visited in reversed
         *      order.
         *
         * @returns {Iterator}
         *  The model iterator. The result object of the iterator will contain
         *  the following value properties on success:
         *  - {ColRowModel} value
         *      The entry model currently visited. For gaps between existing
         *      entry models, the default entry model will be passed here.
         *  - {Interval} interval
         *      The index interval actually covered by the entry model, AND by
         *      the interval passed to this method.
         *  - {Number} offsetHmm
         *      The start position of the result interval (the result property
         *      'interval'), in 1/100 of millimeters.
         *  - {Number} offset
         *      The start position of the result interval (the result property
         *      'interval'), in pixels.
         */
        function createModelIterator(interval, options) {

            // whether to visit visible entries only
            var visible = Utils.getBooleanOption(options, 'visible', false);
            // whether to iterate in reversed order
            var reverse = Utils.getBooleanOption(options, 'reverse', false);
            // current entry index during iteration
            var index = reverse ? interval.last : interval.first;
            // the array index of the entry model containing the start index
            var arrayIndex = getModelArrayIndex(index, reverse);
            // the iterator object returned by this method
            var iterator = new Iterator();

            // returns the iterator result for the passed entry model and interval
            function createResult(entryModel, stepInterval) {

                // move index to the next interval for the next iterator step
                index = reverse ? (stepInterval.first - 1) : (stepInterval.last + 1);

                // next index out of interval: nothing to visit in the next iteration step
                if (!interval.containsIndex(index)) {
                    iterator.abort();
                }

                // shorten the passed interval to the interval visited by the iterator
                return { done: false, value: entryModel, interval: stepInterval.intersect(interval) };
            }

            // returns the iterator result for the passed entry model
            function createModelResult(entryModel) {

                // create an iterator result for the passed entry model
                var result = createResult(entryModel, entryModel.interval);

                // add the start positions of the interval actually covered by the result
                var relIndex = result.interval.first - entryModel.interval.first;
                result.offsetHmm = entryModel.offsetHmm + entryModel.sizeHmm * relIndex;
                result.offset = entryModel.offset + entryModel.size * relIndex;

                // return the iterator result object
                return result;
            }

            // returns the iterator result for the passed interval of a gap between entry models
            function createGapResult(first, last) {

                // create an iterator result with the default entry model used for the gaps
                var result = createResult(defaultModel, new Interval(first, last));

                // start position of the gap is the end offset of the preceding model visited before
                // (reverse mode: of the next model to be visited)
                var gapOffsetModel = reverse ? entryModels[arrayIndex] : entryModels[arrayIndex - 1];

                // start position of the gap, as index, and in 1/100mm and pixels
                var gapIndex = gapOffsetModel ? (gapOffsetModel.interval.last + 1) : 0;
                var gapOffsetHmm = gapOffsetModel ? gapOffsetModel.getEndOffsetHmm() : 0;
                var gapOffset = gapOffsetModel ? gapOffsetModel.getEndOffset() : 0;

                // add the start positions of the interval actually covered by the result
                var relIndex = result.interval.first - gapIndex;
                result.offsetHmm = gapOffsetHmm + defaultModel.sizeHmm * relIndex;
                result.offset = gapOffset + defaultModel.size * relIndex;

                // return the iterator result object
                return result;
            }

            // the iterator implementation
            iterator.next = function () {

                // empty collection: single iterator step on the full interval
                if (entryModels.length === 0) {
                    return createGapResult(0, maxIndex);
                }

                // fail-safety: check for invalid source intervals
                if (reverse ? (index < 0) : (index > maxIndex)) {
                    Utils.error('ColRowCollection.createModelIterator(): invalid source interval: ' + interval.stringifyAs(columns));
                    return iterator.abort();
                }

                // the current entry model (may be undefined, e.g. for leading/trailing gap)
                var entryModel = entryModels[arrayIndex];

                // visit the gap after the last (reverse: before the first) entry model
                if (!entryModel) {
                    return reverse ?
                        createGapResult(0, entryModels[0].interval.first - 1) :
                        createGapResult(_.last(entryModels).interval.last + 1, maxIndex);
                }

                // the complete interval of the next entry model
                var modelInterval = entryModel.interval;

                // visit the gap before (reverse: behind) the next entry model
                if (!modelInterval.containsIndex(index)) {
                    return reverse ?
                        createGapResult(modelInterval.last + 1, index) :
                        createGapResult(index, modelInterval.first - 1);
                }

                // prepare array index for the following (reverse: preceding) entry model for next iterator step
                arrayIndex += (reverse ? -1 : 1);

                // visit the entry model
                return createModelResult(entryModel);
            };

            // filter the visible entry models if specified
            return visible ? new FilterIterator(iterator, 'size') : iterator;
        }

        /**
         * Updates the effective offsets and sizes of all columns/rows in this
         * collection, after the zoom factor of the sheet has been changed.
         */
        function changeViewAttributesHandler(event, attributes) {
            if ('zoom' in attributes) {
                updateZoomSettings();
                updateModelSize(defaultModel);
                updateModelGeometry(0, true);
            }
        }

        /**
         * Creates and returns the single-index intervals representing the
         * adjacent columns/rows of all passed index intervals.
         *
         * @param {IntervalArray} intervals
         *  The interval array to be processed.
         *
         * @param {Boolean} leading
         *  Whether to create an interval array for the leading adjacent index
         *  entries (true), or for the trailing index entries (false).
         *
         * @returns {IntervalArray}
         *  The intervals representing the adjacent columns/rows of all passed
         *  index intervals.
         */
        function getAdjacentIntervals(intervals, leading) {
            return IntervalArray.map(intervals, leading ?
                function (interval) { return (interval.first > 0) ? new Interval(interval.first - 1) : null; } :
                function (interval) { return (interval.last < maxIndex) ? new Interval(interval.last + 1) : null; }
            );
        }

        /**
         * Returns whether the passed tagged style intervals can be merged to a
         * single tagged style intervals. The intervals must lay exactly side
         * by side, and the otional style properties "s" and "attrs" must be
         * equal (or equally missing).
         *
         * @param {Interval} interval1
         *  The first interval to be compared with the second interval.
         *
         * @param {Interval} interval2
         *  The second interval to be compared with the first interval.
         *
         * @returns {Boolean}
         *  Whether the first interval precedes the second interval without a
         *  gap, and the style properties "s" and "attrs" of both intervals are
         *  equal.
         */
        function canMergeTaggedIntervals(interval1, interval2) {
            return (interval1.last + 1 === interval2.first) &&
                autoStyles.areEqualStyleIds(interval1.s, interval2.s) &&
                _.isEqual(interval1.attrs, interval2.attrs);
        }

        /**
         * Extends the last tagged style interval in the array, if the passed
         * tagged style interval follows directly, and the style settings
         * (auto-style, and formatting attributes) of both intervals are equal.
         * Otherwise, the passed interval will be appended to the array.
         *
         * @param {IntervalArray} intervals
         *  The sorted interval array to be extended with the passed interval.
         *  Each interval object in the array may contain the additional
         *  properties "s" and "attrs". The last array element will be modified
         *  in-place if possible.
         *
         * @param {Interval} interval
         *  The new interval to be inserted into the interval array. MUST be
         *  located behind the last interval in the array. May contain the
         *  additional properties "s" and "attrs". If neither of the properties
         *  exists, the interval will NOT be inserted into the array.
         *
         * @returns {IntervalArray}
         *  The passed interval array, for convenience.
         */
        function appendTaggedInterval(intervals, interval) {
            if (('s' in interval) || ('attrs' in interval)) {
                var lastInterval = intervals.last();
                if (lastInterval && canMergeTaggedIntervals(lastInterval, interval)) {
                    lastInterval.last = interval.last;
                } else {
                    intervals.push(interval);
                }
            }
            return intervals;
        }

        /**
         * Merges adjacent tagged style intervals with equal style settings in
         * the passed interval array if possible.
         *
         * @param {IntervalArray} intervals
         *  The sorted interval array whose intervals will be merged in-place
         *  if possible. Each interval object in the array may contain the
         *  additional properties "s" and "attrs".
         *
         * @returns {IntervalArray}
         *  The passed interval array, for convenience.
         */
        function mergeTaggedIntervals(intervals) {
            intervals.forEachReverse(function (interval, index) {
                var nextInterval = intervals[index + 1];
                if (nextInterval && canMergeTaggedIntervals(nextInterval, interval)) {
                    interval.last = nextInterval.last;
                    intervals.splice(index + 1, 1);
                }
            });
            return intervals;
        }

        /**
         * Creates a new document operation to change the passed interval (a
         * 'changeColumns' or 'changeRows' operation, according to the
         * orientation of this collection).
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the new operation.
         *
         * @param {Interval} interval
         *  The tagged style interval for the document operation, with the
         *  additional optional properties "s" and "attrs".
         *
         * @param {Object} [options]
         *  Optional parameters to be passed to the operations generator.
         */
        function createChangeIntervalOperation(generator, interval, options) {
            var properties = _.pick(interval, 's', 'attrs');
            if (!_.isEmpty(properties)) {
                generator.generateIntervalOperation(CHANGE_OPERATION_NAME, interval, properties, options);
            }
        }

        /**
         * Generates the operations, and the undo operations, to change the
         * auto-styles and formatting attributes of multiple columns or rows.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval, to
         *  generate operations for. Each interval may contain the following
         *  additional properties:
         *  - {Object} [fillData]
         *      Additional formatting settings for the interval. These data
         *      objects will be collected in an array, and will be passed to
         *      the callback function (see below).
         *
         * @param {Function} callback
         *  A callback function that will be invoked for each index interval of
         *  the partition of the passed index intervals. Receives the style
         *  data objects extracted from the original intervals covered by the
         *  current partition interval as simple array in the first parameter.
         *  MUST return the contents object for that interval with formatting
         *  attributes and other settings, as described in the method
         *  ColRowCollection.generateFillOperations(). Additionally, the
         *  following internal properties are supported:
         *  - {String} [contents.cacheKey]
         *      A unique key for the formatting attributes and auto-style
         *      carried in this contents object. If set, an internal cache will
         *      be filled with the resulting auto-style identifiers, and will
         *      be used while processing the index intervals (performance
         *      optimization).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved (with the index intervals that have
         *  really been changed) when all operations have been generated, or
         *  that will be rejected with an object with 'cause' property set to
         *  one of the following error codes:
         *  - 'cols:overflow': Too many columns in the index intervals.
         *  - 'rows:overflow': Too many rows in the index intervals.
         */
        var generateChangeIntervalOperations = SheetUtils.profileAsyncMethod('ColRowCollection.generateChangeIntervalOperations()', function (generator, intervals, callback) {
            SheetUtils.log('intervals=' + intervals.stringifyAs(columns));

            // ensure to work with an array, early exit for empty arrays
            intervals = IntervalArray.get(intervals);
            if (intervals.empty()) { return self.createResolvedPromise(new IntervalArray()); }

            // the collected intervals with new formatting and/or auto-style
            var newIntervals = new IntervalArray();
            // the collected undo intervals to restore the old formatting and/or auto-style
            var undoIntervals = new IntervalArray();
            // do not waste time to collect the fill data of the original intervals, if the callback does not use them
            var collectFillData = callback.length > 0;

            // create an iterator that visits a (sorted) partition of the intervals, and adds a
            // 'contents' property to the result objects
            var iterator = new TransformIterator(intervals.partition(), function (interval, result) {

                // collect the fill data of all source ranges covered by the current row band and column interval
                // (performance optimization: do not collect fill data, if callback function does not use them)
                var fillDataArray = collectFillData ? _.pluck(interval.coveredBy, 'fillData').filter(_.identity) : null;

                // create the cell contents entry for the current partition interval
                result.contents = callback.call(self, fillDataArray);
                return result;
            });

            // create an iterator that visits the different entry models in an interval, and forwards the 'contents' property
            iterator = new NestedIterator(iterator, createModelIterator, function (arrayResult, modelResult) {
                modelResult.contents = arrayResult.contents;
                return modelResult;
            });

            // whether the effective size of the collection entries will be changed by the operations
            var resizeEntries = false;

            // create the style intervals (intervals with effective attributes and auto-style identifiers)
            var promise = self.iterateSliced(iterator, function (entryModel, result) {

                // the interval object with new formatting attrtibutes and/or auto-style
                var newInterval = result.interval.clone();
                // the undo interval object to restore the old state
                var undoInterval = result.interval.clone();
                // the contents object to be applied to the current interval
                var contents = result.contents;

                // find/create the auto-style to be applied to the current interval
                var oldStyleId = entryModel.hasAutoStyle() ? entryModel.style : autoStyles.getDefaultStyleId();
                var newStyleId = generator.generateAutoStyle(oldStyleId, contents);

                // add auto-style to the interval objects, if changed
                if (!autoStyles.areEqualStyleIds(oldStyleId, newStyleId)) {
                    newInterval.s = newStyleId;
                    undoInterval.s = oldStyleId;
                }

                // get the column/row attributes, add the 'customFormat' row attribute, if the formatting changes
                var newAttributes = _.isEmpty(contents.attrs) ? null : _.clone(contents.attrs);
                if (!columns && ('s' in newInterval)) {
                    (newAttributes || (newAttributes = {})).customFormat = true;
                }

                // convert column/row size in 1/100mm to operation units according to file format
                if (newAttributes && !contents.nativeSize && (typeof newAttributes[SIZE_ATTR_NAME] === 'number')) {
                    newAttributes[SIZE_ATTR_NAME] = convertSizeToUnit(newAttributes[SIZE_ATTR_NAME]);
                }

                // create the resulting attribute set to be applied at the current entry model
                var newAttributeSet = _.isEmpty(newAttributes) ? null : entryModel.getReducedAttributeSet(Utils.makeSimpleObject(FAMILY_NAME, newAttributes));

                // add formatting attributes to the interval objects, if changed
                if (!_.isEmpty(newAttributeSet)) {

                    // bug 56758: OOXML needs the default column width when showing hidden columns
                    var newColAttrs = (columns && ooxml) ? newAttributeSet.column : null;
                    if (newColAttrs && (newColAttrs.visible === true) && !('width' in newColAttrs)) {
                        var oldColAttrs = entryModel.getMergedEntryAttributes();
                        if (oldColAttrs.width === 0) {
                            newColAttrs.width = convertSizeToUnit(defaultModel.sizeHmm);
                            newColAttrs.customWidth = false;
                        }
                    }

                    newInterval.attrs = newAttributeSet;
                    undoInterval.attrs = entryModel.getUndoAttributeSet(newAttributeSet);

                    // detect resized collection entries
                    resizeEntries = resizeEntries || _.some(newAttributeSet[FAMILY_NAME], function (value, name) {
                        return name in RESIZE_ATTRIBUTE_NAMES;
                    });
                }

                // add the style interval, if it contains changed formatting attributes, or a new auto-style
                appendTaggedInterval(newIntervals, newInterval);
                appendTaggedInterval(undoIntervals, undoInterval);

            }, 'ColRowCollection.generateChangeIntervalOperations');

            // generate the column/row operations, and simultaneously update the column/row collection
            promise = promise.then(function () {

                // generate the undo operations synchronously
                undoIntervals.forEach(function (interval) {
                    createChangeIntervalOperation(generator, interval, { undo: true });
                });

                // generate the column/row operations, and simultaneously update the column/row collection
                return self.iterateArraySliced(newIntervals, function (interval) {
                    createChangeIntervalOperation(generator, interval);
                }, 'ColRowCollection.generateChangeIntervalOperations');
            });

            // update the positions of all drawing objects in the sheet, after the columns/rows have been resized
            promise = promise.then(function () {
                if (resizeEntries) {
                    return sheetModel.generateRefreshAnchorOperations(generator);
                }
            });

            // return the positions of all changed intervals
            return promise.then(function () {
                SheetUtils.log('changed=' + newIntervals.stringifyAs(columns));
                return newIntervals;
            });
        });

        /**
         * Generates the undo operations to restore the auto-style and explicit
         * formatting attributes of the specified intervals.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the undo operations.
         *
         * @param {IntervalArray} intervals
         *  The index intervals to generate the undo operations for.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all undo operations have been
         *  generated.
         */
        function generateRestoreIntervalOperations(generator, intervals) {

            // create all undo intervals, and try to merge them
            var undoIntervals = new IntervalArray();
            var promise = self.iterateArraySliced(intervals, function (interval) {
                return self.iterateSliced(createModelIterator(interval), function (entryModel, result) {

                    // the undo interval, the additional properties for the interval operation
                    var undoInterval = result.interval.clone();

                    // restore non-default auto-styles
                    if (entryModel.hasAutoStyle() && !autoStyles.isDefaultStyleId(entryModel.style)) {
                        undoInterval.s = entryModel.style;
                    }

                    // restore explicit column/row attributes
                    var attrSet = entryModel.getExplicitAttributeSet();
                    if (!_.isEmpty(attrSet)) {
                        undoInterval.attrs = attrSet;
                    }

                    // create a change operation, if an auto-style or explicit attributes need to be restored
                    appendTaggedInterval(undoIntervals, undoInterval);

                }, 'ColRowCollection.generateRestoreIntervalOperations');
            }, 'ColRowCollection.generateRestoreIntervalOperations');

            // create the change operations from the merged undo intervals
            return promise.done(function () {
                undoIntervals.forEach(function (undoInterval) {
                    createChangeIntervalOperation(generator, undoInterval, { undo: true });
                });
            });
        }

        /**
         * Returns the internal contents of this collection, needed for cloning
         * into another collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @returns {Object}
         *  The internal contents of this collection.
         */
        this._getCloneData = function () {
            return { entryModels: entryModels };
        };

        // operation implementations ------------------------------------------

        /**
         * Callback handler for the document operation 'copySheet'. Clones all
         * contents from the passed collection into this collection.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         *
         * @param {ColRowCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyCopySheetOperation = function (context, collection) {

            // the internal contents of the source collection
            var cloneData = collection._getCloneData();

            entryModels = _.invoke(cloneData.entryModels, 'clone', sheetModel);
        };

        /**
         * Callback handler for the document operations 'insertColumns' and
         * 'insertRows'. Inserts new columns/rows into this collection, and
         * triggers an 'insert:entries' event containing the index interval of
         * the inserted columns/rows.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyInsertOperation = function (context) {

            // the column/row interval to be inserted
            var interval = context.getJSONInterval(columns);
            // new formatting attributes for the inserted entries
            var attributeSet = context.getOptObj('attrs');
            // the identifier of a new cell auto-style (may be an empty string for the default style)
            var styleId = context.getOptStr('s', '', true);
            // the array index of the first entry model to be moved
            var arrayIndex = getModelArrayIndex(interval.first);
            // the current entry model
            var entryModel = entryModels[arrayIndex];
            // the number of columns/rows to be inserted
            var delta = interval.size();

            // split entry model if it covers the first moved column/row
            if (entryModel && (entryModel.interval.first < interval.first)) {
                splitModel(arrayIndex, interval.first - entryModel.interval.first);
                arrayIndex += 1;
            }

            // update intervals of following entries
            for (var index = arrayIndex; index < entryModels.length; index += 1) {

                // update index interval of the entry
                entryModel = entryModels[index];
                entryModel.interval.first += delta;
                entryModel.interval.last += delta;

                // delete following entries moved outside the collection limits
                if (entryModel.interval.first > maxIndex) {
                    deleteModels(index);
                } else if (entryModel.interval.last >= maxIndex) {
                    deleteModels(index + 1);
                    entryModel.interval.last = maxIndex;
                }
            }

            // create a new entry model, if formatting attributes or an auto-style have been set
            if (attributeSet || styleId) {

                // insert a new entry model, try to merge with adjacent entries
                var newModel = new ColRowModel(sheetModel, interval, columns, attributeSet, styleId, false);
                updateModelSize(newModel);
                insertModel(arrayIndex, newModel);
                mergeModel(arrayIndex, { prev: true, next: true });
            }

            // update pixel offsets of the entries
            updateModelGeometry(interval.first);

            // notify insert listeners
            this.trigger('insert:entries', interval);

            // the insert operation implicitly shifts the cells in the cell collection
            var operation = docModel.makeFullRange(interval, columns).toJSON();
            operation.dir = SheetUtils.getDirection(!columns, false).toJSON();
            sheetModel.getCellCollection().applyMoveCellsOperation(new SheetOperationContext(docModel, operation));
        };

        /**
         * Callback handler for the document operations 'deleteColumns' and
         * 'deleteRows'. Removes existing columns/rows from this collection,
         * and triggers a 'delete:entries' event containing the index interval
         * of the deleted columns/rows.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyDeleteOperation = function (context) {

            // the column/row interval to be deleted
            var interval = context.getJSONInterval(columns);
            // the array index of the first entry model to be deleted
            var arrayIndex = getModelArrayIndex(interval.first);
            // the current entry model
            var entryModel = entryModels[arrayIndex];
            // the number of elements to be deleted from the array
            var deleteCount = 0;
            // the number of columns/rows to be removed
            var delta = interval.size();

            // split the part of the entry located before the interval
            if (entryModel && (entryModel.interval.first < interval.first)) {
                entryModel = splitModel(arrayIndex, interval.first - entryModel.interval.first);
                arrayIndex += 1;
            }

            // remove and move the remaining entries
            for (var index = arrayIndex; index < entryModels.length; index += 1) {

                // update number of entries to be deleted, update index interval of moved entries
                entryModel = entryModels[index];
                if (entryModel.interval.last <= interval.last) {
                    deleteCount += 1;
                } else {
                    entryModel.interval.first = Math.max(interval.first, entryModel.interval.first - delta);
                    entryModel.interval.last -= delta;
                }
            }

            // delete the array elements
            if (deleteCount > 0) {
                deleteModels(arrayIndex, deleteCount);
            }

            // try to merge with previous entry
            mergeModel(arrayIndex, { prev: true });

            // update pixel offsets of the entries
            updateModelGeometry(interval.first);

            // notify delete listeners
            this.trigger('delete:entries', interval);

            // the delete operation implicitly shifts the cells in the cell collection
            var operation = docModel.makeFullRange(interval, columns).toJSON();
            operation.dir = SheetUtils.getDirection(!columns, true).toJSON();
            sheetModel.getCellCollection().applyMoveCellsOperation(new SheetOperationContext(docModel, operation));
        };

        /**
         * Callback handler for the document operations 'changeColumns' and
         * 'changeRows'. Changes the formatting attributes of all entries
         * covered by the operation, and triggers a 'change:entries' event
         * containing the index interval of the changed columns/rows.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyChangeOperation = function (context) {

            // the column/row interval to be modified
            var interval = context.getJSONInterval(columns);
            // the column/row attributes to be set (or cleared)
            var attributeSet = context.getOptObj('attrs');
            // the identifier of a new cell auto-style (may be an empty string for the default style)
            var styleId = context.has('s') ? context.getStr('s', true) : null;

            // nothing to do without attributes and auto-style
            if (!attributeSet && !_.isString(styleId)) { return; }

            // current column/row index for the next invocation
            var index = interval.first;
            // array index of the next collection entry to be visited
            var arrayIndex = getModelArrayIndex(index);
            // whether any entry model has been changed
            var changed = false;
            // additional information passed to the event listeners
            var changeInfo = { sizeChanged: false, visibilityChanged: false };
            // index of the first column/row with dirty pixel offset due to changed sizes
            var dirtyIndex = null;

            // applies the attributes and auto-style at the passed entry model
            function updateEntryModel(entryModel, lastIndex) {

                // update the cell auto-style of the current entry model
                if (_.isString(styleId) && !autoStyles.areEqualStyleIds(entryModel.style, styleId)) {
                    entryModel.style = styleId;
                    changed = true;
                }

                // rescue old size of the entry model, in 1/100 mm and in pixels
                var oldSizeHmm = entryModel.sizeHmm;
                var oldSize = entryModel.size;

                // set the column/row attributes; post-processing only if any attributes have changed
                if (attributeSet && entryModel.setAttributes(attributeSet)) {

                    // update effective size of the entry model
                    updateModelSize(entryModel);
                    changed = true;

                    // collect changed size and visibility
                    var changedSize = (oldSizeHmm !== entryModel.sizeHmm) || (oldSize !== entryModel.size);
                    changeInfo.sizeChanged = changeInfo.sizeChanged || changedSize;
                    changeInfo.visibilityChanged = changeInfo.visibilityChanged || ((oldSize === 0) !== (entryModel.size === 0));

                    // set index of entries with dirty pixel offsets, if size has been changed
                    if (changedSize && (dirtyIndex === null)) {
                        dirtyIndex = entryModel.interval.last + 1;
                    }
                }

                // continue with next sub-interval
                index = lastIndex + 1;
            }

            // invokes the callback function for the gap before a collection entry
            function processGap(lastIndex) {

                // the preceding entry model, needed to calculate the offsets
                var prevModel = entryModels[arrayIndex - 1];
                // relative start index in the gap
                var relIndex = prevModel ? (index - prevModel.interval.last - 1) : index;
                // start position of the gap, in 1/100mm and in pixels
                var gapOffsetHmm = prevModel ? prevModel.getEndOffsetHmm() : 0;
                var gapOffset = prevModel ? prevModel.getEndOffset() : 0;

                // initialize the new entry model for the gap
                var entryModel = defaultModel.clone(sheetModel);
                entryModel.interval = Interval.create(index, lastIndex);
                entryModel.offsetHmm = gapOffsetHmm + entryModel.sizeHmm * relIndex;
                entryModel.offset = gapOffset + entryModel.size * relIndex;

                // insert new entry into array in modifying mode
                insertModel(arrayIndex, entryModel);
                arrayIndex += 1;

                // set the formatting attributes and auto-style
                updateEntryModel(entryModel, lastIndex);

                // remove entry from the collection, if it does not contain any explicit attributes
                if (isDefaultFormatted(entryModel)) {
                    arrayIndex -= 1;
                    deleteModels(arrayIndex, 1);
                    return;
                }

                // try to merge the new entry with its predecessor
                // (arrayIndex points behind the new entry, after merging, it has to be decreased)
                if (mergeModel(arrayIndex - 1, { prev: true }).prev) {
                    arrayIndex -= 1;
                }
            }

            // split the first entry model not covered completely by the interval
            if ((arrayIndex < entryModels.length) && (entryModels[arrayIndex].interval.first < index)) {
                splitModel(arrayIndex, index - entryModels[arrayIndex].interval.first);
                arrayIndex += 1;
            }

            // process all existing collection entries and the gaps covered by the interval
            while (true) {

                // the collection entry to be visited, exit the loop if the interval is done
                var entryModel = entryModels[arrayIndex];
                if (!entryModel || (interval.last < entryModel.interval.first)) { break; }

                // visit the gap between current index and start of the next entry model
                if (index < entryModel.interval.first) {
                    processGap(entryModel.interval.first - 1);
                }

                // split the last entry model not covered completely by the interval
                var lastIndex = Math.min(interval.last, entryModel.interval.last);
                if (lastIndex < entryModel.interval.last) {
                    splitModel(arrayIndex, interval.last - index + 1);
                }

                // set the formatting attributes and auto-style
                updateEntryModel(entryModel, lastIndex);

                // Remove entry model from the collection, if it does not contain any explicit attributes
                // anymore. On success, do not change the array index (it already points to the next
                // entry model after deleting the current entry model).
                if (isDefaultFormatted(entryModel)) {
                    deleteModels(arrayIndex, 1);
                    continue;
                }

                // Try to merge the entry model with its predecessor. On success, do not change the
                // array index (it already points to the next entry model after merging the entries).
                if (mergeModel(arrayIndex, { prev: true }).prev) {
                    continue;
                }

                // go to next array element
                arrayIndex += 1;
            }

            // visit the gap after the last existing entry model
            if (index <= interval.last) {
                processGap(interval.last);
            }

            // try to merge last visited entry model with its successor
            // (arrayIndex already points to the successor of the last visited entry)
            mergeModel(arrayIndex, { prev: true });

            // update pixel offsets of following entry models, if the size has been changed
            if (dirtyIndex !== null) {
                updateModelGeometry(dirtyIndex);
            }

            // notify change listeners
            if (changed) {
                changeInfo.attributes = (attributeSet && attributeSet[FAMILY_NAME]) || null;
                changeInfo.styleId = styleId || null;
                this.trigger('change:entries', interval, changeInfo);
            }
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the maximum valid column/row index available for an entry in
         * this collection.
         *
         * @returns {Number}
         *  The maximum column/row index.
         */
        this.getMaxIndex = function () {
            return maxIndex;
        };

        /**
         * Returns a new column/row interval covering the entire collection.
         *
         * @returns {Interval}
         *  A new column/row interval covering the entire collection.
         */
        this.getFullInterval = function () {
            return new Interval(0, maxIndex);
        };

        /**
         * Returns the total size of all columns/rows represented by this
         * collection, in 1/100 of millimeters, independent from the current
         * sheet zoom factor.
         *
         * @returns {Number}
         *  The total size of all columns/rows in the sheet, in 1/100 of
         *  millimeters.
         */
        this.getTotalSizeHmm = function () {

            // the last existing entry model
            var lastModel = _.last(entryModels);
            // the start index of the gap after the last entry model
            var index = lastModel ? (lastModel.interval.last + 1) : 0;
            // the end position of the last entry model
            var offset = lastModel ? lastModel.getEndOffsetHmm() : 0;

            // add the size of the gap following the last entry model, this becomes the total size
            return offset + defaultModel.sizeHmm * (maxIndex - index + 1);
        };

        /**
         * Returns the total size of all columns/rows represented by this
         * collection, in pixels according to the current sheet zoom factor.
         *
         * @returns {Number}
         *  The total size of all columns/rows in the sheet, in pixels.
         */
        this.getTotalSize = function () {

            // the last existing entry model
            var lastModel = _.last(entryModels);
            // the start index of the gap after the last entry model
            var index = lastModel ? (lastModel.interval.last + 1) : 0;
            // the end position of the last entry model
            var offset = lastModel ? lastModel.getEndOffset() : 0;

            // add the size of the gap following the last entry model, this becomes the total size
            return offset + defaultModel.size * (maxIndex - index + 1);
        };

        /**
         * Returns a descriptor object for the column/row with the specified
         * index.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @returns {ColRowDescriptor}
         *  A descriptor object for the specified column/row.
         */
        this.getEntry = function (index) {

            // collection is empty: create a descriptor with default settings, covering the entire collection
            if (entryModels.length === 0) {
                return new ColRowDescriptor(autoStyles, defaultModel, index, this.getFullInterval());
            }

            // the array index of the covered, or the following entry model
            var arrayIndex = getModelArrayIndex(index);

            // entry points into gap before first entry: use default size
            if ((arrayIndex < 0) || ((arrayIndex === 0) && (index < entryModels[0].interval.first))) {
                return new ColRowDescriptor(autoStyles, defaultModel, index, new Interval(0, entryModels[0].interval.first - 1));
            }

            // index points into a gap between two entries, or to the gap after the last entry
            var entryModel = entryModels[arrayIndex];
            if (!entryModel || (index < entryModel.interval.first)) {
                var prevModel = entryModels[arrayIndex - 1];
                var gapInterval = new Interval(prevModel.interval.last + 1, entryModel ? (entryModel.interval.first - 1) : maxIndex);
                var entryDesc = new ColRowDescriptor(autoStyles, defaultModel, index, gapInterval);
                // adjust entry offsets (default entry is always relative to index 0 instead of start index of current gap)
                var relIndex = index - prevModel.interval.last - 1;
                entryDesc.offsetHmm = prevModel.getEndOffsetHmm() + entryDesc.sizeHmm * relIndex;
                entryDesc.offset = prevModel.getEndOffset() + entryDesc.size * relIndex;
                return entryDesc;
            }

            // index points into a collection entry
            return new ColRowDescriptor(autoStyles, entryModel, index);
        };

        /**
         * Creates an iterator that visits all collection entries in the
         * specified index intervals.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.visible=false]
         *      If set to true, only visible collection entries will be covered
         *      by the iterator. By default, all visible and hidden entries
         *      will be visited.
         *  - {Boolean} [options.reverse=false]
         *      If set to true, the intervals AND the indexes in each interval
         *      will be processed in reversed order.
         *  - {Boolean} [options.unique=false]
         *      If set to true, the iterator will visit only the first entry of
         *      multiple consecutive collection entries with the exact same
         *      auto-style, and formatting attributes.
         *
         * @returns {Iterator}
         *  The new iterator. The result objects will contain the following
         *  value properties:
         *  - {ColRowDescriptor} value
         *      The descriptor of the collection entry currently visited.
         *  - {Interval} orig
         *      The original index interval (from the passed interval array)
         *      containing the current collection entry.
         *  - {Number} index
         *      The array index of the original interval contained in the
         *      property 'interval' of the result object.
         */
        this.createIterator = function (intervals, options) {

            // whether to iterate in reversed order
            var reverse = Utils.getBooleanOption(options, 'reverse', false);
            // whether to iterate entire formatting intervals
            var unique = Utils.getBooleanOption(options, 'unique', false);
            // the interval array iterator
            var arrayIt = IntervalArray.iterator(intervals, { reverse: reverse });

            function createDescriptorResult(result, index) {
                var entryModel = result.value;
                var entryDesc = new ColRowDescriptor(autoStyles, entryModel, index, result.interval);
                var relIndex = index - result.interval.first;
                entryDesc.offsetHmm = result.offsetHmm + entryModel.sizeHmm * relIndex;
                entryDesc.offset = result.offset + entryModel.size * relIndex;
                return { value: entryDesc };
            }

            // returns the appropriate iterator for a single index interval
            function createIntervalIterator(interval) {

                // the model iterator
                var modelIt = createModelIterator(interval, options);

                // unique mode: visit entire intervals in one step
                if (unique) {
                    return new TransformIterator(modelIt, function (entryModel, result) {
                        return createDescriptorResult(result, result.interval.first);
                    });
                }

                // creates an iterator that visits the single entries of a unique interval
                function createIndexIterator(entryModel, modelResult) {
                    return modelResult.interval.iterator({ reverse: reverse });
                }

                // combines the result of the outer model iterator with the inner index iterator for the unique model intervals
                function createIndexResult(modelResult, indexResult) {
                    return createDescriptorResult(modelResult, indexResult.value);
                }

                // crate a nested iterator, combining the outer model iterator and the inner index iterators
                return new NestedIterator(modelIt, createIndexIterator, createIndexResult);
            }

            // combines the result of the outer array iterator with the inner entry iterators for each interval
            function createIntervalResult(arrayResult, entryResult) {
                return { value: entryResult.value, orig: arrayResult.value, index: arrayResult.index };
            }

            // the nested iterator, combining the outer array iterator and the inner entry iterators
            return new NestedIterator(arrayIt, createIntervalIterator, createIntervalResult);
        };

        /**
         * Creates an iterator that visits the intervals with equal auto-style
         * covered by the passed intervals.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.visible=false]
         *      If set to true, only visible collection entries will be covered
         *      by the iterator. By default, all visible and hidden entries
         *      will be visited.
         *  - {Boolean} [options.reverse=false]
         *      If set to true, the intervals AND the equally formatted parts
         *      in each interval will be processed in reversed order.
         *
         * @returns {Iterator}
         *  The new iterator. The result objects will contain the following
         *  value properties:
         *  - {Interval} value
         *      The current index interval.
         *  - {String} style
         *      The auto-style identifier contained by all collection entries
         *      in the interval, or the value null, if a row interval has no
         *      active auto-style due to its formatting attribute 'customStyle'
         *      set to false.
         *  - {Interval} orig
         *      The original index interval (from the passed interval array)
         *      containing the current collection entry.
         *  - {Number} index
         *      The array index of the original interval contained in the
         *      property 'interval' of the result object.
         */
        this.createStyleIterator = function (intervals, options) {

            // returns the appropriate style iterator for a single interval
            function createStyleIterator(interval) {

                // use an internal model iterator (performance: no need to create the entry descriptors)
                var modelIt = createModelIterator(interval, options);

                // create an iterator that transforms the model iterator to interval results with style identifier
                var styleIt = new TransformIterator(modelIt, function (entryModel, result) {
                    return { value: result.interval, style: entryModel.hasAutoStyle() ? entryModel.style : null };
                });

                // return a reducing iterator that merges consecutive intervals with equal auto-style
                return new ReduceIterator(styleIt, function (result1, result2) {
                    var int1 = result1.value, int2 = result2.value;
                    if ((int1.last + 1 === int2.first) && autoStyles.areEqualStyleIds(result1.style, result2.style)) {
                        int1.last = int2.last;
                        return result1;
                    }
                });
            }

            // combines the results of the outer and inner iterators
            function createIteratorResult(arrayResult, styleResult) {
                return { value: styleResult.value, style: styleResult.style, orig: arrayResult.value, index: arrayResult.index };
            }

            // return the nested iterator, combining an outer array iterator with an inner address iterator for each interval
            var reverse = Utils.getBooleanOption(options, 'reverse', false);
            var arrayIt = IntervalArray.iterator(intervals, { reverse: reverse });
            return new NestedIterator(arrayIt, createStyleIterator, createIteratorResult);
        };

        /**
         * Returns the mixed column/row attributes of all entries covered by
         * the passed index intervals.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval.
         *
         * @returns {Object}
         *  The mixed attributes of all columns/rows covered by the passed
         *  intervals, as simple object (NOT mapped as a 'column' or 'row'
         *  sub-object). The result does not contain any cell formatting
         *  attributes. All attributes that could not be resolved unambiguously
         *  will be set to the value null.
         */
        this.getMixedAttributes = function (intervals) {

            // the resulting mixed attributes
            var mixedAttributes = null;
            // whether the default model has been visited once (no need to process it repeatedly)
            var defaultVisited = false;

            // visit all equally formatted intervals, and build the mixed attributes
            Utils.iterateArray(IntervalArray.get(intervals).merge(), function (interval) {
                return Iterator.forEach(createModelIterator(interval), function (entryModel) {

                    // do not visit the default model representing all gaps in the collection multiple times
                    if (entryModel === defaultModel) {
                        if (defaultVisited) { return; }
                        defaultVisited = true;
                    }

                    // the merged attributes of the current interval (type column/row only)
                    var attributes = entryModel.getMergedEntryAttributes();
                    // whether any attribute is still unambiguous
                    var hasNonNull = false;

                    // first visited interval: store initial attributes
                    if (!mixedAttributes) {
                        mixedAttributes = _.copy(attributes, true);
                        return;
                    }

                    // mix the attributes of the visited interval into the result
                    _.each(attributes, function (value, name) {
                        if (_.isEqual(value, mixedAttributes[name])) {
                            hasNonNull = true;
                        } else {
                            mixedAttributes[name] = null;
                        }
                    });

                    // stop iteration, if all attributes are ambiguous
                    if (!hasNonNull) { return Utils.BREAK; }
                });
            });

            return mixedAttributes;
        };

        /**
         * Returns whether the column/row with the specified index is visible.
         *
         * @param {Number} index
         *  The zero-based index of a column/row.
         *
         * @returns {Boolean}
         *  Whether the column/row is visible.
         */
        this.isEntryVisible = function (index) {
            return this.getEntry(index).size > 0;
        };

        /**
         * Returns information about the column/row with the passed index, if
         * it is visible; otherwise about the nearest visible column/row that
         * follows the column/row with the passed index.
         *
         * @param {Number} index
         *  The zero-based index of a column/row.
         *
         * @param {Number} [lastIndex]
         *  If specified, searching for a visible entry will stop after the
         *  column/row with this index. The index MUST be equal or greater than
         *  the passed index. If omitted, searches to the end of the sheet.
         *
         * @returns {ColRowDescriptor|Null}
         *  A descriptor object for the visible column/row; or null, if no more
         *  visible columns/rows are available.
         */
        this.getNextVisibleEntry = function (index, lastIndex) {
            if (index > maxIndex) { return null; }
            // use an iterator to visit the first visible entry following the passed index
            var result = this.createIterator(new Interval(index, _.isNumber(lastIndex) ? lastIndex : maxIndex), { visible: true }).next();
            return result.done ? null : result.value;
        };

        /**
         * Returns information about the column/row with the passed index, if
         * it is visible; otherwise about the nearest visible column/row that
         * precedes the column/row with the passed index.
         *
         * @param {Number} index
         *  The zero-based index of a column/row.
         *
         * @param {Number} [firstIndex]
         *  If specified, searching for a visible entry will stop before the
         *  column/row with this index. The index MUST be equal or less than
         *  the passed index. If omitted, searches to the beginning of the
         *  sheet.
         *
         * @returns {ColRowDescriptor|Null}
         *  A descriptor object for the visible column/row; or null, if no more
         *  visible columns/rows are available.
         */
        this.getPrevVisibleEntry = function (index, firstIndex) {
            if (index < 0) { return null; }
            // use a reverse iterator to visit the first visible entry preceding the passed index
            var result = this.createIterator(new Interval(_.isNumber(firstIndex) ? firstIndex : 0, index), { visible: true, reverse: true }).next();
            return result.done ? null : result.value;
        };

        /**
         * Returns information about the column/row with the passed index, if
         * it is visible; optionally about a visible column/row located near
         * the passed index.
         *
         * @param {Number} index
         *  The zero-based index of a column/row.
         *
         * @param {String} [method='exact']
         *  Specifies how to look for another column/row, if the specified
         *  column/row is hidden. The following lookup methods are supported:
         *  - 'exact':
         *      Does not look for other columns/rows. If the specified
         *      column/row is hidden, returns null.
         *  - 'next':
         *      Looks for a visible column/row following the passed index.
         *  - 'prev':
         *      Looks for a visible column/row preceding the passed index.
         *  - 'nextPrev':
         *      First, looks for visible column/row following the passed index.
         *      If there is none available, looks for a visible column/row
         *      preceding the passed index.
         *  - 'prevNext':
         *      First, looks for visible column/row preceding the passed index.
         *      If there is none available, looks for a visible column/row
         *      following the passed index.
         *
         * @param {Interval} [boundInterval]
         *  If specified, a bounding index interval to restrict the result to.
         *  If no visible entry could be found inside that interval, null will
         *  be returned.
         *
         * @returns {ColRowDescriptor|Null}
         *  A descriptor object for the visible column/row; or null, if no
         *  visible columns/rows are available.
         */
        this.getVisibleEntry = function (index, method, boundInterval) {

            // the resulting column/row descriptor
            var entryDesc = null;

            function isValidDescriptor() {
                return _.isObject(entryDesc) && (entryDesc.size > 0) && (!boundInterval || boundInterval.containsIndex(entryDesc.index));
            }

            switch (method) {
                case 'next':
                    entryDesc = this.getNextVisibleEntry(index);
                    break;
                case 'prev':
                    entryDesc = this.getPrevVisibleEntry(index);
                    break;
                case 'nextPrev':
                    entryDesc = this.getNextVisibleEntry(index);
                    if (!isValidDescriptor()) { entryDesc = this.getPrevVisibleEntry(index); }
                    break;
                case 'prevNext':
                    entryDesc = this.getPrevVisibleEntry(index);
                    if (!isValidDescriptor()) { entryDesc = this.getNextVisibleEntry(index); }
                    break;
                default:
                    entryDesc = this.getEntry(index);
            }

            return isValidDescriptor() ? entryDesc : null;
        };

        /**
         * Returns whether all columns/rows in the passed interval are hidden.
         *
         * @param {Interval} interval
         *  The index interval to be checked.
         *
         * @returns {Boolean}
         *  Whether all columns/rows in the passed interval are hidden.
         */
        this.isIntervalHidden = function (interval) {
            return this.getIntervalPosition(interval).size === 0;
        };

        /**
         * Returns the visible column/row intervals contained in the passed
         * interval.
         *
         * @param {Interval} interval
         *  The column/row interval to be processed.
         *
         * @returns {IntervalArray}
         *  An array with all visible column/row intervals contained in the
         *  passed interval. If the passed interval is completely hidden, the
         *  returned array will be empty.
         */
        this.getVisibleIntervals = function (interval) {

            // the resulting visible intervals
            var visibleIntervals = new IntervalArray();
            // the last inserted result interval
            var lastInterval = null;

            // use an internal model iterator (performance: do not use public iterator, no need to create the descriptors)
            var iterator = createModelIterator(interval, { visible: true });

            // collect all visible intervals
            Iterator.forEach(iterator, function (entryModel, result) {

                // try to expand last existing interval in the result
                if (lastInterval && (lastInterval.last + 1 === result.interval.first)) {
                    lastInterval.last = result.interval.last;
                } else {
                    visibleIntervals.push(lastInterval = result.interval.clone());
                }
            });

            return visibleIntervals;
        };

        /**
         * Returns the number of visible columns or rows contained in the
         * passed index interval.
         *
         * @param {Interval} interval
         *  The column/row interval to be processed.
         *
         * @returns {Number}
         *  The number of visible columns or rows contained in the passed
         *  interval.
         */
        this.countVisibleIndexes = function (interval) {

            // use an internal model iterator (performance: do not use public iterator, no need to create the descriptors)
            var iterator = createModelIterator(interval, { visible: true });

            // sum up the sizes of all visible intervals
            return Iterator.reduce(0, iterator, function (count, entryModel, result) {
                return count + result.interval.size();
            });
        };

        /**
         * Returns a copy of the passed column/row interval. If the first entry
         * of the passed interval is hidden, the returned interval will start
         * at the first visible entry contained in the interval. Same applies
         * to the last entry of the passed interval. If the entire interval is
         * hidden, null will be returned.
         *
         * @param {Interval} interval
         *  The column/row interval to be processed.
         *
         * @returns {Interval|Null}
         *  The visible part of the passed interval (but may contain inner
         *  hidden columns/rows); or null, if the entire interval is hidden.
         */
        this.shrinkIntervalToVisible = function (interval) {

            // first visible collection entry
            var firstEntry = this.getNextVisibleEntry(interval.first, interval.last);
            // last visible collection entry
            var lastEntry = firstEntry ? this.getPrevVisibleEntry(interval.last, interval.first) : null;

            // return the resulting interval
            return lastEntry ? new Interval(firstEntry.index, lastEntry.index) : null;
        };

        /**
         * Returns a column/row interval that covers the passed interval, and
         * that has been expanded to all hidden columns/rows directly preceding
         * and following the passed interval.
         *
         * @param {Interval} interval
         *  The column/row interval to be expanded.
         *
         * @returns {Interval}
         *  An expanded column/row interval including all leading and trailing
         *  hidden columns/rows.
         */
        this.expandIntervalToHidden = function (interval) {

            // nearest visible collection entry preceding the interval
            var prevEntry = (interval.first > 0) ? this.getPrevVisibleEntry(interval.first - 1) : null;
            // nearest visible collection entry following the interval
            var nextEntry = (interval.last < maxIndex) ? this.getNextVisibleEntry(interval.last + 1) : null;

            // return the resulting interval
            return new Interval(prevEntry ? (prevEntry.index + 1) : interval.first, nextEntry ? (nextEntry.index - 1) : interval.last);
        };

        /**
         * Returns merged column/row intervals covering the visible parts of
         * the passed intervals. First, the intervals will be expanded to the
         * leading and trailing hidden columns/rows. This may reduce the number
         * of intervals in the result, if there are only hidden columns/rows
         * between the passed intervals.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval.
         *
         * @returns {IntervalArray}
         *  The merged intervals, shrunken to leading/trailing visible
         *  columns/rows (but may contain inner hidden columns/rows).
         */
        this.mergeAndShrinkIntervals = function (intervals) {

            // expand passed intervals to hidden columns/rows
            intervals = IntervalArray.map(intervals, this.expandIntervalToHidden, this);

            // merge and sort the intervals (result may be smaller due to expansion to hidden columns/rows)
            intervals = intervals.merge();

            // reduce merged intervals to visible parts (filter out intervals completely hidden)
            return IntervalArray.map(intervals, this.shrinkIntervalToVisible, this);
        };

        /**
         * Returns the absolute sheet offset of a specific position inside a
         * column/row in 1/100 mm, independent from the current sheet zoom
         * factor.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @param {Number} offsetHmm
         *  The relative offset inside the column/row, in 1/100 mm. If this
         *  offset is larger than the total size of the column/row, the
         *  resulting absolute sheet offset will be located at the end position
         *  of the column/row.
         *
         * @returns {Number}
         *  The absolute position of the specified entry offset, in 1/100 mm.
         */
        this.getEntryOffsetHmm = function (index, offsetHmm) {
            var entryDesc = this.getEntry(index);
            return entryDesc.offsetHmm + Utils.minMax(offsetHmm, 0, entryDesc.sizeHmm);
        };

        /**
         * Returns the absolute sheet offset of a specific position inside a
         * column/row in pixels, according to the current sheet zoom factor.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @param {Number} offsetHmm
         *  The relative offset inside the column/row, in 1/100 mm. If this
         *  offset is larger than the total size of the column/row, the
         *  resulting absolute sheet offset will be located at the end position
         *  of the column/row.
         *
         * @returns {Number}
         *  The absolute position of the specified entry offset, in pixels.
         */
        this.getEntryOffset = function (index, offsetHmm) {
            var entryDesc = this.getEntry(index);
            return entryDesc.offset + Utils.minMax(sheetModel.convertHmmToPixel(offsetHmm), 0, entryDesc.size);
        };

        /**
         * Returns information about the column/row covering the passed offset
         * in the sheet in pixels according to the current sheet zoom factor,
         * or in 1/100 of millimeters.
         *
         * @param {Number} offset
         *  The absolute offset in the sheet, in pixels, or in 1/100 of
         *  millimeters, according to the 'pixel' option (see below).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.pixel=false]
         *      If set to true, the parameter 'offset' is interpreted as length
         *      in pixels according to the current sheet zoom factor.
         *      Otherwise, the offset is interpreted in 1/100 of millimeters.
         *  - {Boolean} [options.outerHidden=false]
         *      If set to true, and the passed offset is outside the sheet area
         *      (less than 0, or greater than the last available offset in the
         *      sheet), returns a descriptor for the very first or very last
         *      column/row of the sheet, regardless if this column/row is
         *      hidden or visible.
         *
         * @returns {ColRowDescriptor}
         *  A descriptor object for the column/row, with the additional
         *  properties 'relOffsetHmm' and 'relOffset' (relative offset inside
         *  the column/row). If the passed offset is less than 0, returns
         *  information about the first visible column/row in the sheet; if the
         *  passed offset is greater than the total size of the sheet, returns
         *  information about the last visible column/row in the sheet (unless
         *  the option 'outerHidden' has been set, see above).
         */
        this.getEntryByOffset = function (offset, options) {

            // whether to use pixels instead of 1/100 mm
            var pixel = Utils.getBooleanOption(options, 'pixel', false);
            // the property names for offset and size
            var OFFSET_NAME = pixel ? 'offset' : 'offsetHmm';
            var SIZE_NAME = pixel ? 'size' : 'sizeHmm';
            // the total size of the sheet
            var totalSize = pixel ? self.getTotalSize() : self.getTotalSizeHmm();
            // the passed offset, restricted to the valid sheet dimension
            var currOffset = Utils.minMax(offset, 0, totalSize - 1);
            // the index of the first entry model after the offset
            var arrayIndex = _.sortedIndex(entryModels, Utils.makeSimpleObject(OFFSET_NAME, currOffset + 1), OFFSET_NAME);
            // the current entry model
            var entryModel = entryModels[arrayIndex - 1];
            // the end offset of the previous entry
            var endOffset = !entryModel ? 0 : pixel ? entryModel.getEndOffset() : entryModel.getEndOffsetHmm();
            // the column/row index, relative to the entry or the gap
            var relIndex = 0;
            // the resulting entry descriptor
            var entryDesc = null;

            // special handling if all entries are hidden: return first entry
            if (totalSize === 0) {
                entryDesc = self.getEntry(0);
                entryDesc.relOffset = entryDesc.relOffsetHmm = 0;
                return entryDesc;
            }

            // offset outside sheet limits: always return first/last entry model if specified
            if (Utils.getBooleanOption(options, 'outerHidden', false)) {
                if (offset < 0) {
                    entryDesc = self.getEntry(0);
                } else if (offset >= totalSize) {
                    entryDesc = self.getEntry(maxIndex);
                }
                if (entryDesc && (entryDesc.size === 0)) {
                    entryDesc.relOffset = entryDesc.relOffsetHmm = 0;
                    return entryDesc;
                }
            }

            // offset points into the previous entry
            if (entryModel && (currOffset < endOffset)) {
                relIndex = Math.floor((currOffset - entryModel[OFFSET_NAME]) / entryModel[SIZE_NAME]);
                entryDesc = new ColRowDescriptor(autoStyles, entryModel, entryModel.interval.first + relIndex);
            } else {
                // offset points into the gap before the entry model at arrayIndex (entryModel is located before the offset)
                var nextModel = entryModels[arrayIndex];
                var gapInterval = new Interval(entryModel ? (entryModel.interval.last + 1) : 0, nextModel ? (nextModel.interval.first - 1) : maxIndex);
                relIndex = Math.floor((currOffset - endOffset) / defaultModel[SIZE_NAME]);
                entryDesc = new ColRowDescriptor(autoStyles, defaultModel, (entryModel ? (entryModel.interval.last + 1) : 0) + relIndex, gapInterval);
                // adjust offsets (default entry is relative to index 0, not to start index of current gap)
                entryDesc.offsetHmm = (entryModel ? entryModel.getEndOffsetHmm() : 0) + entryDesc.sizeHmm * relIndex;
                entryDesc.offset = (entryModel ? entryModel.getEndOffset() : 0) + entryDesc.size * relIndex;
            }

            // add relative offset properties
            if (pixel) {
                entryDesc.relOffset = Utils.minMax(offset - entryDesc.offset, 0, entryDesc.size);
                entryDesc.relOffsetHmm = Utils.minMax(sheetModel.convertPixelToHmm(entryDesc.relOffset), 0, entryDesc.sizeHmm);
            } else {
                entryDesc.relOffsetHmm = Utils.minMax(offset - entryDesc.offsetHmm, 0, entryDesc.sizeHmm);
                entryDesc.relOffset = Utils.minMax(sheetModel.convertHmmToPixel(entryDesc.relOffsetHmm), 0, entryDesc.size);
            }

            return entryDesc;
        };

        /**
         * Converts the passed offset to a scroll anchor position which
         * contains the entry index, and a floating-point ratio describing the
         * exact position inside the entry.
         *
         * @param {Number} offset
         *  The absolute offset in the sheet, in pixels, or in 1/100 of
         *  millimeters, according to the 'pixel' option (see below).
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.pixel=false]
         *      If set to true, the parameter 'offset' is interpreted as length
         *      in pixels according to the current sheet zoom factor.
         *      Otherwise, the offset is interpreted in 1/100 of millimeters.
         *  - {Boolean} [options.outerHidden=false]
         *      If set to true, and the passed offset is outside the sheet area
         *      (less than 0, or greater than the last available offset in the
         *      sheet), returns a scroll anchor for the very first or very last
         *      column/row of the sheet, regardless if this column/row is
         *      hidden or visible.
         *
         * @returns {Number}
         *  A scroll anchor position, as floating-point number. The integral
         *  part represents the zero-based column/row index of the collection
         *  entry that contains the passed offset. The fractional part
         *  represents the ratio inside the collection entry.
         */
        this.getScrollAnchorByOffset = function (offset, options) {
            var entryDesc = this.getEntryByOffset(offset, options);
            return entryDesc.index + ((entryDesc.sizeHmm > 0) ? (entryDesc.relOffsetHmm / entryDesc.sizeHmm) : 0);
        };

        /**
         * Converts the passed absolute offset in the sheet given in pixels
         * (according to the current sheet zoom factor) to an absolute offset
         * in 1/100 mm (independent from the current sheet zoom factor).
         *
         * @param {Number} offset
         *  An absolute offset in the sheet in pixels (according to the current
         *  sheet zoom factor).
         *
         * @returns {Number}
         *  The absolute offset in the sheet in 1/100 mm (independent from the
         *  current sheet zoom factor).
         */
        this.convertOffsetToHmm = function (offset) {
            var entryDesc = this.getEntryByOffset(offset, { pixel: true });
            return entryDesc.offsetHmm + entryDesc.relOffsetHmm;
        };

        /**
         * Converts the passed absolute offset in the sheet given in 1/100 mm
         * (independent from the current sheet zoom factor) to an absolute
         * offset in pixels (according to the current sheet zoom factor).
         *
         * @param {Number} offsetHmm
         *  An absolute offset in the sheet in 1/100 mm (independent from the
         *  current sheet zoom factor).
         *
         * @returns {Number}
         *  The absolute offset in the sheet in pixels (according to the
         *  current sheet zoom factor).
         */
        this.convertOffsetToPixel = function (offsetHmm) {
            var entryDesc = this.getEntryByOffset(offsetHmm);
            return entryDesc.offset + entryDesc.relOffset;
        };

        /**
         * Calculates the absolute offset of the passed scroll anchor, in 1/100
         * of millimeters.
         *
         * @param {Number} scrollAnchor
         *  A valid scroll anchor value as floating-point number. See return
         *  value of the method ColRowCollection.getScrollAnchorByOffset() for
         *  details.
         *
         * @returns {Number}
         *  The absolute offset of the passed scroll anchor, in 1/100 mm.
         */
        this.convertScrollAnchorToHmm = function (scrollAnchor) {
            var entryDesc = this.getEntry(Math.floor(scrollAnchor));
            return entryDesc.offsetHmm + Math.round(entryDesc.sizeHmm * (scrollAnchor % 1));
        };

        /**
         * Calculates the absolute offset of the passed scroll anchor, in
         * pixels according to the current sheet zoom factor.
         *
         * @param {Number} scrollAnchor
         *  A valid scroll anchor value as floating-point number. See return
         *  value of the method ColRowCollection.getScrollAnchorByOffset() for
         *  details.
         *
         * @returns {Number}
         *  The absolute offset of the passed scroll anchor, in pixels.
         */
        this.convertScrollAnchorToPixel = function (scrollAnchor) {
            var entryDesc = this.getEntry(Math.floor(scrollAnchor));
            return entryDesc.offset + Math.round(entryDesc.size * (scrollAnchor % 1));
        };

        /**
         * Returns the position and size of the specified column/row interval.
         *
         * @param {Interval} interval
         *  The index interval of the columns/rows, with the zero-based
         *  properties 'first' and 'last'.
         *
         * @returns {Object}
         *  An object with the following properties:
         *  - {Number} position.offsetHmm
         *      The absolute position of the first column/row in 1/100 mm,
         *      independent from the current sheet zoom factor.
         *  - {Number} position.sizeHmm
         *      The total size of the interval in 1/100 mm, independent form
         *      the current sheet zoom factor.
         *  - {Number} position.offset
         *      The absolute position of the first column/row in pixels,
         *      according to the current sheet zoom factor.
         *  - {Number} position.size
         *      The total size of the interval in pixels, according to the
         *      current sheet zoom factor.
         */
        this.getIntervalPosition = function (interval) {

            // descriptor for the first column/row of the interval
            var firstEntryDesc = this.getEntry(interval.first);
            // descriptor for the first column/row following the interval
            var lastEntryDesc = this.getEntry(interval.last + 1);

            return {
                offsetHmm: firstEntryDesc.offsetHmm,
                sizeHmm: lastEntryDesc.offsetHmm - firstEntryDesc.offsetHmm,
                offset: firstEntryDesc.offset,
                size: lastEntryDesc.offset - firstEntryDesc.offset
            };
        };

        // operation generators -----------------------------------------------

        /**
         * Merges the passed index intervals arrays. The interval objects may
         * contain additional style properties that need to be merged.
         *
         * @param {IntervalArray} oldIntervals
         *  An array of index intervals. The intervals MUST be sorted, and they
         *  MUST NOT overlap each other (e.g. an array as returned by the
         *  methods IntervalArray.merge() or IntervalArray.partition()). The
         *  intervals may contain a 'style' property with an auto-style
         *  identifier, and/or an 'attrs' property with explicit formatting
         *  attributes (flat map of key/value pairs).
         *
         * @param {IntervalArray} newIntervals
         *  An array of index intervals that will be merged over the old index
         *  intervals. The intervals MUST be sorted, and they MUST NOT overlap
         *  each other (e.g. an array as returned by the methods
         *  IntervalArray.merge() or IntervalArray.partition()). The interval
         *  objects may contain a 'style' property with an auto-style
         *  identifier, and/or an 'attrs' property with explicit formatting
         *  attributes (flat map of key/value pairs).
         *
         * @returns {IntervalArray}
         *  A new array of index intervals (sorted, without overlapping
         *  intervals). The interval objects contain the merged settings for
         *  the 'style' and 'attrs' properties of the old and new index
         *  intervals.
         */
        this.mergeStyleIntervals = function (sourceIntervals, targetIntervals) {

            // returns a clone of the passed interval with style properties
            function cloneStyleInterval(interval) {
                var newInterval = interval.clone();
                if ('style' in interval) { newInterval.style = interval.style; }
                if ('attrs' in interval) { newInterval.attrs = interval.attrs; }
                return newInterval;
            }

            // the resulting intervals returned by this method
            var resultIntervals = new IntervalArray();

            // arrays will be modified in-place for simplicity, work on clones of cloned objects
            sourceIntervals = IntervalArray.map(sourceIntervals, cloneStyleInterval);
            targetIntervals = IntervalArray.map(targetIntervals, cloneStyleInterval);

            // merge the source intervals array over the target intervals array
            sourceIntervals.some(function (sourceInterval, sourceIndex) {

                // index of last target interval completely preceding the current source interval
                var lastIndex = Utils.findLastIndex(targetIntervals, function (targetInterval) {
                    return targetInterval.last < sourceInterval.first;
                }, { sorted: true });

                // move all preceding target intervals into the result
                if (lastIndex >= 0) {
                    resultIntervals.append(targetIntervals.splice(0, lastIndex + 1));
                }

                // merge all covering target intervals over the current source interval
                while (true) {

                    // no more intervals available in target array: append all remaining source intervals and exit the loop
                    if (targetIntervals.length === 0) {
                        // push current interval separately (may be modified, see below)
                        resultIntervals.append(sourceInterval, sourceIntervals.slice(sourceIndex + 1));
                        return true; // exit the 'some()' array loop
                    }

                    // first remaining interval in the target array
                    var targetInterval = targetIntervals[0];
                    // temporary interval, inserted into the result array
                    var tempInterval = null;

                    // entire source interval fits into the gap before the current target interval
                    if (sourceInterval.last < targetInterval.first) {
                        resultIntervals.push(sourceInterval);
                        // continue the 'some()' array loop with next source interval
                        return;
                    }

                    // one interval starts before the other interval: split the leading interval
                    if (sourceInterval.first < targetInterval.first) {
                        tempInterval = cloneStyleInterval(sourceInterval);
                        tempInterval.last = targetInterval.first - 1;
                        resultIntervals.push(tempInterval);
                        sourceInterval.first = targetInterval.first;
                    } else if (targetInterval.first < sourceInterval.first) {
                        tempInterval = cloneStyleInterval(targetInterval);
                        tempInterval.last = sourceInterval.first - 1;
                        resultIntervals.push(tempInterval);
                        targetInterval.first = sourceInterval.first;
                    }

                    // both intervals start at the same position now: create a new interval with merged style settings
                    tempInterval = new Interval(sourceInterval.first, Math.min(sourceInterval.last, targetInterval.last));
                    resultIntervals.push(tempInterval);

                    // prefer auto-style of target interval over auto-style of source interval
                    if ('style' in targetInterval) {
                        tempInterval.style = targetInterval.style;
                    } else if ('style' in sourceInterval) {
                        tempInterval.style = sourceInterval.style;
                    }

                    // copy or merge the formatting attributes
                    if (('attrs' in sourceInterval) || ('attrs' in targetInterval)) {
                        tempInterval.attrs = _.extend({}, sourceInterval.attrs, targetInterval.attrs);
                    }

                    // source and target intervals have equal size: continue with next source and target interval
                    if (sourceInterval.last === targetInterval.last) {
                        targetIntervals.shift();
                        return;
                    }

                    // source interval ends before target interval: shorten the target interval
                    if (sourceInterval.last < targetInterval.last) {
                        targetInterval.first = sourceInterval.last + 1;
                        return;
                    }

                    // otherwise: shorten source interval, continue with next target interval
                    sourceInterval.first = targetInterval.last + 1;
                    targetIntervals.shift();
                }
            });

            // append remaining target intervals
            resultIntervals.append(targetIntervals);

            // merge adjacent intervals with equal style settings
            return mergeTaggedIntervals(resultIntervals);
        };

        /**
         * Generates the operations, and the undo operations, to fill the
         * passed intervals in this collection with some individual contents.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index interval, or a single index interval. Each
         *  interval object MUST contain an additional property 'fillData' with
         *  properties to be applied at all entries in the respective interval.
         *  See method ColRowCollection.generateFillOperations() for more
         *  details about the expected properties. Overlapping intervals MUST
         *  NOT contain the same properties with different values (e.g.: set
         *  auto-style 'a1' or auto-style 'a2' to the same entries in
         *  overlapping intervals), otherwise the result will be undefined.
         *  However, incomplete property objects of overlapping intervals will
         *  be merged together for the overlapping parts of the intervals
         *  (e.g.: setting the auto-style 'a1' to the column interval A:B, and
         *  the column attribute {visible:true} to the column interval B:C will
         *  result in setting both the auto-style and the column attribute to
         *  column B).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved (with the index intervals that have
         *  really been changed) when all operations have been generated, or
         *  that will be rejected with an object with 'cause' property set to
         *  one of the following error codes:
         *  - 'cols:overflow': Too many columns in the index intervals.
         *  - 'rows:overflow': Too many rows in the index intervals.
         */
        this.generateIntervalOperations = function (generator, intervals) {

            // bug 34641: restrict maximum number of modified columns/rows
            if (intervals.size() > MAX_CHANGE_COUNT) {
                return SheetUtils.makeRejected(columns ? 'cols:overflow' : 'rows:overflow');
            }

            // generate the operations for all intervals, merge content objects of overlapping intervals
            return generateChangeIntervalOperations(generator, intervals, function (fillDataArray) {
                return fillDataArray.reduce(_.extend, {});
            });
        };

        /**
         * Generates the operations, and the undo operations, to fill the same
         * contents into the specified columns/rows.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index interval, or a single index interval.
         *
         * @param {Object} contents
         *  The new properties to be applied at all index intervals:
         *  - {Object} [contents.attrs]
         *      Explicit column/row attributes to be set at the intervals, as a
         *      simple attribute map.
         *  - {String} [contents.s]
         *      The name of a new auto-style to be applied at the intervals. If
         *      this and the property 'a' have been omitted, the current
         *      auto-styles of the intervals will not be changed.
         *  - {Object} [contents.a]
         *      An attribute set with explicit cell formatting attributes to be
         *      applied at the intervals. If this and the property 's' have
         *      been omitted, the current auto-styles of the intervals will not
         *      be changed.
         *  - {Boolean} [contents.nativeSize=false]
         *      If set to true, the size attribute (either "width" or "height")
         *      in the "attrs" property is expected to be in native operation
         *      units according to the file format (OOXML: standard digit count
         *      for column width, font points for height; ODF: 1/100 mm). By
         *      default, the size attribute is always expected in 1/100 of
         *      millimeters, and will be converted to the native operation
         *      units.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved (with the index intervals that have
         *  really been changed) when all operations have been generated, or
         *  that will be rejected with an object with 'cause' property set to
         *  one of the following error codes:
         *  - 'cols:overflow': Too many columns in the index intervals.
         *  - 'rows:overflow': Too many rows in the index intervals.
         */
        this.generateFillOperations = function (generator, intervals, contents) {

            // ensure an array, merge the intervals
            intervals = IntervalArray.get(intervals).merge();
            // bug 34641: restrict maximum number of modified columns/rows
            if (intervals.size() > MAX_CHANGE_COUNT) {
                return SheetUtils.makeRejected(columns ? 'cols:overflow' : 'rows:overflow');
            }

            // add an arbitrary cache key to the contents object that causes internal auto-style caching
            // (all intervals will be formatted with the same new formatting attributes)
            contents = _.clone(contents);
            contents.cacheKey = _.uniqueId('key');

            // process all intervals with the same contents object
            return generateChangeIntervalOperations(generator, intervals, _.constant(contents));
        };

        /**
         * Generates the interval operations, and the undo operations, to fill
         * the columns/rows in the sheet with outer and/or inner borders.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index interval, or a single index interval.
         *
         * @param {Object} borderAttributes
         *  The border attributes to be applied at all intervals. May contain
         *  regular border attributes, as supported by the document operations
         *  ('borderTop', 'borderBottom', 'borderLeft', 'borderRight') which
         *  will be applied at the outer boundaries of the passed intervals;
         *  and the pseudo attributes 'borderInsideHor' or 'borderInsideVert',
         *  which will be applied to the inner entries of the intervals.
         *
         * @param {String} cacheKey
         *  The root cache key used to optimize creation of new auto-styles,
         *  intended to be reused in different generators (columns, rows, and
         *  cells) while creating the operations for the same border settings.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved (with the index intervals that have
         *  really been changed) when all operations have been generated, or
         *  that will be rejected with an object with 'cause' property set to
         *  one of the following error codes:
         *  - 'cols:overflow': Too many columns in the index intervals.
         *  - 'rows:overflow': Too many rows in the index intervals.
         */
        this.generateBorderOperations = (function () {

            // single names of border attributes along the columns/rows
            var LEADING_BORDER_KEY = SheetUtils.getOuterBorderKey(columns, true);
            var TRAILING_BORDER_KEY = SheetUtils.getOuterBorderKey(columns, false);
            var INNER_BORDER_KEY = SheetUtils.getInnerBorderKey(columns);

            // border keys, and single names of border attributes in crossing direction
            var CROSS_LEADING_BORDER_KEY = SheetUtils.getOuterBorderKey(!columns, true);
            var CROSS_TRAILING_BORDER_KEY = SheetUtils.getOuterBorderKey(!columns, false);
            var CROSS_INNER_BORDER_KEY = SheetUtils.getInnerBorderKey(!columns);

            // Returns the parts of the passed intervals that will receive a new border attribute.
            // Parameter 'leading' specifies the position of the cell border to be modified. The
            // resulting intervals will contain an additional property 'fillData' with border settings
            // as expected by the helper method SheetOperationGenerator.createBorderContents().
            function getChangedBorderIntervals(intervals, borderAttributes, leading) {

                // the resulting intervals with additional border information
                var resultIntervals = new IntervalArray();

                // adds the passed settings to the passed intervals, and appends them to the resulting interval array
                function appendIntervals(newIntervals, fillData) {
                    return resultIntervals.append(Utils.addProperty(newIntervals, 'fillData', fillData));
                }

                // get the border attributes, nothing to do without any border attribute
                var outerKey = leading ? LEADING_BORDER_KEY : TRAILING_BORDER_KEY;
                var outerBorder = borderAttributes[SheetUtils.getBorderName(outerKey)];
                var innerBorder = borderAttributes[SheetUtils.getBorderName(INNER_BORDER_KEY)];
                var crossBorder = borderAttributes[SheetUtils.getBorderName(CROSS_INNER_BORDER_KEY)];
                if (!outerBorder && !innerBorder && !crossBorder) { return null; }

                // the crossing inner border attribute to be set as both crossing outer borders
                if (crossBorder) {
                    appendIntervals(intervals.clone(true), { key: CROSS_LEADING_BORDER_KEY,  border: crossBorder, cacheType: 'outer' });
                    appendIntervals(intervals.clone(true), { key: CROSS_TRAILING_BORDER_KEY, border: crossBorder, cacheType: 'outer' });
                }

                // add the adjacent intervals whose opposite borders need to be deleted while setting an outer border
                // (if the outer border will not be changed, the borders of the adjacent entries will not be modified neither)
                if (outerBorder) {
                    // the adjacent ranges at the specified range border whose existing borders will be deleted
                    var adjacentIntervals = getAdjacentIntervals(intervals, leading);
                    // opposite borders will be deleted, e.g. the right border left of a column interval (but not if
                    // they are equal to the outer border set at the column, to reduce the amount of changed entries)
                    var oppositeKey = leading ? TRAILING_BORDER_KEY : LEADING_BORDER_KEY;
                    appendIntervals(adjacentIntervals, { key: oppositeKey, keepBorder: outerBorder });
                }

                // shortcut (outer and inner borders are equal): return the entire intervals
                if (outerBorder && innerBorder && Border.isEqual(outerBorder, innerBorder)) {
                    return appendIntervals(intervals.clone(true), { key: outerKey, border: outerBorder, cacheType: 'outer' });
                }

                // outer border will be set to the outer columns/rows only
                if (outerBorder) {
                    var indexProp = leading ? 'first' : 'last';
                    appendIntervals(IntervalArray.map(intervals, function (interval) {
                        return new Interval(interval[indexProp]);
                    }), { key: outerKey, border: outerBorder, cacheType: 'outer' });
                }

                // inner border will be set to the remaining columns/rows
                if (innerBorder) {
                    var offsetL = leading ? 1 : 0;
                    var offsetT = leading ? 0 : -0;
                    appendIntervals(IntervalArray.map(intervals, function (interval) {
                        return interval.single() ? null : new Interval(interval.first + offsetL, interval.last + offsetT);
                    }), { key: outerKey, border: innerBorder, cacheType: 'inner' });
                }

                return resultIntervals;
            }

            // the actual implementation of the public method returned from local scope
            function generateBorderOperations(generator, intervals, borderAttributes, cacheKey) {
                SheetUtils.log('intervals=' + intervals.stringifyAs(columns) + ' attributes=', borderAttributes);

                // ensure an array, remove duplicates (but do not merge the intervals)
                intervals = IntervalArray.get(intervals).unify();

                // prepare the interval arrays with the parts of the passed intervals that will get a new single border
                var intervalsL = getChangedBorderIntervals(intervals, borderAttributes, true);
                var intervalsT = getChangedBorderIntervals(intervals, borderAttributes, false);

                // the intervals to be processed, as single array
                var fillIntervals = new IntervalArray(intervalsL, intervalsT);
                // bug 34641: restrict maximum number of modified columns/rows
                if (fillIntervals.merge().size() > MAX_CHANGE_COUNT) {
                    return SheetUtils.makeRejected(columns ? 'cols:overflow' : 'rows:overflow');
                }

                // generate the operations for all intervals, use the border attributes resolved above
                return generateChangeIntervalOperations(generator, fillIntervals, function (fillDataArray) {
                    return generator.createBorderContents(fillDataArray, cacheKey);
                });
            }

            return SheetUtils.profileAsyncMethod('ColRowCollection.generateBorderOperations()', generateBorderOperations);
        }());

        /**
         * Generates the operations, and the undo operations, to change the
         * formatting of existing border lines of columns/rows in the sheet.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index interval, or a single index interval.
         *
         * @param {Border} border
         *  A border attribute value, which may be incomplete. All properties
         *  contained in this object (color, line style, line width) will be
         *  set for all visible borders in the columns/rows. Omitted border
         *  properties will not be changed.
         *
         * @param {String} cacheKey
         *  The root cache key used to optimize creation of new auto-styles,
         *  intended to be reused in different generators (columns, rows, and
         *  cells) while creating the operations for the same border settings.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved (with the index intervals that have
         *  really been changed) when all operations have been generated, or
         *  that will be rejected with an object with 'cause' property set to
         *  one of the following error codes:
         *  - 'cols:overflow': Too many columns in the index intervals.
         *  - 'rows:overflow': Too many rows in the index intervals.
         */
        this.generateVisibleBorderOperations = (function () {

            // border keys according to collection orientation
            var LEADING_BORDER_KEY = SheetUtils.getOuterBorderKey(columns, true);
            var TRAILING_BORDER_KEY = SheetUtils.getOuterBorderKey(columns, false);

            // Returns the adjacent intervals of the passed intervals whose border will be changed additionally to the
            // covered intervals. The resulting intervals will contain an additional property 'fillData' with border
            // settings as expected by the helper method SheetOperationGenerator.createVisibleBorderContents().
            function getAdjacentBorderIntervals(intervals, leading) {
                var adjacentIntervals = getAdjacentIntervals(intervals, leading).difference(intervals);
                var oppositeKey = leading ? TRAILING_BORDER_KEY : LEADING_BORDER_KEY;
                return Utils.addProperty(adjacentIntervals, 'fillData', { keys: oppositeKey });
            }

            // the actual implementation of the public method returned from local scope
            function generateVisibleBorderOperations(generator, intervals, border, cacheKey) {
                SheetUtils.log('intervals=' + intervals.stringifyAs(columns) + ' border=', border);

                // ensure an array, merge the intervals
                intervals = IntervalArray.get(intervals).merge();
                // bug 34641: restrict maximum number of modified columns/rows
                if (intervals.size() > MAX_CHANGE_COUNT) {
                    return SheetUtils.makeRejected(columns ? 'cols:overflow' : 'rows:overflow');
                }

                // all entries need to be changed inside the passed intervals
                var fillIntervals = Utils.addProperty(intervals.clone(true), 'fillData', { keys: 'tblrdu' });
                // the adjacent entries outside the intervals (the adjoining borders will be changed too)
                fillIntervals.append(getAdjacentBorderIntervals(intervals, true));
                fillIntervals.append(getAdjacentBorderIntervals(intervals, false));

                // generate the operations for all intervals, use the border attributes resolved above
                return generateChangeIntervalOperations(generator, fillIntervals, function (fillDataArray) {
                    return generator.createVisibleBorderContents(fillDataArray, border, cacheKey);
                });
            }

            return SheetUtils.profileAsyncMethod('ColRowCollection.generateVisibleBorderOperations()', generateVisibleBorderOperations);
        }());

        /**
         * Generates the operations, and the undo operations, to insert new, or
         * to delete existing columns/rows from this collection.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {MoveDescriptor} moveDesc
         *  A descriptor with information about the inserted or deleted index
         *  intervals. Its property 'insert' specifies whether to insert new
         *  entries into, or delete entries from this collection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateMoveOperations = SheetUtils.profileAsyncMethod('ColRowCollection.generateMoveOperations()', function (generator, moveDesc) {
            SheetUtils.log('intervals=' + moveDesc.targetIntervals.stringifyAs(columns));

            // create the delete operations (insertion mode, in reversed order), or the insert operations (deletion mode) for undo
            if (moveDesc.insert) {
                moveDesc.targetIntervals.forEachReverse(function (interval) {
                    generator.generateIntervalOperation(DELETE_OPERATION_NAME, interval, null, { undo: true });
                });
            } else {
                moveDesc.targetIntervals.forEach(function (interval) {
                    generator.generateIntervalOperation(INSERT_OPERATION_NAME, interval, null, { undo: true });
                });
            }

            // create undo operations to restore the current attributes and auto-styles of the deleted intervals
            var promise = generateRestoreIntervalOperations(generator, moveDesc.deleteIntervals);

            // deletion mode: generate the delete operations (in reversed order), and immediately apply them
            if (!moveDesc.insert) {
                return promise.then(function () {
                    return self.iterateArraySliced(moveDesc.targetIntervals, function (interval) {
                        generator.generateIntervalOperation(DELETE_OPERATION_NAME, interval);
                    }, 'ColRowCollection.generateMoveOperations', { reverse: true });
                });
            }

            // insertion mode: create the operations for all intervals, expand formatting into the new inserted intervals
            return promise.then(function () {
                return self.iterateArraySliced(moveDesc.targetIntervals, function (interval) {

                    // operation properties (the operations will be applied immediately)
                    var properties = {};

                    // expand formatting according to the preceding collection entry
                    if (interval.first > 0) {

                        var prevEntryDesc = self.getEntry(interval.first - 1);
                        var thisEntryDesc = self.getEntry(interval.first);

                        // create the auto-style with correct border settings
                        var styleId = generator.generateInsertedAutoStyle(prevEntryDesc.style, thisEntryDesc.style, columns);
                        if (!autoStyles.isDefaultStyleId(styleId)) { properties.s = styleId; }

                        // copy the explicit attributes (ensure that the inserted entries are always visible)
                        var attrs = _.clone(prevEntryDesc.explicit);
                        if (!prevEntryDesc.merged.visible) { attrs.visible = true; }
                        if (!_.isEmpty(attrs)) { properties.attrs = Utils.makeSimpleObject(FAMILY_NAME, attrs); }
                    }

                    // generate and immediately apply an insert operation
                    generator.generateIntervalOperation(INSERT_OPERATION_NAME, interval, properties);

                }, 'ColRowCollection.generateMoveOperations');
            });
        });

        /**
         * Generates the operations, and the undo operations, to automatically
         * fill complete columns or rows.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Interval} interval
         *  The column/row interval containing the source entries to be copied.
         *
         * @param {Direction} direction
         *  The direction in which the specified interval will be expanded. The
         *  values LEFT and UP will cause to auto-fill in front of the passed
         *  interval, regardless of the actual orientation of this collection.
         *
         * @param {Number} count
         *  The number of columns/rows to be generated next to the specified
         *  source interval. MUST be small enough to not leave the available
         *  index space of this collection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateAutoFillOperations = SheetUtils.profileAsyncMethod('ColRowCollection.generateAutoFillOperations()', function (generator, interval, direction, count) {
            SheetUtils.log('source=' + interval.stringifyAs(columns) + ' count=' + count);

            // bug 34641: restrict maximum number of modified columns/rows
            if (count > SheetUtils.MAX_AUTOFILL_COL_ROW_COUNT) {
                return SheetUtils.makeRejected(columns ? 'cols:overflow' : 'rows:overflow');
            }

            // whether to expand/shrink the leading or trailing border
            var leading = SheetUtils.isLeadingDir(direction);

            // special case: if the entire source intervals has equal formatting, simply fill the target interval
            var entryDesc = this.getEntry(interval.first);
            if (entryDesc.uniqueInterval.contains(interval)) {
                var targetInterval = leading ? new Interval(interval.first - count, interval.first - 1) : new Interval(interval.last + 1, interval.last + count);
                return this.generateFillOperations(generator, targetInterval, { attrs: entryDesc.merged, s: entryDesc.style, nativeSize: true });
            }

            // an iterator that visits all models in the source interval in an endless loop
            var iterator = new NestedIterator(new IndexIterator(Number.POSITIVE_INFINITY), function () {
                return createModelIterator(interval, { reverse: leading });
            });

            // build the settings to be generated for the target interval
            var targetIntervals = new IntervalArray();
            var index = leading ? (interval.first - 1) : (interval.last + 1);
            var promise = this.repeatSliced(function () {
                if (count === 0) { return Utils.BREAK; }

                // fetch the next source interval model
                var iterResult = iterator.next();
                var entryModel = iterResult.value;

                // create the target interval
                var size = Math.min(count, iterResult.interval.size());
                var targetInterval = leading ? new Interval(index - size + 1, index) : new Interval(index, index + size - 1);
                targetInterval.fillData = { attrs: entryModel.getMergedEntryAttributes(), s: entryModel.style, nativeSize: true };
                targetIntervals.push(targetInterval);

                // prepare next iteration cycle
                count -= size;
                index = leading ? (targetInterval.first - 1) : (targetInterval.last + 1);

            }, 'ColRowCollection.generateAutoFillOperations');

            // generate the operations for the target interval
            promise = promise.then(function () {
                return generateChangeIntervalOperations(generator, targetIntervals, function (fillDataArray) {
                    return fillDataArray[0];
                });
            });

            return promise;
        });

        // initialization -----------------------------------------------------

        // initialize column/row size of the default entry model
        updateZoomSettings();
        updateModelSize(defaultModel);

        // update entry sizes, after the default column/row size defined by the sheet has been changed
        sheetModel.on({
            'change:attributes': updateDefaultSize,
            'change:viewattributes': changeViewAttributesHandler
        });

        // update entry sizes after the document has been loaded (the collection may
        // be destroyed before import finishes via the operation 'deleteSheet')
        this.waitForImportSuccess(function (alreadyImported) {
            if (!alreadyImported) { updateDefaultSize(); }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(entryModels, 'destroy');
            defaultModel.destroy();
            self = docModel = autoStyles = sheetModel = entryModels = defaultModel = null;
        });

    } }); // class ColRowCollection

    // exports ================================================================

    return ColRowCollection;

});
