/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/presentation/view/dialog/insertdatedialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/dialog/basedialog',
    'gettext!io.ox/office/presentation/main',
    'less!io.ox/office/presentation/view/dialog/presentationdialogs'
], function (Utils, Forms, DateUtils, LocaleData, BaseDialog, gt) {

    'use strict';

    // Global constants and variables
    var insertFieldTitle = /*#. Insert date and time from dialog menu*/ gt('Date and time');

    var dialogWidth = Utils.SMALL_DEVICE ? 320 : 420;

    // var fieldTypesList = [{ option: 'date', value: gt('Current date') }];

    // class InsertDateFieldDialog ============================================

    /**
     * The insert date field modal dialog.
     * Provides localized date formats available for inserting into Presentation document.
     *
     *
     * The changes will be applied to the current document after clicking the 'Ok' button.
     *
     * The dialog itself is shown by calling this.execute. The caller can react on successful
     * finalization of the dialog within an appropriate done-Handler, chained to the
     * this.execute function call.
     *
     * @constructor
     *
     * @extends BaseDialog
     *
     * @param {PresentationView} docView
     *  The view instance containing this dialog.
     */
    var InsertDateFieldDialog = BaseDialog.extend(function (docView) {

        var docModel = docView.getDocModel(),
            numberFormatter = docModel.getNumberFormatter(),
            // container node for first column (category drop-down, fields type list)
            controlRootNode = $('<div class="list">'),
            // fetch locale date formats
            categoryCodesDate = numberFormatter.getCategoryCodes('date'),
            $fieldSelectBox = null,
            $fieldFormat = null,
            autoUpdateDateCheckbox = null;

        // base constructor ---------------------------------------------------

        BaseDialog.call(this, docView, { title: insertFieldTitle, width: dialogWidth });

        // private methods ----------------------------------------------------

        /**
         * Helper function for getting formatted current date.
         *
         * @param {String} formatCode
         *  Format of the date to be created.
         *
         * @return {String}
         *  Current, formatted date.
         */
        function getDateTimeRepresentation(formatCode) {
            var parsedFormat = numberFormatter.getParsedFormat(formatCode);
            var serial = numberFormatter.convertDateToNumber(DateUtils.makeUTCNow());
            return numberFormatter.formatValue(parsedFormat, serial);
        }

        /**
         * Generate field values for selected value (date), and insert them into list.
         *
         */
        function populateFieldFormats() {
            var listFormatsMarkup,
                localDateFormatList = [];

            if (categoryCodesDate.length) {
                _.each(categoryCodesDate, function (formatCode) {
                    localDateFormatList.push({ option: formatCode.value, value: getDateTimeRepresentation(formatCode.value) });
                });
            } else {
                localDateFormatList.push({ option: LocaleData.SHORT_DATE, value: getDateTimeRepresentation(LocaleData.SHORT_DATE) },
                    { option: LocaleData.LONG_DATE, value: getDateTimeRepresentation(LocaleData.LONG_DATE) });
            }
            // generate markup for selection list
            _.each(localDateFormatList, function (format) {
                listFormatsMarkup += Forms.createElementMarkup('option', {
                    attributes: { value: format.option },
                    content: format.value
                });
            });

            $fieldSelectBox = $fieldFormat.find('select');
            $fieldSelectBox[0].innerHTML = listFormatsMarkup;
            $fieldSelectBox.find('option').first().attr('selected', 'selected');
            $fieldSelectBox.attr('size', localDateFormatList.length > 13 ? 13 : localDateFormatList.length);
        }

        /**
         * Initialize controls of the dialog.
         */
        function initControls() {
            //var listMarkup = '';
            var aBtn = Forms.createButtonMarkup({ attributes: { class: 'auto-update-field', role: 'checkbox' }, label: gt('Update automatically') });

            $fieldFormat = $(Forms.createSelectBoxMarkup([], {
                name: 'Field format',
                label: //#. 'Insert Date Field' dialog: label for a list box containing date field formats
                    gt.pgettext('field-dialog', 'Select format'),
                classes: 'control date-field-format',
                size: Utils.SMALL_DEVICE ? 4 : 10
            }));

            controlRootNode.append($fieldFormat, aBtn);
            autoUpdateDateCheckbox = controlRootNode.find('.auto-update-field');
            Forms.checkButtonNodes(autoUpdateDateCheckbox, false, { design: 'boxed', ambiguous: false });
            Forms.setButtonKeyHandler(autoUpdateDateCheckbox);
            populateFieldFormats();
        }

        // initialization -----------------------------------------------------

        // close dialog when losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        // initialize the body element of the dialog
        this.getBody()
            .addClass('io-ox-office-presentation-dialog-insert-date')
            .toggleClass('mobile', !!Utils.SMALL_DEVICE)
            .append(controlRootNode);

        // create the layout of the dialog
        initControls();

        this.setFocusNode($fieldFormat);

        // block keyboard input during applying of operations
        this.on('show', function () {
            docModel.setBlockKeyboardEvent(true);
        });

        // handler for the OK button
        this.setOkHandler(function () {
            var formatIndex = $fieldSelectBox[0] ? ($fieldSelectBox[0].selectedIndex + 1) : null;
            var options = { fieldFormat: $fieldSelectBox.val(), automatic: Forms.isCheckedButtonNode(autoUpdateDateCheckbox), formatIndex: formatIndex };
            var dateTimeType = docModel.getApp().isODF() ? 'date' : 'datetime';
            return docModel.getFieldManager().insertField(dateTimeType, options);
        });

        // listen on checkbox to change icon
        autoUpdateDateCheckbox.on('click', function () {
            Forms.checkButtonNodes(this, !Forms.isCheckedButtonNode(this), { design: 'boxed', ambiguous: false });
        });

        this.registerDestructor(function () {
            docModel.setBlockKeyboardEvent(false);
            docModel = numberFormatter = controlRootNode = categoryCodesDate = null;
            $fieldSelectBox = $fieldFormat = autoUpdateDateCheckbox = null;
        });

    }); // class InsertDateFieldDialog

    // exports ================================================================

    return InsertDateFieldDialog;

});
