/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/model/updatedocumentmixin', [
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/dom'
], function (DrawingFrame, Config, Utils, DOM) {

    'use strict';

    // mix-in class UpdateDocumentMixin ======================================

    /**
     * A mix-in class for the 'updateDocumentFormatting' function.
     *
     * @param app
     *  The application object.
     *
     * @constructor
     */
    function UpdateDocumentMixin(app) {

        var // self reference for local functions
            self = this,
            // the page node
            pagediv = null;

        // private methods ----------------------------------------------------

        /**
         * Dumps profiling information to browser console.
         */
        function dumpProfilingInformation() {

            var tableCount,
                cellCount,
                paragraphCount,
                spanCount,
                drawingCount;

            function dumpUpdateFormattingCount(styleSheets, elementCount, elementName) {
                var updateCount = styleSheets.DBG_COUNT || 0,
                    perElementCount = (elementCount === 0) ? 0 : Utils.round(updateCount / elementCount, 0.01);
                if ((elementCount > 0) || (updateCount > 0)) {
                    Utils.info('Editor.updateDocumentFormatting(): ' + elementCount + ' ' + elementName + ' updated ' + updateCount + ' times (' + perElementCount + ' per element)');
                }
            }

            function dumpParagraphCount(methodName, debugCount) {
                var updateCount = debugCount || 0,
                    perElementCount = (paragraphCount === 0) ? 0 : Utils.round(updateCount / paragraphCount, 0.01);
                Utils.info('Editor.' + methodName + '(): called ' + updateCount + ' times (' + perElementCount + ' per paragraph)');
            }

            if (!Config.DEBUG) { return $.noop; }

            tableCount = pagediv.find(DOM.TABLE_NODE_SELECTOR).length;
            cellCount = pagediv.find('td').length;
            paragraphCount = pagediv.find(DOM.PARAGRAPH_NODE_SELECTOR).length;
            spanCount = pagediv.find('span').filter(function () { return DOM.isPortionSpan(this) || DOM.isTextComponentNode(this.parentNode); }).length;
            drawingCount = pagediv.find(DrawingFrame.NODE_SELECTOR).length;

            dumpUpdateFormattingCount(self.getCharacterStyles(), spanCount, 'text spans');
            dumpUpdateFormattingCount(self.getParagraphStyles(), paragraphCount, 'paragraphs');
            dumpUpdateFormattingCount(self.getTableCellStyles(), cellCount, 'table cells');
            dumpUpdateFormattingCount(self.getTableStyles(), tableCount, 'tables');
            dumpUpdateFormattingCount(self.getDrawingStyles(), drawingCount, 'drawing objects');
            dumpParagraphCount('validateParagraphNode', self.getValidateParagraphNode().DBG_COUNT);
        }

        /**
         * Registering a listener function for the 'tab:activated' event, that is triggered after
         * the first finished activation of a toolbar tab.
         */
        function registerFirstTabActivatedHandler() {

            function activateThumbNails() {
                // adding delay for slide formatting, so that toolbar is really rendered by the browser
                self.executeDelayed(function () {
                    self.trigger('slideModel:init:thumbnail'); // starting creation of thumbnails in slide pane
                }, 'SlideFormatManager.registerFirstTabActivatedHandler', 200);
            }

            if (app.getUserSettingsValue('showToolbars', true)) {
                app.getView().getToolBarTabs().one('tab:activated', activateThumbNails);
            } else {
                self.waitForImportSuccess(activateThumbNails); // alternative event, triggered earlier
            }
        }

        /**
         * A helper function that collects all functionality that needs to be done, after the
         * active slide (if it exists) is set.
         */
        function postActiveSlideSettings() {

            // making the (empty) slide pane visible, so that the slide is positioned correctly
            app.getView().getSlidePane().setSidePaneWidthDuringLoad(self.getSlidePaneWidthInDocAttributes());

            // checking the visibility of the page (not visible for empty views)
            self.checkPageVisibility();

            // making the slide (immediately) visible
            return app.leaveBusyDuringImport(false, { immediateShow: true, zoomType: (Config.AUTOTEST ? null : 'slide') }).always(function () {

                // Setting contenteditable 'false' to pagecontent node in IE (46752)
                Utils.handleReadOnly(DOM.getPageContentNode(pagediv));

                // writing some load logging information into the console (only in debug mode)
                dumpProfilingInformation();

                // registering the handler for the activation of the first tab (then creating of thumbnails in slide pane can start)
                registerFirstTabActivatedHandler();

                // trigger event to initialize the (empty) SlidePane, before the controller updates the GUI (looks better than vice versa)
                // -> triggering synchronously so that empty slide pane is immediately visible together with first slide
                // -> the generation of thumbnails needs its own event ('slideModel:init:thumbnail'), because otherwise the
                //    visibility of the first slide is deferred dramatically. An alternative would be, to trigger this event
                //    with a delay, but then slide thumbnail generation competes with toolbar generation.
                self.trigger('slideModel:init', { isMasterView: self.isMasterView(), activeSlideId: self.getActiveSlideId(), width: self.getSlidePaneWidth(), slideRatio: self.getSlideRatio() });
            });
        }

        // public methods -----------------------------------------------------

        /**
         * Updates the formatting of all elements, after the import operations
         * of the document have been applied.
         * This function handles also the case, if the document was loaded from
         * local storage.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the formatting has been
         *  updated successfully, or rejected when an error has occurred.
         */
        this.updateDocumentFormatting = function () {

            var // an object containing all slide IDs and slides
                allSlides = null,
                // the id of the active slide (TODO: This is not always the first slide in standard view)
                // -> this can also be undefined, if there is not slide in the view
                activeSlide = '',
                // the format manager
                slideFormatManager = self.getSlideFormatManager(),
                // the page styles object
                pageStyles = self.getPageStyles(),
                // the object containing the page attributes
                pageAttributes = null,
                // whether the document is loaded from local storage
                isLocalStorageImport = self.isLocalStorageImport();

            // setting globals
            pagediv = self.getNode();
            allSlides = self.getAllSlides();
            activeSlide = self.getIdOfFirstSlideOfActiveView();
            pageAttributes = self.getPageStyles().getElementAttributes(pagediv);

            // setting tasks at the slide format manager
            slideFormatManager.initializeManager(self.getIdTypeConnectionModel(), self.getStandardSlideOrder(true), self.getMasterSlideOrder(true), { localStorage: isLocalStorageImport });

            // hiding all slides
            _.invoke(allSlides, 'addClass', 'invisibleslide');

            // localStorage: Removing superfluous classes at slides that could not be removed when saving file to local storage
            if (isLocalStorageImport) { _.invoke(allSlides, 'removeClass', 'notselectable'); }

            // resetting an existing selection, after all operations are applied
            self.getSelection().resetSelection();

            // saving page attributes at model for convenience reasons
            self.setPageAttributes(pageAttributes);

            // update the root page node
            pageStyles.updateElementFormatting(pagediv);

            // Performance: Removing the marker class set from fast load
            pagediv.removeClass(DOM.HIDDENAFTERATTACH_CLASSNAME);

            // in localStorage the slides need to be formatted, because of Canvas and because they might have never been
            // formatted before (example: slide_100 is often not formatted before the document is closed and written into
            // local storage -> support of class 'slideformatted' in slideFormatManager).

            if (activeSlide) {
                return slideFormatManager.forceSlideFormatting(activeSlide).done(function () {
                    // activating the first slide after formatting it and its parents completely
                    self.setActiveSlideId(activeSlide, { forceUpdate: true });
                }).then(postActiveSlideSettings);
            } else {
                postActiveSlideSettings();
                return $.when();
            }
        };

        /**
         * Handler for loading empty default documents with fast load. It applies html string,
         * then actions, starts formatting the document, and at the end, leaves busy mode.
         *
         * @param {String} markup
         *  The HTML mark-up to be shown as initial document contents.
         *
         * @param {Array} actions
         *  The operation actions to be applied to finalize the fast import.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the actions have been applied.
         */
        this.fastEmptyLoadHandler = function (markup, actions) {
            // markup is in form of object, parse and get data from mainDocument
            markup = JSON.parse(markup);
            self.setFullModelNode(markup.mainDocument);

            self.trigger('fastemptyload:done'); // updating the model

            return self.applyActions(actions, { external: true }).then(function () {
                return self.updateDocumentFormatting();
            });
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class UpdateDocumentMixin

    // exports ================================================================

    return UpdateDocumentMixin;

});
