/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/stylesheetpicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/baseframework/app/appobjectmixin'
], function (Utils, LocaleData, RadioList, AppObjectMixin) {

    'use strict';

    // class StyleSheetPicker =================================================

    /**
     * A drop-down list control used to select a style sheet from a list. The
     * drop-down list entries will visualize the formatting attributes of the
     * style sheet if possible.
     *
     * @constructor
     *
     * @extends RadioList
     * @extends AppObjectMixin
     *
     * @param {EditView} docView
     *  The document view instance containing this control.
     *
     * @param {String} family
     *  The attribute family of the style sheets visualized by this control.
     *
     * @param {Object} [initOptions]
     *  Optional parameters passed to the RadioList constructor. Supports all
     *  options of the RadioList class, and the following additional options:
     *  - {String|Array<String>} [initOptions.previewFamilies]
     *      The attribute families used to get the formatting options of the
     *      list items representing the style sheets. If omitted, only
     *      attributes of the family specified by the 'family' parameter will
     *      be used.
     *  - {Array<String>} [initOptions.sections]
     *      An array specifying the order of all predefined section
     *      identifiers.
     *  - {String} [initOptions.i18nModulePath]
     *      The module path to the localized JSON object containing the
     *      configuration of translated style sheet names. Each property of the
     *      map must be a string with the English style sheet name as key.
     *      These keys may contain regular expressions which must be placed in
     *      capturing groups (in parentheses). The values of the map are the
     *      translated style sheet names. The placeholders '$1', '$2', etc.
     *      will match the capturing groups defined in the key patterns.
     *  - {Boolean} [initOptions.addSectionLabel=false]
     *      Whether to add a label to the sections.
     *  - {Array<Object>} [initOptions.additionalItems]
     *      Additional stylesheet items. The array contains objects with the
     *      style attributes.
     */
    function StyleSheetPicker(docView, family, initOptions) {

        // self reference
        var self = this;

        // the model instance
        var docModel = docView.getDocModel();

        // the theme model that has been used to create the scheme color table
        var themeModel = docModel.getThemeModel();

        // the style collection
        var styleCollection = docModel.getStyleCollection(family);

        // the path to the translated style sheet names
        var modulePath = Utils.getStringOption(initOptions, 'i18nModulePath', '');

        // the configuration for translated style sheet names
        var translationDatabase = {};

        // the cache with translated style sheet names
        var translationCache = {};

        // whether the preview table needs to be refreshed
        var dirty = true;

        var addSectionLabel = Utils.getBooleanOption(initOptions, 'sectionLabel', false);

        var additionalItems = Utils.getArrayOption(initOptions, 'additionalItems', []);

        // base constructors --------------------------------------------------

        RadioList.call(this, docView, Utils.extendOptions({
            itemDesign: 'grid',
            sortItems: true,
            updateCaptionMode: 'none'
        }, initOptions));

        AppObjectMixin.call(this, docView.getApp());

        // private methods ----------------------------------------------------

        /**
         * Returns the translation of the passed style sheet name for the
         * current locale. Additionally, the passed style name will be adjusted
         * further for GUI display. All words will be capitalized, automatic
         * line breaks before numbers will be prevented by converting the
         * preceding space characters to NBSP characters.
         *
         * @param {String} styleName
         *  The original (English) style sheet name.
         *
         * @returns {String}
         *  The translated and adjusted style sheet name. If no translation is
         *  available, the passed style sheet name will be adjusted and
         *  returned.
         */
        function translateStyleName(styleName) {

            // the lower-case style name, used as database key
            var styleKey = styleName.toLowerCase();
            // the translated style name
            var translatedName = null;

            // first, check the cache of the translation database
            translatedName = translationCache[styleKey];

            if (!translatedName) {
                translatedName = Utils.translateWordFromDatabase(styleName, translationDatabase);
                // adjust the resulting style name for GUI display
                translatedName = translatedName.replace(/ (\d)/g, '\xa0$1');
                // put the translated name into the cache
                translationCache[styleKey] = translatedName;
            }
            return _.noI18n(translatedName);
        }

        /**
         * Writes the translated name of the specified style sheet into the
         * drop-down button.
         */
        function updateHandler(styleId) {
            var styleName = styleId ? styleCollection.getName(styleId) : null;
            if (!styleName) {
                _.find(additionalItems, function (style) {
                    if (style.id === styleId) {
                        styleName = style.name;
                        return true;
                    }
                });
            }
            self.setLabel(styleName ? translateStyleName(styleName) : '');
        }

        function createOptionButton(styleName, styleId, category, priority) {

            // the sort index stored at the button for lexicographical sorting
            var sortIndex = String((priority < 0) ? (priority + 0x7FFFFFFF) : priority);

            //TODO: hacked code for functions like "create new paragraph stylesheet"
            //we could use the category for
            if (styleId.indexOf('\x00') < 0) {
                // translate and adjust style name
                styleName = translateStyleName(styleName);
            }

            // build a sorting index usable for lexicographical comparison:
            // 1 digit for priority sign, 10 digits positive priority,
            // followed by lower-case translated style sheet name
            sortIndex = ('000000000' + sortIndex).substr(-10);
            sortIndex = ((priority < 0) ? '0' : '1') + sortIndex + styleName.toLowerCase();

            // create the list item, pass sorting index
            self.createOptionButton(styleId, { section: category, label: styleName, tooltip: styleName, sortIndex: sortIndex, priority: priority });
        }

        /**
         * Creates a new button in the drop-down menu for the specified style
         * sheet.
         */
        function createStyleCollectionButton(styleName, styleId) {

            // nothing to do for hidden styles
            if (styleCollection.isHidden(styleId)) { return; }

            // the section identifier for the style sheet
            var category = styleCollection.getUICategory(styleId) || '';
            // sorting priority
            var priority = Math.floor(styleCollection.getUIPriority(styleId));

            createOptionButton(styleName, styleId, category, priority);
        }

        /**
         * Create a new additional button.
         * @param {Object} button see@ initOptions.additionalItems
         */
        function createAdditionalButton(button) {
            createOptionButton(button.name, button.id, button.category, button.priority);
        }

        /**
         * Fills the drop-down list with all known style names, and adds
         * preview CSS formatting to the list items.
         */
        function initializeStyleTable() {

            var newThemeModel = docModel.getThemeModel();
            dirty = dirty || (themeModel !== newThemeModel);
            themeModel = newThemeModel;
            if (!dirty) { return; }

            // bug 33737, 33747: restore browser focus
            self.getMenu().guardFocusedListItem(function () {
                self.clearMenu();
                // configure the sections
                _.each(Utils.getArrayOption(initOptions, 'sections'), function (sectionId) {
                    if (addSectionLabel) {
                        var translatedName = translateStyleName(sectionId);
                        self.createMenuSection(sectionId, { label: translatedName ? translatedName : sectionId });
                    } else {
                        self.createMenuSection(sectionId);
                    }
                });
                // insert the style sheets
                _.each(styleCollection.getStyleSheetNames({ skipHidden: true }), createStyleCollectionButton);

                _.each(additionalItems, createAdditionalButton);
            });

            dirty = false;
        }

        /**
         * Handles inserted or modified style sheets in the style collection.
         */
        function insertStyleHandler(event, styleId) {
            // bug 33737, 33747: restore browser focus
            self.getMenu().guardFocusedListItem(function () {

                // Bug 39578: indices are temporary wrong because new button is always at the end and sorting is debounced
                // collecting old index
                // Bug 40403 double quotes must be escaped in attribute selector
                var btnSelector = 'a[data-value="' + Utils.escapeHTML(styleId) + '"]';
                var menuNode = self.getMenu().getNode();
                var oldCellNode = menuNode.find(btnSelector).parent();
                var oldIndex = oldCellNode.index();

                self.deleteOptionButton(styleId);
                createStyleCollectionButton(styleCollection.getName(styleId), styleId);

                //compare old with new index an correct if required
                var newCellNode = menuNode.find(btnSelector).parent();
                var newIndex = newCellNode.index();

                if (newIndex !== oldIndex) {
                    var parent = newCellNode.parent()[0];
                    if (oldIndex === 0) {
                        newCellNode.insertBefore(parent.childNodes[oldIndex]);
                    } else {
                        newCellNode.insertAfter(parent.childNodes[oldIndex - 1]);
                    }
                }
            });
        }

        /**
         * Handles deleted style sheets in the style collection.
         */
        function deleteStyleHandler(event, styleId) {
            self.deleteOptionButton(styleId);
        }

        // initialization -----------------------------------------------------

        // add CSS class to menu button and drop-down list for special formatting
        this.getNode().addClass('style-picker family-' + family);
        this.getMenuNode().addClass('app-' + docView.getApp().getDocumentType() + ' style-picker family-' + family);

        // update translated style name in drop-down button
        this.registerUpdateHandler(updateHandler);

        // handle all changes in the style sheet collection
        this.listenTo(styleCollection, 'insert:stylesheet', insertStyleHandler);
        this.listenTo(styleCollection, 'delete:stylesheet', deleteStyleHandler);
        this.listenTo(styleCollection, 'change:stylesheet', insertStyleHandler);

        // repaint the color table if the theme target chain has changed
        this.getMenu().on('popup:beforeshow', initializeStyleTable);

        // reinitialize the preview if theme settings have been changed
        this.listenTo(docModel.getThemeCollection(), 'triggered', function () { dirty = true; });

        // load the translated style names if specified
        if (modulePath.length > 0) {
            this.waitForSuccess(LocaleData.loadResource(modulePath, { merge: true }), function (data) {
                translationDatabase = data;
                translationCache = {};
                if (this.isImportSucceeded()) { dirty = true; }
            }, this);
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = initOptions = docView = docModel = styleCollection = null;
        });

    } // class StyleSheetPicker

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: StyleSheetPicker });

});
